import os
from adari_core.data_libs.master_raw_cuts import MasterRawCutsReport
from adari_core.data_libs.master_sky_flat import MasterSkyFlatReport
from adari_core.plots.text import TextPlot
from .sphere_utils import SphereSetupInfo, SphereReportMixin

class SphereImgFlatReport(SphereReportMixin, MasterRawCutsReport, MasterSkyFlatReport):

    def __init__(self):
        super().__init__("sphere_img_flat")
        self.files_needed = [
            "IRD_FLAT_FIELD",
            "ZPL_INT_FLAT_FIELD_IMAGING_CAM1",
            "ZPL_INT_FLAT_FIELD_IMAGING_CAM2",
            "ZPL_INT_FLAT_FIELD_MASTER_CAM1",
            "ZPL_INT_FLAT_FIELD_MASTER_CAM2",
        ]
        self.center_size = 200
        self.min = 0.8
        self.max = 1.2
        self.img = False
        self.multi = False

    def parse_sof(self):

        master_flat = []
        self.multi = False
        self.img = False

        for filename, catg in self.inputs:
            if catg in self.files_needed:
                master_flat.append(filename)
            if (
                catg == "ZPL_INT_FLAT_FIELD_MASTER_CAM1"
                or catg == "ZPL_INT_FLAT_FIELD_MASTER_CAM2"
            ):
                self.multi = True
            if (
                catg == "ZPL_INT_FLAT_FIELD_IMAGING_CAM1"
                or catg == "ZPL_INT_FLAT_FIELD_IMAGING_CAM2"
            ):
                self.multi = True
                self.img = True

        # Build and return the file name list
        file_lists = []
        for item in master_flat:
            file_lists.append(
                {
                    "master_im": item,
                }
            )

        return file_lists

    def generate_panels(self, **kwargs):
        panels = {}
        if self.img:
            exts = ["IFRAME.IMAGE", "PFRAME.IMAGE"]
        else:
            exts = ["IMAGE"]

        vspace = 0.3
        for ext in exts:
            new_panels = super().generate_panels(
                master_im_ext=ext,
                cut_clipping="val",
                cut_n_clipping=[0.8, 1.2],
                collapse_clipping="val",
                collapse_n_clipping=[0.8, 1.2],
            )
            for i, (panel, panel_descr) in enumerate(new_panels.items()):
                panel_descr["report_description"] = (
                    f"SPHERE img flat panel - "
                    f"{os.path.basename(panel_descr['master_im'])}, "
                    f"{panel_descr['master_im_ext']}"
                )

                master_im = self.hdus[i]["master_im"]
                hdr = master_im["PRIMARY"].header
                procatg = str(hdr.get("HIERARCH ESO PRO CATG"))

                # Text Plot
                fname = os.path.basename(str(master_im.filename()))
                t1 = TextPlot(columns=1, v_space=vspace)

                col1 = (
                    str(master_im["PRIMARY"].header.get("INSTRUME")),
                    "EXTNAME: " + ext,
                    "PRO CATG: " + procatg,
                    "FILE NAME: " + fname,
                    "RAW1 NAME: "
                    + str(
                        master_im["PRIMARY"].header.get(
                            "HIERARCH ESO PRO REC1 RAW1 NAME"
                        )
                    ),
                )
                t1.add_data(col1)
                panel.assign_plot(t1, 0, 0, xext=2)

                t2 = TextPlot(columns=1, v_space=vspace, xext=1)
                if self.multi:
                    self.metadata = SphereSetupInfo.zpl_int_flat(master_im)
                else:
                    self.metadata = SphereSetupInfo.ird_flat(master_im)
                col2 = self.metadata
                t2.add_data(col2)
                panel.assign_plot(t2, 2, 0, xext=1)
                if procatg == "IRD_FLAT_FIELD":
                    master_plot = panel.retrieve(0, 1)
                    master_center = panel.retrieve(0, 2)
                    cut = panel.retrieve(1, 1)
                    cut_cent = panel.retrieve(1, 2)
                    cut.add_data(
                        master_plot,
                        master_plot.get_data_coord(master_plot.data.shape[1] // 4, "x"),
                        label="master",
                        color="red",
                    )
                    cut_cent.add_data(
                        master_center,
                        master_center.get_data_coord(
                            master_center.data.shape[1] // 1.25, "x"
                        ),
                        label="master",
                        color="red",
                    )

            panels = {**panels, **new_panels}

        return panels


rep = SphereImgFlatReport()
