/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_ifs_cal_background.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ifs_keywords.h"
#include "sph_ifs_tags.h"
#include "sph_common_keywords.h"
#include "sph_shared_irdifs.h"
#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include "sph_fitting.h"
#include "sph_keyword_manager.h"
#include "sph_ifs_subtract_dark_scaled.h"
#include "sph_framecombination.h"

#include <cpl.h>
#include <math.h>
#include <string.h>


extern sph_error_code
sph_subtract_dark_from_raws(cpl_frameset* rawframes, sph_master_frame* dark,
        int check_dit, int check_rom);

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IFS_CAL_BACKGROUND_GENERAL;
extern sph_error_code SPH_IFS_CAL_BACKGROUND_FRAMES_MISSING;

static sph_master_frame*
sph_ifs_create_polyfit(sph_master_frame* mframe, int order, int nsamples) {
    sph_master_frame* polframe = NULL;
    int nx = 0;
    int ny = 0;
    cpl_vector* xvals = NULL;
    cpl_vector* yvals = NULL;
    cpl_vector* zvals = NULL;
    int xx = 0;
    int yy = 0;
    double val = 0.0;
    gsl_rng* pRNG = NULL;
    int cc = 0;
    int bpix = 0;
    cpl_polynomial* poly = NULL;

    nx = cpl_image_get_size_x(mframe->image);
    ny = cpl_image_get_size_y(mframe->image);

    pRNG = gsl_rng_alloc(gsl_rng_taus);
    if (!pRNG)
        return NULL;
    polframe = sph_master_frame_duplicate(mframe);
    if (!polframe)
        return NULL;
    xvals = cpl_vector_new(nsamples);
    yvals = cpl_vector_new(nsamples);
    zvals = cpl_vector_new(nsamples);
    if (!xvals || !yvals || !zvals)
        return NULL;

    cpl_image_delete(polframe->ncombmap);
    polframe->ncombmap = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_delete(polframe->badpixelmap);
    polframe->badpixelmap = cpl_image_new(nx, ny, CPL_TYPE_INT);

    xx = (int) gsl_ran_flat(pRNG, 0.0, nx);
    yy = (int) gsl_ran_flat(pRNG, 0.0, ny);

    while (cc < nsamples) {
        xx = (int) gsl_ran_flat(pRNG, 0.0, nx);
        yy = (int) gsl_ran_flat(pRNG, 0.0, ny);
        while (cpl_image_get(mframe->badpixelmap, xx + 1, yy + 1, &bpix) > 0) {
            xx = (int) gsl_ran_flat(pRNG, 0.0, nx);
            yy = (int) gsl_ran_flat(pRNG, 0.0, ny);
        }
        val = cpl_image_get(mframe->image, xx + 1, yy + 1, &bpix);
        cpl_vector_set(xvals, cc, xx);
        cpl_vector_set(yvals, cc, yy);
        cpl_vector_set(zvals, cc, val);
        val = cpl_image_get(polframe->ncombmap, xx + 1, yy + 1, &bpix);
        cpl_image_set(polframe->ncombmap, xx + 1, yy + 1, val + 1);
        cc++;
    }

    poly = sph_fitting_fit_poly2d(xvals, yvals, zvals, NULL, order, order);
    if (poly) {
        double vvxy[2];
        cpl_vector* vv = cpl_vector_wrap(2, vvxy);
        for (yy = 0; yy < ny; ++yy) {
            for (xx = 0; xx < nx; ++xx) {
                vvxy[0] = xx;
                vvxy[1] = yy;
                val = cpl_polynomial_eval(poly, vv);
                cpl_image_set(polframe->image, xx + 1, yy + 1, val);
                val -= cpl_image_get(mframe->image, xx + 1, yy + 1, &bpix);
                val *= val;
                cpl_image_set(polframe->rmsmap, xx + 1, yy + 1, val);
            }
        }
        cpl_vector_unwrap(vv);
    }
    cpl_vector_delete(xvals);
    xvals = NULL;
    cpl_vector_delete(yvals);
    yvals = NULL;
    cpl_vector_delete(zvals);
    zvals = NULL;
    cpl_polynomial_delete(poly);
    poly = NULL;
    gsl_rng_free(pRNG);
    SPH_RAISE_CPL_RESET;
    return polframe;
}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ifs_cal_background_run Create Master Dark Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master dark.
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ifs_cal_background.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ifs_cal_background recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ifs_cal_background_run(sph_ifs_cal_background* self) {
    int recipe_error = CPL_ERROR_NONE;
    cpl_errorstate prestate = cpl_errorstate_get();
    sph_master_frame* cal_background_mframe = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* fitresult = NULL;
    char cal_background_filename[256];
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_propertylist* rawpl = NULL;
    double rms = 0.0;
    double expdark = 0.0;
    double expbg = 0.0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (self->cal_background_outfilename == NULL) {
        SPH_ERROR_RAISE_ERR( SPH_IFS_CAL_BACKGROUND_GENERAL,
                "No output filename was provided "
                "for product. "
                "Something must be wrong with "
                "the passed parameterlist.\n");
        goto EXIT;
    }

    cal_background_mframe = sph_framecombination_master_frame_from_cpl_frameset(
            self->rawframes, self->coll_alg, self->framecomb_parameterlist);

    if (cal_background_mframe == NULL) {
        SPH_ERROR_RAISE_ERR( SPH_IFS_CAL_BACKGROUND_GENERAL,
                "Could not create the collapsed frame."
                "Please check the previous "
                "error messages.\n");
        goto EXIT;
    }

    rawpl = sph_keyword_manager_load_properties(
            cpl_frame_get_filename(cpl_frameset_get_first(self->rawframes)), 0);

    expbg = cpl_propertylist_get_double(rawpl, SPH_COMMON_KEYWORD_SEQ1DIT);

    if (expbg <= 0) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT, "Bad DIT");
        goto EXIT;
    }

    if (self->master_dark_frame) {
        dark = sph_master_frame_load_(self->master_dark_frame, 0);
        if (dark) {
            rerr = sph_ifs_subtract_dark_scaled(cal_background_mframe, dark);
            expdark = cpl_propertylist_get_double(dark->properties,
                    SPH_COMMON_KEYWORD_SEQ1DIT);
            if (expdark < expbg) {
                sph_master_frame_divide_double(cal_background_mframe,
                        expbg - expdark);
            } else {
                SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT, "Could not "
                "divide background frame by exposure time, "
                "since raw frames do not have exposures times "
                "larger than the master dark."
                "This recipe requires raws with very"
                "long exposure times.");
                goto EXIT;
            }
            sph_master_frame_delete(dark);
            dark = NULL;
        } else {
            rerr = CPL_ERROR_FILE_IO;
        }
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERROR_RAISE_ERR( rerr, "Error when subtracting dark.");
            goto EXIT;
        }
    }

    (void)strcpy(cal_background_filename, self->cal_background_outfilename);

    recipe_error = sph_master_frame_quality_check(cal_background_mframe);
    if (recipe_error != CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_UNSPECIFIED,
                "Could not do the quality check-- "
                "Please check the previous error messages.\n");
        recipe_error = CPL_ERROR_UNSPECIFIED;
        goto EXIT;
    }

    recipe_error = sph_master_frame_mask_sigma(cal_background_mframe,
            self->sigma_clip);

    if (recipe_error != CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_ERR( SPH_IFS_CAL_BACKGROUND_GENERAL,
                "Could not mask sigma -- "
                "Please check the previous "
                "error messages.\n");
        goto EXIT;
    }

    recipe_error = sph_master_frame_quality_check(cal_background_mframe);

    if (recipe_error != CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_ERR( SPH_IFS_CAL_BACKGROUND_GENERAL,
                "Could not do the quality check-- "
                "Please check the previous error messages.\n");
        goto EXIT;
    }

    if (self->smooth > 0.0) {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Interpolating bad pixels...");
        sph_master_frame_interpolate_bpix(cal_background_mframe);
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "Smoothing with FWHM = %f ...", self->smooth);
        sph_master_frame_smooth(cal_background_mframe, self->smooth);
    }

    cpl_propertylist_append_double(cal_background_mframe->qclist,
            SPH_IFS_KEYWORD_CAL_BG_QC_SMOOTHING_LENGTH, self->smooth);
    cpl_propertylist_append_double(cal_background_mframe->qclist,
                                   SPH_IFS_KEYWORD_CAL_BG_QC_MEAN,
            sph_master_frame_get_mean(cal_background_mframe, &rms));
    cpl_propertylist_append_double(cal_background_mframe->qclist,
                                   SPH_IFS_KEYWORD_CAL_BG_QC_RMS, rms);
    cpl_propertylist_append_double(cal_background_mframe->qclist,
                                   SPH_IFS_KEYWORD_CAL_BG_QC_MEDIAN,
            sph_master_frame_get_median(cal_background_mframe, &rms));

    sph_irdifs_qc_persistance(cal_background_mframe->qclist,
                              cal_background_mframe, rawpl,
                              640, 1470, 720, 1570,
                              770, 1370, 850, 1440);
    sph_irdifs_bg_qc_rate(cal_background_mframe->qclist, rawpl,
                          SPH_IFS_KEYWORD_CAL_BG_QC_MEDIAN);

    sph_master_frame_save_dfs(cal_background_mframe, cal_background_filename,
            self->inframes, NULL, self->inparams,
            SPH_IFS_TAG_CAL_BACKGROUND_CALIB,
            SPH_RECIPE_NAME_IFS_CAL_BACKGROUND, SPH_PIPELINE_NAME_IFS, NULL);

    if (self->fitflag) {
        fitresult = sph_ifs_create_polyfit(cal_background_mframe, self->order,
                self->nsamples);
    }

    sph_master_frame_delete(cal_background_mframe);
    cal_background_mframe = NULL;

    if (fitresult) {
        sph_master_frame_save_dfs(fitresult, self->fit_filename, self->inframes,
                NULL, self->inparams, SPH_IFS_TAG_CAL_BACKGROUND_FIT_CALIB,
                SPH_RECIPE_NAME_IFS_CAL_BACKGROUND, SPH_PIPELINE_NAME_IFS, NULL);
        sph_master_frame_delete(fitresult);
        fitresult = NULL;
    }

    cpl_ensure_code( cpl_errorstate_is_equal( prestate ),
                     cpl_error_get_code());

    cpl_propertylist_delete(rawpl);
    rawpl = NULL;
    EXIT: if (self->cal_background_framelist != NULL) {
        sph_framelist_delete(self->cal_background_framelist);
        self->cal_background_framelist = NULL;
    }
    if (self->cal_background_cube != NULL) {
        sph_cube_delete(self->cal_background_cube);
    }

    return (int) cpl_error_get_code();
}

/**@}*/
