/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_filemanager.h"
#include "sph_error.h"
#include "sph_ifs_distortion_map.h"
#include "sph_common_keywords.h"
#include "sph_ifs_tags.h"
#include "sph_ifs_keywords.h"
#include "sph_utils.h"
#include "sph_ifs_science.h"
#include "sph_keyword_manager.h"
#include <time.h>
#include "sph_distortion_model.h"
#include "sph_point_pattern.h"

#define SPH_IFS_DISTORTION_MAP_POINT_LIMIT 6

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ifs_distortion_map Distortion Map Recipe for IFS
 *
 * This module provides the algorithm inplementation for the creation of the
 * the distortion map for IFS.
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ifs_distortion_map.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
cpl_error_code sph_ifs_distortion_map_run(sph_ifs_distortion_map* self) 
{
    const cpl_frame* background_dark_frame
        = self->master_background_frame != NULL
        ? self->master_background_frame
        : self->master_dark_frame;
    sph_ifs_lenslet_model* lensmodel = NULL;
    sph_master_frame* result = NULL;
    cpl_propertylist* plist = NULL;
    sph_point_pattern* point_pattern = NULL;
    sph_distortion_model* distmap = NULL;
    cpl_image* pimage = NULL;
    double centx = -10000.0;
    double centy = -10000.0;
    sph_point_pattern* obs_pp = NULL;
    cpl_table        * dist_table = NULL;
    double rot = 0.0;
    double scale = 1.0;
    cpl_image* pp_image = NULL;
    cpl_frame* distframe_qc = NULL;
    double avg_centx, avg_centy;

    plist = sph_keyword_manager_load_properties(
        cpl_frame_get_filename(self->wave_calib_frame), 0);
    lensmodel = sph_ifs_lenslet_model_new_from_propertylist(plist);

    SPH_RAISE_CPL_RESET

    result = sph_ifs_science_reduce_and_collapse(self->raw_frameset, NULL,
        self->inparams, self->coll_alg, lensmodel, background_dark_frame,
        self->static_badpixel_frame, self->wave_calib_frame,
        self->preamp_flat, self->master_ifu_flat_frame, NULL,
        self->master_dff_long1_frame, self->master_dff_long2_frame,
        self->master_dff_long3_frame, self->master_dff_long4_frame,
        self->master_dff_longbb_frame, self->master_dff_short_frame,
        SPH_RECIPE_NAME_IFS_DISTORTION_MAP,
        SPH_IFS_TAG_DISTORTION_MAP_REDUCED);

    if (result) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Collapsed wavelength cube.");
        sph_master_frame_mask_tolerance(result, 0.0, 400000);
        pimage = sph_master_frame_extract_image(result, 1);
        cpl_image_fill_rejected(pimage, 0.0);
        cpl_image_accept_all(pimage);
        cpl_image_save(pimage, "collapsed_distortion_map.fits", CPL_TYPE_DOUBLE,
            NULL, CPL_IO_CREATE);
        if (!self->point_pattern_frame) {
            SPH_INFO_MSG("No point pattern provided. "
                "Cant proceed to create distortion map and will"
                " create point pattern from collapsed LDT instead.");
            point_pattern = sph_point_pattern_new_from_image(pimage, NULL,
                                                          self->threshold_sigma,
                                                             3.0, 3.0, 3.0, 0,
                                                             NULL);
            if (!point_pattern) {
                SPH_ERR("Could not create point pattern. Can't proceed.");
                return CPL_ERROR_ILLEGAL_INPUT;
            }

            sph_point_pattern_centralise_wrt_central_point(point_pattern);
            rot = lensmodel->bigre_rot + self->rotate;
            if (rot) {
                sph_point_pattern_rotate(point_pattern, 0.0, 0.0, rot);
                SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                    "Rotating output pattern by %f degree", rot);
            }
            if (cpl_propertylist_has(result->properties,
                SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE)) {
                scale = cpl_propertylist_get_double(result->properties,
                    SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE);
            scale = scale * lensmodel->bigre_scale;
            SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "Scaling output pattern by factor %f", scale);
            sph_point_pattern_scale_around(point_pattern, 0.0, 0.0, scale);
        } else
        SPH_ERROR_RAISE_WARNING(
            CPL_ERROR_DATA_NOT_FOUND,
            "Missing keyword %s. Not scaling pattern.", SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE);

        sph_point_pattern_save_dfs(point_pattern, self->pp_outfilename,
            self->inframes, NULL, self->inparams,
            SPH_IFS_TAG_POINT_PATTERN,
            SPH_RECIPE_NAME_IFS_DISTORTION_MAP, SPH_PIPELINE_NAME_IFS,
            plist);
        self->point_pattern_frame = sph_utils_extract_frame(self->inframes,
            SPH_IFS_TAG_POINT_PATTERN);
    }
    cpl_image_delete(pimage);
    pimage = NULL;

    if (point_pattern == NULL) {
        point_pattern = sph_point_pattern_new_from_table(
            self->point_pattern_frame, 1);
        if (point_pattern == NULL) {
            cpl_error_reset();
            point_pattern = sph_point_pattern_load_ascii(
                cpl_frame_get_filename(self->point_pattern_frame));
        }
    }
    if (!point_pattern) {
        SPH_ERR("Could not load point pattern. Cant proceed.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if (sph_point_pattern_get_size(point_pattern) < SPH_IFS_DISTORTION_MAP_POINT_LIMIT) {
        SPH_ERROR_RAISE_ERR(
            SPH_ERROR_GENERAL,
            "Point pattern has too few points. "
            "Only found %d but need at least %d. "
            "Cant proceed.", sph_point_pattern_get_size(point_pattern), SPH_IFS_DISTORTION_MAP_POINT_LIMIT);
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    sph_point_pattern_centralise_wrt_central_point(point_pattern);

    rot = (lensmodel->bigre_rot + self->rotate) * -1.0;
    if (rot) {
        sph_point_pattern_rotate(point_pattern, 0.0, 0.0, rot);
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
            "Rotating input pattern by %f degree", rot);
    }
    if (cpl_propertylist_has(result->properties,
        SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE)) {
        scale = cpl_propertylist_get_double(result->properties,
            SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE);
    scale = scale * lensmodel->bigre_scale;
    SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
        "Scaling input pattern by factor %f", 1.0/scale);
    sph_point_pattern_scale_around(point_pattern, 0.0, 0.0,
        1.0 / scale);
    } else {
        SPH_ERROR_RAISE_WARNING(
            CPL_ERROR_DATA_NOT_FOUND,
            "Missing keyword %s. Not scaling pattern.", SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE);
    }
    sph_point_pattern_add_offset(point_pattern,146.0,146.0);
    pp_image = sph_point_pattern_create_image(point_pattern,291,291,1.0);
    cpl_image_save(pp_image, "input_point_pattern_map.fits", CPL_TYPE_DOUBLE,
        NULL, CPL_IO_CREATE);
    cpl_image_delete( pp_image ); pp_image = NULL;

    sph_point_pattern_save_ascii(point_pattern, "testpp.txt");

    cpl_propertylist_delete(plist);
    plist = NULL;
    plist = cpl_propertylist_new();

    distmap = sph_distortion_model_from_grid_image_general(result,
        point_pattern, self->polyfit_order, self->threshold_sigma,
        self->threshold_length, plist, 0.0, 0.0, &centx, &centy,
        &avg_centx, &avg_centy, &obs_pp, &dist_table, 0);
    if (cpl_propertylist_has(result->properties,
        SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE)) {
        scale = cpl_propertylist_get_double(result->properties,
            SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE);
    SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
        "Scaling the distortion map units from pixels to "
        "microns, using a factor of %f microns per pixel. ", scale);
    sph_distortion_model_scale_units(distmap, scale);
    } else {
        scale = 1.0;
    }

    if ( obs_pp ) {
        sph_point_pattern* obs_pp1 = sph_point_pattern_duplicate(obs_pp);

        sph_point_pattern_centralise_wrt_central_point(obs_pp);
        sph_point_pattern_add_offset(obs_pp,146.0,146.0);
        pp_image = sph_point_pattern_create_image(obs_pp,291,291,1.0);
        cpl_image_save(pp_image, "output_point_pattern_map.fits", CPL_TYPE_DOUBLE,
                       NULL, CPL_IO_CREATE);
        sph_point_pattern_save_ascii( obs_pp, "output_point_pattern.txt");
        cpl_image_delete( pp_image ); pp_image = NULL;

#if 0
        sph_point_pattern_draw_save(obs_pp1,  result->image, NULL, NULL,
                                    NULL, NULL, 0.0, 0.0, NULL, 0.0, 0.0, 5.0,
                                    "output_point_pattern_map.fits");
#endif
        sph_point_pattern_delete(obs_pp1);
    }

    if (distmap) {
        cpl_propertylist_update_int(plist,
            SPH_IFS_KEYWORD_DISTMAP_NPOINTS_OBS,
            sph_point_pattern_get_size(obs_pp));

        cpl_propertylist_update_double(plist,
            SPH_IFS_KEYWORD_DISTMAP_OPTICAL_AXIS_X, centx);
        cpl_propertylist_update_double(plist,
            SPH_IFS_KEYWORD_DISTMAP_OPTICAL_AXIS_Y, centy);

        sph_distortion_model_save_dfs(distmap,
            self->distortion_map_filename, self->inframes,
            cpl_frameset_get_first(self->raw_frameset), self->inparams,
            SPH_IFS_TAG_DISTORTION_MAP_CALIB,
            SPH_RECIPE_NAME_IFS_DISTORTION_MAP, SPH_PIPELINE_NAME_IFS,
            plist, cpl_image_get_size_x(result->image),
            cpl_image_get_size_y(result->image), scale, 1, NULL, NULL, NULL);
        sph_distortion_model_delete(distmap);
        distmap = NULL;
    } else {
        SPH_ERR("Could not create product frame.");
    }
    if (cpl_error_get_code() == CPL_ERROR_NONE && self->qc_consist_check) {
        distframe_qc = cpl_frameset_find(self->inframes,
            SPH_IFS_TAG_DISTORTION_MAP_CALIB);
        sph_master_frame_delete(result);
        result = NULL;
        if (distframe_qc) {
            SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "Running consistency check... "
                "distortion map is applied to inputs. ");
            result = sph_ifs_science_reduce_and_collapse(self->raw_frameset,
                NULL, self->inparams, self->coll_alg, lensmodel,
                background_dark_frame, self->static_badpixel_frame,
                self->wave_calib_frame, self->preamp_flat,
                self->master_ifu_flat_frame, distframe_qc,
                self->master_dff_long1_frame,
                self->master_dff_long2_frame,
                self->master_dff_long3_frame,
                self->master_dff_long4_frame,
                self->master_dff_longbb_frame,
                self->master_dff_short_frame,
                SPH_RECIPE_NAME_IFS_DISTORTION_MAP,
                SPH_IFS_TAG_DISTORTION_MAP_REDUCED);
            if (result) {
                sph_master_frame_save(result, "qc_distmap.fits", NULL);

                distmap = sph_distortion_model_from_grid_image(
                    result, point_pattern, self->polyfit_order,
                    self->threshold_sigma,
                    self->threshold_length,
                    plist, 0.0, 0.0, &centx, &centy, &avg_centx, &avg_centy,
                    &obs_pp, &dist_table);
                sph_distortion_model_scale_units(distmap, scale);
                sph_distortion_model_save_dfs(distmap,
                    "qc_distmap_residual.fits", self->inframes,
                    cpl_frameset_get_first(self->raw_frameset),
                    self->inparams, SPH_IFS_TAG_DISTORTION_MAP_CALIB,
                    SPH_RECIPE_NAME_IFS_DISTORTION_MAP,
                    SPH_PIPELINE_NAME_IFS, plist,
                    cpl_image_get_size_x(result->image),
                    cpl_image_get_size_y(result->image), scale, 1, NULL, NULL, NULL);
                sph_distortion_model_delete(distmap);
                distmap = NULL;

                sph_master_frame_delete(result);
                result = NULL;
            }
        }
    }

    sph_master_frame_delete(result);
    result = NULL;
    sph_point_pattern_delete(point_pattern);
    point_pattern = NULL;
    }

    if (self->qc_consist_check == 0)
        sph_filemanager_clean();
    cpl_propertylist_delete(plist);
    plist = NULL;
    sph_ifs_lenslet_model_delete(lensmodel);
    lensmodel = NULL;
    sph_point_pattern_delete(obs_pp);
    obs_pp = NULL;
    cpl_table_delete(dist_table);
    return cpl_error_get_code();
}

/**@}*/
