/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_ifs_wave_calib.h"
#include "sph_master_frame.h"
#include "sph_spectral_region.h"
#include "sph_ldt.h"
#include "sph_dataset_stats.h"
#include "sph_dataset.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_fits.h"
#include "sph_fitting.h"
#include "sph_ifs_keywords.h"
#include "sph_ifs_subtract_dark_scaled.h"
#include "sph_ifs_tags.h"
#include "sph_framecombination.h"
#include "sph_extract_angles.h"

#include <cpl.h>
#include <math.h>
#include <string.h>
#include <assert.h>

/*---- Forward declaration of private functions*/
static inline sph_error_code save_as_cube(sph_ifs_wave_calib * self,
        sph_pixel_description_table *pdt, sph_ifs_lenslet_model * model,
        cpl_propertylist * plist, cpl_frame * template,  const char * fname,
        const char * tag, sph_master_frame* mframe, const double ang);

static inline cpl_boolean are_all_raws_ifs(const cpl_frameset * fset);

static inline const char *
safe_extract_from_plist(const cpl_propertylist * plist, const char * card_name);

static const double SPH_IFS_ASTROM_OFFSETANGLE_1 = 0.0;
static const double SPH_IFS_ASTROM_OFFSETANGLE_2 = 0.0;

static const cpl_size RESOLVING_POWER_EDGE_TRIM = 60 ;
static const cpl_size SPH_IFS_PRISM_KW_LENGTH_YJ = 7;
static const cpl_size SPH_IFS_PRISM_KW_LENGTH_YJH = 8;

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IFS_WAVE_CALIB_GENERAL;

#define SPH_IFS_WAVE_CALIB_LAMBTOL 3 * model->dispersion
#define SPH_IFS_WAVE_CALIB_LAMBTOL2 2 * model->dispersion
#define SPH_IFS_WAVE_CALIB_LAMBTOL1 1 * model->dispersion

#define SIGMA_TO_FWHM_FACTOR 2.0 * sqrt(2.0 * CPL_MATH_LN2)

static
int sph_ifs_wave_calib_set_calib_wavelengths_from_header(sph_ifs_wave_calib* self) 
{
    cpl_vector* lambdas = NULL;

    cpl_propertylist* plist = 
        sph_keyword_manager_load_properties(cpl_frame_get_filename( self->specpos_frame), 0 );
    if ( !cpl_propertylist_has( plist, SPH_IFS_KEYWORD_PRISM_MODE) ) {
        SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL,
            "Could not find keywords needed to distinguish PRISM settings."
            "Either provide a frame with that information or switch off autmaatic wavelenngth"
            "association.");
        return SPH_ERROR_GENERAL;
    }
    if ( strcmp( cpl_propertylist_get_string(plist, SPH_IFS_KEYWORD_PRISM_MODE), 
                  SPH_IFS_KEYWORD_VALUE_PRISM_MODE_J) == 0 ) {
        
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO, 
            "Prism Y-J mode detected setting wavelengths automatically");
        self->number_lines = 3;
        self->calib_wavelengths = cpl_vector_new(self->number_lines);
        
        lambdas = self->calib_wavelengths;
        cpl_vector_set(lambdas, 0, 0.98772);
        cpl_vector_set(lambdas, 1, 1.12371);
        cpl_vector_set(lambdas, 2, 1.30937);
    }
    if ( strcmp( cpl_propertylist_get_string(plist, SPH_IFS_KEYWORD_PRISM_MODE), 
                  SPH_IFS_KEYWORD_VALUE_PRISM_MODE_JH) == 0 ) {
        
        SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO, 
            "Prism Y-H mode detected setting wavelengths automatically");
        self->number_lines = 4;
        self->calib_wavelengths = cpl_vector_new(self->number_lines);

        lambdas = self->calib_wavelengths;
        cpl_vector_set(lambdas, 0, 0.98772);
        cpl_vector_set(lambdas, 1, 1.12371);
        cpl_vector_set(lambdas, 2, 1.30937);
        cpl_vector_set(lambdas, 3, 1.5451);
    }
    return CPL_ERROR_NONE;
}

static
int sph_ifs_wave_calib_set_calib_wavelengths(sph_ifs_wave_calib* self) {
    cpl_vector* lambdas = NULL;

    if (self->number_lines == 0) {
        return sph_ifs_wave_calib_set_calib_wavelengths_from_header(self);
    }

    if (self->number_lines < 2) {
        sph_error_raise(SPH_IFS_WAVE_CALIB_GENERAL, __FILE__, __func__,
                __LINE__, SPH_ERROR_ERROR,
                "Need to have at least 2 calibration wavelengths! "
                        "But number of lines was %d", self->number_lines);
        goto ERROR_EXIT;
    }

    self->calib_wavelengths = cpl_vector_new(self->number_lines);
    lambdas = self->calib_wavelengths;

    if (self->wavelength_line1 > 0.0) {
        cpl_vector_set(lambdas, 0, self->wavelength_line1);
    } else {
        sph_error_raise(SPH_IFS_WAVE_CALIB_GENERAL, __FILE__, __func__,
                __LINE__, SPH_ERROR_ERROR,
                "Need to have at least 2 calibration wavelengths! "
                        "But wavelength of line 1 was %lf",
                self->wavelength_line1);
        goto ERROR_EXIT;
    }
    if (self->wavelength_line2 > 0.0) {
        cpl_vector_set(lambdas, 1, self->wavelength_line2);
    } else {
        sph_error_raise(SPH_IFS_WAVE_CALIB_GENERAL, __FILE__, __func__,
                __LINE__, SPH_ERROR_ERROR,
                "Need to have at least 2 valid calibration wavelengths! "
                        "But wavelength of line 2 was %lf",
                self->wavelength_line1);
        goto ERROR_EXIT;
    }
    if (self->wavelength_line3 > 0.0 && self->number_lines > 2) {
        cpl_vector_set(lambdas, 2, self->wavelength_line3);
    }
    if (self->wavelength_line4 > 0.0 && self->number_lines > 3) {
        cpl_vector_set(lambdas, 3, self->wavelength_line4);
    }
    if (self->wavelength_line5 > 0.0 && self->number_lines > 4) {
        cpl_vector_set(lambdas, 4, self->wavelength_line5);
    }

    return CPL_ERROR_NONE;

    ERROR_EXIT: if (self->calib_wavelengths) {
        cpl_vector_delete(self->calib_wavelengths);
        self->calib_wavelengths = NULL;
    }
    return SPH_IFS_WAVE_CALIB_GENERAL;

}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ifs_wave_calib_run Run the Instrument Flat Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * instrument flat recipe for IFS.
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ifs_wave_calib.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list
 @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static sph_pixel_description_table*
sph_ifs_wave_calib_process_prism(sph_ifs_wave_calib* self,
        cpl_frameset* rawframes, sph_pixel_description_table* pdt,
        sph_ifs_lenslet_model* model, sph_master_frame* iff,
        sph_master_frame* dark, cpl_mask* badspecmask, cpl_propertylist* pl) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_master_frame* mframe = NULL;
    sph_spectral_region* reg = NULL;
    sph_spectrum* spec = NULL;
    cpl_vector* wavs = NULL;
    cpl_vector* diffs = NULL;
    cpl_vector* minw = NULL;
    cpl_vector* maxw = NULL;
    cpl_vector* deltaw = NULL;
    cpl_vector* resolving_power = NULL;
    cpl_vector* centw = NULL;
    cpl_image* dispim = NULL;
    double dispersion = 0.0;
    double flux = 0.0;
    double flux_stdev = 0.0;
    int sid = 0;
    int badcount = 0;
    int badfitcount = 0;
    int badf = 0;
    int badf1 = 0;
    int badf2 = 0;
    int goodcount = 0;

    cpl_mask* tmpmask = NULL;
    cpl_image* specidimage = NULL;
    double exptime = 1.0;

    if (!pdt || !model) {
        SPH_ERR("Absolutely need the PDT to proceed");
        return NULL;
    }
    mframe = sph_framecombination_master_frame_from_cpl_frameset(rawframes,
            self->coll_alg, self->framecomb_parameterlist);
    if (!mframe) {
        SPH_ERR("Could not collapse raws.");
        return NULL;
    }
    if (dark) {
        rerr = sph_ifs_subtract_dark_scaled(mframe, dark);
        if (rerr != CPL_ERROR_NONE) {
            SPH_INFO_MSG("Could not subtract dark.");
            SPH_RAISE_CPL_RESET;
        }
    }
    if (iff) {
        rerr = sph_master_frame_divide_master_frame(mframe, iff);
        if (rerr != CPL_ERROR_NONE) {
            SPH_ERR("Could not divide flat.");
            sph_master_frame_delete(mframe);
            mframe = NULL;
            return NULL;
        }
    }

    flux = sph_master_frame_get_mean(mframe, &flux_stdev);
    exptime = sph_utils_get_exposure_time(cpl_frameset_get_first(rawframes));

    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX,
            flux / exptime);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_FLAT_LAMP_FLUX_STDEV,
            flux_stdev / exptime);

    maxw = cpl_vector_new(pdt->nregions);
    minw = cpl_vector_new(pdt->nregions);
    centw = cpl_vector_new(pdt->nregions);
    deltaw = cpl_vector_new(pdt->nregions);
    resolving_power = cpl_vector_new(pdt->nregions);

    cpl_vector_fill(minw, 0.0);
    cpl_vector_fill(maxw, 0.0);
    cpl_vector_fill(centw, 0.0);
    cpl_vector_fill(deltaw, 0.0);
    cpl_vector_fill(resolving_power, 0.0);
    sph_pixel_description_table_setup_regions(pdt);

    specidimage = sph_pixel_description_table_get_specid_image(pdt);
    for (sid = 0; sid < pdt->nregions; ++sid) {
        reg = sph_pixel_description_table_get_region(pdt, sid + 1);
        if (reg) {
            /*sph_error_raise(SPH_IFS_WAVE_CALIB_GENERAL, __FILE__, __func__,
                    __LINE__, SPH_ERROR_INFO,
                    "Processing spectrum region %d/%d, specid %d, lensid %d",
                    sid + 1, pdt->nregions, reg->specid, reg->lensid);*/
            spec = sph_pixel_description_table_extract_spectrum_mframe(pdt,
                    reg->specid, mframe, NULL);
            if (spec) {
                wavs = sph_spectrum_fit_wavs(spec, self->calib_wavelengths,
                        self->polyfit_order, self->fit_window_size, &diffs,
                        NULL, NULL);
                if (wavs && diffs) {
                    if (cpl_error_get_code() == CPL_ERROR_SINGULAR_MATRIX) {
                        badfitcount++;
                        SPH_RAISE_CPL_RESET;
                        if (badspecmask) {
                            tmpmask = cpl_mask_threshold_image_create(
                                    specidimage, spec->specid - 0.1,
                                    spec->specid + 0.1);
                            cpl_mask_or(badspecmask, tmpmask);
                            cpl_mask_delete(tmpmask);
                            tmpmask = NULL;
                        }
                    } else if (fabs(
                            cpl_vector_get(wavs, 0)
                                    - model->minlambda) > SPH_IFS_WAVE_CALIB_LAMBTOL ||
                                    fabs(cpl_vector_get(wavs,cpl_vector_get_size(wavs)-1) - model->maxlambda) > SPH_IFS_WAVE_CALIB_LAMBTOL) {
                        cpl_vector_set(minw, goodcount + badf,
                                cpl_vector_get(spec->wavelengths, 0));
                        cpl_vector_set(
                                maxw,
                                goodcount + badf,
                                cpl_vector_get(
                                        spec->wavelengths,
                                        cpl_vector_get_size(spec->wavelengths)
                                                - 1));
                        cpl_vector_set(
                                centw,
                                goodcount + badf,
                                cpl_vector_get(
                                        spec->wavelengths,
                                        (cpl_vector_get_size(spec->wavelengths)
                                                - 1) / 2));
                        cpl_vector_set(resolving_power, goodcount + badf,
                                sph_spectrum_get_resolving_power(spec));
                        badf++;
                        sph_pixel_description_table_set_spectra_wavelengths(pdt,
                                reg->specid, wavs, 0); //temp fix
                        if (badspecmask) {
                            tmpmask = cpl_mask_threshold_image_create(
                                    specidimage, spec->specid - 0.1,
                                    spec->specid + 0.1);
                            cpl_mask_or(badspecmask, tmpmask);
                            cpl_mask_delete(tmpmask);
                            tmpmask = NULL;
                        }
                    } else {
                        SPH_ERROR_RAISE_INFO(
                                CPL_ERROR_CONTINUE,
                                "Found good spectrum %f, %f", cpl_vector_get(wavs, 0), cpl_vector_get(wavs, cpl_vector_get_size(wavs)-1));
                        sph_pixel_description_table_set_spectra_wavelengths(pdt,
                                reg->specid, wavs, self->no_spline_interpol);
                        cpl_vector_set(minw, goodcount + badf,
                                cpl_vector_get(wavs, 0));
                        cpl_vector_set(
                                maxw,
                                goodcount + badf,
                                cpl_vector_get(wavs,
                                        cpl_vector_get_size(wavs) - 1));
                        cpl_vector_set(
                                centw,
                                goodcount + badf,
                                cpl_vector_get(wavs,
                                        (cpl_vector_get_size(wavs) - 1) / 2));
                        cpl_vector_set(resolving_power, goodcount + badf,
                                sph_spectrum_get_resolving_power(spec));
                        goodcount++;
                    }
                    if (fabs(cpl_vector_get(wavs, 0) - model->minlambda)
                            > SPH_IFS_WAVE_CALIB_LAMBTOL / 3.0
                            || fabs(
                                    cpl_vector_get(wavs,
                                            cpl_vector_get_size(wavs) - 1)
                                            - model->maxlambda)
                                    > SPH_IFS_WAVE_CALIB_LAMBTOL / 3.0) {
                        badf1++;
                    }
                    if (fabs(cpl_vector_get(wavs, 0) - model->minlambda)
                            > 2.0 * SPH_IFS_WAVE_CALIB_LAMBTOL / 3.0
                            || fabs(
                                    cpl_vector_get(wavs,
                                            cpl_vector_get_size(wavs) - 1)
                                            - model->maxlambda)
                                    > 2.0 * SPH_IFS_WAVE_CALIB_LAMBTOL / 3.0) {
                        badf2++;
                    }
                } else {
                    badcount++;
                    if (cpl_error_get_code() == CPL_ERROR_SINGULAR_MATRIX) {
                        SPH_RAISE_CPL_RESET;
                    }
                    cpl_vector_fill(spec->bads, 1.0);
                    cpl_vector_fill(spec->ncomb, 0.0);
                    cpl_vector_fill(spec->rms, 0.0);
                    cpl_vector_fill(spec->wavelengths, 0.0);
                    cpl_vector_fill(spec->spec, 0.0);
                    sph_pixel_description_table_set_spectra_wavelengths(pdt,
                            reg->specid, spec->wavelengths,
                            self->no_spline_interpol);
                    if (cpl_error_get_code() == CPL_ERROR_INCOMPATIBLE_INPUT) {
                        SPH_RAISE_CPL_RESET;
                    }
                    if (badspecmask) {
                        tmpmask = cpl_mask_threshold_image_create(specidimage,
                                reg->specid - 0.1, reg->specid + 0.1);
                        cpl_mask_or(badspecmask, tmpmask);
                        cpl_mask_delete(tmpmask);
                        tmpmask = NULL;
                    }
                }
                cpl_vector_delete(wavs);
                wavs = NULL;
                cpl_vector_delete(diffs);
                diffs = NULL;
                sph_spectrum_delete(spec);
                spec = NULL;
            } else {
                cpl_error_reset();
            }
        }
    }
    dispim = sph_pixel_description_table_get_dispimage(pdt);
    dispersion = cpl_image_get_median(dispim);
    cpl_image_delete(dispim);
    dispim = NULL;
    cpl_vector_set_size(minw, goodcount + badf);
    cpl_vector_set_size(maxw, goodcount + badf);
    cpl_vector_set_size(centw, goodcount + badf);
    cpl_vector_set_size(resolving_power, goodcount + badf);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_QC_WAVE_NSINGFITS,
            badfitcount);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_QC_WAVE_NOUTOFSPEC,
            badf);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_QC_WAVE_N2DISP, badf2);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_QC_WAVE_NDISP, badf1);
    cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_QC_WAVE_BADSPEC,
            badcount);
    cpl_propertylist_append_double(pl,
            SPH_COMMON_KEYWORD_MEDIAN_RESOLVING_POWER,
            cpl_vector_get_median(resolving_power));
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_MEDIAN_DISPERSION,
            dispersion);
    SPH_INFO_MSG("I just put this value into ESO QC MEDIAN DISPERSION @ L451:");
    printf("%.10f\n", dispersion);
//    getchar();
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_MEDIAN_MINWAVEL,
            cpl_vector_get_median(minw) - 0.5 * dispersion);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_MEDIAN_MAXWAVEL,
            cpl_vector_get_median(maxw) + 0.5 * dispersion);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_MEDIAN_CENTWAVEL,
            cpl_vector_get_median(centw));

    cpl_vector_delete(maxw);
    maxw = NULL;
    cpl_vector_delete(minw);
    minw = NULL;
    cpl_vector_delete(centw);
    centw = NULL;
    cpl_vector_delete(deltaw);
    deltaw = NULL;
    cpl_vector_delete(resolving_power);
    resolving_power = NULL;
    cpl_image_delete(specidimage);
    specidimage = NULL;
    sph_error_raise(SPH_IFS_WAVE_CALIB_GENERAL, __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO,
            "Had %d good spectra, with %d inaccurate by more than %f, "
                    "%d inaccurate by more than %f. "
                    "Had %d bad spectra, out of spec fits that were "
                    "inaccurate by more than %f: %d, "
                    "no fit: %d out of %d",
            pdt->nregions - badcount,
            badf1, SPH_IFS_WAVE_CALIB_LAMBTOL1,
            badf2, SPH_IFS_WAVE_CALIB_LAMBTOL2,
            badcount, SPH_IFS_WAVE_CALIB_LAMBTOL, badf,
            badfitcount, pdt->nregions);
    sph_master_frame_delete(mframe);
    mframe = NULL;
    return pdt;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list
 @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ifs_wave_calib_run(sph_ifs_wave_calib* self) {
    const cpl_frame* background_dark_frame
        = self->master_background_frame != NULL
        ? self->master_background_frame
        : self->master_dark_frame;
    sph_pixel_description_table* specpos = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_master_frame* iff = NULL;
    sph_master_frame* dark = NULL;
    cpl_mask* bmask = NULL;
    cpl_image* badim = NULL;
    sph_ifs_lenslet_model* model = NULL;
    cpl_propertylist* plist = NULL;
    cpl_propertylist* pli   = NULL;
    cpl_propertylist* extpl = NULL;
    const char* filename;

    sph_error_code rerr = CPL_ERROR_NONE;

    double ang = 0;
    {

        cpl_frame * frame =
        		cpl_frameset_get_position(self->rawframes, 0);
        cpl_vector * parangs =
        		sph_extract_angles_from_cube( frame,
        				SPH_IFS_ASTROM_OFFSETANGLE_1,
						SPH_IFS_ASTROM_OFFSETANGLE_2);
        if (!parangs){
            SPH_ERROR_RAISE_WARNING(SPH_ERROR_GENERAL,"Could not determine parallactic angles!");
            cpl_error_reset();
            cpl_size nplanes = sph_fits_get_nplanes(cpl_frame_get_filename(frame), 0);
            parangs = cpl_vector_new(nplanes);
            for (cpl_size ii = 0; ii < nplanes; ++ii) {
                cpl_vector_set(parangs, ii, 0.0);
            }
        }
        ang = cpl_vector_get(parangs, cpl_vector_get_size(parangs) - 1);
        cpl_vector_delete(parangs);
    }

    /*------------------------------------------------------------------
     -  Process calibration wavelengths
     --------------------------------------------------------------------*/
    rerr = sph_ifs_wave_calib_set_calib_wavelengths(self);
    if (rerr != CPL_ERROR_NONE) {
        sph_error_raise(SPH_IFS_WAVE_CALIB_GENERAL, __FILE__, __func__,
                __LINE__, SPH_ERROR_ERROR,
                "Could not process calibration wavelengths. "
                        "Will not process frames.");

        return rerr;
    }
    if (sph_utils_check_exposure_times_are_same(self->rawframes) == 0) {
        SPH_ERR("The exposure times of the input frames are not the same."
        "Can not process these.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    if (background_dark_frame) {
        dark = sph_master_frame_load_(background_dark_frame, 0);
    }
    if (self->master_iff_frame) {
        iff = sph_master_frame_load_(self->master_iff_frame, 0);
    }

    rerr = cpl_error_get_code();

    if(!rerr && !are_all_raws_ifs(self->rawframes)){
        SPH_ERR("The raw files are not IFS");
    	rerr = CPL_ERROR_ILLEGAL_INPUT;
    }

    if(rerr){
        SPH_ERR("Input file I/O failed.");
        sph_master_frame_delete(dark);
        sph_master_frame_delete(iff);
    	return rerr;
    }


    // read template property list of first frame to copy singular keys later
    filename =
        cpl_frame_get_filename(cpl_frameset_get_first_const(self->rawframes));
    pli = cpl_propertylist_load(filename, 0);

    SPH_INFO_MSG("Loading specpos frame...");
    assert( self->specpos_frame != NULL );
    // Read in the spectra positions from the calibration recipe
    filename = cpl_frame_get_filename(self->specpos_frame);
    specpos = sph_pixel_description_table_load(filename);
    plist = sph_keyword_manager_load_properties(filename, 0);
    // Generate the initial model from the input spectra positions
    model = sph_ifs_lenslet_model_new_from_propertylist(plist);
    cpl_propertylist_delete(plist);
    plist = NULL;
    /*------------------------------------------------------------------
     -  Now process all sets of prism frames separately
     --------------------------------------------------------------------*/

    plist = cpl_propertylist_new();
    bmask = cpl_mask_new(specpos->nx, specpos->ny);
    // Use the observed wavelength calib frames to adjust models
    pdt = sph_ifs_wave_calib_process_prism(self, self->rawframes, specpos,
            model, iff, dark, bmask, plist);
    if (pdt) {
        sph_utils_simple_copy_singular(pli,plist);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Created final PDT. Measuring dispersion...");
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Now saving...");
        if (sph_pixel_description_table_save_dfs(pdt, self->wave_calib_filename,
                model, self->inframes, NULL, self->inparams,
                SPH_IFS_TAG_WAVE_CALIB_CALIB, SPH_RECIPE_NAME_IFS_WAVE_CALIB,
                SPH_PIPELINE_NAME_IFS, plist) != CPL_ERROR_NONE) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO,
                    "Could not write main product.");
        } else {
            if (bmask) {
                badim = cpl_image_new_from_mask(bmask);
                extpl=cpl_propertylist_new();
                cpl_propertylist_update_string(extpl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_PDT_BADIMAGE_EXTNAME);

                cpl_image_save(badim, self->wave_calib_filename,
                        CPL_BPP_16_UNSIGNED, extpl, CPL_IO_EXTEND);
                cpl_image_delete(badim);
                badim = NULL;
                cpl_propertylist_delete(extpl);
                extpl=NULL;
            }
        }
    }
    else{
    	rerr = cpl_error_get_code();
    }

    /*Save wavelength calibration cube*/
    if(pdt && self->save_addprod && !rerr){
        SPH_INFO_MSG("Working on wavelength calibration cube...");

        SPH_INFO_MSG("  Creating master frame");
        sph_master_frame* mframe =
                sph_framecombination_master_frame_from_cpl_frameset(self->rawframes,
                        self->coll_alg, self->framecomb_parameterlist);
        SPH_INFO_MSG("  Getting cube name");
        char  * cube_fname = sph_filemanager_new_filename_from_base(self->wave_calib_filename,
                "cube");

        SPH_INFO_MSG("  Saving cube as:");
        printf("%s\n", cube_fname);
        rerr = save_as_cube(
                self,  // sph_ifs_wave_calib
                pdt,   // sph_pixel_description_table
                model, // sph_ifs_lenslet_model
                plist, // cpl_propertylist
                NULL,  // cpl_frame template
                cube_fname,  // filename
                SPH_IFS_TAG_WAVE_CALIB_CUBE,  // tag (char)
                mframe,  // sph_master_frame
                ang  // ang (double)
                );

        // PIPE-7692 - Insert a call here to compute actual spectral resolution
        // as per Wolfgang Hummel's instructions
        // Use the self->inframes as the data source (these are what end up in the
        // cube product)

        // We need to do this by re-opening the cube here, analysing it, and adding
        // in new header keywords
        // It would be better to be able to run this during the generation of the cube
        // to prevent further IO penalties; however, the functions that do this write
        // (save_to_cube --> sph_ldt_save_cube) are called elsewhere, so we shouldn't
        // mangle those

        cpl_msg_info(cpl_func, "Re-opening cube for QC dispersion calculation");

        // Need to re-open the cube
        cpl_propertylist * sdp_plist = cpl_propertylist_load(cube_fname, 0);
        cpl_propertylist * new_plist = cpl_propertylist_new();
        int xint = cpl_propertylist_get_int(sdp_plist, "NAXIS1");
        int yint = cpl_propertylist_get_int(sdp_plist, "NAXIS2");
        int zint = cpl_propertylist_get_int(sdp_plist, "NAXIS3");
        cpl_size xx = xint;
        cpl_size yy = yint;
        cpl_size zz = zint;
        cpl_imagelist * sdp_product_imgs = cpl_imagelist_load_window(
                cube_fname, CPL_TYPE_DOUBLE, 0,
                RESOLVING_POWER_EDGE_TRIM + 1 , RESOLVING_POWER_EDGE_TRIM + 1 ,
                xx - RESOLVING_POWER_EDGE_TRIM , yy - RESOLVING_POWER_EDGE_TRIM );
        cpl_size no_images = cpl_imagelist_get_size(sdp_product_imgs);

        // Generate a median collapse spectrum, by collapsing in all spatial dimensions
        // excluding 60 pix around the FOV edge
        cpl_vector * median_collapse_spec = cpl_vector_new(zz);
        for (cpl_size i = 0; i < no_images; i++) {
            // Grab the image
            // (Don't need to dealloc these, will be handled on imagelist delete)
            cpl_image * curr_img = cpl_imagelist_get(sdp_product_imgs, i);
            // This, however, will need a cpl_mask_delete when we're done
            cpl_mask * curr_bpm = cpl_mask_load_window( // No imagelist equiv. for masks
                    cube_fname, i, 1,
                    RESOLVING_POWER_EDGE_TRIM + 1, RESOLVING_POWER_EDGE_TRIM + 1,
                    xx - RESOLVING_POWER_EDGE_TRIM, yy - RESOLVING_POWER_EDGE_TRIM);
            cpl_mask * old_bpm = cpl_image_set_bpm(curr_img, curr_bpm);
            cpl_mask_delete(old_bpm);

            // Compute the single-pixel median
            // NOTE: We can't do this with cpl_image_collapse_median_create in one direction,
            // then the other. This isn't mathematically valid.
            // The correct function is cpl_image_get_median_window
            // Remember that pixel indexing starts at *1*
            double sigma;
            double this_median = cpl_image_get_median(curr_img);
//            printf("This image had a median of %f\n", this_median);
            cpl_vector_set(median_collapse_spec, i, this_median) ;

        }

        // Fit each arc line (3 for YJ, 4 for YJH)
        // We can extract the calib wavelengths automatically from the 'self' sph_ifs_wave_calib object

        int no_wavl = cpl_propertylist_get_int(sdp_plist, "NAXIS3");
        double wavl0 = cpl_propertylist_get_double(sdp_plist, "CRVAL3");
        cpl_vector * sdp_wavls = cpl_vector_new(no_wavl);
        double wavl_delta = cpl_propertylist_get_double(sdp_plist, "CD3_3");
        for (int i=0; i < no_wavl; i++) {
            cpl_vector_set(sdp_wavls, i, wavl0 + (i * wavl_delta));
        }

        cpl_vector * fitted_wavls = cpl_vector_new(1);
        cpl_vector * good_respow = cpl_vector_new(1);
        cpl_size no_of_good_wavls = 0;
        // For each arc line (3 in YJ, 4 in YJH):
        for (int i=0; i < cpl_vector_get_size(self->calib_wavelengths); i++) {
            // Find the fitting area bounds
            cpl_size start, stop;
            if (i == 0) {
                start = 0;
            } else {
                start = cpl_vector_find(sdp_wavls,
                                        (cpl_vector_get(self->calib_wavelengths, i-1) +
                                         cpl_vector_get(self->calib_wavelengths, i)) / 2.0);
            }
            if (i == cpl_vector_get_size(self->calib_wavelengths) - 1) {
                stop = zint - 1 ;
            } else {
                stop = cpl_vector_find(sdp_wavls,
                                        (cpl_vector_get(self->calib_wavelengths, i) +
                                         cpl_vector_get(self->calib_wavelengths, i+1)) / 2.0);
            }

            // Fit the line
            cpl_vector * xpts = cpl_vector_extract(sdp_wavls, start, stop, 1);
            cpl_vector * ypts = cpl_vector_extract(median_collapse_spec, start, stop, 1);
            double x0 = cpl_vector_get(self->calib_wavelengths, i), sigma = 0.01, area = 10.0, offset = cpl_vector_get(ypts, 0);
            cpl_error_code fit_error_code = cpl_vector_fit_gaussian(
                xpts, NULL, // no sigmas
                ypts, NULL, // no sigmas
                CPL_FIT_ALL,
//                CPL_FIT_CENTROID | CPL_FIT_STDEV | CPL_FIT_AREA,
                &x0, &sigma, &area, &offset,
                NULL, NULL, NULL // mse, red_chisq, covariance
            );
            if (fit_error_code != CPL_ERROR_NONE) {
                SPH_INFO_MSG("Unable to fit laser line!");
                x0 = -999.0;
                sigma = -999.0;
                area = -999.0;
                offset = -999.0;
            } else {
                cpl_msg_info(cpl_func, "Laser line Gaussian fit results: x0 = %f, sigma = %f, area = %f, offset = %f",
                             x0, sigma, area, offset);
                // Store the wavelength value of the line
                cpl_vector_set(fitted_wavls, no_of_good_wavls, x0);
                no_of_good_wavls++;
                cpl_vector_set_size(fitted_wavls, no_of_good_wavls + 1);
            }
            cpl_vector_delete(xpts);
            cpl_vector_delete(ypts);

            // Store the integrated flux in each line as QC.FLUX.LASERn
            char * kw_lasern = cpl_sprintf("%s%d", SPH_IFS_KEYWORD_LASER_LINE_FLUX, i+1);
            cpl_propertylist_append_double(new_plist, kw_lasern, area);

            // Store the wavelength of the laser.
            kw_lasern = cpl_sprintf("%s%d", SPH_IFS_KEYWORD_LASER_WAVELENGTH, i+1);
            cpl_propertylist_append_double(new_plist, kw_lasern, x0);

            // Store the resolving power:
            // lambda0/delta_lambda
            // lambda0 = nominal wavelength of laser lamp
            // delta_lambda = Gaussian fit FWHM
            // as QC.RESPOW.LASERn
            kw_lasern = cpl_sprintf("%s%d", SPH_IFS_KEYWORD_LASER_RESOVLING_POWER, i+1);
            if (sigma <= 0.0) {
                cpl_propertylist_append_double(new_plist, kw_lasern,
                                              -999.0);
            } else {
                cpl_propertylist_append_double(new_plist, kw_lasern,
                                               cpl_vector_get(self->calib_wavelengths, i) /
                                               (SIGMA_TO_FWHM_FACTOR * sigma));
                cpl_vector_set(good_respow, no_of_good_wavls - 1, cpl_vector_get(self->calib_wavelengths, i) /
                                                                  (SIGMA_TO_FWHM_FACTOR * sigma));
                cpl_vector_set_size(good_respow, no_of_good_wavls + 1);
            }

        }
        // Remove the 'waiting' element of fitted_wavls
        cpl_vector_set_size(fitted_wavls, no_of_good_wavls);
        cpl_vector_set_size(good_respow, no_of_good_wavls);

        cpl_propertylist * wavecalib_plist = cpl_propertylist_new();
        if (no_of_good_wavls < 2) {
            SPH_WARNING("Insufficient fits to laser wavelengths to determine dispersion");
            cpl_propertylist_append_double(new_plist, SPH_IFS_KEYWORD_LASER_MIN_LAMBDA,
                                           -999.0);
            cpl_propertylist_append_double(new_plist, SPH_IFS_KEYWORD_LASER_DISPERSION,
                                           -999.0);
        } else {
            // Find the relation between the nominal laser wavelength values and the
            // Gaussian centres
            // To do this, we will create a cpl_bivector object
            cpl_vector *pixl = cpl_vector_new(no_wavl);
            for (int i = 1; i <= no_wavl; i++) {
                cpl_vector_set(pixl, i - 1, (double) i);
            }
            cpl_bivector *wavl_to_pixl = cpl_bivector_wrap_vectors(sdp_wavls, pixl);

            cpl_vector *fitted_pixl = cpl_vector_new(
                    cpl_vector_get_size(fitted_wavls)
            );
            cpl_vector *copy_of_calib_wavelengths = cpl_vector_new(cpl_vector_get_size(self->calib_wavelengths));
            cpl_bivector *fitted_wavl_to_pixl = cpl_bivector_wrap_vectors(
                    fitted_wavls,
                    cpl_vector_new(cpl_vector_get_size(fitted_wavls))
            );
            cpl_bivector_interpolate_linear(fitted_wavl_to_pixl, wavl_to_pixl);
            cpl_bivector_delete(wavl_to_pixl);

            // Make a linear fit to the points we have determined
            double chisq;
            cpl_polynomial *linfit = sph_fitting_fit_poly1d(
                    cpl_bivector_get_y(fitted_wavl_to_pixl), // x
//                    cpl_bivector_get_x(fitted_wavl_to_pixl), // y
                    self->calib_wavelengths, // y
                    NULL, // NULL
                    0, 1, // min and max degree
                    CPL_FALSE, 1.0, &chisq // allowshift, guess, chisq
            );
            const cpl_size zero = 0;
            const cpl_size one = 1;

            // PIPE-7454 - compute RMS (r^2_{xy})
            // Easy ones
            double xbar = cpl_vector_get_mean(cpl_bivector_get_y(fitted_wavl_to_pixl));
            double ybar = cpl_vector_get_mean(self->calib_wavelengths);
            // Ones requiring multiplcation
            cpl_vector* x2 = cpl_vector_duplicate(cpl_bivector_get_y(fitted_wavl_to_pixl));
            cpl_vector_multiply(x2, cpl_bivector_get_y(fitted_wavl_to_pixl));
            double x2bar = cpl_vector_get_mean(x2);
            cpl_vector* y2 = cpl_vector_duplicate(self->calib_wavelengths);
            cpl_vector_multiply(y2, self->calib_wavelengths);
            double y2bar = cpl_vector_get_mean(y2);
            cpl_vector* xy = cpl_vector_duplicate(cpl_bivector_get_y(fitted_wavl_to_pixl));
            cpl_vector_multiply(xy, self->calib_wavelengths);
            double xybar = cpl_vector_get_mean(xy);
            cpl_vector_delete(x2);
            cpl_vector_delete(y2);
            cpl_vector_delete(xy);
            double r2xy = pow(xybar - (xbar * ybar), 2) / ((x2bar - pow(xbar, 2)) * (y2bar - pow(ybar, 2)) );
            printf("I have xbar = %f, x2bar = %f, ybar = %f, y2bar = %f, xybar = %f ==> r2xy = %.15f\n",
                   xbar, x2bar, ybar, y2bar, xybar, r2xy);
//            getchar();

            // There's also a second formula that uses the fitted parameters
            // Uncomment this code block to use as a sanity check
            /* START SANITY CHECK */
//            cpl_vector* xpos ;
//            cpl_vector* fi = cpl_vector_new(cpl_vector_get_size(self->calib_wavelengths));
//            for (cpl_size i = 0; i < cpl_vector_get_size(self->calib_wavelengths); ++i) {
//                xpos = cpl_vector_new(1);
//                // Set x pos
//                cpl_vector_set(xpos, 0, cpl_vector_get(cpl_bivector_get_y(fitted_wavl_to_pixl), i));
//                // Compute fi at this pos
//                double fii = cpl_polynomial_eval(linfit, xpos);
//                cpl_vector_set(fi, i, fii);
//            }
//            cpl_vector_delete(xpos);
//            cpl_vector* r_numer = cpl_vector_duplicate(self->calib_wavelengths);
//            cpl_vector_subtract(r_numer, fi);
//            cpl_vector_power(r_numer, 2.0);
//            cpl_vector* r_denom = cpl_vector_duplicate(self->calib_wavelengths);
//            cpl_vector_subtract_scalar(r_denom, cpl_vector_get_mean(self->calib_wavelengths));
//            cpl_vector_power(r_denom, 2.0);
//            double r2xy_2 = 1.0 - (cpl_vector_get_sum(r_numer) / cpl_vector_get_sum(r_denom));
//            printf("I have a check value of r2xy = %.15f\n", r2xy_2);
//            getchar();
//            cpl_vector_delete(r_numer);
//            cpl_vector_delete(r_denom);
//            cpl_vector_delete(fi);
            /* END SANITY CHECK */


            cpl_bivector_delete(fitted_wavl_to_pixl);

            // Store the following info from the above calculation:
            // QC MIN LAMBDA - wavelengths in the first pixel/wavelength bin
            // Note this is the wavelength at pixel ONE, not the x-intercept (pixel zero)
            cpl_propertylist_append_double(new_plist, SPH_IFS_KEYWORD_LASER_MIN_LAMBDA,
                                           cpl_polynomial_get_coeff(linfit, &zero) + cpl_polynomial_get_coeff(linfit, &one));
            // QC DISPERSION - computed dispersion
            cpl_propertylist_append_double(new_plist, SPH_IFS_KEYWORD_LASER_DISPERSION,
                                           cpl_polynomial_get_coeff(linfit, &one));
            // QC DISPERSION RMS - computed dispersion RMS (actually r2xy - see PIPE-7454)
            cpl_propertylist_append_double(new_plist, SPH_IFS_KEYWORD_LASER_DISPERSION_RMS,
                                           r2xy);
            // Loop through the laser flux keywords and divide by dispersion
            cpl_msg_info(cpl_func, "QC dispersion = %f, min lambda = %f",
                         cpl_polynomial_get_coeff(linfit, &one),
                         cpl_polynomial_get_coeff(linfit, &zero) + cpl_polynomial_get_coeff(linfit, &one));
            for (int i = 0; i < 4; i++) {
                cpl_property * laser_flux = cpl_propertylist_get_property(new_plist,
                                                                 cpl_sprintf("%s%d", SPH_IFS_KEYWORD_LASER_LINE_FLUX, i+1));
                if (laser_flux != NULL) {
                    double laser_flux_val = cpl_property_get_double(laser_flux);
                    cpl_property_set_double(laser_flux, laser_flux_val / cpl_polynomial_get_coeff(linfit, &one));
                }
            }

            // Create and write an additional propertylist to write
            // a few parameters back to the IFS_WAVECALIB product
            cpl_propertylist_append_double(wavecalib_plist, SPH_IFS_KEYWORD_LASER_DISPERSION,
                                           cpl_polynomial_get_coeff(linfit, &one));
            cpl_propertylist_append_double(wavecalib_plist, SPH_IFS_KEYWORD_LASER_MIN_LAMBDA,
                                           cpl_polynomial_get_coeff(linfit, &zero) + cpl_polynomial_get_coeff(linfit, &one));

            // QC DISPERSION RMS - computed dispersion RMS (actually r2xy - see PIPE-7454)
            cpl_propertylist_append_double(wavecalib_plist, SPH_IFS_KEYWORD_LASER_DISPERSION_RMS,
                                           r2xy);

            // QC MEDIAN RESOLVING POWER - updated for both products as per WH (PIPE-9821)
            cpl_propertylist_append_double(wavecalib_plist, SPH_COMMON_KEYWORD_MEDIAN_RESOLVING_POWER,
                                           cpl_vector_get_median(good_respow));
            cpl_propertylist_append_double(new_plist, SPH_COMMON_KEYWORD_MEDIAN_RESOLVING_POWER,
                                           cpl_vector_get_median(good_respow));
        }

        // Write the updated propertylist back to the file
        // There's a SPHERE utility function for this
        for (int i = 0; i < cpl_propertylist_get_size(new_plist); i++) {
            cpl_property *pr = cpl_propertylist_get(new_plist, i);
            sph_fits_update_property(cube_fname, pr, 0);
        }

        for (int i = 0; i < cpl_propertylist_get_size(wavecalib_plist); i++) {
            cpl_property *pr = cpl_propertylist_get(wavecalib_plist, i);
            sph_fits_update_property(self->wave_calib_filename, pr, 0);
        }

        // Clean-up
        cpl_imagelist_delete(sdp_product_imgs);
        cpl_vector_delete(good_respow);
        cpl_vector_delete(median_collapse_spec);
        cpl_propertylist_delete(sdp_plist);
        cpl_propertylist_delete(new_plist);
        cpl_propertylist_delete(wavecalib_plist);
        // END PIPE-7692

        SPH_INFO_MSG(" Cube save complete, about to free the cube name variable...");
        cpl_free(cube_fname);

        if(rerr){
            SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO,
                    "Could not write wavelength calibrated cube.");
        }
        SPH_INFO_MSG("  Deleting cube from memory");
        sph_master_frame_delete(mframe);
        SPH_INFO_MSG("Cube complete!");
    }

    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    if(pli){
        cpl_propertylist_delete(pli);
        pli = NULL;
    }
    cpl_propertylist_delete(plist);
    plist = NULL;
    sph_master_frame_delete(iff);
    iff = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    cpl_mask_delete(bmask);
    bmask = NULL;
    cpl_vector_delete(self->calib_wavelengths);
    self->calib_wavelengths = NULL;

    return rerr;
}


/* Implementation of private functions */

static inline const char *
safe_extract_from_plist(const cpl_propertylist * plist, const char * card_name){

	if(!cpl_propertylist_has(plist, card_name)) return "";
	return cpl_propertylist_get_string(plist, card_name);
}

static inline cpl_boolean are_all_raws_ifs(const cpl_frameset * fset){
	for(cpl_size i = 0; i < cpl_frameset_get_size(fset); ++i){

		const cpl_frame * f = cpl_frameset_get_position_const(fset, i);
		const char * fname = cpl_frame_get_filename(f);
		cpl_propertylist * plist = cpl_propertylist_load(fname, 0);
		const char * detector_name =
				safe_extract_from_plist(plist, SPH_COMMON_KEYWORD_DETECTOR_NAME);
		cpl_boolean is_ifs =
				strcmp(detector_name, SPH_COMMON_KEYWORD_IFS_DETECTOR_NAME) == 0;
		cpl_propertylist_delete(plist);

		if(!is_ifs) return CPL_FALSE;
	}

	return CPL_TRUE;
}

static inline sph_error_code save_as_cube(sph_ifs_wave_calib * self,
        sph_pixel_description_table *pdt, sph_ifs_lenslet_model * model,
        cpl_propertylist * plist, cpl_frame * template, const char * fname,
        const char *tag, sph_master_frame* mframe, const double ang){

    if(pdt == NULL || mframe == NULL) return CPL_ERROR_NONE;

    SPH_INFO_MSG("Forming LDT1 for SDP product");
    sph_ldt * ldt1 = sph_ldt_new_from_pdt_master_frame(pdt,
    		sph_ifs_lenslet_model_duplicate(model), mframe);

    SPH_INFO_MSG("Saving LDT cube for SDP product");
    sph_error_code fail = sph_ldt_save_cube(ldt1, fname, self->inframes,
                                            self->inframes, template,
            self->inparams, tag, SPH_RECIPE_NAME_IFS_WAVE_CALIB,
            SPH_PIPELINE_NAME_IFS, SPH_IFS_PIXEL_SCALE,
            SPH_IFS_ASTROM_ROTATIONSIGN * ang, plist);

    sph_ldt_delete(ldt1);

    SPH_INFO_MSG("Completed LDT SDP product save");
    return fail;
}

/**@}*/
