/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_instrument_flat.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_create_flat.h"
#include "sph_filemanager.h"
#include "sph_ird_instrument_model.h"
#include "sph_keyword_manager.h"
#include "sph_fitting.h"
#include "sph_background.h"
#include "sph_fft.h"
#include "sph_framecombination.h"

#include <cpl.h>
#include <math.h>

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

static const double SPH_IRD_INSTRUMENT_FLAT_SMOOTHLENGTH = 2.0;

static sph_master_frame* sph_ird_iff_select_dark(const sph_ird_instrument_flat*)
    CPL_ATTR_ALLOC;

static
void sph_ird_instrument_flat_analyse(sph_master_frame* temp_master,
        cpl_image* nonlin_image, double* mean, double* fpn, double *rms,
        double* nonlin) {
    cpl_mask* nonlin_mask = NULL;
    cpl_image* smoothed = NULL;
    cpl_image* tmp = NULL;

    *mean = sph_master_frame_get_mean(temp_master, rms);
    if (nonlin_image) {
        nonlin_mask = sph_master_frame_get_badpixelmask(temp_master);
        if (nonlin_mask) {
            cpl_image_reject_from_mask(nonlin_image, nonlin_mask);
        }
        *nonlin = cpl_image_get_mean(nonlin_image);
    }
    if (fpn) {
        tmp = sph_master_frame_extract_image(temp_master, 1);
        smoothed = cpl_image_duplicate(tmp);
        sph_fft_smoothe_image(smoothed, NULL, SPH_IRD_INSTRUMENT_FLAT_SMOOTHLENGTH);
        cpl_image_subtract(tmp, smoothed);
        cpl_image_delete(smoothed);
        smoothed = NULL;
        *fpn = cpl_image_get_stdev(tmp);
        *fpn = *fpn / *mean; // express FPN as perecentage of mean
        cpl_image_delete(tmp);
        tmp = NULL;
    }
    cpl_mask_delete(nonlin_mask);
    nonlin_mask = NULL;

}


/*----------------------------------------------------------------------------*/
/**
 @brief    Select dark frame to use from input
 @param    self   the recipe structure
 @return   Pointer to dark master frame to use, or NULL if none

 This is a helper to check the dark and background frames provided and select the
 highest priority one...

 */
/*----------------------------------------------------------------------------*/
static
sph_master_frame* sph_ird_iff_select_dark(const sph_ird_instrument_flat* self)
{
    const cpl_frame*  dark_frame     = NULL;
    sph_master_frame* out_dark_frame = NULL;

    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (self->insbg_fit_frame != NULL) {
        dark_frame = self->insbg_fit_frame;
        cpl_msg_info(cpl_func,"Using INS_BG_FIT frame as background!");
    } else if (self->insbg_frame != NULL) {
        dark_frame = self->insbg_frame;
        cpl_msg_info(cpl_func,"Using INS_BG frame as background!");
    } else if (self->master_dark != NULL) {
        dark_frame = self->master_dark;
        cpl_msg_info(cpl_func,"Using master dark as background!");
    }

    if (dark_frame != NULL) {
    	out_dark_frame = sph_master_frame_load_(dark_frame, 0);
        if (cpl_error_get_code()) {
            (void)cpl_error_set_where(cpl_func);
        }
    }
    return out_dark_frame;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_instrument_flat recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_instrument_flat_run(sph_ird_instrument_flat* self) {
    cpl_image* hot_image = NULL;
    cpl_mask* badpix = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* temp_master = NULL;
    sph_ird_instrument_model* model = NULL;
    cpl_propertylist* plist = NULL;
    double rms = 0.0;
    cpl_propertylist* pl = NULL;
    cpl_image* nonlin_image_all = NULL;
    cpl_image* nonlin_image_left = NULL;
    cpl_image* nonlin_image_right = NULL;
    cpl_mask* nonlin_mask = NULL;
    cpl_mask* left_mask = NULL;
    cpl_mask* right_mask = NULL;
    sph_master_frame* left_flat = NULL;
    sph_master_frame* right_flat = NULL;
    cpl_vector* lampflux_left = NULL;
    cpl_vector* lampflux_right = NULL;
    cpl_vector* counts_left = NULL;
    cpl_vector* counts_right = NULL;
    cpl_vector* lampfluxstdev_right = NULL;
    cpl_vector* lampfluxstdev_left = NULL;
    double mean = 0.0;
    double fpn = 0.0;
    double frms = 0.0;
    double nonlin = 0.0;
    cpl_mask* tmpmask = NULL;
    cpl_frameset* bg_subtracted_frames = NULL;

    /*------------------------------------------------------------------
     -  Placing master dark in cube structure for later use
     --------------------------------------------------------------------*/

    dark = sph_ird_iff_select_dark(self);

    /*------------------------------------------------------------------
     -  Now process all sets of wavelength frames separately
     --------------------------------------------------------------------*/


    if (self->robust_fit) {
        if (cpl_frameset_get_size(self->rawframes) < 5) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                    "The robust fitting is enabled. "
                    "For this to work, a minimum of 5 raw frames needs"
                    "to be provided. Either provide more frames or"
                    "switch robust fitting off.");
            return CPL_ERROR_ILLEGAL_INPUT;
        }
    }

    if (self->static_badpixel_frame) {
        hot_image = cpl_image_load(
                cpl_frame_get_filename(self->static_badpixel_frame),
                CPL_TYPE_INT, 0, 0);
        if (hot_image) {
            badpix = cpl_mask_threshold_image_create(hot_image, 0.5, 10000);
            cpl_image_delete(hot_image);
            hot_image = NULL;
        }
    }
    if (self->master_dark_frameset) {
        bg_subtracted_frames = sph_background_apply_to_raw_cubes(
                self->rawframes, self->master_dark_frameset);
        if(dark)
        	sph_master_frame_delete(dark);
        dark = NULL;
        cpl_msg_info(cpl_func,"Raw dark frames found for ideal dark calibration!");
        cpl_msg_info(cpl_func,"Disabled any additional dark!");

    }
    if (bg_subtracted_frames == NULL){
        bg_subtracted_frames = self->rawframes;
        cpl_msg_info(cpl_func,"No Raw dark frames found for ideal dark calibration!");
        cpl_msg_info(cpl_func,"Using standard darks if present!");
    }
    if (self->model_frame) {
        model = sph_ird_instrument_model_load(
                cpl_frame_get_filename(self->model_frame));
        if (model) {
            SPH_ERROR_RAISE_INFO(
                    SPH_ERROR_GENERAL,
                    "Using a custom IRDIS instrument model from %s.", cpl_frame_get_filename(self->model_frame));
        } else {
            SPH_ERROR_RAISE_WARNING(
                    SPH_ERROR_GENERAL,
                    "Could not load custom IRDIS instrument model from %s."
                    "Am using default instead.", cpl_frame_get_filename(self->model_frame));
            cpl_error_reset();
        }
    }
    if (!model) {
        model = sph_ird_instrument_model_new();
    }

    plist = sph_ird_instrument_model_get_as_propertylist(model);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Collapsing raw frames...");
    temp_master = sph_framecombination_master_frame_from_frameset(
            bg_subtracted_frames, self->coll_alg, self->framecomb_parameterlist,
            0);
    SPH_ERROR_ENSURE_GOTO_EXIT( temp_master, cpl_error_get_code( ));
    if (model->detsize_pixels_x != cpl_image_get_size_x(temp_master->image)
            || model->detsize_pixels_y
                    != cpl_image_get_size_y(temp_master->image)) {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_ILLEGAL_INPUT,
                "The pixel size of the "
                "model (%d x %d) and the input raw "
                "images (%d x %d) is not the same."
                " There is something wrong either with the "
                "configuration of the "
                "data reduction recipe or the settings "
                "in the detector readout.", model->detsize_pixels_x, model->detsize_pixels_y, (int)cpl_image_get_size_x(temp_master->image), (int)cpl_image_get_size_y(temp_master->image));
        goto EXIT;
    }

    mean = sph_master_frame_get_mean(temp_master, &rms);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Finding illuminated regions. "
    "All regions with a value of %f times the mean "
    "are selected. In this case this works out as "
    "a threshold at %f counts.", self->threshold, mean * self->threshold);
    if (mean < 1000.0) {
        SPH_ERROR_RAISE_WARNING(CPL_ERROR_ILLEGAL_INPUT,
                "The mean of a flat is %f. "
                "Using flats with mean counts less than 1000 is dangerous."
                "The recipe will proceed but be aware that the master"
                " flat created will most likely not be of high quality.", mean);
    }
    sph_master_frame_mask_tolerance(temp_master, mean * 0.1, FLT_MAX);
    tmpmask = sph_master_frame_get_badpixelmask(temp_master);
    sph_master_frame_delete(temp_master);
    temp_master = NULL;

    left_mask = sph_ird_instrument_model_get_mask_leftwin(model);

    if (badpix)
        cpl_mask_or(left_mask, badpix);

    if (tmpmask)
        cpl_mask_or(left_mask, tmpmask);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Determining flat for the left FOV");
    left_flat = sph_create_flat(bg_subtracted_frames, self->robust_fit,
            left_mask, dark, &nonlin_image_all, self->badpix_lowtolerance,
            self->badpix_uptolerance, self->badpix_chisqtolerance,
            &lampflux_left, &counts_left, 0, &lampfluxstdev_left);
    temp_master = sph_ird_instrument_model_extract_left_master_frame(model,
            left_flat);
    sph_master_frame_delete(left_flat);
    left_flat = temp_master;
    nonlin_image_left = sph_ird_instrument_model_extract_left_image(model,
            nonlin_image_all);
    cpl_image_delete(nonlin_image_all);
    nonlin_image_all = NULL;

    right_mask = sph_ird_instrument_model_get_mask_rightwin(model);

    if (badpix)
        cpl_mask_or(right_mask, badpix);

    if (tmpmask)
        cpl_mask_or(right_mask, tmpmask);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Determining flat for the right FOV");
    right_flat = sph_create_flat(bg_subtracted_frames, self->robust_fit,
            right_mask, dark, &nonlin_image_all, self->badpix_lowtolerance,
            self->badpix_uptolerance, self->badpix_chisqtolerance,
            &lampflux_right, &counts_right, 0, &lampfluxstdev_right);
    temp_master = sph_ird_instrument_model_extract_right_master_frame(model,
            right_flat);
    sph_master_frame_delete(right_flat);
    right_flat = temp_master;
    nonlin_image_right = sph_ird_instrument_model_extract_right_image(model,
            nonlin_image_all);
    cpl_image_delete(nonlin_image_all);
    nonlin_image_all = NULL;
    if (left_flat && right_flat) {
        temp_master = sph_ird_instrument_model_assemble(model, left_flat,
                right_flat);
        if (temp_master != NULL)
            sph_ird_instrument_raw_add_qc_counts(temp_master->qclist,
                                                 self->rawframes,
                                                 left_mask, right_mask,
                                                 CPL_FALSE);
    } else {
        temp_master = NULL;
    }
    cpl_mask_delete(left_mask);
    cpl_mask_delete(right_mask);
    left_mask = NULL;
    right_mask = NULL;

    if (nonlin_image_left && nonlin_image_right) {
        nonlin_image_all = sph_ird_instrument_model_assemble_image(model,
                nonlin_image_left, nonlin_image_right);
    }

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Determining QC keywords and saving...");
    if (temp_master) {

        pl = cpl_propertylist_new();
        cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NUMBER_BADPIXELS,
                sph_master_frame_get_nbads(temp_master));
        sph_ird_instrument_flat_analyse(temp_master, temp_master->rmsmap, &mean,
                &fpn, &frms, &nonlin);
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_FLAT_MEAN_COUNT,
                mean);
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_FLAT_FPN, fpn);
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_FLAT_RMS, frms);
        cpl_propertylist_update_double(pl,
                SPH_COMMON_KEYWORD_FLAT_NONLIN_FACTOR, nonlin);
        sph_ird_instrument_flat_analyse(left_flat, left_flat->rmsmap, &mean,
                &fpn, &frms, &nonlin);
        cpl_propertylist_update_double(pl, SPH_IRD_KEYWORD_FLAT_MEAN_COUNT_LEFT,
                mean);
        cpl_propertylist_update_double(pl, SPH_IRD_KEYWORD_FLAT_FPN_LEFT, fpn);
        cpl_propertylist_update_double(pl, SPH_IRD_KEYWORD_FLAT_RMS_LEFT, frms);
        cpl_propertylist_update_double(pl,
                SPH_IRD_KEYWORD_FLAT_NONLIN_FACTOR_LEFT, nonlin);
        sph_keyword_manager_vector2proplist(pl, lampflux_left,
                SPH_IRD_KEYWORD_FLAT_LAMP_FLUX_LEFT);
        sph_keyword_manager_vector2proplist(pl, counts_left,
                SPH_IRD_KEYWORD_FLAT_LAMP_COUNTS_LEFT);
        sph_keyword_manager_vector2proplist(pl, lampfluxstdev_left,
                SPH_IRD_KEYWORD_FLAT_LAMP_FLUX_STDEV_LEFT);

        sph_ird_instrument_flat_analyse(right_flat, right_flat->rmsmap, &mean,
                &fpn, &frms, &nonlin);
        cpl_propertylist_update_double(pl,
                SPH_IRD_KEYWORD_FLAT_MEAN_COUNT_RIGHT, mean);
        cpl_propertylist_update_double(pl, SPH_IRD_KEYWORD_FLAT_FPN_RIGHT, fpn);
        cpl_propertylist_update_double(pl, SPH_IRD_KEYWORD_FLAT_RMS_RIGHT,
                frms);
        cpl_propertylist_update_double(pl,
                SPH_IRD_KEYWORD_FLAT_NONLIN_FACTOR_RIGHT, nonlin);
        sph_keyword_manager_vector2proplist(pl, lampflux_right,
                SPH_IRD_KEYWORD_FLAT_LAMP_FLUX_RIGHT);
        sph_keyword_manager_vector2proplist(pl, counts_right,
                SPH_IRD_KEYWORD_FLAT_LAMP_COUNTS_RIGHT);
        sph_keyword_manager_vector2proplist(pl, lampfluxstdev_right,
                SPH_IRD_KEYWORD_FLAT_LAMP_FLUX_STDEV_RIGHT);

        cpl_propertylist_append_double(pl, SPH_IRD_KEYWORD_FLUX_LEFT,
                cpl_vector_get_mean(lampflux_left));
        cpl_propertylist_append_double(pl, SPH_IRD_KEYWORD_FLUX_RIGHT,
                cpl_vector_get_mean(lampflux_right));
        cpl_propertylist_append(pl, plist);
        sph_master_frame_save_dfs(temp_master, self->outfilename,
                self->inframes, NULL, self->inparams, SPH_IRD_TAG_FLAT_CALIB,
                SPH_RECIPE_NAME_IRD_INSTRUMENT_FLAT, SPH_PIPELINE_NAME_IRDIS,
                pl);
        sph_master_frame_delete(temp_master);
        cpl_image_delete(nonlin_image_left);
        nonlin_image_left = NULL;
        cpl_image_delete(nonlin_image_right);
        nonlin_image_right = NULL;
        cpl_mask_delete(nonlin_mask);
        nonlin_mask = NULL;
        cpl_propertylist_delete(pl);
        pl = NULL;
    }
    if (nonlin_image_all) {
    	if(self->save_addprod){
    		pl = cpl_propertylist_new();
    		cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NUMBER_BADPIXELS,
                (int) cpl_image_get_flux(nonlin_image_all));
    		cpl_propertylist_append_string(pl, SPH_COMMON_KEYWORD_PRO_CATG,
                SPH_IRD_TAG_NON_LINEAR_PIXELMAP_CALIB);
    		cpl_propertylist_append(pl, plist);
                sph_utils_remove_wcs_3d(pl);
                /* FIXME: Set WCS to dummy (pixel) value for now */
                sph_utils_reset_wcs_12d(pl);
                /* Make any final header updates */
                sph_utils_update_header(pl);
    		cpl_dfs_save_image(self->inframes, NULL, self->inparams, self->inframes,
                NULL, nonlin_image_all, CPL_BPP_8_UNSIGNED,
                SPH_RECIPE_NAME_IRD_INSTRUMENT_FLAT, pl, NULL,
                SPH_PIPELINE_NAME_IRDIS, self->badpix_filename);
    		cpl_propertylist_delete(pl);
    	}
        pl = NULL;
        cpl_image_delete(nonlin_image_all);
        nonlin_image_all = NULL;

    }
    EXIT: sph_master_frame_delete(dark);
    dark = NULL;
    cpl_mask_delete(badpix);
    badpix = NULL;
    sph_master_frame_delete(left_flat);
    left_flat = NULL;
    sph_master_frame_delete(right_flat);
    right_flat = NULL;

    cpl_vector_delete(lampflux_left);
    lampflux_left = NULL;
    cpl_vector_delete(lampflux_right);
    lampflux_right = NULL;
    cpl_vector_delete(counts_left);
    counts_left = NULL;
    cpl_vector_delete(counts_right);
    counts_right = NULL;
    cpl_vector_delete(lampfluxstdev_left);
    lampfluxstdev_left = NULL;
    cpl_vector_delete(lampfluxstdev_right);
    lampfluxstdev_right = NULL;
    cpl_propertylist_delete(plist);
    plist = NULL;
    sph_ird_instrument_model_delete(model);
    sph_filemanager_clean();

    cpl_mask_delete(tmpmask);

    if (bg_subtracted_frames != self->rawframes) {
        cpl_frameset_delete(bg_subtracted_frames);
    }

    return (int) cpl_error_get_code();
}

/**@}*/

