/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include <math.h>
#include "sph_ird_loci.h"
#include "sph_extract_angles.h"
#include "sph_ird_tags.h"
#include "sph_ird_instrument_model.h"
#include "sph_fctable.h"
#include "sph_filter_table.h"
#include "sph_loci.h"
#include "sph_filemanager.h"
#include "sph_ird_common_science.h"
#include "sph_common_science.h"
#include "sph_keyword_manager.h"
#include "sph_common_keywords.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
cpl_error_code sph_ird_loci_run(sph_ird_loci* self) {
    sph_master_frame* result = NULL;
    cpl_vector* angles = NULL;
    cpl_vector* angles_stop = NULL;
    cpl_propertylist* plist = NULL;
    cpl_frameset* total_frameset = NULL;
    cpl_frameset* fctable_frames = NULL;
    cpl_frameset* speckle_frameset = NULL;
    cpl_array* speck_assoc = NULL;
    sph_fctable* fctab = NULL;
    sph_loci* loci = NULL;
    cpl_image* tmpim = NULL;
    cpl_image* badmask = NULL;
    int ff = 0;
    double cx = 0.0;
    double cy = 0.0;
    cpl_frameset* processed = NULL;
    cpl_frame* aframe = NULL;
    sph_ird_common_science* sci = NULL;
    cpl_vector* tmpangles = NULL;

    sci = sph_ird_common_science_new(self->inframes, self->inparams,
            self->dark_frame, self->flat_frame, self->static_badpixel_frame,
            self->distmap_frame, self->filter_frame, self->flat_frame,
            NULL,
            SPH_RECIPE_NAME_IRD_LOCI, SPH_IRD_TAG_LOCI_CALIB);

    if (self->fcasciis != NULL) {
        fctable_frames = cpl_frameset_duplicate(self->fcasciis);
        if (fctable_frames == NULL) {
            goto EXIT;
        }
    } else {
        fctable_frames = sph_ird_common_science_create_fctables(self->rawframes,
                self->fcframe, self->keep_fctable, 1);
    }

    sph_ird_common_science_process_cubes(sci, self->rawframes, fctable_frames,
                                         NULL, NULL, &total_frameset,
                                         NULL, NULL, NULL);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO_MESSAGE,
            "rawframes: %d", (int)cpl_frameset_get_size(self->rawframes));
    SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO_MESSAGE,
            "frameset: %d", (int)cpl_frameset_get_size(fctable_frames));
    SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO_MESSAGE,
            "total_frameset: %d", (int)cpl_frameset_get_size(total_frameset));

    angles = sph_common_science_extract_angles_from_fctables(fctable_frames);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    angles_stop = cpl_vector_duplicate(angles);

    for (ff = 0; ff < cpl_vector_get_size(angles) - 1; ++ff) {
        cpl_vector_set(angles_stop, ff, cpl_vector_get(angles, ff + 1));
    }

    cpl_vector_set(
            angles_stop,
            cpl_vector_get_size(angles) - 1,
            2.0 * cpl_vector_get(angles, cpl_vector_get_size(angles) - 1)
                    - cpl_vector_get(angles, cpl_vector_get_size(angles) - 2));

    cx = sci->irdmodel->window_size_y / 2.0;
    cy = sci->irdmodel->window_size_y / 2.0;

    loci = sph_loci_new(self->Na, self->W, self->Ndelta, cx, cy, self->g,
            self->minR, self->maxR);
    plist = cpl_propertylist_new();

    if (loci) {
        processed = cpl_frameset_new();
        sph_loci_set_frameset(loci, total_frameset);
        sph_loci_init_subsections(loci, self->dr, self->div_scheme);

        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "LOCI has been set up to process %d frames."
                "The image(s) are divided into %d segements in %d annuli. ", (int)cpl_frameset_get_size(total_frameset), loci->nsubsections, (int)cpl_vector_get_size(loci->rings_radii));

        cpl_propertylist_update_int(plist, SPH_COMMON_KEYWORD_QC_LOCI_NRINGS,
                (int) cpl_vector_get_size(loci->rings_radii));

        cpl_propertylist_update_int(plist,
                SPH_COMMON_KEYWORD_QC_LOCI_NSUBSECTIONS, loci->nsubsections);
        for (ff = 0; ff < cpl_vector_get_size(angles); ++ff) {
            tmpim = sph_loci_process_single(loci, ff,
                    sci->irdmodel->window_size_y, sci->irdmodel->window_size_y,
                    angles, angles_stop);
            if (tmpim) {
                SPH_ERROR_RAISE_INFO(
                        SPH_ERROR_GENERAL,
                        "Performing"
                        " LOCI on image %d of %d", ff, (int)cpl_vector_get_size(angles));
                badmask = cpl_image_new_from_mask(cpl_image_get_bpm(tmpim));
                cpl_detector_interpolate_rejected(tmpim);
                sph_transform_apply_to_images(sci->transform, &tmpim, &badmask,
                        NULL, NULL, NULL, cx, cy, cpl_vector_get(angles, ff),
                        1.0, sci->irdmodel);
                aframe = sph_filemanager_create_temp_frame(
                        SPH_RECIPE_NAME_IRD_LOCI, SPH_IRD_TAG_LOCI_CALIB,
                        CPL_FRAME_GROUP_CALIB);
                cpl_image_save(tmpim, cpl_frame_get_filename(aframe),
                        CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
                cpl_image_save(badmask, cpl_frame_get_filename(aframe),
                        CPL_BPP_8_UNSIGNED, NULL, CPL_IO_EXTEND);
                cpl_frameset_insert(processed, aframe);
                cpl_image_delete(tmpim);
                tmpim = NULL;
                cpl_image_delete(badmask);
                badmask = NULL;
            } else {
                SPH_ERROR_RAISE_ERR(cpl_error_get_code(),
                        "Could not process image %d", ff);
                break;
            }
        }
        if (cpl_error_get_code() == CPL_ERROR_NONE) {
            result = sph_common_science_combine(processed, self->coll_alg, 0, 1,
                    -1);
            cpl_propertylist_update_int(plist,
                    SPH_COMMON_KEYWORD_QC_LOCI_NBADSECTIONS, loci->nbads);

            if (result) {
                sph_master_frame_save_dfs(result, self->loci_outfilename,
                        self->inframes, cpl_frameset_get_first(self->rawframes),
                        self->inparams, SPH_IRD_TAG_LOCI_CALIB,
                        SPH_RECIPE_NAME_IRD_LOCI, SPH_PIPELINE_NAME_IRDIS,
                        plist);
            } else {
                SPH_ERR("Could not create product frame.");
            }
        }
    }
    EXIT: sph_master_frame_delete(result);
    result = NULL;
    cpl_propertylist_delete(plist);
    plist = NULL;
    cpl_array_delete(speck_assoc);
    speck_assoc = NULL;
    cpl_frameset_delete(total_frameset);
    total_frameset = NULL;
    cpl_frameset_delete(speckle_frameset);
    speckle_frameset = NULL;
    cpl_frameset_delete(processed);
    processed = NULL;
    cpl_frameset_delete(fctable_frames);
    fctable_frames = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    sph_loci_delete(loci);
    loci = NULL;
    cpl_vector_delete(angles);
    angles = NULL;
    cpl_vector_delete(tmpangles);
    tmpangles = NULL;
    cpl_vector_delete(angles_stop);
    angles_stop = NULL;
    sph_ird_common_science_delete(sci);
    sph_filemanager_clean();
    return (int) cpl_error_get_code();

}

/**@}*/
