/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include <string.h>
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_science_imaging.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_simple_adi.h"
#include "sph_utils.h"
#include "sph_ird_instrument_model.h"
#include "sph_framecombination.h"
#include "sph_common_science.h"
#include "sph_ird_common_science.h"
#include "sph_filemanager.h"
#include "sph_keyword_manager.h"
#include "sph_differential_imaging.h"
#include "sph_strehl.h"
#include "sph_phot.h"


const char* const SPH_IRD_STAR_CENTER_FILE_NAME = "WAFFFRAME";
const char* const SPH_IRD_STAR_CENTER_ANGLE_NAME = "ANGLE";

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_SCIENCE_IMAGING_GENERAL;
extern sph_error_code SPH_IRD_SCIENCE_IMAGING_FRAMES_MISSING;


/*-----------------------------------------------------------------------------
 Private functions forward declaration
 -----------------------------------------------------------------------------*/
static cpl_error_code
calc_flux_and_possibly_strehl(cpl_propertylist  * plist_left,
                            cpl_propertylist  * plist_right,
                            const cpl_image   * img_left,
                            const cpl_image   * img_right,
                            const cpl_frame   * raw_frame,
                            const cpl_frame   * filter_frame,
                            const char        * recipe,
                            cpl_parameterlist * parlist,
                            double            * flux_left,
                            double            * flux_right);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_science_imaging_run
 *
 * This module provides the core recipe for the IRDIS science recipe
 * in the classical imaging mode.
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_science_imaging.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_science_imaging recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code sph_ird_science_imaging_run(sph_ird_science_imaging* self) {
    sph_master_frame* result = NULL;
    sph_master_frame* leftFOV = NULL;
    sph_master_frame* rightFOV = NULL;
    cpl_propertylist* plist = NULL;
    cpl_propertylist* plist_left = NULL;
    cpl_propertylist* plist_right = NULL;
    cpl_propertylist* pli   = NULL;
    sph_ird_common_science* sci = NULL;
    sph_transform* transform = NULL;
    cpl_frameset* total_frameset = NULL;
    cpl_frameset* left_frameset_out = NULL;
    cpl_frameset* right_frameset_out = NULL;
    cpl_frameset* adi_frameset = NULL;
    cpl_frameset* fctable_frames = NULL;
    cpl_vector* angles = NULL;
    int master_dark_present = 0;


    /*------------------------------------------------------------------
     -  selecting the right dark
     --------------------------------------------------------------------*/
    if(self->dark_frame)
    	master_dark_present = 1;

    if ((self->skybg_fit_frame || self->skybg_frame || self->insbg_fit_frame || self->insbg_frame) && self->dark_frame)
	      cpl_frame_delete(self->dark_frame); // delete dark because it's going to be re-assigne below

    if(self->skybg_fit_frame){
       	self->dark_frame = cpl_frame_duplicate(self->skybg_fit_frame);
       	cpl_msg_info(cpl_func,"Using SKY_BG_FIT frame as background!");
    }
    else {
    	if(self->skybg_frame){
    		self->dark_frame = cpl_frame_duplicate(self->skybg_frame);
    		cpl_msg_info(cpl_func,"Using SKY_BG frame as background!");
           	}
    	else {
    		if(self->insbg_fit_frame){
    			self->dark_frame = cpl_frame_duplicate(self->insbg_fit_frame);
    			cpl_msg_info(cpl_func,"Using INS_BG_FIT frame as background!");
    		}
           	else{
           		if(self->insbg_frame){
           			self->dark_frame = cpl_frame_duplicate(self->insbg_frame);
           		    cpl_msg_info(cpl_func,"Using INS_BG frame as background!");
           		}
           		else {
           			if(self->dark_frame){
           				if(master_dark_present)
           					cpl_msg_info(cpl_func,"Using master dark as background!");
           			}
           			else {
           				sph_error_raise( SPH_IRD_SCIENCE_IMAGING_FRAMES_MISSING,
                                __FILE__, __func__, __LINE__ ,
                                SPH_ERROR_ERROR,
           				                         "No dark or background whatsoever supplied! "
           				                         "Must have one of %s, %s, %s, %s, or %s!",
           				                         SPH_IRD_TAG_INS_BG_FIT_CALIB,
           				                         SPH_IRD_TAG_INS_BG_CALIB,
           				                         SPH_IRD_TAG_SKY_BG_FIT_CALIB,
           				                         SPH_IRD_TAG_SKY_BG_CALIB,
           				                         SPH_IRD_TAG_DARK_CALIB);
           				cpl_ensure_code(0, cpl_error_get_code());

           			}

           		}
           	}
    	}
    }
    const char * raw_fname = cpl_frame_get_filename(cpl_frameset_get_first(self->rawframes));
    pli = cpl_propertylist_load(raw_fname, 0); // read template property list of first frame to copy singular keys later

    if (self->transform_method == 0) {
        if (self->filter_method == 0) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_NONE, 0.0,
                    0.0);
        } else if (self->filter_method == 1) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH,
                    self->filter_radius, 0.0);
        } else if (self->filter_method == 2) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_FERMI,
                    self->filter_radius, self->fermi_temp);
        } else if (self->filter_method == 3) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_BUTTER,
                    self->butter_pass, self->butter_stop);
        }
    } else if (self->transform_method == 1) {
        transform = sph_transform_new_cpl_warp();
    }

    sci = sph_ird_common_science_new(self->inframes, self->inparams,
            self->dark_frame, self->flat_frame, self->static_badpixel_frame,
            self->distmap_frame, NULL, self->flat_frame,
            fctable_frames,
            SPH_RECIPE_NAME_IRD_SCIENCE_IMAGING,
            SPH_IRD_TAG_SCIENCE_IMAGING_CALIB);
    if (sci == NULL) {
        (void)cpl_error_set_where(cpl_func);
        goto EXIT;
    }

    /*
    if (self->tff_frame) {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Setting telescope flat %s. "
        "Note that this flat is applied after "
        "shifting/de-rotation", cpl_frame_get_filename( self->tff_frame ));
        sph_ird_common_science_set_tff(sci, self->tff_frame);
    }*/
    sci->flag_sdi = 0;
    sci->rotate_flag = 0;

    if (transform)
        sph_ird_common_science_set_transform(sci, transform);

    if (self->fcasciis != NULL) {
        fctable_frames = cpl_frameset_duplicate(self->fcasciis);
        if (fctable_frames == NULL) {
            goto EXIT;
        }
    } else {
        if (self->fcframe) {
            fctable_frames = sph_ird_common_science_create_fctables(
                    self->rawframes, self->fcframe, self->keep_fctable, 1);
            if (fctable_frames == NULL) {
                goto EXIT;
            }
        }
    }

    if (sph_ird_common_science_process_cubes(sci, self->rawframes,
            fctable_frames, &left_frameset_out, &right_frameset_out,
                                             &total_frameset,
                                             NULL, NULL, NULL) == CPL_ERROR_NONE) {

        if (self->flag_adi) {
            angles = sph_common_science_extract_angles_from_fctables(
                    fctable_frames);

            adi_frameset = sph_simple_adi_process_cubes(total_frameset,
                    self->rawframes, angles, sci->transform, sci->irdmodel, 0,
                    1, 2, self->fullset);
            result = sph_common_science_combine(adi_frameset, self->coll_alg, 0,
                    1, -1);
            cpl_frameset_delete(adi_frameset);

            adi_frameset = sph_simple_adi_process_cubes(left_frameset_out,
                    self->rawframes, angles, sci->transform, sci->irdmodel, 0,
                    1, 2, self->fullset);
            leftFOV = sph_common_science_combine(adi_frameset, self->coll_alg,
                    0, 1, -1);
            cpl_frameset_delete(adi_frameset);

            adi_frameset = sph_simple_adi_process_cubes(right_frameset_out,
                    self->rawframes, angles, sci->transform, sci->irdmodel, 0,
                    1, 2, self->fullset);
            rightFOV = sph_common_science_combine(adi_frameset, self->coll_alg,
                    0, 1, -1);
            cpl_frameset_delete(adi_frameset);
            cpl_vector_delete(angles);
            angles = NULL;
        } else {
            result = sph_common_science_combine(total_frameset, self->coll_alg,
                    0, 1, -1);
            leftFOV = sph_common_science_combine(left_frameset_out,
                    self->coll_alg, 0, 1, -1);
            rightFOV = sph_common_science_combine(right_frameset_out,
                    self->coll_alg, 0, 1, -1);
        }

        if (result) {
            cpl_errorstate okstate = cpl_errorstate_get();
            cpl_image * img_left = sph_master_frame_extract_image(leftFOV, 1);
            cpl_image * img_right = sph_master_frame_extract_image(rightFOV, 1);
            plist = cpl_propertylist_new();
            plist_left = cpl_propertylist_new();
            plist_right = cpl_propertylist_new();
            double flux_left = 0;
            double flux_right = 0;
            const cpl_boolean strehl_failed = self->filter_frame != NULL &&
            		calc_flux_and_possibly_strehl(plist_left, plist_right, img_left, img_right,
                                 cpl_frameset_get_first(self->rawframes),
                                 self->filter_frame,
                                 "science_imaging", self->inparams,
								 &flux_left, &flux_right);
            if (strehl_failed) {
                cpl_msg_warning(cpl_func, "Strehl estimation failed:");
                cpl_errorstate_dump(okstate, CPL_FALSE,
                                    cpl_errorstate_dump_one_warning);
                cpl_errorstate_set(okstate);
            } else {
            	cpl_propertylist_append(plist, plist_left);
            	cpl_propertylist_append(plist, plist_right);
            }

            cpl_image_delete(img_left);
            cpl_image_delete(img_right);

            if (!cpl_errorstate_is_equal(okstate)) {
                cpl_msg_warning(cpl_func, "Strehl estimation error reset:");
                cpl_errorstate_dump(okstate, CPL_FALSE,
                                    cpl_errorstate_dump_one_warning);
                cpl_errorstate_set(okstate);
            }

            cpl_propertylist* pl_phot_right = cpl_propertylist_new();
            cpl_propertylist* pl_phot_left = cpl_propertylist_new();
            cpl_error_code phot_failed = CPL_ERROR_NONE;

            if(!strehl_failed)
            	phot_failed = sph_phot_irdis(pl_phot_left, pl_phot_right,
            						 flux_left, flux_right, pli,
                                     cpl_frameset_get_first(self->rawframes),
                                     self->star_frame,
                                     self->filter_frame);

            if (!phot_failed) {
                cpl_propertylist_append(plist, pl_phot_left);
                cpl_propertylist_append(plist, pl_phot_right);
                cpl_propertylist_append(plist_right, pl_phot_right);
                cpl_propertylist_append(plist_left, pl_phot_left);
            } else {
                const char * target =
                    cpl_propertylist_has(pli, SPH_COMMON_TARGET_NAME_KEY) &&
                    cpl_propertylist_get_type(pli, SPH_COMMON_TARGET_NAME_KEY) ==
                    CPL_TYPE_STRING
                    ? cpl_propertylist_get_string(pli, SPH_COMMON_TARGET_NAME_KEY)
                    : "<missing>";
                cpl_boolean no_error = cpl_errorstate_is_equal(okstate);

                cpl_msg_warning(cpl_func, "Photometry computation failed (%s)! "
                                "It is ok if this target (%s) is not a "
                                "standard star...",
                                no_error ? "without error" : "with error(s)",
                                target);
                if (!no_error) {
                    cpl_errorstate_dump(okstate, CPL_FALSE,
                                        cpl_errorstate_dump_one_warning);
                    cpl_errorstate_set(okstate);
                }
            }

            cpl_propertylist_delete(pl_phot_right);
            cpl_propertylist_delete(pl_phot_left);

        	sph_utils_simple_copy_singular(pli, plist);
        	sph_utils_simple_copy_singular(pli, plist_left);
        	sph_utils_simple_copy_singular(pli, plist_right);

        	sph_ird_common_science_put_wcs(plist,512.0,512.0);
        	sph_ird_common_science_put_wcs(plist_left,512.0,512.0);
        	sph_ird_common_science_put_wcs(plist_right,512.0,512.0);


        	sph_master_frame_save_dfs(result, self->science_imaging_outfilename,
                    self->inframes, cpl_frameset_get_first(self->rawframes),
                    self->inparams, SPH_IRD_TAG_SCIENCE_IMAGING_CALIB,
                    SPH_RECIPE_NAME_IRD_SCIENCE_IMAGING,
                    SPH_PIPELINE_NAME_IRDIS, plist);
        } else {
            SPH_ERR("Could not create product frame.");
        }
        if (leftFOV) {
        	if (self->save_addprod) {
               sph_master_frame_save_dfs(leftFOV,
                    self->science_imaging_outfilename_left, self->inframes,
                    cpl_frameset_get_first(self->rawframes), self->inparams,
                    SPH_IRD_TAG_SCIENCE_IMAGING_CALIB_LEFT,
                    SPH_RECIPE_NAME_IRD_SCIENCE_IMAGING,
                    SPH_PIPELINE_NAME_IRDIS, plist_left);
        	}
        }
        if (rightFOV) {
        	if (self->save_addprod) {
                sph_master_frame_save_dfs(rightFOV,
                    self->science_imaging_outfilename_right, self->inframes,
                    cpl_frameset_get_first(self->rawframes), self->inparams,
                    SPH_IRD_TAG_SCIENCE_IMAGING_CALIB_RIGHT,
                    SPH_RECIPE_NAME_IRD_SCIENCE_IMAGING,
                    SPH_PIPELINE_NAME_IRDIS, plist_right);
        	}
        }
    }
    EXIT: sph_master_frame_delete(result);
    result = NULL;
    sph_master_frame_delete(leftFOV);
    result = NULL;
    sph_master_frame_delete(rightFOV);
    result = NULL;
    cpl_propertylist_delete(plist);
    cpl_propertylist_delete(plist_left);
	cpl_propertylist_delete(plist_right);
    cpl_propertylist_delete(pli);
    cpl_frameset_delete(total_frameset);
    cpl_frameset_delete(left_frameset_out);
    cpl_frameset_delete(right_frameset_out);
    cpl_frameset_delete(adi_frameset);
    sph_ird_common_science_delete(sci);
    sph_filemanager_clean();
    return (int) cpl_error_get_code();
}

static cpl_boolean _sph_ird_science_imaging_plist_contains_value(const cpl_propertylist * list,
        const char * key, const char * expected_value){

    if(!cpl_propertylist_has(list, key)) return CPL_FALSE;
    const char * content = cpl_propertylist_get_string(list, key);

    return strcmp(content, expected_value) == 0;
}


static cpl_boolean
_sph_ird_science_imaging_is_ao_enabled_from_list(const cpl_propertylist * list){

    const char * ao_keywords[] = {"ESO AOS HOLOOP STATE",
            "ESO AOS IRLOOP STATE", "ESO AOS PUPLOOP STATE",
             "ESO AOS TTLOOP STATE"};

    const char * expected_ao_value = "CLOSED_LOOP";

    cpl_size sz = sizeof(ao_keywords) / sizeof(ao_keywords[0]);

    for(cpl_size i = 0; i < sz; ++i){
        if(!_sph_ird_science_imaging_plist_contains_value(list,
                ao_keywords[i], expected_ao_value)) return CPL_FALSE;
    }

    return CPL_TRUE;
}


static cpl_boolean is_ao_enabled(const cpl_frame * raw_frame){

    cpl_propertylist * list = cpl_propertylist_load(cpl_frame_get_filename(raw_frame), 0);
    const cpl_boolean to_ret = _sph_ird_science_imaging_is_ao_enabled_from_list(list);
    cpl_propertylist_delete(list);
    return to_ret;
}

static cpl_error_code
calc_flux_and_possibly_strehl(cpl_propertylist  * plist_left,
                            cpl_propertylist  * plist_right,
                            const cpl_image   * img_left,
                            const cpl_image   * img_right,
                            const cpl_frame   * raw_frame,
                            const cpl_frame   * filter_frame,
                            const char        * recipe,
                            cpl_parameterlist * parlist,
                            double            * flux_left,
                            double            * flux_right){

    const cpl_boolean ao_enabled = is_ao_enabled(raw_frame);

    if(ao_enabled)
        cpl_msg_info(cpl_func, "Adaptive optics is enabled, STREHL ratio computation will be attempted");
    else
        cpl_msg_info(cpl_func, "Adaptive optics is disabled, only zeropoint computation will be attempted");

    if(ao_enabled)
        return sph_strehl_irdis_and_append(plist_left, plist_right, img_left, img_right,
                raw_frame,
                filter_frame,
                recipe, parlist,
                flux_left, flux_right);

    double irrelevant;
    cpl_error_code fail =  sph_strehl_disabled_ao_flux_irdis(img_left, raw_frame, parlist, recipe, flux_left, &irrelevant);
    if(!fail)
        fail = sph_strehl_disabled_ao_flux_irdis(img_right, raw_frame, parlist, recipe, flux_right, &irrelevant);

    return fail;
}

/**@}*/

