/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_wave_calib.h"
#include "sph_ird_instrument_model.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_keyword_manager.h"
#include "sph_cube.h"
#include "sph_fitting.h"
#include "sph_utils.h"
#include "sph_filemanager.h"
#include "sph_dataset_stats.h"
#include "sph_framecombination.h"
#include <string.h>
#include <strings.h>
#include <stdio.h>

static sph_error_code
sph_ird_wave_calib_load_data__(sph_ird_wave_calib* self);

static
int sph_ird_wave_calib_set_calib_wavelengths__(sph_ird_wave_calib* self);

static cpl_vector*
sph_ird_wave_calib_spectrum_fit_wavs__(sph_ird_wave_calib* self,
        cpl_vector* peaks, int np, cpl_vector** pdiffs, cpl_vector** pderiv,
        cpl_vector* coeffs, double* chisq);

static sph_error_code
sph_ird_wave_calib_transform_to_fitcoords__(sph_ird_wave_calib* self,
        cpl_vector* xpoints_in_out, cpl_vector* lambdas_in_out, double lam0,
        double y0);

static
int
sph_vector_find_unsorted__(cpl_vector* v, double l);

static sph_error_code
sph_ird_wave_calib_qc_wavelengths__(sph_ird_wave_calib* self,
        sph_spectral_region* reg, cpl_vector* peaks, cpl_vector* lam,
        cpl_vector* rms);

static sph_error_code
sph_ird_wave_calib_read_wavs_from_inskeys__(sph_ird_wave_calib* self);
static sph_error_code
sph_ird_wave_calib_fill_pdt__(sph_ird_wave_calib* self, char side, int specid);
static sph_pixel_description_table*
sph_ird_wave_calib_create_wave_tab__(sph_ird_wave_calib* self);
static
double
sph_ird_wave_calib_eval_poly__(sph_ird_wave_calib* self, cpl_polynomial* poly,
        int yy, double lam0, double y0, double *pdl);

static cpl_vector*
sph_ird_wave_calib_find_linepeaks__(sph_master_frame* mframe,
        sph_spectral_region* region, double threshold);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_wave_calib_run IRDIS wavelength calibration recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master wavelength calibration file
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_wave_calib.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_wave_calib recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_wave_calib_run(sph_ird_wave_calib* self) {
	cpl_propertylist* extpl = NULL;
	cpl_propertylist* pli   = NULL;

	pli = cpl_propertylist_load(
	                    cpl_frame_get_filename(cpl_frameset_get_first(self->rawframes)),
	                    0); // read template property list of first frame to copy singular keys later


    sph_ird_wave_calib_load_data__(self);

    if (self->use_inskeys)
        sph_ird_wave_calib_read_wavs_from_inskeys__(self);

    self->qclist = cpl_propertylist_new();

    if (self->grism_mode) {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "Grism mode is ON (Prism mode OFF). "
                "Setting c2 = c3 = c4 = 0.");
    } else {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "Grism mode is OFF (Prism mode ON)");
    }

    self->pdt = sph_ird_wave_calib_create_wave_tab__(self);

    if (self->pdt) {
        sph_utils_simple_copy_singular(pli, self->qclist);

        sph_pixel_description_table_save_dfs(self->pdt, self->outfilename, NULL,
                self->inframes, NULL, self->inparams,
                SPH_IRD_TAG_WAVE_CALIB_CALIB, SPH_RECIPE_NAME_IRD_WAVE_CALIB,
                SPH_PIPELINE_NAME_IRDIS, self->qclist);
    } else {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_OUTPUT,
                "Could not create the PDT.");
    }
    if (self->rawmframe) {
    	extpl = cpl_propertylist_new();
    	cpl_propertylist_update_string(extpl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_IMAGE_EXTNAME);
        if (self->pdt) {
            cpl_image_save(self->rawmframe->image, self->outfilename,
                    CPL_TYPE_DOUBLE, extpl, CPL_IO_EXTEND);
        } else {
            cpl_image_save(self->rawmframe->image, self->outfilename,
                    CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        }
        cpl_propertylist_update_string(extpl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_BADPIXMAP_EXTNAME);
        cpl_image_save(self->rawmframe->badpixelmap, self->outfilename,
                CPL_TYPE_DOUBLE, extpl, CPL_IO_EXTEND);
        cpl_propertylist_update_string(extpl,SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_MASTER_FRAME_RMSMAP_EXTNAME);
        cpl_image_save(self->rawmframe->rmsmap, self->outfilename,
                CPL_TYPE_DOUBLE, extpl, CPL_IO_EXTEND);
        cpl_propertylist_delete(extpl);
        extpl=NULL;
    }

    sph_pixel_description_table_delete(self->pdt);
    self->pdt = NULL;
    sph_master_frame_delete(self->dark);
    self->dark = NULL;
    sph_master_frame_delete(self->bgmframe);
    self->bgmframe = NULL;
    sph_master_frame_delete(self->rawmframe);
    self->rawmframe = NULL;
    sph_master_frame_delete(self->flat);
    self->flat = NULL;
    sph_ird_instrument_model_delete(self->model);
    self->model = NULL;
    if (pli) {
    	cpl_propertylist_delete(pli);
    	pli=NULL;
    }
    cpl_propertylist_delete(self->qclist);
    self->qclist = NULL;
    cpl_vector_delete(self->calib_wavelengths);
    self->calib_wavelengths = NULL;
    cpl_image_delete(self->maskimage);
    self->maskimage = NULL;

    //sph_filemanager_clean();
    return (int) cpl_error_get_code();
}

static sph_error_code sph_ird_wave_calib_load_data__(sph_ird_wave_calib* self) {
    cpl_propertylist* plist = NULL;
    if (self->master_dark_frame) {
        self->dark = sph_master_frame_load_(self->master_dark_frame, 0);
    }
    if (self->master_flat_frame) {
        self->flat = sph_master_frame_load_(self->master_flat_frame, 0);
    }
    if (self->mask_frame) {
        self->maskimage = cpl_image_load(
                cpl_frame_get_filename(self->mask_frame), CPL_TYPE_INT, 0, 0);
    }
    if (self->bg_raw) {
        self->bgmframe = sph_master_frame_load_(self->bg_raw, 0);
        if (self->dark) {
            sph_master_frame_subtract_master_frame(self->bgmframe, self->dark);
        }
    }

    plist = sph_keyword_manager_load_properties(
            cpl_frame_get_filename(self->master_flat_frame), 0);

    self->model = sph_ird_instrument_model_new_from_propertylist(plist);
    cpl_propertylist_delete(plist);
    plist = NULL;
    sph_ird_wave_calib_set_calib_wavelengths__(self);

    if (cpl_vector_get_size(self->calib_wavelengths) < self->fitorder + 1) {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_ILLEGAL_INPUT,
                "The fitorder is too high (%d) -- only have %d lines.", self->fitorder, (int)cpl_vector_get_size(self->calib_wavelengths));
        cpl_ensure_code(0, CPL_ERROR_ILLEGAL_INPUT);
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static sph_pixel_description_table*
sph_ird_wave_calib_create_wave_tab__(sph_ird_wave_calib* self) {
    sph_master_frame* mframe = NULL;
    int left_ok = 0;
    int right_ok = 0;
    int left_index = 0;
    int right_index = 0;
    int ii = 0;
    cpl_image* regim = NULL;

    mframe = sph_framecombination_master_frame_from_cpl_frameset(
            self->rawframes, self->coll_alg, self->framecomb_parameterlist);

    cpl_ensure(mframe, cpl_error_get_code(), NULL);

    self->rawmframe = mframe;

    if (self->bgmframe)
        sph_master_frame_subtract_master_frame(mframe, self->bgmframe);
    if (self->dark)
        sph_master_frame_subtract_master_frame(mframe, self->dark);
    if (self->flat)
        sph_master_frame_divide_master_frame(mframe, self->flat);
    if (self->maskimage) {
        sph_master_frame_set_bads_from_image(mframe, self->maskimage);
        if (self->flat) {
            sph_master_frame_set_bads_from_image(self->flat, self->maskimage);
        }
    }

    if (self->flat) {
        regim = sph_master_frame_get_badpixelmap(self->flat);
    } else if (self->maskimage) {
        regim = cpl_image_duplicate(self->maskimage);
    } else {
        SPH_ERROR_RAISE_ERR( CPL_ERROR_ILLEGAL_INPUT,
                "Need either a flat or a maskimage in input!");
        sph_master_frame_delete(mframe);
        mframe = NULL;
        self->rawmframe = NULL;
        return NULL;
    }
    // make sure bad pixels are below 0 and goods are 0...
    cpl_image_multiply_scalar(regim, -1.0);

    // regions are all pixels above -0.5
    self->pdt = sph_pixel_description_table_new_from_image(regim, 0.0, 0.0,
            -0.5);

    sph_pixel_description_table_setup_regions(self->pdt);
    for (ii = 0; ii < self->pdt->nregions; ++ii) {
        if (self->pdt->regions[ii]->maxx < self->pdt->nx / 2) {
            if (self->pdt->regions[ii]->maxx - self->pdt->regions[ii]->minx
                    > self->pdt->nx / 4) {
                left_index = ii;
            }
        } else {
            if (self->pdt->regions[ii]->maxx - self->pdt->regions[ii]->minx
                    > self->pdt->nx / 4) {
                right_index = ii;
            }
        }
    }

    cpl_ensure(self->pdt, CPL_ERROR_ILLEGAL_OUTPUT, NULL);

    if (self->smooth > 0.0) {
        sph_master_frame_interpolate_bpix(self->rawmframe);
        sph_master_frame_smooth(self->rawmframe, self->smooth);
        sph_master_frame_set_bads_from_image(self->rawmframe, regim);
    }

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Processing left side...");
    left_ok = sph_ird_wave_calib_fill_pdt__(self, 'L', left_index + 1);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Processing right side...");
    right_ok = sph_ird_wave_calib_fill_pdt__(self, 'R', right_index + 1);
    if (left_ok != CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_ERR(
                SPH_ERROR_GENERAL,
                "Could not perform correct wavelength calibration for left side.");
        sph_pixel_description_table_delete(self->pdt);
        self->pdt = NULL;
    }
    if (right_ok != CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_ERR(
                SPH_ERROR_GENERAL,
                "Could not perform correct wavelength calibration for right side.");
        sph_pixel_description_table_delete(self->pdt);
        self->pdt = NULL;
    }
    cpl_image_delete(regim);
    regim = NULL;
    return self->pdt;
}

static sph_error_code sph_ird_wave_calib_fill_pdt__(sph_ird_wave_calib* self,
        char side, int specid) {
    cpl_vector* pdiffs = NULL;
    cpl_vector* pderiv = NULL;
    cpl_vector* wavs = NULL;
    sph_spectral_region* reg = NULL;
    sph_spectral_region* dumreg = NULL;
    cpl_array* coeffs0 = NULL;
    cpl_array* coeffs1 = NULL;
    cpl_array* chi = NULL;
    int nbadlines = 0;
    int nbadfit = 0;
    int ncount = 0;
    char stri[256];
    int xx;
    cpl_vector* coeffs = NULL;
    cpl_vector* peaks = NULL;
    cpl_vector* lam = NULL;
    cpl_vector* rms = NULL;
    double chisq = 0.0;
    int ii = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;cpl_ensure_code(
            self->fitorder > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code( self->pdt, CPL_ERROR_NULL_INPUT);

    dumreg = sph_spectral_region_new();
    reg = sph_pixel_description_table_get_region(self->pdt, specid);
    dumreg->specid = reg->specid;
    dumreg->lensid = reg->lensid;
    for (xx = reg->minx; xx <= reg->maxx; xx += self->column_w) {
        ncount++;
    }
    coeffs0 = cpl_array_new(ncount, CPL_TYPE_DOUBLE);
    coeffs1 = cpl_array_new(ncount, CPL_TYPE_DOUBLE);
    chi = cpl_array_new(ncount, CPL_TYPE_DOUBLE);
    ncount = 0;
    for (xx = reg->minx; xx <= reg->maxx; xx += self->column_w) {
        dumreg->minx = xx;
        dumreg->maxx =
                xx + self->column_w > reg->maxx ?
                        reg->maxx : xx + self->column_w;
        dumreg->miny = reg->miny;
        dumreg->maxy = reg->maxy;
        self->pdt->regions[specid - 1] = dumreg;

        peaks = sph_ird_wave_calib_find_linepeaks__(self->rawmframe, dumreg,
                self->threshold);

        if (peaks) {
            if (cpl_vector_get_size(peaks) == self->number_lines) {
                cpl_vector_sort(peaks, CPL_SORT_DESCENDING);

                coeffs = cpl_vector_new(self->fitorder + 1);

                chisq = 0.0;

                wavs = sph_ird_wave_calib_spectrum_fit_wavs__(self, peaks,
                        dumreg->maxy - dumreg->miny, &pdiffs, &pderiv, coeffs,
                        &chisq);

                if (wavs) {
                    if (sph_pixel_description_table_set_spectra_wavelengths(
                            self->pdt, specid, wavs, 0) != CPL_ERROR_NONE) {
                        SPH_ERROR_RAISE_WARNING(
                                SPH_ERROR_GENERAL,
                                "Could not set the wavelengths for spectral region.");
                        nbadfit++;
                    } else {
                        cpl_array_set_double(coeffs0, ncount,
                                dumreg->miny + cpl_vector_get(coeffs, 0));
                        cpl_array_set_double(coeffs1, ncount,
                                cpl_vector_get(coeffs, 1));
                        cpl_array_set_double(chi, ncount, chisq);

                        sph_keyword_manager_update_double_char_int(self->qclist,
                                SPH_IRD_KEYWORD_WAVECALIB_Y0, side, ncount + 1,
                                dumreg->miny + cpl_vector_get(coeffs, 0));
                        sph_keyword_manager_update_double_char_int(self->qclist,
                                SPH_IRD_KEYWORD_WAVECALIB_C1, side, ncount + 1,
                                cpl_vector_get(coeffs, 1));
                        sph_keyword_manager_update_double_char_int(self->qclist,
                                SPH_IRD_KEYWORD_WAVECALIB_CHI, side, ncount + 1,
                                chisq);
                        ncount++;
                        cpl_vector_delete(pdiffs);
                        pdiffs = NULL;
                        cpl_vector_delete(pderiv);
                        pderiv = NULL;
                    }
                    cpl_vector_delete(wavs);
                } else {
                    SPH_ERROR_RAISE_WARNING(
                            SPH_ERROR_GENERAL,
                            "Could not fit the wavelengths for spectral region.");
                    cpl_array_set_invalid(coeffs0, ncount);
                    cpl_array_set_invalid(coeffs1, ncount);
                    cpl_array_set_invalid(chi, ncount);
                    nbadfit++;
                    ncount++;
                }

                lam = cpl_vector_duplicate(self->calib_wavelengths);
                rms = cpl_vector_duplicate(self->calib_wavelengths);

                sph_ird_wave_calib_qc_wavelengths__(self, dumreg, peaks, lam,
                        rms);

                for (ii = 0; ii < cpl_vector_get_size(lam); ++ii) {
                    sph_keyword_manager_update_double_char_int2(self->qclist,
                            SPH_IRD_KEYWORD_WAVECALIB_QC_LAM, side, ncount + 1,
                            ii + 1, cpl_vector_get(lam, ii));
                    sph_keyword_manager_update_double_char_int2(self->qclist,
                            SPH_IRD_KEYWORD_WAVECALIB_QC_LAMRMS, side,
                            ncount + 1, ii + 1, cpl_vector_get(rms, ii));
                }

                cpl_vector_delete(lam);
                lam = NULL;
                cpl_vector_delete(rms);
                rms = NULL;
                cpl_vector_delete(coeffs);
                coeffs = NULL;
                cpl_vector_delete(peaks);
                peaks = NULL;
            } else {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_GENERAL,
                        "Found not the right number of lines."
                        "Got %d, expected %d."
                        "Can not process this column!", (int)cpl_vector_get_size(peaks), self->number_lines);
                cpl_array_set_invalid(coeffs0, ncount);
                cpl_array_set_invalid(coeffs1, ncount);
                cpl_array_set_invalid(chi, ncount);
                nbadlines++;
                ncount++;
            }
        } else {
            SPH_ERROR_RAISE_WARNING(SPH_ERROR_GENERAL,
                    "Could not extract spectrum around x = %d", xx);
            cpl_array_set_invalid(coeffs0, ncount);
            cpl_array_set_invalid(coeffs1, ncount);
            cpl_array_set_invalid(chi, ncount);
            nbadlines++;
            ncount++;
        }
        self->pdt->regions[specid - 1] = reg;
    }
    if (ncount) {
        sprintf(stri, "%s%c", SPH_IRD_KEYWORD_WAVECALIB_Y0_MEAN, side);
        sph_keyword_manager_update_double_char(self->qclist,
                SPH_IRD_KEYWORD_WAVECALIB_Y0_MEAN, side,
                cpl_array_get_mean(coeffs0));
        if (ncount > 1)
            sph_keyword_manager_update_double_char(self->qclist,
                    SPH_IRD_KEYWORD_WAVECALIB_Y0_RMS, side,
                    cpl_array_get_stdev(coeffs0));

        sph_keyword_manager_update_double_char(self->qclist,
                SPH_IRD_KEYWORD_WAVECALIB_C1_MEAN, side,
                cpl_array_get_mean(coeffs1));
        if (ncount > 1)
            sph_keyword_manager_update_double_char(self->qclist,
                    SPH_IRD_KEYWORD_WAVECALIB_C1_RMS, side,
                    cpl_array_get_stdev(coeffs1));
        if (ncount > 1)
            SPH_ERROR_RAISE_INFO(
                    SPH_ERROR_GENERAL,
                    "Obtained the fit values of Y0 = %5.3f+/-%4.3f.", cpl_array_get_mean(coeffs0), cpl_array_get_stdev(coeffs0))
        else
            SPH_ERROR_RAISE_INFO(
                    SPH_ERROR_GENERAL,
                    "Obtained the fit values of Y0 = %5.3f.", cpl_array_get_mean(coeffs0));

        if (ncount > 1)
            SPH_ERROR_RAISE_INFO(
                    SPH_ERROR_GENERAL,
                    "Obtained the fit values of C1 = %5.3f+/-%4.3f.", cpl_array_get_mean(coeffs1), cpl_array_get_stdev(coeffs1))
        else
            SPH_ERROR_RAISE_INFO(
                    SPH_ERROR_GENERAL,
                    "Obtained the fit values of C1 = %5.3f", cpl_array_get_mean(coeffs1));

        sph_keyword_manager_update_int_char(self->qclist,
                SPH_IRD_KEYWORD_WAVECALIB_NGOODLINES, side, ncount);
        sph_keyword_manager_update_int_char(self->qclist,
                SPH_IRD_KEYWORD_WAVECALIB_NBADLINES, side, nbadlines);
        sph_keyword_manager_update_int_char(self->qclist,
                SPH_IRD_KEYWORD_WAVECALIB_NNOFITLINES, side, nbadfit);
    }
    cpl_array_delete(coeffs0);
    coeffs0 = NULL;
    cpl_array_delete(coeffs1);
    coeffs1 = NULL;
    cpl_array_delete(chi);
    chi = NULL;
    sph_spectral_region_delete(dumreg);
    dumreg = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static cpl_vector*
sph_ird_wave_calib_spectrum_fit_wavs__(sph_ird_wave_calib* self,
        cpl_vector* peaks, int np, cpl_vector** pdiffs, cpl_vector** pderiv,
        cpl_vector* coeffs, double* chisq) {
    cpl_vector* result = NULL;
    cpl_vector* calib_lines_transformed = NULL;
    cpl_vector* xpoints_transformed = NULL;
    int ii = 0;
    int nlines = 0;
    cpl_vector* xpoints = NULL;
    cpl_polynomial* poly = NULL;
    double val = 0.0;
    double dl = 0.0;
    double y0 = 0.0;
    cpl_size pows[1];
    double lambda0 = 1545.07;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    cpl_ensure(peaks, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(np > 10, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(self->calib_wavelengths, CPL_ERROR_NULL_INPUT, NULL);

    if (coeffs) {
        cpl_ensure( cpl_vector_get_size(coeffs) == self->fitorder + 1,
                CPL_ERROR_ILLEGAL_INPUT, NULL);
    }

    nlines = cpl_vector_get_size(self->calib_wavelengths);

    if (nlines < 3 || nlines > 8) {
        SPH_ERR("Need between 3 and 8 lines to "
        "calibrated spectrum.");
        return NULL;
    }
    y0 = cpl_vector_get(peaks,
            sph_vector_find_unsorted__(self->calib_wavelengths, lambda0));

    xpoints_transformed = cpl_vector_duplicate(peaks);
    calib_lines_transformed = cpl_vector_duplicate(self->calib_wavelengths);
    //cpl_vector_dump( xpoints_transformed, stdout );

    sph_ird_wave_calib_transform_to_fitcoords__(self, xpoints_transformed,
            calib_lines_transformed, lambda0, y0);

    //cpl_vector_dump( xpoints_transformed, stdout );
    //cpl_vector_dump( calib_lines_transformed, stdout );

    poly = sph_fitting_fit_poly1d(xpoints_transformed, calib_lines_transformed,
            NULL, 1, self->fitorder, 0, 0.0, chisq);

    if (!poly) {
        SPH_ERROR_RAISE_ERR(cpl_error_get_code(), "Could not perform fit.");
    } else {
        result = cpl_vector_new(np);
        if (pdiffs) {
            *pdiffs = cpl_vector_new(np);
        }
        if (pderiv) {
            *pderiv = cpl_vector_new(np);
        }
        for (ii = 0; ii < np; ++ii) {
            val = sph_ird_wave_calib_eval_poly__(self, poly, ii, lambda0, y0,
                    &dl);
            cpl_vector_set(result, ii, val);
            if (pderiv)
                cpl_vector_set(*pderiv, ii, dl);
        }
    }
    cpl_vector_delete(xpoints_transformed);
    xpoints_transformed = NULL;
    cpl_vector_delete(calib_lines_transformed);
    calib_lines_transformed = NULL;

    if (coeffs) {
        for (ii = 0; ii <= self->fitorder; ++ii) {
            pows[0] = ii;
            cpl_vector_set(coeffs, ii, cpl_polynomial_get_coeff(poly, pows));
        }
        pows[0] = 0;
        cpl_vector_set(coeffs, 0, y0);
    }

    if (xpoints)
        cpl_vector_delete(xpoints);
    xpoints = NULL;
    cpl_polynomial_delete(poly);
    poly = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

static sph_error_code sph_ird_wave_calib_qc_wavelengths__(
        sph_ird_wave_calib* self, sph_spectral_region* reg, cpl_vector* peaks,
        cpl_vector* lam, cpl_vector* rms) {
    int pp = 0;
    int xx = 0;
    double pos = 0;
    double dpos = 0;
    double val = 0.0;
    int ipos = 0;
    cpl_vector* lams = NULL;
    sph_pixel_descriptor* descr = NULL;
    sph_pixel_descriptor* descr2 = NULL;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( peaks, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( reg, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( lam, CPL_ERROR_NULL_INPUT);

    for (pp = 0; pp < cpl_vector_get_size(peaks); ++pp) {
        lams = cpl_vector_new(reg->maxx - reg->minx);
        for (xx = reg->minx; xx < reg->maxx; ++xx) {
            pos = cpl_vector_get(peaks, pp);
            ipos = (int) pos;
            dpos = pos - ipos;
            descr = sph_pixel_description_table_get_descriptor(self->pdt, xx,
                    ipos + reg->miny);
            descr2 = sph_pixel_description_table_get_descriptor(self->pdt, xx,
                    ipos + 1 + reg->miny);
            if (descr && descr2) {
                val = (1.0 - dpos) * descr->wavelength
                        + dpos * descr2->wavelength;
                cpl_vector_set(lams, xx - reg->minx, val);
            }
        }

        cpl_vector_set(lam, pp, cpl_vector_get_mean(lams));

        if (rms) {
            cpl_vector_set(
                    rms,
                    pp,
                    cpl_vector_get_mean(lams)
                            - cpl_vector_get(self->calib_wavelengths, pp));
        }
        cpl_vector_delete(lams);
        lams = NULL;

    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
int sph_ird_wave_calib_set_calib_wavelengths__(sph_ird_wave_calib* self) {
    cpl_vector* lambdas = NULL;

    if (self->number_lines < 2) {
        sph_error_raise(CPL_ERROR_ILLEGAL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Need to have at least 2 calibration wavelengths! "
                        "But number of lines was %d", self->number_lines);
        goto ERROR_EXIT;
    }

    self->calib_wavelengths = cpl_vector_new(self->number_lines);
    lambdas = self->calib_wavelengths;

    if (self->wavelength_line1 > 0.0) {
        cpl_vector_set(lambdas, 0, self->wavelength_line1);
    } else {
        sph_error_raise(CPL_ERROR_ILLEGAL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Need to have at least 2 calibration wavelengths! "
                        "But wavelength of line 1 was %lf",
                self->wavelength_line1);
        goto ERROR_EXIT;
    }
    if (self->wavelength_line2 > 0.0) {
        cpl_vector_set(lambdas, 1, self->wavelength_line2);
    } else {
        sph_error_raise(CPL_ERROR_ILLEGAL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Need to have at least 2 valid calibration wavelengths! "
                        "But wavelength of line 2 was %lf",
                self->wavelength_line1);
        goto ERROR_EXIT;
    }
    if (self->wavelength_line3 > 0.0 && self->number_lines > 2) {
        cpl_vector_set(lambdas, 2, self->wavelength_line3);
    }
    if (self->wavelength_line4 > 0.0 && self->number_lines > 3) {
        cpl_vector_set(lambdas, 3, self->wavelength_line4);
    }
    if (self->wavelength_line5 > 0.0 && self->number_lines > 4) {
        cpl_vector_set(lambdas, 4, self->wavelength_line5);
    }
    if (self->wavelength_line6 > 0.0 && self->number_lines > 5) {
        cpl_vector_set(lambdas, 5, self->wavelength_line6);
    }

    return CPL_ERROR_NONE;

    ERROR_EXIT: if (self->calib_wavelengths) {
        cpl_vector_delete(self->calib_wavelengths);
        self->calib_wavelengths = NULL;
    }
    return cpl_error_get_code();

}

static cpl_vector*
sph_ird_wave_calib_find_linepeaks__(sph_master_frame* mframe,
        sph_spectral_region* region, double threshold) {
    sph_dataset* dataset = NULL;
    int npix = 0;
    sph_dataset_stats* datstat = NULL;
    cpl_vector* peaks = NULL;
    double median = 0.0;
    double mean = 0.0;
    double threshval = 0.0;
    cpl_image* threshimage = NULL;
    cpl_image* winim = NULL;

    cpl_ensure(mframe, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(region, CPL_ERROR_NULL_INPUT, NULL);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    threshimage = sph_master_frame_extract_image(mframe, 1);
    cpl_image_fill_rejected(threshimage, 0.0);

    winim = cpl_image_extract(threshimage, region->minx + 1, region->miny + 1,
            region->maxx, region->maxy);

    cpl_ensure(winim, cpl_error_get_code(), NULL);
    npix = cpl_image_get_size_y(winim);
    median = cpl_image_get_median(winim);
    mean = cpl_image_get_mean(winim);

    if (threshold > 0) {
        threshval = threshold;
    } else {
        threshval = median + 0.5 * (mean - median);
    }
    cpl_image_threshold(winim, threshval, FLT_MAX, 0.0, FLT_MAX);
#ifdef SPH_SAVE_NONDFS
    cpl_image_save(winim, "threshimage.fits", CPL_TYPE_FLOAT, NULL,
            CPL_IO_CREATE);
#endif
    dataset = sph_dataset_new(10, 0.0, 10.0);
    sph_dataset_collapse(dataset, winim, 0.0, npix - 1);
    datstat = sph_dataset_stats_new(dataset);
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Finding lines with threshold %f, median being %f.", threshval, median);
    sph_dataset_stats_find_peaks(datstat, threshval, 3);
    peaks = sph_dataset_stats_get_peaks_as_cpl_vector(datstat);

    sph_dataset_stats_delete(datstat);
    cpl_image_delete(threshimage);
    cpl_image_delete(winim);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;cpl_ensure(
            cpl_vector_sort(peaks,-1) == CPL_ERROR_NONE, cpl_error_get_code(),
            NULL);
    return peaks;
}

static sph_error_code sph_ird_wave_calib_transform_to_fitcoords__(
        sph_ird_wave_calib* self, cpl_vector* xpoints_in_out,
        cpl_vector* lambdas_in_out, double lam0, double y0) {
    double lambda = 0.0;
    double lambdat = 0.0;
    double dy = 0.0;
    double c2 = 0.0;
    double c3 = 0.0;
    double c4 = 0.0;
    cpl_vector* lambda_vector = NULL;
    cpl_vector* xpoints_vector = NULL;
    int ii = 0;
    double y = 0.0;

    if (!self->grism_mode) {
        c2 = self->c2;
        c3 = self->c3;
        c4 = self->c4;
    }
    xpoints_vector = cpl_vector_duplicate(xpoints_in_out);
    lambda_vector = cpl_vector_duplicate(lambdas_in_out);
    for (ii = 0; ii < cpl_vector_get_size(xpoints_vector); ++ii) {
        y = cpl_vector_get(xpoints_vector, ii);
        lambda = cpl_vector_get(lambda_vector, ii);
        dy = (y - y0) / 100.0;
        lambdat = lambda - lam0 - c2 * dy * dy - c3 * dy * dy * dy
                - c4 * dy * dy * dy * dy;
        cpl_vector_set(lambdas_in_out, ii, lambdat);
        cpl_vector_set(xpoints_in_out, ii, dy);
    }

    cpl_vector_delete(xpoints_vector);
    xpoints_vector = NULL;
    cpl_vector_delete(lambda_vector);
    lambda_vector = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
static
double sph_ird_wave_calib_eval_poly__(sph_ird_wave_calib* self,
        cpl_polynomial* poly, int yy, double lam0, double y0, double *pdl) {
    double dy = 0.0;
    double val = 0.0;
    double c2 = 0.0;
    double c3 = 0.0;
    double c4 = 0.0;

    if (!self->grism_mode) {
        c2 = self->c2;
        c3 = self->c3;
        c4 = self->c4;
    }

    dy = (yy - y0) / 100.0;
    val = cpl_polynomial_eval_1d(poly, dy, pdl);
    val = val + lam0 + c2 * dy * dy + c3 * dy * dy * dy
            + c4 * dy * dy * dy * dy;
    if (pdl) {
        *pdl = *pdl + lam0 + c2 * dy * dy + c3 * dy * dy * dy
                + c4 * dy * dy * dy * dy;
    }
    return val;
}

static
int sph_vector_find_unsorted__(cpl_vector* v, double l) {
    int ii = 0;
    int bestind = -1;
    double bestdiff = 1e10;

    cpl_ensure(v, CPL_ERROR_NULL_INPUT, -1);

    for (ii = 0; ii < cpl_vector_get_size(v); ++ii) {
        if (fabs(cpl_vector_get(v, ii) - l) < bestdiff) {
            bestdiff = fabs(cpl_vector_get(v, ii) - l);
            bestind = ii;
        }
    }
    return bestind;
}

static sph_error_code sph_ird_wave_calib_read_wavs_from_inskeys__(
        sph_ird_wave_calib* self) {
    cpl_propertylist* plist = NULL;
    cpl_propertylist* plist_dup = NULL;
    int n = 0;
    const char* sensname = NULL;
    const char* sensid = NULL;
    double tmp_wavelengths_array[10];
    int integer_wavelength;
    int lid = 0;
    int sensnum = 0;
    char sensornamekey[256];
    cpl_frame* aframe = NULL;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->rawframes, CPL_ERROR_NULL_INPUT);

    aframe = cpl_frameset_get_first(self->rawframes);

    plist = sph_keyword_manager_load_properties(cpl_frame_get_filename(aframe),
            0);
    cpl_ensure_code(plist, cpl_error_get_code());
    plist_dup = cpl_propertylist_duplicate(plist);
    cpl_propertylist_erase_regexp(plist, "ESO INS4 SENS.* ID", 1);
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Reading INS sensor propertylist from %s."
            "Found %d keywords in total and %d SENS keywords.", cpl_frame_get_filename(aframe), (int)cpl_propertylist_get_size(plist_dup), (int)cpl_propertylist_get_size(plist));
    if (cpl_propertylist_get_size(plist) > 0) {
        for (n = 0; n < cpl_propertylist_get_size(plist); ++n) {
            sensid = cpl_propertylist_get_string(plist,
                    cpl_property_get_name(cpl_propertylist_get(plist, n)));
            if (strncasecmp(sensid, "CBL", 3) == 0) {
                if (sensid[4] == 'I') {
                    sensnum = 0;
                    sensname = cpl_property_get_name(
                            cpl_propertylist_get(plist, n));
                    if (sensname) {
                        sscanf(sensname, "ESO INS4 SENS%d ID", &sensnum);
                    }
                    sprintf(sensornamekey, "ESO INS4 SENS%d NAME", sensnum);
                    sensname = cpl_propertylist_get_string(plist_dup,
                            sensornamekey);

                    if (sensname) {
                        sscanf(sensname, "Laser %d -", &integer_wavelength);
                        tmp_wavelengths_array[lid++] =
                                ((double) integer_wavelength);
                    }
                }
            }
        }
    }

    cpl_propertylist_delete(plist_dup);
    plist_dup = NULL;
    cpl_propertylist_delete(plist);
    plist = NULL;

    if (lid != self->number_lines) {
        SPH_ERROR_RAISE_WARNING(
                SPH_ERROR_GENERAL,
                "Could not find all the necessary INS keywords in "
                "file %s (found %d of %d). "
                "Will ignore request to use INS keywords for "
                "wavelengths and use esorex command line "
                "parameters instead.", cpl_frame_get_filename(aframe), lid, self->number_lines);
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
    }

    for (n = 0; n < lid; ++n) {
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "Setting wavelength %d from INS keywords to %f.", n, tmp_wavelengths_array[n]);
        if (n == 0)
            self->wavelength_line1 = tmp_wavelengths_array[n];
        if (n == 1)
            self->wavelength_line2 = tmp_wavelengths_array[n];
        if (n == 2)
            self->wavelength_line3 = tmp_wavelengths_array[n];
        if (n == 3)
            self->wavelength_line4 = tmp_wavelengths_array[n];
        if (n == 4)
            self->wavelength_line5 = tmp_wavelengths_array[n];
        if (n == 5)
            self->wavelength_line6 = tmp_wavelengths_array[n];
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/**@}*/
