/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * sph_zpl_framegroupe.c
 *
 *  Created on: Apr 11, 2013
 *      Author: pavlov
 */


#include "sph_zpl_framegroups.h"
#include "sph_zpl_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_error.h"

#include <string.h>
#include <cpl.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_framegroups Groupes of two zpl frames (plus and minus)
 *
 * @par Descirption:
 * This module provides the functionality to create grope of frames from
 * a given input frameset based one the two keywords (plus, minus). This is used by recipes
 * to create groupes of frames for +Q and -Q or +U and -U.
 * So, the frames are grouped into 2 sets with similar properties as
 * identified by the values of the given keyword.
 *
 * @par Synopsis:
 * @code
 * typedef struct _sph_zpl_framegroups_
 * {
 *   cpl_frameset*  frameplus;
 *   cpl_frameset*  frameminus;
 *   char*		   valueplus;
 *   char*		   valueminus;
 *   char*          keyword;
 * } sph_zpl_framegroupes;
 *
 * @endcode
 *
 */

sph_error_code SPH_FRAMEGROUPS_FRAMES_GENERAL = SPH_FRAMEGROUPS_ERR_START + 0;
sph_error_code SPH_FRAMEGROUPS_FRAMES_MISSING = SPH_FRAMEGROUPS_ERR_START + 1;

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new empty framegroups
 *
 * @return pointer to newly created framegroups or NULL on error
 *
 * This creates a new empty framegroups.
 */
/*----------------------------------------------------------------------------*/

sph_zpl_framegroups* sph_zpl_framegroups_new(void) {
    sph_zpl_framegroups* self = (sph_zpl_framegroups*)cpl_malloc(sizeof(*self));

    self->framesplus = NULL;
    self->framesminus = NULL;
    self->propertyplus = NULL;
    self->propertyminus = NULL;

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set frames as plus frames
 *
 */
/*----------------------------------------------------------------------------*/

sph_error_code sph_zpl_framegroups_set_framesplus(sph_zpl_framegroups* self,
        cpl_frameset* framesplus) {
    sph_error_code rerr = CPL_ERROR_NONE;

    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(framesplus, CPL_ERROR_NULL_INPUT);

    self->framesplus = cpl_frameset_duplicate(framesplus);
    if (self->framesplus == NULL) {
        SPH_ERR("Can't duplicate framesplus!");
        rerr = sph_error_get_last_code();
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set frames as plus frames
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_zpl_framegroups_set_framesminus(sph_zpl_framegroups* self,
        cpl_frameset* framesminus) {
    sph_error_code rerr = CPL_ERROR_NONE;

    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(framesminus, CPL_ERROR_NULL_INPUT);

    self->framesminus = cpl_frameset_duplicate(framesminus);
    if (self->framesminus == NULL) {
        SPH_ERR("Can't duplicate framesminus!");
        rerr = sph_error_get_last_code();
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create new zimpol stokes-groups of frames based on the given input
 frames.

 @param    inframes     The input frameset
 @param    czKeyword    The keyword name whose value is to be used
 @param 	kvalue_plus  The keyword value of the "plus" stokes parameter
 @param    kvalue_minus The keyword value of the "minus" stokes parameter
 @return   Pointer to the new created sph_zpl_framegroups or null in the case of error.

 Using the input frameset new zimpol stokes-groups of frames is created where the input
 frameset is filtered according to the criteria specified by the two stokes parameters
 keyword value. All frames that have keyword defined with the given name and have
 the keyword values of the specified values are selected in plus and minus set
 (framesplus and framesminus).
 */
/*----------------------------------------------------------------------------*/
sph_zpl_framegroups* sph_zpl_framegroups_create(const cpl_frameset* inframes,
        const char* czKeyword, const char* kvalue_plus,
        const char* kvalue_minus) {
    sph_zpl_framegroups* results;
    const cpl_frame* curframe = NULL;
    cpl_propertylist* plist = NULL;
    cpl_error_code rerr = CPL_ERROR_NONE;

    cpl_ensure( inframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( czKeyword, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( kvalue_plus, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( kvalue_minus, CPL_ERROR_NULL_INPUT, NULL);

    results = sph_zpl_framegroups_new();

    curframe = cpl_frameset_get_first_const(inframes);
    if (!curframe) {
        sph_error_raise(SPH_FRAMEGROUPS_FRAMES_MISSING, __FILE__, __func__,
                __LINE__, SPH_ERROR_ERROR,
                "No framegroups are created: cant find any input frames.");
        sph_zpl_framegroups_delete(results);
        return NULL;
    }

    results->propertyplus = cpl_property_new(czKeyword, CPL_TYPE_STRING);
    cpl_property_set_string(results->propertyplus, kvalue_plus);

    results->propertyminus = cpl_property_new(czKeyword, CPL_TYPE_STRING);
    cpl_property_set_string(results->propertyminus, kvalue_minus);

    results->framesplus = cpl_frameset_new();
    results->framesminus = cpl_frameset_new();

    while (curframe) {
        plist = sph_keyword_manager_load_properties(
                cpl_frame_get_filename(curframe), 0);

        if (!plist) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not read the propertylist of frame %s.",
                    cpl_frame_get_filename(curframe));
            sph_zpl_framegroups_delete(results);
            return NULL;
        }

        if (!cpl_propertylist_has(plist, czKeyword)) {
            sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING,
                    "Keyword %s is not presented in the current frame %s",
                    czKeyword, cpl_frame_get_filename(curframe));
            if (plist) {
                cpl_propertylist_delete(plist);
                plist = NULL;
            }
            curframe = cpl_frameset_get_next_const(inframes);
            continue;
        }
        if (!strcmp(cpl_propertylist_get_string(plist, czKeyword),
                kvalue_plus)) {

            rerr = cpl_frameset_insert(results->framesplus,
                    cpl_frame_duplicate(curframe));
            if (rerr) {
                SPH_ERR("cant insert frame into the frameplus set");
                if (plist) {
                    cpl_propertylist_delete(plist);
                    plist = NULL;
                }
                sph_zpl_framegroups_delete(results);
                return NULL;
            }
        }
        if (!strcmp(cpl_propertylist_get_string(plist, czKeyword),
                kvalue_minus)) {

            rerr = cpl_frameset_insert(results->framesminus,
                    cpl_frame_duplicate(curframe));
            if (rerr) {
                SPH_ERR("cant insert frame into the frameminus group");
                if (plist) {
                    cpl_propertylist_delete(plist);
                    plist = NULL;
                }
                sph_zpl_framegroups_delete(results);
                return NULL;
            }
        }
        if (plist) {
            cpl_propertylist_delete(plist);
            plist = NULL;
        }
        curframe = cpl_frameset_get_next_const(inframes);
    }

    if (cpl_frameset_is_empty(results->framesminus)
            && cpl_frameset_is_empty(results->framesplus)) {
        SPH_WARNING(
                "no groups were created from the given keyword: framesets (frameminus and frameplus) are empty!");
        if (plist) {
            cpl_propertylist_delete(plist);
            plist = NULL;
        }
        sph_zpl_framegroups_delete(results);
        return NULL;
    }

    SPH_ERROR_RAISE_INFO(
            SPH_FRAMEGROUPS_FRAMES_GENERAL,
            "plus-stokes group contains %d frames", (int) cpl_frameset_get_size(results->framesplus));
    SPH_ERROR_RAISE_INFO(
            SPH_FRAMEGROUPS_FRAMES_GENERAL,
            "minus-stokes group contains %d frames", (int) cpl_frameset_get_size(results->framesminus));

    if ((cpl_frameset_get_size(results->framesplus)
            + cpl_frameset_get_size(results->framesminus))
            != cpl_frameset_get_size(inframes)) {
        SPH_ERROR_RAISE_WARNING(
                SPH_FRAMEGROUPS_FRAMES_GENERAL,
                "The size of the all frames in the groups is different from the size of the inframes which is %d", 
                (int) cpl_frameset_get_size(inframes));
    }

    if (cpl_frameset_is_empty(results->framesminus)) {
        cpl_frameset_delete(results->framesminus);
        results->framesminus = NULL;
    }
    if (cpl_frameset_is_empty(results->framesplus)) {
        cpl_frameset_delete(results->framesplus);
        results->framesplus = NULL;
    }

    return results;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create new zimpol stokes-groups of frames for the +/-Q frames.

 @param    inframes     The input frameset
 @return   Pointer to the new created sph_zpl_framegroups or null in the case of error.

 All frames that have keyword defined with the given name and have
 the keyword values of the default values for +/-Q are selected in plus and minus set
 (framesplus and framesminus).
 */
/*----------------------------------------------------------------------------*/
sph_zpl_framegroups* sph_zpl_framegroups_create_Q(const cpl_frameset* inframes) {
    return sph_zpl_framegroups_create(inframes, SPH_ZPL_KEYWORD_POL_STOKES_NAME,
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QPLUS,
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QMINUS);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create new zimpol stokes-groups of frames for the +/-U frames.

 @param    inframes     The input frameset
 @return   Pointer to the new created sph_zpl_framegroups or null in the case of error.

 All frames that have keyword defined with the given name and have
 the keyword values of the default values for +/-U are selected in plus and minus set
 (framesplus and framesminus).
 */
/*----------------------------------------------------------------------------*/
sph_zpl_framegroups* sph_zpl_framegroups_create_U(const cpl_frameset* inframes) {
    return sph_zpl_framegroups_create(inframes, SPH_ZPL_KEYWORD_POL_STOKES_NAME,
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_UPLUS,
            SPH_ZPL_KEYWORD_VALUE_POL_STOKES_UMINUS);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the framegroups
 @param    self        the framegroups to delete
 @return   void

 Deallocate the framegroups, the associated framesets and the corresponding values.
 */
/*----------------------------------------------------------------------------*/
void sph_zpl_framegroups_delete(sph_zpl_framegroups* self) {

    if (self != NULL) {
        cpl_frameset_delete(self->framesminus);
        cpl_frameset_delete(self->framesplus);
        cpl_property_delete(self->propertyminus);
        cpl_property_delete(self->propertyplus);
        cpl_free(self);
    }
}
