/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <math.h>
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_badpixelco.h"

/*********************************
 * Static function declarations
 ********************************/
static
int sph_badpixelco_set_tmp_value(
		const cpl_image* badpix_image,
		cpl_size badpix_col_coord,
		cpl_size badpix_row_coord,
		const cpl_image* image_to_correct,
		double* tmp,
		int countpix,
		double medpix,
		double medpix_other,
		int override);
static
int sph_badpixelco_set_tmp_col(int countpix, const cpl_image* badpix_image,
		cpl_size badpix_col_coord, cpl_size badpix_row_coord,
		const cpl_image* image_to_correct, double* tmp, double medpix,
		double medpix_left,
		int override);



/*********************************
 * Main function
 ********************************/

/*----------------------------------------------------------------------------*/
/**
 * @brief Correct badpixels in a raw cube
 * @param allframes			the complete set of frames used
 * @param raw_cube_frame 	the raw cube to correct
 * @param border			the number of pixels as border
 * @param thresh_factor     factor for median threshold
 * @param thresh_absolute   absolte median threshold
 *
 * @return error code
 *
 * Description:
 * This corrects the raw cube using the badpixel correction
 * as implemented in Dino Mesas IDL script.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_badpixelco_apply_to_raw_cube(
		cpl_frameset* allframes,
		cpl_frame* raw_cube_frame,
		cpl_size border,
		double thresh_factor,
		double thresh_absolute)
{
	cpl_imagelist* resultlist = cpl_imagelist_new();
	cpl_imagelist* imlist = NULL;
	cpl_frameset* usedframes = NULL;
	cpl_parameterlist* parlist = NULL;
	cpl_propertylist* applist = NULL;

	cpl_ensure_code(allframes, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(raw_cube_frame, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(thresh_factor >= 0.0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(thresh_absolute >= 0.0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(border >= 0, CPL_ERROR_ILLEGAL_INPUT);

	imlist = cpl_imagelist_load(
				cpl_frame_get_filename(raw_cube_frame),
				CPL_TYPE_UNSPECIFIED, 0);

	cpl_ensure_code(imlist, cpl_error_get_code());

	for ( cpl_size plane = 0; plane < cpl_imagelist_get_size(imlist); ++plane )
	{
		cpl_image* image = cpl_imagelist_get(imlist,plane);

		if ( image ) {
			cpl_image* corrected_image = sph_badpixelco_correct_image(border,
					thresh_factor, thresh_absolute, image);
 		cpl_imagelist_set(resultlist, corrected_image, plane);
		}
	}

	usedframes = cpl_frameset_new();
	cpl_frameset_insert(usedframes,cpl_frame_duplicate(raw_cube_frame));

	parlist = cpl_parameterlist_new();
	applist = cpl_propertylist_new();
	cpl_propertylist_append_string(applist,CPL_DFS_PRO_CATG,"IFS BADPIXELCORRECTED");

	cpl_dfs_save_imagelist(allframes,NULL,parlist,usedframes,raw_cube_frame,resultlist,CPL_TYPE_DOUBLE,"TESTRECIPE",applist,NULL,"SPHERE","test.fits");

	cpl_imagelist_delete(resultlist);
	cpl_imagelist_delete(imlist);
	cpl_parameterlist_delete(parlist);
	cpl_propertylist_delete(applist);
	cpl_frameset_delete(usedframes);
	SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Correct badpixels in a single image in place
 * @param border			the number of pixels as border
 * @param thresh_factor     factor for median threshold
 * @param thresh_absolute   absolte median threshold
 * @param image             the input image (unmodified)
 *
 * @return cpl_error_code
 *
 * Description:
 * This corrects the input image with the badpixel correction
 * as implemented in Dino Mesas IDL script.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_badpixelco_correct_image_in_place(sph_raw_image_corrector_data* data,cpl_image** image)
{
	// TODO: should really have an inherit in-place method that does not need to
	//       create a new image.
	cpl_image* corrected_image = sph_badpixelco_correct_image(data->badpixco_border,
			data->badpixco_thresh_factor, data->badpixco_thresh_absolute, *image);
	cpl_image_delete(*image);
	*image = corrected_image;
	SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Correct badpixels in a single image
 * @param border			the number of pixels as border
 * @param thresh_factor     factor for median threshold
 * @param thresh_absolute   absolte median threshold
 * @param image             the input image (unmodified)
 *
 * @return the corrected image
 *
 * Description:
 * This corrects the input image with the badpixel correction
 * as implemented in Dino Mesas IDL script.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image* sph_badpixelco_correct_image(cpl_size border,
		double thresh_factor, double thresh_absolute,
		cpl_image* image)
{
	cpl_size x,y,nx,ny;
	cpl_vector* medians = NULL;
	cpl_image* corrected_image = NULL;

	cpl_ensure(thresh_factor >= 0.0, CPL_ERROR_ILLEGAL_INPUT,NULL);
	cpl_ensure(thresh_absolute >= 0.0, CPL_ERROR_ILLEGAL_INPUT,NULL);
	cpl_ensure(border >= 0, CPL_ERROR_ILLEGAL_INPUT,NULL);

	medians = sph_badpixelco_get_medians_for_rows(image, border);
	corrected_image = cpl_image_duplicate(image);

	nx = cpl_image_get_size_x(image);
	ny = cpl_image_get_size_y(image);
        for (y = 1; y < ny - 1; ++y) {
            for (x = 1; x < nx - 1; ++x) {
			sph_badpixelco_correct_single_pixel_entire_row(image, medians,
					thresh_factor, thresh_absolute, x + 1, y + 1,
					corrected_image);
		}
	}
	cpl_vector_delete(medians);
	medians = NULL;
	return corrected_image;
}

/*********************************
 * Helper functions
 * These are used by the main function
 * above, but should not normally
 * be used otherwise.
 ********************************/
cpl_vector*
sph_badpixelco_get_medians_for_rows( cpl_image* image, cpl_size border ) {
	cpl_size min = border;
	cpl_size max = cpl_image_get_size_x(image) - border;

	cpl_vector* result = NULL;

	cpl_ensure(min >=0, CPL_ERROR_ILLEGAL_INPUT, NULL);
	cpl_ensure(max > min, CPL_ERROR_ILLEGAL_INPUT, NULL);

	result = cpl_vector_new( cpl_image_get_size_y(image) );

	for ( cpl_size row = 0; row < cpl_image_get_size_y(image); ++row ) {
		cpl_vector_set( result, row, cpl_image_get_median_window(image,min + 1,row + 1,max + 1,row + 1) );
	}

	return result;
}
cpl_error_code
sph_badpixelco_correct_single_pixel_entire_row(
		const cpl_image* image_to_correct,
		const cpl_vector* medians,
		double thresh_factor,
		double thresh_absolute,
		cpl_size x,
		cpl_size y,
		cpl_image* result_image)
{
	double median = 0.0;
	int badpix = 0;
	cpl_vector* tmp = cpl_vector_new(8);

	cpl_ensure_code( image_to_correct != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code( medians != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code( result_image != NULL, CPL_ERROR_NULL_INPUT);

	median = cpl_vector_get(medians,y);

	cpl_vector_set( tmp, 0,
			cpl_image_get(image_to_correct, x - 1, y - 1, &badpix) *
				median / cpl_vector_get(medians, y - 2) );
	cpl_vector_set( tmp, 1,
			cpl_image_get(image_to_correct, x, y - 1, &badpix) *
				median / cpl_vector_get(medians, y - 2) );
	cpl_vector_set( tmp, 2,
			cpl_image_get(image_to_correct, x + 1, y - 1, &badpix) *
				median / cpl_vector_get(medians,y - 2) );
	cpl_vector_set( tmp, 3,
			cpl_image_get(image_to_correct, x - 1, y, &badpix) *
				median / cpl_vector_get(medians, y - 1) );
	cpl_vector_set( tmp, 4,
			cpl_image_get(image_to_correct, x + 1, y, &badpix) *
				median / cpl_vector_get(medians,y - 1) );
	cpl_vector_set( tmp, 5,
			cpl_image_get(image_to_correct, x - 1, y + 1, &badpix) *
				median / cpl_vector_get(medians, y) );
	cpl_vector_set( tmp, 6,
			cpl_image_get(image_to_correct, x, y + 1, &badpix) *
				median / cpl_vector_get(medians, y) );
	cpl_vector_set( tmp, 7,
			cpl_image_get(image_to_correct, x + 1, y + 1, &badpix) *
				median / cpl_vector_get(medians,y) );

	if ( cpl_image_get( image_to_correct, x, y, &badpix) > thresh_factor * cpl_vector_get_median(tmp) &&
		 cpl_image_get( image_to_correct, x, y, &badpix) > thresh_absolute )
	{
		cpl_image_set( result_image, x, y, cpl_vector_get_median(tmp));
	}
	cpl_vector_delete(tmp); tmp = NULL;
	SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}



cpl_error_code
sph_badpixelco_correct_single_pixel(
		const cpl_image* badpix_image,
		const cpl_image* image_to_correct,
		cpl_size badpix_row_coord,
		cpl_size badpix_col_coord,
		cpl_image* result_image)
{
	cpl_ensure_code( badpix_image != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code( image_to_correct != NULL, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code( result_image != NULL, CPL_ERROR_NULL_INPUT);
	{
		double* tmp = cpl_calloc(8,sizeof(double));
		int countpix = 0;
		cpl_size size = cpl_image_get_size_x(image_to_correct);
		cpl_size miny = 101;
		cpl_size maxy = size - (miny - 1);

		double medpix =  cpl_image_get_median_window(image_to_correct,badpix_col_coord,miny,badpix_col_coord,maxy);
		double medpix_left =  cpl_image_get_median_window(image_to_correct,badpix_col_coord-1,miny,badpix_col_coord-1,maxy);
		double medpix_right =  cpl_image_get_median_window(image_to_correct,badpix_col_coord+1,miny,badpix_col_coord+1,maxy);
		double newval = 0.0;

		countpix = sph_badpixelco_set_tmp_col(countpix, badpix_image,
				badpix_col_coord-1, badpix_row_coord, image_to_correct, tmp,
				medpix, medpix_left,0);
		countpix = sph_badpixelco_set_tmp_col(countpix, badpix_image,
				badpix_col_coord, badpix_row_coord, image_to_correct, tmp,
				medpix, medpix,0);
		countpix = sph_badpixelco_set_tmp_col(countpix, badpix_image,
				badpix_col_coord+1, badpix_row_coord, image_to_correct, tmp,
				medpix, medpix_right,0);
		if ( countpix == 0 ) {
			countpix = sph_badpixelco_set_tmp_col(countpix, badpix_image,
					badpix_col_coord-1, badpix_row_coord, image_to_correct, tmp,
					medpix, medpix_left, 1);
			countpix = sph_badpixelco_set_tmp_col(countpix, badpix_image,
					badpix_col_coord, badpix_row_coord, image_to_correct, tmp,
					medpix, medpix, 1);
			countpix = sph_badpixelco_set_tmp_col(countpix, badpix_image,
					badpix_col_coord+1, badpix_row_coord, image_to_correct, tmp,
					medpix, medpix_right, 1);
		}
		{
			cpl_vector* v = cpl_vector_wrap(8,tmp);
			newval = cpl_vector_get_median(v);
			cpl_vector_unwrap(v);
		}
		cpl_image_set(result_image,badpix_col_coord,badpix_row_coord,newval);
		cpl_free(tmp);
	}
	SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


cpl_size
sph_badpixelco_fill_cx_cy_vectors(
		const cpl_image* badimage,
		cpl_vector** cx,
		cpl_vector** cy )
{
	cpl_ensure( badimage != NULL, CPL_ERROR_NULL_INPUT, 0);
	cpl_ensure( cx != NULL, CPL_ERROR_NULL_INPUT, 0);
	cpl_ensure( cy != NULL, CPL_ERROR_NULL_INPUT, 0);
	{
		const int* dat = cpl_image_get_data_const(badimage);
		cpl_size c = 0;
		cpl_size size = cpl_image_get_size_x(badimage);
		cpl_size badcount = 0;

		for ( c = 0; c < size*size; ++c ) if (dat[c] == 1) badcount++;

		if ( badcount < 1 ) {
			*cx = cpl_vector_new(badcount + 1);
			*cy = cpl_vector_new(badcount + 1);
		}
		else {
			*cx = cpl_vector_new(badcount);
			*cy = cpl_vector_new(badcount);
			badcount = 0;
			for ( c = 0; c < size*size; ++c ) {
				if (dat[c] == 1) {
					cpl_size row = c / size;
					cpl_vector_set(*cy,badcount,(double)row);
					cpl_vector_set(*cx,badcount,(double)(c % size));
					badcount++;
				}
			}

		}

		return badcount;
	}
}

static
int sph_badpixelco_set_tmp_value(
		const cpl_image* badpix_image,
		cpl_size badpix_col_coord,
		cpl_size badpix_row_coord,
		const cpl_image* image_to_correct,
		double* tmp,
		int countpix,
		double medpix,
		double medpix_other,
		int override)
{
	int bpix = 0;
	if (override ||
			cpl_image_get(badpix_image, badpix_col_coord, badpix_row_coord,
					&bpix) != 1)
	{
		double value = cpl_image_get(image_to_correct, badpix_col_coord,
				badpix_row_coord, &bpix);
		tmp[countpix] = value * medpix / medpix_other;
		countpix++;
	}
	return countpix;
}

static
int sph_badpixelco_set_tmp_col(int countpix, const cpl_image* badpix_image,
		cpl_size badpix_col_coord, cpl_size badpix_row_coord,
		const cpl_image* image_to_correct, double* tmp, double medpix,
		double medpix_left,
		int override)
{
	countpix = sph_badpixelco_set_tmp_value(badpix_image, badpix_col_coord - 1,
			badpix_row_coord - 1, image_to_correct, tmp, countpix, medpix,
			medpix_left,override);
	countpix = sph_badpixelco_set_tmp_value(badpix_image, badpix_col_coord - 1,
			badpix_row_coord, image_to_correct, tmp, countpix, medpix,
			medpix_left,override);
	countpix = sph_badpixelco_set_tmp_value(badpix_image, badpix_col_coord - 1,
			badpix_row_coord + 1, image_to_correct, tmp, countpix, medpix,
			medpix_left,override);
	return countpix;
}
