/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_common_science.h"

#include "sph_error.h"
#include "sph_master_frame.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_filemanager.h"
#include "sph_keyword_manager.h"
#include "sph_fft.h"
#include "sph_fits.h"
#include "sph_fctable.h"
#include "sph_smart_imagelist.h"
#include "sph_zpl_exposure.h"
#include "sph_zpl_exposure_imaging.h"

#include <math.h>
#include <string.h>
#include <assert.h>

#define SPH_VAL_BADISBAD 0.000001
#define SPH_COMMON_SCIENCE_MAX_BADPIX 10000000
#define SPH_COMMON_SCIENCE_OK_BADPIX 100000


static cpl_frameset* sph_common_science_current_frameset__ = NULL;
static cpl_frame* sph_common_science_current_frame__ = NULL;
static int sph_common_science_iscube__ = 0;
static int sph_common_science_nplanes__ = 0;
static int sph_common_science_currentplane__ = 0;

static int sph_common_science_general_get_next_raw(cpl_frameset* rawframes, int ext);
static sph_error_code
sph_common_science_treat_weighted_masterframe( sph_master_frame* result, sph_master_frame* mframe);



static
sph_master_frame*
sph_common_science_create_masterframe__( cpl_imagelist* imlist,
        cpl_image* imim, cpl_image* bpixim, cpl_image* wim) {
    sph_master_frame*       result = NULL;

    const size_t nx = cpl_image_get_size_x(imim);
    const size_t ny = cpl_image_get_size_y(imim);
    const size_t nz = cpl_imagelist_get_size(imlist);

    result = sph_master_frame_new_empty();
    result->image = cpl_image_duplicate(imim);
    if ( bpixim ) {
        result->badpixelmap = cpl_image_duplicate(bpixim);
        result->ncombmap = cpl_image_cast(bpixim, CPL_TYPE_DOUBLE);
        cpl_image_multiply_scalar(result->ncombmap,-1.0);
        cpl_image_add_scalar(result->ncombmap,nz);
        //cpl_image_multiply_scalar(result->badpixelmap,0.0);
        cpl_image_threshold(result->badpixelmap,nz-0.5,nz-0.4,0.0,1.0);
    }
    else {
        result->badpixelmap = cpl_image_new(nx,
                ny,CPL_TYPE_INT);
        result->ncombmap = cpl_image_new(nx,
                ny,CPL_TYPE_DOUBLE);
        cpl_image_add_scalar(result->ncombmap,
                nz);
    }

    result->rmsmap = cpl_image_new(
            nx,
            ny,
            CPL_TYPE_DOUBLE);
    if ( wim ) {
        sph_master_frame_set_rms_from_weightmap(result,wim,1);
    } else {

        if (cpl_image_get_type(imim) == CPL_TYPE_DOUBLE &&
            cpl_image_get_type(result->rmsmap) == CPL_TYPE_DOUBLE &&
            cpl_image_get_type((cpl_imagelist_get_const(imlist, 0))) ==
            CPL_TYPE_DOUBLE) {
            const double*     pimim = cpl_image_get_data_double_const(imim);
            const cpl_binary* pbpmm = !cpl_image_get_bpm_const(imim) ? NULL
                : cpl_mask_get_data_const(cpl_image_get_bpm_const(imim));
            double*           prms  = cpl_image_get_data_double(result->rmsmap);
            /* The contribution counter. FIXME: Only needed with BPMs */
            cpl_image*        cci   = cpl_image_new(nx, ny, CPL_TYPE_INT);
            int*              pcci  = cpl_image_get_data_int(cci);

            /* Initialize rmsmap */
            (void)memset(prms, 0, nx * ny * sizeof(*prms));

            for (size_t zz = 0; zz < nz; zz++) {
                const cpl_image* imgi  = cpl_imagelist_get_const(imlist, zz);
                const double*    pimgi = cpl_image_get_data_double_const(imgi);
                const cpl_binary*pbpmi = !cpl_image_get_bpm_const(imgi) ? NULL
                    : cpl_mask_get_data_const(cpl_image_get_bpm_const(imgi));

                for (size_t ii = 0; ii < nx * ny; ii++) {
                    if ((!pbpmm || !pbpmm[ii]) &&
                        (!pbpmi || !pbpmi[ii])) {
                        const double delta =
                            pimgi[ii] -
                            pimim[ii];

                        prms[ii] += delta * delta;
                        pcci[ii]++;
                    }
                }
            }

            for (size_t ii = 0; ii < nx * ny; ii++) {
                if (pcci[ii] > 1) {
                    prms[ii] /= (double)pcci[ii] - 1.5;
                }
                prms[ii] = sqrt(prms[ii]);
            }
            cpl_image_delete(cci);
        } else {
            cpl_msg_warning(cpl_func, "Computing RMS for non-double data");

            for (size_t yy = 0; yy < ny; ++yy) {
                for (size_t xx = 0; xx < nx; ++xx) {
                    int          bpix = 0;
                    const double mean =
                        cpl_image_get(imim, xx + 1, yy + 1, &bpix);

                    if (bpix == 0) {
                        double rmstot = 0.0;
                        int cc = 0;
                        for (size_t zz = 0; zz < nz; ++zz) {
                            const double rms =
                                cpl_image_get(cpl_imagelist_get(imlist,zz),
                                              xx+1, yy+1, &bpix);
                            if ( bpix == 0 ) {
                                rmstot += (rms - mean ) * ( rms - mean );
                                cc++;
                            }
                        }
                        if (cc > 1)
                            rmstot /= (double)cc - 1.5;
                        cpl_image_set(result->rmsmap,xx+1,yy+1,sqrt(rmstot));
                    } else {
                        cpl_image_set(result->rmsmap, xx+1, yy+1, 0.0);
                    }
                }
            }
        }
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Remap the badpixels to rotated, shifted and distortion coords.
 * @param inbadbpix  the original badpixelmap
 * @param angle     the rotation angle
 * @param centx     rotation centre in pixel coordinates
 * @param centy     rotation centre in pixel coordinates
 * @param dx        shift in x
 * @param dy        shift in y
 * @param distmap   the polynomial distortion model to apply (maybe NULL)
 * @param irdis_model    IRDIS model, set to NULL for all other instruments.
 *
 * @return the new badpixel image or NULL on error.
 *
 * This function creates a new badpixel image of the same type as the input
 * image (usually CPL_TYPE_INT). It maps all pixels with values >= 1 to
 * a new image using the transformation specified (shift, rotation and,
 * if pointer to a model is given, distortion).
 * The transformation is purely geometric -- the pixel is transformed to
 * a polygon (rectangle) that is rotated and mapped onto the output image.
 * In the output image, all pixels that overlap the original bad pixel are
 * set to the value 1.
 *
 * The special input parameter "irdis_model" is should be set
 * if this function is to rotate a badpixel map from IRDIS in DBI or DPI mode,
 * as it correctly handles the badpixels at the edge of the images due to
 * the circular aperture. Set to NULL for any other instrument.
 *
 * Note that the function will not work correctly for
 * extreme distortions (with maximal distortions of more than a few pixels).
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_common_science_remap_badpix(cpl_image* inbadpix,
        double angle, double centx,
        double centy,double dx0, double dy0,
        sph_distortion_model* distmap,
        double      scale,
        sph_ird_instrument_model* irdmodel)
{
    const cpl_size      nx = cpl_image_get_size_x(inbadpix);
    const cpl_size      ny = cpl_image_get_size_y(inbadpix);
    cpl_vector*         badpixpos_x = NULL;
    cpl_vector*         badpixpos_y = NULL;
    cpl_image*          imout = NULL;
    cpl_image*          tmpim = NULL;
    int                 xx       = 0;
    int                 yy      = 0;
    int                 bpcount = 0;
    int                 bpix = 0;
    sph_polygon*        poly = NULL;
    sph_polygon*        poly2 = NULL;
    double              cosangle = cos(angle * CPL_MATH_RAD_DEG);
    double              sinangle = sin(angle * CPL_MATH_RAD_DEG);
    int                 ok = 1;
    double              dx = dx0;
    double              dy = dy0;
    static int          __already_warned = 0;
    sph_polygon_workspace*      ws = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure(inbadpix,CPL_ERROR_NULL_INPUT,NULL);
    imout = cpl_image_new(nx,
                          ny,
                          cpl_image_get_type(inbadpix));
    badpixpos_x = cpl_vector_new(SPH_COMMON_SCIENCE_MAX_BADPIX);
    badpixpos_y = cpl_vector_new(SPH_COMMON_SCIENCE_MAX_BADPIX);
    cpl_ensure(imout,CPL_ERROR_ILLEGAL_INPUT,NULL);

    ws = sph_polygon_workspace_new();

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            ok = 1;
            if ( irdmodel ) {
                ok = sph_ird_instrument_model_check_in_dbi_fov(irdmodel, xx, yy,
                                                               ok, 10.0);
            }
            if ( ok )
            {
                if ( cpl_image_get(inbadpix,xx+1,yy+1,&bpix) >= 1 ) {
                    cpl_vector_set(badpixpos_x,bpcount,xx * 1.0);
                    cpl_vector_set(badpixpos_y,bpcount,yy * 1.0);
                    bpcount++;
                    if ( bpcount >= SPH_COMMON_SCIENCE_MAX_BADPIX ) {
                        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                                "There are in total %d badpixels. "
                                "This recipe can not cope with such bad quality data."
                                "Please check your dark and flat quality.",
                                bpcount);
                        return NULL;
                    }
                }
            }
            else {
                // Dont do anything here -- pixels that are outside the
                // IRDIS FOV are not changed in their badpixel properties
                // and not rotated.
                cpl_image_set(imout,xx + 1, yy + 1, cpl_image_get(inbadpix,xx+1,yy+1,&bpix));
            }
        }
    }
    if ( bpcount >= SPH_COMMON_SCIENCE_OK_BADPIX && __already_warned == 0) {
        SPH_ERROR_RAISE_WARNING(SPH_ERROR_GENERAL,
                "There are in total %d badpixels. "
                "This is so large that the performance is suffering. "
                "I will continue, but please check your dark and flat quality.",bpcount);
        __already_warned = 1;
    }
    if ( bpcount > 0 ) {
        cpl_vector_set_size(badpixpos_x,bpcount);
        cpl_vector_set_size(badpixpos_y,bpcount);
        poly = sph_polygon_new();
        sph_polygon_add_point(poly,0.0,0.0);
        sph_polygon_add_point(poly,0.0,1.0);
        sph_polygon_add_point(poly,1.0,1.0);
        sph_polygon_add_point(poly,1.0,0.0);
        poly2 = sph_polygon_new();
        sph_polygon_add_point(poly2,0.0,0.0);
        sph_polygon_add_point(poly2,0.0,1.0);
        sph_polygon_add_point(poly2,1.0,1.0);
        sph_polygon_add_point(poly2,1.0,0.0);
        for (bpix = 0; bpix < cpl_vector_get_size(badpixpos_x); ++bpix) {
            double              midx = 0.0;
            double              midy = 0.0;
            poly->points[0].x = cpl_vector_get(badpixpos_x,bpix);
            poly->points[0].y = cpl_vector_get(badpixpos_y,bpix);
            poly->points[1].x = cpl_vector_get(badpixpos_x,bpix);
            poly->points[1].y = cpl_vector_get(badpixpos_y,bpix) + 1 - 0.00000000001;
            poly->points[2].x = cpl_vector_get(badpixpos_x,bpix) + 1 - 0.00000000001;
            poly->points[2].y = cpl_vector_get(badpixpos_y,bpix) + 1 - 0.00000000001;
            poly->points[3].x = cpl_vector_get(badpixpos_x,bpix) + 1 - 0.00000000001;
            poly->points[3].y = cpl_vector_get(badpixpos_y,bpix);
            sph_polygon_get_midxy_(poly, &midx, &midy);
            if ( distmap ) {
                sph_polygon_shift(poly,-centx,-centy);
                sph_distortion_model_apply_poly(distmap,poly);
                sph_polygon_shift(poly,centx,centy);
            }
            if ( scale != 1.0 && scale != 0.0 ) {
                dx = dx0 + (midx-centx) * ( scale - 1.0);
                dy = dy0 + (midy-centy) * ( scale - 1.0);
            }
            sph_polygon_rotate_around(poly,centx,centy,cosangle,sinangle,dx,dy);
            sph_polygon_get_midxy_(poly, &midx, &midy);
            for (yy = (int)midy-1; yy <= (int)midy+1; ++yy) {
                for (xx = (int)midx-1; xx <= (int)midx+1; ++xx) {
                    if (xx >= 0 &&
                            xx < cpl_image_get_size_x(imout) &&
                            yy >= 0 &&
                            yy < cpl_image_get_size_y(imout) ) {
                        double overlap;
                        poly2->points[0].x = xx;
                        poly2->points[0].y = yy;
                        poly2->points[1].x = xx;
                        poly2->points[1].y = yy + 1.0;
                        poly2->points[2].x = xx + 1.0;
                        poly2->points[2].y = yy + 1.0;
                        poly2->points[3].x = xx + 1.0;
                        poly2->points[3].y = yy;
                        overlap = sph_polygon_calculate_overlap(poly,poly2,ws);
                        if ( overlap > SPH_VAL_BADISBAD ) {
                            cpl_ensure(cpl_image_set(imout,xx+1,yy+1,1.0)==CPL_ERROR_NONE,cpl_error_get_code(),NULL);
                        }
                    }
                }
            }
        }
    }
    cpl_vector_delete(badpixpos_x); badpixpos_x = NULL;
    cpl_vector_delete(badpixpos_y); badpixpos_y = NULL;
    sph_polygon_delete(poly); poly = NULL;
    sph_polygon_delete(poly2); poly2 = NULL;
    sph_polygon_workspace_delete(ws); ws = NULL;
    if (tmpim ) {
        cpl_image_delete(tmpim);
    }
    sph_polygon_free_all();
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return imout;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Set badpixel weights in a list of weight images to zero
 *
 * @param  weights   the imagelist to set badpixels to zero in
 * @param  badpix    imagelist of badpixels
 *
 * @return error code of the operation
 *
 * This function takes two imagelists of the same size. It goes through all
 * images in both lists. For all pixels that are between 0.5 and INT_MAX in the
 * badpix imagelist the corresponding pixel value in the weights imagelist is
 * set to zero.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_common_science_set_weights_zero(cpl_imagelist* weights,
                                    const cpl_imagelist* badpixs)
{
    cpl_size    nz;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    cpl_ensure_code( weights != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( badpixs != NULL, CPL_ERROR_NULL_INPUT);
    nz = cpl_imagelist_get_size(weights);
    cpl_ensure_code( nz > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_imagelist_get_size(badpixs) == nz,
                    CPL_ERROR_ILLEGAL_INPUT );

    for (cpl_size ii = 0; ii < nz; ++ii) {
        cpl_mask*   mask = 
            cpl_mask_threshold_image_create(cpl_imagelist_get_const(badpixs, ii),
                                            0.5, INT_MAX);
        if ( mask ) {
            cpl_image* img = cpl_imagelist_get(weights, ii);
            cpl_image_reject_from_mask(img, mask);
            cpl_image_fill_rejected(img, 0.0 );
            cpl_image_accept_all(img);
            cpl_mask_delete(mask);
        }
    }

    return cpl_error_set_where(cpl_func);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Calculate the weights from a list of images
 * @param imlist        the list of images to create weightmaps for
 * @param badpixs       an optional image list of badpixel images (maybe NULL)
 * @param medim         a comparison image (usually median of imlist, maybe NULL)
 *
 * @return imagelist of weightmaps or NULL on error
 *
 * This function calculates the weightmaps corresponding to each of the
 * images in the input imagelist. For each image in the imagelist, the difference
 * to the medim image is taken (if medim is non NULL, otherwise medim is first
 * created as the median image of the input image list). This difference image
 * is then squared and the inverse is then stored in the resulting image list as
 * the weightmap of the image.
 * If the badpixs image list is provided, all pixels that are between 0.5 and INT_MAX
 * in the badpix image are set to zero in the corresponding weightmap.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_imagelist*
sph_common_science_calc_weights(cpl_imagelist* imlist,
                                cpl_imagelist* badpixs,
                                const cpl_image* medim)
{
    cpl_imagelist*   weights = NULL;
    cpl_image*       einsimage = NULL;
    cpl_image*       medimint  = NULL;
    const cpl_image* usemedim = medim;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    cpl_ensure( imlist , CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( cpl_imagelist_get_size(imlist) > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    if ( badpixs ) {
        cpl_ensure(cpl_imagelist_get_size(imlist) == cpl_imagelist_get_size(badpixs), CPL_ERROR_ILLEGAL_INPUT, NULL );
    }
    if (medim == NULL) {
        medimint = cpl_imagelist_collapse_median_create(imlist);
        usemedim = medimint;
    }
    weights = cpl_imagelist_new();

    einsimage = cpl_image_new(
            cpl_image_get_size_x(cpl_imagelist_get_const(imlist,0)),
            cpl_image_get_size_y(cpl_imagelist_get_const(imlist,0)),
            CPL_TYPE_DOUBLE);

    cpl_image_add_scalar(einsimage,1.0);

    for (int ii = 0; ii < cpl_imagelist_get_size(imlist); ++ii) {
        cpl_mask*           maskA = NULL;
        cpl_image*          wim = NULL;
        cpl_image*          diffim = NULL;
        diffim = cpl_image_duplicate(cpl_imagelist_get_const(imlist,ii));
        cpl_image_subtract(diffim,usemedim);
        cpl_image_multiply(diffim,diffim);
        if ( cpl_image_get_absflux(diffim) > 0.0 ) {
            wim = cpl_image_divide_create(einsimage,diffim);
        }
        else {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,"The "
                    "collapsed median image is identical "
                    "to one of the input images. These frames"
                    " can not be processed. Either you are using "
                    "synthetic data or something is seriously wrong with "
                    "your input data. Please provide something else or "
                    "deselect the weight calculation.");
            cpl_ensure(0,cpl_error_get_code(),NULL);
        }
        cpl_ensure(wim,cpl_error_get_code(),NULL);
        if ( badpixs ) {
            maskA = cpl_mask_threshold_image_create(
                    cpl_imagelist_get_const(badpixs,ii),
                    0.5, INT_MAX);
            cpl_mask_or(maskA,cpl_image_get_bpm_const(wim));
        }
        else {
            if ( cpl_image_get_bpm_const(wim) ) {
                maskA = cpl_mask_duplicate(cpl_image_get_bpm_const(wim));
            }
        }
        if ( maskA ) {
            cpl_image_reject_from_mask(wim, maskA );
            cpl_image_fill_rejected(wim,0.0);
            cpl_mask_delete(maskA);
        }
        cpl_imagelist_set(weights,wim,ii);
        cpl_image_delete(diffim);
        SPH_RAISE_CPL;
    }

    cpl_image_delete(medimint);
    cpl_image_delete(einsimage);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    return weights;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Reject bad pixels in imagelist
 * @param imlist    the imagelist where baxpixels should be rejected
 * @param badpixs   a list of images with the badpixels
 *
 * @return error code of the operation
 *
 * Marks all pixels in the input imagelist as rejected that have
 * values between 0.5 and INT_MAX in the corresponding image in the badpixs
 * imagelist.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_common_science_reject_badpix_image(
        cpl_image* im,
        cpl_image* badpixs)
{
    cpl_mask*   mask = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;

    cpl_ensure_code( im , CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( badpixs , CPL_ERROR_NULL_INPUT);
    mask = cpl_mask_threshold_image_create(badpixs, 0.5, INT_MAX);
    if ( mask ) {
        cpl_image_reject_from_mask(im, mask );
        cpl_mask_delete(mask); mask = NULL;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Reject bad pixels in imagelist
 * @param imlist    the imagelist where baxpixels should be rejected
 * @param badpixs   a list of images with the badpixels
 *
 * @return error code of the operation
 *
 * Marks all pixels in the input imagelist as rejected that have
 * values between 0.5 and INT_MAX in the corresponding image in the badpixs
 * imagelist.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_common_science_reject_badpix(
        cpl_imagelist* imlist,
        cpl_imagelist* badpixs)
{
    cpl_ensure_code( imlist , CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( badpixs , CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( cpl_imagelist_get_size(imlist) > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_imagelist_get_size(imlist) == cpl_imagelist_get_size(badpixs), CPL_ERROR_ILLEGAL_INPUT );
    for (int ii = 0; ii < cpl_imagelist_get_size(imlist); ++ii) {
        cpl_mask*   mask
            = cpl_mask_threshold_image_create(
                cpl_imagelist_get_const(badpixs,ii), 0.5, INT_MAX);
        if ( mask ) {
            cpl_image_reject_from_mask(
                    cpl_imagelist_get(imlist,ii), mask );
            cpl_mask_delete(mask);
        }
    }
    return cpl_error_get_code();
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Combine calibrated frames
 * @param inframeset        the frameset of calibrated frames
 * @param coll_alg          the collapse algorithm to use
 * @param imlist            imagelist
 * @param bp                badpixel list (maybe NULL)
 * @param wm                weightmap list (maybe NULL)
 * @param result            placeholder for the result master frame
 * @return combined master frame
 *
 * Combines all the imagelists to a new master_frame.
 * @see sph_common_science_combine() for more info.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_common_science_combine_imlists(
        sph_collapse_algorithm coll_alg,
        cpl_imagelist* imlist,
        cpl_imagelist* bpixlist,
        cpl_imagelist* wlist,
        sph_master_frame** result)
{
    cpl_image*          bpixim = NULL;
    cpl_image*          imim = NULL;
    cpl_image*          wim = NULL;
    cpl_imagelist*      wlist_intern = NULL;

    cpl_ensure_code(imlist,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(result,CPL_ERROR_NULL_INPUT);
    SPH_ERROR_ENSURE_GOTO_EXIT(imlist,cpl_error_get_code());

    if ( bpixlist )
    {
        //sph_common_science_reject_badpix(imlist,bpixlist);
        bpixim = cpl_image_new_from_accepted(imlist);
        cpl_image_multiply_scalar(bpixim,-1);
        cpl_image_add_scalar(bpixim,cpl_imagelist_get_size(bpixlist));
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
    }
    if ( coll_alg == SPH_COLL_ALG_WGT_MEAN ) {
        if ( wlist == NULL ) {
            wlist = sph_common_science_calc_weights(imlist,bpixlist,NULL);
            wlist_intern = wlist;
        }
        wim = cpl_imagelist_collapse_create(wlist);
        cpl_image_multiply_scalar(wim,cpl_imagelist_get_size(wlist));
        cpl_imagelist_multiply(imlist,wlist);
        imim = cpl_imagelist_collapse_create(imlist);
        cpl_image_multiply_scalar(imim,cpl_imagelist_get_size(wlist));
        cpl_image_divide(imim,wim);
        cpl_image_divide_scalar(wim,cpl_imagelist_get_size(wlist));
    }
    else if ( coll_alg == SPH_COLL_ALG_MEAN ) {
        imim = cpl_imagelist_collapse_create(imlist);
    }
    else if ( coll_alg == SPH_COLL_ALG_MEDIAN )
    {
        imim = cpl_imagelist_collapse_median_create(imlist);
    }
    else {
        SPH_ERROR_ENSURE_GOTO_EXIT(0,cpl_error_get_code());
    }

    // At this point I should have an image, possibly a badpixelmap
    // and possibly a weightmap.
    // Now need to construct a master frame from those.

    *result = sph_common_science_create_masterframe__(imlist,imim,bpixim,wim);

EXIT:
    cpl_imagelist_delete(wlist_intern); wlist_intern = NULL;
    cpl_image_delete(wim); wim = NULL;
    cpl_image_delete(bpixim); bpixim = NULL;
    cpl_image_delete(imim); imim = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Combine calibrated images in a single cube
 * @param cube_frame        the frames of the cube
 * @param coll_alg          the collapse algorithm to use
 * @param ext_im            image extension
 * @param ext_bp            badpixel extension
 * @param ext_wm            weightmap extension
 *
 * @return combined master frame
 *
 * Combines all the calibrated frames to a new master_frame.
 * @see sph_common_science_combine()
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_common_science_combine_single_cube(
        cpl_frame* cube_frame,
        sph_collapse_algorithm coll_alg,
        unsigned int ext_im,
        int ext_bp,
        int ext_wm)
{
    cpl_frameset*           tmpset = NULL;
    sph_master_frame*       result = NULL;
    cpl_ensure(cube_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    tmpset = cpl_frameset_new();
    cpl_frameset_insert(tmpset,cpl_frame_duplicate(cube_frame));
    result = sph_common_science_combine(tmpset,coll_alg,ext_im,ext_bp,ext_wm);
    cpl_frameset_delete(tmpset);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Combine calibrated frames
 * @param inframeset        the frameset of calibrated frames
 * @param coll_alg          the collapse algorithm to use
 * @param ext_im            image extension
 * @param ext_bp            badpixel extension
 * @param ext_wm            weightmap extension
 *
 * @return combined master frame
 *
 * Combines all the calibrated frames to a new master_frame.
 * The inframeset should contain a set of calibrated frames. These frames
 * should be FITS files with at least one plane (cubes are allowed) and
 * at least 2 extensions. The extension ext_im is assumed to contain the
 * image data and the extension ext_bp is assumed to contain a
 * map of bad pixels where all bad pixels have the value 1 and all good
 * pixels the value 0.
 *
 * Frames are combined either using a SPH_COLL_ALG_MEAN
 * a SPH_COLL_ALG_MEDIAN or a SPH_COLL_ALG_WGT_MEAN method.
 *
 * If the weighted mean is chosen, and the ext_wm parameter is set to -1,
 * a list of weight maps is first calculated from
 * differences to the median image (@see sph_common_science_calc_weights).
 * Pixels that are flagged in the bad pixel maps get assigned zero weight.
 *
 * Otherwise weightmaps are used from the input frameset with the specified
 * extension being the weightmap extensions.
 * The weights are used to create a weighted mean of the input image list.
 *
 * If the median method is chosen the resulting image is simply the median
 * of the input imagelist. Weights and bad pixels are then not considered.
 *
 * The resulting master frame contains the main image, the combined
 * badpixel map (showing as bad only pixels that were bad in all input images,
 * or had a total resulting weight of zero),
 * the ncombmap which gives the total number of good pixels that went into
 * the resulting pixel (this can be used to see how many badpixels were in the
 * input image list at a specific location), and the rms map which is the square
 * root of the inverse of the mean resulting weightmap.
 *
 * In case that a median combination was chosen (and therefore no
 * weights were used), the rms is calculated as the standard deviation
 * to the median or mean image.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_common_science_combine(
        cpl_frameset* inframeset,
        sph_collapse_algorithm coll_alg,
        unsigned int ext_im,
        int ext_bp,
        int ext_wm)
{
    sph_master_frame*       result = NULL;
    sph_smart_imagelist*    slist = NULL;
    cpl_imagelist*          imlist = NULL;
    cpl_imagelist*          bpixlist = NULL;
    cpl_imagelist*          wlist = NULL;
    cpl_image*              wim = NULL;
    sph_master_frame**      comb_frames = NULL;
    int                     rr = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;


    if ( !(coll_alg == SPH_COLL_ALG_MEAN ||
           coll_alg == SPH_COLL_ALG_MEDIAN ||
           coll_alg == SPH_COLL_ALG_WGT_MEAN) ) {

            sph_error_raise( SPH_ERROR_GENERAL,
                    __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Unsupported coll alg: %d!", coll_alg );
    }

    cpl_ensure(coll_alg == SPH_COLL_ALG_MEAN ||
            coll_alg == SPH_COLL_ALG_MEDIAN ||
            coll_alg == SPH_COLL_ALG_WGT_MEAN,
            CPL_ERROR_UNSUPPORTED_MODE,NULL);

    slist = sph_smart_imagelist_create(inframeset, ext_im);
    cpl_ensure(slist,cpl_error_get_code(),NULL);

    comb_frames = cpl_calloc(slist->nregions,sizeof(sph_master_frame*));

    for (rr = 0; rr < slist->nregions; ++rr)
    {
        slist->extension = ext_im;

        imlist = sph_smart_imagelist_get_set(slist,rr);
        SPH_ERROR_ENSURE_GOTO_EXIT(imlist,cpl_error_get_code());
        if ( ext_bp >= 0 )
        {
            slist->extension = ext_bp;
            bpixlist = sph_smart_imagelist_get_set(slist,rr);
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        }
        if ( ext_wm >= 0 ) {
            slist->extension = ext_wm;
            wlist = sph_smart_imagelist_get_set(slist,rr);
            SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        }

        SPH_ERROR_ENSURE_GOTO_EXIT(
                sph_common_science_combine_imlists(coll_alg,
                imlist,bpixlist,wlist,&comb_frames[rr])
            == CPL_ERROR_NONE, cpl_error_get_code());

        cpl_imagelist_delete(imlist); imlist = NULL;
        cpl_imagelist_delete(bpixlist);bpixlist = NULL;
        cpl_imagelist_delete(wlist); wlist = NULL;
    }
    SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
    result = sph_smart_imagelist_reassemble_master_frame(slist,comb_frames);


    if (result) {
        cpl_propertylist*       plist = NULL;
        slist->extension = ext_im;
            sph_smart_imagelist_get_raw_median_properties(slist,
                    result->properties);

            plist = sph_keyword_manager_load_properties
                (cpl_frame_get_filename(cpl_frameset_get_first(inframeset)),
                 ext_im);

            cpl_propertylist_append( result->properties, plist );
            cpl_propertylist_delete(plist);
    }
    SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;

EXIT:
    cpl_imagelist_delete(imlist); imlist = NULL;
    cpl_imagelist_delete(bpixlist); bpixlist = NULL;
    cpl_imagelist_delete(wlist); wlist = NULL;
    cpl_image_delete(wim); wim = NULL;
    if ( comb_frames ) {
        for (rr = 0; rr < slist->nregions; ++rr) {
            sph_master_frame_delete(comb_frames[rr]);
            comb_frames[rr] = NULL;
        }
        cpl_free(comb_frames); comb_frames = NULL;
    }

    sph_smart_imagelist_delete(slist); slist = NULL;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}



/*----------------------------------------------------------------------------*/
/**
 * @brief Calibrate a raw image
 * @param inimage        the raw image
 * @param dark            a master dark frame to subtract (maybe NULL)
 * @param flat            a master flat field to use (maybe NULL)
 * @return calibrated master frame or NULL on error
 *
 * Description:
 * Calibrates a raw image, subtracting a dark, dividing a flat field .
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_common_science_calibrate_raw( const cpl_image* rawimage,
                                  const sph_master_frame* dark,
                                  const sph_master_frame* flat )
{
    sph_master_frame*        mframe = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure( rawimage, CPL_ERROR_NULL_INPUT, NULL );
    mframe = sph_master_frame_new_from_cpl_image(rawimage);
    if ( dark == NULL && flat == NULL) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"No dark or flat");
        return mframe;
    }
    if ( dark ) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"Subtracting dark...");
        sph_master_frame_subtract_master_frame( mframe, dark );
    }
    if ( flat ) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"Dividing flat...");
        sph_master_frame_divide_master_frame( mframe, flat );
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return mframe;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the total number images in frameset
 * @param rawframes     the frameset to obtain the total no. images from.
 * @param ext           the extension to read from
 *
 * @return the total number of images or -1 on error
 *
 * This returns the total number of images in the frameset. All planes
 * in all frames are counted.
 *
 *
 */
/*----------------------------------------------------------------------------*/
int sph_common_science_get_nraws(const cpl_frameset* rawframes, int ext) {
    int total = 0;

    cpl_ensure(rawframes != NULL, CPL_ERROR_NULL_INPUT, -1);
 
    for (int ff = 0; ff < cpl_frameset_get_size(rawframes); ++ff) {
        const cpl_frame* iframe = cpl_frameset_get_position_const(rawframes,ff);
        const char* filename = cpl_frame_get_filename(iframe);

        total += sph_fits_test_iscube(filename, ext) == 1
            ? sph_fits_get_nplanes(filename, ext)
            : 1;

    }
    return total;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Get the next raw image
 * @param rawframes     the frameset to obtain the next image from
 * @param ext           the extension to read from
 *
 * @return the next raw science image or NULL if none left or error.
 *
 * This returns the next raw image unprocessed.
 * The function always reads from the extension currently set in self,
 * and automatically detects if the frame is a cube or not. It will go
 * through all images (planes) in all frames.
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_common_science_get_next_raw(cpl_frameset* rawframes, int ext) {
    int         readnextplane = 0;
    cpl_image*  im = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    readnextplane = sph_common_science_general_get_next_raw( rawframes,  ext );
    if (readnextplane == -1) return NULL;

    im = cpl_image_load(cpl_frame_get_filename(sph_common_science_current_frame__),
            CPL_TYPE_DOUBLE,
            sph_common_science_currentplane__,
            ext);
    if (cpl_error_get_code()) {
        (void)cpl_error_set_where(cpl_func);
    } else if ( readnextplane ) {
        sph_common_science_currentplane__++;
    }
    return im;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return pointer to current raw frame
 *
 * @return pointer to current raw frame or NULL if not initialised
 *
 * This function returns a pointer to the current raw frame that the image
 * from the last get_next_raw call was loaded from.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frame*
sph_common_science_get_current_raw_frame(void) {
    return sph_common_science_current_frame__;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return current plane number in current frame
 *
 * @return plane number in current frame or -1 on error
 *
 * This function returns the current plane number in the current
 * frame. It corresponds to the number of planes already read.
 *
 */
/*----------------------------------------------------------------------------*/
int
sph_common_science_get_current_plane(void) {
    return sph_common_science_currentplane__;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Get the next zpl exposure pre-processed frame
 * @param rawframes     the frameset to obtain the next image from
 * @param ext           the extension to read from
 *
 * @return the next zpl exposure frame or NULL if none left or error.
 *
 * This returns the next zpl exposure pre-processed raw frame.
 * The function always reads from the extension currently set in self,
 * and automatically detects if the frame is a cube or not. It will go
 * through all zpl exposure (planes) in all frames.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_zpl_exposure*
sph_common_science_get_next_zpl_exposure(cpl_frameset* rawframes) {
    int                 readnextplane = 0;
    sph_zpl_exposure*   zplexp = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    readnextplane = sph_common_science_general_get_next_raw( rawframes,  0 );
    if (readnextplane == -1) return NULL;

    zplexp = sph_zpl_exposure_load( sph_common_science_current_frame__, sph_common_science_currentplane__ );

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if ( readnextplane ) sph_common_science_currentplane__++;
    return zplexp;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the next zpl exposure imaging pre-processed frame
 * @param rawframes     the frameset to obtain the next image from
 * @param ext           the extension to read from
 *
 * @return the next zpl exposure imaging frame or NULL if none left or error.
 *
 * This returns the next zpl exposure imaging pre-processed raw frame.
 * The function always reads from the extension currently set in self,
 * and automatically detects if the frame is a cube or not. It will go
 * through all zpl exposure imaging (planes) in all frames.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_zpl_exposure_imaging*
sph_common_science_get_next_zpl_exposure_imaging(cpl_frameset* rawframes) {
    int                         readnextplane = 0;
    sph_zpl_exposure_imaging*   zplexp = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    readnextplane = sph_common_science_general_get_next_raw( rawframes,  0 );
    if (readnextplane == -1) return NULL;

    zplexp = sph_zpl_exposure_imaging_load( sph_common_science_current_frame__, sph_common_science_currentplane__ );

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if ( readnextplane ) sph_common_science_currentplane__++;
    return zplexp;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Get the next double imagepre-processed frame
 * @param frames     the frameset to obtain the next double image from
 * @param ext           the extension to read from
 *
 * @return the next double image frame or NULL if none left or error.
 *
 * This returns the next double image frame.
 * The function always reads from the extension currently set in self,
 * and automatically detects if the frame is a cube or not. It will go
 * through all double image (planes) in all frames.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_double_image*
sph_common_science_get_next_double_image(cpl_frameset* frames) {
    int                 readnextplane = 0;
    sph_double_image*   doubleimage = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    readnextplane = sph_common_science_general_get_next_raw( frames,  0 );

    if (readnextplane == -1) return NULL;

    doubleimage = sph_double_image_load(
            cpl_frame_get_filename(sph_common_science_current_frame__), sph_common_science_currentplane__ );

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if ( readnextplane ) sph_common_science_currentplane__++;
    return doubleimage;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Get the next master processed frame
 * @param frames     the frameset to obtain the next master_frame from
 * @param ext           the extension to read from
 *
 * @return the next master_frame or NULL if none left or error.
 *
 * This returns the next master_frame frame.
 * The function always reads from the extension currently set in self,
 * and automatically detects if the frame is a cube or not. It will go
 * through all planes in all frames.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_common_science_get_next_master_frame(cpl_frameset* frames) {
    int                 readnextplane = 0;
    sph_master_frame*   mframe = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    readnextplane = sph_common_science_general_get_next_raw( frames,  0 );

    if (readnextplane == -1) return NULL;

    mframe = sph_master_frame_load_(sph_common_science_current_frame__,
                    sph_common_science_currentplane__ );

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if ( readnextplane ) sph_common_science_currentplane__++;
    return mframe;
}



/*----------------------------------------------------------------------------*/
/**
 * @brief Create weighted mean double image from the double image frames/cubes
 *
 * @param inframes  the input cpl_frameset with double image frames/cubes to combine
 *
 * @return pointer to the newly created double image or NULL in case of error.
 *
 * This function creates a new sph_double_image from a set of double image
 * frames (or cubes), given as a cpl_frameset.
 *
 * @note  <x> = sum(x_i/sigma_i^2) / sum(1/sigma_i^2); sigma_i = rms
 *
 */
/*----------------------------------------------------------------------------*/
sph_double_image*
sph_common_science_combine_weight_double_image_frameset(
        cpl_frameset* inframes)
{
    sph_double_image*        curdimage        = NULL;
    sph_double_image*         result            = NULL;



    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure(inframes ,CPL_ERROR_NULL_INPUT, NULL);

    SPH_INFO_MSG("weight combine...")

     curdimage = sph_common_science_get_next_double_image( inframes );
    SPH_INFO_MSG("curdimage loaded")

    if ( curdimage != NULL ) {
         result = sph_double_image_new( cpl_image_get_size_x( curdimage->iframe->image),
                                       cpl_image_get_size_y( curdimage->iframe->image) );
    }

    while ( curdimage ){

        sph_common_science_treat_weighted_masterframe( result->iframe, curdimage->iframe );
        sph_common_science_treat_weighted_masterframe( result->pframe, curdimage->pframe );

        sph_double_image_delete( curdimage );
         curdimage = sph_common_science_get_next_double_image( inframes );
    }

    cpl_image_divide( result->iframe->image, result->iframe->rmsmap );
    cpl_image_power( result->iframe->rmsmap, 0.5);
    cpl_image_divide( result->pframe->image, result->pframe->rmsmap );
    cpl_image_power( result->pframe->rmsmap, 0.5);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function to extract vector of angles from set of FCtables
 * @param cpl_frameset          a set of fctables
 *
 * @return vector of angles
 *
 * Extracts all angles and returns as a single vector in the
 * order in which they appear in the input.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_vector*
sph_common_science_extract_angles_from_fctables( const cpl_frameset* fctables ) {
    const cpl_frame*    aframe     = NULL;
    cpl_vector*         all_angles = NULL;
    cpl_size             np         = 0;

    cpl_ensure(fctables, CPL_ERROR_NULL_INPUT, NULL);

    aframe = cpl_frameset_get_first_const(fctables);
    while (aframe)
    {
        const char*  fname   = cpl_frame_get_filename(aframe);
        sph_fctable* fctable = sph_filemanager_is_fits(fname)
            ? sph_fctable_load_fits(fname) : NULL;
        cpl_vector*  angles  = NULL;

        if ( fctable == NULL ) {
            cpl_error_reset();
            fctable = sph_fctable_load_ascii( fname );
            cpl_ensure(fctable != NULL, cpl_error_get_code(), NULL);
        }

        angles = sph_fctable_get_angles(fctable);

        if ( angles ) {
            const cpl_size v0 = np;

            np += cpl_vector_get_size(angles);

            if (v0 == 0) {
                assert(all_angles == NULL);
                all_angles = cpl_vector_new(np);
            } else {
                cpl_vector_set_size(all_angles, np);
            }

            for (cpl_size vv = v0; vv < np; ++vv)
            {
                cpl_vector_set( all_angles, vv,
                                cpl_vector_get(angles, vv - v0) );
            }
        }
        sph_fctable_delete(fctable); fctable = NULL;
        cpl_vector_delete(angles); angles = NULL;
        aframe = cpl_frameset_get_next_const(fctables);
    }

    if (cpl_error_get_code()) {
        (void)cpl_error_set_where(cpl_func);
        cpl_vector_delete(all_angles);
        all_angles = NULL;
    }

    return all_angles;
}


static sph_error_code
sph_common_science_treat_weighted_masterframe( sph_master_frame* result, sph_master_frame* mframe){
    sph_master_frame*        mframe_copy = NULL;
    cpl_image*                im            = NULL;
    cpl_mask*                mask            = NULL;
    cpl_mask*                resmask            = NULL;


    cpl_ensure_code( mframe ,CPL_ERROR_NULL_INPUT );
    cpl_ensure_code( result ,CPL_ERROR_NULL_INPUT );
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    mframe_copy= sph_master_frame_duplicate(mframe);
    //set a big value (SPH_MASTER_FRAME_BAD_RMS) into rmsmap for bad pixels
    sph_master_frame_set_bads_from_image( mframe, mframe->badpixelmap );
    sph_master_frame_set_bads_from_image( mframe_copy, mframe->badpixelmap );

    // (x_i/sigma_i^2)
    cpl_image_multiply( mframe_copy->rmsmap, mframe_copy->rmsmap );
    cpl_image_divide( mframe_copy->image, mframe_copy->rmsmap );


    //mask =  cpl_image_get_bpm( mframe_copy->image );
    //if ( cpl_mask_count( mask ) != 0 )  {
    //    SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "Division by zero...rmsmap should not have zero elements...continue anyway")
    //}
    //cpl_mask_delete( mask ); mask = NULL;


    //adding to the result double image
    cpl_image_add( result->image, mframe_copy->image ); //set/add result image

    resmask = sph_master_frame_get_badpixelmask( result );
    mask = sph_master_frame_get_badpixelmask( mframe_copy );
    cpl_mask_and( resmask, mask);


    //set result badpixelmap and corresponding big values for rms
    sph_master_frame_set_bads_from_mask( result, resmask);
    cpl_mask_delete( resmask ); resmask = NULL;
    cpl_mask_delete( mask ); mask = NULL;

    //retrieve badpixel map from current image as an image and inverse it,
    //i.e. values = 1 (bad pixels) set to 0, values = 0 (good pixels) set to 1
    //image of inverted bad pixels is added to ncombmap
    im = sph_master_frame_get_badpixelmap( mframe_copy );
    cpl_image_add_scalar(im, -1.0);
    cpl_image_multiply_scalar( im, -1.0);
    cpl_image_add( result->ncombmap, im );
    cpl_image_delete( im ); im = NULL;


    // 1/sigma_i^2
    sph_utils_cpl_image_fill_double( mframe->rmsmap, 1.0 );
    cpl_image_divide( mframe->rmsmap, mframe_copy->rmsmap );
    cpl_image_add( result->rmsmap, mframe->rmsmap );


    sph_master_frame_delete( mframe_copy );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;

}



/*----------------------------------------------------------------------------*/
/**
 * @brief Get the next plane to read
 * @param rawframes     the frameset to obtain the next image from
 * @param ext           the extension to read from
 *
 * @return 1 if next plane to read, 0 if not or -1 if error.
 *
 * This returns true if there is a next plane to read.
 *
 */
/*----------------------------------------------------------------------------*/
static
int sph_common_science_general_get_next_raw(cpl_frameset* rawframes, int ext) {
    int         readnextplane = 0;

    cpl_ensure(rawframes,CPL_ERROR_NULL_INPUT,-1);
    cpl_ensure(ext >= 0,CPL_ERROR_ILLEGAL_INPUT,-1);
#   ifdef _OPENMP
#   pragma omp critical
#endif
    {
        if ( rawframes != sph_common_science_current_frameset__ ) {
            sph_common_science_current_frameset__ = rawframes;
            sph_common_science_current_frame__ = NULL;
            sph_common_science_iscube__ = 0;
            sph_common_science_nplanes__ = 0;
            sph_common_science_currentplane__ = 0;
        }

        if ( sph_common_science_iscube__ &&
                sph_common_science_currentplane__ <
                sph_common_science_nplanes__ ) {
            readnextplane = 1;
        }
        else {
            int ok = 1;
            if ( sph_common_science_current_frame__ == NULL ) {
                sph_common_science_current_frame__ = cpl_frameset_get_first(rawframes);
                if ( sph_common_science_current_frame__ == NULL ) {
                    cpl_error_reset();
                    ok = 0;
                }
            }
            else {
                sph_common_science_current_frame__ = cpl_frameset_get_next(rawframes);
                if ( sph_common_science_current_frame__ == NULL ) {
                    sph_common_science_current_frameset__ = NULL;
                    sph_common_science_current_frame__ = NULL;
                    sph_common_science_iscube__ = 0;
                    sph_common_science_nplanes__ = 0;
                    sph_common_science_currentplane__ = 0;
                    cpl_error_reset();
                    ok = 0;
                }
            }
            if ( ok ) {
                sph_common_science_iscube__ = sph_fits_test_iscube(
                        cpl_frame_get_filename(
                                sph_common_science_current_frame__), ext);
                sph_common_science_currentplane__ = 0;
                sph_common_science_nplanes__ = 0;
                if ( sph_common_science_iscube__ ) {
                    sph_common_science_nplanes__ = sph_fits_get_nplanes(
                            cpl_frame_get_filename(sph_common_science_current_frame__), ext );
                    readnextplane = 1;
                }
            }
            else {
                readnextplane = -1;
            }
        }
    }
    return readnextplane;
}

// PIPE-6081
/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the specified raw frame and any other frame tagged differently
 * @param self          The frameset containing the raw frame to copy
 * @param raw           The frame to copy
 * @return The extracted frameset (to be deallocated) or NULL on error
 */
/*----------------------------------------------------------------------------*/
cpl_frameset* sph_common_science_frameset_extract_raw_single(const cpl_frameset* self,
                                                             const cpl_frame* raw) {
    cpl_boolean ok = CPL_FALSE;
    const char* rawtag  = cpl_frame_get_tag(raw);
    const char* rawname = cpl_frame_get_filename(raw);
    cpl_frameset* other = NULL;
    const cpl_frame* frame;
    cpl_frameset_iterator *it  = NULL;

//    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    if (self == NULL) {
        return NULL;
    }
    cpl_ensure(raw  != NULL, CPL_ERROR_NULL_INPUT, NULL);

    other = cpl_frameset_new();

    it  = cpl_frameset_iterator_new(self);
    frame = cpl_frameset_iterator_get(it);

    while (frame) {
        if (strcmp(cpl_frame_get_tag(frame), rawtag) != 0 ||
        strcmp(cpl_frame_get_filename(frame), rawname) == 0) {
            if (strcmp(cpl_frame_get_filename(frame), rawname) == 0)
                ok = CPL_TRUE;
            cpl_frameset_insert(other, cpl_frame_duplicate(frame));
            cpl_msg_warning(cpl_func, "Copied %s (%s <=> %s)",
                            cpl_frame_get_filename(frame),
                            cpl_frame_get_tag(frame), rawtag);
        } else {
            cpl_msg_warning(cpl_func, "Not copied %s (%s <=> %s)",
                            cpl_frame_get_filename(frame),
                            cpl_frame_get_tag(frame), rawtag);
        }

        cpl_frameset_iterator_advance(it, 1);
        frame = cpl_frameset_iterator_get(it);
    }

    cpl_frameset_iterator_delete(it);

    if (!ok) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                    "Raw frame %s with tag %s not found",
                                    cpl_frame_get_filename(raw),
                                    rawtag);
        cpl_frameset_delete(other);
        other = NULL;
    }

    return other;
                                                     }


/**@}*/
