/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_master_frame.h"
#include "sph_cube.h"
#include "sph_create_flat.h"
#include "sph_filemanager.h"
#include "sph_smart_imagelist.h"
#include "sph_common_keywords.h"
#include "sph_fitting.h"
#include "sph_keyword_manager.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides ...
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*-----------------------------------------------------------------------------
                        Private function prototypes
 -----------------------------------------------------------------------------*/

static sph_master_frame*
sph_create_flat_process_imlist(cpl_imagelist*,
                               cpl_vector*,
                               const sph_master_frame*,
                               int) CPL_ATTR_ALLOC;

static
cpl_vector* sph_create_flat_calculate_means(const cpl_frameset*,
                                            const sph_master_frame*,
                                            const cpl_mask*,
                                            cpl_vector**,
                                            cpl_vector**,
                                            int,
                                            cpl_vector**)
    CPL_ATTR_ALLOC;

/*-----------------------------------------------------------------------------
                              Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief        Process an imagelist to obtain flat field

 @param     imlist      The input imagelist
 @param     means       The sampling points? One for each input image
 @param     dark        A dark of the same x and y pixel size as imlist or NULL
 @param     do_robust   Iff true use robust method, otherwise polynomial fit

 @return    new master frame or NULL

 Description:
 Processes the imagelist obtaining a master frame giving the flat field values.

 */
/*----------------------------------------------------------------------------*/
static sph_master_frame*
sph_create_flat_process_imlist(cpl_imagelist* imlist, /* FIXME: const */
                               cpl_vector* means,     /* FIXME: const */
                               const sph_master_frame* dark,
                               int do_robust) {
    sph_master_frame* result = NULL;
    cpl_imagelist* coeffs = NULL;
    cpl_image* errimage = NULL;
    cpl_image* lin_coeff = NULL;
    double m = 0.0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    cpl_ensure( imlist, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( means, CPL_ERROR_NULL_INPUT, NULL);
    if (cpl_imagelist_get_size(imlist) <= 1) {
        cpl_msg_error(cpl_func, "Please provide at least 2 raw flats.");
    }
    cpl_ensure( cpl_imagelist_get_size(imlist) > 1, CPL_ERROR_ILLEGAL_INPUT,
            NULL);
    cpl_ensure( cpl_imagelist_get_size(imlist) == cpl_vector_get_size(means),
            CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    if (dark)
        cpl_imagelist_subtract_image(imlist, dark->image);
    errimage = cpl_image_new(cpl_image_get_size_x(cpl_imagelist_get(imlist, 0)),
            cpl_image_get_size_y(cpl_imagelist_get(imlist, 0)),
            CPL_TYPE_DOUBLE);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    m = cpl_vector_get_mean(means);
    if (do_robust) {
        lin_coeff = sph_fitting_fit_robust_imlist(imlist, means, 0.1, 10.0,
                NULL, errimage);
    } else {
        coeffs = cpl_fit_imagelist_polynomial(means, imlist, 0, 1, CPL_FALSE,
                CPL_TYPE_DOUBLE, errimage);
        lin_coeff = cpl_imagelist_unset(coeffs, 1);
        cpl_imagelist_delete(coeffs);
        coeffs = NULL;
        cpl_image_divide_scalar(errimage, m * m);
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    result = sph_master_frame_new_from_cpl_image(lin_coeff);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_image_delete(result->rmsmap);
    result->rmsmap = errimage;
    cpl_image_delete(lin_coeff);
    lin_coeff = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

static
cpl_vector* sph_create_flat_calculate_means(const cpl_frameset* rawframes,
                                            const sph_master_frame* dark,
                                            const cpl_mask* mask,
                                            cpl_vector** lampflux_out,
                                            cpl_vector** counts_out,
                                            int extnum,
                                            cpl_vector** lampfluxstdev_out) {
    cpl_vector* means = NULL;
    int ff = 0;
    double mean = 0.0;
    double mean_stdev = 0.0;
    cpl_image* tmpim = NULL;
    sph_master_frame* temp_master = NULL;
    cpl_propertylist* pl = NULL;
    int naxis = 1;
    int plane = 0;
    int count = 0;
    double exptime = 0.0;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    means = cpl_vector_new(1);
    if (lampflux_out) {
        *lampflux_out = cpl_vector_new(1);
    }
    if (counts_out) {
        *counts_out = cpl_vector_new(1);
    }
    if (lampfluxstdev_out) {
        *lampfluxstdev_out = cpl_vector_new(1);
    }
    for (ff = 0; ff < cpl_frameset_get_size(rawframes); ++ff) {
        // FIXME: The bit below should be changed to use common
        // code to the sph_common_get_next_raw and associated
        // functions.
        const cpl_frame* frame = cpl_frameset_get_position_const(rawframes, ff);
        pl = cpl_propertylist_load(cpl_frame_get_filename(frame), 0);
        if (cpl_propertylist_get_int(pl, "NAXIS") == 3) {
            naxis = cpl_propertylist_get_int(pl, "NAXIS3");
        } else
            naxis = 1;
        cpl_propertylist_delete(pl);
        pl = NULL;
        pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(frame),
                0);
        exptime = 1.0;
        if (cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_SEQ1DIT))
            exptime = cpl_propertylist_get_double(pl,
            		SPH_COMMON_KEYWORD_SEQ1DIT);
        cpl_propertylist_delete(pl);
        pl = NULL;

        for (plane = 0; plane < naxis; ++plane) {
            mean = mean_stdev = 0.0;

            tmpim = cpl_image_load(cpl_frame_get_filename(frame),
                    CPL_TYPE_DOUBLE, plane, extnum);
            if (tmpim) {
                temp_master = sph_master_frame_new_from_cpl_image(tmpim);
                cpl_image_delete(tmpim);
            }
            if (temp_master) {
                if (dark) {
                    sph_master_frame_subtract_master_frame(temp_master, dark);
                }
                if (mask) {
                    sph_master_frame_set_bads_from_mask(temp_master, mask);
                }
                mean = sph_master_frame_get_mean(temp_master, &mean_stdev);
                sph_master_frame_delete(temp_master);
                temp_master = NULL;
                count = cpl_vector_get_size(means);
                cpl_vector_set(means, count - 1, mean);
                cpl_vector_set_size(means, count + 1);
                if (counts_out) {
                    cpl_vector_set(*counts_out, count - 1, mean);
                    cpl_vector_set_size(*counts_out, count + 1);
                }
                if (lampflux_out) {
                    cpl_vector_set(*lampflux_out, count - 1, mean / exptime);
                    cpl_vector_set_size(*lampflux_out, count + 1);
                }
                if (lampfluxstdev_out) {
                    cpl_vector_set(*lampfluxstdev_out, count - 1,
                            mean_stdev / exptime);
                    cpl_vector_set_size(*lampfluxstdev_out, count + 1);
                }
            } else {
                SPH_ERROR_RAISE_ERR(
                        cpl_error_get_code(),
                        "Could not create/load flat frame from file %s and plane %d", cpl_frame_get_filename(frame), plane);
                cpl_vector_delete(means);
                return NULL;
            }
        }
    }
    count = cpl_vector_get_size(means);
    cpl_vector_set_size(means, count - 1);
    if (counts_out) {
        count = cpl_vector_get_size(*counts_out);
        cpl_vector_set_size(*counts_out, count - 1);
    }
    if (lampflux_out) {
        count = cpl_vector_get_size(*lampflux_out);
        cpl_vector_set_size(*lampflux_out, count - 1);
    }
    if (lampfluxstdev_out) {
        count = cpl_vector_get_size(*lampfluxstdev_out);
        cpl_vector_set_size(*lampfluxstdev_out, count - 1);
    }
    if (cpl_error_get_code()) {
        if (counts_out) {
            cpl_vector_delete(*counts_out);
            *counts_out = NULL;
        }
        if (lampflux_out) {
            cpl_vector_delete(*lampflux_out);
            *lampflux_out = NULL;
        }
        if (lampfluxstdev_out) {
            cpl_vector_delete(*lampfluxstdev_out);
            *lampfluxstdev_out = NULL;
        }
        cpl_vector_delete(means);
        means = NULL;
        (void)cpl_error_set_where(cpl_func);
    }
    return means;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create a master flat field
 @param    rawframes        the raw flat frames
 @param    framecomb_parameterlist     the parameters for framecombination
 @param    badpix    an optional mask of static badpixels
 @param    dark        an optional dark frame to subtract
 @param    linbadpix an optional pointer to a linearity badpix map to be created
 @param    badpix_lowtolerance  lower tolerance to use for the creation of linbadpix
 @param    badpix_uptolerance  upper tolerance to use for the creation of linbadpix
 @param    badpix_rmstolerance  chisq tolerance to use for the creation of linbadpix

 @return   pointer to new flat or NULL on error.

 This function creates a new flat field from the raw frames and an optional dark
 and optional static badpixel map.
 Currently the framecombination is always a CLEAN_MEAN and the parameters used
 are the ones given in framecomb_parameterlist.
 If given, the linbadpix pointer is filled with a new linearity badpixel image
 that is calculated using the flat field fits and the three tolerance parameters
 provided.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_create_flat(const cpl_frameset* rawframes, int do_robust,
                const cpl_mask* badpix,
                const sph_master_frame* dark, cpl_image** linbadpix,
                double badpix_lowtolerance, double badpix_uptolerance,
                double badpix_rmstolerance,
                cpl_vector** lampflux_out,
                cpl_vector** counts_out,
                int extension,
                cpl_vector** lampfluxstdev_out) {
    sph_master_frame* result = NULL;
    sph_smart_imagelist* smart_list = NULL;
    int current_set = 0;
    cpl_vector* means = NULL;
    sph_master_frame** fit_frames = NULL;
    cpl_mask* dark_bads = NULL;
    cpl_mask* fitmask = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    smart_list = sph_smart_imagelist_create(rawframes, extension);
    cpl_ensure(smart_list, cpl_error_get_code(), NULL);
    smart_list->extension = extension;

    if (dark) {
        dark_bads = sph_master_frame_get_badpixelmask(dark);
        if (dark_bads == NULL) {
            (void)cpl_error_set_where(cpl_func);
            sph_smart_imagelist_delete(smart_list);
            return NULL;
        }
    }
    means = sph_create_flat_calculate_means(rawframes, dark, badpix,
            lampflux_out, counts_out, extension, lampfluxstdev_out);
    if (means == NULL) {
        (void)cpl_error_set_where(cpl_func);
        cpl_mask_delete(dark_bads);
        sph_smart_imagelist_delete(smart_list);
        return NULL;
    }

    fit_frames = cpl_calloc(smart_list->nregions, sizeof(sph_master_frame*));
    for (current_set = 0; current_set < smart_list->nregions; ++current_set) {
        cpl_imagelist* imlist =
            sph_smart_imagelist_get_set(smart_list, current_set);
        sph_master_frame* dark_window = NULL;
        if (dark) {
            dark_window = sph_master_frame_new_extract(dark,
                    smart_list->regions[current_set].ll_x + 1,
                    smart_list->regions[current_set].ll_y + 1,
                    smart_list->regions[current_set].ur_x + 1,
                    smart_list->regions[current_set].ur_y + 1);
        }
        fit_frames[current_set] = sph_create_flat_process_imlist(imlist, means,
                dark_window, do_robust);
        cpl_imagelist_delete(imlist);
        sph_master_frame_delete(dark_window);
    }
    cpl_vector_delete(means);
    result = sph_smart_imagelist_reassemble_master_frame(smart_list,
            fit_frames);
    if (result) {
        sph_master_frame_mask_tolerance(result, badpix_lowtolerance,
                badpix_uptolerance);
        if (badpix_rmstolerance > 0.0) {
            fitmask = cpl_mask_threshold_image_create(result->rmsmap, -1.0,
                    badpix_rmstolerance * badpix_rmstolerance);
            cpl_mask_not(fitmask);
            sph_master_frame_set_bads_from_mask(result, fitmask);
            cpl_mask_delete(fitmask);
            fitmask = NULL;
        }

        if (linbadpix)
            *linbadpix = sph_master_frame_get_badpixelmap(result);
        if (badpix)
            sph_master_frame_set_bads_from_mask(result, badpix);
        if (dark_bads)
            sph_master_frame_set_bads_from_mask(result, dark_bads);

        /* The flat value on any bad pixel position is set to 1.
           This improves the visual appearance (and allows for
           dividing with the flat without taking into account the
           bad pixel map). */
        cpl_image_fill_rejected(result->image, 1.0);

        sph_master_frame_quality_check(result);

        if (dark) {
            cpl_image_add(result->badpixelmap, dark->badpixelmap);
        }
    }

    if (fit_frames) {
        for (current_set = 0; current_set < smart_list->nregions;
                ++current_set) {
            sph_master_frame_delete(fit_frames[current_set]);
        }
        cpl_free(fit_frames);
    }

    sph_smart_imagelist_delete(smart_list);
    cpl_mask_delete(dark_bads);

    if (result == NULL) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_OUTPUT,
                            "Could not create the flat frame.");
    } else if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL
            sph_master_frame_delete(result);
        result = NULL;
    }
    return result;
}
/**@}*/
