/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include "../Recipes/IFS/sph_ifs_tags.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include <assert.h>
#include "sph_crosstalkco.h"
#include "sph_fft.h"
#include "sph_filemanager.h"
#include <cpl.h>

static
cpl_image* sph_crosstalkco_small_scale_crosstalk( const cpl_image* image_in,
                                                  cpl_size sepmax, 
                                                  double bfac, double powfac )
    CPL_ATTR_ALLOC;

static
cpl_image* sph_crosstalkco_large_scale_crosstalk( const cpl_image* image_in,
                                                  cpl_size dimsub,
                                                  double threshold,
                                                  double smoothing_fwhm,
                                                  double stephist)
    CPL_ATTR_ALLOC;

/*----------------------------------------------------------------------------*/
/**
 * @brief Correct crosstalk in a raw cube
 * @param allframes			the complete set of frames used
 * @param raw_cube_frame 	the raw cube to correct
 * @param sepmax			the seperation (border) to the image edge
 * @param bfac				the bfac parameter
 * @param powfac			the powfac parameter
 * @param lg_scale_corr_window     the large scale correction window to use
 * @param threshold         threshold for pixels to replace with median
 * @param smoothing_fwhm    the fwhm for the smooting
 * @param stephist          the stepsize to use for the histrogram
 *
 * @return error code
 *
 * Description:
 * This corrects the raw cube using the badpixel correction
 * as implemented in Dino Mesas IDL script.
 *
 * The bfac and powfac parameters are applied in the subtraction correction amount
 * for the small scale crosstalk correction
 * using the equation: 1.0/(1.0 + pow(rdist/bfac,powfac))
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_crosstalkco_apply_to_raw_cube(
		cpl_frameset* allframes,
		cpl_frame* raw_cube_frame,
		cpl_size sepmax,
		double bfac,
		double powfac,
		cpl_size lg_scale_corr_window,
		double threshold,
		double smoothing_fwhm,
		double stephist)
{
	cpl_imagelist* resultlist = NULL;
	cpl_imagelist* imlist = NULL;

	cpl_frameset* usedframes = NULL;
	cpl_parameterlist* parlist = NULL;
	cpl_propertylist* applist = NULL;
	const char* filename = NULL;
	char* newfilename = NULL;


	cpl_ensure_code(allframes, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(raw_cube_frame, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(bfac >= 0.0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(powfac >= 0.0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(sepmax > 0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(lg_scale_corr_window > 0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(lg_scale_corr_window < 500, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(stephist > 0.0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(stephist < 5000.0, CPL_ERROR_ILLEGAL_INPUT);
	cpl_ensure_code(threshold > 0.0, CPL_ERROR_ILLEGAL_INPUT);
        /* FIXME: Unused */
	cpl_ensure_code(smoothing_fwhm > 0.0, CPL_ERROR_ILLEGAL_INPUT);

	imlist = cpl_imagelist_load(
				cpl_frame_get_filename(raw_cube_frame),
				CPL_TYPE_UNSPECIFIED, 0);

	cpl_ensure_code(imlist, cpl_error_get_code());

        resultlist = cpl_imagelist_new();

	for ( cpl_size plane = 0; plane < cpl_imagelist_get_size(imlist); ++plane )
	{
		cpl_image* image = cpl_imagelist_get(imlist,plane);
		if ( image ) {
			cpl_image* sc_corrected_image =
					sph_crosstalkco_small_scale_crosstalk( image, sepmax, bfac, powfac );
			/*cpl_image* corrected_image =
					sph_crosstalkco_large_scale_crosstalk(
							sc_corrected_image, lg_scale_corr_window,threshold,
							smoothing_fwhm, stephist); */
			cpl_image* corrected_image = cpl_image_duplicate(sc_corrected_image);
			cpl_image_delete(sc_corrected_image); sc_corrected_image = NULL;

			cpl_imagelist_set(resultlist,corrected_image,plane);

		}
	}

	usedframes = cpl_frameset_new();
	cpl_frameset_insert(usedframes,cpl_frame_duplicate(raw_cube_frame));

	parlist = cpl_parameterlist_new();
	applist = cpl_propertylist_new();

	cpl_propertylist_append_string(applist,
			CPL_DFS_PRO_CATG,
			SPH_IFS_TAG_NEW_XTALK_CORRECTION);

	filename = cpl_frame_get_filename(raw_cube_frame);

	newfilename = sph_filemanager_new_filename_from_base(filename,"xtcorr");

	cpl_dfs_save_imagelist(allframes,
			NULL,
			parlist,
			usedframes,
			raw_cube_frame,
			resultlist,
			CPL_TYPE_DOUBLE,
			SPH_RECIPE_NAME_IFS_NEW_XTALK_CORRECTION,
			applist,
			NULL,
			SPH_PIPELINE_NAME_IFS,
			newfilename);

	cpl_imagelist_delete(imlist); imlist = NULL;
	cpl_imagelist_delete(resultlist); resultlist = NULL;
	cpl_frameset_delete(usedframes); usedframes = NULL;
	cpl_parameterlist_delete(parlist); parlist = NULL;
	cpl_propertylist_delete(applist); applist = NULL;
	cpl_free(newfilename); newfilename = NULL;
	SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Correct crosstalk in a single image in place
 * @param data			    the crosstalk correction parameters
 * @param image             the input image (unmodified)
 *
 * @return cpl_error_code
 *
 * Description:
 * This corrects the input image with the crosstalk correction
 * as implemented in Dino Mesas IDL script.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_crosstalkco_correct_image_in_place(sph_raw_image_corrector_data* data,cpl_image** image)
{
	// TODO: should really have an inherit in-place method that does not need to
	//       create a new image.
    cpl_image* corrected_image = NULL;

	cpl_image* sc_corrected_image =
			sph_crosstalkco_small_scale_crosstalk( *image, data->crosstalkco_sepmax, data->crosstalkco_bfac, data->crosstalkco_powfac );
	if (data->largescalecrosstalkco_apply){
		corrected_image =
			sph_crosstalkco_large_scale_crosstalk(
					sc_corrected_image, data->crosstalkco_lg_scale_corr_window,
					data->crosstalkco_threshold,
					data->crosstalkco_smoothing_fwhm,
					data->crosstalkco_stephist);
	} else {
		corrected_image = cpl_image_duplicate(sc_corrected_image);
	}
	cpl_image_delete(sc_corrected_image); sc_corrected_image = NULL;
	cpl_image_delete(*image);
	*image = corrected_image;
	SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


static
cpl_image* sph_crosstalkco_small_scale_crosstalk( const cpl_image* image_in,
                                                  cpl_size sepmax,
                                                  double bfac,
                                                  double powfac )
{
	cpl_size dimmat = sepmax * 2 + 1;
	cpl_size dimimg = cpl_image_get_size_x(image_in);
	cpl_image* windowed = NULL;
	cpl_size k,j,totsiz;
	cpl_vector* vec = NULL;
	cpl_vector* px = NULL;
	cpl_vector* py = NULL;
	cpl_image* corrected_image = NULL;
	double* data = NULL;

	double** matsub = cpl_calloc(dimmat,sizeof(double*));

	cpl_ensure( cpl_image_get_size_y(image_in) == dimimg, CPL_ERROR_ILLEGAL_INPUT, NULL);

	for ( k = 0; k < dimmat; ++k ) {
		matsub[k] = cpl_calloc(dimmat,sizeof(double));
		for ( j = 0; j < dimmat; ++j ) {
			if ( sepmax - k > 1 || k - sepmax > 1 || sepmax - j > 1 || j - sepmax > 1 ) {
				double rdist = sqrt(((double)(k - sepmax)) * ((double)(k - sepmax)) + ((double)(j - sepmax)) * ((double)(j - sepmax)));
				matsub[k][j] = 1.0/(1.0 + pow(rdist/bfac,powfac));
			}
			else {
				matsub[k][j] = 0.0;
			}
		}
	}

	windowed = cpl_image_extract(
			image_in,
			sepmax + 1,
			sepmax + 1,
			cpl_image_get_size_x(image_in) - sepmax,
			cpl_image_get_size_y(image_in) - sepmax);

	totsiz = cpl_image_get_size_x(windowed) * cpl_image_get_size_y(windowed);

	assert(totsiz > 10);

	vec = cpl_vector_wrap(
			totsiz,
			cpl_image_unwrap(windowed));

	cpl_vector_sort(vec,CPL_SORT_DESCENDING);

	px = cpl_vector_new( cpl_vector_get_size(vec) );
	py = cpl_vector_new( cpl_vector_get_size(vec) );

	corrected_image = cpl_image_duplicate(image_in);

	data = cpl_image_get_data(corrected_image);

	for ( k = 0; k < cpl_vector_get_size(vec); ++k ) {
            const double sr = cpl_vector_get(vec,k);
            double pyval = floor(sr / (double)( dimimg - sepmax * 2));
            const double pxval = sr - pyval * (double)(dimimg - sepmax * 2)
                + (double)sepmax;
            cpl_size minx, maxx, miny, maxy;
            double val;
            int bpix = 0;

            pyval += (double)sepmax;
            cpl_vector_set(px, k, pxval);
            cpl_vector_set(py, k, pyval);
            /* Subtract from subwindow */

            minx = CPL_MAX((cpl_size)pxval - sepmax, 0);
            maxx = CPL_MIN((cpl_size)pxval + sepmax, dimimg - 1);
            miny = CPL_MAX((cpl_size)pyval - sepmax, 0);
            maxy = CPL_MIN((cpl_size)pyval + sepmax, dimimg - 1);
            val = cpl_image_get(image_in, (cpl_size)pxval + 1,
                                (cpl_size)pyval + 1, &bpix);
            for ( cpl_size x = minx; x < maxx; ++x) {
                for ( cpl_size y = miny; y < maxy; ++y ) {
                    data[y * dimimg + x] -= val * matsub[x - minx][y - miny];
                }
            }
        }
	cpl_vector_delete(vec); vec = NULL;
	cpl_vector_delete(px); px = NULL;
	cpl_vector_delete(py); py = NULL;
	for ( k = 0; k < dimmat; ++k ) {
		cpl_free(matsub[k]);
	}

	cpl_free(matsub); matsub = NULL;

	return corrected_image;
}

static
cpl_image* sph_crosstalkco_large_scale_crosstalk( const cpl_image* image_in,
                                                  cpl_size dimsub,
                                                  double threshold,
                                                  double smoothing_fwhm,
                                                  double stephist)
{
	cpl_size dimimg = cpl_image_get_size_x(image_in);
	cpl_size dimimgct = dimimg / dimsub;
	cpl_vector* valinix;
	cpl_vector* valfinx;
	cpl_vector* valiniy;
	cpl_vector* valfiny;
	cpl_image* corrected_image;
	cpl_image* imgct;
	const double median = cpl_image_get_median(image_in);
	cpl_size konta = 0;
	cpl_size x,y,k,j;

	cpl_ensure(smoothing_fwhm > 0.0, CPL_ERROR_ILLEGAL_INPUT, NULL);

	valinix = cpl_vector_new( dimimgct * dimimgct );
	valfinx = cpl_vector_new( dimimgct * dimimgct );
	valiniy = cpl_vector_new( dimimgct * dimimgct );
	valfiny = cpl_vector_new( dimimgct * dimimgct );
	corrected_image = cpl_image_duplicate(image_in);
	imgct = cpl_image_new(dimimg, dimimg, CPL_TYPE_DOUBLE);

	for ( j = 0; j < dimimgct; ++j ) {
		for ( k = 0; k < dimimgct; ++k) {
			cpl_vector_set(valinix,konta, (double)k * (double)dimsub );
		    cpl_vector_set(valfinx,konta, cpl_vector_get(valinix,konta) + (double)dimsub - 1.0);
			cpl_vector_set(valiniy,konta, (double)j * (double)dimsub );
			cpl_vector_set(valfiny,konta, cpl_vector_get(valiniy,konta) + (double)dimsub - 1.0);
		    konta = konta + 1;
		}
	}


	cpl_image_threshold(corrected_image,-threshold,threshold,median,median);

	for ( k = 0; k < cpl_vector_get_size(valinix); ++k ) {
		cpl_image* imgsub = cpl_image_extract(corrected_image,
				(cpl_size)cpl_vector_get(valinix,k) + 1,
				(cpl_size)cpl_vector_get(valiniy,k) + 1,
				(cpl_size)cpl_vector_get(valfinx,k) + 1,
				(cpl_size)cpl_vector_get(valfiny,k) + 1);
		cpl_size wx = cpl_image_get_size_x(imgsub);
		cpl_size wy = cpl_image_get_size_y(imgsub);

		double minhist = cpl_image_get_min(imgsub);
		double maxhist = cpl_image_get_max(imgsub);

		cpl_size nsteps = (int)round((maxhist-minhist) / stephist );
		cpl_vector* vcount = cpl_vector_new(nsteps + 2);
		cpl_vector* ncount = cpl_vector_new(nsteps + 2);

                for ( y = 0; y < wy; ++y) {
                    for (x = 0; x < wx; ++x) {
                            int bpix = 0;
                            double val = cpl_image_get(imgsub, x + 1, y + 1, &bpix);
                            cpl_size index = (int)((val - minhist) / stephist) + 1;
                            if ( val <= maxhist && val >= minhist) {
                                cpl_vector_set(ncount, index, cpl_vector_get(ncount,index) + 1.0);
                            }
			}
		}
		{
			cpl_size maxc = 0;
			double valct = 0.0;
			for (cpl_size t = 0; t < nsteps; ++t) {
				if ( (cpl_size)cpl_vector_get(ncount,t) > maxc) {
					double vini = minhist + (double)t * stephist;
					double vfin = vini + stephist;
					maxc = (cpl_size)cpl_vector_get(ncount,t);
					valct = 0.5 * (vini + vfin);
				}
			}
			cpl_image_fill_window( imgct,
					(cpl_size)cpl_vector_get(valinix,k) + 1,
					(cpl_size)cpl_vector_get(valiniy,k) + 1,
					(cpl_size)cpl_vector_get(valfinx,k) + 1,
					(cpl_size)cpl_vector_get(valfiny,k) + 1, valct);
		}
		cpl_image_delete(imgsub); imgsub = NULL;
		cpl_vector_delete(vcount); vcount = NULL;
		cpl_vector_delete(ncount); ncount = NULL;

	}

	sph_fft_smoothe_image(imgct, NULL, smoothing_fwhm);
	cpl_image_subtract(corrected_image,imgct);

	cpl_vector_delete(valinix); valinix = NULL;
	cpl_vector_delete(valfinx); valfinx = NULL;
	cpl_vector_delete(valiniy); valiniy = NULL;
	cpl_vector_delete(valfiny); valfiny = NULL;

	cpl_image_delete(imgct);

	cpl_image_save(corrected_image,"dummy.fits",CPL_TYPE_UNSPECIFIED,NULL,CPL_IO_DEFAULT);

	return corrected_image;
}
