/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include <cpl.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "sph_master_frame.h"
#include "sph_fits.h"
#include "sph_cube.h"
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_pixel_descriptor.h"
#include "sph_pixel_description_table.h"
#include "sph_framelist.h"
#include "sph_pixel_polyfit_table.h"
#include "sph_smart_imagelist.h"
#include "sph_dataset.h"
#include "sph_dataset_stats.h"
#include "sph_keyword_manager.h"
#include "sph_utils.h"
#include "sph_filemanager.h"
#define SPH_CUBE_UNLINK_EXTS

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_cube Master cube object
 * @par Synopsis:
 * @code
 * typedef struct _sph_cube_
 * {
 *    cpl_imagelist       *   imlist;
 *    cpl_table           *   assoc_table;
 *    cpl_type                type;
 *    sph_interpol_alg        interpol_alg;
 *    cpl_propertylist    *   proplist;
 *    const char          *   filename;
 *    short int               file_exists;
 *    short int               image_extno;
 *    short int               badpixel_extno;
 *    short int               rmsmap_extno;
 *    short int               ncombmap_extno;
 *    short int               tbl_extno;
 *    short int               nx;
 *    short int               ny;
 * } sph_cube;
 * @endcode
 * @par Desciption:
 * This module provides functions to define and operate on a cube with
 * three dimensions. The sph_cube holds information of a datacube with
 * a third axis (image planes) where the image planes can be separated
 * a non-constant step-size apart. In this way it differs from the
 * Euro3D cube defined in other ESO reduction software.
 * The information about the location of the third axis is saved in a
 * FITS table extension, which contains a list of the planes and the
 * associated z-value.
 * To retrieve information from the cube, some function methods exist that
 * use interpolation between planes if necessary.
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/
sph_error_code SPH_CUBE_GENERAL             = SPH_CUBE_ERR_START + 0;
sph_error_code SPH_CUBE_NO_IMAGE_EXT        = SPH_CUBE_ERR_START + 1;
sph_error_code SPH_CUBE_NO_BADPIXEL_EXT     = SPH_CUBE_ERR_START + 2;
sph_error_code SPH_CUBE_ERR_INVALID_FITMODE = SPH_CUBE_ERR_START + 3;
sph_error_code SPH_CUBE_FAILED_PIXHIST      = SPH_CUBE_ERR_START + 4;


/*-----------------------------------------------------------------------------
 Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new sphere master cube object
 * @param filename      the filename to be used to store the cube
 *                      If the file already exists it is overwritten
 *                      as soon as the master cube is filled with data
 *
 * This creates a new empty sphere master cube associated to a given filename.
 * Note that the file is not created immediately after this function is called,
 * but rather only when the cube is filled with actual data, by appending images
 * or master frames for example.
 *
 * @return      pointer to the newly created sph_cube or null if an
 *              error occured
 */
/*----------------------------------------------------------------------------*/
sph_cube* sph_cube_new(const char* filename)
{
    sph_cube* self = NULL;

    if (filename == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        unlink(filename);
        self = cpl_calloc( 1, sizeof(*self));

        self->assoc_table = cpl_table_new( 0);
        self->proplist = cpl_propertylist_new();
        self->type = CPL_TYPE_UNSPECIFIED;

        self->filename = cpl_strdup(filename);

        self->file_exists = 0;
        self->nx = 0;
        self->ny = 0;
    }

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new sphere master cube object
 * @param filename      the filename to be used to store the cube
 *                      If the file already exists it is overwritten
 *                      as soon as the master cube is filled with data
 *
 * This creates a new empty sphere master cube associated to a given filename.
 * Note that the file is not created immediately after this function is called,
 * but rather only when the cube is filled with actual data, by appending images
 * or master frames for example.
 *
 * @return      pointer to the newly created sph_cube or null if an
 *              error occured
 */
/*----------------------------------------------------------------------------*/
sph_cube* sph_cube_new_dfs(const char* filename,
        const cpl_frameset* allframes,
        const cpl_frame* template_frame,
        const cpl_parameterlist* params,
        const char* tag,
        const char* recipe,
        const char* pipename,
        const cpl_propertylist* plist)
{
    sph_cube* self = NULL;

    if (filename == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (pipename == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (recipe == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (tag == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        self = sph_cube_new(filename);

        self->dfs_allframes      = allframes;
        self->dfs_template_frame = template_frame;
        self->dfs_paramlist      = params;
        self->dfs_pipename       = cpl_strdup(pipename);
        self->dfs_recipe         = cpl_strdup(recipe);
        self->dfs_tag            = cpl_strdup(tag);

        if ( plist ) {
            cpl_propertylist_append(self->proplist, plist);
            /* Update the header if required */
            sph_utils_update_header(self->proplist);
        }
    }
    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Append an master frame to the cube
 * @param self      The cube
 * @param mframe    The master frame to append
 * @param zvalue    The zvalue of the plane where the image is located on
 *
 * This function appends a master frame to the data cube. It adds the image,
 * badpixelmap, ncombmap and the RMSMAP to the cube. Any extension that
 * does not exist yet is created.
 *
 * @return Error code of the operation
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_cube_append_master(sph_cube* self,
                       const sph_master_frame* mframe,
                       float zvalue)
{
    const int nx = cpl_image_get_size_x(mframe->image);
    const int ny = cpl_image_get_size_y(mframe->image);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code(self   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mframe != NULL, CPL_ERROR_NULL_INPUT);

    if ( !self->file_exists ) {
    	self->nx = nx;
    	self->ny = ny;
    	self->file_exists = 1;
    } else if ( self->nx != nx || self->ny != ny ) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "Cube and master frame have incompatible "
                                     "sizes: (%d,%d <=> (%d,%d)",
                                     self->nx, nx, self->ny, ny);
    }
    sph_cube_append_image( self->filename,
                           mframe->image, self->proplist,
                           SPH_CUBE_IMAGE_EXT );
    sph_cube_append_image( self->filename,
                           mframe->badpixelmap, NULL,
                           SPH_CUBE_BADPIX_EXT );
    sph_cube_append_image( self->filename,
                           mframe->rmsmap,NULL,
                           SPH_CUBE_RMSMAP_EXT );
    sph_cube_append_image( self->filename,
                           mframe->ncombmap,NULL,
                           SPH_CUBE_NCOMBMAP_EXT );
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Append an image to the cube
 * @param fname      The filename of the file to append to
 * @param image    The image to append
 * @param pl        Propertylist to write (see note below!!)
 * @param ext    The extension of where the append the image to
 * This function appends an image to the data cube. Any extension that
 * does not exist yet is created.
 *
 * @return Error code of the operation
 * @note: the propertylist (if passed as non NULL) is written
 * directly without passing it through the keyword manager.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_cube_append_image(const char* fname,
                      const cpl_image* image,
                      const cpl_propertylist* pl,
                      int extend)
{
    cpl_error_code rerr;
    char* ftmpname;
    char basename[1024];
    char tail[256];
    FILE* fp = NULL;
    int file_exists = 0;
    cpl_imagelist* imlist = NULL;

    cpl_ensure_code(fname != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(image != NULL, CPL_ERROR_NULL_INPUT);

    sph_filemanager_split(fname, basename, tail);
    ftmpname = cpl_sprintf("%s_e%02d.%s", basename, extend, tail);
    fp = fopen(ftmpname,"r");
    if (fp) {
        fclose(fp); fp = NULL;
        file_exists = 1;
        if (pl != NULL) {
            const cpl_size plsz = cpl_propertylist_get_size(pl);
            cpl_msg_warning(cpl_func, "Ignoring %d-propertylist for extension "
                            "%d in file %s", (int)plsz, extend, fname);
            pl = NULL;
        }
    } else if (pl != NULL) {
        const cpl_size plsz = cpl_propertylist_get_size(pl);
        cpl_msg_info(cpl_func, "Using %d-propertylist for extension "
                     "%d in file %s", (int)plsz, extend, fname);
    } else {
        cpl_msg_info(cpl_func, "Using no propertylist for extension "
                     "%d in file %s", extend, fname);
    }

    imlist =  cpl_imagelist_new();
    CPL_DIAG_PRAGMA_PUSH_IGN(-Wcast-qual);
    cpl_imagelist_set(imlist, (cpl_image*)image, 0);
    CPL_DIAG_PRAGMA_POP;

    /* Make any final header updates */
    cpl_propertylist* tmp_pl = NULL;
    if (pl != NULL) {
      tmp_pl = cpl_propertylist_duplicate(pl);
      sph_utils_update_header(tmp_pl);
    }
    rerr = cpl_imagelist_save(imlist, ftmpname, CPL_TYPE_UNSPECIFIED, tmp_pl,
                              file_exists ? CPL_IO_APPEND : CPL_IO_CREATE);
    cpl_propertylist_delete(tmp_pl);

    (void)cpl_imagelist_unset(imlist, 0);
    cpl_imagelist_delete(imlist); 
    cpl_free(ftmpname);

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Finalise all extensions saved to a cube
 * @param fname      The filename base of the saved imagelists
 *
 * Create a new multi-extension file from all files that have
 * the same basename (given as the fname parameter) but end with
 *  "_eXX.fits" where X is the extension number.
 *
 * @return Error code of the operation
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_cube_finalise_file(const char* fname)
{
    char* ftmpname = NULL;
    char* basename = NULL;
    char* tail = NULL;
    FILE* fp = NULL;
    int file_exists = 1;
    cpl_imagelist* imlist = NULL;
    cpl_imagelist* imlist2 = NULL;
    cpl_image*      im = NULL;
    int     extend = 0;
    int     nextend = 0;
    int     pp = 0;
    int     p0 = 0;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* pl2 = NULL;
    cpl_ensure_code(fname, CPL_ERROR_NULL_INPUT);
    basename = cpl_calloc(strlen(fname) + 1,sizeof(char));
    tail = cpl_calloc(strlen(fname) + 1,sizeof(char));
    ftmpname = cpl_calloc(strlen(fname) + 40,sizeof(char));

    sph_filemanager_split(fname,basename,tail);
    while ( file_exists ) {
        file_exists = 0;
        sprintf(ftmpname,"%s_e%02d.%s",basename,nextend,tail);
        fp = fopen(ftmpname,"r");
        if (fp) {
            fclose(fp); fp = NULL;
            file_exists = 1;
            nextend++;
        }
    }
    fp = fopen(fname,"r");
    file_exists = 0;
    if (fp && nextend > 0 ) {
        fclose(fp); fp = NULL;
        if ( cpl_fits_count_extensions(fname) != nextend - 1) {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_INCOMPATIBLE_INPUT,
                    "A file %s already exists but has "
                    "different number of extensions(%d).", fname,
                    (int)cpl_fits_count_extensions(fname));
            goto EXIT;
        }
        file_exists = 1;
    }
    if ( file_exists ) {
        for (extend = 0; extend < nextend; ++extend) {
            sprintf(ftmpname,"%s_e%02d.%s",basename,extend,tail);
            imlist = cpl_imagelist_load(fname,CPL_TYPE_UNSPECIFIED,extend);
            pl = cpl_propertylist_load(fname,extend);
            imlist2 = cpl_imagelist_load(ftmpname,CPL_TYPE_UNSPECIFIED,0);
            pl2 = cpl_propertylist_load(ftmpname,0);
            SPH_ERROR_ENSURE_GOTO_EXIT(imlist2,cpl_error_get_code());
            SPH_ERROR_ENSURE_GOTO_EXIT(imlist,cpl_error_get_code());
            p0 = cpl_imagelist_get_size(imlist);
            for (pp = 0; pp < cpl_imagelist_get_size(imlist2); ++pp) {
                im = cpl_imagelist_get(imlist2,pp);
                cpl_imagelist_set(imlist,cpl_image_duplicate(im),pp + p0);
                SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
            }
            cpl_imagelist_delete(imlist2);imlist2 = NULL;

            cpl_propertylist_append( pl, pl2 );
            /* Make any final header updates */
            sph_utils_update_header(pl);
            cpl_imagelist_save(imlist,ftmpname,CPL_TYPE_UNSPECIFIED,pl,CPL_IO_CREATE);
            cpl_propertylist_delete(pl);pl = NULL;
            cpl_propertylist_delete(pl2);pl2 = NULL;
            cpl_imagelist_delete(imlist); imlist = NULL;
        }
    }
    for (extend = 0; extend < nextend; ++extend) {
        sprintf(ftmpname,"%s_e%02d.%s",basename,extend,tail);
        imlist = cpl_imagelist_load(ftmpname,CPL_TYPE_UNSPECIFIED,0);
        SPH_ERROR_ENSURE_GOTO_EXIT(imlist,cpl_error_get_code());

        pl = cpl_propertylist_load(ftmpname,0);
        /* Make any final header updates */
        sph_utils_update_header(pl);
        if ( extend == 0 ) {
            cpl_imagelist_save(imlist,fname,CPL_TYPE_UNSPECIFIED,pl,CPL_IO_CREATE);
        }
        else cpl_imagelist_save(imlist,fname,CPL_TYPE_UNSPECIFIED,pl,CPL_IO_EXTEND);
        cpl_propertylist_delete(pl);pl = NULL;
        cpl_imagelist_delete(imlist); imlist = NULL;
#ifdef SPH_CUBE_UNLINK_EXTS
#ifdef _OPENMP
#pragma omp critical
#endif
        {
        	unlink(ftmpname);
        }
#endif
    }
EXIT:
    cpl_free(basename); basename = NULL;
    cpl_free(ftmpname); ftmpname = NULL;
    cpl_free(tail); tail = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Finalise all files in frameset
 * @param frameset
 *
 * @return error code of the oepration
 *
 * This function will finalise all filles that are inside the
 * given frameset and are valid for finalisation. The function
 * does this by just looking at all the filenames and
 * checking if corresponding "eXX" files exist. These
 * are then finalised.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_cube_finalise_all(const cpl_frameset* frameset) {
    const cpl_size nf = cpl_frameset_get_size(frameset);

    cpl_ensure_code(frameset != NULL, CPL_ERROR_NULL_INPUT);

    for (cpl_size i = 0; i < nf; ++i) {
        const cpl_frame* aframe = cpl_frameset_get_position_const(frameset, i);
        const char* fname = cpl_frame_get_filename(aframe);
        sph_cube_finalise_file(fname);
    }

    return cpl_error_set_where(cpl_func);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Append an master frame to the cube
 * @param self      The cube
 * @param dimage    The double image to append
 * @param zvalue    The zvalue of the plane where the image is located on
 *
 * This function appends a double_image to the data cube. It adds the image,
 * badpixelmap, ncombmap and the RMSMAP to the cube. Any extension that
 * does not exist yet is created.
 *
 * @return Error code of the operation
 *
 */
/*----------------------------------------------------------------------------*/
int sph_cube_append_double_image(sph_cube* self,
                                 const sph_double_image* dimage,
                                 float zvalue)
{
    int nx = 0;
    int ny = 0;

    cpl_ensure_code(self   != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(dimage != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(dimage->iframe != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(dimage->pframe != NULL, CPL_ERROR_NULL_INPUT);

    nx = cpl_image_get_size_x(dimage->iframe->image);
    ny = cpl_image_get_size_y(dimage->iframe->image);

    if ( (self->nx != nx || self->ny != ny) && self->file_exists )
    {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
        		"Dont have the same size of cube "
                "and double image!");
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if ( !self->file_exists )
    {
        self->nx = nx;
        self->ny = ny;
        self->file_exists = 1;

    }
    sph_cube_append_image( self->filename,
            dimage->iframe->image,NULL,
            SPH_CUBE_IMAGE_EXT );

    sph_cube_append_image( self->filename,
            dimage->iframe->badpixelmap,NULL,
            SPH_CUBE_BADPIX_EXT );

    sph_cube_append_image( self->filename,
            dimage->iframe->rmsmap, NULL,
            SPH_CUBE_RMSMAP_EXT );

    sph_cube_append_image( self->filename,
            dimage->iframe->ncombmap, NULL,
            SPH_CUBE_NCOMBMAP_EXT );


    sph_cube_append_image( self->filename,
            dimage->pframe->image, NULL,
            SPH_CUBE_IMAGE2_EXT );

    sph_cube_append_image( self->filename,
            dimage->pframe->badpixelmap, NULL,
            SPH_CUBE_BADPIX2_EXT );

    sph_cube_append_image( self->filename,
            dimage->pframe->rmsmap, NULL,
            SPH_CUBE_RMSMAP2_EXT );

    sph_cube_append_image( self->filename,
            dimage->pframe->ncombmap, NULL,
            SPH_CUBE_NCOMBMAP2_EXT );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete the master cube.
  @param    self     the master cube to delete, may be NULL

  Deallocate the master cube and all associated memory.
  The named file or any extensions in it are NOT modified.

 */
/*----------------------------------------------------------------------------*/
void sph_cube_delete(sph_cube* self)
{
    if (self) {
        cpl_table_delete(self->assoc_table);
        cpl_free(self->filename);
        cpl_imagelist_delete(self->imlist);
        cpl_propertylist_delete(self->proplist);
        cpl_free(self->dfs_tag);
        cpl_free(self->dfs_pipename);
        cpl_free(self->dfs_recipe);
        cpl_free(self);
    }
}

/**@}*/
