/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_dataset.h"
#include "sph_error.h"
#include <float.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_dataset dataset object
 *
 * @par Synopsis:
 * @code
 *   typedef struct _sph_dataset_
 * {
 *   int                     npoints;
 *   cpl_vector*             spec;
 *   cpl_vector*             xvalues;
 * } sph_dataset;
 *
 * @endcode
 *
 * @par Descirption:
 * This object represents an individual dataset. It consists of
 * a region in absolute detector coordinates where the dataset is
 * defined, and a cpl_image with the region.
 * The dataset structure associates itself with an image, which is a two
 * dimensional representation of the dataset. It is usually a cut-out from
 * a detector image the datasets was extracted from.
 * With the sph_datasets_collapse function, a one dimensional representation
 * of a dataset can also be obtained -- with the relevant information of the
 * two axis (x-values and value) being saved in two vectors, yvalues and xvalues,
 * with the same size.
 *
 * The functions in this module allow datasets to be created and extraced from
 * detector images.
 *
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

sph_error_code SPH_DATASET_GENERAL          = SPH_DATASET_ERR_START + 0;
sph_error_code SPH_DATASET_CANT_COLLAPSE    = SPH_DATASET_ERR_START + 1;
sph_error_code SPH_DATASET_OUT_OF_BOUNDS    = SPH_DATASET_ERR_START + 2;
sph_error_code SPH_DATASET_NOT_COLLAPSED    = SPH_DATASET_ERR_START + 3;


/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new (empty) dataset
 *
 * @return pointer to newly created dataset or NULL on error
 *
 * This creates a new empty dataset.
 */
/*----------------------------------------------------------------------------*/
sph_dataset*
sph_dataset_new( int npoints, double minxvalues, double maxxvalues ) {
    sph_dataset*   self    = NULL;
    double          dx      = 0.0;
    int             ii      = 0;

    self = cpl_calloc(1, sizeof(sph_dataset));

    if ( !self ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not allocate dataset" );
        return NULL;
    }
    if ( npoints < 2 ) {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Cant create "
                                 "dataset with 1 or 0 points!" );
        return NULL;
    }

    self->npoints = npoints;
    self->xvalues = cpl_vector_new( npoints );

    self->yvalues = cpl_vector_new( npoints );

    dx = ( maxxvalues - minxvalues ) / ( npoints - 1 );
    if ( dx == 0.0 ) {
        //dx = 1.0;
    }
    for ( ii = 0; ii < cpl_vector_get_size( self->yvalues ); ++ii) {
        cpl_vector_set( self->xvalues, ii, minxvalues + dx * ii );
        cpl_vector_set( self->yvalues, ii, 0 );
    }

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new dataset loading it from a FITS binary table
 *
 * @param czFilename        the filename of the FITS table to load from
 *
 * @return pointer to newly created dataset or NULL on error
 *
 * This creates a new dataset from the specified FITS filename. The table
 * must be in the 1st extension.
 */
/*----------------------------------------------------------------------------*/
sph_dataset* sph_dataset_load( const char* czFilename ) {
    sph_dataset*          self        = NULL;
    cpl_table*              tab         = NULL;
    int                     npoints     = 0;
    double                  lmin        = 0.0;
    double                  lmax        = 0.0;
    double                  val         = 0.0;
    int                     rej         = 0;
    int                     ii          = 0;

    tab = cpl_table_load( czFilename, 1, 0 );
    if ( tab == NULL ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "Could not load the datasets table from file.");
        return NULL;
    }
    if ( cpl_table_get_ncol( tab ) != 2 ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "datasets table does not have 2 columns!");
        return NULL;
    }
    npoints = cpl_table_get_nrow( tab );
    if ( npoints < 2 ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "datasets table has less than 2 rows!");
        return NULL;
    }
    lmin = cpl_table_get( tab, "x-values", 0, &rej );
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "Could not get the minimum xvalues from table."
                         "Check that the \"x-values\" column is there.");
        return NULL;
    }
    lmax = cpl_table_get( tab, "x-values", npoints - 1, &rej );
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "Could not get the minimum xvalues from table."
                         "Check that the \"x-values\" column is there.");
        return NULL;
    }
    self = sph_dataset_new( npoints, lmin, lmax );

    if ( self ) {
        for (ii = 0; ii < npoints; ++ii) {
            rej = 0;
            val = cpl_table_get(tab, "y-values", ii, &rej );
            if ( rej == 0 ) {
                cpl_vector_set( self->yvalues, ii, val);
            }
        }
    }
    cpl_table_delete(tab);
    return self;

}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new dataset from image region
 *
 * @param image     The image from which the dataset is extracted
 *
 * @param minx      The x-pixel of the bottom left corner of the window
 * @param miny      The y-pixel of the bottom left corner of the window
 * @param maxx      The x-pixel of the top right corner of the window
 * @param maxy      The y-pixel of the top right corner of the window
 *
 * @return pointer to newly created dataset or NULL on error
 *
 * This creates a new dataset from the image in the specified rectangular region.
 * Note that the coordinates of minx, maxx, miny and maxy are such that the
 * the bottom left pixel of the full image has value 0,0 in those coordinates
 * (this is different from the CPL_IMAGE definition where the bottom left pixel
 * has 1,1 ).
 */
/*----------------------------------------------------------------------------*/
sph_dataset*
sph_dataset_new_extract( const cpl_image* image, int minx, int miny,
                         int maxx, int maxy ) {
    sph_dataset* self = NULL;
    cpl_image*   img  = cpl_image_extract( image, minx + 1, miny + 1,
                                           maxx + 1, maxy + 1 );

    if (img == NULL) {
        (void)cpl_error_set_where(cpl_func);
    } else {

        self = cpl_calloc(1, sizeof(*self));

        if ( maxx - minx > maxy - miny ) {
            sph_dataset_collapse( self, img, minx, maxx );
        } else {
            sph_dataset_collapse( self, img, miny, maxy );
        }
        cpl_image_delete(img);
    }

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new dataset from image region
 *
 * @param image     The image from which the dataset is extracted
 *
 * @param npoints   The number of bins in the histogram.
 *
 * @return pointer to newly created dataset or NULL on error
 *
 * This creates a new dataset from the image by doing a histogram of
 * pixel values
 *
 */
/*----------------------------------------------------------------------------*/
sph_dataset*
sph_dataset_new_pixel_histogram( const cpl_image* image, double min, double max,
                                 int npoints ) {
    sph_dataset*    self    = NULL;

    if ( !image ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "No image given as input." );
    } else {
        self = sph_dataset_new( npoints, min, max );

        if ( !self ) {
            sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                             __func__, __LINE__,
                             SPH_ERROR_ERROR, "Could not allocate dataset" );
        } else {
            const cpl_size nx = cpl_image_get_size_x( image );
            const cpl_size ny = cpl_image_get_size_y( image );

            for (cpl_size xx = 0; xx < nx; ++xx) {
                for (cpl_size yy = 0; yy < ny; ++yy) {
                    int rej = 0;
                    const double val =
                        cpl_image_get( image, xx + 1, yy + 1, &rej );

                    if ( !rej && val >= min && val <= max ) {
                        sph_dataset_add_value( self, val, 1.0 );
                    }
                }
            }
        }
    }
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Collapse dataset and recalculate 1D dataset
 *
 * @param self  The dataset to collapse
 *
 * @return Error code of the operation
 *
 * The dataset is (re-)collapsed into one dimension.
 *
 */
/*----------------------------------------------------------------------------*/
int sph_dataset_collapse( sph_dataset* self,
                          const cpl_image* image,
                          double minl,
                          double maxl ) {
    int             rerr    = CPL_ERROR_NONE;
    int             xx      = 0;
    int             yy      = 0;
    int             nx      = 0;
    int             ny      = 0;
    double          rowsum  = 0.0;
    double          value   = 0.0;
    double            dx        = 0.0;
    int             badpix  = 0;
    int             nrows   = 0;
    int             goodcount = 0;

    if ( !self ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "No self." );
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !image ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "No dataset image in self." );
        return CPL_ERROR_NULL_INPUT;
    }

    nx = cpl_image_get_size_x(image);
    ny = cpl_image_get_size_y(image);

    if ( !self->yvalues ) {
        self->yvalues = cpl_vector_new( 1 );
    }
    else {
        cpl_vector_delete( self->yvalues );
        self->yvalues = cpl_vector_new( 1 );
    }
    rerr |= cpl_vector_set_size(self->yvalues, 1);
    if ( rerr ) {
        sph_error_raise(SPH_DATASET_CANT_COLLAPSE,
                        __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Cant collapse datasets"
                        " from datasetsl image "
                        " -- cant create input cpl vector.\n"
                        "CPL says: %s", cpl_error_get_message());

        return (int)cpl_error_get_code();
    }

    for (yy = 0; yy < ny; ++yy) {
        rowsum = 0.0;
        goodcount = 0;
        for (xx = 0; xx < nx; ++xx) {
            value = cpl_image_get(image, xx + 1, yy + 1, &badpix);
            if ( badpix == 0) {
                rowsum = rowsum + value;
                goodcount = goodcount + 1;
            }
        }
        nrows = nrows + 1;
        rerr |= cpl_vector_set_size( self->yvalues, nrows);
        if ( !rerr && goodcount > 0 ) {
            rerr |= cpl_vector_set( self->yvalues, nrows - 1, rowsum / goodcount );
        }
        else if (rerr) {
            sph_error_raise(SPH_DATASET_CANT_COLLAPSE,
                            __FILE__, __func__, __LINE__,
                            SPH_ERROR_ERROR, "Cant collapse datasets"
                            " from datasetsl image "
                            " -- cant expand  input cpl vector.\n"
                            "CPL says: %s", cpl_error_get_message());

            return (int)cpl_error_get_code();
        }
        else {
            rerr |= cpl_vector_set( self->yvalues, nrows - 1, 0.0 );
        }
    }
    if ( self->xvalues ) {
        cpl_vector_delete( self->xvalues );
        self->xvalues = NULL;
    }
    self->xvalues = cpl_vector_new( cpl_vector_get_size( self->yvalues ) );

    dx = ( maxl - minl ) / ( cpl_vector_get_size( self->yvalues ) - 1 );
    if ( dx == 0.0 ) {
        //dx = 1.0;
    }
    for (xx = 0; xx < cpl_vector_get_size( self->yvalues ); ++xx) {
        cpl_vector_set( self->xvalues, xx, minl + dx * xx );
    }

    self->npoints = cpl_vector_get_size( self->yvalues );
    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Fit a gaussian to the 1-D dataset
 *
 * @param self  The dataset to fit
 * @param x0    The best fit peak position
 * @param sigma The best fit sigma of the gaussian
 * @param chi   The reduced chi-squared of the best fit
 *
 * @return Error code of the operation
 *
 * A gaussian fit to the 1D dataset is performed and the best fit values
 * returned
 *
 */
/*----------------------------------------------------------------------------*/
int sph_dataset_fit_gauss( const sph_dataset* self,
                           double*      x0,
                           double*      sigma,
                           cpl_vector*  errs,
                           double*      chi ) {
    int             rerr    = CPL_ERROR_NONE;
    double          area    = 0.0;
    double          offset  = 0.0;

    if ( !self ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "No self." );
        return CPL_ERROR_NULL_INPUT;
    }

    if ( !self->yvalues ) {
        sph_error_raise( SPH_DATASET_NOT_COLLAPSED, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "First collapse the dataset!" );
        return SPH_DATASET_NOT_COLLAPSED;
    }
    if ( !self->xvalues ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                __func__, __LINE__,
                SPH_ERROR_ERROR, "could not set 1D dataset" );
        return CPL_ERROR_NULL_INPUT;
    }


    rerr = cpl_vector_fit_gaussian( self->xvalues, NULL, self->yvalues,
                             errs, CPL_FIT_ALL, x0, sigma,
                             &area, &offset, NULL, chi, NULL);

    if ( rerr != CPL_ERROR_NONE && rerr != CPL_ERROR_CONTINUE ) {
        SPH_RAISE_CPL;
    }
    return rerr;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief get the x value of the maximal y yvalues in the dataset
 *
 * @param self      The dataset
 *
 * @return x-value of the max in dataset, -1 on error.
 *
 * Returns the x-value of the maximum of the dataset.
 */
/*----------------------------------------------------------------------------*/
double sph_dataset_get_max_xpos( const sph_dataset* self ) {
    int             ind         = 0;
    int             ii          = 0;
    double          result      = 0.0;
    double          maxvalue    = 0.0;
    if ( !self ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "No self." );
        return CPL_ERROR_NULL_INPUT;
    }

    if ( !self->yvalues ) {
            sph_error_raise( SPH_DATASET_NOT_COLLAPSED, __FILE__,
                             __func__, __LINE__,
                             SPH_ERROR_ERROR, "First collapse the dataset!" );
            return SPH_DATASET_NOT_COLLAPSED;
    }
    if ( !self->xvalues ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                __func__, __LINE__,
                SPH_ERROR_ERROR, "could not set 1D dataset" );
        return CPL_ERROR_NULL_INPUT;
    }

    maxvalue = cpl_vector_get( self->yvalues, 0 );
    //TODO: This is too simple and inefficient! Need to use GSL at some point.
    for (ii = 0; ii < self->npoints; ++ii) {
        if ( cpl_vector_get( self->yvalues, ii ) > maxvalue ) {
            maxvalue = cpl_vector_get( self->yvalues, ii );
            ind = ii;
        }
    }

    if ( ind < 0 || ind >= cpl_vector_get_size( self->yvalues ) ) {
        sph_error_raise( SPH_DATASET_OUT_OF_BOUNDS,
                        __FILE__, __func__,
                        __LINE__, SPH_ERROR_ERROR,
                        "The x-values specified was out of bounds --"
                        "no coressponding value found.");
        return -1;
    }
    result = cpl_vector_get( self->xvalues, ind );
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief add to the value at a given x-value
 *
 * @param self      The dataset
 * @param xvalue    The x-value
 * @param toadd     The value to add
 *
 * @return error code of the operation.
 *
 * Adss a value to the value of the dataset at the given x-values. If the
 * x-values requested is outside the boundaries, the routine will
 * return and raise an error.
 */
/*----------------------------------------------------------------------------*/
int sph_dataset_add_value( sph_dataset* self, double value, double toadd ) {
    int             ind     = 0;
    double          result  = 0.0;

    ind = sph_dataset_get_index( self, value );

    if ( ind < 0 ) {
        sph_error_raise( SPH_DATASET_OUT_OF_BOUNDS,
                        __FILE__, __func__,
                        __LINE__, SPH_ERROR_ERROR,
                        "The x-values specified was out of bounds --"
                        "no coressponding value found.");
        return SPH_DATASET_OUT_OF_BOUNDS;
    }

    result = cpl_vector_get( self->yvalues, ind );

    result = result + toadd;

    cpl_vector_set( self->yvalues, ind, result );

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief get the value at a given x-values
 *
 * @param self      The dataset
 * @param xvalues    The x-values
 *
 * @return value of the dataset, -1 on error.
 *
 * Returns the value of the dataset at the given x-values. If the
 * x-values requested is outside the boundaries, the routine will
 * return -1 and rais an error.
 */
/*----------------------------------------------------------------------------*/
double sph_dataset_get_value( const sph_dataset* self, double value ) {
    int                ind        = 0;
    double          result  = 0.0;

    ind = sph_dataset_get_index( self, value );

    if ( ind < 0 ) {
        sph_error_raise( SPH_DATASET_OUT_OF_BOUNDS,
                        __FILE__, __func__,
                        __LINE__, SPH_ERROR_ERROR,
                        "The x-values specified was out of bounds --"
                        "no coressponding value found.");
        return -1;
    }
    result = cpl_vector_get( self->yvalues, ind );
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief get the index at a given x-values
 *
 * @param self       The dataset
 * @param xvalues    The x-values
 *
 * @return value of index, -1 on error.
 *
 * Returns the value of the dataset index at the given x-values. If the
 * x-values requested is outside the boundaries, the routine will
 * return -1 and raise an error.
 */
/*----------------------------------------------------------------------------*/
int sph_dataset_get_index( const sph_dataset* self, double value ) {
    int             ind     = 0;
    double          result  = 0.0;
    double          xmax    = 0.0;
    double          xmin    = 0.0;

    if ( !self ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "No self." );
        return CPL_ERROR_NULL_INPUT;
    }

    if ( !self->yvalues ) {
            sph_error_raise( SPH_DATASET_NOT_COLLAPSED, __FILE__,
                             __func__, __LINE__,
                             SPH_ERROR_ERROR, "First collapse the dataset!" );
            return SPH_DATASET_NOT_COLLAPSED;
    }
    if ( !self->xvalues ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                __func__, __LINE__,
                SPH_ERROR_ERROR, "could not set 1D dataset" );
        return CPL_ERROR_NULL_INPUT;
    }

    xmax = cpl_vector_get_max( self->xvalues);
    xmin = cpl_vector_get_min( self->xvalues);

    if ( value > xmax + DBL_EPSILON * fabs( value ) ||
         value < xmin - DBL_EPSILON * fabs( value ) ) {
        sph_error_raise( SPH_DATASET_OUT_OF_BOUNDS,
                        __FILE__, __func__,
                        __LINE__, SPH_ERROR_ERROR,
                        "The x-value %f specified was out of bounds.", value);
        return -1;
    }

    if ( fabs( value - xmax ) < DBL_EPSILON * fabs( value ) ) {
        ind = self->npoints - 1;
    }
    else {
        result = ( value - xmin ) * (double)self->npoints / ( xmax - xmin );
        ind = (int)result;
    }



    if ( ind < 0 || ind >= cpl_vector_get_size( self->yvalues ) ) {
        sph_error_raise( SPH_DATASET_OUT_OF_BOUNDS,
                        __FILE__, __func__,
                        __LINE__, SPH_ERROR_ERROR,
                        "The x-values specified was out of bounds --"
                        "no coressponding value found.");
        return -1;
    }
    return ind;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Save the dataset as ascii.
  @param    self        the dataset to save
  @param    filename    the filename to save as
  @return   0 on succes, -1 if input pointer was NULL.

  Save the dataset as a FITS table.
 */
/*----------------------------------------------------------------------------*/
int sph_dataset_save_ascii( const sph_dataset* self, const char* filename) {
    cpl_table*          tab     = NULL;
    int                 ii      = 0;
    int                 np      = 0;
    FILE*               fp = NULL;

    if ( self == NULL ) {
        return -1;
    }

    fp = fopen(filename,"w");
    cpl_ensure_code(fp,CPL_ERROR_FILE_IO);

    np = cpl_vector_get_size( self->xvalues );

    tab = cpl_table_new( np );

    cpl_table_new_column( tab, "x-values", CPL_TYPE_DOUBLE );
    cpl_table_new_column( tab, "y-values", CPL_TYPE_DOUBLE );

    for (ii = 0; ii < np; ++ii) {
        cpl_table_set( tab, "x-values", ii, cpl_vector_get( self->xvalues, ii ) );
        cpl_table_set( tab, "y-values", ii, cpl_vector_get( self->yvalues, ii ) );
    }

    cpl_table_dump( tab, 0, 1000000000, fp );
    cpl_table_delete( tab );
    fclose(fp);
    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Save the dataset.
  @param    self        the dataset to save
  @param    filename    the filename to save as
  @return   0 on succes, -1 if input pointer was NULL.

  Save the dataset as a FITS table.
 */
/*----------------------------------------------------------------------------*/
int sph_dataset_save( sph_dataset* self, const char* filename) {
    cpl_table*          tab     = NULL;
    int                 ii      = 0;
    int                 np      = 0;

    if ( self == NULL ) {
        return -1;
    }


    np = cpl_vector_get_size( self->xvalues );

    tab = cpl_table_new( np );

    cpl_table_new_column( tab, "x-values", CPL_TYPE_DOUBLE );
    cpl_table_new_column( tab, "y-values", CPL_TYPE_DOUBLE );

    for (ii = 0; ii < np; ++ii) {
        cpl_table_set( tab, "x-values", ii, cpl_vector_get( self->xvalues, ii ) );
        cpl_table_set( tab, "y-values", ii, cpl_vector_get( self->yvalues, ii ) );
    }

    cpl_table_save( tab, NULL, NULL, filename, CPL_IO_DEFAULT );
    cpl_table_delete( tab );
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete the dataset.
  @param    self        the dataset to delete
  @return   0 on succes, -1 if input pointer was NULL.

  Deallocate the dataset, the associated image and the 1D dataset.
 */
/*----------------------------------------------------------------------------*/
int sph_dataset_delete( sph_dataset* self) {
    if ( self == NULL ) {
        return -1;
    }

    if ( self->yvalues ) {
        cpl_vector_delete(self->yvalues);
    }
    if ( self->xvalues ) {
        cpl_vector_delete(self->xvalues);
    }
    cpl_free(self);
    return 0;
}


/**@}*/
