/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <math.h>
#include "sph_error.h"
#include "sph_common_science.h"
#include "sph_master_frame.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_filemanager.h"
#include "sph_keyword_manager.h"
#include "sph_fits.h"
#include "sph_fft.h"
#include "sph_smart_imagelist.h"
#include "sph_differential_imaging.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Create speckles images
 * @param all_frames    the input frameset of (reduced) frames
 * @param maxframes     the maximum number of frames to combine per speckleframe
 * @param coll_alg      the collapse algorithm
 * @param assoc_array   optional cpl_array (output) to associate frames.
 *
 * @return new frameset of all speckle frames or NULL on error
 *
 * Creates a new list of speckle images from the input set of reduced frames. For
 * each set of maxframes of the input frames a new speckle frame is created
 * by combining the frames using the collapse algorithm, which should be either
 * MEAN or MEDIAN. No transformations are performed and the input images
 * should already be reduced and centered correctly.
 *
 * To combine all frames in the framelist to a single speckle frame, set maxframes
 * to MAX_INT or cpl_frameset_get_size(all_frames).
 *
 * To help use of the produced speckle frames, which are saved as master frames,
 * an optional output of an association array is produced. This array contains
 * for each element the corresponding number of the speckle frame in the output
 * speckle frameset.
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_differential_imaging_calculate_speckles(
        const cpl_frameset* all_frames, int maxframes,
        sph_collapse_algorithm coll_alg,
        cpl_array** assoc_array) {
    const int       nz = cpl_frameset_get_size(all_frames);
    int             ff      = 0;
    int             mf      = 0;
    cpl_frame*      aframe  = NULL;
    cpl_frameset*   speckle_frames  = NULL;
    cpl_frameset*   tempframeset    = NULL;
    sph_master_frame*   speckles = NULL;
    cpl_frame*          tmpframe = NULL;

    cpl_ensure( all_frames, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( maxframes > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure( coll_alg == SPH_COLL_ALG_MEAN || coll_alg == SPH_COLL_ALG_MEDIAN,
            CPL_ERROR_ILLEGAL_INPUT, NULL);

    speckle_frames = cpl_frameset_new();
    if ( assoc_array ) {
        *assoc_array = cpl_array_new(nz, CPL_TYPE_INT);
        cpl_ensure(*assoc_array,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    }
    for (ff = 0; ff < nz; ++ff) {
        if ( mf == 0 ) {
            tempframeset = cpl_frameset_new();
        }
        aframe = cpl_frame_duplicate( cpl_frameset_get_position_const(all_frames,
                                                                      ff) );
        if ( assoc_array ) {
            cpl_array_set(*assoc_array,ff,cpl_frameset_get_size(speckle_frames));
        }
        cpl_frameset_insert(tempframeset,aframe);
        mf = cpl_frameset_get_size(tempframeset);
        if ( mf >= maxframes ||
               ff == nz - 1 ) {
            speckles = sph_common_science_combine(tempframeset,coll_alg,0,1,-1);
            tmpframe = sph_filemanager_create_temp_frame("DIFF_IMAGING_SPECKLE","SPECKLE_FRAME",CPL_FRAME_GROUP_CALIB);
            sph_master_frame_save(speckles,cpl_frame_get_filename(tmpframe),NULL);
            cpl_frameset_insert(speckle_frames,tmpframe);
            tmpframe = NULL;
            sph_master_frame_delete(speckles); speckles = NULL;
            cpl_frameset_delete(tempframeset);tempframeset = NULL;
            mf = 0;
        }
    }
//    cpl_frameset_delete(tempframeset); tempframeset = NULL;
    return speckle_frames;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Create new differential imaging processing
 * @param raws raw frameset (maybe NULL)
 *
 * @return new pointer to sph_differential_imaging or NULL on error
 *
 * Creates a new object for differential imaging processing.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_differential_imaging*
sph_differential_imaging_new(cpl_frameset* rawframes) {
    sph_differential_imaging* self = 
        cpl_calloc(1, sizeof(*self));
    self->extension = 0;
    self->rawframes = rawframes;
    self->fft = sph_fft_new(SPH_FFT_FFTW3_DP);
    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief  Set the processing flags
 * @param self      the differential imaging processing object
 * @param ditherin  1 to switch dithering on, 0 for off.
 * @param scalein  1 to switch scaling on, 0 for off.
 * @param rotatein  1 to switch rotation on, 0 for off.
 *
 * @return error code
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_differential_imaging_set_flags( sph_differential_imaging* self,
        int ditherin, int scalein, int rotatein)
{
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    self->scalef = scalein;
    self->ditherf = ditherin;
    self->rotatef = rotatein;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the transform parameters all in one go.
 * @param self      the differential imaging processing object
 * @param dx        the dithering in x
 * @param dx        the dithering in y
 * @param scale     the scaling (left frame relative to right)
 * @param angle     the angle (clockwise in degrees)
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_differential_imaging_set_transforms( sph_differential_imaging* self,
        double dx, double dy, double scale, double angle) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    self->dx = dx;
    self->dy = dy;
    self->scale = scale;
    self->angle = angle;
    self->ditherf = 1;
    self->scalef = 1;
    self->rotatef = 1;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the extension in RAW frames to read image data from.
 * @param self      the differential imaging processing object
 * @param extn      the extension number (starting from 0)
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_differential_imaging_set_extension( sph_differential_imaging* self,
        unsigned int extn) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    self->extension = extn;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the transform parameters all in one go.
 * @param self      the differential imaging processing object
 * @param dx        the dithering in x
 * @param dx        the dithering in y
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_differential_imaging_set_dither( sph_differential_imaging* self,
        double dx, double dy) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    self->dx = dx;
    self->dy = dy;
    self->ditherf = 1;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the scaling parameter.
 * @param self      the differential imaging processing object
 * @param scale     the scaling (left frame relative to right)
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_differential_imaging_set_scale( sph_differential_imaging* self,
        double scale) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    self->scale = scale;
    self->scalef = 1;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the rotation parameter.
 * @param self      the differential imaging processing object
 * @param angle     the angle (clockwise in degrees)
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_differential_imaging_set_angle( sph_differential_imaging* self,
        double angle) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    self->angle = angle;
    self->rotatef = 1;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set calibration files to apply
 * @param self      the differential imaging processing object
 * @param dark      a dark to subtract (maybe NULL)
 * @param flat      a flat to apply (maybe NULL)
 * @param distmap   a distortion model (maybe NULL)
 *
 * @return error code of the operation
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_differential_imaging_set_calibrations( sph_differential_imaging* self,
        sph_master_frame* dark, sph_master_frame* flat,
        sph_distortion_model* distmap) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    self->dark = dark;
    self->flat = flat;
    self->distmap = distmap;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the next raw science image
 * @param self
 *
 * @return the next raw science frame or NULL if none left or error.
 *
 * This returns the next raw science image unprocessed.
 * The function always reads from the extension currently set in self,
 * and automatically detects if the frame is a cube or not. It will go
 * through all images (planes) in all frames.
 *
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_differential_imaging_get_next_raw(sph_differential_imaging* self) {
    int         readnextplane = 0;
    cpl_image*  im = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(self->rawframes,CPL_ERROR_ILLEGAL_INPUT,NULL);
    if ( self->iscube && self->currentplane < self->nplanes ) {
        readnextplane = 1;
    }
    else {
        if ( self->current_frame == NULL ) {
            self->current_frame = cpl_frameset_get_first(self->rawframes);
            if ( self->current_frame == NULL ) {
                cpl_error_reset();
                return NULL;
            }
        }
        else {
            self->current_frame = cpl_frameset_get_next(self->rawframes);
            if ( self->current_frame == NULL ) {
                cpl_error_reset();
                return NULL;
            }
        }
        self->iscube = sph_fits_test_iscube(
        		cpl_frame_get_filename(
        				self->current_frame), 0);
        self->currentplane = 0;
        self->nplanes = 0;
        if ( self->iscube ) {
            self->nplanes = sph_fits_get_nplanes(
            		cpl_frame_get_filename(self->current_frame), 0 );
            readnextplane = 1;
        }
    }
    im = cpl_image_load(cpl_frame_get_filename(self->current_frame),
                        CPL_TYPE_DOUBLE,
                        self->currentplane,
                        self->extension);
    if (cpl_error_get_code()) {
        (void)cpl_error_set_where(cpl_func);
    } else if ( readnextplane ) {
        self->currentplane++;
    }
    return im;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Get the next calibrated science image as master frame
 * @param self
 *
 * @return the next calibrated science master frame or NULL if none left or error.
 *
 * This returns the next calibrated science image as master frame.
 * The function always reads from the image from extension set in self.
 * The routine automatically detects if the frame is a cube or not. It will go
 * through all images (planes) in all frames and apply the calibration files
 * that are set (dark, flat, ...).
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_differential_imaging_get_next_calibrated_raw_master(
        sph_differential_imaging* self)
{
    sph_master_frame*   mframe = NULL;
    cpl_image*      rawim;

    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(self->rawframes,CPL_ERROR_ILLEGAL_INPUT,NULL);

    rawim = sph_differential_imaging_get_next_raw(self);
    if ( rawim ) {
        mframe = sph_common_science_calibrate_raw(rawim, self->dark,
                                                  self->flat);
        cpl_image_delete(rawim);
        if (cpl_error_get_code()) (void)cpl_error_set_where(cpl_func);
    }
    return mframe;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the next calibrated science image
 * @param self
 * @param bpixmap reference to output badpixels (maybe NULL)
 * @param wmap reference to output weights (maybe NULL)
 *
 * @return the next calibrated science frame or NULL if none left or error.
 *
 * This returns the next calibrated science image.
 * The function always reads from the image from extension set in self.
 * The routine automatically detects if the frame is a cube or not. It will go
 * through all images (planes) in all frames and apply the calibration files
 * that are set (dark, flat, ...).
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_differential_imaging_get_next_calibrated_raw(
        sph_differential_imaging* self,
        cpl_image** bpixmap, cpl_image** wmap)
{
    cpl_image*        rawim = NULL;
    sph_master_frame* mframe;

    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(self->rawframes,CPL_ERROR_ILLEGAL_INPUT,NULL);

    mframe = sph_differential_imaging_get_next_calibrated_raw_master(self);
    if ( mframe ) {
        if ( bpixmap ) {
            *bpixmap = sph_master_frame_get_badpixelmap(mframe);
        }
        if ( wmap ) {
            *wmap = sph_master_frame_get_weights(mframe);
        }
        rawim = sph_master_frame_extract_image(mframe,1);
        sph_master_frame_delete(mframe);
    }
    return rawim;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the left image
 * @param self
 * @param leftimage     the left image
 * @param lefbadpix     the badpixel image (maybe NULL)
 * @return error code
 *
 * This sets the left image to use for differential imaging. The image
 * input is assumed to be calibrated (but not transformed regarding rotation or
 * dithering).
 * The left image is always the image that is subtracted from in
 * the get_difference_image function.
 * An optional badpixel image belonging to the image may be provided.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_differential_imaging_set_left_image( sph_differential_imaging* self,
        cpl_image* leftframe, cpl_image* leftbadpix) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    if ( leftframe ) {
        self->leftimage = leftframe;
    }
    if ( leftbadpix ) {
        self->leftbpix = leftbadpix;
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the right image
 * @param self
 * @param rightimage     the right image
 * @param rightbadpix     the badpixel image (maybe NULL)
 * @return error code
 *
 * This sets the right image to use for differential imaging. The image
 * input is assumed to be calibrated (but not transformed regarding rotation or
 * dithering).
 * The right image is always the image that is subtracted in
 * the get_difference_image function.
 * An optional badpixel image belonging to the image may be provided.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_differential_imaging_set_right_image( sph_differential_imaging* self,
        cpl_image* rightframe, cpl_image* rightbadpix) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    if ( rightframe ) {
        self->rightimage = rightframe;
    }
    if ( rightbadpix ) {
        self->leftbpix = rightbadpix;
    }
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Delete object
 * @param self
 *
 *
 */
/*----------------------------------------------------------------------------*/
void
sph_differential_imaging_delete( sph_differential_imaging* self ) {
    if ( self ) {
        if ( self->fft ) {
            sph_fft_delete(self->fft); self->fft = NULL;
        }
        cpl_free(self);
    }
}


/**@}*/
