/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifndef SPH_DISTORTION_MODEL_H_
#define SPH_DISTORTION_MODEL_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   New types
 -----------------------------------------------------------------------------*/

#include <cpl.h>

typedef struct _sph_distortion_model_ {
    cpl_polynomial* polyx;
    cpl_polynomial* polyy;
    cpl_vector*     workv;
    double          offx;
    double          offy;
    double          workxy[2];
} sph_distortion_model;

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_error.h"
#include "sph_distortion_map.h"
#include "sph_point_pattern.h"
#include "sph_polygon.h"
#include "sph_master_frame.h"

/*-----------------------------------------------------------------------------
 Declaration of Functions
 -----------------------------------------------------------------------------*/

cpl_error_code
sph_distortion_model_set_coeff_x(
        sph_distortion_model*,
        cpl_size,
        cpl_size,
        double);

cpl_error_code
sph_distortion_model_set_coeff_y(
        sph_distortion_model*,
        cpl_size,
        cpl_size,
        double);

cpl_error_code
sph_distortion_model_add_coeff_x(
        sph_distortion_model*,
        cpl_size,
        cpl_size,
        double);

cpl_error_code
sph_distortion_model_add_coeff_y(
        sph_distortion_model*,
        cpl_size,
        cpl_size,
        double);

sph_point_pattern*
sph_distortion_model_apply_pp(const sph_distortion_model*,
                                         const sph_point_pattern*)
    CPL_ATTR_ALLOC;

sph_distortion_model*
sph_distortion_model_new( const cpl_polynomial* polyx,
                                     const cpl_polynomial* polyy )
    CPL_ATTR_ALLOC;

sph_distortion_model*
sph_distortion_model_load(const char*,
                          int,
                          const char*,
                          const char*)
    CPL_ATTR_ALLOC;

sph_distortion_model*
sph_distortion_model_load_opt(const char*,
                              int,
                              const char*,
                              const char*,
                              const char*,
                              const char*,
                              double *,
                              double *,
                              int *)
    CPL_ATTR_ALLOC;

sph_distortion_model*
sph_distortion_model_new_from_proplist(const cpl_propertylist* proplist,
                                                       const char* c_x_tag,
                                                       const char* c_y_tag)
    CPL_ATTR_ALLOC;

sph_distortion_model*
sph_distortion_model_from_grid_image(const sph_master_frame*,
                                                const sph_point_pattern*,
                                                int,
                                                double,
                                                double,
                                                cpl_propertylist*,
                                                double,
                                                double,
                                                double *,
                                                double *,
                                                double *,
                                                double *,
                                                sph_point_pattern**,
                                                cpl_table **);

sph_distortion_model*
sph_distortion_model_from_grid_image_general(const sph_master_frame*,
                                                        const sph_point_pattern*,
                                                        int,
                                                        double,
                                                        double,
                                                        cpl_propertylist*,
                                                        double,
                                                        double,
                                                        double *,
                                                        double *,
                                                        double *,
                                                        double *,
                                                        sph_point_pattern**,
                                                        cpl_table **,
                                                        int);

sph_error_code
sph_polynomial_distortion_set_offset(sph_distortion_model* self,
                                     double offx,
                                     double offy);

sph_error_code sph_distortion_model_save(
        const sph_distortion_model* self,
        const char* filename,
        int nx,
        int ny, unsigned mode,
        const cpl_propertylist* pl,
        const char* postfix,
		const char* c_x_tag,
		const char* c_y_tag);

cpl_polynomial*
sph_distortion_model_get_poly_x( sph_distortion_model* self );

cpl_polynomial*
sph_distortion_model_get_poly_y( sph_distortion_model* self );

cpl_propertylist*
sph_distortion_model_get_proplist_override_tags(
		const sph_distortion_model* self,
		const char * coeff_x_tag,
		const char * coeff_y_tag) CPL_ATTR_ALLOC;

cpl_propertylist*
sph_distortion_model_get_proplist( const sph_distortion_model*)
    CPL_ATTR_ALLOC;

sph_point_pattern*
sph_distortion_model_create_point_pattern(const sph_distortion_model* self,
                                          int nx, int ny, int np)
    CPL_ATTR_ALLOC CPL_ATTR_NONNULL;

cpl_error_code sph_distortion_model_fill_image_xy
(const sph_distortion_model*,
 cpl_image*,
 cpl_image*);

sph_distortion_map*
sph_distortion_model_fit_distortion_map(
        sph_distortion_model* self,
        const sph_distortion_map* dmap,
        int maxdegree,
        double* pchiX,
        double* pchiY);

void
sph_distortion_model_delete(sph_distortion_model*);
sph_error_code
sph_distortion_model_save_dfs(
        sph_distortion_model* self,
        const char* outfilename,
        cpl_frameset* allframes,
        cpl_frame* template_frame,
        cpl_parameterlist* params,
        const char* tag,
        const char* recipe,
        const char* pipename,
        cpl_propertylist* qclist,
        int nx,
        int ny,
        double pixscale,
        int changescale,
        const char* postfix,
		const char* c_x_tag,
		const char* c_y_tag);

void sph_distortion_model_get_dist(const sph_distortion_model*,
                                   const double[],
                                   double[])
    CPL_ATTR_NONNULL;

void sph_distortion_model_get_qc_dist(const sph_distortion_model*,
                                      const double[],
                                      double[])
    CPL_ATTR_NONNULL;

cpl_image*
sph_distortion_model_apply(
        const sph_distortion_model* self,
        const cpl_image* image,
        cpl_kernel kerneltype,
        double radin,
        const char * qckeyr,
        const char * qckeyl,
        cpl_propertylist* qclist)
    CPL_ATTR_ALLOC;

cpl_error_code
sph_distortion_model_apply_1(cpl_image*,
                             const sph_distortion_model*,
                             const cpl_image*,
                             cpl_kernel,
                             double,
                             const char *,
                             const char *,
                             cpl_propertylist*);
sph_error_code
sph_distortion_model_apply_poly(
        const sph_distortion_model* self,
        sph_polygon* poly );


sph_error_code
sph_distortion_model_scale_units(
        sph_distortion_model* self,
        double scale);

#endif /* SPH_DISTORTION_MODEL_H_ */
