/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <strings.h>
#include "sph_double_image.h"
#include "sph_master_frame.h"
#include "sph_zpl_exposure_imaging.h"
#include "sph_smart_imagelist.h"
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_utils.h"
#include "sph_version.h"
#include <cpl.h>
#include <string.h>
#include <math.h>

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/
sph_error_code SPH_DOUBLE_IMAGE_GENERAL         = SPH_DOUBLE_IMAGE_ERR_START + 0;
sph_error_code SPH_DOUBLE_IMAGE_BAD_ALGORITHM   = SPH_DOUBLE_IMAGE_ERR_START + 1;
sph_error_code SPH_DOUBLE_IMAGE_NO_ALGORITHM    = SPH_DOUBLE_IMAGE_ERR_START + 2;
sph_error_code SPH_DOUBLE_IMAGE_BAD_TYPE        = SPH_DOUBLE_IMAGE_ERR_START + 3;
sph_error_code SPH_DOUBLE_IMAGE_NO_TYPE         = SPH_DOUBLE_IMAGE_ERR_START + 4;
sph_error_code SPH_DOUBLE_IMAGE_BAD_IMAGE        = SPH_DOUBLE_IMAGE_ERR_START + 5;

const char* SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_LEFT = "EXTNAME ADDITION LEFT";
const char* SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT = "EXTNAME ADDITION RIGHT";
static
sph_error_code
sph_double_image_update_string_prefix(
        cpl_propertylist* proplist,
        const char* keyname,
        const char* prefix,
        const char* value );
static sph_error_code sph_double_image_interpolate_y_adjust_boundarylines_zpl_imaging(
        sph_double_image* self,
        cpl_boolean add_single_line,
        cpl_boolean set_badpxl );
static sph_error_code sph_double_image_interpolate_y_adjust_boundarylines_zpl_polarimetry(
        sph_double_image* self,
        cpl_boolean set_badpxl );

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_double_image  A structure to store double image products
 *
 * This structure represents a data product with two values for each pixel.
 * It is mainly used to store polarisation data.
 * Apart from the normal value field, it contains, just like sph_master_frame,
 * the following extensions:
 * @par Descirption:
 * <ol>
 * <li> RMSMAP. A map of the same dimension and number of pixels as the
 *      image giving the inverse of the RMS of the image pixel</li>
 * <li> NCombmap. A map of the same dimension and number of pixels as the
 *      image giving the number of pixels that were used to obtain the image</li>
 * <li> Badpixelmap. A map of the same dimension and number of pixels as the
 *      image flagging pixels that were impossible to reduce from the raw frames
 *      either because all input raw pixels at that position were bad or because
 *      the statistics is not calculable.</li>
 * <li> Quality Control Parameter list. A list of quality control parameters.
 *      What quality parameters are listed depends on the recipe used to create
 *      the frame.
 * </ol>
 * Each of these extensions appear twice now, one for each of the two images.
 *
 *
 * @par Synopsis:
 * @code
 * typedef struct _sph_double_image_
 * {
 *    sph_master_frame*        iframe;
 *    sph_master_frame*        pframe;
 * } sph_double_image;
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_double_image.
  @return   a pointer to the newly created sph_double_image or NULL if
              unsuccessful.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_double_image
 */
/*----------------------------------------------------------------------------*/
sph_double_image* sph_double_image_new_empty(void) {
    sph_double_image* spmi = cpl_calloc( 1, sizeof(sph_double_image) );

    return spmi;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_double_image.
  @return   a pointer to the newly created sph_double_image or NULL if
              unsuccessful.
  @param    mf_odd        a pointer to the first(odd) master frame
  @param    mf_even        a pointer to the second(even) master frame

  @note Input master frames are not duplicated, so don't delete them
  @note A CPL error is raised in case there was a problem.

  Construct a new sph_double_image
 */
/*----------------------------------------------------------------------------*/
sph_double_image* sph_double_image_new_from_master_frames( sph_master_frame* mf_odd, sph_master_frame* mf_even)
{
    sph_double_image* spmi = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    if ( mf_odd == NULL || mf_even == NULL ){
        SPH_ERR("Could not create double image: input master frame(s) has(have) null pointer(s).")
        return NULL;
    }
    spmi = cpl_calloc( 1, sizeof(sph_double_image) );

    spmi->properties = NULL;
    spmi->qclist = NULL;
    spmi->iframe =  mf_odd;
    spmi->pframe =  mf_even;


    //properties
    if ( mf_odd->properties && mf_even->properties ) {
        spmi->properties = cpl_propertylist_new();

        cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_IFRAME_RAW_MEDIANS_AVG,
                cpl_propertylist_get_double( mf_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_AVG) );
        cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_IFRAME_RAW_MEDIANS_MED,
                cpl_propertylist_get_double( mf_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MED) );
        cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_IFRAME_RAW_MEDIANS_MIN,
                cpl_propertylist_get_double( mf_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MIN) );
        cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_IFRAME_RAW_MEDIANS_MAX,
                cpl_propertylist_get_double( mf_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MAX) );
        if ( cpl_propertylist_has(mf_odd->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_IFRAME_RAW_MEDIANS_RANGE ))
             cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_IFRAME_RAW_MEDIANS_RANGE,
                cpl_propertylist_get_double( mf_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE) );

        //even (PFRAME)
        cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_PFRAME_RAW_MEDIANS_AVG,
                cpl_propertylist_get_double( mf_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_AVG) );
        cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_PFRAME_RAW_MEDIANS_MED,
                cpl_propertylist_get_double( mf_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MED) );
        cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_PFRAME_RAW_MEDIANS_MIN,
                cpl_propertylist_get_double( mf_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MIN) );
        cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_PFRAME_RAW_MEDIANS_MAX,
                cpl_propertylist_get_double( mf_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MAX) );
        if ( cpl_propertylist_has(mf_odd->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_PFRAME_RAW_MEDIANS_RANGE ))
             cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_DOUBLE_IMAGE_PFRAME_RAW_MEDIANS_RANGE,
                cpl_propertylist_get_double( mf_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE) );

        if ( cpl_error_get_code() != CPL_ERROR_NONE  ){
            SPH_WARNING("Some keywords from raw_medians group were not properly updated. It is probably ok. "
                    "Delete double image propertylist, and reset error system.");
            cpl_propertylist_delete( spmi->properties ); spmi->properties = NULL;
            cpl_error_reset();
        }


    }
    if ( spmi->pframe == NULL || spmi->iframe == NULL )
    {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not allocate double image" );
        cpl_free(spmi);
        return NULL;
    }
    return spmi;
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Copy of the odd sph_master_frame from the given sph_double_image.

  @param self   the sph_doule_image frame which odd sph_master_frame is to
                retrieve from

  @return a pointer to the newly copied odd sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new copied  odd sph_master_frame as copy of the odd
  sph_master_frame  from the sph_double_image.

 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_double_image_get_iframe_master_frame_duplicate( const sph_double_image* self )
{
    return self && self->iframe
        ? sph_master_frame_duplicate(self->iframe) : NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get iframe(odd) sph_master_frame from the given sph_double_image.

  @param self   the sph_doule_image frame which iframe(odd) sph_master_frame is
                to retrieve from
  @return a pointer to the iframe (odd) sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return an iframe(odd) sph_master_frame from the sph_double_image without
  duplicate.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_double_image_get_iframe_master_frame(const sph_double_image* self ) {
    return self ? self->iframe : NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Copy of the even sph_master_frame from the given sph_double_image.

  @param self   the sph_doule_image frame which even sph_master_frame is to
                retrieve from

  @return a pointer to the newly copied even sph_master_frame or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new copied  even sph_master_frame as copy of the even
  sph_master_frame from the sph_double_image.
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_double_image_get_pframe_master_frame_duplicate(const sph_double_image* self ) {
    return self && self->pframe
        ? sph_master_frame_duplicate( self->pframe ) : NULL;
}

sph_master_frame* sph_double_image_get_pframe_master_frame(const sph_double_image* self ) {
    return self ? self->pframe : NULL;
}

sph_master_frame*
sph_double_image_calculate_pframe_slash_iframe_master_frame( const sph_double_image* self){
    sph_master_frame*    pframe_slash = NULL;
    if ( self && self->pframe && self->iframe ){
        pframe_slash = sph_double_image_get_pframe_master_frame_duplicate( self );
        sph_master_frame_divide_master_frame( pframe_slash, self->iframe );
        if ( pframe_slash->properties ){
            cpl_propertylist_delete( pframe_slash->properties );
            pframe_slash->properties = NULL;
        }
    }
    return pframe_slash;
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_double_image.
  @param    nx        the pixel size in x
  @param    ny        the pixel size in y
  @return   a pointer to the newly created sph_double_image or NULL if
              unsuccessful.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_double_image
 */
/*----------------------------------------------------------------------------*/
sph_double_image*
sph_double_image_new( int nx, int ny ) {
    sph_double_image* spmi;

    cpl_ensure(nx > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(ny > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    spmi = (sph_double_image*)cpl_calloc( 1, sizeof(*spmi) );

    spmi->qclist = NULL;
    spmi->properties = NULL;
    spmi->iframe = sph_master_frame_new(nx, ny);
    spmi->pframe = sph_master_frame_new(nx, ny);

    return spmi;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_double_image from zpl exposure imaging
  @return   a pointer to the newly created sph_double_image or NULL if
              unsuccessful.
  @param    zplexpimg     a pointer to the sph_zpl_exposure_imaging

  @note Input master frames are not duplicated, so don't delete them
  @note A CPL error is raised in case there was a problem.

  Construct a new sph_double_image
 */
/*----------------------------------------------------------------------------*/
sph_double_image*
sph_double_image_new_from_zpl_exposure_imaging( sph_zpl_exposure_imaging* zplexpimg)
{
    sph_double_image*        doubleimage        = NULL;

    cpl_ensure(zplexpimg, CPL_ERROR_NULL_INPUT, NULL);

    doubleimage = sph_double_image_new_empty();

    //copy properties
    doubleimage->properties = cpl_propertylist_duplicate( zplexpimg->properties );

    //iframe (in the case of imaging iframe contains a real imaging signal)
    doubleimage->iframe = sph_master_frame_new_empty();
    doubleimage->iframe->image = cpl_image_duplicate( zplexpimg->image_odd );
    if ( doubleimage->iframe->image ) {
        doubleimage->iframe->badpixelmap = cpl_image_new( cpl_image_get_size_x(doubleimage->iframe->image),
                cpl_image_get_size_y( doubleimage->iframe->image), CPL_TYPE_INT);
        doubleimage->iframe->ncombmap = cpl_image_new( cpl_image_get_size_x(doubleimage->iframe->image),
                cpl_image_get_size_y( doubleimage->iframe->image), CPL_TYPE_DOUBLE);
        doubleimage->iframe->rmsmap = cpl_image_new( cpl_image_get_size_x(doubleimage->iframe->image),
                cpl_image_get_size_y( doubleimage->iframe->image), CPL_TYPE_DOUBLE);
    }

    //pframe (in the case of the imaging pframe contains a dark signal (covered lines)
    doubleimage->pframe = sph_master_frame_new_empty();
    doubleimage->pframe->image = cpl_image_duplicate( zplexpimg->image_even );
    if ( doubleimage->pframe->image ) {
        doubleimage->pframe->badpixelmap = cpl_image_new( cpl_image_get_size_x(doubleimage->pframe->image),
                cpl_image_get_size_y( doubleimage->pframe->image), CPL_TYPE_INT);
        doubleimage->pframe->ncombmap = cpl_image_new( cpl_image_get_size_x(doubleimage->pframe->image),
                cpl_image_get_size_y( doubleimage->pframe->image), CPL_TYPE_DOUBLE);
        doubleimage->pframe->rmsmap = cpl_image_new( cpl_image_get_size_x(doubleimage->pframe->image),
                cpl_image_get_size_y( doubleimage->pframe->image), CPL_TYPE_DOUBLE);
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    return doubleimage;

}



/*----------------------------------------------------------------------------*/
/**
 * @brief        add a double image from another double image
 *
 * @param        self        the double image to add to
 * @param        diframe        the double image to add
 *
 * @return        error code of the operation
 *
 * This function adds double image "diframe" to self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/

sph_error_code
sph_double_image_add_double_image( sph_double_image* self,
                                   const sph_double_image* diframe){
    const sph_error_code rerr =
        sph_master_frame_add_master_frame( self->iframe, diframe->iframe ) ||
        sph_master_frame_add_master_frame( self->pframe, diframe->pframe );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        subtract a double image from another double image
 *
 * @param        self        the double image to subtract from
 * @param        diframe        the double image to subtract
 *
 * @return        error code of the operation
 *
 * This function subtracts double image "diframe" from  self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_double_image_subtract_double_image( sph_double_image* self,
                                        const sph_double_image* diframe){
    const sph_error_code rerr =
        sph_master_frame_subtract_master_frame( self->iframe, diframe->iframe ) ||
        sph_master_frame_subtract_master_frame( self->pframe, diframe->pframe );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        multiply double image by scalar double value
 *
 * @param        self       the double image to multiply
 * @param        value      the double value
 *
 * @return        error code of the operation
 *
 * This function multiplies the double image from self by the double "value".
 * Self is changed in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_double_image_multiply_double( sph_double_image* self, double value){
    const sph_error_code rerr =
        sph_master_frame_multiply_double( self->iframe, value ) ||
        sph_master_frame_multiply_double( self->pframe, value );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        divide a double image by another double image
 *
 * @param        self        the double image to divide
 * @param        dimage         the double image to divide by
 *
 * @return        error code of the operation
 *
 * This function divides the double image self by the "dimage". Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_double_image_divide_double_image( sph_double_image* self,
                                      const sph_double_image* dimage){
    const sph_error_code rerr =
        sph_master_frame_divide_master_frame( self->iframe, dimage->iframe ) ||
        sph_master_frame_divide_master_frame( self->pframe, dimage->pframe );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        divide double image by scalar double value
 *
 * @param        self        the double image to divide by   (dividend)
 * @param        value        the double value (divisor)
 *
 * @return        error code of the operation
 *
 * This function divides double image from self by double "value".
 * Self is changed in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_double_image_divide_double( sph_double_image* self, double value){
    const sph_error_code rerr =
        sph_master_frame_divide_double( self->iframe, value ) ||
        sph_master_frame_divide_double( self->pframe, value );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        combine a double image (+Q or +U)) with another
 *                 double image (-Q or -U  )
 *
 * @param        self                the "plus" double image to combine
 * @param        diframe_minus        the "minus" image to combine
 *
 * @return        error code of the operation
 *
 * This function combines "polarimetrically" double image "diframe_minus" with self -- plus double
 * image. Self is changed in place. The combination of the master double image from +Q measurement
 * with master double image from -Q measurement follows the following formulas:
 * I = [I(+Q) + I(-Q)]/2.  (the same for +U and -U)
 * P = [P(+Q) - P(-Q)]/2.  (the same for +U and -U)
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_double_image_combine_double_image (sph_double_image* self,
                                       const sph_double_image* diframe_minus){
    const sph_error_code rerr =
        sph_master_frame_add_master_frame( self->iframe,
                                           diframe_minus->iframe ) ||
        sph_master_frame_multiply_double( self->iframe, 0.5 ) ||
        sph_master_frame_subtract_master_frame( self->pframe,
                                                diframe_minus->pframe ) ||
        sph_master_frame_multiply_double( self->pframe, 0.5 );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}


const sph_double_image_qc_names default_iframe_names = {
	SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME,
	SPH_COMMON_KEYWORD_QC_MEDIAN_DOUBLEIMAGE_IFRAME,
	SPH_COMMON_KEYWORD_QC_RMS_DOUBLEIMAGE_IFRAME,
	SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD
};

const sph_double_image_qc_names default_pframe_names = {
	SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME,
	SPH_COMMON_KEYWORD_QC_MEDIAN_DOUBLEIMAGE_PFRAME,
	SPH_COMMON_KEYWORD_QC_RMS_DOUBLEIMAGE_PFRAME,
	SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_EVEN
};

cpl_error_code sph_double_image_quality_check(sph_double_image* self){

	return sph_double_image_quality_check_names(self, NULL, NULL);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Perform a quality control on the double image

  @param    self        	the double image
  @param    iframe_names    qc names for IFRAME, if NULL use defaults
  @param    pframe_names    qc names for PFRAME, if NULL use defaults
  @return   the error code
  @note     A CPL error is raised in case there was a problem.

  Performs a quality control calculation on the double image. This means, that
  certain statistics are calculated and added to the qclist first to the each
  master frame of the quad image, and then put it into the main qclist of the

  Currently these statistics are: Mean, median and rms of the image.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_double_image_quality_check_names(sph_double_image* self ,
											  const sph_double_image_qc_names * iframe_names,
											  const sph_double_image_qc_names * pframe_names){

    int            rerr            = CPL_ERROR_NONE;

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    //IFRAME (ODD)
    rerr = sph_master_frame_quality_check( self->iframe );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by calculating quality parameters for the "
                "di->iframe master frame.")
        return     rerr;
    }
    if ( self->qclist == NULL ) {
        self->qclist = cpl_propertylist_new();
    }

    if(!iframe_names) iframe_names = &default_iframe_names;
    if(!pframe_names) pframe_names = &default_pframe_names;

    //Mean
    if ( cpl_propertylist_has( self->qclist, iframe_names->mean_name) ){
        rerr = cpl_propertylist_update_double( self->qclist, iframe_names->mean_name,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, iframe_names->mean_name,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEAN of DOUBLE IMAGE IFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //Median
    if ( cpl_propertylist_has( self->qclist, iframe_names->median_name) ){
        rerr = cpl_propertylist_update_double( self->qclist, iframe_names->median_name,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, iframe_names->median_name,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEDIAN of DOUBLE IMAGE IFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //RMS
    if ( cpl_propertylist_has( self->qclist, iframe_names->rms_name) ){
        rerr = cpl_propertylist_update_double( self->qclist, iframe_names->rms_name,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, iframe_names->rms_name,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC RMS of DOUBLE IMAGE IFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }

    //PFRAME (EVEN)
    rerr = sph_master_frame_quality_check( self->pframe );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by calculating quality parameters for the "
                "di->pframe master frame.")
        return     rerr;
    }
    //Mean
    if ( cpl_propertylist_has( self->qclist, pframe_names->mean_name) ){
        rerr = cpl_propertylist_update_double( self->qclist, pframe_names->mean_name,
                cpl_propertylist_get_double( self->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, pframe_names->mean_name,
                cpl_propertylist_get_double( self->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEAN of DOUBLE IMAGE PFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //Median
    if ( cpl_propertylist_has( self->qclist, pframe_names->median_name) ){
        rerr = cpl_propertylist_update_double( self->qclist, pframe_names->median_name,
                cpl_propertylist_get_double( self->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, pframe_names->median_name,
                cpl_propertylist_get_double( self->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEDIAN of DOUBLE IMAGE PFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //RMS
    if ( cpl_propertylist_has( self->qclist, pframe_names->rms_name) ){
        rerr = cpl_propertylist_update_double( self->qclist, pframe_names->rms_name,
                cpl_propertylist_get_double( self->pframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, pframe_names->rms_name,
                cpl_propertylist_get_double( self->pframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC RMS of DOUBLE IMAGE PFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }

    //add number of the badpixels QC keywords to the qclist of the double image
    rerr = cpl_propertylist_update_long(self->qclist, iframe_names->bad_pixels_name,
             (long) cpl_image_get_flux(self->iframe->badpixelmap));
    rerr = cpl_propertylist_update_long(self->qclist,  pframe_names->bad_pixels_name,
             (long) cpl_image_get_flux(self->pframe->badpixelmap));

    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC keywords of DOUBLE IMAGE NUMBER BADPIXELS")
        SPH_RAISE_CPL;
        return    rerr;
    }

    return rerr;

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Perform a quality control on the double image

  @param    self         the double image
  @param    keysuffix    string to add to the standard quality keyword names
  @return   the error code
  @note     A CPL error is raised in case there was a problem.

  Performs a quality control calculation on the double image. This means, that
  certain statistics are calculated and added to the qclist first to the each
  master frame of the quad image, and then put it into the main qclist of the

  Currently these statistics are: Mean, median and rms of the image.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_double_image_quality_check_keysuffix( sph_double_image* self, const char* keysuffix ){
    int            rerr            = CPL_ERROR_NONE;
    char           qkeyname[1024];

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    //IFRAME (ODD)
    rerr = sph_master_frame_quality_check( self->iframe );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by calculating quality parameters for the "
                "di->iframe master frame.")
        return     rerr;
    }
    if ( self->qclist == NULL ) {
        self->qclist = cpl_propertylist_new();
    }

    //Mean
    strcpy(qkeyname,"");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME, keysuffix );
    rerr = cpl_propertylist_update_double( self->qclist, qkeyname,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEAN of DOUBLE IMAGE IFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }

    //Median
    strcpy(qkeyname,"");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEDIAN_DOUBLEIMAGE_IFRAME, keysuffix );
    rerr = cpl_propertylist_update_double( self->qclist, qkeyname,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEDIAN of DOUBLE IMAGE IFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }

    //RMS
    strcpy(qkeyname,"");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_RMS_DOUBLEIMAGE_IFRAME, keysuffix );
    rerr = cpl_propertylist_update_double( self->qclist, qkeyname,
                cpl_propertylist_get_double( self->iframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC RMS of DOUBLE IMAGE IFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }

    //PFRAME (EVEN)
    rerr = sph_master_frame_quality_check( self->pframe );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by calculating quality parameters for the "
                "di->pframe master frame.")
        return     rerr;
    }
    //Mean
    strcpy(qkeyname,"");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME, keysuffix );
    rerr = cpl_propertylist_update_double( self->qclist, qkeyname,
                cpl_propertylist_get_double( self->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEAN of DOUBLE IMAGE PFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //Median
    strcpy(qkeyname,"");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_MEDIAN_DOUBLEIMAGE_PFRAME, keysuffix );
    rerr = cpl_propertylist_update_double( self->qclist, qkeyname,
                cpl_propertylist_get_double( self->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEDIAN of DOUBLE IMAGE PFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }
    //RMS
    strcpy(qkeyname,"");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_RMS_DOUBLEIMAGE_PFRAME, keysuffix );
    rerr = cpl_propertylist_update_double( self->qclist, qkeyname,
                cpl_propertylist_get_double( self->pframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC RMS of DOUBLE IMAGE PFRAME.")
        SPH_RAISE_CPL;
        return     rerr;
    }

    //add number of the badpixels QC keywords to the qclist of the double image
    strcpy(qkeyname,"");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_ODD, keysuffix );
    rerr = cpl_propertylist_update_long(self->qclist,  qkeyname,
             (long) cpl_image_get_flux(self->iframe->badpixelmap));

    strcpy(qkeyname,"");
    sprintf( qkeyname, "%s %s", SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_DOUBLE_IMAGE_EVEN, keysuffix );
    rerr = cpl_propertylist_update_long(self->qclist,  qkeyname,
             (long) cpl_image_get_flux(self->pframe->badpixelmap));

    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC keywords of DOUBLE IMAGE NUMBER BADPIXELS")
        SPH_RAISE_CPL;
        return    rerr;
    }

    return rerr;

}



/*----------------------------------------------------------------------------*/
/**
  @brief    Create the bad pixel maps for the quad image
  @param    self        the quad image
  @param    sigma       the sigma level for masking
  @return   the error code

  @note     A CPL error is raised in case there was a problem.

  - Create (calculate) the bad pixel maps for the given double image
  This function is based on the associated function of the master frame which
  current algorithm is an algorithm that sets the bad pixels
  to be all those that are further than sigma * the total RMS of the whole
  master frame away from the median of the master frame.
  - QC keyword value are used from the qclist of each master frame not from the
  quad image qclist
 */
/*----------------------------------------------------------------------------*/

cpl_error_code       sph_double_image_mask_sigma( sph_double_image* self, double sigma)
{
    int            rerr         = CPL_ERROR_NONE;

    rerr = sph_master_frame_mask_sigma( self->iframe, sigma);
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by creating mask sigma for the double image -> iframe.");
        SPH_RAISE_CPL;
        return cpl_error_get_code();
    }

    rerr = sph_master_frame_mask_sigma( self->pframe, sigma);
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by creating mask sigma for the double image -> pframe.");
        SPH_RAISE_CPL;
        return cpl_error_get_code();
    }

    return rerr;
}


/*----------------------------------------------------------------------------*/
/**
   @brief Copy constructor function for sph_double_image

   @param    self   the double image to duplicate

   @return   a pointer to the newly created sph_master_frame or NULL if
            unsuccessfully.
   @note     A CPL error is raised in case there was a problem.

  Construct a new sph_double_image as copy of the input master frame.
 */
/*----------------------------------------------------------------------------*/
sph_double_image* sph_double_image_duplicate( const sph_double_image* self) {
    sph_double_image*  new_doubleimage = NULL;

    if (self == NULL) {
        sph_error_raise( SPH_DOUBLE_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "An initial double image to copy has a null pointer."
                                         "Nothing to copy." );
        return NULL;
    }

    //create an empty double image
    new_doubleimage = sph_double_image_new_empty();
    if (new_doubleimage == NULL) {
           sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_ERROR, "Could not allocate memory for the double image" );
                return NULL;
    }

    //duplicate properties and qclist if exists
    if ( self->properties != NULL ){
        new_doubleimage->properties = cpl_propertylist_duplicate( self->properties );
    }
    if ( self->qclist != NULL ){
        new_doubleimage->qclist = cpl_propertylist_duplicate( self->qclist );
    }

    //duplicate iframe and pframe as a single master frame
    if (self->iframe !=NULL){
        new_doubleimage->iframe = sph_master_frame_duplicate(self->iframe);
    } else {
        sph_error_raise( SPH_DOUBLE_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "iframe from the initial double image to copy has a null pointer." );
    }

    if (self->pframe != NULL){
        new_doubleimage->pframe = sph_master_frame_duplicate(self->pframe);
    } else {
        sph_error_raise( SPH_DOUBLE_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "pframe from the initial double image to copy has a null pointer." );
    }

    return new_doubleimage;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new double image frameset from cpl_frameset of zpl exp img TYPE
 *
 * @param inframe  the input cpl_frameset (of ZPL_EXP_IMG)
 *
 * @return pointer to the newly created frameset with double_image or NULL in
 * case of error.
 *
 * This function creates new sph_double_image frameset from a given cpl_frameset
 * given as a cpl_frameset of type ZPL_EXP_IMG TYPE.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frameset* sph_double_image_create_double_image_frameset_from_zplexp_imaging_frameset( const cpl_frameset* inframes )
{
    cpl_frameset*            result            = NULL;
    const cpl_frame*         curframe;
    cpl_frame*               doubleframe        = NULL;
    //cpl_frame*                tmpframe    = NULL;

    cpl_propertylist*        pl                = NULL;
    //cpl_propertylist*     pl_i    = NULL;
    //cpl_propertylist*     pl_p    = NULL;

    //sph_master_frame*     mframe            = NULL;
    sph_keyword_manager*    keym            = NULL;

    const char*            czFrame            = NULL;
    //char                    czFrameNew[256];
    const char*            czType            = NULL;
    short                   ok                = 0;


    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);

    result = cpl_frameset_new();
    if ( !result ) {
        return NULL;
    }
    keym = sph_keyword_manager_new();
    if ( !keym ) {
        return NULL;
    }
    curframe = cpl_frameset_get_first_const( inframes );

    while ( curframe ) {
        czFrame = cpl_frame_get_filename( curframe );
        //strncpy( czFrameNew, czFrame, 256 );
        //strcat( czFrameNew, ".tmp");
        doubleframe = sph_filemanager_create_temp_frame( czFrame, "NONE", CPL_FRAME_GROUP_CALIB );
        //sph_keyword_manager_load_keywords( czFrame ); // this will read the keywords from ext 0 of the file
        //czType = sph_keyword_manager_get_string( SPH_COMMON_KEYWORD_SPH_TYPE ); // get the value of the  SPH_COMMON_KEYWORD_SPH_TYPE - soft. repres.
        pl = sph_keyword_manager_load_properties( czFrame, 0 );
        czType = cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_SPH_TYPE);
        ok = 0;

        if ( !strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP_IMG ) ) {
            sph_double_image* doubleimage = sph_double_image_new_empty();

            if ( doubleimage ){

                // double image iframe
                doubleimage->iframe = sph_master_frame_new_empty();
                if ( doubleimage->iframe ){
                      doubleimage->iframe->image = cpl_image_load(czFrame, CPL_TYPE_DOUBLE, 0, 0 );

                    if ( doubleimage->iframe->image ){
                        doubleimage->iframe->badpixelmap = cpl_image_new( cpl_image_get_size_x(doubleimage->iframe->image),
                                         cpl_image_get_size_y( doubleimage->iframe->image ), CPL_TYPE_INT );
                        doubleimage->iframe->ncombmap = cpl_image_new( cpl_image_get_size_x(doubleimage->iframe->image),
                                          cpl_image_get_size_y( doubleimage->iframe->image ), CPL_TYPE_INT );
                        doubleimage->iframe->rmsmap = cpl_image_new( cpl_image_get_size_x(doubleimage->iframe->image),
                                        cpl_image_get_size_y( doubleimage->iframe->image ), CPL_TYPE_DOUBLE );
                        //pl_zero_i = sph_keyword_manager_load_properties(czFrame, 0); //read property list from ext 0
                        ok = ok + 1;
                    }
                }

                // zero- image pframe
                doubleimage->pframe = sph_master_frame_new_empty();
                if ( doubleimage->pframe ){
                      doubleimage->pframe->image = cpl_image_load(czFrame, CPL_TYPE_DOUBLE, 0, 1 );

                    if ( doubleimage->pframe->image ){
                        doubleimage->pframe->badpixelmap = cpl_image_new( cpl_image_get_size_x( doubleimage->pframe->image ),
                                         cpl_image_get_size_y( doubleimage->pframe->image ), CPL_TYPE_INT );
                        doubleimage->pframe->ncombmap = cpl_image_new( cpl_image_get_size_x( doubleimage->pframe->image ),
                                          cpl_image_get_size_y( doubleimage->pframe->image ), CPL_TYPE_INT );
                        doubleimage->pframe->rmsmap = cpl_image_new( cpl_image_get_size_x( doubleimage->pframe->image ),
                                        cpl_image_get_size_y( doubleimage->pframe->image ), CPL_TYPE_DOUBLE );
                        //pl_zero_p = sph_keyword_manager_load_properties(czFrame, 0);
                        // read property list from ext 1, but there is no possibility to do in keyword manager
                        ok = ok + 1;
                    }
                }



            } // if doubleimage

            if ( ok == 2 ){
                //sph_double_image_save( doubleimage, czFrameNew, pl ,CPL_IO_CREATE);
                sph_double_image_save( doubleimage, cpl_frame_get_filename( doubleframe ), pl,CPL_IO_CREATE);
                sph_double_image_delete( doubleimage ); doubleimage = NULL;

                //doubleframe = cpl_frame_new();
                //cpl_frame_set_filename( doubleframe, czFrameNew );
                cpl_frame_set_tag( doubleframe, "None");
                cpl_frameset_insert( result, doubleframe );

                sph_error_raise( SPH_DOUBLE_IMAGE_GENERAL,
                    __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,  " DOUBLE IMAGE frame with filename = %s is created \n"
                            "(from the PREPROC ZPL EXP IMG frame with filename = %s)", cpl_frame_get_filename ( doubleframe ), czFrame );

            } else {
                sph_error_raise( SPH_DOUBLE_IMAGE_BAD_IMAGE,
                    __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING,  " Frame bad image(s)...ignore this frame (filename: %s)", czFrame );
            }

        } else {             //if czType
            sph_error_raise( SPH_DOUBLE_IMAGE_BAD_TYPE,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_WARNING,  " Frame bad Type  = %s...ignore this frame (filename: %s)", czType, czFrame );

        }

        cpl_propertylist_delete( pl );

        curframe = cpl_frameset_get_next_const( inframes );

    } // end while
    return result;

}



/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new double image frameset from the 'cube(s)' of zpl exp img
 *
 * @param inframe  the input cube(s) cpl_frameset  (of ZPL_EXP_IMG)
 *
 * @return pointer to the newly created frameset with double_image or NULL in
 * case of error.
 *
 * This function creates new sph_double_image frameset from a given cpl_frameset
 * given as a cpl_frameset of type ZPL_EXP_IMG TYPE.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_double_image_create_double_image_frameset_from_zplexp_imaging_cubes( const cpl_frameset* inframes ){
    const cpl_frame*              curframe;
    cpl_frame*                    expframe            = NULL;
    cpl_frameset*                 expframes           = NULL;
    cpl_frameset*                 result              = NULL;
    cpl_frameset*                 result_expframes    = NULL;
    sph_zpl_exposure_imaging*   zplexp                 = NULL;
    cpl_propertylist*            pl                  = NULL;
    cpl_propertylist*            pl_proper           = NULL;
    sph_keyword_manager*        keym                = NULL;
    int                         nz                  = 0;
    sph_error_code                rerr                = CPL_ERROR_NONE;


    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);

    result = cpl_frameset_new();
    if ( !result ) {
        SPH_ERR("Can't create frameset for the result.");
        return NULL;
     }

    keym = sph_keyword_manager_new();
    if ( !keym ) {
        SPH_ERR("Can't create keyword manager.");
        cpl_frameset_delete( result ); result = NULL;
        return NULL;
    }

    curframe = cpl_frameset_get_first_const( inframes );
    while ( curframe ) {

        expframes = cpl_frameset_new();
        if ( !expframes ) {
            SPH_ERR("Can't create frameset for the expframes.")
            cpl_frameset_delete(result); result = NULL;
            if (keym) {
                sph_keyword_manager_delete(); keym = NULL;
            }
            return NULL;
        }

        pl = sph_keyword_manager_load_properties( cpl_frame_get_filename( curframe ), 0 );

        pl_proper = cpl_propertylist_load(  cpl_frame_get_filename( curframe ), 0);
        if ( cpl_propertylist_has( pl_proper, "NAXIS3") ) {
           nz = cpl_propertylist_get_int( pl_proper, "NAXIS3");
        } else {
           sph_error_raise(SPH_ERROR_WARNING,
                   __FILE__, __func__, __LINE__,
                   SPH_ERROR_WARNING, "An input frame is not a cube: %s ;"
                                      " proceed further anyway",
                                       cpl_frame_get_filename( curframe ) );
           nz = 1;
        }

        for ( int i = 0; i < nz; i++){
            zplexp = sph_zpl_exposure_imaging_load( curframe, i );
            if (!zplexp) {
                sph_error_raise(SPH_ERROR_WARNING,
                         __FILE__, __func__, __LINE__,
                         SPH_ERROR_WARNING, "Could not load an individual zpl exp imaging frame: %s. Goto the next cube",
                                             cpl_frame_get_filename( curframe ) );
                 break;
            }
            expframe = sph_filemanager_create_temp_frame( cpl_frame_get_filename( curframe ),
                                               "NONE", CPL_FRAME_GROUP_NONE );

            //add to the global eso property list of the zpl exp cube particular information about an
            //individual frame such as time stamp, PA, H, ...
            //this particular information must be retrieved(calculated), based on the general keywords
            //of the cube (start time, end time, number of zplexp plane, etc...)

            rerr = sph_zpl_exposure_imaging_save( zplexp, expframe, pl);
            if (rerr != CPL_ERROR_NONE) {
                   sph_error_raise(SPH_ERROR_WARNING,
                           __FILE__, __func__, __LINE__,
                           SPH_ERROR_WARNING, "Could not save an individual zpl exp frame: %s. Goto the next cube",
                                               cpl_frame_get_filename( expframe ) );
                   if (zplexp) {
                       sph_zpl_exposure_imaging_delete( zplexp ); zplexp = NULL;
                   }
                   break;
            }
            sph_zpl_exposure_imaging_delete( zplexp );
            rerr = cpl_frameset_insert( expframes, expframe);
            if (rerr != CPL_ERROR_NONE) {
                   sph_error_raise(SPH_ERROR_ERROR,
                           __FILE__, __func__, __LINE__,
                           SPH_ERROR_ERROR, "Could not insert expframe with the filename -- %s -- into expframes",
                                               cpl_frame_get_filename( expframe ) );
                     if (keym) {
                         sph_keyword_manager_delete(); keym = NULL;
                     }
                     if (expframes) {
                         cpl_frameset_delete( expframes); expframes = NULL;
                     }
                     cpl_propertylist_delete(pl); pl = NULL;
                     cpl_propertylist_delete(pl_proper); pl_proper = NULL;
                  return NULL;
            }

        } //end for

        result_expframes = sph_double_image_create_double_image_frameset_from_zplexp_imaging_frameset( expframes );

        if (result_expframes) {
            cpl_frameset_join(result, result_expframes);
            cpl_frameset_delete( result_expframes ); result_expframes = NULL;

        } else {
            SPH_ERR("Could not create quad image result_frameset from expframes")
        }

        if (expframes) {
            cpl_frameset_delete( expframes); expframes = NULL;
        }


        cpl_propertylist_delete( pl );
        cpl_propertylist_delete( pl_proper );
        curframe = cpl_frameset_get_next_const( inframes );
    } //end while

    if (cpl_frameset_get_size(result) == 0) {
        sph_error_raise( SPH_DOUBLE_IMAGE_GENERAL, __FILE__,  __func__, __LINE__,
                                 SPH_ERROR_WARNING, "The result frameset has no elements." );
        cpl_frameset_delete(result); result = NULL;
    }

    return result;


}


/*----------------------------------------------------------------------------*/
/**
  @brief        Verify whether double image is squared
  @param self   the sph_zpl_exposure to
  @return       cpl boolean (CPL_TRUE or CPL_FALSE)

 * This function checks if the double image is squared.
 * Verification is done only for image extensions.
 */
/*----------------------------------------------------------------------------*/
cpl_boolean sph_double_image_is_squared(const sph_double_image* self) {

    return self && self->iframe && self->pframe &&
        self->iframe->image && self->pframe->image &&
        cpl_image_get_size_x(self->iframe->image) ==
        cpl_image_get_size_y(self->iframe->image) &&
        cpl_image_get_size_x(self->pframe->image) ==
        cpl_image_get_size_y(self->pframe->image)
        ? CPL_TRUE : CPL_FALSE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief        Expand double image in y direction by factor 2 duplicating each line

  @param self   the sph_double_image to expand

  @return       return sphere error code
  @note         A CPL error is raised in case there was a problem.

 * This function expands double image by factor 2 duplicating each line .
 * Self is changed in place.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_double_image_expand2y( sph_double_image* self ){
    sph_double_image*           dimage  = NULL;
    cpl_image*                  im      = NULL;
    int                         llx     = 0;
    int                         lly     = 0;
    int                         urx     = 0;
    int                         ury     = 0;
    int                         nx      = 0;
    int                         ny      = 0;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );


    dimage = sph_double_image_duplicate( self );
    nx = cpl_image_get_size_x( self->iframe->image );
    ny = cpl_image_get_size_y( self->iframe->image );
    ny = ny*2;

    //printf("nx=%d, ny=%d \n", nx, ny);
    //delete images from the self double image
    cpl_image_delete(self->iframe->image);
    cpl_image_delete(self->iframe->badpixelmap);
    cpl_image_delete(self->iframe->ncombmap);
    cpl_image_delete(self->iframe->rmsmap);
    cpl_image_delete(self->pframe->image);
    cpl_image_delete(self->pframe->badpixelmap);
    cpl_image_delete(self->pframe->ncombmap);
    cpl_image_delete(self->pframe->rmsmap);


    //re-define the sizes of the images
    self->iframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->iframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    self->pframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->pframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);


    llx = 1; urx = nx;
    for (int yy = 0; yy < ny; yy=yy+2 ){

        lly = (yy/2) + 1;
        ury = lly;

        //printf("yy=%d, llx=%d, lly=%d, urx=%d, ury=%d\n", yy, llx, lly, urx, ury );
        //iframe
        im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->image, im, llx, yy+1 );
        cpl_image_copy( self->iframe->image, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->badpixelmap, im, llx, yy+1 );
        cpl_image_copy( self->iframe->badpixelmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->ncombmap, im, llx, yy+1 );
        cpl_image_copy( self->iframe->ncombmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->rmsmap, im, llx, yy+1 );
        cpl_image_copy( self->iframe->rmsmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        //pframe
        im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->image, im, llx, yy+1 );
        cpl_image_copy( self->pframe->image, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->badpixelmap, im, llx, yy+1 );
        cpl_image_copy( self->pframe->badpixelmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->ncombmap, im, llx, yy+1 );
        cpl_image_copy( self->pframe->ncombmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->rmsmap, im, llx, yy+1 );
        cpl_image_copy( self->pframe->rmsmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;
    }
    sph_double_image_delete( dimage );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;

}

sph_error_code sph_double_image_interpolate_y_zpl_imaging( sph_double_image* self, cpl_boolean set_badpxl){

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );

    sph_double_image_interpolate_y( self, set_badpxl );
    sph_double_image_interpolate_y_adjust_boundarylines_zpl_imaging( self, CPL_TRUE, set_badpxl );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;

}

sph_error_code sph_double_image_interpolate_y_zpl_polarimetry( sph_double_image* self, cpl_boolean set_badpxl){
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );

    sph_double_image_interpolate_y( self, set_badpxl );
    sph_double_image_interpolate_y_adjust_boundarylines_zpl_polarimetry( self, set_badpxl );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;

}



/*----------------------------------------------------------------------------*/
/**
  @brief        Interpolate a double image in the y axis (linear interpolation)

  @param self       The original sph_double_image to interpolate
  @param set_badpxl FIXME: Input parameter
  @return       return sphere error code
  @note         A CPL error is raised in case there was a problem.

 * This function performs a linear interpolation in the y-axis of the double image.
 * Self is changed in place.
 * This scheme accounts for the light focusing by the cylindrical pixel lenses and
 * preserves the total count in the data. See for details "ZIMPOL DRH astrometry
 * and de-rotation" document from Hans-Martin Schmid.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_double_image_interpolate_y( sph_double_image* self,
                                               cpl_boolean set_badpxl) {
    sph_double_image*           dimage  = NULL;
    cpl_image*                  im      = NULL;
    cpl_image*                  im_a    = NULL;
    cpl_image*                  im_b    = NULL;
    int                         llx     = 0;
    int                         lly     = 0;
    int                         urx     = 0;
    int                         ury     = 0;
    int                         nx      = 0;
    int                         ny      = 0;
    int                         ny_init = 0;
    int                         lly_b   = 0;
    int                         ury_b   = 0;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );


    dimage = sph_double_image_duplicate( self );
    nx = cpl_image_get_size_x( self->iframe->image );
    ny_init = cpl_image_get_size_y( self->iframe->image );
    ny = ny_init*2-1;
    //printf("nx=%d, ny=%d \n", nx, ny);

    //delete images from the self double image
    cpl_image_delete(self->iframe->image);
    cpl_image_delete(self->iframe->badpixelmap);
    cpl_image_delete(self->iframe->ncombmap);
    cpl_image_delete(self->iframe->rmsmap);
    cpl_image_delete(self->pframe->image);
    cpl_image_delete(self->pframe->badpixelmap);
    cpl_image_delete(self->pframe->ncombmap);
    cpl_image_delete(self->pframe->rmsmap);


    //re-define the sizes of the images
    self->iframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->iframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    self->pframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->pframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    llx = 1; urx = nx;
    for (int yy = 0; yy < ny; yy=yy+2 ){

        lly = (yy/2) + 1;
        ury = lly;


        lly_b = lly + 1;
        ury_b = lly_b;
        //printf("yy=%d, llx=%d, lly=%d, urx=%d, ury=%d\n", yy, llx, lly, urx, ury );
        //printf("lly_b=%d, ury_b=%d\n", lly_b, ury_b );

        //iframe
        im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
        if ( lly_b <= ny_init ) {
            im_a = cpl_image_duplicate((const cpl_image*) im);
            cpl_image_divide_scalar( im_a, 4.0 );
            im_b = cpl_image_extract( dimage->iframe->image, llx, lly_b, urx, ury_b );
            cpl_image_divide_scalar( im_b, 4.0 );
            cpl_image_add( im_a, (const cpl_image*) im_b);
        }
        cpl_image_divide_scalar( im, 2.0 );
        cpl_image_copy( self->iframe->image, im, llx, yy+1 );
        if ( lly_b <= ny_init ) cpl_image_copy( self->iframe->image, im_a, llx, yy+2 );
        if ( im ) cpl_image_delete( im ); im = NULL;
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;
        if ( im_b ) cpl_image_delete( im_b ); im_b = NULL;

        im_a = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->badpixelmap, im_a, llx, yy+1 );
        if ( lly_b <= ny_init ) {
            im_b = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly_b, urx, ury_b );
            cpl_image_add( im_b, (const cpl_image*) im_a);
            cpl_image_copy( self->iframe->badpixelmap, im_b, llx, yy+2 );
            if (im_b) cpl_image_delete( im_b ); im_b = NULL;
        }
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;

        im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->ncombmap, im, llx, yy+1 );
        if (lly_b <= ny_init )cpl_image_copy( self->iframe->ncombmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im_a = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->rmsmap, im_a, llx, yy+1 );
        if ( lly_b <= ny_init ){
            im_b =  cpl_image_extract( dimage->iframe->rmsmap, llx, lly_b, urx, ury_b );
            cpl_image_add( im_b, (const cpl_image*) im_a);
            cpl_image_copy( self->iframe->rmsmap, im_b, llx, yy+2 );
            if (im_b) cpl_image_delete( im_b ); im_b = NULL;
        }
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;

        //pframe
        im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
        if ( lly_b <= ny_init ) {
            im_a = cpl_image_duplicate((const cpl_image*) im);
            cpl_image_divide_scalar( im_a, 4.0 );
            im_b = cpl_image_extract( dimage->pframe->image, llx, lly_b, urx, ury_b );
            cpl_image_divide_scalar( im_b, 4.0 );
            cpl_image_add( im_a, (const cpl_image*) im_b);
        }
        cpl_image_divide_scalar( im, 2.0 );
        cpl_image_copy( self->pframe->image, im, llx, yy+1 );
        if ( lly_b <= ny_init ) cpl_image_copy( self->pframe->image, im_a, llx, yy+2 );
        if ( im ) cpl_image_delete( im ); im = NULL;
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;
        if ( im_b ) cpl_image_delete( im_b ); im_b = NULL;

        im_a = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->badpixelmap, im_a, llx, yy+1 );
        if ( lly_b <= ny_init ) {
            im_b = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly_b, urx, ury_b );
            cpl_image_add( im_b, (const cpl_image*) im_a);
            cpl_image_copy( self->pframe->badpixelmap, im_b, llx, yy+2 );
            if (im_b) cpl_image_delete( im_b ); im_b = NULL;
        }
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;

        im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->ncombmap, im, llx, yy+1 );
        if (lly_b <= ny_init) cpl_image_copy( self->pframe->ncombmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im_a = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->rmsmap, im_a, llx, yy+1 );
        if ( lly_b <= ny_init ){
            im_b =  cpl_image_extract( dimage->pframe->rmsmap, llx, lly_b, urx, ury_b );
            cpl_image_add( im_b, (const cpl_image*) im_a);
            cpl_image_copy( self->pframe->rmsmap, im_b, llx, yy+2 );
            if (im_b) cpl_image_delete( im_b ); im_b = NULL;
        }
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;


    }
    sph_double_image_delete( dimage );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;

}

static sph_error_code sph_double_image_interpolate_y_adjust_boundarylines_zpl_imaging
                       ( sph_double_image* self,
                         cpl_boolean add_single_line,
                         cpl_boolean set_badpxl ){
    sph_double_image*           dimage  = NULL;
    cpl_image*                  im      = NULL;
    int                         llx     = 0;
    int                         lly     = 0;
    int                         urx     = 0;
    int                         ury     = 0;
    int                         nx      = 0;
    int                         ny      = 0;
    int               badpxl_dummyvalue = 1;
    double           image_dummyvalue   = 0.0;
    int                         inysize = 2;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );


    if ( add_single_line ) inysize = 1;
    dimage = sph_double_image_duplicate( self );
    nx = cpl_image_get_size_x( self->iframe->image );
    ny = cpl_image_get_size_y( self->iframe->image );
    ny = ny + inysize;

    //printf("nx=%d, ny=%d \n", nx, ny);
    //delete images from the self double image
    cpl_image_delete(self->iframe->image);
    cpl_image_delete(self->iframe->badpixelmap);
    cpl_image_delete(self->iframe->ncombmap);
    cpl_image_delete(self->iframe->rmsmap);
    cpl_image_delete(self->pframe->image);
    cpl_image_delete(self->pframe->badpixelmap);
    cpl_image_delete(self->pframe->ncombmap);
    cpl_image_delete(self->pframe->rmsmap);


    //re-define the sizes of the images
    self->iframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->iframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    self->pframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->pframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);


    llx = 1; urx = nx;
    lly = 1; ury = lly;

    //duplicate the first line from the bottom (dividev by 2)

    //iframe
    im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
    cpl_image_divide_scalar( im, 2.0);
    cpl_image_copy( self->iframe->image, im, llx, lly );
    cpl_image_delete( im ); im = NULL;

    if ( set_badpxl ) {
        for (int xx = 0; xx < nx; ++xx) {
            cpl_image_set( self->iframe->badpixelmap, xx + 1, lly, badpxl_dummyvalue );
            cpl_image_reject( self->iframe->image, xx + 1, lly ); //set badpixel in the image struct itself
            cpl_image_set( self->iframe->rmsmap, xx + 1, lly, SPH_MASTER_FRAME_BAD_RMS );
            cpl_image_set( self->iframe->ncombmap, xx + 1, lly, image_dummyvalue );
        }
    }  else {
        im = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->badpixelmap, im, llx, lly );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->ncombmap, im, llx, lly );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->rmsmap, im, llx, lly );
        cpl_image_delete( im ); im = NULL;
    }

    //pframe
    im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
    cpl_image_divide_scalar( im, 2.0);
    cpl_image_copy( self->pframe->image, im, llx, lly );
    cpl_image_delete( im ); im = NULL;

    if ( set_badpxl ) {
        for (int xx = 0; xx < nx; ++xx) {
            cpl_image_set( self->pframe->badpixelmap, xx + 1, lly, badpxl_dummyvalue );
            cpl_image_reject( self->pframe->image, xx + 1, lly ); //set badpixel in the image struct itself
            cpl_image_set( self->pframe->rmsmap, xx + 1, lly, SPH_MASTER_FRAME_BAD_RMS );
            cpl_image_set( self->pframe->ncombmap, xx + 1, lly, image_dummyvalue );
        }

    } else {
        im = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->badpixelmap, im, llx, lly );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->ncombmap, im, llx, lly );
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->rmsmap, im, llx, lly );
        cpl_image_delete( im ); im = NULL;
    }


    //copy original image into the new created
    //iframe
    llx=1; lly = 2;
    cpl_image_copy( self->iframe->image, dimage->iframe->image, llx, lly );
    cpl_image_copy( self->iframe->badpixelmap, dimage->iframe->badpixelmap, llx, lly );
    cpl_image_copy( self->iframe->ncombmap, dimage->iframe->ncombmap, llx, lly );
    cpl_image_copy( self->iframe->rmsmap, dimage->iframe->rmsmap, llx, lly );
    //pframe
    cpl_image_copy( self->pframe->image, dimage->pframe->image, llx, lly );
    cpl_image_copy( self->pframe->badpixelmap, dimage->pframe->badpixelmap, llx, lly );
    cpl_image_copy( self->pframe->ncombmap, dimage->pframe->ncombmap, llx, lly );
    cpl_image_copy( self->pframe->rmsmap, dimage->pframe->rmsmap, llx, lly );


    //add (duplicate) top line (counts divided by 2.0) to the new image
    //iframe
    if ( !add_single_line ) {
        const int shifty = 2;
        llx = 1; lly = cpl_image_get_size_y( dimage->iframe->image );
        urx = nx; ury = lly;

        //iframe
        im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
        cpl_image_divide_scalar( im, 2.0);
        cpl_image_copy( self->iframe->image, im, llx, lly + shifty );
        cpl_image_delete( im ); im = NULL;

        if ( set_badpxl ) {
            for (int xx = 0; xx < nx; ++xx) {
                cpl_image_set( self->iframe->badpixelmap, xx + 1, lly + shifty, badpxl_dummyvalue );
                cpl_image_reject( self->iframe->image, xx + 1, lly + shifty); //set badpixel in the image struct itself
                cpl_image_set( self->iframe->rmsmap, xx + 1, lly + shifty, SPH_MASTER_FRAME_BAD_RMS );
                cpl_image_set( self->iframe->ncombmap, xx + 1, lly + shifty, image_dummyvalue );
            }
        }  else {
            im = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
            cpl_image_copy( self->iframe->badpixelmap, im, llx, lly + shifty);
            cpl_image_delete( im ); im = NULL;

            im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
            cpl_image_copy( self->iframe->ncombmap, im, llx, lly + shifty);
            cpl_image_delete( im ); im = NULL;

            im = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
            cpl_image_copy( self->iframe->rmsmap, im, llx, lly + shifty);
            cpl_image_delete( im ); im = NULL;
        }

        //pframe
        im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
        cpl_image_divide_scalar( im, 2.0);
        cpl_image_copy( self->pframe->image, im, llx, lly + shifty);
        cpl_image_delete( im ); im = NULL;

        if ( set_badpxl ) {
            for (int xx = 0; xx < nx; ++xx) {
                cpl_image_set( self->pframe->badpixelmap, xx + 1, lly + shifty, badpxl_dummyvalue );
                cpl_image_reject( self->pframe->image, xx + 1, lly + shifty); //set badpixel in the image struct itself
                cpl_image_set( self->pframe->rmsmap, xx + 1, lly + shifty, SPH_MASTER_FRAME_BAD_RMS );
                cpl_image_set( self->pframe->ncombmap, xx + 1, lly + shifty, image_dummyvalue );
            }

        } else {
            im = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
            cpl_image_copy( self->pframe->badpixelmap, im, llx, lly + shifty);
            cpl_image_delete( im ); im = NULL;

            im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
            cpl_image_copy( self->pframe->ncombmap, im, llx, lly  + shifty);
            cpl_image_delete( im ); im = NULL;

            im = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
            cpl_image_copy( self->pframe->rmsmap, im, llx, lly + shifty);
            cpl_image_delete( im ); im = NULL;
        }
    }


    sph_double_image_delete( dimage );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static sph_error_code sph_double_image_interpolate_y_adjust_boundarylines_zpl_polarimetry
                       ( sph_double_image* self, cpl_boolean set_badpxl ){

    sph_double_image*           dimage  = NULL;
    cpl_image*                  im      = NULL;
    int                         llx     = 0;
    int                         lly     = 0;
    int                         urx     = 0;
    int                         ury     = 0;
    int                         nx      = 0;
    int                         ny      = 0;
    const int                   shifty  = 1;
    int               badpxl_dummyvalue = 1;
    double           image_dummyvalue   = 0.0;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );


    dimage = sph_double_image_duplicate( self );
    nx = cpl_image_get_size_x( self->iframe->image );
    ny = cpl_image_get_size_y( self->iframe->image );
    ny = ny + 3;

    //printf("nx=%d, ny=%d \n", nx, ny);
    //delete images from the self double image
    cpl_image_delete(self->iframe->image);
    cpl_image_delete(self->iframe->badpixelmap);
    cpl_image_delete(self->iframe->ncombmap);
    cpl_image_delete(self->iframe->rmsmap);
    cpl_image_delete(self->pframe->image);
    cpl_image_delete(self->pframe->badpixelmap);
    cpl_image_delete(self->pframe->ncombmap);
    cpl_image_delete(self->pframe->rmsmap);


    //re-define the sizes of the images
    self->iframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->iframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    self->pframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->pframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    //add the first dummy line into the new image
    for (int xx = 0; xx < nx; ++xx) {
        for (int yy = 0; yy < 1; ++yy) {
            cpl_image_set( self->iframe->image, xx + 1, yy + 1, image_dummyvalue );
            cpl_image_set( self->iframe->badpixelmap, xx + 1, yy + 1, badpxl_dummyvalue );
            cpl_image_reject( self->iframe->image, xx + 1, yy + 1 ); //set badpixel in the image struct itself
            cpl_image_set( self->iframe->rmsmap, xx+1, yy+1, SPH_MASTER_FRAME_BAD_RMS );

            cpl_image_set( self->pframe->image, xx + 1, yy + 1, image_dummyvalue );
            cpl_image_set( self->pframe->badpixelmap, xx + 1, yy + 1, badpxl_dummyvalue );
            cpl_image_reject( self->pframe->image, xx + 1, yy + 1 ); //set badpixel in the image struct itself
            cpl_image_set( self->pframe->rmsmap, xx+1, yy+1, SPH_MASTER_FRAME_BAD_RMS );
        }
    }


    llx = 1; urx = nx;
    lly = 1; ury = lly;

    //duplicate the first line from the bottom (dividev by 2)

    //iframe
    im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
    cpl_image_divide_scalar( im, 2.0);
    cpl_image_copy( self->iframe->image, im, llx, lly + shifty);
    cpl_image_delete( im ); im = NULL;

    if ( set_badpxl ) {
        for (int xx = 0; xx < nx; ++xx) {
            cpl_image_set( self->iframe->badpixelmap, xx + 1, lly + shifty, badpxl_dummyvalue );
            cpl_image_reject( self->iframe->image, xx + 1, lly + shifty); //set badpixel in the image struct itself
            cpl_image_set( self->iframe->rmsmap, xx + 1, lly + shifty, SPH_MASTER_FRAME_BAD_RMS );
            cpl_image_set( self->iframe->ncombmap, xx + 1, lly + shifty, image_dummyvalue );
        }
    }  else {
        im = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->badpixelmap, im, llx, lly + shifty);
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->ncombmap, im, llx, lly + shifty);
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->rmsmap, im, llx, lly + shifty);
        cpl_image_delete( im ); im = NULL;
    }

    //pframe
    im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
    cpl_image_divide_scalar( im, 2.0);
    cpl_image_copy( self->pframe->image, im, llx, lly + shifty);
    cpl_image_delete( im ); im = NULL;

    if ( set_badpxl ) {
        for (int xx = 0; xx < nx; ++xx) {
            cpl_image_set( self->pframe->badpixelmap, xx + 1, lly + shifty, badpxl_dummyvalue );
            cpl_image_reject( self->pframe->image, xx + 1, lly + shifty ); //set badpixel in the image struct itself
            cpl_image_set( self->pframe->rmsmap, xx + 1, lly + shifty, SPH_MASTER_FRAME_BAD_RMS );
            cpl_image_set( self->pframe->ncombmap, xx + 1, lly + shifty, image_dummyvalue );
        }

    } else {
        im = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->badpixelmap, im, llx, lly + shifty);
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->ncombmap, im, llx, lly + shifty);
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->rmsmap, im, llx, lly + shifty );
        cpl_image_delete( im ); im = NULL;
    }


    //copy original image into the new created
    //iframe
    llx=1; lly = 3;
    cpl_image_copy( self->iframe->image, dimage->iframe->image, llx, lly );
    cpl_image_copy( self->iframe->badpixelmap, dimage->iframe->badpixelmap, llx, lly );
    cpl_image_copy( self->iframe->ncombmap, dimage->iframe->ncombmap, llx, lly );
    cpl_image_copy( self->iframe->rmsmap, dimage->iframe->rmsmap, llx, lly );
    //pframe
    cpl_image_copy( self->pframe->image, dimage->pframe->image, llx, lly );
    cpl_image_copy( self->pframe->badpixelmap, dimage->pframe->badpixelmap, llx, lly );
    cpl_image_copy( self->pframe->ncombmap, dimage->pframe->ncombmap, llx, lly );
    cpl_image_copy( self->pframe->rmsmap, dimage->pframe->rmsmap, llx, lly );


    //add (duplicate) top line (counts divided by 2.0) to the new image
    //iframe
    llx = 1; lly = cpl_image_get_size_y( dimage->iframe->image );
    urx = nx; ury = lly;

    //iframe
    im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
    cpl_image_divide_scalar( im, 2.0);
    cpl_image_copy( self->iframe->image, im, llx, ny );
    cpl_image_delete( im ); im = NULL;

    if ( set_badpxl ) {
        for (int xx = 0; xx < nx; ++xx) {
            cpl_image_set( self->iframe->badpixelmap, xx + 1, ny, badpxl_dummyvalue );
            cpl_image_reject( self->iframe->image, xx + 1, ny); //set badpixel in the image struct itself
            cpl_image_set( self->iframe->rmsmap, xx + 1, ny, SPH_MASTER_FRAME_BAD_RMS );
            cpl_image_set( self->iframe->ncombmap, xx + 1, ny, image_dummyvalue );
        }
    }  else {
        im = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->badpixelmap, im, llx, ny);
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->ncombmap, im, llx, ny);
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->rmsmap, im, llx, ny);
        cpl_image_delete( im ); im = NULL;
    }

    //pframe
    im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
    cpl_image_divide_scalar( im, 2.0);
    cpl_image_copy( self->pframe->image, im, llx, ny);
    cpl_image_delete( im ); im = NULL;

    if ( set_badpxl ) {
        for (int xx = 0; xx < nx; ++xx) {
            cpl_image_set( self->pframe->badpixelmap, xx + 1, ny, badpxl_dummyvalue );
            cpl_image_reject( self->pframe->image, xx + 1, ny); //set badpixel in the image struct itself
            cpl_image_set( self->pframe->rmsmap, xx + 1, ny, SPH_MASTER_FRAME_BAD_RMS );
            cpl_image_set( self->pframe->ncombmap, xx + 1, ny, image_dummyvalue );
        }

    } else {
        im = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->badpixelmap, im, llx, ny);
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->ncombmap, im, llx, ny);
        cpl_image_delete( im ); im = NULL;

        im = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->rmsmap, im, llx, ny);
        cpl_image_delete( im ); im = NULL;
    }

    sph_double_image_delete( dimage );

    return cpl_error_set_where(cpl_func);
}




/*----------------------------------------------------------------------------*/
/**
  @brief        Create a squared double image (linear interpolation)

  @param self   the original sph_double_image to interpolate (zpl format [1024x511]

  @return       return sphere error code
  @note         A CPL error is raised in case there was a problem.

 * This function performs a linear interpolation in the y-axis of the image creating the image
 * of the size [1024x1022]. It also adds  two "dummy" lines with zero values
 * (marked as bad one in the badpixel map) in order to get a final squared image with the size [1024x1024].
 * This scheme  account for the light focusing by the cylindrical pixel lenses and preserves the total count in the data.
 * See for details "ZIMPOL DRH astrometry and de-rotation" document from Hans-Martin Schmid.
 * Self is changed in place.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_double_image_create_squared( sph_double_image* self ){
    sph_double_image*           dimage  = NULL;
    cpl_image*                  im      = NULL;
    cpl_image*                  im_a    = NULL;
    cpl_image*                  im_b    = NULL;
    int                         llx     = 0;
    int                         lly     = 0;
    int                         urx     = 0;
    int                         ury     = 0;
    int                         nx      = 0;
    int                         ny      = 0;
    int                         ny_init = 0;
    int                         lly_b   = 0;
    int                         ury_b   = 0;
    double           image_dummyvalue   = 0.0;
    int               badpxl_dummyvalue = 1;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );


    dimage = sph_double_image_duplicate( self );
    nx = cpl_image_get_size_x( self->iframe->image );
    ny_init = cpl_image_get_size_y( self->iframe->image );
    ny = ny_init*2 + 2;


    //printf("nx=%d, ny=%d \n", nx, ny);
    //delete images from the self double image
    cpl_image_delete(self->iframe->image);
    cpl_image_delete(self->iframe->badpixelmap);
    cpl_image_delete(self->iframe->ncombmap);
    cpl_image_delete(self->iframe->rmsmap);
    cpl_image_delete(self->pframe->image);
    cpl_image_delete(self->pframe->badpixelmap);
    cpl_image_delete(self->pframe->ncombmap);
    cpl_image_delete(self->pframe->rmsmap);


    //re-define the sizes of the images
    self->iframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->iframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    self->pframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->pframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);


    //add two first dummy lines to get a squared image
    for (int xx = 0; xx < nx; ++xx) {
        for (int yy = 0; yy < 3; ++yy) {
            cpl_image_set( self->iframe->image, xx + 1, yy + 1, image_dummyvalue );
            cpl_image_set( self->iframe->badpixelmap, xx + 1, yy + 1, badpxl_dummyvalue );
            cpl_image_reject( self->iframe->image, xx + 1, yy + 1 ); //set badpixel in the image struct itself
            cpl_image_set( self->iframe->rmsmap, xx+1, yy+1, SPH_MASTER_FRAME_BAD_RMS );

            cpl_image_set( self->pframe->image, xx + 1, yy + 1, image_dummyvalue );
            cpl_image_set( self->pframe->badpixelmap, xx + 1, yy + 1, badpxl_dummyvalue );
            cpl_image_reject( self->pframe->image, xx + 1, yy + 1 ); //set badpixel in the image struct itself
            cpl_image_set( self->pframe->rmsmap, xx+1, yy+1, SPH_MASTER_FRAME_BAD_RMS );

        }
    }


    llx = 1; urx = nx;
    for (int yy = 2; yy < ny; yy=yy+2 ){

        lly = (yy/2);
        ury = lly;


        lly_b = lly + 1;
        ury_b = lly_b;
        //printf("yy=%d, llx=%d, lly=%d, urx=%d, ury=%d\n", yy, llx, lly, urx, ury );
        //printf("lly_b=%d, ury_b=%d\n", lly_b, ury_b );

        //iframe
        im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
        if ( lly_b <= ny_init ) {
            im_a = cpl_image_duplicate((const cpl_image*) im);
            cpl_image_divide_scalar( im_a, 4.0 );
            im_b = cpl_image_extract( dimage->iframe->image, llx, lly_b, urx, ury_b );
            cpl_image_divide_scalar( im_b, 4.0 );
            cpl_image_add( im_a, (const cpl_image*) im_b);
        }
        cpl_image_divide_scalar( im, 2.0 );
        cpl_image_copy( self->iframe->image, im, llx, yy+1 );
        if ( lly_b <= ny_init ) cpl_image_copy( self->iframe->image, im_a, llx, yy+2 );
        if ( im ) cpl_image_delete( im ); im = NULL;
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;
        if ( im_b ) cpl_image_delete( im_b ); im_b = NULL;

        im_a = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->badpixelmap, im_a, llx, yy+1 );
        if ( lly_b <= ny_init ) {
            im_b = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly_b, urx, ury_b );
            cpl_image_add( im_b, (const cpl_image*) im_a);
            cpl_image_copy( self->iframe->badpixelmap, im_b, llx, yy+2 );
            if (im_b) cpl_image_delete( im_b ); im_b = NULL;
        }
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;

        im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->ncombmap, im, llx, yy+1 );
        cpl_image_copy( self->iframe->ncombmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im_a = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->iframe->rmsmap, im_a, llx, yy+1 );
        if ( lly_b <= ny_init ){
            im_b =  cpl_image_extract( dimage->iframe->rmsmap, llx, lly_b, urx, ury_b );
            cpl_image_add( im_b, (const cpl_image*) im_a);
            cpl_image_copy( self->iframe->rmsmap, im_b, llx, yy+2 );
            if (im_b) cpl_image_delete( im_b ); im_b = NULL;
        }
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;

        //pframe
        im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
        if ( lly_b <= ny_init ) {
            im_a = cpl_image_duplicate((const cpl_image*) im);
            cpl_image_divide_scalar( im_a, 4.0 );
            im_b = cpl_image_extract( dimage->pframe->image, llx, lly_b, urx, ury_b );
            cpl_image_divide_scalar( im_b, 4.0 );
            cpl_image_add( im_a, (const cpl_image*) im_b);
        }
        cpl_image_divide_scalar( im, 2.0 );
        cpl_image_copy( self->pframe->image, im, llx, yy+1 );
        if ( lly_b <= ny_init ) cpl_image_copy( self->pframe->image, im_a, llx, yy+2 );
        if ( im ) cpl_image_delete( im ); im = NULL;
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;
        if ( im_b ) cpl_image_delete( im_b ); im_b = NULL;

        im_a = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->badpixelmap, im_a, llx, yy+1 );
        if ( lly_b <= ny_init ) {
            im_b = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly_b, urx, ury_b );
            cpl_image_add( im_b, (const cpl_image*) im_a);
            cpl_image_copy( self->pframe->badpixelmap, im_b, llx, yy+2 );
            if (im_b) cpl_image_delete( im_b ); im_b = NULL;
        }
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;

        im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->ncombmap, im, llx, yy+1 );
        cpl_image_copy( self->pframe->ncombmap, im, llx, yy+2 );
        cpl_image_delete( im ); im = NULL;

        im_a = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
        cpl_image_copy( self->pframe->rmsmap, im_a, llx, yy+1 );
        if ( lly_b <= ny_init ){
            im_b =  cpl_image_extract( dimage->pframe->rmsmap, llx, lly_b, urx, ury_b );
            cpl_image_add( im_b, (const cpl_image*) im_a);
            cpl_image_copy( self->pframe->rmsmap, im_b, llx, yy+2 );
            if (im_b) cpl_image_delete( im_b ); im_b = NULL;
        }
        if ( im_a ) cpl_image_delete( im_a ); im_a = NULL;


    }
    sph_double_image_delete( dimage );

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;

}




/*----------------------------------------------------------------------------*/
/**
  @brief        Add new top and bottom lines to the given double image

  @param self   the sph_double_image to add lines to

  @return       return sphere error code
  @note         A CPL error is raised in case there was a problem.

 * This function adds new top and bottom lines to the given double image.
 * Self is changed in place.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_double_image_add_topbottom_lines( sph_double_image* self ){
    sph_double_image*            dimage    = NULL;
    cpl_image*                    im        = NULL;
    int                            llx        = 0;
    int                         lly        = 0;
    int                         urx        = 0;
    int                         ury        = 0;
    int                         nx         = 0;
    int                            ny        = 0;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );


    dimage = sph_double_image_duplicate( self );
    nx = cpl_image_get_size_x( self->iframe->image );
    ny = cpl_image_get_size_y( self->iframe->image );
    ny = ny + 2;

    //printf("nx=%d, ny=%d \n", nx, ny);
    //delete images from the self double image
    cpl_image_delete(self->iframe->image);
    cpl_image_delete(self->iframe->badpixelmap);
    cpl_image_delete(self->iframe->ncombmap);
    cpl_image_delete(self->iframe->rmsmap);
    cpl_image_delete(self->pframe->image);
    cpl_image_delete(self->pframe->badpixelmap);
    cpl_image_delete(self->pframe->ncombmap);
    cpl_image_delete(self->pframe->rmsmap);


    //re-define the sizes of the images
    self->iframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->iframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    self->pframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->pframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);




       llx = 1; urx = nx;
       lly = 1; ury = 1;

       //add upper line to the new image
       //iframe
       im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->image, im, llx, lly );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->badpixelmap, im, llx, lly );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->ncombmap, im, llx, lly );
    cpl_image_delete( im ); im = NULL;

    im = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
    cpl_image_copy( self->iframe->rmsmap, im, llx, lly );
    cpl_image_delete( im ); im = NULL;

    //pframe
    im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
    cpl_image_copy( self->pframe->image, im, llx, lly );
    cpl_image_delete( im ); im = NULL;

    im = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
    cpl_image_copy( self->pframe->badpixelmap, im, llx, lly );
    cpl_image_delete( im ); im = NULL;

    im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
    cpl_image_copy( self->pframe->ncombmap, im, llx, lly );
    cpl_image_delete( im ); im = NULL;

    im = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
    cpl_image_copy( self->pframe->rmsmap, im, llx, lly );
    cpl_image_delete( im ); im = NULL;

    //copy original image into the new created
    //iframe
    llx=1; lly = 2;
    cpl_image_copy( self->iframe->image, dimage->iframe->image, llx, lly );
    cpl_image_copy( self->iframe->badpixelmap, dimage->iframe->badpixelmap, llx, lly );
    cpl_image_copy( self->iframe->ncombmap, dimage->iframe->ncombmap, llx, lly );
    cpl_image_copy( self->iframe->rmsmap, dimage->iframe->rmsmap, llx, lly );
    //pframe
    cpl_image_copy( self->pframe->image, dimage->pframe->image, llx, lly );
    cpl_image_copy( self->pframe->badpixelmap, dimage->pframe->badpixelmap, llx, lly );
    cpl_image_copy( self->pframe->ncombmap, dimage->pframe->ncombmap, llx, lly );
    cpl_image_copy( self->pframe->rmsmap, dimage->pframe->rmsmap, llx, lly );


       //add bottom line to the new image
       //iframe
    llx = 1; lly = cpl_image_get_size_y( dimage->iframe->image );
    urx = nx; ury = cpl_image_get_size_y( dimage->iframe->image );

       im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->image, im, llx, ny );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->badpixelmap, im, llx, ny );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->ncombmap, im, llx, ny );
    cpl_image_delete( im ); im = NULL;

    im = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
    cpl_image_copy( self->iframe->rmsmap, im, llx, ny );
    cpl_image_delete( im ); im = NULL;

    //pframe
    im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
    cpl_image_copy( self->pframe->image, im, llx, ny );
    cpl_image_delete( im ); im = NULL;

    im = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
    cpl_image_copy( self->pframe->badpixelmap, im, llx, ny );
    cpl_image_delete( im ); im = NULL;

    im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
    cpl_image_copy( self->pframe->ncombmap, im, llx, ny );
    cpl_image_delete( im ); im = NULL;

    im = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
    cpl_image_copy( self->pframe->rmsmap, im, llx, ny );
    cpl_image_delete( im ); im = NULL;

    sph_double_image_delete( dimage );

       SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
  @brief        Cut first and last columns in the all images of sph_double_image

  @param self   the sph_double_image to be cut

  @return       return sphere error code
  @note         A CPL error is raised in case there was a problem.

 * This function cuts first and last columns in the all images of sph_double_image
 * Self is changed in place.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_double_image_cutedge_columns( sph_double_image* self ){
    sph_double_image*            dimage    = NULL;
    cpl_image*                    im        = NULL;
    int                            llx        = 0;
    int                         lly        = 0;
    int                         urx        = 0;
    int                         ury        = 0;
    int                         nx         = 0;
    int                            ny        = 0;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT );


    dimage = sph_double_image_duplicate( self );
    nx = cpl_image_get_size_x( self->iframe->image );
    nx = nx - 2;
    ny = cpl_image_get_size_y( self->iframe->image );

    //printf("nx=%d, ny=%d \n", nx, ny);
    //delete images from the self double image
    cpl_image_delete(self->iframe->image);
    cpl_image_delete(self->iframe->badpixelmap);
    cpl_image_delete(self->iframe->ncombmap);
    cpl_image_delete(self->iframe->rmsmap);
    cpl_image_delete(self->pframe->image);
    cpl_image_delete(self->pframe->badpixelmap);
    cpl_image_delete(self->pframe->ncombmap);
    cpl_image_delete(self->pframe->rmsmap);


    //re-define the sizes of the images
    self->iframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->iframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->iframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    self->pframe->image = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->badpixelmap = cpl_image_new(nx,ny, CPL_TYPE_INT);
    self->pframe->ncombmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    self->pframe->rmsmap = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);


    llx = 2; lly = 1;
    urx = cpl_image_get_size_x( dimage->iframe->image )-1;
    ury = cpl_image_get_size_y( dimage->iframe->image );

    //iframe
    im = cpl_image_extract( dimage->iframe->image, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->image, im, 1, 1 );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->iframe->badpixelmap, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->badpixelmap, im, 1, 1 );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->iframe->ncombmap, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->ncombmap, im, 1, 1 );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->iframe->rmsmap, llx, lly, urx, ury );
       cpl_image_copy( self->iframe->rmsmap, im, 1, 1 );
       cpl_image_delete( im ); im = NULL;

    //pframe
    im = cpl_image_extract( dimage->pframe->image, llx, lly, urx, ury );
       cpl_image_copy( self->pframe->image, im, 1, 1 );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->pframe->badpixelmap, llx, lly, urx, ury );
       cpl_image_copy( self->pframe->badpixelmap, im, 1, 1 );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->pframe->ncombmap, llx, lly, urx, ury );
       cpl_image_copy( self->pframe->ncombmap, im, 1, 1 );
       cpl_image_delete( im ); im = NULL;

       im = cpl_image_extract( dimage->pframe->rmsmap, llx, lly, urx, ury );
       cpl_image_copy( self->pframe->rmsmap, im, 1, 1 );
       cpl_image_delete( im ); im = NULL;

       sph_double_image_delete( dimage );

       SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;

}



/*----------------------------------------------------------------------------*/
/**
 * @brief        Save the double image
 *
 * @param        self        the double_image to save
 * @param        czFilename  the filename to save under
 * @param        pli         a propertylist to append to main header (can be NULL)
 * @return        the error code
 *
 * Save the double_image as a FITS file. This saves the double_image as a FITS
 * file with 8 extensions, one each for the image, one for the badpixel map, one
 * for the number of pixels map and one for the rmsmap. First come all extension
 * for frame I, then for frame P.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_double_image_save( const sph_double_image* self,
                       const char* czFilename,
                       const cpl_propertylist* pli,
                       unsigned mode )
{
    sph_error_code            rerr       = CPL_ERROR_NONE;
    cpl_propertylist*         mainpl     = NULL;
    cpl_propertylist*         extpl      = NULL;
    cpl_propertylist*         bufpl      = NULL;
    const char*                  addition_l = NULL;
    const char*                  addition_r = NULL;

    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->iframe || !self->pframe ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    extpl = cpl_propertylist_new(); // not really used for the moment; it might be used later



    if ( pli )
    {
        mainpl = sph_keyword_manager_trans_keywords( pli );
    }
    else {
        mainpl = cpl_propertylist_new();
    }


    //add qclist created to the main property list
    if ( self->qclist ) {
        rerr = cpl_propertylist_append( mainpl, self->qclist );
    }

    bufpl =  cpl_propertylist_duplicate(mainpl);
    if ( cpl_propertylist_has(mainpl, SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_LEFT) ) {
        addition_l = cpl_propertylist_get_string(bufpl,SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_LEFT);
        cpl_propertylist_erase(mainpl,SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_LEFT);
    }
    else {
        addition_l = "IFRAME";
    }

    if ( cpl_propertylist_has(mainpl, SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT) ) {
            addition_r = cpl_propertylist_get_string(bufpl,SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT);
            cpl_propertylist_erase(mainpl,SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT);
    }
    else {
            addition_r = "PFRAME";
    }

    /* Update the header if required */
    sph_utils_update_header(mainpl);

    rerr = cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE );
    sph_double_image_update_string_prefix(mainpl, SPH_COMMON_KEYWORD_EXTNAME, addition_l, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_IMAGE_EXTNAME);
    rerr = cpl_image_save( self->iframe->image, czFilename, CPL_TYPE_FLOAT, mainpl, mode );
    sph_double_image_update_string_prefix(extpl, SPH_COMMON_KEYWORD_EXTNAME, addition_l, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_BADPIXMAP_EXTNAME);
    rerr = cpl_image_save( self->iframe->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    sph_double_image_update_string_prefix(extpl, SPH_COMMON_KEYWORD_EXTNAME, addition_l, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_NCOMBMAP_EXTNAME);
    rerr = cpl_image_save( self->iframe->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    sph_double_image_update_string_prefix(extpl, SPH_COMMON_KEYWORD_EXTNAME, addition_l, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_RMSMAP_EXTNAME);
    rerr = cpl_image_save( self->iframe->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );


    sph_double_image_update_string_prefix(extpl, SPH_COMMON_KEYWORD_EXTNAME, addition_r, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_IMAGE_EXTNAME);
    rerr = cpl_image_save( self->pframe->image, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    sph_double_image_update_string_prefix(extpl, SPH_COMMON_KEYWORD_EXTNAME, addition_r, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_BADPIXMAP_EXTNAME);
    rerr = cpl_image_save( self->pframe->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    sph_double_image_update_string_prefix(extpl, SPH_COMMON_KEYWORD_EXTNAME, addition_r, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_NCOMBMAP_EXTNAME);
    rerr = cpl_image_save( self->pframe->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    sph_double_image_update_string_prefix(extpl, SPH_COMMON_KEYWORD_EXTNAME, addition_r, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_RMSMAP_EXTNAME);
    rerr = cpl_image_save( self->pframe->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    if ( mainpl ) {
        cpl_propertylist_delete( mainpl );mainpl = NULL;
    }
    if ( extpl ) {
        cpl_propertylist_delete( extpl );extpl = NULL;
    }
    if ( bufpl ) {
            cpl_propertylist_delete( bufpl );bufpl = NULL;
        }


    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by savin double image.")
        return   rerr;
    }

    return rerr;
}

#ifdef SPH_DOUBLE_SAVE_WITHOUT_QC
static sph_error_code sph_double_image_save_without_qc( sph_double_image* doubleimage,
                                                        const char* czFilename,
                                                       cpl_propertylist* pli )
{
    sph_error_code        rerr              = CPL_ERROR_NONE;
    cpl_propertylist*     pli_regexp        = NULL;

    cpl_ensure_code(doubleimage,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(czFilename,CPL_ERROR_NULL_INPUT);

    pli_regexp = cpl_propertylist_new();
    if ( pli ) {
        cpl_propertylist*     pltemp = cpl_propertylist_new();
        rerr = cpl_propertylist_copy_property_regexp( pltemp, pli, ".*QC.*", 1 );
        rerr = cpl_propertylist_copy_property_regexp( pli_regexp, pltemp, ".*COMMENT.*", 1 );
        cpl_propertylist_delete(pltemp);pltemp = NULL;
    }

    rerr = sph_double_image_save( doubleimage, czFilename, pli_regexp,CPL_IO_CREATE);
    if (rerr != CPL_ERROR_NONE ) {
        SPH_ERR("Couldn't  save the current double image")
    }
    cpl_propertylist_delete(pli_regexp);
    return rerr;
}
#endif

/*----------------------------------------------------------------------------*/
/**
 * @brief        Update property list of the double image DFS product
 *               by SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE
 *
 * @param       czFilename  the filename to load and save double image with
 *              correctly updated SPH_COMMON_KEYWORD_SPH_TYPE
 * @return      the error code
 *
 * @note This function is used as a final step to save double image as a
 * DFS product (see function sph_double_image_save_dfs). It updates
 * SPH_COMMON_KEYWORD_SPH_TYPE by SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE.
 * It is necessary because cpl_dsf_save_image updates the double image property
 * list by the property list from the first input frames.
 *
 */
/*----------------------------------------------------------------------------*/
static
sph_error_code sph_double_image_update_prod_type( const char* czFilename )
{
    sph_error_code        rerr              = CPL_ERROR_NONE;
    sph_double_image*     doubleimage       = NULL;
    cpl_propertylist*     pli               = NULL;
    int                   plane             = 0;

    doubleimage = sph_double_image_load(  czFilename, plane);

    if ( !doubleimage ){
        SPH_ERR("No double image is loaded.")
        return sph_error_get_last_code();
    }

#ifdef SPH_DOUBLE_SAVE_WITHOUT_QC
    pli = cpl_propertylist_load_regexp( czFilename, 0, ".*QC.*", 1);
    rerr = sph_double_image_save_without_qc( doubleimage, czFilename, pli);
#else
    pli = cpl_propertylist_load( czFilename, 0);
    rerr = sph_double_image_save(doubleimage, czFilename, pli, CPL_IO_CREATE );
#endif

    if (rerr != CPL_ERROR_NONE ) {
        SPH_ERR("Couldn't  save the current double image")
    }
    sph_double_image_delete( doubleimage );
    cpl_propertylist_delete(pli);pli = NULL;
    return rerr;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief       Save the double image as DFS product
 *
 * @param       self            the double image to save
 * @param       czFilename      the filename to save under
 * @param       allframes       the frameset used to construct the product
 * @param       usedframes      the frameset used to construct the product
 * @param       template_frame  the frame to use as template for header
 * @param       param           the parameterlist used for product
 * @param       tag             the tag of the product
 * @param       recipe          the recipe creating this product
 * @param       pipename        the pipeline creating this product
 * @param       pl              a propertylist to append to main header (can be NULL)
 * @return      the error code
 *
 * Save the master frame as a FITS file according to ESO DFS.
 * This saves the master frame as a FITS file with 16 extensions (4 master frames ).
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_double_image_save_dfs(
        const sph_double_image* self,
        const char* outfilename,
        cpl_frameset* allframes,
        const cpl_frame* template_frame,
        const cpl_parameterlist* params,
        const char* tag,
        const char* recipe,
        const char* pipename,
        const cpl_propertylist* pli)
{
    sph_error_code            rerr            = CPL_ERROR_NONE;
    cpl_propertylist*         mainpl          = NULL;
    cpl_propertylist*         extpl           = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->iframe || !self->pframe  ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    extpl = cpl_propertylist_new();

    if ( pli ) {
        SPH_INFO_MSG("Property list given as an external parameter, load it.")
        mainpl = sph_keyword_manager_trans_keywords( pli );
    } else if ( self->properties ) {
     SPH_INFO_MSG("Property list already exists , load it.");
     mainpl = sph_keyword_manager_trans_keywords( self->properties );
    } else {
        mainpl = cpl_propertylist_new();
        SPH_INFO_MSG("No property list presents, a new one is created")
    }

    if ( cpl_propertylist_has( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE )){
        rerr = cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE );
    } else {
        cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE );
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    /*
      * Attention: SPH_COMMON_KEYWORD_SPH_TYPE will be changed by cpl_dfs_save_image function.
      * Unfortunately this keyword will be later updated automatically in cpl_dfs_save_image.
      * In order to avoid this  a function, called sph_double_image_update_prod_type is used
      * as soon as the dfs product is created (see below)
      *
      */

     //add qclist created to the main property list
     if ( self->qclist ) {
         rerr = cpl_propertylist_append( mainpl, self->qclist );
         SPH_INFO_MSG("QC parameter list added to the property list")
     }
     SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

     rerr |= cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_PRO_CATG, tag);
     SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

     if ( rerr != CPL_ERROR_NONE ){
         SPH_ERR("Error occured by updating keywords");
         SPH_RAISE_CPL;
         return rerr;
     }

     if (self->iframe->image == NULL) {
         SPH_ERR("self->iframe->image = NULL");
     }

     //printf( "SPH_COMMON_KEYWORD_SPH_TYPE = %s \n", SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_DOUBLE_IMAGE);
     SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

     cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPNAME, SPH_QC_COMPNAME );
     cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPVERN, SPH_QC_COMPVERN );
     cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPREVN, SPH_QC_COMPREVN );
     cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPDATE, SPH_QC_COMPDATE );

     //cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_IMAGE_EXTNAME);

     sph_utils_remove_wcs_3d(mainpl);
     /* FIXME: Set WCS to dummy (pixel) value for now */
     sph_utils_reset_wcs_12d(mainpl);

     /* Update the header if required */
     sph_utils_update_header(mainpl);

     rerr = cpl_dfs_save_image( allframes, NULL, params,
             allframes,
             template_frame,
             self->iframe->image,
             CPL_TYPE_FLOAT, recipe, mainpl, NULL, pipename, outfilename );

     SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;



     if ( rerr == CPL_ERROR_NONE ){
         // iframe
         //rerr |= cpl_image_save( self->iframe->image, outfilename, CPL_TYPE_FLOAT, mainpl, CPL_IO_DEFAULT ); //already saved in cpl_dfs_save_image

         //cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_BADPIXMAP_EXTNAME);
         rerr |= cpl_image_save( self->iframe->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );

         //cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_NCOMBMAP_EXTNAME);
         rerr |= cpl_image_save( self->iframe->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

         //cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_IFRAME_RMSMAP_EXTNAME);
         rerr |= cpl_image_save( self->iframe->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
         //pframe
         //cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_IMAGE_EXTNAME);
         rerr |= cpl_image_save( self->pframe->image, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

         //cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_BADPIXMAP_EXTNAME);
         rerr |= cpl_image_save( self->pframe->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );

         //cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_NCOMBMAP_EXTNAME);
         rerr |= cpl_image_save( self->pframe->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

         //cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_DOUBLE_IMAGE_PFRAME_RMSMAP_EXTNAME);
         rerr |= cpl_image_save( self->pframe->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
     } else {
         SPH_ERR("Could not save first image of the double image (self->iframe->image) using cpl_dfs_save_image");
     }

     if ( mainpl ) {
           cpl_propertylist_delete( mainpl );mainpl = NULL;
     }
     if ( extpl ) {
           cpl_propertylist_delete( extpl );extpl = NULL;
     }

     // Attention: Update of the created dfs double image product by the
     // correct SPH_COMMON_KEYWORD_SPH_TYPE  keyword (dirty, dirty...)
     //SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
     rerr |= sph_double_image_update_prod_type( outfilename );
     SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

     SPH_RAISE_CPL;
     return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief load a double image from a file
 *
 * @param    filename        the filename to load from
 * @param     plane            the plane to laod from ( 0..Nplanes)
 *
 * This function creates a new double image and loads the data from a FITS
 * file with at least one plane and at least 8 extensions (the first eight
 * extensions are read). This function does not check the format is actually
 * a sph_double_image.
 */
/*----------------------------------------------------------------------------*/
sph_double_image*
sph_double_image_load( const char* czFilename, int plane ) {
    sph_double_image*        result        = NULL;
    cpl_propertylist*        pl_eso        = NULL;


    result = sph_double_image_new_empty();

    if ( !result ) {
        return NULL;
    }

    pl_eso = cpl_propertylist_load_regexp( czFilename, 0, ".*ESO.*", 0 );

    if ( pl_eso != NULL ){
        cpl_error_code rerr = CPL_ERROR_NONE;
        //initializing and loading properties, qclist for the newly created quad image
        result->properties = cpl_propertylist_new();
        rerr = cpl_propertylist_copy_property_regexp( result->properties, pl_eso, ".*QC.*", 1 );
        if ( rerr != CPL_ERROR_NONE){
            sph_error_raise( SPH_DOUBLE_IMAGE_GENERAL,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_WARNING,  "Copy regexp properties for the double image is failed" );
            rerr = CPL_ERROR_NONE;
            cpl_error_reset();
            cpl_propertylist_delete( result->properties ); result->properties = NULL;
        }
        result->qclist = cpl_propertylist_new();
        rerr = cpl_propertylist_copy_property_regexp( result->qclist, pl_eso, ".*QC.*", 0 );
        if ( rerr != CPL_ERROR_NONE){
            sph_error_raise( SPH_DOUBLE_IMAGE_GENERAL,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_WARNING,  "Copy regexp qclist for the double image is failed" );
            cpl_error_reset();
            cpl_propertylist_delete( result->qclist ); result->qclist = NULL;
        }
        cpl_propertylist_delete( pl_eso );
    } else {
        SPH_RAISE_CPL;
    }

    result->iframe = sph_master_frame_new_empty();
    result->pframe = sph_master_frame_new_empty();
    if ( !result->iframe || !result->pframe ) {
        sph_double_image_delete( result );
        return NULL;
    }
    result->iframe->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 0 );
    if ( !result->iframe->image ) {
        sph_double_image_delete( result );
        return NULL;
    }
    result->iframe->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 1 );
    if ( !result->iframe->badpixelmap ) {
        sph_double_image_delete( result );
        return NULL;
    }
    result->iframe->ncombmap = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 2 );
    if ( !result->iframe->ncombmap ) {
        sph_double_image_delete( result );
        return NULL;
    }
    result->iframe->rmsmap = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 3 );
    if ( !result->iframe->rmsmap ) {
        sph_double_image_delete( result );
        return NULL;
    }
    result->pframe->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 4 );
    if ( !result->pframe->image ) {
        sph_double_image_delete( result );
        return NULL;
    }
    result->pframe->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 5 );
    if ( !result->pframe->badpixelmap ) {
        sph_double_image_delete( result );
        return NULL;
    }
    result->pframe->ncombmap = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 6 );
    if ( !result->pframe->ncombmap ) {
        sph_double_image_delete( result );
        return NULL;
    }
    result->pframe->rmsmap = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 7 );
    if ( !result->pframe->rmsmap ) {
        sph_double_image_delete( result );
        return NULL;
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    delete the sph_double image and release resources
  @param    self The double image to delete
  @return   void
 */
/*----------------------------------------------------------------------------*/
void
sph_double_image_delete( sph_double_image* self ) {
    if ( self ) {
        sph_master_frame_delete( self->iframe );
        sph_master_frame_delete( self->pframe );
        cpl_propertylist_delete( self->properties );
        cpl_propertylist_delete ( self->qclist );
        cpl_free( self );
    }
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief    Change the named property to the prefixed value
  @param    self    The propertylist to modify
  @param    keyname The name of the property to modify
  @param    prefix  The prefix
  @param    value   The value to prefix
  @return   CPL_ERROR_NONE on success or else the relevant CPL error code

 */
/*----------------------------------------------------------------------------*/
static sph_error_code
sph_double_image_update_string_prefix(cpl_propertylist* self,
                                      const char* keyname,
                                      const char* prefix,
                                      const char* value )
{
    char* string;
    cpl_error_code code;

    /* cpl_sprintf() cannot detect non-format NULL arguments */
    cpl_ensure_code(prefix != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(value  != NULL, CPL_ERROR_NULL_INPUT);

    string = cpl_sprintf("%s.%s", prefix, value);

    code = cpl_propertylist_update_string(self, keyname, string);
    cpl_free(string);

    return code ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/**@}*/

