/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifndef SPH_DOUBLE_IMAGE_H_
#define SPH_DOUBLE_IMAGE_H_
#include "sph_master_frame.h"
#include "sph_zpl_exposure_imaging.h"
#include "sph_error.h"

typedef struct _sph_double_image_
{
	sph_master_frame* 		iframe;
	sph_master_frame*		pframe;
	cpl_propertylist*		properties;
	cpl_propertylist*		qclist;
} sph_double_image;

/*QC Names provider*/
typedef struct{
	const char * mean_name;
	const char * median_name;
	const char * rms_name;
	const char * bad_pixels_name;
}sph_double_image_qc_names;

extern const char* SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_LEFT;
extern const char* SPH_DOUBLE_IMAGE_EXTNAME_ADDITION_RIGHT;

/*-----------------------------------------------------------------------------
 Function prototypes
 -----------------------------------------------------------------------------*/

sph_double_image* sph_double_image_new( int nx, int ny )
    CPL_ATTR_ALLOC;
sph_double_image* sph_double_image_new_empty( void )
    CPL_ATTR_ALLOC;
sph_double_image* sph_double_image_new_from_master_frames( sph_master_frame* mf_odd, sph_master_frame* mf_even)
    CPL_ATTR_ALLOC;
sph_double_image* sph_double_image_new_from_zpl_exposure_imaging( sph_zpl_exposure_imaging* zplexpimg )
    CPL_ATTR_ALLOC;

sph_double_image* sph_double_image_load( const char* czFilename, int plane )
    CPL_ATTR_ALLOC;
sph_double_image* sph_double_image_duplicate ( const sph_double_image* self)
    CPL_ATTR_ALLOC;
sph_error_code    sph_double_image_add_double_image( sph_double_image* self,
                                                     const sph_double_image* diframe);
sph_error_code    sph_double_image_subtract_double_image( sph_double_image* self,
                                                          const sph_double_image* diframe);
sph_error_code    sph_double_image_multiply_double(sph_double_image* self, double value);
sph_error_code    sph_double_image_divide_double_image( sph_double_image* self,
                                                        const sph_double_image* dimage);
sph_error_code    sph_double_image_divide_double( sph_double_image* self, double value);
sph_error_code    sph_double_image_combine_double_image (sph_double_image* self,
                                                         const sph_double_image* diframe_minus);

cpl_error_code    sph_double_image_quality_check( sph_double_image* self );
cpl_error_code    sph_double_image_quality_check_names(sph_double_image* self ,
											  const sph_double_image_qc_names * iframe_names,
											  const sph_double_image_qc_names * pframe_names);
cpl_error_code    sph_double_image_quality_check_keysuffix( sph_double_image* self,
                                                            const char* keysuffix );

cpl_error_code 	  sph_double_image_mask_sigma( sph_double_image* self, double sigma);

sph_master_frame* sph_double_image_get_iframe_master_frame (const sph_double_image* self)
    CPL_ATTR_ALLOC;
sph_master_frame* sph_double_image_get_pframe_master_frame (const sph_double_image* self)
    CPL_ATTR_ALLOC;
sph_master_frame* sph_double_image_get_iframe_master_frame_duplicate (const sph_double_image* self)
    CPL_ATTR_ALLOC;
sph_master_frame* sph_double_image_get_pframe_master_frame_duplicate (const sph_double_image* self)
    CPL_ATTR_ALLOC;
sph_master_frame*
sph_double_image_calculate_pframe_slash_iframe_master_frame( const sph_double_image* self)
    CPL_ATTR_ALLOC;

cpl_frameset*
sph_double_image_create_double_image_frameset_from_zplexp_imaging_cubes( const cpl_frameset* inframes )
    CPL_ATTR_ALLOC;
cpl_frameset*
sph_double_image_create_double_image_frameset_from_zplexp_imaging_frameset( const cpl_frameset* inframes )
    CPL_ATTR_ALLOC;
sph_error_code    sph_double_image_expand2y( sph_double_image* self );
sph_error_code    sph_double_image_create_squared( sph_double_image* self );
sph_error_code    sph_double_image_interpolate_y( sph_double_image* self, cpl_boolean set_badpxl );
sph_error_code    sph_double_image_interpolate_y_zpl_imaging( sph_double_image* self, cpl_boolean set_badpxl );
sph_error_code    sph_double_image_interpolate_y_zpl_polarimetry( sph_double_image* self, cpl_boolean set_badpxl );
cpl_boolean       sph_double_image_is_squared( const sph_double_image* self );
sph_error_code    sph_double_image_add_topbottom_lines( sph_double_image* self );
sph_error_code    sph_double_image_cutedge_columns( sph_double_image* self );

sph_error_code
sph_double_image_save_dfs(
        const sph_double_image* self,
        const char* outfilename,
        cpl_frameset* allframes,
        const cpl_frame* template_frame,
        const cpl_parameterlist* params,
        const char* tag,
        const char* recipe,
        const char* pipename,
        const cpl_propertylist* pli);

sph_error_code sph_double_image_save( const sph_double_image* self,
                                      const char* czFilename,
                                      const cpl_propertylist* pli,
                                      unsigned mode );
void sph_double_image_delete( sph_double_image* self );

#endif /* SPH_DOUBLE_IMAGE_H_ */
