/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_extract_angles.h"
#include "sph_time.h"
#include "sph_fits.h"
#include "sph_skycalc.h"
#include "sph_coord.h"
#include "sph_keyword_manager.h"
#include "sph_common_keywords.h"


/*----------------------------------------------------------------------------*/
/**
 * @brief Extract the angles from a raw data cube
 * @param self     Frame from which to extract the angles
 * @param offsetA  First  angle to add to the result
 * @param offsetB  Second angle to add to the result
 * @return error code of the operation
 *
 * This function extracts parallactic angles for every plane in an input data cube.
 * The angles are returned as doubles and defined in the same way as for
 * skycalc (see the corresponding skycalc documention in sph_skycalc).
 *
 * The frame containing the cube is passed as input must contain the keywords
 * for RA, DEC MJD-OBS and UTC.
 */
/*----------------------------------------------------------------------------*/
cpl_vector* sph_extract_angles_from_cube(const cpl_frame* self,
                                         double offsetA,
                                         double offsetB)
{
    cpl_vector*        result;
    cpl_size           nplanes = 0;
    cpl_vector*        timev = NULL;
    sph_skycalc_input  skycal_inp[7];
    sph_skycalc_output skycal_out;
    cpl_propertylist*  pl = NULL;
    cpl_error_code     code;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);

    nplanes = (cpl_size)sph_fits_get_nplanes( cpl_frame_get_filename(self), 0);
    cpl_ensure(nplanes > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    timev = sph_time_extract_times_from_frame( self );
    cpl_ensure(timev != NULL, CPL_ERROR_ILLEGAL_INPUT, NULL);

    pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(self), 0 );

    for (size_t i = 0; i < 7; ++i) {
        skycal_inp[i].command = ' ';
        snprintf(skycal_inp[i].param_string, 80, " ");
    }

    code = sph_coord_initialise_skycalc(pl, skycal_inp,
                                        cpl_vector_get(timev, 0),
                                        &skycal_out);

    cpl_propertylist_delete(pl);
    result = cpl_vector_new(nplanes);

    for (cpl_size ii = 0; code == CPL_ERROR_NONE && ii < nplanes; ++ii) {
        const double time = cpl_vector_get(timev, ii);
        code = sph_coord_do_skycalc(skycal_inp, time, &skycal_out);
        cpl_vector_set(result, ii, skycal_out.pa + offsetA + offsetB);
    }

    cpl_vector_delete(timev);

    if (code) {
        (void)cpl_error_set_where(cpl_func);
        cpl_vector_delete(result);
        result = NULL;
    }

    return result;
}
