/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include <complex.h>
#include "sph_fft.h"
#include "sph_error.h"
#include "sph_common_keywords.h"

#include <math.h>
#include <string.h>
#include <fftw3.h>
#include <gsl/gsl_errno.h>
#include <assert.h>

static cpl_image*
sph_fftw_rotate_image(sph_fft* self,const cpl_image* image,double degrees)
    CPL_ATTR_ALLOC;

static double sph_fft_rotate_quart(cpl_image*, double) CPL_ATTR_NONNULL;

static void sph_fft_shift_fftw_array( const sph_fft* self, fftw_complex* in,
                                      double dx, double dy ) CPL_ATTR_NONNULL;

static void
sph_fft_shift_fftw_triplet(const sph_fft*, double complex*, double complex*,
                           double complex*, double, double) CPL_ATTR_NONNULL;

static cpl_image* sph_fft_smoothe_create(cpl_size, cpl_size, double)
    CPL_ATTR_ALLOC;

static
sph_error_code
sph_fft_forward_row( sph_fft* self, int row ) {
    int    nx = self->nx;
    if ( self->method == SPH_FFT_GSL_RADIX2 ) {
        gsl_fft_complex_radix2_forward( &self->complexdata[ 2 * row * nx ], 1, nx );
    }
    else if ( self->method == SPH_FFT_GSL_MIXEDRADIX ) {
        gsl_fft_complex_forward( &self->complexdata[ 2 * row * nx ], 1, nx, self->wavetab, self->workspace );
    }
    return CPL_ERROR_NONE;
}
static
sph_error_code
sph_fft_inverse_row( sph_fft* self, int row ) {
    int    nx = self->nx;
    if ( self->method == SPH_FFT_GSL_RADIX2 ) {
        gsl_fft_complex_radix2_inverse( &self->complexdata[ 2 * row * nx ], 1, nx );
    }
    else if ( self->method == SPH_FFT_GSL_MIXEDRADIX ) {
        gsl_fft_complex_inverse( &self->complexdata[ 2 * row * nx ], 1, nx, self->wavetab, self->workspace );
    }
    return CPL_ERROR_NONE;
}
static
sph_error_code
sph_fft_forward_col( sph_fft* self, int col ) {
    int    nx = self->nx;
    int ny = self->ny;
    int    errcode = 0;
    if ( self->method == SPH_FFT_GSL_RADIX2 ) {
        gsl_fft_complex_radix2_forward( &self->complexdata[ 2 * col ], nx, ny );
    }
    else if ( self->method == SPH_FFT_GSL_MIXEDRADIX ) {
        if ( ( errcode = gsl_fft_complex_forward( &self->complexdata[ 2 * col ], nx, ny, self->wavetab, self->workspace )) != 0 ) {
            sph_error_raise(SPH_ERROR_GENERAL,__FILE__,__func__,__LINE__,SPH_ERROR_ERROR,"Error from GSL: %s", gsl_strerror(errcode));
        }
    }
    return CPL_ERROR_NONE;
}
static
sph_error_code
sph_fft_inverse_col( sph_fft* self, int col ) {
    int    nx = self->nx;
    int ny = self->ny;
    if ( self->method == SPH_FFT_GSL_RADIX2 ) {
        gsl_fft_complex_radix2_inverse( &self->complexdata[ 2 * col ], nx, ny );
    }
    else if (self->method == SPH_FFT_GSL_MIXEDRADIX ) {
        gsl_fft_complex_inverse( &self->complexdata[ 2 * col ], nx, ny, self->wavetab, self->workspace );
    }
    return CPL_ERROR_NONE;
}

static
sph_error_code
sph_fft_shift_row( sph_fft* self, int row, double delta ) {
    const double y0 = ( self->ny - 1 ) / 2;
    const double x0 = ( self->nx - 1 ) / 2;

    for (int xx = 0; xx < self->nx; ++xx) {
        const double shift  = delta * ( row - y0 - 0.5 ) + self->shiftx;
        const double kval   = xx <= x0 ? xx : xx - self->nx;
        const double dshift = (double)shift * ( kval / (double)(self->nx) );
        gsl_complex mulfac  = gsl_complex_polar(1.0, CPL_MATH_2PI * dshift);
        gsl_complex* valp   =
            (gsl_complex*)&self->complexdata[ 2 * ( row * self->nx + xx ) ];
        gsl_complex mulresult = gsl_complex_mul(*valp, mulfac);

        GSL_REAL(*valp) = GSL_REAL(mulresult);
        GSL_IMAG(*valp) = GSL_IMAG(mulresult);
    }
    return CPL_ERROR_NONE;
}

static
sph_error_code
sph_fft_filter_row( sph_fft* self, int row, sph_fft_filter_method method, double delta, double param1, double param2 ) {
    int    nx = self->nx;
    int    ny = self->ny;
    int     xx    = 0;
    double    y0 =0.0;
    double    x0    = 0.0;
    double kvaly = 0.0;
    double kvalx = 0.0;
    double dshift = 0.0;
    double  r = 0.0;
    double factor = 0.0;
    double order = 0.0;
    gsl_complex*        valp    = NULL;

    y0 = ( self->ny - 1 ) / 2;
    x0 = ( self->nx - 1 ) / 2;
    if ( row <= y0 ) {
        kvaly = row;
    }
    else {
        kvaly = row - ny;
    }
    for (xx = 0; xx < nx; ++xx) {
        if ( xx <= x0 ) {
            kvalx = xx;
        }
        else {
            kvalx = xx - nx;
        }
        dshift =  kvalx / (double)nx * kvalx / (double)nx +
                    kvaly / (double)ny * kvaly / (double)ny;
        valp = (gsl_complex*)&self->complexdata[ 2 * ( row * nx + xx ) ];

        if ( method == SPH_FFT_FILTER_METHOD_TH ) {
            if ( 4.0 * dshift > 1.0 - delta * delta ) {
                GSL_REAL(*valp) = 0.0;//exp( -dshift/(delta*delta));
                GSL_IMAG(*valp) = 0.0;//exp( -dshift/(delta*delta));
            }
        }
        if ( method == SPH_FFT_FILTER_METHOD_FERMI ) {
            r = 1.0 - delta * delta;
            factor = 1.0 / ( 1 + exp( (4 * dshift - r) / param1 ) );
            GSL_REAL(*valp) = GSL_REAL(*valp) * factor;
            GSL_IMAG(*valp) = GSL_IMAG(*valp) * factor;
        }
        if ( method == SPH_FFT_FILTER_METHOD_BUTTER ) {
            order = 2 * log( 0.882 / sqrt(10.624 * 10.624 - 1.0) ) / log( param1 / param2 );
            r = param1 / pow(0.882, 2.0 / order);
            factor = 1.0 / sqrt( 1 + pow(4 * dshift / r, order ) );
            GSL_REAL(*valp) = GSL_REAL(*valp) * factor;
            GSL_IMAG(*valp) = GSL_IMAG(*valp) * factor;
        }
    }
    return CPL_ERROR_NONE;
}
static
sph_error_code
sph_fft_shift_col( sph_fft* self, int col, double delta ) {
    int    nx = self->nx;
    int ny = self->ny;
    int     yy    = 0;
    gsl_complex            mulfac;
    gsl_complex            mulresult;
    gsl_complex*        valp    = NULL;
    double    x0;
    double y0;
    double    shift    = 0;
    double kval = 0.0;
    double dshift = 0.0;

    x0 = ( self->nx - 1 ) / 2;
    y0 = ( self->ny - 1 ) / 2;
    for (yy = 0; yy < ny; ++yy) {
        shift = delta * ( col - x0 - 0.5 ) + self->shifty;
        if ( yy <= y0 ) {
            kval = yy;
        }
        else {
            kval = yy - ny;
        }
        dshift = (double)shift * ( kval / (double)ny );
        mulfac = gsl_complex_polar( 1.0, CPL_MATH_2PI * dshift  );
        valp = (gsl_complex*)&self->complexdata[ 2 * ( yy * nx + col ) ];
        mulresult = gsl_complex_mul( *valp, mulfac );
        GSL_REAL(*valp) = GSL_REAL(mulresult);
        GSL_IMAG(*valp) = GSL_IMAG(mulresult);
    }
    return CPL_ERROR_NONE;
}

static sph_error_code sph_fft_prepare_gsl_radix2(sph_fft* self,
                                                 const cpl_image* im,
                                                 int newx,
                                                 int newy)
{
    const int  N      = cpl_image_get_size_x(im);
    cpl_image* workim = cpl_image_new( newx, newy, cpl_image_get_type(im) );

    if ( !workim ) return SPH_ERROR_GENERAL;
    cpl_free(self->complexdata);
    self->complexdata = cpl_calloc(2 * newx * newy, sizeof(double) );

    if ( !self->complexdata ) return SPH_ERROR_GENERAL;

    SPH_RAISE_CPL_RESET;
    cpl_image_copy( workim, im, ( newx - N ) / 2 + 1, ( newy - N ) / 2 + 1 );
    SPH_RAISE_CPL_RESET;

    for (int xx = 0; xx < newx; ++xx) {
        for (int yy = 0; yy < newy; ++yy) {
            int   bpix   = 0;
            self->complexdata[2 * (yy * newx + xx) ]
                = cpl_image_get( workim, xx + 1, yy + 1, &bpix );
        }
    }

    self->nx = newx;
    self->ny = newy;

    cpl_image_delete(workim);
    return CPL_ERROR_NONE;
}
static
void
sph_fft_alloc_mixedradix(sph_fft* self) {
    if ( self->wavetab ) gsl_fft_complex_wavetable_free(self->wavetab);
    if ( self->workspace ) gsl_fft_complex_workspace_free(self->workspace);
    self->wavetab = gsl_fft_complex_wavetable_alloc( self->nx );
    self->workspace = gsl_fft_complex_workspace_alloc( self->nx );
}

static
sph_error_code
sph_fft_prepare_gsl_mixed_radix( sph_fft* self, const cpl_image* im ) {
    int                    bpix = 0;
    int                    N    = 0;
    int                    xx = 0;
    int                    yy = 0;

    N = cpl_image_get_size_x(im);
    if ( cpl_image_get_size_y(im) != N ) {
        SPH_ERR("Not the same size in x and y of image in FFT.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    if ( self->complexdata ) {
        cpl_free(self->complexdata);self->complexdata = NULL;
    }
    self->complexdata = cpl_calloc(2 * N * N, sizeof(double) );

    if ( !self->complexdata ) return SPH_ERROR_GENERAL;
    for (yy = 0; yy < N; ++yy) {
        for (xx = 0; xx < N; ++xx) {
            self->complexdata[2 * (yy * N + xx) ] = cpl_image_get( im, xx + 1, yy + 1, &bpix );
        }
    }
    self->nx = N;
    self->ny = N;
    sph_fft_alloc_mixedradix(self);
    return CPL_ERROR_NONE;
}
sph_fft*
sph_fft_new( sph_fft_method method ) {
    sph_fft*     result = cpl_calloc( 1, sizeof(sph_fft) );
    result->method = method;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Perform a skew only
 * @param self
 * @param im the input image
 * @param degree    the angle of the skew in degrees
 *
 * @return error code of the operation
 *
 * This performs a skew in x ( rows only shifted). The operation
 * corresponds to
 * [ 1    - tan (angle / 2 ) ]     *  [ x ]
 * [ 0    1                  ]     *  [ y ]
 *
 * Use the function sph_fft_rotate_complex2image to return the image.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fft_skewx( sph_fft* self, const cpl_image* im, double degrees ) {
    int                    N        = 0;
    double                angle    = degrees * CPL_MATH_RAD_DEG;
    double                tantananglehalf    = tan( angle / 2.0 ) * tan( angle / 2.0 );
    int                    newx    = 2;
    int                    newy    = 2;
    int                    xx        = 0;
    int                    yy        = 0;
    int                    bpix    = 0;
    double                delta    = 0.0;
    cpl_image*            workim    = NULL;
    sph_error_code        rerr    = CPL_ERROR_NONE;


    if ( !im ) return SPH_ERROR_GENERAL;
    if ( degrees > 90.0 || degrees < -90.0 ) {
        return SPH_ERROR_GENERAL;
    }
    if ( cpl_image_get_size_x(im) != cpl_image_get_size_y(im) ) {
        return SPH_ERROR_GENERAL;
    }
    N = cpl_image_get_size_x( im );
    self->angle    = angle;
    while ( newx <= N + N * tantananglehalf ) {
        newx = newx * 2;
    }
    newy = newx;
    if ( newx > 2048 * 4 || newy > 2048 * 4 ) {
        return SPH_ERROR_GENERAL; // this is too big to handle
    }

    workim = cpl_image_new( newx, newy, CPL_TYPE_DOUBLE );
    if ( !workim ) return SPH_ERROR_GENERAL;
    self->complexdata = cpl_calloc(2 * newx * newy, sizeof(double) );
    if ( !self->complexdata ) return SPH_ERROR_GENERAL;

    rerr = cpl_image_copy( workim, im, ( newx - N ) / 2 + 1, ( newy - N ) / 2 + 1);

    for (yy = 0; yy < newy; ++yy) {
        for (xx = 0; xx < newx; ++xx) {
            self->complexdata[2 * (yy * newx + xx) ] = cpl_image_get( workim, xx + 1, yy + 1, &bpix );
        }
    }
    self->Npp = newy * 2;
    self->Np = newx;
    self->nx = newx;
    self->ny = newy;

    delta = -tan(angle/2.0);
    for (yy = 0; yy < newy; ++yy) {
        sph_fft_forward_row( self, yy );
        sph_fft_shift_row( self, yy, delta );
        sph_fft_inverse_row( self, yy );
    }
    cpl_image_delete(workim); workim = NULL;
    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Perform a skew only
 * @param self
 * @param im the input image
 * @param degree    the angle of the skew in degrees
 *
 * @return error code of the operation
 *
 * This performs a skew in y ( columns only shifted). The operation
 * corresponds to
 * [ 1                 0 ]     *  [ x ]
 * [ -tan(angle/2)    1 ]     *  [ y ]
 *
 * Use the function sph_fft_rotate_complex2image to return the image.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fft_skewy( sph_fft* self, const cpl_image* im, double degrees ) {
    int                    N        = 0;
    double                angle    = degrees * CPL_MATH_RAD_DEG;
    double                tantananglehalf    = tan( angle / 2.0 ) * tan( angle / 2.0 );
    int                    newx    = 2;
    int                    newy    = 2;
    int                    xx        = 0;
    int                    yy        = 0;
    int                    bpix    = 0;
    double                delta    = 0.0;
    cpl_image*            workim    = NULL;
    sph_error_code        rerr    = CPL_ERROR_NONE;


    if ( !im ) return SPH_ERROR_GENERAL;
    if ( degrees > 90.0 || degrees < -90.0 ) {
        return SPH_ERROR_GENERAL;
    }
    if ( cpl_image_get_size_x(im) != cpl_image_get_size_y(im) ) {
        return SPH_ERROR_GENERAL;
    }
    N = cpl_image_get_size_x( im );
    self->angle    = angle;
    while ( newx <= N + N * tantananglehalf ) {
        newx = newx * 2;
    }
    newy = newx;
    if ( newx > 2048 * 4 || newy > 2048 * 4 ) {
        return SPH_ERROR_GENERAL; // this is too big to handle
    }

    workim = cpl_image_new( newx, newy, CPL_TYPE_DOUBLE );
    if ( !workim ) return SPH_ERROR_GENERAL;
    self->complexdata = cpl_calloc(2 * newx * newy, sizeof(double) );
    if ( !self->complexdata ) return SPH_ERROR_GENERAL;

    rerr = cpl_image_copy( workim, im, ( newx - N ) / 2 + 1, ( newy - N ) / 2 + 1 );
    //rerr = cpl_image_copy( workim, im, 1, 1 );

    for (yy = 0; yy < newy; ++yy) {
        for (xx = 0; xx < newx; ++xx) {
            self->complexdata[2 * (yy * newx + xx) ] = cpl_image_get( workim, xx + 1, yy + 1, &bpix );
        }
    }
    self->Npp = newy * 2;
    self->Np = newx;
    self->nx = newx;
    self->ny = newy;

    delta = -tan(angle/2.0);
    for (xx = 0; xx < newx; ++xx) {
        sph_fft_forward_col( self, xx );
        sph_fft_shift_col( self, xx, delta );
        sph_fft_inverse_col( self, xx );
    }
    cpl_image_delete(workim); workim = NULL;
    return rerr;
}
sph_error_code
sph_fft_set_complex(sph_fft* self, int kx ,int ky, double real, double imag) {
    int xx = 0;
    int yy = 0;
    gsl_complex*        valp    = NULL;


    if ( kx < 0 ) {
        xx = self->nx + kx;
    }
    else {
        xx = kx;
    }
    if ( ky < 0 ) {
        yy = self->ny + ky;
    }
    else {
        yy = ky;
    }
    if ( xx >= 0 && yy >=0 && yy < self->ny && xx < self->nx ) {
        valp = (gsl_complex*)&self->complexdata[ 2 * ( yy * self->nx + xx ) ];
        GSL_REAL(*valp) = real;
        GSL_IMAG(*valp) = imag;
    }
    else return CPL_ERROR_ILLEGAL_INPUT;

    return CPL_ERROR_NONE;

}
gsl_complex
sph_fft_get_complex(sph_fft* self, int kx ,int ky ) {
    int xx = 0;
    int yy = 0;
    gsl_complex*        valp    = NULL;
    gsl_complex result;

    GSL_REAL(result)=0.0;
    GSL_IMAG(result)=0.0;

    if ( kx < 0 ) {
        xx = self->nx + kx;
    }
    else {
        xx = kx;
    }
    if ( ky < 0 ) {
        yy = self->ny + ky;
    }
    else {
        yy = ky;
    }
    if ( xx >= 0 && yy >=0 && yy < self->ny && xx < self->nx ) {
        valp = (gsl_complex*)&self->complexdata[ 2 * ( yy * self->nx + xx ) ];
        GSL_REAL(result) = GSL_REAL(*valp);
        GSL_IMAG(result) = GSL_IMAG(*valp);
    }
    else {
        SPH_ERR("Access out of bounds.");
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Perform a forward FFT
 * @param self
 *
 * @return error code of the operation
 *
 * This performs a forward FFT.
 * The sph_fft object must have been prepared using
 * one of the the sph_fft_prepare function. The resulting image
 * can be obtained with the sph_fft_complex2image functions.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fft_forward( sph_fft* self ) {
    int        kk        = 0;


    for (kk = 0; kk < self->ny; ++kk) {
        sph_fft_forward_row( self, kk );
    }
    for (kk = 0; kk < self->nx; ++kk) {
        sph_fft_forward_col( self, kk );
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Perform an inverse FFT
 * @param self
 *
 * @return error code of the operation
 *
 * This performs an inverse FFT.
 * The sph_fft object must have been prepared using
 * one of the the sph_fft_prepare function. The resulting image
 * can be obtained with the sph_fft_complex2image functions.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fft_inverse( sph_fft* self ) {
    int        kk        = 0;


    for (kk = 0; kk < self->ny; ++kk) {
        sph_fft_inverse_col( self, kk );
    }
    for (kk = 0; kk < self->nx; ++kk) {
        sph_fft_inverse_row( self, kk );
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
  @internal
  @brief Perform a shift of three images, each in the Fourier space
  @param self  The FFT object ?
  @param in1   The first  image to shift in place
  @param in2   The second image to shift in place
  @param in3   The third  image to shift in place
  @param dx    The shift in X
  @param dy    The shift in Y
  @return CPL_ERROR_NONE on success or the relevant CPL error code
  @note Due to the Real-to-Complex transform do only the leftmost N/2+1 columns
  @see cpl_fft_image()

 */
/*----------------------------------------------------------------------------*/

static void
sph_fft_shift_fftw_triplet(const sph_fft* self,
                           double complex* in1,
                           double complex* in2,
                           double complex* in3,
                           double dx,
                           double dy) {

    const size_t N = (size_t)self->nx;
    const size_t nxh = N/2 + 1; /* With R2C & C2R: only half the columns */

    /* FIXME: Does odd N need to be supported ? */

    for (size_t yy = 0; yy < N; yy++) {
        const double ky = yy < (N+1)/2 ? (double)yy : (double)yy - (double)N;
        for (size_t xx = 0; xx < nxh; xx++) {
            const double kx = xx < (N+1)/2 ? (double)xx : (double)xx - (double)N;

            const double shift_xy = -dx * kx - dy * ky;

            const double complex scale
                = cexp((CPL_MATH_2PI * shift_xy /(double)N) * I)
                / (double)(N * N); /* Apply also the FFT scaling */

            in1[yy * nxh + xx] *= scale;
            in2[yy * nxh + xx] *= scale;
            in3[yy * nxh + xx] *= scale;
        }
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief Shift a triplet of square, double-type cpl_images with no BPM using FFT

  @param self  The FFT object ?
  @param img1  The first  image to shift in place
  @param img2  The second image to shift in place
  @param img3  The third  image to shift in place
  @param dx    The shift in X
  @param dy    The shift in Y
 
  @return CPL_ERROR_NONE on success or the relevant CPL error code
  @see cpl_fft_image()

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_fft_shift_triplet(const sph_fft* self,
                                     cpl_image* img1,
                                     cpl_image* img2,
                                     cpl_image* img3,
                                     double dx,
                                     double dy) {

    const size_t    N  = (size_t)cpl_image_get_size_x(img1);
    const size_t    nxh = N/2 + 1; /* With R2C & C2R: only half the columns */
    double complex* in1;
    double complex* in2;
    double complex* in3;
    cpl_image*    fftimg1;
    cpl_image*    fftimg2;
    cpl_image*    fftimg3;
    cpl_imagelist* fftlist;
    cpl_imagelist* imglist;

    cpl_msg_info(cpl_func, "SHIFT(%u): x=%g. y=%g", (unsigned)N, dx, dy);

    cpl_ensure_code((size_t)cpl_image_get_size_y(img1) == N,
                    CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code((size_t)cpl_image_get_size_x(img2) == N,
                    CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code((size_t)cpl_image_get_size_y(img2) == N,
                    CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code((size_t)cpl_image_get_size_x(img3) == N,
                    CPL_ERROR_INCOMPATIBLE_INPUT);
    cpl_ensure_code((size_t)cpl_image_get_size_y(img3) == N,
                    CPL_ERROR_ILLEGAL_INPUT);

    cpl_ensure_code(cpl_image_get_type(img1) == CPL_TYPE_DOUBLE,
                    CPL_ERROR_TYPE_MISMATCH);
    cpl_ensure_code(cpl_image_get_type(img2) == CPL_TYPE_DOUBLE,
                    CPL_ERROR_TYPE_MISMATCH);
    cpl_ensure_code(cpl_image_get_type(img3) == CPL_TYPE_DOUBLE,
                    CPL_ERROR_TYPE_MISMATCH);

    assert( sizeof(double complex) == sizeof(fftw_complex));

    fftimg1 = cpl_image_wrap_double_complex(nxh, N,
                                            cpl_malloc(N * nxh * sizeof(*in1)));
    fftimg2 = cpl_image_wrap_double_complex(nxh, N,
                                            cpl_malloc(N * nxh * sizeof(*in2)));
    fftimg3 = cpl_image_wrap_double_complex(nxh, N,
                                            cpl_malloc(N * nxh * sizeof(*in3)));
    imglist = cpl_imagelist_new();
    cpl_imagelist_set(imglist, img1, 0);
    cpl_imagelist_set(imglist, img2, 1);
    cpl_imagelist_set(imglist, img3, 2);

    fftlist = cpl_imagelist_new();
    cpl_imagelist_set(fftlist, fftimg1, 0);
    cpl_imagelist_set(fftlist, fftimg2, 1);
    cpl_imagelist_set(fftlist, fftimg3, 2);

    in1 = cpl_image_get_data_double_complex(fftimg1);
    in2 = cpl_image_get_data_double_complex(fftimg2);
    in3 = cpl_image_get_data_double_complex(fftimg3);

    cpl_fft_imagelist(fftlist, imglist,  CPL_FFT_FORWARD);

    /* Each of the three images are now in the fourier domain: shift */

    sph_fft_shift_fftw_triplet(self, in1, in2, in3, dx, dy);

    cpl_fft_imagelist(imglist, fftlist, CPL_FFT_BACKWARD |  CPL_FFT_NOSCALE);

    cpl_imagelist_delete(fftlist);
    cpl_imagelist_unset(imglist, 2);
    cpl_imagelist_unset(imglist, 1);
    cpl_imagelist_unset(imglist, 0);
    cpl_imagelist_delete(imglist);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Perform a shifting using FFT of a cpl_image
 * @param self
 * @param dx
 * @param dy
 *
 * @return shifted images or NULL
 *
 * This creates a new shifted version of the input image.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_fft_shift_image(sph_fft* self, const cpl_image* im, double dx, double dy) {

    const int    N  = cpl_image_get_size_x(im);
    cpl_image*   tim = NULL;

    cpl_ensure(cpl_image_get_size_y(im) == N, CPL_ERROR_ILLEGAL_INPUT, NULL);

    if ( self->method == SPH_FFT_GSL_RADIX2 ||
         self->method == SPH_FFT_GSL_MIXEDRADIX )
    {
        sph_fft_prepare(self, im);
        sph_fft_forward(self);
        sph_fft_shift(self,dx,dy);
        sph_fft_inverse(self);
        if ( self->method == SPH_FFT_GSL_RADIX2 ) {
            sph_fft_crop_complex_array(self,self->nx/2,0);
        }
        tim = sph_fft_complex2image_real(self);
    }
    else if ( self->method == SPH_FFT_FFTW3_DP ) {
        fftw_complex* in = sph_fft_create_fftw_complex(self, im, N, N);
        fftw_plan     p  = fftw_plan_dft_2d(N, N, in, in,
                                            FFTW_FORWARD, FFTW_ESTIMATE);
        int           xx, yy;
        double*       ptim = (double*)cpl_malloc((size_t)(N * N)
                                                 * sizeof(*ptim));

       
        fftw_execute(p);
        fftw_destroy_plan(p);
        sph_fft_shift_fftw_array(self, in, dx, dy);
        p = fftw_plan_dft_2d(N, N, in, in, FFTW_BACKWARD, FFTW_ESTIMATE);
        fftw_execute(p);
        fftw_destroy_plan(p);

        for (yy = 0; yy < N; yy++) {
            for (xx = 0; xx < N; xx++) {
                ptim[yy * N + xx] = creal(in[yy * N + xx]) / (double)( N * N );
            }
        }
        fftw_free(in); in = NULL;

        tim = cpl_image_wrap_double(N, N, ptim);
    }
    else {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_UNSUPPORTED_MODE,
                "This FFT mode is not supported for shifting");
    }
    return tim;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Perform a shifting using FFT of a cpl_image
 * @param self
 * @param dx
 * @param dy
 *
 * @return shifted images or NULL
 *
 * This creates a new shifted version of the input image.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_fft_freq_filter( sph_fft* self, const cpl_image* im,
        sph_fft_filter_method method, double radius,
        double param1, double param2 )
{
    cpl_image*    tim = NULL;
    fftw_complex*           in = NULL;
    fftw_plan               p;
    int                     N = 0;
    int                     xx = 0;
    int                     yy = 0;
    int                     kk = 0;

    if ( self->method == SPH_FFT_GSL_RADIX2 ||
            self->method == SPH_FFT_GSL_MIXEDRADIX )
    {
        sph_fft_prepare(self,im);
        for (kk = 0; kk < self->nx; ++kk) {
            sph_fft_forward_col( self, kk );
        }
        for (kk = 0; kk < self->ny; ++kk) {
            sph_fft_forward_row( self, kk );
            sph_fft_filter_row( self, kk, method, radius, param1, param2 );
            sph_fft_inverse_row( self, kk );
        }
        for (kk = 0; kk < self->nx; ++kk) {
            sph_fft_inverse_col( self, kk );
        }
        if ( self->method == SPH_FFT_GSL_RADIX2 ) {
            sph_fft_crop_complex_array(self,self->nx/2,0);
        }
        tim = sph_fft_complex2image_real(self);
    }
    else if ( self->method == SPH_FFT_FFTW3_DP ) {
        N = cpl_image_get_size_x(im);
        cpl_ensure(cpl_image_get_size_y(im) == N,CPL_ERROR_ILLEGAL_INPUT,NULL);
        in = sph_fft_create_fftw_complex(self,im,N,N);
        p = fftw_plan_dft_2d(N,N,in,in,
                FFTW_FORWARD,FFTW_ESTIMATE);
        fftw_execute(p);
        fftw_destroy_plan(p);
        sph_fft_filter_fftw_array(self,in,method,radius,param1,param2);
        p = fftw_plan_dft_2d(N,N,in,in,
                FFTW_BACKWARD,FFTW_ESTIMATE);
        fftw_execute(p);
        fftw_destroy_plan(p);
        tim = cpl_image_duplicate(im);
        for (yy = 0; yy < N; ++yy) {
            for (xx = 0; xx < N; ++xx) {
                cpl_image_set(tim,xx+1,yy+1,creal(in[yy * N + xx])/ ( N * N ) );
            }
        }
        fftw_free(in); in = NULL;
    }
    else {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_UNSUPPORTED_MODE,
                "This FFT mode is not supported for shifting");
    }
    return tim;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Perform the scaling algorithm
 * @param self
 *
 * @return error code of the operation
 *
 * This performs the actual scaling.
 * The sph_fft object must have been prepared using
 * the sph_fft_scale_prepare function. The resulting image
 * can be obtained with the sph_rotate_complex2image function.
 *
 * See sph_fft_scale_prepare for more info.
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_fft_scale( sph_fft* self, const cpl_image* im, double scale ) {
    int nxreal        = 0;
    int nxfreq        = 0;
    int    xx            = 0;
    int    yy            = 0;
    int    nx            = 0;
    int    ny            = 0;
    double fact     = 0.0;
    double    dshift     = 0.0;
    cpl_image*        imout = NULL;
    sph_error_code    rerr = CPL_ERROR_NONE;
    fftw_complex*           in = NULL;
    fftw_complex*           arr2 = NULL;
    fftw_plan               p;
    double                  acc = 0.0;
    double              minacc  = 1000000.0;
    if ( !im ) {
        SPH_NO_SELF;
        return NULL;
    }
    nx = cpl_image_get_size_x(im);
    ny = cpl_image_get_size_y(im);
    acc = 0.5 / ( scale * nx );
    if ( nx != ny) {
        SPH_ERR("X and Y dimensions of image for scale operations must be same.");
        return NULL;
    }

    for (yy = nx; yy < 3*nx; yy+=2) {
        for (xx = nx; xx < 3 * nx; xx+=2) {
            fact = (double)xx/(double)yy;
            if ( fabs( fact - scale) < minacc ) {
                nxreal = yy;
                nxfreq = xx;
                minacc = fabs( fact-scale);
            }
        }
    }
    if ( nxreal < 1 || nxfreq < 1 ) {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,
                "Could not obtain good padding sizes for accuracy %f.",
                acc);
        return NULL;
    }
    else {
        sph_error_raise(SPH_ERROR_GENERAL,__FILE__,__func__,
                __LINE__,SPH_ERROR_INFO,
                "Am scaling from %d to %d pixels in real "
                "space and %d pixels in freqency space, acc was %f, scaling ac: %f.",nx,nxreal,nxfreq,acc,minacc);
    }
    if ( self->method == SPH_FFT_GSL_MIXEDRADIX ||
            self->method == SPH_FFT_GSL_RADIX2 )
    {
        rerr = sph_fft_prepare(self,im);
        rerr |= sph_fft_pad_complex_array_real_space(self,nxreal);
        rerr |= sph_fft_forward(self);
        if ( nxfreq < nxreal ) {
            rerr |= sph_fft_crop_complex_array(self,nxfreq,1);
        }
        else {
            rerr |= sph_fft_pad_complex_array_freq_space(self,nxfreq);
        }
        dshift = ( 1.0 - (double)nxfreq/(double)nxreal ) * -0.5;
        sph_error_raise(SPH_ERROR_GENERAL,__FILE__,__func__,
                __LINE__,SPH_ERROR_INFO,
                "Am using a phase shift of %f pixels.",dshift);
        rerr |= sph_fft_shift(self,dshift,dshift);
        rerr |= sph_fft_inverse(self);
        rerr |= sph_fft_crop_complex_array(self,nx,0);
        if ( rerr == CPL_ERROR_NONE ) {
            imout = sph_fft_complex2image_real(self);
        }
        else {
            SPH_ERR("Problem when performing scale operations.");
            return NULL;
        }
    }
    else if ( self->method == SPH_FFT_FFTW3_DP ) {
        in = sph_fft_create_fftw_complex(self,im,nxreal,nxreal);
        p = fftw_plan_dft_2d(nxreal,nxreal,in,in,
                FFTW_FORWARD,FFTW_ESTIMATE);
        fftw_execute(p);
        fftw_destroy_plan(p);
        dshift = ( 1.0 - (double)nxfreq/(double)nxreal ) * -0.5;
        if ( nxfreq < nxreal ) {
            arr2 = sph_fft_crop_fftw_array(self,in,nxfreq,1);
            fftw_free(in);
            in = arr2;arr2 = NULL;
        }
        else {
            arr2 = sph_fft_pad_fftw_array_freq_space(self,in,nxfreq);
            fftw_free(in);
            in = arr2;arr2 = NULL;
        }
        sph_fft_shift_fftw_array(self,in,dshift,dshift);
        p = fftw_plan_dft_2d(self->nx,self->ny,in,in,
                FFTW_BACKWARD,FFTW_ESTIMATE);
        fftw_execute(p);
        fftw_destroy_plan(p);
        arr2 = sph_fft_crop_fftw_array(self,in,nx,0);
        fftw_free(in);
        in = arr2;arr2 = NULL;
        imout = cpl_image_new(self->nx, self->ny, cpl_image_get_type(im));
        for (yy = 0; yy < self->ny; ++yy) {
            for (xx = 0; xx < self->nx; ++xx) {
                cpl_image_set(imout,xx+1,yy+1,(double)self->nx / (double)nxreal * (double)self->nx / (double)nxreal * creal(in[yy * self->nx + xx])/(scale * scale * self->nx * self->nx));
            }
        }
        fftw_free(in);
    }
    return imout;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Rotate an image
 * @param self      sph_fft workspace
 * @param image     input image (unchanged)
 * @param angle     rotation angle in degrees
 *
 * @return Created rotated image or NULL on error
 *
 * @see sph_fft_rotate_prepare
 *
 * This function is a top level wrapper for the other rotation
 * fft functions. It rotates a copy of an input image
 * using FFT and returns it.
 * @Note: this function returns and error message if the sph_fft
 * was not initialised with SPH_FFT_GSL_RADIX2.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_fft_rotate_image( sph_fft* self, const cpl_image* image, double degrees ) {
    cpl_image* result = cpl_image_duplicate(image);

    if (result == NULL) {
        (void)cpl_error_set_where(cpl_func);
    } else {

        degrees = sph_fft_rotate_quart(result, degrees);

        if (degrees != 0.0) {

            if (self == NULL) {
                (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
                cpl_image_delete(result);
                result = NULL;
            } else if ( self->method == SPH_FFT_FFTW3_DP ) {
                cpl_image* temp = result;
                result = sph_fftw_rotate_image(self, temp, degrees);
                cpl_image_delete(temp);
            } else if (self->method != SPH_FFT_GSL_RADIX2) {
                (void)cpl_error_set(cpl_func, CPL_ERROR_UNSUPPORTED_MODE);
                cpl_image_delete(result);
                result = NULL;
            } else if (sph_fft_rotate_prepare(self, result, -degrees,
                                              0.0, 0.0) ||
                       sph_fft_rotate(self)) {
                (void)cpl_error_set_where(cpl_func);
                cpl_image_delete(result);
                result = NULL;
            } else {
                cpl_image_delete(result);
                result = sph_fft_rotate_complex2image(self);

                if (result == NULL) {
                    (void)cpl_error_set_where(cpl_func);
                }
            }
        }
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a smoothing operator for an image convolution
 * @return The newly allocated object
 * @see sph_fft_operand_delete
 *
 */
/*----------------------------------------------------------------------------*/
sph_fft_operand* sph_fft_operand_new(void)
{
    sph_fft_operand* self = (sph_fft_operand*)cpl_malloc(sizeof(*self));

    self->image = NULL;
    self->nuse  = 0;

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Smoothe an image by convolution with a Gaussian
 * @param self     Image to smoothe in place
 * @param convolve NULL or an FFT operator for smoothing repetions
 * @param fwhm    The Full Width at Half Maximum of the Gaussian to smoothe with
 * @return CPL_ERROR_NONE on success, otherwise the relevant CPL error code
 * @see sph_fft_operand_new
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_fft_smoothe_image(cpl_image*       self,
                                     sph_fft_operand* convolve,
                                     double           fwhm)
{

    if (self == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (cpl_image_get_type(self) != CPL_TYPE_DOUBLE) {
        return cpl_error_set(cpl_func, CPL_ERROR_UNSUPPORTED_MODE);
    } else if (fwhm <= 0.0) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "FWHM = %g <= 0", fwhm);
    } else {
        const cpl_size nx = cpl_image_get_size_x(self);
        const cpl_size ny = cpl_image_get_size_y(self);
        const cpl_size nx2 = nx / 2 + 1;
        cpl_image*     imconv;
        cpl_image*     selffft = cpl_image_wrap_double_complex
            (nx2, ny, (double complex*)cpl_malloc(nx2 * ny *
                                                  sizeof(double complex)));
        cpl_error_code code;


        /* FFT self, either alone or together with the convolution operator */
        if (convolve != NULL && convolve->image != NULL) {
            imconv = convolve->image;

            code = cpl_fft_image(selffft, self, CPL_FFT_FORWARD);
            assert(!code);

        } else {
            cpl_image*     imreal  = sph_fft_smoothe_create(nx, ny, fwhm);
            cpl_imagelist* imlreal = cpl_imagelist_new();
            cpl_imagelist* imlfft  = cpl_imagelist_new();

            imconv = cpl_image_wrap_double_complex
                (nx2, ny, cpl_malloc(nx2 * ny * sizeof(double complex)));

            cpl_imagelist_set(imlreal, imreal, 0);
            cpl_imagelist_set(imlreal, self,   1);

            cpl_imagelist_set(imlfft, imconv, 0);
            cpl_imagelist_set(imlfft, selffft, 1);

            /* Transform the two images together */
            code = cpl_fft_imagelist(imlfft, imlreal, CPL_FFT_FORWARD);
            assert(!code);

            (void)cpl_imagelist_unset(imlreal, 1);
            (void)cpl_imagelist_unset(imlfft,  1);
            (void)cpl_imagelist_unset(imlfft,  0);
            cpl_imagelist_delete(imlreal); /* Delete also imreal */
            cpl_imagelist_delete(imlfft);

            if (convolve != NULL) {
                assert( convolve->image == NULL);
                convolve->image = imconv;
            }
        }

        /* Perform the convolution in the Fourier domain - with scaling */
        code = cpl_image_multiply(selffft, imconv);
        assert(!code);

        /* Inverse FFT self */
        code = cpl_fft_image(self, selffft, CPL_FFT_BACKWARD | CPL_FFT_NOSCALE);
        assert(!code);

        cpl_image_delete(selffft);
        if (convolve == NULL) {
            cpl_image_delete(imconv);
        } else {
            assert(imconv == convolve->image);
            convolve->nuse++;
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Deallocate a convolution object
 * @see sph_fft_operand_new
 *
 */
/*----------------------------------------------------------------------------*/
void sph_fft_operand_delete(sph_fft_operand* self)
{
    if (self != NULL) {
        if (self->nuse) {
            cpl_msg_info(cpl_func, "FFT-operator (%d X %d) used %u time(s)",
                         (int)cpl_image_get_size_x(self->image),
                         (int)cpl_image_get_size_y(self->image),
                         self->nuse);
            assert( self->image != NULL );
            cpl_image_delete(self->image);
        } else {
            cpl_msg_info(cpl_func, "FFT-operator not used");
            assert( self->image == NULL );
        }
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a smoothing image, FFT-scaled but still to be FFT'ed
 * @param nx    The number of columns in image to be smoothed
 * @param ny    The number of rows    in image to be smoothed
 * @param fwhm  The Full Width at Half Maximum of the Gaussian to smoothe with
 * @return CPL_ERROR_NONE on success, otherwise the relevant CPL error code
 * @note Limited error checking in this static function
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_image* sph_fft_smoothe_create(cpl_size nx,
                                         cpl_size ny,
                                         double fwhm)
{
    const double    sigma = fwhm * CPL_MATH_SIG_FWHM;
    const double    x0    = (double)nx / 2.0;
    const double    y0    = (double)ny / 2.0;
    cpl_error_code  code;

    double*         pself = (double*)cpl_malloc(nx * ny * sizeof(*pself));
    cpl_image*      iself = cpl_image_wrap_double(nx, ny, pself);

    const double*   pswap = (double*)cpl_malloc(nx * ny * sizeof(*pswap));
    cpl_image*      swap  = cpl_image_wrap_double(nx, ny, (double*)pswap);


    cpl_msg_info(cpl_func, "Smoothing %d X %d image with FWHM=%g gaussian",
                 (int)nx, (int)ny, fwhm);

    code = cpl_image_fill_gaussian(swap, x0 + 1.0, y0 + 1.0, 1.0, sigma, sigma);
    assert( !code );

    code = cpl_image_multiply_scalar(swap, 1.0 /
                                     ((double)(nx * ny) *
                                      cpl_image_get_absflux(swap)));
    assert( !code );

    /* Swap quadrants */
    for (size_t yy = 0; yy < (size_t)ny / 2; ++yy) {
        (void)memcpy(pself + nx * yy,
                     pswap + nx * (yy + ny / 2) + nx / 2,
                     nx / 2 * sizeof(*pswap));
        (void)memcpy(pself + nx * yy + nx / 2,
                     pswap + nx * (yy + ny / 2),
                     (nx - nx / 2) * sizeof(*pswap));
    }
    for (size_t yy = ny / 2; yy < (size_t)ny; ++yy) {
        (void)memcpy(pself + nx * yy,
                     pswap + nx * (yy - ny / 2) + nx / 2,
                     nx / 2 * sizeof(*pswap));
        (void)memcpy(pself + nx * yy + nx / 2,
                     pswap + nx * (yy - ny / 2),
                     (nx - nx / 2) * sizeof(*pswap));
    }

    cpl_image_delete(swap);

    return iself;

}

static cpl_image* sph_fftw_rotate_image(sph_fft* self,
                                        const cpl_image* image,
                                        double degrees) {
    const size_t  nx = cpl_image_get_size_x(image);
    cpl_image*    result = cpl_image_duplicate(image);
    const int     nxy    = (int)nx;
    const int     embed  = nxy * nxy;
    fftw_complex* in     =
        sph_fft_create_fftw_complex(self, result, nxy, nxy);

    assert(self->method == SPH_FFT_FFTW3_DP);
    assert(-90.0 < degrees);
    assert(degrees < 90.0);

    if (cpl_image_get_size_y(image) != (cpl_size)nx) {
        fftw_free(in);
        cpl_image_delete(result);
        (void)cpl_error_set(cpl_func, CPL_ERROR_UNSUPPORTED_MODE);
        result = NULL;
    } else if (in == NULL) {
        cpl_image_delete(result);
        (void)cpl_error_set_where(cpl_func);
        result = NULL;
    } else if (nx * nx != (size_t)embed) {
        fftw_free(in);
        cpl_image_delete(result);
        (void)cpl_error_set(cpl_func, CPL_ERROR_UNSUPPORTED_MODE);
        result = NULL;
    } else {
        const double  angle  = self->angle = -degrees * CPL_MATH_RAD_DEG;
        const double  delta1 = tan(angle * -0.5);
        const double  delta2 = sin(angle);
        fftw_plan     pforwr, pforwc, pbackr, pbackc;

        // Forward rows
        pforwr = fftw_plan_many_dft(1, &nxy, nxy, in, &embed, 1,
                                    nxy, in, &embed, 1, nxy,
                                    FFTW_FORWARD, FFTW_ESTIMATE);
        fftw_execute(pforwr);

        // Shift rows
        for (size_t yy = 0; yy < nx; ++yy) {
            for (size_t xx = 0; xx < nx; ++xx) {
                const double k = 2 * xx < nx ? (double)xx : xx - (double)nx;

                const double shift = (delta1 * ( yy - ((double)nx-1.0) / 2.0 )
                                      + self->shiftx) * (k / (double)nx);

                in[xx + yy * nx] *= cexp(shift * CPL_MATH_2PI * I);
            }
        }

        // Forward cols
        pforwc = fftw_plan_many_dft(1, &nxy, nxy, in, &embed, nxy,
                                    1, in, &embed, nxy, 1,
                                    FFTW_FORWARD, FFTW_ESTIMATE);
        fftw_execute(pforwc);
        fftw_destroy_plan(pforwc);


        // Backward rows
        pbackr = fftw_plan_many_dft(1, &nxy, nxy, in, &embed, 1,
                                    nxy, in, &embed, 1, nxy,
                                    FFTW_BACKWARD, FFTW_ESTIMATE);
        fftw_execute(pbackr);

        // Shift cols
        for (size_t yy = 0; yy < nx; ++yy) {
            const double k = 2 * yy < nx ? (double)yy : yy - (double)nx;
            for (size_t xx = 0; xx < nx; ++xx) {
                const double shift = (delta2 * ( xx - ((double)nx-1.0)/2.0 )
                                      + self->shifty)  * (k / (double)nx);

                in[xx + yy * nx] *= cexp(shift * CPL_MATH_2PI * I);
            }
        }

        // Forward rows
        fftw_execute(pforwr);
        fftw_destroy_plan(pforwr);

        // Backward cols
        pbackc = fftw_plan_many_dft(1, &nxy, nxy, in, &embed, nxy,
                                    1, in, &embed, nxy, 1,
                                    FFTW_BACKWARD, FFTW_ESTIMATE);
        fftw_execute(pbackc);
        fftw_destroy_plan(pbackc);

        // Shift rows
        for (size_t yy = 0; yy < nx; ++yy) {
            for (size_t xx = 0; xx < nx; ++xx) {
                const double k = 2 * xx < nx ? (double)xx : xx - (double)nx;
                const double shift = delta1 * ( yy - ((double)nx-1.0)/2.0  )
                    * ( k / (double)nx);

                in[xx + yy * nx] *= cexp(shift * CPL_MATH_2PI * I);
            }
        }

        // Backward rows
        fftw_execute(pbackr);
        fftw_destroy_plan(pbackr);

        if (cpl_image_get_type(result) == CPL_TYPE_DOUBLE) {
            /* Handle the typical case more efficiently */
            double* presult = cpl_image_get_data_double(result);

            assert( presult != NULL);

            for (size_t i = 0; i < (size_t)embed; ++i) {
                presult[i] = creal(in[i]) / (double)(nx * nx * nx);
            }
        } else {
            for (size_t yy = 0; yy < nx; ++yy) {
                for (size_t xx = 0; xx < nx; ++xx) {
                    cpl_image_set(result, xx+1, yy+1,
                                  creal(in[xx + yy * nx]) / (nx * nx * nx));
                }
            }
        }

        fftw_free(in);
        if (cpl_error_get_code()) {
            (void)cpl_error_set_where(cpl_func);
            cpl_image_delete(result);
            result = NULL;
        }
    }

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Perform the 3-step skew rotation algorithm
 * @param self
 *
 * @return error code of the operation
 *
 * This performs the actual rotation according to the 3-step skew
 * algorithm. The sph_fft object must have been prepared using
 * the sph_fft_rotate_prepare function. The resulting image
 * can be obtained with the sph_rotate_complex2image function.
 *
 * See sph_fft_rotate_prepare for more info.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fft_rotate( sph_fft* self ) {
    int        kk        = 0;
    double                delta    = 0.0;
    double                dx        = self->shiftx;
    double                dy        = self->shifty;
    double                fac = 1.0; //should become 2 when image doubling used

    delta = -tan(self->angle/2.0);
    //self->shiftx = self->shiftx + 0.707;
    //self->shifty = self->shifty - 0.5;
    for (kk = 0; kk < self->ny; ++kk) {
        sph_fft_forward_row( self, kk );
        sph_fft_shift_row( self, kk, delta );
    }
    for (kk = 0; kk < self->nx; ++kk) {
        sph_fft_forward_col( self, kk );
    }
    //sph_fft_rotate_double_sizey( self );
    delta = fac * sin(self->angle);
    for (kk = 0; kk < self->ny; ++kk) {
        sph_fft_inverse_row( self, kk );
    }
    for (kk = 0; kk < self->nx; ++kk) {
        sph_fft_shift_col( self, kk, delta );
    }
    self->shiftx = 0.0;
    self->shifty= 0.0;
    for (kk = 0; kk < self->ny; ++kk) {
        sph_fft_forward_row( self, kk );
    }
    delta = -tan(self->angle/2.0) / fac;
    for (kk = 0; kk < self->nx; ++kk) {
        sph_fft_inverse_col( self, kk );
    }
    for (kk = 0; kk < self->ny; ++kk) {
        sph_fft_shift_row( self, kk, delta );
        sph_fft_inverse_row( self, kk );
    }
    self->shiftx = dx;
    self->shifty = dy;
    return CPL_ERROR_NONE;
}

/* static  */
/* sph_error_code */
/* sph_fft_rotate_double_sizey( sph_fft* self ) { */
/*     gsl_complex_packed_array newarr; */

/*     newarr = cpl_calloc( 4 * self->Np * self->Npp, sizeof(double) ); */
/*     memcpy( newarr, self->complexdata, self->Npp * self->Np * sizeof(double)); */
/*     memcpy( &newarr[self->Np * self->Npp ], self->complexdata, self->Npp * self->Np * sizeof(double) ); */
/*     //memcpy( &newarr[self->Np * self->Npp / 2], self->complexdata, self->Npp * self->Np * sizeof(double) ); */
/*     cpl_free(    self->complexdata); */
/*     self->complexdata = newarr; */
/*     self->ny = self->Npp; */
/*     self->nx = self->Np; */
/*     return CPL_ERROR_NONE; */
/* } */
/*----------------------------------------------------------------------------*/
/**
 * @brief Create an image from the complex array
 * @param self
 *
 * @return the new image
 *
 * Ths converts the real part of the complex array
 * to a cpl_image without croppinh and
 * returns it.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_fft_complex2image_real( const sph_fft* self ) {
    cpl_image* result = NULL;

    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const size_t nx = (size_t)self->nx;
        const size_t ny = (size_t)self->ny;
        double* presult = (double*)cpl_malloc( nx * ny * sizeof(*presult));

        result = cpl_image_wrap_double( nx, ny, presult );

        for (size_t j = 0; j < ny; ++j) {
            for (size_t i = 0; i < nx; ++i) {
                presult[i + j * nx] = self->complexdata[ 2 * (i + j * nx) ];
            }
        }
    }

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create an image from the complex array
 * @param self
 *
 * @return the new image
 *
 * Ths converts the imaginary part of the complex array
 * to a cpl_image without croppinh and
 * returns it.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_fft_complex2image_imag( const sph_fft* self ) {
    cpl_image* result = NULL;

    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const size_t nx = (size_t)self->nx;
        const size_t ny = (size_t)self->ny;
        double* presult = (double*)cpl_malloc( nx * ny * sizeof(*presult));

        result = cpl_image_wrap_double( nx, ny, presult );

        for (size_t j = 0; j < ny; ++j) {
            for (size_t i = 0; i < nx; ++i) {
                presult[i + j * nx] = self->complexdata[ 2 * (i + j * nx) + 1 ];
            }
        }
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create an image from the complex array
 * @param self
 *
 * @return the new image
 *
 * Ths converts the real part of the complex array prepared
 * for in sph_fft_rotate_prepare to a cpl_image, crops it
 * to the original size and returns it.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_image*
sph_fft_rotate_complex2image( const sph_fft* self ) {
    cpl_image* result = NULL;

    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const size_t nx = (size_t)self->nx;
        const size_t ny = (size_t)self->ny;
        double* presult = (double*)cpl_malloc( nx/2 * ny/2 * sizeof(*presult));

        result = cpl_image_wrap_double( nx/2, ny/2, presult );

        for (size_t j = 0; j < ny / 2; ++j) {
            for (size_t i = 0; i < nx / 2; ++i) {
                presult[i + j * nx/2] =
                    self->complexdata[ 2 * ( ( j + ny / 4 ) * nx + i + nx / 4)];
            }
        }
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Prepare FFT module for FFT operations
 * @param self
 * @param im        the image to FFT (unmodified by sph_fft)
 *
 * @return error code of the operation
 *
 * This function prepares the sph_fft module for a FFT
 * operation. Space for a complex array of at least double the size of the image
 * is allocated.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_fft_prepare( sph_fft* self, const cpl_image* im ) {
    int                    newx    = 2;
    int                    newy    = 2;
    sph_error_code        rerr    = CPL_ERROR_NONE;
    int                    N = 0;

    if ( !im ) return SPH_ERROR_GENERAL;
    if ( cpl_image_get_size_x(im) != cpl_image_get_size_y(im) ) {
        return SPH_ERROR_GENERAL;
    }
    if ( self->method == SPH_FFT_GSL_RADIX2 ) {
        N = cpl_image_get_size_x( im );
        while ( newx <= N ) {
            newx = newx * 2;
        }
        newy = newx;
        if ( newx > 2048 * 4 || newy > 2048 * 4 ) {
            return SPH_ERROR_GENERAL; // this is too big to handle
        }
        sph_fft_prepare_gsl_radix2(self, im, newx, newy);
        self->Npp = newy * 2;
        self->Np = newx;
    }
    else if ( self->method == SPH_FFT_GSL_MIXEDRADIX ) {
        sph_fft_prepare_gsl_mixed_radix(self, im);
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Prepare FFT module for rotation
 * @param self
 * @param im        the image to rotate (unmodified by sph_fft)
 * @param degree    the angle in degrees
 * @param dx        additional constant shift in x
 * @param dy        additional constant shift in y
 *
 * @return error code of the operation
 *
 * This function prepares the sph_fft module for a rotation/shift
 * operation. Space for a complex array of at least double the size of the image
 * is allocated.
 * The rotation algorithm is the 3-phase skew algorithm as described
 * in Owen et al. 1996 and implemented in other languages like IDL or MATLAB
 * ( see http://eeg.sourceforge.net/doc_m2html/bioelectromagnetism/RotateImage.html)
 * Currently the implementation here does leaves out a step that is important
 * in many other applications: the doubling of the image size and masking to avoid
 * aliasing. This may be implemented in future if this is found to be a problem.
 * Note that you may have to exclude a border of width equal to the pixel
 * shift when using the created images. Also, there is a "bug" which invalidates
 * the top pixel row even when dx and dy are zero.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_fft_rotate_prepare(sph_fft* self,
                                      const cpl_image* im,
                                      double degrees,
                                      double dx,
                                      double dy) {
    int          N               = cpl_image_get_size_x( im );
    const double angle           = degrees * CPL_MATH_RAD_DEG;
    const double tantananglehalf = tan( angle / 2.0 ) * tan( angle / 2.0 );
    int                    newx  = 2;
    int                    newy  = 2;

    if ( !im ) return SPH_ERROR_GENERAL;

    if ( degrees > 90.0 || degrees < -90.0 ) {
        return SPH_ERROR_GENERAL;
    }
    if ( N != cpl_image_get_size_y(im) ) {
        return SPH_ERROR_GENERAL;
    }

    self->angle    = angle;

    while ( newx <= N + N * tantananglehalf ) {
        newx = newx * 2;
    }
    while ( newy <= 2 * N ) {
        newy = newy * 2;
    }
    newy = newy / 2;
    if ( newx > 2048 * 4 || newy > 2048 * 4 ) {
        return SPH_ERROR_GENERAL; // this is too big to handle
    }

    self->shiftx = -dx;
    self->shifty = -dy;

    sph_fft_prepare_gsl_radix2(self, im, newx, newy);

    self->Npp = newy * 2;
    self->Np  = newx;

    return CPL_ERROR_NONE;
}


sph_error_code
sph_fft_filter_fftw_array( const sph_fft* self, fftw_complex* in,
                           sph_fft_filter_method method, double delta,
                           double param1, double param2 ) {
    int         xx = 0;
    int         yy = 0;
    int         nx = 0;
    int         ny = 0;
    double      kx = 0.0;
    double      ky = 0.0;
    double      dshift = 0.0;
    double  r = 0.0;
    double factor = 0.0;
    double order = 0.0;

    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in,CPL_ERROR_NULL_INPUT);
    nx = self->nx;
    ny = self->ny;

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            if ( xx < ((double)nx)/2.0 ) {
                kx = (double)xx;
            }
            else {
                kx = (double)xx - (double)nx;
            }
            if ( yy < ((double)ny)/2.0 ) {
                ky = (double)yy;
            }
            else {
                ky = (double)yy - (double)ny;
            }

            dshift = (kx/(double)nx) * (kx/(double)nx) +
                    (ky/(double)ny) * (ky/(double)ny);
            if ( method == SPH_FFT_FILTER_METHOD_TH ) {
                if ( 4.0 * dshift > 1.0 - delta * delta ) {
                    in[yy * nx + xx] = 0.0;//exp( -dshift/(delta*delta));
                    in[yy * nx + xx] = 0.0;//exp( -dshift/(delta*delta));
                }
            }
            if ( method == SPH_FFT_FILTER_METHOD_FERMI ) {
                r = 1.0 - delta * delta;
                factor = 1.0 / ( 1 + exp( (4 * dshift - r) / param1 ) );
                in[yy * nx + xx] *=  factor;
                in[yy * nx + xx] *=  factor;
            }
            if ( method == SPH_FFT_FILTER_METHOD_BUTTER ) {
                order = 2 * log( 0.882 / sqrt(10.624 * 10.624 - 1.0) ) / log( param1 / param2 );
                r = param1 / pow(0.882, 2.0 / order);
                factor = 1.0 / sqrt( 1 + pow(4 * dshift / r, order ) );
                in[yy * nx + xx] *= factor;
                in[yy * nx + xx] *= factor;
            }

            if ( 4.0 * dshift > 1.0 - delta * delta ) {
                in[yy * nx + xx] = 0.0;
                in[yy * nx + xx] = 0.0;
            }
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
static void
sph_fft_shift_fftw_array( const sph_fft* self, fftw_complex* in,
                          double dx, double dy ) {

    const int n = self->nx;
    int xx, yy;

    /* FIXME: Does odd n need to be supported ? */

    for (yy = 0; yy < n; yy++) {
        for (xx = 0; xx < n; xx++) {
            const double kx = (double)((xx < (n+1)/2) ? xx : xx - n);
            const double ky = (double)((yy < (n+1)/2) ? yy : yy - n);

            const double shift_xy = -dx * kx - dy * ky;

            in[yy * n + xx] *= cexp((CPL_MATH_2PI * shift_xy /(double)n) * I);
        }
    }
}

fftw_complex*
sph_fft_crop_fftw_array( sph_fft* self, const fftw_complex* in,int newx, int shiftflag )
{
    fftw_complex*            newdata = NULL;
    int xx;
    int yy;
    int xxold;
    int yyold;
    int newy = newx;

    cpl_ensure(in,CPL_ERROR_ILLEGAL_INPUT,NULL);
    cpl_ensure(newx <= self->nx, CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    cpl_ensure(newy <= self->ny, CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    newdata = (fftw_complex*)fftw_malloc(sizeof(fftw_complex) * newx * newy);
    cpl_ensure(newdata,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    if ( shiftflag == 0 ) {
        for (yy = 0; yy < newy; ++yy) {
            (void)memcpy(newdata + yy * newx, in + 
                         (yy + (self->ny - newy) / 2) * self->nx +
                         (self->nx - newx) / 2,
                         newx * sizeof(*in));
        }
    }
    else {
        for (yy = 0; yy < newy/2; ++yy) {
            for (xx = 0; xx < newx/2; ++xx) {
                xxold = xx;
                yyold = yy;
                newdata[yy * newx + xx] = in[yyold * self->nx + xxold];
            }
            for (xx = newx/2; xx < newx; ++xx) {
                xxold = xx + self->nx - newx;
                yyold = yy;
                newdata[yy * newx + xx] = in[yyold * self->nx + xxold];
            }
        }
        for (yy = newy/2; yy < newy; ++yy) {
            for (xx = 0; xx < newx/2; ++xx) {
                xxold = xx;
                yyold = yy + self->ny - newy;
                newdata[yy * newx + xx] = in[yyold * self->nx + xxold];
            }
            for (xx = newx/2; xx < newx; ++xx) {
                xxold = xx + self->nx - newx;
                yyold = yy + self->ny - newy;
                newdata[yy * newx + xx] = in[yyold * self->nx + xxold];
            }
        }
    }
    self->nx = newx;
    self->ny = newy;
    return newdata;
}
fftw_complex*
sph_fft_pad_fftw_array_freq_space( sph_fft* self, const fftw_complex* in, int newn )
{
    fftw_complex*            newdata = NULL;
    int newx = newn;
    int newy = newn;
    int xx;
    int yy;
    int xxold;
    int yyold;
    int shiftx = newx-self->nx;
    int shifty = newy-self->ny;

    cpl_ensure(in,CPL_ERROR_ILLEGAL_INPUT,NULL);
    cpl_ensure(newx >= self->nx, CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    cpl_ensure(newy >= self->ny, CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
    newdata = (fftw_complex*)fftw_malloc(sizeof(fftw_complex) * newx * newy);
    cpl_ensure(newdata,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    for (yy = 0; yy < newy; ++yy) {
        /* FIXME: Use memset() */
        for (xx = 0; xx < newx; ++xx) {
            newdata[yy * newx + xx ] = 0;
        }
    }
    for (xx = 0; xx < ( newx - shiftx )/ 2 ; ++xx) {
        xxold = xx;
        for (yy = 0; yy < ( newy - shifty ) / 2; ++yy) {
            yyold = yy;
            newdata[yy * newx + xx] = in[yyold * self->nx + xxold];
        }
        for (yy = ( newy + shifty ) / 2; yy < newy; ++yy) {
            yyold = self->ny/2 + yy - (newy+shifty)/2;
            newdata[yy * newx + xx] = in[yyold * self->nx + xxold];
        }
    }
    for (xx = ( newx + shiftx )/ 2; xx < newx ; ++xx) {
        xxold = self->nx/2 + xx - (newx+shiftx)/2;
        for (yy = 0; yy < ( newy - shifty ) / 2; ++yy) {
            yyold = yy;
            newdata[yy * newx + xx] = in[yyold * self->nx + xxold];
        }
        for (yy = ( newy + shifty ) / 2; yy < newy; ++yy) {
            yyold = self->ny/2 + yy - (newy+shifty)/2;
            newdata[yy * newx + xx] = in[yyold * self->nx + xxold];
        }
    }
    self->nx = newx;
    self->ny = newy;
    return newdata;
}

sph_error_code
sph_fft_crop_complex_array( sph_fft* self, int newx, int shiftflag )
{
    const size_t newy = newx;
    double* newdata = (double*)cpl_calloc(2 * newx * newy, sizeof(*newdata));

    if ( shiftflag == 0 ) {
        for (size_t yy = 0; yy < (size_t)newy; yy++) {
            (void)memcpy(newdata + 2 * yy * newx,
                         self->complexdata +
                         2 * ((yy + ( self->ny - newy ) / 2) * self->nx
                              + ( self->nx - newx ) / 2),
                         2 * newx * sizeof(double));
        }
    }
    else {
        for (size_t yy = 0; yy < (size_t)newy/2; yy++) {
            (void)memcpy(newdata + 2 * yy * newx,
                         self->complexdata + 2 * (yy * self->nx),
                         newx * sizeof(double));
            (void)memcpy(newdata + 2 * (yy * newx + newx / 2),
                         self->complexdata + 2 * (yy * self->nx + newx/2 +
                                                  self->nx - newx),
                         newx * sizeof(double));
        }
        for (size_t yy = (size_t)newy/2; yy < (size_t)newy; yy++) {
            (void)memcpy(newdata + 2 * yy * newx,
                         self->complexdata +
                         2 * ((yy + self->ny - newy) * self->nx),
                         newx * sizeof(double));
            (void)memcpy(newdata + 2 * (yy * newx + newx / 2),
                         self->complexdata +
                         2 * ((yy + self->ny - newy) * self->nx
                              + newx / 2 + self->nx - newx),
                         newx * sizeof(double));
        }
    }
    cpl_free(self->complexdata);
    self->complexdata = newdata;
    self->nx = newx;
    self->ny = newy;
    if ( self->method == SPH_FFT_GSL_MIXEDRADIX ) sph_fft_alloc_mixedradix(self);

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Obtain scaling information using a row collapse of image
 * @param self      ftt module
 * @param im1       the image to obtain scaling info from
 *
 * @return period along the rows of the image.
 *
 * This routine measures the period along the x-axis (along the rows) of
 * the input image. It returns the period in pixels along with an estimate of
 * the error in the variable sigma.
 *
 *
 */
/*----------------------------------------------------------------------------*/
double
sph_fft_get_scale_vector(sph_fft* self, const cpl_image* im1, double *sigma )
{
    cpl_image*        powerim = NULL;
    cpl_image*        imcut = NULL;
    int                xx = 0;
    int                yy = 0;
    cpl_vector*    fftv = NULL;
    cpl_vector*    fftvx = NULL;
    const cpl_size nlx   = cpl_image_get_size_x(im1);
    const cpl_size nvec  = nlx / 2 - 1;
    gsl_complex*    valp = NULL;
    double  xm = 0.0;
    double    area = 0.0;
    double    offset = 0.0;
    cpl_error_code code;


    sph_fft_prepare(self,im1);
    sph_fft_forward(self);

    powerim = cpl_image_new( self->nx, self->ny, CPL_TYPE_DOUBLE );
    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx ; ++xx) {
            valp = (gsl_complex*)&self->complexdata[ 2 * ( yy * self->nx + xx ) ];
            cpl_image_set(powerim,xx+1,yy+1,gsl_complex_abs2(*valp));
        }
    }
    imcut = cpl_image_extract( powerim, 2,  1, nlx/2, 1);
    fftv = cpl_vector_new_from_image_row(imcut,1);
    fftvx = cpl_vector_duplicate(fftv);

    for (xx = 0; xx < nvec; ++xx) {
        cpl_vector_set(fftvx,xx,xx+1.0);
    }
    code = cpl_vector_fit_gaussian(fftvx,NULL,fftv,NULL,CPL_FIT_ALL,&xm,sigma,
                                   &area,&offset,NULL,NULL,NULL);
    cpl_image_delete(imcut);
    cpl_image_delete(powerim);
    cpl_vector_delete(fftv);
    cpl_vector_delete(fftvx);

    if (code) {
        xm = 0.0;
        (void)cpl_error_set_where(cpl_func);
    } else {
        cpl_msg_info(cpl_func, "Fitted %d-element vector: %g %g %g %g",
                     (int)nvec, xm, *sigma, area, offset);
        *sigma /= xm;
        xm = nlx / xm;
        *sigma *=  xm;
    }

    return xm;
}
sph_error_code
sph_fft_set_shift(sph_fft* self, double dx, double dy) {
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    self->shiftx = dx;
    self->shifty = dy;
    return CPL_ERROR_NONE;
}

fftw_complex*
sph_fft_create_fftw_complex( sph_fft* self, const cpl_image* im,
                             int newx, int newy )
{
    const int nx = cpl_image_get_size_x(im);
    const int ny = cpl_image_get_size_y(im);
    const int shiftx = newx - nx;
    const int shifty = newy - ny;

    const double * pim = cpl_image_get_data_double_const(im);

    int xx;
    int yy;
    fftw_complex*           in = NULL;

    cpl_ensure(self != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(pim  != NULL, CPL_ERROR_NULL_INPUT, NULL);

    cpl_ensure(shiftx >= 0,  CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(shifty >= 0,  CPL_ERROR_ILLEGAL_INPUT, NULL);

    in = (fftw_complex*)fftw_malloc(sizeof(*in) * (size_t)(newx * newy));

    cpl_ensure(in != NULL, CPL_ERROR_ILLEGAL_OUTPUT, NULL);

    if ( shiftx || shifty ) {
        for (yy = 0; yy < newy; ++yy) {
            (void)memset(in + (size_t)(yy * newx), 0,
                         sizeof(*in) * (size_t)newx);
        }
    }

    for (yy = shifty/2; yy < newy - shifty/2; ++yy) {
        const int yyold = yy - shifty/2;
        for (xx = shiftx/2; xx < newx - shiftx/2; ++xx) {
            const int xxold = xx - shiftx/2;
            in[yy * newx + xx] = pim[yyold * nx + xxold];
        }
    }

    self->nx = newx;
    self->ny = newy;

    return in;
}
sph_error_code
sph_fft_pad_complex_array_real_space( sph_fft* self, int newn )
{
    double*            newdata = NULL;
    int newx = newn;
    int newy = newn;
    int xx;
    int yy;
    int xxold;
    int yyold;
    int shiftx = newx-self->nx;
    int shifty = newx-self->ny;

    if ( self->nx >= newx || self->ny >= newy ) return CPL_ERROR_NONE;

    newdata = cpl_calloc(2 * newx * newy, sizeof(double) );
    for (xx = shiftx/2; xx < newx - shiftx/2; ++xx)
    {
        for (yy = shifty/2; yy < newy - shifty/2; ++yy)
        {
            xxold = xx - shiftx/2;
            yyold = yy - shifty/2;
            newdata[2 * ( yy * newx + xx)] = self->complexdata[2*(yyold * self->nx + xxold)];
            newdata[2 * ( yy * newx + xx) + 1] = self->complexdata[2*(yyold * self->nx + xxold) + 1];
        }
    }
    cpl_free(self->complexdata);
    self->complexdata = newdata;
    self->nx = newx;
    self->ny = newy;
    if ( self->method == SPH_FFT_GSL_MIXEDRADIX ) sph_fft_alloc_mixedradix(self);
    return CPL_ERROR_NONE;
}

sph_error_code
sph_fft_shift( sph_fft* self, double dx, double dy )
{
    int xx;
    int yy;
    double    y0 =0.0;
    double    x0    = 0.0;
    double kvalx = 0.0;
    double kvaly = 0.0;
    double dshiftx = 0.0;
    double dshifty = 0.0;
    gsl_complex            mulfac;
    gsl_complex            mulresult;
    gsl_complex*        valp    = NULL;

    x0 = ( self->nx - 1 ) / 2;
    y0 = ( self->ny - 1 ) / 2;
    for (yy = 0; yy < self->ny; ++yy) {
        for (xx = 0; xx < self->nx ; ++xx) {
            if ( yy <= y0 ) {
                kvaly = yy;
            }
            else {
                kvaly = yy - self->ny;
            }
            dshiftx = -1.0 * (double)dy * ( kvaly / (double)self->ny );
            mulfac = gsl_complex_polar( 1.0, CPL_MATH_2PI * dshiftx  );
            valp = (gsl_complex*)&self->complexdata[ 2 * ( yy * self->nx + xx ) ];
            mulresult = gsl_complex_mul( *valp, mulfac );
            GSL_REAL(*valp) = GSL_REAL(mulresult);
            GSL_IMAG(*valp) = GSL_IMAG(mulresult);
            if ( xx <= x0 ) {
                kvalx = xx;
            }
            else {
                kvalx = xx - self->nx;
            }
            dshifty = -1.0 * (double)dx * ( kvalx / (double)self->nx );
            mulfac = gsl_complex_polar( 1.0, CPL_MATH_2PI * dshifty  );
            valp = (gsl_complex*)&self->complexdata[ 2 * ( yy * self->nx + xx ) ];
            mulresult = gsl_complex_mul( *valp, mulfac );
            GSL_REAL(*valp) = GSL_REAL(mulresult);
            GSL_IMAG(*valp) = GSL_IMAG(mulresult);
        }
    }
    return CPL_ERROR_NONE;
}

sph_error_code
sph_fft_pad_complex_array_freq_space( sph_fft* self, int newn )
{
    double*            newdata = NULL;
    int newx = newn;
    int newy = newn;
    int xx;
    int yy;
    int xxold;
    int yyold;
    int shiftx = newx-self->nx;
    int shifty = newy-self->ny;

    if ( self->nx >= newx || self->ny >= newy ) return CPL_ERROR_NONE;

    newdata = cpl_calloc(2 * newx * newy, sizeof(double) );
    for (xx = 0; xx < ( newx - shiftx )/ 2 ; ++xx) {
        xxold = xx;
        for (yy = 0; yy < ( newy - shifty ) / 2; ++yy) {
            yyold = yy;
            newdata[2 * ( yy * newx + xx)] = self->complexdata[2*(yyold * self->nx + xxold)];
            newdata[2 * ( yy * newx + xx) + 1] = self->complexdata[2*(yyold * self->nx + xxold) + 1];
        }
        for (yy = ( newy + shifty ) / 2; yy < newy; ++yy) {
            yyold = self->ny/2 + yy - (newy+shifty)/2;
            newdata[2 * ( yy * newx + xx)] = self->complexdata[2*(yyold * self->nx + xxold)];
            newdata[2 * ( yy * newx + xx) + 1] = self->complexdata[2*(yyold * self->nx + xxold) + 1];
        }
    }
    for (xx = ( newx + shiftx )/ 2; xx < newx ; ++xx) {
        xxold = self->nx/2 + xx - (newx+shiftx)/2;
        for (yy = 0; yy < ( newy - shifty ) / 2; ++yy) {
            yyold = yy;
            newdata[2 * ( yy * newx + xx)] = self->complexdata[2*(yyold * self->nx + xxold)];
            newdata[2 * ( yy * newx + xx) + 1] = self->complexdata[2*(yyold * self->nx + xxold) + 1];
        }
        for (yy = ( newy + shifty ) / 2; yy < newy; ++yy) {
            yyold = self->ny/2 + yy - (newy+shifty)/2;
            newdata[2 * ( yy * newx + xx)] = self->complexdata[2*(yyold * self->nx + xxold)];
            newdata[2 * ( yy * newx + xx) + 1] = self->complexdata[2*(yyold * self->nx + xxold) + 1];
        }
    }
    cpl_free(self->complexdata);
    self->complexdata = newdata;
    self->nx = newx;
    self->ny = newy;
    if ( self->method == SPH_FFT_GSL_MIXEDRADIX ) sph_fft_alloc_mixedradix(self);
    return CPL_ERROR_NONE;
}

void sph_fft_delete( sph_fft* self ) {
    if ( self ) {

        if ( self->complexdata) cpl_free( self->complexdata );
        if ( self->wavetab ) gsl_fft_complex_wavetable_free(self->wavetab);
        if ( self->workspace ) gsl_fft_complex_workspace_free(self->workspace);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief   Rotate an square image by a multiple of 90 degrees clockwise.
  @param   self     The image to rotate in place
  @param   degrees  The rotation in degrees
  @return  Remaining degrees to be rotated
  @see cpl_image_turn()
  @note Error checking disabled for this function, assumes non-NULL input

  On success, the remainder is strictly within +/- 90 degrees,
  this remainder is still to be rotated. Thus, a reduced value can be returned
  with no rotation having taken place.

*/
/*----------------------------------------------------------------------------*/
static double sph_fft_rotate_quart(cpl_image* self, double degrees)
{
    if (degrees >= 360.0) {
        degrees -= 360.0 * floor(degrees / 360.0);
    } else if (degrees <= -360.0) {
        degrees += 360.0 * ceil(degrees / 360.0);
    }

    if (degrees >= 270.0 ) {
        cpl_image_turn( self, -3 );
        degrees -= 270.0;
    } else if (degrees >= 180.0 ) {
        cpl_image_turn( self, -2 );
        degrees -= 180.0;
    } else if (degrees >= 90.0 ) {
        cpl_image_turn( self, -1 );
        degrees -= 90.0;
    } else if ( degrees <= -270.0 ) {
        cpl_image_turn( self, 3 );
        degrees += 270.0;
    } else if ( degrees <= -180.0 ) {
        cpl_image_turn( self, 2 );
        degrees += 180.0;
    } else if ( degrees <= -90.0 ) {
        cpl_image_turn( self, 1 );
        degrees += 90.0;
    }

    assert( -90.0 < degrees);
    assert( degrees < 90.0 );

    return degrees;
}
