/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_filter_table.h"

#include <cpl.h>
#include <math.h>
#include <strings.h>
#include <string.h>

#ifdef SPH_TABLE_ASCII
/* Read one line from standard input, */
/* copying it to line array (but no more than max chars). */
/* Does not place terminating \n in line array. */
/* Returns line length, or 0 for empty line, or EOF for end-of-file. */
static
int sph_filter_table_getline__(FILE* f,char line[], int max)
{
    int nch = 0;
    int c;
    max = max - 1;          /* leave room for '\0' */

    while((c = fgetc(f)) != EOF)
    {
        if(c == '\n')
            break;

        if(nch < max)
        {
            line[nch] = c;
            nch = nch + 1;
        }
    }

    if(c == EOF && nch == 0)
        return EOF;

    line[nch] = '\0';
    return nch;
}
#endif

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
static
void
sph_filter_table_increase_size__(sph_filter_table* self) {
    int n = 0;

    n = sph_filter_table_get_size(self) + 1;

    cpl_vector_set_size(self->left_lambda,n+1);
    cpl_vector_set_size(self->right_lambda,n+1);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new table of field centres and angles
 * @param
 *
 * @return new pointer
 *
 * Creates a new table of field centres and angles.
 *
 */
/*----------------------------------------------------------------------------*/
sph_filter_table*  sph_filter_table_new(void) {
    sph_filter_table*        fctab = NULL;

    fctab = cpl_calloc(1,sizeof(sph_filter_table));
    fctab->filterNames = cpl_calloc(256,sizeof(char*));
    fctab->left_lambda = cpl_vector_new(1);
    fctab->right_lambda = cpl_vector_new(1);
    return fctab;
}

int sph_filter_table_get_size(const sph_filter_table* self) {
    cpl_ensure(self,CPL_ERROR_ILLEGAL_INPUT,-1);
    return cpl_vector_get_size(self->left_lambda)-1;
}

sph_error_code
sph_filter_table_add_row(sph_filter_table* self,
        const char* filter_name,
        double left_lambda, double right_lambda) {
    int         n = 0;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    n = sph_filter_table_get_size(self);
    self->filterNames[n] = cpl_calloc(256,sizeof(char));
    (void)strcpy(self->filterNames[n],filter_name);
    cpl_vector_set(self->left_lambda,n,left_lambda);
    cpl_vector_set(self->right_lambda,n,right_lambda);
    sph_filter_table_increase_size__(self);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

#ifdef SPH_TABLE_ASCII
sph_error_code
sph_filter_table_save_ascii( const sph_filter_table* self,
                             const char* filename ) {
    const int   n = sph_filter_table_get_size(self);
    FILE*       f = NULL;

    cpl_ensure_code( self     != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( filename != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( n > 0,            CPL_ERROR_ILLEGAL_INPUT);

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    f = fopen( filename, "w" );
    cpl_ensure_code( f != NULL,        CPL_ERROR_FILE_IO);

    fprintf(f,"# Filter table\n");
    fprintf(f,"# \n");
    fprintf(f,"# This table gives the list of filters\n");
    fprintf(f,"# for use in IRDIS science recipes.\n");
    fprintf(f,"# For any frame that has no filter name keyword in the header\n");
    fprintf(f,"# or a filter name that does not appear in this list,\n");
    fprintf(f,"# the first entry is taken as the default entry.\n");
    fprintf(f,"#\n");
    fprintf(f,"# Format of this table is: \n");
    fprintf(f,"# Filtername                  lambda of left FOV (um)   lambda of right FOV (um)\n");
    fprintf(f,"# \n");

    for (int ii = 0; ii < n; ++ii) {
        char* line = cpl_sprintf(" %27s %9.2f                 %9.2f",
                                 self->filterNames[ii],
                                 cpl_vector_get(self->left_lambda,ii),
                                 cpl_vector_get(self->right_lambda,ii) );
        fprintf(f,"%s\n",line);
        cpl_free(line);
    }
    fclose(f); f = NULL;

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
#endif


sph_filter_table*
sph_filter_table_load_fits(const char* filename) {
    const char*       name;
    cpl_size          i;
    double            left_lambda  = 0.0;
    double            right_lambda = 0.0;
    cpl_table         *raw_table   = NULL;
    sph_filter_table* self         = NULL;

    cpl_ensure(filename,CPL_ERROR_NULL_INPUT,NULL);
    self = sph_filter_table_new();
    raw_table = cpl_table_load(filename,1,0);
    if (raw_table==NULL){
    	SPH_ERROR_RAISE_ERR(cpl_error_get_code(),"Could not read filter table.")
    	sph_filter_table_delete(self); self = NULL;
    	return NULL;
    }
    if (!cpl_table_has_column(raw_table,"filter_id")){
    	SPH_ERROR_RAISE_ERR(cpl_error_get_code(),"Filter table has no filter_id column!")
    	sph_filter_table_delete(self); self = NULL;
    	return NULL;
    }
    if (!cpl_table_has_column(raw_table,"lambda_left")){
        	SPH_ERROR_RAISE_ERR(cpl_error_get_code(),"Filter table has no lambda_left column!")
        	sph_filter_table_delete(self); self = NULL;
        	return NULL;
        }
    if (!cpl_table_has_column(raw_table,"lambda_right")){
        	SPH_ERROR_RAISE_ERR(cpl_error_get_code(),"Filter table has no lambda_right column!")
        	sph_filter_table_delete(self); self = NULL;
        	return NULL;
        }
    if (cpl_table_get_nrow(raw_table) < 1){
    	SPH_ERROR_RAISE_ERR(cpl_error_get_code(),"Filter table has no entries!")
    	sph_filter_table_delete(self); self = NULL;
    	return NULL;
    }


    for (i=0;i<cpl_table_get_nrow(raw_table);i++) {
    	name = cpl_table_get_string(raw_table,"filter_id",i);
    	left_lambda = cpl_table_get_double(raw_table,"lambda_left",i,NULL);
    	right_lambda = cpl_table_get_double(raw_table,"lambda_right",i,NULL);
    	sph_filter_table_add_row(self,name,left_lambda,right_lambda);
    }
    SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO,
    		             "Read %d lines from filter table %s!",
    		             sph_filter_table_get_size(self),
    		             filename)

    return self;
}

#ifdef SPH_TABLE_ASCII
sph_filter_table*
sph_filter_table_load_ascii(const char* filename) {
    FILE*       f = NULL;
    char       line[256];
    char       name[256];
    double      left_lambda = 0.0;
    double      right_lambda = 0.0;
    int         nmatches = 0;
    int         linesize = 0;
    sph_filter_table*    self = NULL;
    cpl_ensure(filename,CPL_ERROR_NULL_INPUT,NULL);
    f = fopen(filename,"r");
    cpl_ensure(f,CPL_ERROR_FILE_NOT_FOUND,NULL);
    self = sph_filter_table_new();
    while ( (linesize = sph_filter_table_getline__(f,line,256)) != EOF ) {
        if ( line[0]== '#' || linesize == 0 ) {
            ;
        }
        else {
            nmatches = sscanf(line,"%255s %lf %lf",
                    name,&left_lambda,&right_lambda);
            if ( nmatches == 3 ) {
                sph_filter_table_add_row(self,name,left_lambda,right_lambda);
            }
            else {
                SPH_ERROR_ENSURE_GOTO_EXIT(0,CPL_ERROR_BAD_FILE_FORMAT);
            }
        }
    }
    fclose(f); f = NULL;

    return self;
EXIT:
    sph_filter_table_delete(self); self = NULL;
    return NULL;
}
#endif

cpl_vector*
sph_filter_table_get_left_lambdas(sph_filter_table* self) {
    cpl_vector* vector = NULL;
    int         n = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    n = sph_filter_table_get_size(self);
    cpl_ensure(n>0,CPL_ERROR_ILLEGAL_INPUT,NULL);
    vector = cpl_vector_duplicate(self->left_lambda);
    cpl_vector_set_size(vector,n);
    return vector;
}

cpl_vector*
sph_filter_table_get_right_lambdas(sph_filter_table* self) {
    cpl_vector* vector = NULL;
    int         n = 0;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    n = sph_filter_table_get_size(self);
    cpl_ensure(n>0,CPL_ERROR_ILLEGAL_INPUT,NULL);
    vector = cpl_vector_duplicate(self->right_lambda);
    cpl_vector_set_size(vector,n);
    return vector;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Look up the left and right lambda from the filter table
  @param    self   The filter table
  @param    filter The name of the filter
  @param    pleft  On success, set to the left lambda
  @param    pright On success, set to the right lambda
  @return   CPL_ERROR_NONE or the relevant CPL error code on error
  @note If the filter is not found both left and right lambda are set to -1
        and a warning is raised

  Possible CPL error codes set in this function:
  - CPL_ERROR_NULL_INPUT if an input pointer is NULL
  - CPL_ERROR_ILLEGAL_INPUT if the table is empty
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_filter_table_get_left_right_lambda(const sph_filter_table* self,
                                       const char* filtername,
                                       double* pleft, double* pright) {

    int ii, n;

    cpl_ensure_code(self       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(filtername != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pleft      != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pright     != NULL, CPL_ERROR_NULL_INPUT);

    n = sph_filter_table_get_size(self);
    cpl_ensure_code(n > 0,              CPL_ERROR_ILLEGAL_INPUT);

    for (ii = 0; ii < n; ++ii) {
        if (strcasecmp(self->filterNames[ii], filtername) == 0) break;
    }

    if (ii < n) {
        *pleft  = cpl_vector_get(self->left_lambda,  ii);
        *pright = cpl_vector_get(self->right_lambda, ii);
    } else {
        *pleft  = -1.0;
        *pright = -1.0;
        SPH_ERROR_RAISE_WARNING(CPL_ERROR_DATA_NOT_FOUND, "Filter '%s' not "
                            "found among %d table value(s)", filtername, n);
    }

    return CPL_ERROR_NONE;
}

void sph_filter_table_delete(sph_filter_table* self) {
    int ii;
    if ( self ) {
        for (ii = 0; ii < sph_filter_table_get_size(self); ++ii) {
            cpl_free(self->filterNames[ii]);
        }
        cpl_vector_delete(self->left_lambda);
        cpl_vector_delete(self->right_lambda);
        cpl_free(self->filterNames);
        cpl_free(self);
    }
}
/**@}*/
