/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_framelist.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_dataset.h"
#include "sph_dataset_stats.h"
#include "sph_keyword_manager.h"
#include <math.h>
#include <cpl.h>
const int SPH_FRAMELIST_LAMDA_TOLERANCE     = 0.00000001;
const int SPH_FRAMELIST_MAX_SETS            = 1000;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_framelist Lists of Frames
 *
 * @par Descirption:
 * This module provides the functionality to create lists of frames from
 * a given input framelist with various criteria. This is used by recipes
 * to create lists of frames each with constant exposure time, wavelength
 * or offset.
 * These frame lists store sets of frames, where each set is associated with
 * one specific value. In this way, frames can be grouped into sets with
 * similar properties as identified by the keywords.
 *
 * @par Synopsis:
 * @code
 * typedef struct _sph_framelist_
 * {
 *    cpl_frameset**  framesets;
 *    double*         values;
 *    int             nsets;
 * } sph_framelist;
 * @endcode
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

sph_error_code SPH_FRAMELIST_FRAMES_GENERAL     = SPH_FRAMELIST_ERR_START + 0;
sph_error_code SPH_FRAMELIST_FRAMES_MISSING     = SPH_FRAMELIST_ERR_START + 1;
sph_error_code SPH_FRAMELIST_SET_OUT_OF_BOUNDS  = SPH_FRAMELIST_ERR_START + 2;
sph_error_code SPH_FRAMELIST_NO_VALID_DITS      = SPH_FRAMELIST_ERR_START + 3;
sph_error_code SPH_FRAMELIST_FAILED_PIXHIST     = SPH_FRAMELIST_ERR_START + 4;
sph_error_code SPH_FRAMELIST_FRAMES_NOT_FOUND   = SPH_FRAMELIST_ERR_START + 5;
sph_error_code SPH_FRAMELIST_FRAMES_DOUBLE      = SPH_FRAMELIST_ERR_START + 6;
static double sph_framelist_get_lambda( cpl_propertylist* plist, const char* czKeyword ) {
    cpl_type typ = CPL_TYPE_UNSPECIFIED;
    int intval = 0;
    double lambda = 0.0;
    float floatval = 0.0;

    typ     = cpl_propertylist_get_type( plist,
            czKeyword);

    if ( typ == CPL_TYPE_INT) {
        intval = cpl_propertylist_get_int( plist,
                czKeyword);
        lambda = (double)intval;
    }
    else if ( typ == CPL_TYPE_FLOAT) {
        floatval = cpl_propertylist_get_float( plist,
                czKeyword);
        lambda = (double)floatval;
    }
    else if ( typ == CPL_TYPE_DOUBLE) {
        lambda = cpl_propertylist_get_double( plist,
                czKeyword);
    }
    else if ( typ == CPL_TYPE_INVALID ) {
        sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Could not find the keyword %s in the input frames "
                "Cant create framelist.\n", czKeyword );
    }
    return lambda;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new (empty) framelist
 *
 * @return pointer to newly created framelist or NULL on error
 *
 * This creates a new empty framelist.
 */
/*----------------------------------------------------------------------------*/
sph_framelist* sph_framelist_new(void) {
    sph_framelist*   self    = NULL;
    int              ii      = 0;

    self = cpl_calloc(1, sizeof(sph_framelist));

    if ( !self ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not allocate framelist" );
        return NULL;
    }
    self->values = cpl_calloc( SPH_FRAMELIST_MAX_SETS, sizeof( double ));
    self->framesets = cpl_calloc( SPH_FRAMELIST_MAX_SETS, sizeof( cpl_frameset* ) );
    for ( ii = 0; ii < SPH_FRAMELIST_MAX_SETS; ii ++ ) {
        self->values[ii] = 0.0;
        self->framesets[ii] = NULL;
    }

    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return a frame from a set as a cpl image.
 *
 * @param self      The sph_framelist
 * @param set       The number of the set to use (zero = first)
 * @param frameno   The number of the frame in the set to return as image
 *                  (zero = first).
 * param  type      The cpl_type of the images to return
 * @param plane     The plane the image to return is on (in the frame)
 * @param extension The extension the image to return is on (in the frame)
 *
 * @return pointer to newly created cpl_image or NULL on error
 *
 * This creates a new cpl_image from the given frame and frameset
 */
/*----------------------------------------------------------------------------*/
cpl_image*  sph_framelist_create_cpl_image( sph_framelist* self,
                                            int set,
                                            int frameno,
                                            cpl_type type,
                                            int plane,
                                            int extension) {
    cpl_image*      result      = NULL;
    cpl_frame*      aframe      = NULL;
    if ( !self ) {
        SPH_NO_SELF;
        return NULL;
    }

    if ( set >= self->nsets || set < 0) {
        sph_error_raise( SPH_FRAMELIST_SET_OUT_OF_BOUNDS,
                         __FILE__,
                         __func__,
                         __LINE__,
                         SPH_ERROR_WARNING,
                         "The set provided, %d was equal or bigger than the "
                         "number of sets in the sph_framelist, %d.",
                         set, self->nsets );
        return NULL;
    }
    if ( frameno >= cpl_frameset_get_size(self->framesets[set] ) || frameno < 0 ) {
        sph_error_raise( SPH_FRAMELIST_SET_OUT_OF_BOUNDS,
                         __FILE__,
                         __func__,
                         __LINE__,
                         SPH_ERROR_WARNING,
                         "The set provided, %d was equal or bigger than the "
                         "number of sets in the sph_framelist, %d.",
                         set, self->nsets );
        return NULL;
    }

    aframe = cpl_frameset_get_position(self->framesets[set], frameno );

    if ( !aframe ) {
        SPH_RAISE_CPL;
        return NULL;
    }

    result = cpl_image_load( cpl_frame_get_filename( aframe ), type, plane, extension );
    if ( !result ) {
        SPH_RAISE_CPL;
        return NULL;
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a new list of frames given the input frames.

 @param    inframes     The input frameset
 @param    keyword      The keyword whose value is to be used to filter
 @param    tolerance    The tolerance for the filtering.

 @return   Pointer of the new cpl_frameset or null in case of error.

 Using the input frameset a new list of frames is created where the input
 frameset is filtered according to the criteria specified by the keyword
 and the tolerance parameter.
 All frames that have keywords defined with the given name and have
 that keyword value within tolerance of the specified value are selected
 in each set.

 */
/*----------------------------------------------------------------------------*/
sph_framelist* sph_framelist_create( cpl_frameset* inframes,
                                     const char* czKeyword,
                                     double tolerance
                                   )
{
    sph_framelist*      resultset       = NULL;
    int                 rerr            = 0;
    int                 ii              = 0;
    cpl_frame*          currentFrame    = NULL;
    cpl_propertylist*   plist           = NULL;
    int                 frameset_found  = 0;
    double              lambda          = 0.0;

    resultset = sph_framelist_new();

    if ( !resultset ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not allocate framelist"
                       );
        return NULL;
    }

    currentFrame = cpl_frameset_get_first( inframes );

    if ( !currentFrame ) {
        sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                         __FILE__,
                         __func__,
                         __LINE__,
                         SPH_ERROR_ERROR,
                         "Cant find any input frames -- so will return empty list only."
                        );
        return NULL;
    }

    while ( currentFrame ) {
        plist  = sph_keyword_manager_load_properties( cpl_frame_get_filename( currentFrame ),
                                        0 );
        frameset_found = 0;
        if ( czKeyword )
        {
            if ( plist )
            {
                lambda = sph_framelist_get_lambda( plist, czKeyword );

                for ( ii = 0; ii < resultset->nsets; ii ++ ) {
                    if ( fabs( lambda - resultset->values[ii] )
                            <= tolerance )
                    {
                        if ( resultset->framesets[ii] ) {
                            /* TODO: Check that images are uniform */
                            rerr = cpl_frameset_insert( resultset->framesets[ii],
                                    cpl_frame_duplicate( currentFrame ));
                            if ( rerr ) {
                            	SPH_ERR("cant insert frameset into the result set");
                            	goto ERROR_EXIT;
                            }
                            frameset_found = 1;
                        }
                        else {
                            sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                                    __FILE__,
                                    __func__,
                                    __LINE__,
                                    SPH_ERROR_ERROR,
                                    "A frameset in the framelist has gone missing... "
                                    "Cant create framelist.\n");
                            goto ERROR_EXIT;
                        }
                    }
                }
            }
            else {
                sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Could not read the propertylist of frame %s.",
                        cpl_frame_get_filename( currentFrame ) );
                goto ERROR_EXIT;
            }
        }
        else {
            lambda = resultset->nsets;
        }
        if ( !frameset_found ) {
            if ( resultset->nsets >= SPH_FRAMELIST_MAX_SETS ) {
                sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "The maximum frameset count was reached when\n"
                        "adding frames to the framesets.\n"
                        "Can not deal with with more than %d framesets.\n",
                        SPH_FRAMELIST_MAX_SETS );
                goto ERROR_EXIT;
            }
            else {
                resultset->nsets++;
                resultset->values[resultset->nsets - 1] = lambda;
                resultset->framesets[resultset->nsets - 1] =
                        cpl_frameset_new();
                if ( resultset->framesets[ resultset->nsets -1 ] )
                {
                    /* TODO: Check that images are uniform */
                    rerr = cpl_frameset_insert(
                            resultset->framesets[ resultset->nsets -1],
                            cpl_frame_duplicate( currentFrame ) );
                    if ( rerr ) {
                    	SPH_ERR("cant insert frameset into the result set");
                    	goto ERROR_EXIT;
                    }

                }
                else
                {
                    sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                            SPH_ERROR_ERROR,
                            "An allocation error occured when\n"
                            "adding frames to the framesets.\n"
                            "Problem occured after adding %d framesets.\n",
                            resultset->nsets -1 );
                    goto ERROR_EXIT;
                }
            }
        }

        if ( plist ) {
        	cpl_propertylist_delete( plist ); plist = NULL;
        }
        currentFrame = cpl_frameset_get_next(inframes);
    }

    return resultset;

  ERROR_EXIT:
    if ( resultset ) {
        sph_framelist_delete( resultset );
    }
    if ( plist ) {
    	cpl_propertylist_delete( plist ); plist = NULL;
    }
    return NULL;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a new list of frames given the input frames.

 @param    inframes     The input frameset
 @param    dit_list     The valid dit frame

 @return   Pointer of the new cpl_frameset or null in case of error.

 Using the input frameset a new lsit of frames is created where the input
 frameset is filtered according to the DIT ID as obtained from the dit list.

 */
/*----------------------------------------------------------------------------*/
sph_framelist* sph_framelist_create_from_dit( cpl_frameset* inframes,
                                              cpl_frame* dit_frame )
{
    sph_framelist*      resultset       = NULL;
    int                 rerr            = 0;
    int                 ii              = 0;
    cpl_frame*          currentFrame    = NULL;
    long int            dit_id          = 0;
    int                 frameset_found  = 0;

    resultset = sph_framelist_new();

    if ( !resultset ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not allocate framelist"
        );
        return NULL;
    }

    currentFrame = cpl_frameset_get_first( inframes );

    if ( !currentFrame ) {
        sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                         __FILE__,
                         __func__,
                         __LINE__,
                         SPH_ERROR_ERROR,
                         "Cant find any input frames -- so will return empty list only."
        );
        return NULL;
    }

    while ( currentFrame ) {
        dit_id = sph_utils_get_dit_id( currentFrame, dit_frame );

        if ( dit_id != -1 )
        {
            frameset_found = 0;

            for ( ii = 0; ii < resultset->nsets; ii ++ ) {
                if ( dit_id == resultset->values[ii] )
                {
                    if ( resultset->framesets[ii] ) {
                        /* TODO: Check that images are uniform */
                        rerr = cpl_frameset_insert( resultset->framesets[ii],
                                                    cpl_frame_duplicate( currentFrame ));
                        if ( rerr ) {
                        	SPH_ERR("cant insert frameset into the result set");
                        	goto ERROR_EXIT;
                        }

                        frameset_found = 1;
                    }
                    else {
                        sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                                         __FILE__,
                                         __func__,
                                         __LINE__,
                                         SPH_ERROR_ERROR,
                                         "A frameset in the framelist has gone missing... "
                                         "Cant create framelist.\n");
                        goto ERROR_EXIT;
                    }
                }
            }

            if ( !frameset_found ) {
                if ( resultset->nsets >= SPH_FRAMELIST_MAX_SETS ) {
                    sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                                    SPH_ERROR_ERROR,
                                    "The maximum frameset count was reached when\n"
                                    "adding frames to the framesets.\n"
                                    "Can not deal with with more than %d framesets.\n",
                                    SPH_FRAMELIST_MAX_SETS );
                    goto ERROR_EXIT;
                }
                else {
                    resultset->nsets++;
                    resultset->values[resultset->nsets - 1] = dit_id;
                    resultset->framesets[resultset->nsets - 1] =
                        cpl_frameset_new();
                    if ( resultset->framesets[ resultset->nsets -1 ] ) {
                        /* TODO: Check that images are uniform */
                        rerr = cpl_frameset_insert(
                                                   resultset->framesets[ resultset->nsets -1],
                                                   cpl_frame_duplicate( currentFrame ));
                        if ( rerr ) {
                        	SPH_ERR("cant insert frameset into the result set");
                        	goto ERROR_EXIT;
                        }

                    }
                    else {
                        sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                                        SPH_ERROR_ERROR,
                                        "An allocation error occured when\n"
                                        "adding frames to the framesets.\n"
                                        "Problem occured after adding %d framesets.\n",
                                        resultset->nsets -1 );
                        goto ERROR_EXIT;
                    }
                }
            }
        }
        else {
            sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                            SPH_ERROR_WARNING,
                            "Could not get the dit id of frame %s.",
                            cpl_frame_get_filename( currentFrame ) );
        }
        currentFrame = cpl_frameset_get_next(inframes);
        if ( currentFrame == NULL ) {
            cpl_error_reset();
        }
    }

    if ( resultset->nsets < 1 ) {
        sph_error_raise( SPH_FRAMELIST_NO_VALID_DITS, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Could not get any valid DIT frames !" );
        return NULL;
    }

    return resultset;

  ERROR_EXIT:
    if ( resultset ) {
        sph_framelist_delete( resultset );
    }
    return NULL;

}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return the frameset corresponding to a specific value
 *
 * @param self      the sph_framelist to return a frameset from
 * @param value     the value of the frameset to return
 * @param tolerance the tolerance for the match
 *
 * @return the frameset as a cpl_frameset, or NULL on error.
 *
 * This returns the frameset with the given value. The value associated
 * with the frameset has to match the given value within tolerance.
 * If more than one framesets with a match are found, this constitutes
 * aner error and NULL is returned.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frameset* sph_framelist_get_by_value( sph_framelist* self,
                                          double value, double tolerance ) {
    int                 ii      =  0;
    short               found   =  0;
    int                 lf      = -1;

    for (ii = 0; ii < self->nsets; ++ii) {
        if ( fabs( self->values[ii] - value ) <= tolerance ) {
            found++;
            lf = ii;
        }
    }
    if ( found == 0 ) {
        sph_error_raise( SPH_FRAMELIST_FRAMES_NOT_FOUND,
                         __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "No frameset with %lf of the "
                         "value %lf found.", tolerance, value);
        return NULL;
    }
    if ( found > 1 ) {
        sph_error_raise( SPH_FRAMELIST_FRAMES_DOUBLE,
                         __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "%hd matching framesets found. "
                         "Returning none!", found);
        return NULL;
    }

    return self->framesets[lf];
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Create a new list of frames given the input frames.

 @param    inframes     The input frameset
 @param    dit_list     The valid dit frame

 @return   Pointer of the new cpl_frameset or null in case of error.

 Using the input frameset a new lsit of frames is created where the input
 frameset is filtered according to the DITHER position.

 */
/*----------------------------------------------------------------------------*/
sph_framelist* sph_framelist_create_from_dither( cpl_frameset* inframes,
                                                 const char* keyx,
                                                 const char* keyy )
{
    sph_framelist*      resultset       = NULL;
    int                 rerr            = 0;
    int                 ii              = 0;
    cpl_frame*          currentFrame    = NULL;
    int                 dit_id          = 0;
    int                 frameset_found  = 0;
    float               res             = 0.01;

    resultset = sph_framelist_new();

    if ( !resultset ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not allocate framelist"
        );
        return NULL;
    }

    currentFrame = cpl_frameset_get_first( inframes );

    if ( !currentFrame ) {
        sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                         __FILE__,
                         __func__,
                         __LINE__,
                         SPH_ERROR_ERROR,
                         "Cant find any input frames -- so will return empty list only."
        );
        return NULL;
    }

    while ( currentFrame ) {
        dit_id = sph_utils_get_dither_id( currentFrame, res, keyx, keyy );

        if ( dit_id != -1 )
        {
            frameset_found = 0;

            for ( ii = 0; ii < resultset->nsets; ii ++ ) {
                if ( dit_id == resultset->values[ii] )
                {
                    if ( resultset->framesets[ii] ) {
                        /* TODO: Check that images are uniform */
                        rerr = cpl_frameset_insert( resultset->framesets[ii],
                                                    cpl_frame_duplicate( currentFrame ));
                        if ( rerr ) {
                        	SPH_ERR("cant insert frameset into the result set");
                        	goto ERROR_EXIT;
                        }

                        frameset_found = 1;
                    }
                    else {
                        sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                                         __FILE__,
                                         __func__,
                                         __LINE__,
                                         SPH_ERROR_ERROR,
                                         "A frameset in the framelist has gone missing... "
                                         "Cant create framelist.\n");
                        goto ERROR_EXIT;
                    }
                }
            }

            if ( !frameset_found ) {
                if ( resultset->nsets >= SPH_FRAMELIST_MAX_SETS ) {
                    sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                                    SPH_ERROR_ERROR,
                                    "The maximum frameset count was reached when\n"
                                    "adding frames to the framesets.\n"
                                    "Can not deal with with more than %d framesets.\n",
                                    SPH_FRAMELIST_MAX_SETS );
                    goto ERROR_EXIT;
                }
                else {
                    resultset->nsets++;
                    resultset->values[resultset->nsets - 1] = dit_id;
                    resultset->framesets[resultset->nsets - 1] =
                        cpl_frameset_new();
                    if ( resultset->framesets[ resultset->nsets -1 ] ) {
                        /* TODO: Check that images are uniform */
                        rerr = cpl_frameset_insert(
                                                   resultset->framesets[ resultset->nsets -1],
                                                   cpl_frame_duplicate( currentFrame ));
                        if ( rerr ) {
                        	SPH_ERR("cant insert frameset into the result set");
                        	goto ERROR_EXIT;
                        }

                    }
                    else {
                        sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                                        SPH_ERROR_ERROR,
                                        "An allocation error occured when\n"
                                        "adding frames to the framesets.\n"
                                        "Problem occured after adding %d framesets.\n",
                                        resultset->nsets -1 );
                        goto ERROR_EXIT;
                    }
                }
            }
        }
        else {
            sph_error_raise( cpl_error_get_code(), __FILE__, __func__, __LINE__,
                            SPH_ERROR_WARNING,
                            "Could not get the dither id of frame %s.",
                            cpl_frame_get_filename( currentFrame ) );
        }
        currentFrame = cpl_frameset_get_next(inframes);
        if ( currentFrame == NULL ) {
            cpl_error_reset();
        }
    }

    if ( resultset->nsets < 1 ) {
        sph_error_raise( SPH_FRAMELIST_NO_VALID_DITS, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Could not get any valid dithered frames !" );
    }

    return resultset;

  ERROR_EXIT:
    if ( resultset ) {
        sph_framelist_delete( resultset );
    }
    return NULL;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create a new list of frames given the input frames.

 @param    inframes     The input frameset
 @param    tolerance    The tolerance on the mean
 @param    badpix       The bad pixel mask (can be null)
 @param    external_means External means (can be null)

 @return   Pointer of the new cpl_frameset or null in case of error.

 Using the input frameset a new lsit of frames is created where the input
 frameset is filtered according to the mean value as obtained using the
 cpl_image_mean function. All frames with a mean of within the same tolerance
 are added. If given, the bad pixel mask is applied.

 */
/*----------------------------------------------------------------------------*/
sph_framelist* sph_framelist_create_from_mean( cpl_frameset* inframes,
                                               double tolerance,
                                               cpl_mask* badpix,
                                               cpl_vector* external_means )
{
    sph_framelist*      resultset       = NULL;
    int                 rerr            = 0;
    int                 ii              = 0;
    cpl_frame*          currentFrame    = NULL;
    cpl_image*          image           = NULL;
    int                 frameset_found  = 0;
    double              mean            = 0.0;
    int                 count           = 0;

    resultset = sph_framelist_new();

    if ( !resultset ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not allocate framelist"
        );
        return NULL;
    }

    currentFrame = cpl_frameset_get_first( inframes );

    if ( !currentFrame ) {
        sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                         __FILE__,
                         __func__,
                         __LINE__,
                         SPH_ERROR_ERROR,
                         "Cant find any input frames -- so will return empty list only."
        );
        return NULL;
    }

    if ( external_means ) {
        if ( cpl_vector_get_size( external_means ) != cpl_frameset_get_size( inframes ) ) {
            sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT,
                             __FILE__,
                             __func__,
                             __LINE__,
                             SPH_ERROR_ERROR,
                             "The number of frames and the size of the external means vector"
                             "did not match."
            );
            return NULL;
        }
    }

    while ( currentFrame ) {
        image = cpl_image_load( cpl_frame_get_filename( currentFrame), CPL_TYPE_DOUBLE, 0, 0 );
        if ( image == NULL ) {
            sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                             __FILE__,
                             __func__,
                             __LINE__,
                             SPH_ERROR_ERROR,
                             "Cant load an input frame -- will return empty list only."
            );
            return NULL;
        }
        if ( badpix != NULL ) {
            rerr = cpl_image_reject_from_mask( image, badpix );
        }
        if ( rerr ) {
				sph_error_raise( SPH_ERROR_GENERAL,
						__FILE__, __func__, __LINE__,
						SPH_ERROR_ERROR,
						"Error is raised by cpl_image_reject_from_mask:\n"
						"cpl error code is: %d", rerr );
            	goto ERROR_EXIT;
        }
        if ( external_means == NULL ) {
            mean = cpl_image_get_mean( image );
        }
        else {
            mean = cpl_vector_get( external_means, count );
        }
        cpl_image_delete( image );
        frameset_found = 0;

        for ( ii = 0; ii < resultset->nsets; ii ++ ) {
            if ( fabs( mean - resultset->values[ii] ) <= (float)tolerance )
            {
                if ( resultset->framesets[ii] ) {
                    /* TODO: Check that images are uniform */
                    rerr = cpl_frameset_insert( resultset->framesets[ii],
                                                cpl_frame_duplicate( currentFrame ));
                    if ( rerr ) {
                    	SPH_ERR("cant insert frameset into the result set");
                    	goto ERROR_EXIT;
                    }

                    frameset_found = 1;
                }
                else {
                    sph_error_raise( SPH_FRAMELIST_FRAMES_MISSING,
                                     __FILE__,
                                     __func__,
                                     __LINE__,
                                     SPH_ERROR_ERROR,
                                     "A frameset in the framelist has gone missing... "
                                     "Cant create framelist.\n");
                    goto ERROR_EXIT;
                }
            }
        }

        if ( !frameset_found ) {
            if ( resultset->nsets >= SPH_FRAMELIST_MAX_SETS ) {
                sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                                SPH_ERROR_ERROR,
                                "The maximum frameset count was reached when\n"
                                "adding frames to the framesets.\n"
                                "Can not deal with with more than %d framesets.\n",
                                SPH_FRAMELIST_MAX_SETS );
                goto ERROR_EXIT;
            }
            else {
                resultset->nsets++;
                resultset->values[resultset->nsets - 1] = mean;
                resultset->framesets[resultset->nsets - 1] =
                    cpl_frameset_new();
                if ( resultset->framesets[ resultset->nsets -1 ] ) {
                    /* TODO: Check that images are uniform */
                    rerr = cpl_frameset_insert(
                                               resultset->framesets[ resultset->nsets -1],
                                               cpl_frame_duplicate( currentFrame ));
                    if ( rerr ) {
                    	SPH_ERR("cant insert frameset into the result set");
                    	goto ERROR_EXIT;
                    }

                }
                else {
                    sph_error_raise(SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                                    SPH_ERROR_ERROR,
                                    "An allocation error occured when\n"
                                    "adding frames to the framesets.\n"
                                    "Problem occured after adding %d framesets.\n",
                                    resultset->nsets -1 );
                    goto ERROR_EXIT;
                }
            }
        }
        currentFrame = cpl_frameset_get_next(inframes);
        count++;
    }

    return resultset;

 ERROR_EXIT:
    if ( resultset ) {
        sph_framelist_delete( resultset );
    }
    return NULL;

}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete the framelist.
  @param    self        the framelist to delete

  Deallocate the framelist, the associated framesets and the corresponding value
  array.
 */
/*----------------------------------------------------------------------------*/
void sph_framelist_delete( sph_framelist* self) {

    if ( self != NULL ) {
        for (int ii = 0; ii < self->nsets; ii++ ) {
            cpl_frameset_delete( self->framesets[ii]);
        }
        cpl_free(self->framesets);
        cpl_free(self->values);
        cpl_free(self);
    }
}

/**@}*/
