/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include "sph_error.h"
#include "sph_gain_and_ron.h"
#include <math.h>
#include "sph_master_frame.h"
#include "sph_common_keywords.h"
#include "sph_pixel_polyfit_table.h"
#include "sph_fitting.h"
#include "sph_framecombination.h"
#include "sph_ptc.h"
#include "sph_filemanager.h"
#include "sph_common_science.h"
#include "sph_fits.h"
#include "sph_keyword_manager.h"
#include "sph_dataset.h"

static double const MIN_RON = -20.0;
static double const MAX_RON = 20.0;
static double const DELTA_RON = 5.0;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 * @brief Correct offset problem in datacubes
 * @param rawframes     the input uncorrected frameset
 * @param outframes     output frameset, will be filled with corrected cubes
 *
 * @return error code of the operation
 *
 * This function corrects each raw data cube so that the median count
 * of each image in each plane is identical to the median of the whole
 * datacube. This makes sure that any variations in offsets due to
 * imprecise(?) electronics do not affect the determination of the RMS
 * on the pixels for a datacube.
 * The input to this function should be the raw frames, the output will
 * be a list of frames which are new cubes that are corrected. This new
 * frameset should be used for the further RON and GAIN determinations.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_gain_and_ron_preprocessing(const cpl_frameset* rawframes,
                                              cpl_frameset* outframes) {

    const cpl_size nframes = cpl_frameset_get_size(rawframes);

    cpl_ensure_code(rawframes != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(outframes != NULL, CPL_ERROR_NULL_INPUT);

    for (cpl_size ff = 0; ff < nframes; ++ff) {
        const cpl_frame* rawframe =
            cpl_frameset_get_position_const(rawframes, ff);
        const char* filename = cpl_frame_get_filename(rawframe);
        const int nplanes = sph_fits_get_nplanes(filename, 0);
        cpl_frameset* tmpset;
        sph_master_frame* collapsed;
        double median_cube;


        if (nplanes < 0) break;

        tmpset = cpl_frameset_new();
        cpl_frameset_insert(tmpset, cpl_frame_duplicate(rawframe));
        collapsed = sph_framecombination_master_frame_from_cpl_frameset
            (tmpset, SPH_COLL_ALG_MEDIAN, NULL);
        cpl_frameset_delete(tmpset);

        if (collapsed == NULL) break;

        median_cube = sph_master_frame_get_median(collapsed, NULL);
        sph_master_frame_delete(collapsed);

        if (median_cube != 0.0) {
            char* basename = sph_filemanager_remove_dir(filename);
            cpl_frame* aframe = 
                sph_filemanager_create_temp_frame(basename,
                                                  cpl_frame_get_tag(rawframe),
                                                  CPL_FRAME_GROUP_RAW);
            cpl_propertylist* pl = cpl_propertylist_load(filename, 0);

            cpl_free(basename);

            for (int pp = 0; pp < nplanes && !cpl_error_get_code(); ++pp) {
                cpl_image* image =
                    cpl_image_load(filename, CPL_TYPE_FLOAT, pp, 0);
                const double median = cpl_image_get_median(image);
                if (median != 0.0) {
                    cpl_image_multiply_scalar(image, median_cube / median);
                    sph_cube_append_image(cpl_frame_get_filename(aframe),
                                          image, pl, 0);
                } else {
                    (void)cpl_error_set_message(cpl_func,
                                                CPL_ERROR_DIVISION_BY_ZERO,
                                                "Image %d/%d has zero-median: "
                                                "%s", 1+pp, nplanes, filename);
                }
                cpl_image_delete(image);
            }
            cpl_propertylist_delete(pl);
            if (cpl_error_get_code()) return cpl_error_get_code();
            sph_cube_finalise_file(cpl_frame_get_filename(aframe));
            cpl_frameset_insert(outframes, aframe);
        } else {
            return cpl_error_set_message(cpl_func, CPL_ERROR_DIVISION_BY_ZERO,
                                         "Cube has zero-median: %s", filename);
        }
    }

    return cpl_error_set_where(cpl_func);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Calculate the slope and offset fitting a series of frames
 * @param collapsed_framset         a frameset of master frames
 * @param order                     the fitting order
 * @param slope                     the slope (output)
 * @param slope_err                 the error on the slope (output)
 * @param offset                    the offset (output)
 * @param offset_err                the error on the offset (output)
 * @param red_chi                   reduced chi-squared of fit (output)
 * @return error code
 *
 * This function fits the input frameset and returns the slope and offset
 * with the corresponding errors estimates.
 *
 * The values and errors are estimated by fitting combined with a monte-carlo
 * approach to estimate the errors and obtain a better estimate of the mean
 * values for slope and offset.
 */
/*----------------------------------------------------------------------------*/
static sph_error_code sph_gain_and_ron_get_slope_and_offset(
        cpl_frameset* collapsed_frameset, int order, double* slope,
        double* slope_err, double* offset, double* offset_err, double* red_chi) {
    cpl_vector* means = NULL;
    cpl_vector* errs = NULL;
    cpl_vector* vals = NULL;
    cpl_polynomial* poly = NULL;
    cpl_vector* coefferrs = NULL;
    cpl_vector* coeffmeans = NULL;
    cpl_size pows = 0;
    cpl_ensure_code(collapsed_frameset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(slope, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(slope_err, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(offset, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(offset_err, CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    // Am using median not mean.
    // This was done on request by Raffaele Gratton, March 2011.
    // Remains to be seen if this is a good choice...
    means = sph_ptc_calculate_medians(collapsed_frameset, NULL, &errs, &vals);
    cpl_ensure_code(means != NULL && vals != NULL && errs != NULL,
            CPL_ERROR_ILLEGAL_OUTPUT);

    poly = sph_fitting_fit_poly1d(means, vals, errs, 0, order, 0, 1.0, red_chi);
    if (poly) {
        coefferrs = sph_fitting_estimate_error(poly, means, vals, errs,
                &coeffmeans);
        //*slope = cpl_vector_get(coeffmeans,1);
        pows = 1;
        *slope = cpl_polynomial_get_coeff(poly, &pows);
        *slope_err = cpl_vector_get(coefferrs, 1);
        //*offset = cpl_vector_get(coeffmeans,0);
        pows = 0;
        *offset = cpl_polynomial_get_coeff(poly, &pows);
        *offset_err = cpl_vector_get(coefferrs, 0);
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "Result of fit with "
                "order %d: slope: %4.2f+/-%4.2f and offset %4.2f+/-%4.2f.", order, *slope, *slope_err, *offset, *offset_err);
    } else {
        SPH_ERROR_RAISE_ERR(cpl_error_get_code(), "Could not perform fit.");
    }
    cpl_polynomial_delete(poly);
    poly = NULL;
    cpl_vector_delete(means);
    means = NULL;
    cpl_vector_delete(errs);
    errs = NULL;
    cpl_vector_delete(vals);
    vals = NULL;
    cpl_vector_delete(coefferrs);
    coefferrs = NULL;
    cpl_vector_delete(coeffmeans);
    coeffmeans = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Fill vector with "Vacca factors"
 * @param collapsed         the frameset of collapsed frames or the RAW cubes
 * @param gain_factor       vector of factors for slope term (output)
 * @param ron_factor        vector of factors for offset term (output)
 * @return error code
 *
 * This returns two vectors with the factors that are part of the
 * noise equation in Vacca 2004. The gain for frame i
 * can then be obtained by taking the ith element in gain_factor and dividing
 * it by the slope.
 * The corresponding ron^2 is the offset divided by the ith element in ron_factor
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_gain_and_ron_get_vacca_factors(
        cpl_frameset* collapsed_frameset, cpl_vector** gain_factor,
        cpl_vector** ron_factor) {
    int ff;
    cpl_frame* frame = NULL;
    cpl_propertylist* pl = NULL;
    int nsample = 0;
    int rom = 0;

    cpl_ensure_code(gain_factor, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ron_factor, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(collapsed_frameset, CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    *gain_factor = cpl_vector_new(cpl_frameset_get_size(collapsed_frameset));
    *ron_factor = cpl_vector_new(cpl_frameset_get_size(collapsed_frameset));

    for (ff = 0; ff < cpl_frameset_get_size(collapsed_frameset); ++ff) {

        frame = cpl_frameset_get_position(collapsed_frameset, ff);
        pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(frame),
                0);

        if (cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_NDIT)
                && cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_READ_CURID)) {

            if (cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_NDSAMPLES)) {
                nsample = cpl_propertylist_get_int(pl,
                        SPH_COMMON_KEYWORD_NDSAMPLES);
            } else {
                rom = cpl_propertylist_get_int(pl,
                        SPH_COMMON_KEYWORD_READ_CURID);
                if (rom == 2) {
                    nsample = 2;
                } else {
                    SPH_ERROR_RAISE_WARNING( CPL_ERROR_ILLEGAL_INPUT,
                            "Frame %d has bad ROM -- skipping.", ff);
                    nsample = -1;
                }
            }
            if (nsample > 0) {
                cpl_vector_set(
                        *gain_factor,
                        ff,
                        6.0 * (nsample * nsample + 1)
                                / (5.0 * nsample * (nsample + 1)));
                cpl_vector_set(*ron_factor, ff,
                        12.0 * (nsample - 1) / (1.0 * nsample * (nsample + 1)));
            } else {
                cpl_vector_set(*gain_factor, ff, 0.0);
                cpl_vector_set(*ron_factor, ff, 0.0);
            }
        } else {
            SPH_ERROR_RAISE_ERR(
                    SPH_ERROR_GENERAL,
                    "Cant read the required %s and or %s keywords.", SPH_COMMON_KEYWORD_NDIT, SPH_COMMON_KEYWORD_READ_CURID);
            cpl_vector_set(*gain_factor, ff, 0.0);
            cpl_vector_set(*ron_factor, ff, 0.0);
        }
        cpl_propertylist_delete(pl);
        pl = NULL;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Calculates an image of the gain using Vacca 2004 formalism
 * @param collapsed         the frameset of collapsed cubes
 * @param order             the order for the fit
 *
 * @return master frame of the gain
 *
 *
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_gain_and_ron_vacca_gain_image(cpl_frameset* collapsed_frameset) {
    int ff = 0;
    cpl_vector* factor_gain = NULL;
    cpl_vector* factor_ron = NULL;
    cpl_image* var1 = NULL;
    cpl_image* var2 = NULL;
    sph_master_frame* mframe1 = NULL;
    sph_master_frame* mframe2 = NULL;
    cpl_frame* frame1 = NULL;
    cpl_frame* frame2 = NULL;
    cpl_image* dsignal = NULL;
    cpl_frameset* tmpset = NULL;
    cpl_frame* tmpframe = NULL;
    sph_master_frame* result = NULL;
    cpl_parameterlist* framecomb = NULL;

    tmpset = cpl_frameset_new();
    sph_gain_and_ron_get_vacca_factors(collapsed_frameset, &factor_gain,
            &factor_ron);
    for (ff = 1; ff < cpl_frameset_get_size(collapsed_frameset); ++ff) {
        frame1 = cpl_frameset_get_position(collapsed_frameset, ff - 1);
        frame2 = cpl_frameset_get_position(collapsed_frameset, ff);
        mframe1 = sph_master_frame_load_(frame1, 0);
        mframe2 = sph_master_frame_load_(frame2, 0);
        var1 = sph_master_frame_get_variance(mframe1);
        var2 = sph_master_frame_get_variance(mframe2);
        cpl_image_subtract(var1, var2);
        sph_master_frame_subtract_master_frame(mframe1, mframe2);
        dsignal = sph_master_frame_extract_image(mframe1, 1);
        cpl_image_multiply_scalar(dsignal, cpl_vector_get(factor_gain, ff));
        cpl_image_divide(dsignal, var1);
        tmpframe = sph_filemanager_create_temp_frame("TMP", "GAIN",
                CPL_FRAME_GROUP_CALIB);
        cpl_image_save(dsignal, cpl_frame_get_filename(tmpframe),
                CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        cpl_frameset_insert(tmpset, tmpframe);
        sph_master_frame_delete(mframe1);
        mframe1 = NULL;
        sph_master_frame_delete(mframe2);
        mframe2 = NULL;
        cpl_image_delete(var1);
        var1 = NULL;
        cpl_image_delete(var2);
        var2 = NULL;
        cpl_image_delete(dsignal);
        dsignal = NULL;
    }
    framecomb = cpl_parameterlist_new();
    result = sph_framecombination_master_frame_from_cpl_frameset(tmpset,
            SPH_COLL_ALG_MEDIAN, framecomb);
    cpl_parameterlist_delete(framecomb);
    framecomb = NULL;
    cpl_vector_delete(factor_gain);
    factor_gain = NULL;
    cpl_vector_delete(factor_ron);
    factor_ron = NULL;
    cpl_frameset_delete(tmpset);
    tmpset = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Special implementation, following Vacca 2004.
 * @param collapsed     the list of collapsed frames
 * @param order         the fitting order
 * @param gain_out      the vector of gain values
 * @param ron_out       the vector of ron values
 *
 * @return error code
 *
 * This function calculates the gain and RON from the given set of
 * collapsed cubes (assumed to be stored in the master_frame format),
 * which are returned in the vector -- the vectors give the gain and ron
 * values in the order of the input frames respectively.
 *
 * Gain and RON values are calculated according to the formalism derived
 * by Vacca et al. 2004 which takes the read out samples of IR detector
 * multi-readout modes into account.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_gain_and_ron_vacca(cpl_frameset* collapsed_frameset,
        int order, cpl_vector** gain_out, cpl_vector** ron_out) {
    int ff = 0;
    cpl_vector* ron = NULL;
    cpl_vector* gain = NULL;
    cpl_vector* factor_gain = NULL;
    cpl_vector* factor_ron = NULL;
    double result_slope = 0.0;
    double result_offset = 0.0;
    double err_slope = 0.0;
    double err_offset = 0.0;
    double red_chi = 0.0;
    cpl_polynomial* poly = NULL;

    gain = cpl_vector_new(cpl_frameset_get_size(collapsed_frameset));
    ron = cpl_vector_new(cpl_frameset_get_size(collapsed_frameset));

    sph_gain_and_ron_get_vacca_factors(collapsed_frameset, &factor_gain,
            &factor_ron);
    if (sph_gain_and_ron_get_slope_and_offset(collapsed_frameset, order,
            &result_slope, &err_slope, &result_offset, &err_offset, &red_chi)
            == CPL_ERROR_NONE) {
        for (ff = 0; ff < cpl_vector_get_size(factor_ron); ++ff) {
            cpl_vector_set(gain, ff,
                    cpl_vector_get(factor_gain, ff) / result_slope);
            SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                    "Calculated a GAIN of %f e/ADU.", cpl_vector_get(gain,ff));
            if (result_offset > 0.0) {
                SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                        "Calculated a RON of %f ADU", sqrt(result_offset));
                result_offset = result_offset / cpl_vector_get(factor_ron, ff);
                SPH_ERROR_RAISE_INFO(
                        SPH_ERROR_GENERAL,
                        "Calculated a calibrated RON of %f e-", sqrt(result_offset) * cpl_vector_get(gain,ff));
                cpl_vector_set(ron, ff, sqrt(result_offset));
            } else {
                SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                        "Calculated a negative RON");
                cpl_vector_set(ron, ff, 0.0);
            }
        }
        cpl_polynomial_delete(poly);
        poly = NULL;
        *gain_out = cpl_vector_duplicate(gain);
        *ron_out = cpl_vector_duplicate(ron);
    } else {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL, "Could not perform fit.");
    }

    cpl_vector_delete(factor_gain);
    cpl_vector_delete(factor_ron);
    cpl_vector_delete(ron);
    cpl_vector_delete(gain);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

sph_error_code sph_gain_and_ron_calc(cpl_frameset* collapsed_frameset,
        int order, cpl_propertylist* pl) {
    double gain = 0.0;
    double ron = 0.0;
    double rms = 0.0;
    double rms_ron = 0.0;
    double red_chi = 0.0;

    /*------------------------------------------------------------------
     -  Sorting the frames into framesets of equal DIT setup
     --------------------------------------------------------------------*/

    if (cpl_frameset_get_size(collapsed_frameset) < order + 2) {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_ILLEGAL_INPUT,
                "The current fitting order is %d but only %d raw frames"
                " are provided. "
                "Either decrease the fitting order or include at least %d"
                " frames in the input. ", order, (int)cpl_frameset_get_size(collapsed_frameset), order+2);
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Calculating "
    "detector wide PTC...");
    sph_gain_and_ron_get_slope_and_offset(collapsed_frameset, order, &gain,
            &rms, &ron, &rms_ron, &red_chi);

    if (gain > 0.0) {
        rms = rms / gain;
        gain = 1.0 / gain;
        rms = rms * gain;
    }

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "From whole detector fitting, "
    "determined a gain of %f +/- %f e/ADU.", gain, rms);
    if (ron > 0.0) {
        rms_ron = rms_ron / ron;
        ron = sqrt(ron);
        rms_ron = 0.5 * rms_ron * ron;
    }SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "From whole detector fitting, "
    "determined a ron of %f +/- %f e-).", ron * gain, rms_ron * gain);

    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_QC_GAIN, gain);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_QC_GAIN_RMS, rms);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_QC_RON, ron * gain);
    cpl_propertylist_append_double(pl, SPH_COMMON_KEYWORD_QC_RON_RMS,
            rms_ron * gain);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief The main gain recipe code
 * @param rawframes         the input raw frames
 * @param collalg           the collapse algo
 * @param framecomb_params  the parameters for the collapse
 * @param gain_outfilename  the output file for the gain image
 * @param nonlin_outfilename the output file for the nonlinearity map
 * @param nonline_bpixfilename the output file for the nonlin badpixels
 * @param inframes          the complete set of input frames
 * @param inparams          the input parameters
 * @param taggain           the tag for the gain result
 * @param tagnonlin         the tag for the nonlin result
 * @param tagnonlinbad      the tag for the nonlin badpixel map
 * @param recname           the recipe name
 * @param pipname           the pipeline name
 * @param lintol            the tolerance on the linearity
 * @param order             the order of the fit to use
 *
 * @return error code
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_gain_run(cpl_frameset* rawframes,
        sph_collapse_algorithm collalg,
        cpl_parameterlist* framecomb_parameterlist,
        const char* gain_outfilename, const char* nonlin_outfilename,
        const char* nonlin_bpixfilename, cpl_frameset* inframes,
        cpl_parameterlist* inparams, cpl_frame* badpixframe,
        const char* taggain, const char* tagnonlin, const char* tagnonlinbad,
        const char* recname, const char* pipname, double lintolerance,
        int order, int save_addprod) {
    sph_pixel_polyfit_table* fittab = NULL;
    sph_master_frame* gain_mframe = NULL;
    sph_master_frame* ron_mframe = NULL;
    sph_master_frame* nonlin_map = NULL;
    sph_master_frame* tmpframe = NULL;
    cpl_propertylist* pl = NULL;
    cpl_image* badpix = NULL;
    cpl_vector* coefferrs = NULL;
    cpl_vector* coeffmeans = NULL;
    cpl_polynomial* poly = NULL;
    cpl_frameset* collapsed_frameset = NULL;
    int jj = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Collapsing %d raw frames"
            " and calculating detector wide PTC...", (int)cpl_frameset_get_size(rawframes));
    collapsed_frameset = sph_framecombination_collapse_cubes(rawframes, collalg,
            framecomb_parameterlist, recname, taggain);

    if (badpixframe) {
        badpix = cpl_image_load(cpl_frame_get_filename(badpixframe),
                CPL_TYPE_UNSPECIFIED, 0, 0);
        cpl_ensure_code(badpix, cpl_error_get_code());
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "Applying static badpixels from %s", cpl_frame_get_filename(badpixframe));
        for (jj = 0; jj < cpl_frameset_get_size(collapsed_frameset); ++jj) {
            tmpframe = sph_master_frame_load_(cpl_frameset_get_position_const
                                              (collapsed_frameset, jj),
                                              0);
            sph_master_frame_set_bads_from_image(tmpframe, badpix);
            sph_master_frame_save(
                    tmpframe,
                    cpl_frame_get_filename(
                            cpl_frameset_get_position(collapsed_frameset, jj)),
                    NULL);
            sph_master_frame_delete(tmpframe);
            tmpframe = NULL;
        }
        cpl_image_delete(badpix);
        badpix = NULL;
    }

    pl = cpl_propertylist_new();

    sph_gain_and_ron_calc(collapsed_frameset, order, pl);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Collapsing %d raw frames"
            " and calculating PTC for all pixels...", (int)cpl_frameset_get_size(rawframes));

    fittab = sph_ptc_create_full(collapsed_frameset, order);

    if (fittab == NULL) {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create the linear fit to pixels -- "
                        "Please check the previous error messages.\n");
        return SPH_ERROR_GENERAL;
    }SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Extracting gain and ron frames...");

    gain_mframe = sph_pixel_polyfit_table_get_coeff_as_image(fittab, 1);
    ron_mframe = sph_pixel_polyfit_table_get_coeff_as_image(fittab, 0);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Performing quality check...");
    sph_master_frame_quality_check(ron_mframe);

    sph_master_frame_quality_check(gain_mframe);

    // For saving, want to save gain not 1/gain.
    tmpframe = sph_master_frame_inverse(gain_mframe);
    sph_master_frame_delete(gain_mframe);
    gain_mframe = tmpframe;
    sph_master_frame_quality_check(gain_mframe);

    sph_master_frame_save_dfs(gain_mframe, gain_outfilename, inframes, NULL,
            inparams, taggain, recname, pipname, pl);

    if (order > 1 && save_addprod) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Determining non linearity map...");
        nonlin_map = sph_pixel_polyfit_table_get_coeff_as_image(fittab, 2);
        sph_master_frame_quality_check(nonlin_map);
        sph_master_frame_mask_tolerance(nonlin_map, -lintolerance,
                lintolerance);
        sph_master_frame_quality_check(nonlin_map);

        cpl_propertylist_append_int(pl, SPH_COMMON_KEYWORD_NUMBER_BADPIXELS,
                sph_master_frame_get_nbads(nonlin_map));

        sph_master_frame_save_dfs(nonlin_map, nonlin_outfilename, inframes,
                NULL, inparams, tagnonlin, recname, pipname, pl);

        cpl_propertylist_update_string(pl, SPH_COMMON_KEYWORD_PRO_CATG,
                tagnonlinbad);
        cpl_dfs_save_image(inframes, NULL, inparams, inframes, NULL,
                nonlin_map->badpixelmap, CPL_BPP_32_SIGNED, recname, pl, NULL,
                pipname, nonlin_bpixfilename);
    }
    sph_pixel_polyfit_table_delete(fittab);
    fittab = NULL;
    cpl_frameset_delete(collapsed_frameset);
    collapsed_frameset = NULL;
    sph_master_frame_delete(nonlin_map);
    nonlin_map = NULL;
    sph_master_frame_delete(gain_mframe);
    gain_mframe = NULL;
    sph_master_frame_delete(ron_mframe);
    gain_mframe = NULL;
    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_polynomial_delete(poly);
    poly = NULL;
    cpl_vector_delete(coefferrs);
    coefferrs = NULL;
    cpl_vector_delete(coeffmeans);
    coeffmeans = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

sph_master_frame*
sph_ron_simple_single(cpl_frame* aframe,
        cpl_parameterlist* framecomb_parameterlist, double gain, double sigclip,
        double gain_factor, double ron_factor, double* ron_out, double *rms_out) {
    cpl_mask* mask = NULL;
    sph_master_frame* ron_mframe_tmp = NULL;
    double ron = 0.0;
    double rms = 0.0;
    double rms_rms = 0.0;
    double tmp_peak = 0.0;
    double val = 0.0;
    double rms_val = 0.0;
    cpl_image* tmpim = NULL;
    sph_dataset* histogram = NULL;
    cpl_vector* histrms = NULL;
    double ron2 = 0.0;
    double chi = 0.0;
    int ii = 0;
    cpl_ensure(aframe, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(framecomb_parameterlist, CPL_ERROR_NULL_INPUT, NULL);

    ron_mframe_tmp = sph_master_frame_load_(aframe, 0);
    if (ron_mframe_tmp) {
        if (sigclip > 0.0) {
            sph_master_frame_quality_check(ron_mframe_tmp);
            sph_master_frame_mask_sigma(ron_mframe_tmp, sigclip);
            mask = sph_master_frame_get_badpixelmask(ron_mframe_tmp);
        }
        //cpl_image_multiply(ron_mframe_tmp->rmsmap,ron_mframe_tmp->rmsmap);

        // Calculating the value first...
        val = sph_master_frame_get_median(ron_mframe_tmp, &rms_val);
        //cpl_image_subtract_scalar(ron_mframe_tmp->rmsmap,val / gain);
        rms = sph_master_frame_get_median_variance(ron_mframe_tmp, &rms_rms);
        // ron = rms * rms - val / gain;
        rms_val = rms_val * gain_factor / gain;
        ron = rms - val * gain_factor / gain;
        ron = ron / ron_factor;
        //sph_master_frame_sqrt(ron_mframe_tmp);
        if (ron > 0) {
            //rms_rms = sqrt(rms_rms * rms_rms);
            rms_rms = rms_rms / ron_factor;
            rms_rms = rms_rms / ron;
            ron = sqrt(ron);
            rms = 0.5 * rms_rms * ron;
        } else {
            ron = 0.0;
            rms = 0.0;
        }
        // Now creating a RON image so we have something to look at
        // in the resulting FITS file.
        tmpim = sph_master_frame_get_variance(ron_mframe_tmp);
        cpl_image_divide_scalar(ron_mframe_tmp->image, gain / gain_factor);
        // ron = rms * rms - val / gain;
        cpl_image_subtract(tmpim, ron_mframe_tmp->image);
        if (mask)
            cpl_image_reject_from_mask(tmpim, mask);
        cpl_image_delete(ron_mframe_tmp->image);
        ron_mframe_tmp->image = cpl_image_duplicate(tmpim);
        cpl_image_delete(tmpim);
        tmpim = NULL;
        sph_master_frame_mask_tolerance(ron_mframe_tmp, 0.00001,
                10000000000000000.0);
        sph_master_frame_interpolate_bpix(ron_mframe_tmp);
        sph_master_frame_divide_double(ron_mframe_tmp, ron_factor);
        sph_master_frame_sqrt(ron_mframe_tmp);

        histogram = sph_dataset_new_pixel_histogram(ron_mframe_tmp->image,
                MIN_RON, MAX_RON, 500);
        ii = 1;
        while (sph_dataset_get_max_xpos(histogram) >= MAX_RON * ii - DELTA_RON
                && ii < 100) {
            sph_dataset_delete(histogram);
            histogram = NULL;
            histogram = sph_dataset_new_pixel_histogram(ron_mframe_tmp->image,
                    MAX_RON * ii, MAX_RON * (ii + 1), 500);
            ii++;
        }
        if (ii == 100) {
            SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,
                    "There is something pretty wrong with the "
                    "raw RON frames. "
                    "The pixel histogram does not seem to "
                    "have a peak below %f. Please take "
                    "new RON frames.", MAX_RON * 100);
        }
        tmp_peak = sph_dataset_get_max_xpos(histogram);
        sph_dataset_delete(histogram);
        histogram = NULL;
        histogram = sph_dataset_new_pixel_histogram(ron_mframe_tmp->image,
                tmp_peak - 0.5 * (MAX_RON - MIN_RON),
                tmp_peak + 0.5 * (MAX_RON - MIN_RON), 500);
        histrms = cpl_vector_duplicate(histogram->yvalues);
        cpl_vector_sqrt(histrms);
        cpl_vector_add_scalar(histrms, 1.0);
        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "The histogram has peak at: %f", sph_dataset_get_max_xpos(histogram));
        if (sph_dataset_fit_gauss(histogram, &ron2, &rms, histrms, &chi)
                == CPL_ERROR_NONE) {
            SPH_ERROR_RAISE_INFO(
                    SPH_ERROR_GENERAL,
                    "Fitting gauss gave RON %f +/- %f with chi %f", ron, rms, chi);
        } else {
            SPH_ERROR_RAISE_ERR(cpl_error_get_code(),
                    "Could not perform gaus fit.")
        }
        sph_dataset_delete(histogram);
        histogram = NULL;
        cpl_vector_delete(histrms);
        histrms = NULL;
        cpl_mask_delete(mask);
        mask = NULL;
    }
    if (ron_out) {
        *ron_out = ron;
    }
    if (rms_out) {
        *rms_out = rms;
    }
    return ron_mframe_tmp;
}

cpl_error_code sph_ron_run(cpl_frameset* rawframes,
        sph_collapse_algorithm colalg,
        cpl_parameterlist* framecomb_parameterlist, const char* ron_outfilename,
        cpl_frameset* inframes, cpl_parameterlist* inparams,
        const char* taggain, const char* recname, const char* pipname,
        int order, double gain, double sigclip, int vacca) {
    sph_master_frame* ron_mframe = NULL;
    sph_master_frame* ron_mframe_tmp = NULL;
    double rms = 0.0;
    double ron = 0.0;
    cpl_propertylist* pl = NULL;
    int ii = 0;
    cpl_polynomial* poly = NULL;
    char key[256];
    cpl_frame* aframe = NULL;
    int cc = 0;
    cpl_vector* gain_out = NULL;
    cpl_vector* ron_out = NULL;
    cpl_frameset* collapsed_frameset = NULL;
    cpl_vector* gain_factor = NULL;
    cpl_vector* ron_factor = NULL;
    double gainf = 1.0;
    double ronf = 1.0;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    pl = cpl_propertylist_new();
    collapsed_frameset = sph_framecombination_collapse_cubes(rawframes, colalg,
            framecomb_parameterlist, recname, taggain);
    if (gain <= 0.0) {
        if (cpl_frameset_get_size(collapsed_frameset) < 3) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT, "For determination "
            "of gain need to have at least 3 input frames "
            "(with different signal levels). "
            "Either add another input frame or add "
            "the --gain parameter to set the input gain "
            "and skip gain determination by this recipe.");
            cpl_propertylist_delete(pl);
            pl = NULL;
            cpl_frameset_delete(collapsed_frameset);
            collapsed_frameset = NULL;
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
        }
        if (vacca) {
            sph_gain_and_ron_vacca(collapsed_frameset, order, &gain_out,
                    &ron_out);
            cpl_vector_multiply(ron_out, gain_out);
            cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_GAIN,
                    cpl_vector_get_mean(gain_out));
            cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_GAIN_RMS,
                    cpl_vector_get_stdev(gain_out));
            cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_RON,
                    cpl_vector_get_mean(ron_out));
            cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_RON_RMS,
                    cpl_vector_get_stdev(ron_out));
            cpl_vector_delete(ron_out);
            ron_out = NULL;
        } else {
            sph_gain_and_ron_calc(collapsed_frameset, order, pl);
        }
        gain = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_GAIN);
        rms = cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_GAIN_RMS);
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "From whole detector fitting, "
        "determined a gain of %f +/- %f e/ADU.", gain, rms);
    }
    if (gain_out == NULL) {
        gain_out = cpl_vector_new(cpl_frameset_get_size(collapsed_frameset));
        cpl_vector_fill(gain_out, gain);
    }
    /*------------------------------------------------------------------
     -  Now calculate ron again for all frames indivudually.
     --------------------------------------------------------------------*/
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
            "Recalculating RON on all input frames...");

    if (vacca) {
        sph_gain_and_ron_get_vacca_factors(collapsed_frameset, &gain_factor,
                &ron_factor);
    }

    ron_out = cpl_vector_new(cpl_frameset_get_size(collapsed_frameset));

    for (ii = 0; ii < cpl_frameset_get_size(collapsed_frameset); ++ii) {
        aframe = cpl_frameset_get_position(collapsed_frameset, ii);
        if (gain_factor && ron_factor) {
            ronf = cpl_vector_get(ron_factor, ii);
            gainf = cpl_vector_get(gain_factor, ii);
        } else {
            ronf = 1.0;
            gainf = 1.0;
        }
        ron_mframe_tmp = sph_ron_simple_single(aframe, framecomb_parameterlist,
                cpl_vector_get(gain_out, ii), sigclip, gainf, ronf, &ron, &rms);
        if (ron_mframe_tmp) {
            SPH_ERROR_RAISE_INFO(
                    SPH_ERROR_GENERAL,
                    "Determined a ron of %f +/- %f e- for %s.", ron * cpl_vector_get(gain_out,ii), rms * cpl_vector_get(gain_out,ii), cpl_frame_get_filename(aframe));
            sprintf(key, "%s%d", SPH_COMMON_KEYWORD_QC_RON, ii + 1);
            cpl_vector_set(ron_out, ii, ron * cpl_vector_get(gain_out, ii));
            cpl_propertylist_append_double(pl, key,
                    ron * cpl_vector_get(gain_out, ii));
            sprintf(key, "%s%d", SPH_COMMON_KEYWORD_QC_RON_RMS, ii + 1);
            cpl_propertylist_append_double(pl, key,
                    rms * cpl_vector_get(gain_out, ii));
            if (ii == 0)
                ron_mframe = ron_mframe_tmp;
            else {
                sph_master_frame_multiply_double(ron_mframe, gain);
                sph_master_frame_add_master_frame(ron_mframe, ron_mframe_tmp);
                cc++;
                sph_master_frame_delete(ron_mframe_tmp);
            }
        } else {
            SPH_ERROR_ENSURE_GOTO_EXIT( 0, cpl_error_get_code());
        }
        ron_mframe_tmp = NULL;
    }
    if (cc)
        sph_master_frame_divide_double(ron_mframe, (double) cc);
    sph_master_frame_quality_check(ron_mframe);
    if (gain > 0.0) {
        cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_RON,
                cpl_vector_get_mean(ron_out));
        if (cpl_vector_get_size(ron_out) > 1) {
            cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_RON_RMS,
                    cpl_vector_get_stdev(ron_out));
        } else {
            cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_RON_RMS,
                    0.0);
        }
    }
    sph_master_frame_save_dfs(ron_mframe, ron_outfilename, inframes, NULL,
            inparams, taggain, recname, pipname, pl);
    EXIT: cpl_vector_delete(gain_factor);
    gain_factor = NULL;
    cpl_vector_delete(ron_factor);
    ron_factor = NULL;
    sph_master_frame_delete(ron_mframe);
    ron_mframe = NULL;
    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_polynomial_delete(poly);
    poly = NULL;
    cpl_vector_delete(gain_out);
    gain_out = NULL;
    cpl_vector_delete(ron_out);
    ron_out = NULL;
    cpl_frameset_delete(collapsed_frameset);
    collapsed_frameset = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/**@}*/
