/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_ifs_lenslet_model.h"
#include "sph_error.h"
#include "sph_iniparser.h"
#include "sph_image_grid.h"
#include "sph_master_frame.h"
#include "sph_paint_polygon.h"
#include "sph_dictionary.h"
#include "sph_keyword_manager.h"
#include "sph_common_keywords.h"
#include <cpl.h>
#include <math.h>
#include <limits.h>

const double SPH_IFS_INSTRUMENT_MODEL_DEROT_OFFSETSKY = 0.36 - 100.46;
const double SPH_IFS_INSTRUMENT_MODEL_DEROT_OFFSETELEV = 134.1 - 100.46;

const double SPH_IFS_LENSLET_MODEL_PIX_SIZE_MICRONS     = 18.0;
const double SPH_IFS_LENSLET_MODEL_DET_SIZE_PIX         = 2048;
const double SPH_IFS_LENSLET_MODEL_DET_SIZE_MICRONS     = 18.0 * 2048.0;
const double SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_MICRONS  = 39.0 * 18.0; /* 39.5 pixels length */
const double SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_MICRONS   = 5.09 * 18.0; /* 5.09 pixels width */
const double SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_MICRONS = 161.5;
const double SPH_IFS_LENSLET_MODEL_ROTANGLE             = -11.0;
const double SPH_IFS_LENSLET_MODEL_BIGRE_ROT_OFFSET     = -8.7691;
// The default arcsec / micron (on bigre)
const double SPH_IFS_LENSLET_MODEL_BIGRE_SCALE_DEFAULT  = 4.5957e-5;
const double SPH_IFS_LENSLET_MODEL_SCALING_X            = 1.0;
const double SPH_IFS_LENSLET_MODEL_SCALING_Y            = 1.0;
const double SPH_IFS_LENSLET_MODEL_MIN_LAMBDA_J         = 0.950 ;
const double SPH_IFS_LENSLET_MODEL_MAX_LAMBDA_J         = 1.346 ;
const double SPH_IFS_LENSLET_MODEL_MIN_LAMBDA_JH        = 0.953 ;
const double SPH_IFS_LENSLET_MODEL_MAX_LAMBDA_JH        = 1.677 ;
const int SPH_IFS_LENSLET_MODEL_LENSLETS_SIDE           = 145;          /* Number lenslets (side) */
const char* SPH_IFS_LENSLET_MODEL_SECNAME               = "ESO DRS IFS LENSLET MODEL";
const char* SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME         = "ESO DRS IFS PIX SIZE";
const char* SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME         = "ESO DRS IFS DET PIX SIZE";
const char* SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME     = "ESO DRS IFS LENS SIZE";
const char* SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME     = "ESO DRS IFS LENS N SIDE";
const char* SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME      = "ESO DRS IFS SPEC PIX LEN";
const char* SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_AC_NAME      = "ESO DRS IFS SPEC PIX LEN ACTUAL";
const char* SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME       = "ESO DRS IFS SPEC PIX WIDTH";
const char* SPH_IFS_LENSLET_MODEL_ROTANGLE_NAME         = "ESO DRS IFS ROTANGLE";
const char* SPH_IFS_LENSLET_MODEL_BIGRE_ROT_OFFSET_NAME = "ESO DRS IFS BIGRE ROT OFF";
const char* SPH_IFS_LENSLET_MODEL_BIGRE_SCALE_DEFAULT_NAME  = "ESO DRS IFS BIGRE SCALE";
const char* SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_X_NAME    = "ESO DRS IFS OFF X";
const char* SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_Y_NAME    = "ESO DRS IFS OFF Y";
const char* SPH_IFS_LENSLET_MODEL_SCALING_X_NAME        = "ESO DRS IFS SCALE X";
const char* SPH_IFS_LENSLET_MODEL_SCALING_Y_NAME        = "ESO DRS IFS SCALE Y";
const char* SPH_IFS_LENSLET_MODEL_MAX_LAMBDA_NAME       = "ESO DRS IFS MAX LAMBDA";
const char* SPH_IFS_LENSLET_MODEL_MIN_LAMBDA_NAME       = "ESO DRS IFS MIN LAMBDA";
const char* SPH_IFS_LENSLET_MODEL_DISPERSION_NAME       = "ESO DRS IFS DISPERSON";
const sph_error_code SPH_IFS_LENSLET_MODEL_GENERAL      = SPH_IFS_LENSLET_MODEL_ERR_START + 0;
const sph_error_code SPH_IFS_LENSLET_MODEL_OUTSIDE_ARRAY = SPH_IFS_LENSLET_MODEL_ERR_START + 1;
const double SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_Y = 8.0;
const double SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_X = 2.0;

static cpl_error_code
sph_ifs_lenslet_model_paint_specreg_ids__(cpl_image*, const sph_polygon*, int,
                                          const cpl_image* overlapim);
/*----------------------------------------------------------------------------*/
/**
 @brief Create a new default IFS lenslet model


 @return the new IFS lens model or NULL in case of error

 Description: This function creates a new IFS lens model in the standard
 default set-up. The parameters for the standard set-up are from the data sheet
 as found in document VLT-TRE-SPH-14690-0201, the IFS overview, version 1.0

 */
/*----------------------------------------------------------------------------*/
sph_ifs_lenslet_model* sph_ifs_lenslet_model_new(void) {
    sph_ifs_lenslet_model*        ifsmodel        = NULL;
    ifsmodel = cpl_calloc( 1, sizeof(sph_ifs_lenslet_model) );
    if ( ifsmodel ) {
        ifsmodel->current_vv = -ifsmodel->lenslets_per_side;
        ifsmodel->current_uu = 0;
        ifsmodel->detsize_microns = SPH_IFS_LENSLET_MODEL_DET_SIZE_MICRONS;
        ifsmodel->detsize_pixels = SPH_IFS_LENSLET_MODEL_DET_SIZE_PIX;
        ifsmodel->lensize_microns = SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_MICRONS;
        ifsmodel->lenslets_per_side = SPH_IFS_LENSLET_MODEL_LENSLETS_SIDE;
        ifsmodel->pixsize_microns = SPH_IFS_LENSLET_MODEL_PIX_SIZE_MICRONS;
        ifsmodel->speclength_microns = SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_MICRONS;
        ifsmodel->specwidth_microns = SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_MICRONS;
        ifsmodel->rotangle        = SPH_IFS_LENSLET_MODEL_ROTANGLE;
        ifsmodel->bigre_scale        = SPH_IFS_LENSLET_MODEL_BIGRE_SCALE_DEFAULT;
        ifsmodel->bigre_rot        = SPH_IFS_LENSLET_MODEL_BIGRE_ROT_OFFSET;
        ifsmodel->speclength_pixels =
                ifsmodel->speclength_microns /
                ifsmodel->pixsize_microns;
        ifsmodel->specwidth_pixels =
                ifsmodel->specwidth_microns /
                ifsmodel->pixsize_microns;
        ifsmodel->zero_offsetx = SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_X;
        ifsmodel->zero_offsety = SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_Y;
        ifsmodel->stretch_x    = SPH_IFS_LENSLET_MODEL_SCALING_X;
        ifsmodel->stretch_y    = SPH_IFS_LENSLET_MODEL_SCALING_Y;
        ifsmodel->maxlambda = SPH_IFS_LENSLET_MODEL_MAX_LAMBDA_J;
        ifsmodel->minlambda = SPH_IFS_LENSLET_MODEL_MIN_LAMBDA_J;
        ifsmodel->dispersion =
                ( ifsmodel->maxlambda - ifsmodel->minlambda ) /
                ( ifsmodel->speclength_pixels - 1 ); // The -2 due to difference
        // between actual and theoretical spectra length of 1 pixel.
        ifsmodel->pixelpoly = sph_polygon_new();
        sph_polygon_add_point(ifsmodel->pixelpoly,0.0,0.0);
        sph_polygon_add_point(ifsmodel->pixelpoly,1.0,0.0);
        sph_polygon_add_point(ifsmodel->pixelpoly,1.0,1.0);
        sph_polygon_add_point(ifsmodel->pixelpoly,0.0,1.0);
        ifsmodel->detector_distortion = NULL;
        ifsmodel->lenslet_distortion = NULL;
    }
    return ifsmodel;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Copy constructor
 * @param the lenslet model to copy
 *
 * @return the new lenslet model or NULL
 *
 */
/*----------------------------------------------------------------------------*/
sph_ifs_lenslet_model*
sph_ifs_lenslet_model_duplicate( sph_ifs_lenslet_model* old ) {
    cpl_propertylist*        templist    = NULL;
    sph_ifs_lenslet_model*    result        = NULL;
    if ( !old ) {
        SPH_NO_SELF;
        return NULL;
    }
    templist = sph_ifs_lenslet_model_get_as_propertylist( old );
    if ( !templist ) {
        SPH_NULL_ERROR
        return NULL;
    }
    result = sph_ifs_lenslet_model_new_from_propertylist( templist );
    cpl_propertylist_delete( templist ); templist = NULL;
    if ( old->lambdas ) result->lambdas = cpl_vector_duplicate(old->lambdas);
    else result->lambdas = NULL;
    if ( old->lenslet_distortion ) {
        templist =
                sph_distortion_model_get_proplist(
                        old->lenslet_distortion);
        result->lenslet_distortion =
            sph_distortion_model_new_from_proplist(
                                                              templist,
                                                              SPH_COMMON_KEYWORD_DISTMAP_COEFFX,
                                                              SPH_COMMON_KEYWORD_DISTMAP_COEFFY);
        cpl_propertylist_delete(templist); templist = NULL;
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Create a new IFS lenslet model from cpl propertylist

 @param the propertylist

 @return pointer to the new lenslet model or NULL in case of error.

 Description: This creates a new lenslet model, getting the data from a cpl
 propertylist. The propertylist must contain all the important parameters:
 <ul>
   <li>SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME        = "PIXEL SIZE"</li>
   <li>SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME        = "DETECTOR PIXEL SIZE"</li>
   <li>SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME    = "LENSLETS SIZE"</li>
   <li>SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME    = "LENSLETS ON SIDE"</li>
   <li>SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME     = "SPECTRA PIXEL LENGTH"</li>
   <li>SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME      = "SPECTRA PIXEL WIDTH"</li>
 </ul>

 */
/*----------------------------------------------------------------------------*/
sph_ifs_lenslet_model*
sph_ifs_lenslet_model_new_from_propertylist(
        const cpl_propertylist* plist )
{
    sph_ifs_lenslet_model*        model        = NULL;
    int                   ival      = 0;
    double                dval      = 0.0;
    cpl_errorstate        prestate  = cpl_errorstate_get();
    if ( !plist ) {
        return NULL;
    }

    cpl_msg_info(cpl_func, "Creating lenslet model from %d key(s)",
                    (int)cpl_propertylist_get_size(plist));

    model = sph_ifs_lenslet_model_new();

    if ( model ) {
        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME );
        if ( dval ) model->pixsize_microns = dval;

        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_BIGRE_SCALE_DEFAULT_NAME );
        if ( dval ) model->bigre_scale = dval;

        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_BIGRE_ROT_OFFSET_NAME );
        if ( dval ) model->bigre_rot = dval;


        ival = cpl_propertylist_get_int( plist,
                SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME );
        if ( ival ) {
            model->detsize_pixels = ival;
            model->detsize_microns = ival * model->pixsize_microns;
        }


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME );
        if ( dval ) model->lensize_microns = dval;


        ival = cpl_propertylist_get_int( plist,
                SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME );
        if ( ival ) model->lenslets_per_side = ival;


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME );
        if ( dval ) {
            model->speclength_pixels = dval;
            model->speclength_microns = dval * model->pixsize_microns;
        }


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME );
        if ( dval ) {
            model->specwidth_pixels = dval;
            model->specwidth_microns = dval * model->pixsize_microns;
        }


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_ROTANGLE_NAME );
        if ( dval ) {
            model->rotangle = dval;
            model->stretch_x = 2.0 * model->specwidth_microns /
                    ( CPL_MATH_SQRT3 * model->lensize_microns );
            model->stretch_y = 1.05 * model->speclength_microns /
                    ( 3.0 * model->lensize_microns ) ;
        }


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_X_NAME );
        if ( dval ) model->zero_offsetx = dval;


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_Y_NAME );
        if ( dval ) model->zero_offsety = dval;


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_SCALING_X_NAME );
        if ( dval ) model->stretch_x = dval;


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_SCALING_Y_NAME );
        if ( dval ) model->stretch_y = dval;


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_MAX_LAMBDA_NAME );
        if ( dval ) model->maxlambda = dval;


        dval = cpl_propertylist_get_double( plist,
                SPH_IFS_LENSLET_MODEL_MIN_LAMBDA_NAME );
        if ( dval ) model->minlambda = dval;

        model->dispersion = ( model->maxlambda - model->minlambda ) /
                ( model->speclength_pixels - 1 );
        if (cpl_propertylist_has(plist,
                                  SPH_COMMON_KEYWORD_DISTMAP_COEFFX " 0_0")) {
            model->detector_distortion =
                    sph_distortion_model_new_from_proplist(plist,
                                                              SPH_COMMON_KEYWORD_DISTMAP_COEFFX,
                                                              SPH_COMMON_KEYWORD_DISTMAP_COEFFY);
        }
    }
    if (!cpl_errorstate_is_equal(prestate)) {
        cpl_errorstate_dump(prestate, CPL_FALSE, cpl_errorstate_dump_one_info);
        cpl_errorstate_set(prestate);
    }
    return model;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Load a new IFS lenslet model from a FITS or ASCII ini file

 @param the filename of the ASCII file or FITS file

 @return pointer to the new lenslet model or NULL in case of error.

 Description:
 This creates a new lenslet model, loading the data from FITS header or
 an ASCII file that is in an ini keyword = value type file. The section has to be entitled
 [ LENSLET MODEL ].

 Note: currently no distortion parameters are saved!
 */
/*----------------------------------------------------------------------------*/
sph_ifs_lenslet_model*
sph_ifs_lenslet_model_load( const char* czFilename )
{
    sph_ifs_lenslet_model*          model         = NULL;
    sph_dictionary*                     ini           = NULL;
    char                            keyup[2048];
    int                             ival      = 0;
    double                          dval      = 0.0;
    cpl_propertylist*               plist     = NULL;

    plist = sph_keyword_manager_load_properties( czFilename, 0 );
    if ( plist ) {
        model = sph_ifs_lenslet_model_new_from_propertylist(plist);
        cpl_propertylist_delete(plist); plist = NULL;
        if ( ! model ) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO,
                    "Could not load FITS info "
                    "for IFS instrument model from file %s",czFilename);
            return NULL;
        }
        return model;
    }
    cpl_error_reset();

#ifdef SPH_NO_ASCII
    SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO,
            "Could not load header info "
            "for IFS instrument model from file %s",czFilename);
    return NULL;
#endif

    model = sph_ifs_lenslet_model_new();
    if ( model ) {
        ini = sph_iniparser_load( czFilename );
        if ( !sph_iniparser_find_entry( ini,
                SPH_IFS_LENSLET_MODEL_SECNAME )
                )
        {
            sph_ifs_lenslet_model_delete( model );
            return NULL;
        }
        sprintf( keyup, "%s:%s",
                SPH_IFS_LENSLET_MODEL_SECNAME,
                SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->pixsize_microns = dval;

        sprintf( keyup, "%s:%s",
                SPH_IFS_LENSLET_MODEL_SECNAME,
                SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME );
        ival = sph_iniparser_getint( ini, keyup, 0.0 );
        if ( ival ) {
            model->detsize_pixels = ival;
            model->detsize_microns = ival * model->pixsize_microns;
        }

        sprintf( keyup, "%s:%s",
                SPH_IFS_LENSLET_MODEL_SECNAME,
                SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
         if ( dval ) model->lensize_microns = dval;

        sprintf( keyup, "%s:%s",
                SPH_IFS_LENSLET_MODEL_SECNAME,
                SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME );
        ival = sph_iniparser_getint( ini, keyup, 0.0 );
        if ( ival ) model->lenslets_per_side = ival;

        sprintf( keyup, "%s:%s",
                SPH_IFS_LENSLET_MODEL_SECNAME,
                SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) {
            model->speclength_pixels = dval;
            model->speclength_microns = dval * model->pixsize_microns;
        }

        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) {
            model->specwidth_pixels = dval;
            model->specwidth_microns = dval * model->pixsize_microns;
        }

        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_ROTANGLE_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) {
            model->rotangle = dval;
            model->stretch_x = 2.0 * model->specwidth_microns / ( CPL_MATH_SQRT3 * model->lensize_microns );
            model->stretch_y = 1.05 * model->speclength_microns / ( 3.0 * model->lensize_microns ) ;
        }


        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_BIGRE_ROT_OFFSET_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) {
            model->bigre_rot = dval;
        }

        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_BIGRE_SCALE_DEFAULT_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) {
            model->bigre_scale = dval;
        }


        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_X_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->zero_offsetx = dval;

        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_Y_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->zero_offsety = dval;

        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_SCALING_X_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->stretch_x = dval;

        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_SCALING_Y_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->stretch_y = dval;

        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_MAX_LAMBDA_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->maxlambda = dval;

        sprintf( keyup, "%s:%s",SPH_IFS_LENSLET_MODEL_SECNAME, SPH_IFS_LENSLET_MODEL_MIN_LAMBDA_NAME );
        dval = sph_iniparser_getdouble( ini, keyup, 0.0 );
        if ( dval ) model->minlambda = dval;

        model->dispersion = ( model->maxlambda - model->minlambda ) /
        		( model->speclength_pixels - 1 );
        sph_iniparser_freedict(ini);
    }
    SPH_RAISE_CPL;
    return model;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Return the total number of lenslets

 @param self            the lenslet model to return the number of lenslets for

 @return the number of lenslets or -1 on error.

 Description: Returns the total number of lenslets

 */
/*----------------------------------------------------------------------------*/
int
sph_ifs_lenslet_model_get_nlens( sph_ifs_lenslet_model* self ) {
    int                    nrings        = 0;
    int                    nlens        = 0;
    if ( !self ) {
        return -1;
    }
    nrings = self->lenslets_per_side;
    nlens = 1 + 3 * nrings * nrings + 3 * nrings;
    return nlens;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Save the lenslet model to a ASCII file in ini format.

 @param self        the lenslet model to save
 @param czFilename    the filename to save to

 @return error code of the operation

 Description: Saves the lenslet model to a ASCII file in .ini style format.
 The file is created, existing files are overwritten. The file contains
 only one section and under that all the parameters for the model in the
 key = value format.
 NOTE: currently no distortion parameters are saved!

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ifs_lenslet_model_save( const sph_ifs_lenslet_model* self,
                            const char* czFilename )
{
    sph_error_code            rerr        = CPL_ERROR_NONE;
    FILE*                    pFile        = NULL;

    cpl_ensure_code(self       != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(czFilename != NULL, CPL_ERROR_NULL_INPUT);

    pFile = fopen( czFilename, "w" );

    cpl_ensure_code(pFile      != NULL, CPL_ERROR_ASSIGNING_STREAM);

    if (fprintf( pFile, "[ %s ]\n", SPH_IFS_LENSLET_MODEL_SECNAME ) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME,
                 self->detsize_pixels ) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME,
                 self->lenslets_per_side ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME,
                 self->pixsize_microns ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME,
                 self->lensize_microns ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME,
                 self->speclength_pixels) <= 0 ||
        fprintf( pFile, "%s = %d\n", SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_AC_NAME,
                 (int)(self->speclength_pixels - 0.0001)) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME,
                 self->specwidth_pixels ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_ROTANGLE_NAME,
                 self->rotangle ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_BIGRE_SCALE_DEFAULT_NAME,
                 self->bigre_scale ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_BIGRE_ROT_OFFSET_NAME,
                 self->bigre_rot ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_X_NAME,
                 self->zero_offsetx ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_Y_NAME,
                 self->zero_offsety ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_SCALING_X_NAME,
                 self->stretch_x ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_SCALING_Y_NAME,
                 self->stretch_y ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_MAX_LAMBDA_NAME,
                 self->maxlambda ) <= 0 ||
        fprintf( pFile, "%s = %f\n", SPH_IFS_LENSLET_MODEL_MIN_LAMBDA_NAME,
                 self->minlambda ) <= 0) {
        rerr = cpl_error_set(cpl_func, CPL_ERROR_FILE_IO);
    } 

    cpl_ensure_code(fclose( pFile ) == 0, CPL_ERROR_FILE_IO);

    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the instrument lenslet model as a propertylist.

 @param self            the lenslet model

 @return    the propertylist or NULL in case of error.

 Description: This creates a new propertylist containing all the properties of the
 lenslet model.

 */
/*----------------------------------------------------------------------------*/
cpl_propertylist*
sph_ifs_lenslet_model_get_as_propertylist(const sph_ifs_lenslet_model* self )
{
    cpl_propertylist*        plist;
    sph_error_code            rerr        = CPL_ERROR_NONE;

    if ( !self ) {
        return NULL;
    }
    plist = cpl_propertylist_new();
    rerr |= cpl_propertylist_append_int( plist,
                                         SPH_IFS_LENSLET_MODEL_DET_SIZE_NAME, self->detsize_pixels );

    rerr |= cpl_propertylist_append_int( plist,
                                         SPH_IFS_LENSLET_MODEL_LENSLET_SIDE_NAME, self->lenslets_per_side );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_NAME, self->lensize_microns );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_PIX_SIZE_NAME, self->pixsize_microns );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_NAME, self->speclength_pixels );
    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_AC_NAME, (int)(self->speclength_pixels - 0.0001 ) );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_NAME, self->specwidth_pixels );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_ROTANGLE_NAME, self->rotangle );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_BIGRE_SCALE_DEFAULT_NAME, self->bigre_scale );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_BIGRE_ROT_OFFSET_NAME, self->bigre_rot );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_X_NAME, self->zero_offsetx );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_ZERO_OFFSET_Y_NAME, self->zero_offsety );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_SCALING_X_NAME, self->stretch_x );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_SCALING_Y_NAME, self->stretch_y );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_MAX_LAMBDA_NAME, self->maxlambda );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_MIN_LAMBDA_NAME, self->minlambda );

    rerr |= cpl_propertylist_append_double( plist,
                                            SPH_IFS_LENSLET_MODEL_DISPERSION_NAME, self->dispersion );

    if ( self->detector_distortion ) {
        cpl_propertylist* dumlist = 
            sph_distortion_model_get_proplist
            (self->detector_distortion);
        rerr |= cpl_propertylist_append(plist, dumlist);
        cpl_propertylist_delete(dumlist);
    }

    if (rerr) {
        (void)cpl_error_set_where(cpl_func);
        cpl_propertylist_delete(plist);
        plist = NULL;
    }

    return plist;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return wavelength inside a spectral region
 *
 * @param self
 * @param poly    can be NULL
 * @param uu     the uu of spectral region
 * @param vv    the    vv of spectral region
 * @param ypos    the y pixel position
 *
 *
 * This returns the wavelength at a position y assuming
 * the pixel is associated (inside) the spectral region with
 * uu and vv.
 * The function does not check that the pixel is actually
 * inside the spectra region but simply calculates the
 * distance of y to the lower edge of the spectral region
 * specified and then calculate the wavelength from that.
 */
/*----------------------------------------------------------------------------*/
double
sph_ifs_lenslet_model_get_lambda(
        sph_ifs_lenslet_model* self,
        sph_polygon* poly,
        int uu, int vv,
        double    ypos)
{
    double                result            = 0.0;

    if ( poly == NULL ) {
        poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(
                self,uu,vv,0.0,0.0);
    }
    // The 0.5 since minlambda refers to lambda at pixel midpoint
    result = ( ypos - 0.5 - sph_polygon_get_bottom(poly) ) *
            self->dispersion +
            self->minlambda;
    return result;
}


/*----------------------------------------------------------------------------*/
/**
 @brief Return the instrument lenslet model wavelength vector.

 @param self            the lenslet model

 @return    the vector or NULL in case of error.

 Description: Creates and returns a new cpl_vector containing all the
wavelength points for the lenslet model. These can be considered then
the wavelength values for the "ideal" instrument spectrum.
 */
/*----------------------------------------------------------------------------*/
cpl_vector*
sph_ifs_lenslet_model_get_lambda_vector( sph_ifs_lenslet_model* self )
{
  int            ii = 0;
  double         dl = 0.0;
  sph_error_code rerr = CPL_ERROR_NONE;
  if ( !self ) {
    SPH_NULL_ERROR;
    return NULL;
  }

  if ( self->lambdas ) return self->lambdas;

  if ( self->speclength_pixels < 2 ) {
      SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,"Speclength is less than 2 pixels.");
      return NULL;
  }

  self->lambdas = cpl_vector_new( (int)self->speclength_pixels );
  if ( !self->lambdas ) {
    return NULL;
  }
  dl = self->dispersion;
  for ( ii = 0; ii < cpl_vector_get_size( self->lambdas ); ++ii ) {
    rerr |= cpl_vector_set( self->lambdas, ii, self->minlambda + dl * ii );
  }
  if ( rerr != CPL_ERROR_NONE ) {
    sph_error_raise( SPH_ERROR_GENERAL,
                     __FILE__,
                     __func__,
                     __LINE__,
                     SPH_ERROR_ERROR,
                     "Could not set all wavelegnths in"
                     " wavelength vector for lensmodel."
                     );
    SPH_RAISE_CPL;
    cpl_vector_delete( self->lambdas ); self->lambdas = NULL;
    return NULL;
  }
  return self->lambdas;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create all images necessary to build pdt
 * @param self    the lenslet model
 * @param pimage_wav output wavelength image
 * @param pimage_ids output lenslet ids image
 * @param pimage_ill output illumination image
 * @param pimage_dw output differential wavelength image
 *
 * @return error code
 *
 * This function creates all the images necessary for creation of a PDT.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ifs_lenslet_model_create_pdt_images(
        sph_ifs_lenslet_model* self,
        cpl_image** pimage_wav,
        cpl_image** pimage_ids,
        cpl_image** pimage_ill,
        cpl_image** pimage_dw,
        double dx,
        double dy )
{
    sph_master_frame*    mframe_ill    = NULL;
    sph_master_frame*    mframe_ids    = NULL;
    sph_master_frame*    mframe_wav    = NULL;
    cpl_image*            overlapim    = NULL;
    sph_polygon*        poly = NULL;
    sph_polygon_workspace*        ws = NULL;
    int                    vv        = 0;
    int                 nrings    = 0;
    int                 uu        = 0;
    int                    nlens        = 0;
    int                    cc                 = 0;


    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    ws = sph_polygon_workspace_new();

    nrings = self->lenslets_per_side;

    nlens = 1 + 3 * nrings * nrings + 3 * nrings;


    mframe_ill = sph_master_frame_new( self->detsize_pixels,
            self->detsize_pixels );

    if ( pimage_ids ) *pimage_ids = cpl_image_new( self->detsize_pixels,
            self->detsize_pixels, CPL_TYPE_INT );


    mframe_wav = sph_master_frame_new( self->detsize_pixels,
            self->detsize_pixels );
    for ( vv = -nrings;  vv <= nrings; ++vv ) {
        if ( vv >=0 ) {
            for ( uu = -nrings; uu <= nrings - vv; ++uu ) {
                cc = sph_ifs_lenslet_model_get_index( self, uu, vv );
                poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(self,uu,vv,dx,dy);
                overlapim = sph_paint_polygon_get_overlap_image(mframe_ill,poly,ws);
                sph_paint_polygon_on_master(mframe_ill,poly,1.0,0.0,0.0,0.0,overlapim);
                sph_ifs_lenslet_model_set_wavs_specregion( self, poly,
                        uu, vv, mframe_wav);
                if ( pimage_ids ) sph_ifs_lenslet_model_paint_specreg_ids__(*pimage_ids,poly,cc,overlapim);
                //sph_master_frame_paint_polygon_image_avg(mframe_ids,poly,cc);//,0.0,0.0,0.0,overlapim);
                cpl_image_delete(overlapim); overlapim = NULL;
            }
        }
        else {
            for (uu = 0 - ( nrings + vv ); uu <= nrings; ++uu) {
                cc = sph_ifs_lenslet_model_get_index( self, uu, vv );
                poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(self,uu,vv,dx,dy);
                overlapim = sph_paint_polygon_get_overlap_image(mframe_ill,poly,ws);
                sph_paint_polygon_on_master(mframe_ill,poly,1.0,0.0,0.0,0.0,overlapim);
                sph_ifs_lenslet_model_set_wavs_specregion( self,poly,
                        uu, vv, mframe_wav);
                if ( pimage_ids ) sph_ifs_lenslet_model_paint_specreg_ids__(*pimage_ids,poly,cc,overlapim);
                //sph_master_frame_paint_polygon_image_avg(mframe_ids,poly,cc);//,0.0,0.0,0.0,overlapim);
                cpl_image_delete(overlapim); overlapim = NULL;
            }
        }
    }

    cpl_image_divide_scalar(mframe_ill->image,cpl_image_get_max(mframe_ill->image));
    if ( pimage_ill ) *pimage_ill = cpl_image_duplicate(mframe_ill->image);
    cpl_image_threshold(mframe_ill->image,0.9999,1.0,0.0,1.0);
    if ( pimage_wav ) *pimage_wav = cpl_image_multiply_create(mframe_wav->image,mframe_ill->image);
    //*pimage_ids = cpl_image_cast(mframe_ids->image,CPL_TYPE_INT);
    sph_master_frame_delete(mframe_ill);
    sph_master_frame_delete(mframe_ids);
    sph_master_frame_delete(mframe_wav);

    /* Alternative way of doing it: This way seems better when there
     * are spectra overlapping. However: it seems to change spectra also
     * in another way so that the minimum wavelength is badly associated, in
     * particular one of the tests in cutest_sph_pixel_description_table fails.
     * if the below method is used.
     *
    if ( cpl_image_get_median(mframe_ill->image) < 1.0 && cpl_image_get_median(mframe_ill->image) > 0.0 )
    {
        cpl_image_divide_scalar(mframe_ill->image,cpl_image_get_median(mframe_ill->image));
    }
    if ( cpl_image_get_median(mframe_ill->image) <= 0.0 && cpl_image_get_mean(mframe_ill->image) > 0.0 )
    {
        cpl_image_divide_scalar(mframe_ill->image,cpl_image_get_mean(mframe_ill->image));
    }

    *pimage_ill = cpl_image_duplicate(mframe_ill->image);
    // Only want to keep the exact 1's.
    cpl_image_threshold(mframe_ill->image,0.999,1.01,0.0,0.0);
    *pimage_ids = cpl_image_multiply_create(mframe_ids->image,mframe_ill->image);
    *pimage_wav = cpl_image_multiply_create(mframe_wav->image,mframe_ill->image);
    sph_master_frame_delete(mframe_ill);
    sph_master_frame_delete(mframe_ids);
    sph_master_frame_delete(mframe_wav);
     *
     */

    if ( pimage_ill ) cpl_image_threshold(*pimage_ill,0.0,1.0,0.0,1.0);

    if ( pimage_dw) {
        *pimage_dw = cpl_image_duplicate(*pimage_wav);
        cpl_image_shift(*pimage_dw,0,-1);
        cpl_image_subtract(*pimage_dw,*pimage_wav);
    }

    sph_polygon_workspace_delete( ws );
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

sph_point_pattern*
sph_ifs_lenslet_model_predict_spec_loc_pp(
        sph_ifs_lenslet_model* self,
        double dx, double dy )
{
    sph_polygon*        poly = NULL;
    int                 vv        = 0;
    int                 nrings    = 0;
    int                 uu        = 0;
    double              speclength = SPH_IFS_LENSLET_MODEL_SPEC_LENGTH_MICRONS;
    double              specwidth  = SPH_IFS_LENSLET_MODEL_SPEC_WIDTH_MICRONS;
    int                 nlens        = 0;
    double              slength        = 0.0;
    double              stretch_x         = 1.0;
    double              stretch_y         = 1.0;
    int                 cc                 = 0.0;
    sph_point_pattern*	pp = NULL;

    if ( !self ) {
        return NULL;
    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
      SPH_RAISE_CPL_RESET;
    }

    pp = sph_point_pattern_new();
    stretch_x = self->stretch_x;
    stretch_y = self->stretch_y;
    nrings = self->lenslets_per_side;

    nlens = 1 + 3 * nrings * nrings + 3 * nrings;

    slength = self->lensize_microns;

    speclength = self->speclength_microns;
    specwidth = self->specwidth_microns;
    for ( vv = -nrings;  vv <= nrings; ++vv ) {
        if ( vv >=0 ) {
            for ( uu = -nrings; uu <= nrings - vv; ++uu ) {
                double cx, cy;
                cc = sph_ifs_lenslet_model_get_index( self, uu, vv );
                poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(self,uu,vv,dx,dy);
                sph_polygon_get_midxy_(poly, &cx, &cy);
                sph_point_pattern_add_point(pp, cx, cy);
            }
        }
        else {
            for (uu = 0 - ( nrings + vv ); uu <= nrings; ++uu) {
                double cx, cy;
                cc = sph_ifs_lenslet_model_get_index( self, uu, vv );
                poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(self,uu,vv,dx,dy);
                sph_polygon_get_midxy_(poly, &cx, &cy);
                sph_point_pattern_add_point(pp, cx, cy);
            }
        }
    }
    return pp;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Get u and v of first lenslet .

 @param self            the lenslet model
 @param u               output u
 @param v               output v

 Description: Returns the first u and v. Use sph_ifs_lenslet_model_get_next
 to obtain further ones.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ifs_lenslet_model_get_first( sph_ifs_lenslet_model* self,
        int* u, int* v )
{
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);

    self->current_vv = -self->lenslets_per_side;
    self->current_uu = 0;
    *u=self->current_uu;
    *v=self->current_vv;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Get u and v of first lenslet .

 @param self            the lenslet model
 @param u               output u
 @param v               output v

 Description: Returns the next u and v. Use sph_ifs_lenslet_model_get_first
 to get the first and reset the iterator.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ifs_lenslet_model_get_next( sph_ifs_lenslet_model* self,
        int* u, int* v )
{
    int nrings = 0;
    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);

    nrings = self->lenslets_per_side;

    if ( self->current_vv >=0 ) {
        self->current_uu++;
        if ( self->current_uu > nrings - self->current_vv ) {
            self->current_vv++;
            self->current_uu = -nrings;
        }
    }
    else {
        self->current_uu++;
        if ( self->current_uu > nrings) {
            self->current_vv++;
            self->current_uu = 0 - ( nrings + self->current_vv);
        }
    }
    if ( self->current_vv > self->lenslets_per_side) return CPL_ERROR_DATA_NOT_FOUND;
    *u=self->current_uu;
    *v=self->current_vv;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return polygon of spectral area

 @param self            the lenslet description table
 @param u                the u-position of the hex to obtain
 @param v                the v-position of the hex to obtain.

 @return a new polygon representation of the hexagon lenslet spectra projection area

 Description: Returns the polygon (rectangle) of the projected area on the detector
 of the chosen hexagon at u and v. Coordinates
 are such that the central lenslet has u=0, v=0 and dither position 0 is assumed
 for the detector dither position.

 */
/*----------------------------------------------------------------------------*/
sph_polygon*
sph_ifs_lenslet_model_get_spec_region( sph_ifs_lenslet_model* self, int uu, int vv) {
    double                detx        = 0.0;
    double                dety        = 0.0;
    double                 slength        = 0.0;
    double                specwidth    = 0.0;
    double                speclength    = 0.0;
    double                stretch_x        = 1.0;
    double                stretch_y        = 1.0;
    sph_point*            point           = NULL;
    if ( !self ) {
        return NULL;
    }
    slength = self->lensize_microns;
    speclength = self->speclength_microns;
    specwidth = self->specwidth_microns;
    stretch_x = self->stretch_x;
    stretch_y = self->stretch_y;
    self->pixelpoly->points[0].x = -0.5 * specwidth;
    self->pixelpoly->points[0].y = -0.5 * speclength;
    self->pixelpoly->points[1].x = 0.5 *  specwidth;
    self->pixelpoly->points[1].y = -0.5 * speclength;
    self->pixelpoly->points[2].x = 0.5 * specwidth;
    self->pixelpoly->points[2].y = 0.5 * speclength ;
    self->pixelpoly->points[3].x = -0.5 * specwidth;
    self->pixelpoly->points[3].y = 0.5 * speclength ;
    detx = uu * CPL_MATH_SQRT3 * slength + 0.5 * CPL_MATH_SQRT3 * vv * slength;
    dety = 1.5 * vv * slength;
    detx = detx * self->stretch_x;
    dety = dety * self->stretch_y;
    point = sph_point_new(detx,dety);
    sph_point_rotate_around(point,0.0,0.0,cos(self->rotangle * CPL_MATH_RAD_DEG),sin(self->rotangle * CPL_MATH_RAD_DEG),0.0,0.0);
    sph_polygon_shift(  self->pixelpoly, point->x, point->y );
    sph_point_delete(point);
    return self->pixelpoly;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Return polygon of spectral area in pixel coordinates

 @param self            the lenslet description table
 @param u                the u-position of the hex to obtain
 @param v                the v-position of the hex to obtain.
 @param dx                an optional dithering offset
 @param dy                an optional dithering offset

 @return a new polygon representation of the hexagon lenslet spectra projection area

 Description: Returns the polygon (rectangle) of the projected area on the detector
 of the chosen hexagon at u and v. Coordinates
 are such that the central lenslet has u=0, v=0 and dither position 0 is assumed
 for the detector dither position.

 */
/*----------------------------------------------------------------------------*/
sph_polygon*
sph_ifs_lenslet_model_get_spec_region_pixel_coords(
        sph_ifs_lenslet_model* self, int uu, int vv,
        double dx, double dy )
{
    int                    nn        = 0;
    double                  ddx = 0.0;
    double                  ddy = 0.0;
    sph_polygon*        poly    = NULL;

    poly = sph_ifs_lenslet_model_get_spec_region(self,uu,vv);
    for (nn = 0; nn < poly->npoints; ++nn) {
        poly->points[nn].x/=self->pixsize_microns;
        poly->points[nn].y/=self->pixsize_microns;
        poly->points[nn].x+=self->detsize_pixels/2.0;
        poly->points[nn].y+=self->detsize_pixels/2.0;
    }
    sph_polygon_shift( poly, self->zero_offsetx + dx,
            self->zero_offsety + dy );
    if ( self->detector_distortion ) {
        double cxy[2] = {0.0, 0.0};
        double ddxy[2];
        sph_polygon_get_midxy_(poly, cxy, cxy+1);

        sph_distortion_model_get_dist(self->detector_distortion, cxy, ddxy);

        // Distortion map defined in the case here
        // as model minus osberved, so swap sign
        sph_polygon_shift( poly, -ddxy[0], -ddxy[1]);
    }

    return poly;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Return polygon

 @param self            the lenslet description table
 @param u                the u-position of the hex to obtain
 @param v                the v-position of the hex to obtain.

 @return a new polygon representation of the hexagon lenslet

 Description: Returns the polygon of the chosen hexagon at u and v. Coordinates
 are such that the central lenslet has u=0, v=0.

 */
/*----------------------------------------------------------------------------*/
sph_polygon*
sph_ifs_lenslet_model_get_poly( sph_ifs_lenslet_model* self, int u, int v) {
    sph_polygon*        hex        = NULL;
    double                t        = 0.0;
    double              ddx = 0.0;
    double              ddy = 0.0;

    if ( !self ) {
        return NULL;
    }
    t = self->lensize_microns;
    hex = sph_polygon_new( );
    sph_polygon_add_point( hex, 0.0, -t );
    sph_polygon_add_point( hex, 0.5 * CPL_MATH_SQRT3 * t, -0.5 * t );
    sph_polygon_add_point( hex, 0.5 * CPL_MATH_SQRT3 * t, 0.5 * t );
    sph_polygon_add_point( hex, 0.0, t );
    sph_polygon_add_point( hex, -0.5 * CPL_MATH_SQRT3 * t, 0.5 * t );
    sph_polygon_add_point( hex, -0.5 * CPL_MATH_SQRT3 * t, -0.5 * t );
    sph_polygon_shift(hex, u * CPL_MATH_SQRT3 * t + 0.5 * CPL_MATH_SQRT3 * v * t, 1.5 * v * t);
    if ( self->lenslet_distortion ) {
        double cxy[2] = {0.0, 0.0};
        double ddxy[2];
        sph_polygon_get_midxy_(hex, cxy, cxy + 1);
        sph_distortion_model_get_dist(self->lenslet_distortion, cxy, ddxy);
        sph_polygon_shift(hex, ddxy[0], ddxy[1]);
    }

    return hex;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return the centre coordinates of hexagon
 * @param self      the model
 * @param u         the u coordinate
 * @param v         the v coordinate
 * @param x         the centre x coordinate (output)
 * @param y         the centre y coordinate (output)
 * @return error code
 *
 * Fills x and y with the coordinates of the middle of the hexagon
 * of the lenslet at u,v.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_ifs_lenslet_model_get_centre(const sph_ifs_lenslet_model* self,
        int u, int v,
        double* x, double *y
        )
{
    double                t        = 0.0;
    double                ddx      = 0.0;
    double                ddy      = 0.0;
    cpl_ensure_code(self,CPL_ERROR_NONE);
    cpl_ensure_code(x,CPL_ERROR_NONE);
    cpl_ensure_code(y,CPL_ERROR_NONE);
    t = self->lensize_microns;
    *x = u * CPL_MATH_SQRT3 * t + 0.5 * CPL_MATH_SQRT3 * v * t;
    *y = 1.5 * v * t;

    if ( self->lenslet_distortion ) {
        double cxy[2] = {*x, *y};
        double ddxy[2];

        sph_distortion_model_get_dist(self->lenslet_distortion, cxy, ddxy);

        *x += ddxy[0];
        *y += ddxy[1];
    }

    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Return index of lenslet at u and v

 @param self            the lenslet description table
 @param u                the u-position of the hex
 @param v                the v-position of the hex

 @return index of the hex or -1 on error.

 Description: Returns the index of the hex at u and v. Returns -1 if it is
 outside the array.
 */
/*----------------------------------------------------------------------------*/
int sph_ifs_lenslet_model_get_index(sph_ifs_lenslet_model* self, int u, int v)
{
    int result = -1;
    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else {
        const int    nrings = self->lenslets_per_side;
        const int    nside  = 2 * nrings + 1;

        if (self->easymap == NULL) {
            int  cc    = 0;
            int* pimg  = (int*)cpl_malloc((size_t)(nside * nside) *
                                          sizeof(*pimg));

            self->easymap  = cpl_image_wrap_int(nside, nside, pimg);
            self->peasymap = pimg;

            for (int vv = -nrings; vv <= nrings; vv++) {
                if ( vv >=0 ) {
                    for (int uu = -nrings; uu <= nrings - vv; uu++) {
                        pimg[uu + nrings + (vv + nrings) * nside] = ++cc;
                    }
                } else {
                    for (int uu = 0 - ( nrings + vv ); uu <= nrings; uu++) {
                        pimg[uu + nrings + (vv + nrings) * nside] = ++cc;
                    }
                }
            }
        }

        result = self->peasymap[u + nrings + (v + nrings) * nside];
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return u and v coordinates of point

 @param self            the lenslet description table
 @param x                the x-coordinate on lenslet array in microns
 @param y                the y-coordinate on lenslet array in microns
 @param u                the u-position of the hex (output)
 @param v                the v-position of the hex (output).

 @return error code of the operation

 Description: Fills the variables u and v with the u and v index of the hexagon
 that contains the given point. Returns SPH_IFS_LENSLET_MODEL_OUTSIDE_ARRAY if
 no hex contains the point.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ifs_lenslet_model_get_coords( sph_ifs_lenslet_model* self, double xp, double yp, int* u, int* v) {
    sph_polygon*        hex        = NULL;
    double                t        = 0.0;
    double                 z        = 0.0;
    double                dx        = 0.0;
    double                dy        = 0.0;
    double                x        = 0.0;
    double                y        = 0.0;
    int                    ix        = 0;
    int                    iy        = 0;
    int                    iz        = 0;
    int                    s        = 0;
    double                 abs_dx    = 0.0;
    double                 abs_dy    = 0.0;
    double                 abs_dz    = 0.0;
    int                    inside    = 0;
    // NOTE:  Got this code from the web somewhere...
    //

    if ( !self ) {
        return CPL_ERROR_NULL_INPUT;
    }
    t = self->lensize_microns;
    dx = CPL_MATH_SQRT3 * t;
    dy = 1.5 * t;
    x = 1.0 * ( yp - 0.0 ) / dy;
    y = 1.0 * ( xp - 0.0 ) / dx;
    z = -0.5 * x - y;
    y = -0.5 * x + y;
    ix = floor(x+0.5);
    iy = floor(y+0.5);
    iz = floor(z+0.5);
    s = ix+iy+iz;
    if( s )
    {
        abs_dx = fabs(ix-x);
        abs_dy = fabs(iy-y);
        abs_dz = fabs(iz-z);
        if( abs_dx >= abs_dy && abs_dx >= abs_dz )
            ix -= s;
        else if( abs_dy >= abs_dx && abs_dy >= abs_dz )
            iy -= s;
        else
            iz -= s;
    }
    if ((s = iy - iz) < 0)
        iy = s - 1 + ((ix+1) & 1); // this should be !(ix&1), but I
    else                         // haven't checked it, and it might
        iy = s + 1 - (ix & 1);     // be compiler dependant.

    *u = ( iy - ix ) / 2;
    *v = ix;
    hex = sph_ifs_lenslet_model_get_poly( self, *u, *v );
    inside = sph_polygon_test_inside( hex, xp, yp );
    if ( !inside ) {
        if ( *v < 0 ) *u = *u + 1;
        else *u = *u - 1;
    }
    if ( *v < -self->lenslets_per_side || *v > self->lenslets_per_side ) {
        sph_polygon_delete(hex); hex = NULL;
        return SPH_IFS_LENSLET_MODEL_OUTSIDE_ARRAY;
    }
    if ( *v >= 0 ) {
        if( *u < -self->lenslets_per_side || *u > self->lenslets_per_side - *v ) {
            sph_polygon_delete(hex); hex = NULL;
            return SPH_IFS_LENSLET_MODEL_OUTSIDE_ARRAY;
        }
    }
    if ( *v < 0 ) {
        if( *u < 0 - ( self->lenslets_per_side + *v ) || *u > self->lenslets_per_side  ) {
            sph_polygon_delete(hex); hex = NULL;
            return SPH_IFS_LENSLET_MODEL_OUTSIDE_ARRAY;
        }
    }
    sph_polygon_delete(hex); hex = NULL;
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set wavelengths in a spectral region on a master frame
 * @param self
 * @param uu        the u coordinate of lenslet
 * @param vv        the v coordinate of lenslet
 * @param dx        offset in x
 * @param dx        offset in y
 * @param mframe    the master frame
 *
 * Set the wavelengths for the spectral region.
 * The wavelengths inside the spectral region belonging to uu and vv
 * are set on the given master frame using the model to predict the
 * wavelengths.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ifs_lenslet_model_set_wavs_specregion(
        sph_ifs_lenslet_model* self,
        sph_polygon* poly,
        int uu, int vv,
        sph_master_frame* mframe)
{
    const int nx = cpl_image_get_size_x(mframe->image);
    const int ny = cpl_image_get_size_y(mframe->image);
    double polright, poltop, polleft, polbottom;
    int maxx, maxy, minx, miny;

    sph_polygon_get_extent(poly, &poltop, &polbottom, &polleft, &polright);

    maxx = CPL_MIN((int)polright,  nx - 1);
    maxy = CPL_MIN((int)poltop,    ny - 1);
    minx = CPL_MAX((int)polleft,   0);
    miny = CPL_MAX((int)polbottom, 0);

    for (int yy = miny; yy <= maxy; ++yy) {
        for (int xx = minx; xx <= maxx; ++xx) {
            if (!cpl_image_is_rejected(mframe->image,xx+1,yy+1)) {
                /* 0.5 to make it pixel midpoint */
                const double lambda =
                    sph_ifs_lenslet_model_get_lambda(self, poly, uu, vv,
                                                     (double)yy + 0.5);
                cpl_image_set( mframe->image, xx + 1, yy + 1, lambda );
            }
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set the distortion model
 * @param self      lenslet model
 * @param dist      the distortion model to set
 *
 * @return error code
 *
 * This sets the distortion model for the lenslet model. Do not call
 * the delete function on the distortion model afterwards, as ownership
 * is transferred to the lenslet model. The distortion model will be deleted
 * when sph_ifs_lenslet_model_delete is called.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ifs_lenslet_model_set_detector_distortion(
        sph_ifs_lenslet_model* self,
        sph_distortion_model* detector_dist ) {

    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(detector_dist,CPL_ERROR_NULL_INPUT);
    self->detector_distortion = detector_dist;
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the lenslet distortion model
 * @param self      lenslet model
 * @param dist      the lenslet distortion model to set
 *
 * @return error code
 *
 * This sets the distortion model for the lenslet model. Do not call
 * the delete function on the distortion model afterwards, as ownership
 * is transferred to the lenslet model. The distortion model will be deleted
 * when sph_ifs_lenslet_model_delete is called.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ifs_lenslet_model_set_lenslet_distortion(
        sph_ifs_lenslet_model* self,
        sph_distortion_model* lenslet_dist ) {

    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(lenslet_dist,CPL_ERROR_NULL_INPUT);
    self->lenslet_distortion = lenslet_dist;
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Delete the lenslet model.

 @param self     pointer to the lenslet model to delete

 @return nothing

 Description: Deletes the lenslet model, freeing up the space.

 */
/*----------------------------------------------------------------------------*/
void sph_ifs_lenslet_model_delete(sph_ifs_lenslet_model* self) {
    if (self != NULL) {
        cpl_vector_delete(self->lambdas);
        cpl_image_delete( self->easymap );
        sph_polygon_delete(self->pixelpoly);
        sph_distortion_model_delete(self->detector_distortion);
        sph_distortion_model_delete(self->lenslet_distortion);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
 @brief Fill an integer overlap image with a specific value
 @param self      Integer image to fill
 @param poly      The polygon to determine the fill
 @param idval     The fill value
 @param overlapim The double image with the values for the threshold
 @return CPL_ERROR_NONE or the relevant CPL error on error

 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
sph_ifs_lenslet_model_paint_specreg_ids__(cpl_image*         self,
                                          const sph_polygon* poly,
                                          int                idval,
                                          const cpl_image*   overlapim)
{
    const int     nx        = cpl_image_get_size_x(self);
    const int     ny        = cpl_image_get_size_y(self);
    const int     nxo       = cpl_image_get_size_x(overlapim);
    int*          pself     = cpl_image_get_data_int(self);
    const double* poverlap  = cpl_image_get_data_double_const(overlapim);

    double polright, poltop, polleft, polbottom;
    int maxx, maxy, minx, miny;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;
    cpl_ensure_code(self     != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(poly     != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(poverlap != NULL, CPL_ERROR_NULL_INPUT);

    sph_polygon_get_extent(poly, &poltop, &polbottom, &polleft, &polright);

    /* FIXME: The two images can have different sizes.
       How to ensure properly bounded access of the overlap image? */
    maxx = CPL_MIN((int)polright,  nx - 1);
    maxy = CPL_MIN((int)poltop,    ny - 1);
    minx = CPL_MAX((int)polleft,   0);
    miny = CPL_MAX((int)polbottom, 0);

    for (int yy = miny; yy <= maxy; ++yy) {
        for (int xx = minx; xx <= maxx; ++xx) {
            const double overlap = poverlap[xx-minx + (yy-miny) * nxo];

            if (overlap > 0.999999) {
                pself[xx + yy * nx] = idval;
            }
        }
    }

    return CPL_ERROR_NONE;
}

