/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*---------------------------- Includes ------------------------------------*/

#include "sph_keyword_manager.h"
#include "sph_error.h"
#include "sph_dictionary.h"
#include "sph_iniparser.h"
#include <errno.h>
#include <strings.h>
#include <ctype.h>
#include <assert.h>

/*---------------------------- Defines -------------------------------------*/
#define ASCIILINESZ         255
#define STRLINESZ CPL_STRINGIFY(ASCIILINESZ)

#define MAXKEYLEN 236

sph_error_code SPH_KEYWORD_MANAGER_GENERAL                  = SPH_KEYWORD_MANAGER_ERR_START + 1;
sph_error_code SPH_KEYWORD_MANAGER_CANT_READ_CONFIG         = SPH_KEYWORD_MANAGER_ERR_START + 2;
sph_error_code SPH_KEYWORD_MANAGER_NO_ROOT                  = SPH_KEYWORD_MANAGER_ERR_START + 3;
sph_error_code SPH_KEYWORD_MANAGER_NO_SOFTPLIST             = SPH_KEYWORD_MANAGER_ERR_START + 4;
sph_error_code SPH_KEYWORD_MANAGER_NOT_FOUND_IN_SOFTPLIST   = SPH_KEYWORD_MANAGER_ERR_START + 5;
sph_error_code SPH_KEYWORD_MANAGER_NOT_FOUND                   = SPH_KEYWORD_MANAGER_ERR_START + 6;
sph_error_code SPH_KEYWORD_MANAGER_USING_CONFIG                = SPH_KEYWORD_MANAGER_ERR_START + 7;


static sph_keyword_manager*        keymanager       = NULL;
static const char*                 keymanager_cfg   = "sphere.cfg";

static const cpl_property*
sph_keyword_manager_is_keyword_defined( const char* key );

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the keyword manager filename
 * @param self The new keyword manager filename, must be permanent storage
 * @return The previous keyword manager filename
 */
/*----------------------------------------------------------------------------*/
const char* sph_keyword_manager_set_cfg(const char* self)
{
    const char * old = keymanager_cfg;
    keymanager_cfg = self;
    return old;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get the keyword manager filename
 * @return The keyword manager filename
 */
/*----------------------------------------------------------------------------*/
const char* sph_keyword_manager_get_cfg(void)
{
    return keymanager_cfg;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create new sph_keyword_manager object or get pointer to existing one
 *
 * @return pointer of the  sph_keyword_manager
 *
 * This function creates a new sph_keyword_manager singleton if none exists or
 * returns the pointer to the existing one if one does exist.
 * The manager is automatically initialised from the sphere configuration file
 * if such a file exists.
 */
/*----------------------------------------------------------------------------*/
sph_keyword_manager* sph_keyword_manager_new( void ) {

    if ( keymanager == NULL) {
        FILE*               fp;

        keymanager = cpl_calloc( 1, sizeof(sph_keyword_manager) );
        keymanager->current_frame = cpl_frame_new();
        fp = fopen( keymanager_cfg, "r");
        if ( fp ) {
            fclose( fp );
            sph_keyword_manager_read_config( keymanager_cfg );
            sph_error_raise( SPH_KEYWORD_MANAGER_USING_CONFIG, __FILE__,
                             __func__, __LINE__, SPH_ERROR_INFO,
                             "Keyword associations read from: %s",
                             keymanager_cfg);
            if ( !keymanager->root ) {
                sph_keyword_manager_delete();
                return NULL;
            }
        }
    }
    return keymanager;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Traverse the tree recursively and add all with .NAME and .TYPE as last
 * child to plistsoft
 *
 * @param node the node to transverese
 *
 * @return error code
 *
 * Check if this one has the aparam.NAME and .TYPE defined. If so, add a new
 * parameter with the paramatername aparam of the type given by TYPE to the
 * plistsoft.
 * Then call this function for first child and then for all siblings.
 * In this way a paramaterlist representation of the tree from the node is
 * created.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_traverse( sph_keywordnode* node )
{
    sph_error_code          rerr        = CPL_ERROR_NONE;
    cpl_type                type        = CPL_TYPE_UNSPECIFIED;
    sph_keywordnode*        tnode       = NULL;
    sph_keywordnode*        vnode       = NULL;
    char                    nodename[1024];
    const char*                   astring;

    tnode = sph_keywordnode_get_keywordnode_by_name( node, "TYPE" );
    if ( tnode ) {
        astring = sph_keywordnode_get_string(tnode);
        if ( !strncasecmp( astring, "BOOL", 4)) {
            type = CPL_TYPE_BOOL;
        }
        else if ( !strncasecmp( astring, "INT", 3)) {
            type = CPL_TYPE_INT;
        }
        else if ( !strncasecmp( astring, "STRING", 6)) {
            type = CPL_TYPE_STRING;
        }
        else if ( !strncasecmp( astring, "FLOAT", 5)) {
            type = CPL_TYPE_FLOAT;
        }
        else if ( !strncasecmp( astring, "DOUBLE", 6)) {
            type = CPL_TYPE_DOUBLE;
        }
        else {
            sph_error_raise( CPL_ERROR_INVALID_TYPE,
                             __FILE__, __func__,
                             __LINE__,
                             SPH_ERROR_WARNING,
                             "Illegal type %s",
                             sph_keywordnode_get_string(tnode)
            );
        }
    }
    if ( type != CPL_TYPE_UNSPECIFIED ) {
        tnode = sph_keywordnode_get_keywordnode_by_name( node, "NAME" );

        if ( tnode ) {
            rerr = sph_keywordnode_BuildQualifiedName( node, nodename, 1024 );
            if ( rerr > 1 ) { /* This should not really ever happen...*/
                return SPH_KEYWORD_MANAGER_GENERAL;
            }

            if ( type == CPL_TYPE_BOOL ) {
                cpl_propertylist_prepend_bool( keymanager->plistsoft, nodename, 0 );
                vnode = sph_keywordnode_get_keywordnode_by_name( node, "VALUE" );
                if ( vnode ) {
                    cpl_propertylist_set_bool( keymanager->plistsoft, nodename, sph_keywordnode_get_int( vnode ) );
                }
            }
            else if ( type == CPL_TYPE_INT ) {
                cpl_propertylist_prepend_int( keymanager->plistsoft, nodename, -99 );
                vnode = sph_keywordnode_get_keywordnode_by_name( node, "VALUE" );
                if ( vnode ) {
                    cpl_propertylist_set_int( keymanager->plistsoft, nodename, sph_keywordnode_get_int( vnode ) );
                    sph_error_raise( SPH_KEYWORD_MANAGER_USING_CONFIG,
                            __FILE__, __func__, __LINE__,
                            SPH_ERROR_INFO,
                            "Am mapping the keyword %s to %s and set value to %d",
                            nodename,  sph_keywordnode_get_string( tnode ), sph_keywordnode_get_int( vnode ) );
                }
                else {
                    sph_error_raise( SPH_KEYWORD_MANAGER_USING_CONFIG,
                            __FILE__, __func__, __LINE__, SPH_ERROR_INFO,
                            "Am mapping the keyword %s to %s and set no default value.",
                            nodename, sph_keywordnode_get_string( tnode ) );
                }
            }
            else if ( type == CPL_TYPE_STRING ) {
                cpl_propertylist_prepend_string( keymanager->plistsoft, nodename, "NOTHING" );
                vnode = sph_keywordnode_get_keywordnode_by_name( node, "VALUE" );
                if ( vnode ) {
                    cpl_propertylist_set_string( keymanager->plistsoft, nodename, sph_keywordnode_get_string( vnode ) );
                }
            }
            else if ( type == CPL_TYPE_FLOAT ) {
                cpl_propertylist_prepend_float( keymanager->plistsoft, nodename, -99.0 );
                vnode = sph_keywordnode_get_keywordnode_by_name( node, "VALUE" );
                if ( vnode ) {
                    cpl_propertylist_set_float( keymanager->plistsoft, nodename, sph_keywordnode_get_double( vnode ) );
                    sph_error_raise( SPH_KEYWORD_MANAGER_USING_CONFIG,
                            __FILE__, __func__, __LINE__, SPH_ERROR_INFO,
                            "Am mapping the keyword %s to %s and set value to %g",
                            nodename, sph_keywordnode_get_string( tnode ),
                            sph_keywordnode_get_double( vnode ) );
                }
                else {
                    sph_error_raise( SPH_KEYWORD_MANAGER_USING_CONFIG,
                            __FILE__, __func__, __LINE__, SPH_ERROR_INFO,
                            "Am mapping the keyword %s to %s and set no default value.",
                            nodename, sph_keywordnode_get_string( tnode ) );
                }
            }
            else if ( type == CPL_TYPE_DOUBLE ) {
                cpl_propertylist_prepend_double( keymanager->plistsoft, nodename, -99.0 );
                vnode = sph_keywordnode_get_keywordnode_by_name( node, "VALUE" );
                if ( vnode ) {
                    cpl_propertylist_set_double( keymanager->plistsoft, nodename, sph_keywordnode_get_double( vnode ) );
                }
            }
            else {
                sph_error_raise( CPL_ERROR_INVALID_TYPE,
                                 __FILE__, __func__,
                                 __LINE__,
                                 SPH_ERROR_WARNING,
                                 "Illegal type for %s",
                                 sph_keywordnode_get_string(tnode)
                );
            }
        }
    }
    if ( sph_keywordnode_GetFirstChild(node) ) {
        sph_keyword_manager_traverse( sph_keywordnode_GetFirstChild(node) );
    }
    if ( sph_keywordnode_GetNext( node )  ) {
        sph_keyword_manager_traverse( sph_keywordnode_GetNext(node) );
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Read the key structure from a configuration file
 *
 * @param filename the filename of the key structure (in the format of a
 * sph_dtree file).
 *
 * @return the error code
 *
 * Reads the structure of the keywords from a configuration file which
 * has to be in the format as specified in the sph_dtree module.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_read_config( const char* filename ) {
    sph_keywordnode*        node        = NULL;

    if ( keymanager->root ) {
        sph_keywordnode_destroy(keymanager->root);
        keymanager->root = NULL;
    }
    keymanager->root = sph_keywordnode_load_configuration( filename );
    if ( keymanager->root == NULL ) {
        return SPH_KEYWORD_MANAGER_CANT_READ_CONFIG;
    }

    if ( keymanager->plistsoft ) {
        cpl_propertylist_empty( keymanager->plistsoft );
    }
    else {
        keymanager->plistsoft = cpl_propertylist_new( );
    }

    if ( !keymanager->plistsoft ) {
        return CPL_ERROR_UNSPECIFIED;
    }

    node = sph_keywordnode_get_keywordnode_by_name( keymanager->root, "SPH.KEYWORDS" );
    if ( node ) {
        node = sph_keywordnode_GetFirstChild( node );
        sph_keyword_manager_traverse( node );
    }
    else {
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return the name (FITS keyword) of a specific key
 *
 * @param key the software representation of the keyword
 *
 * @return the keyword name used in FITS files to represent the key
 *
 * This function returns the FITS keyword that is associated to the key
 * (which corresponds to the software representation of that keyword).
 * The function returns NULL if the key is not found
 *
 */
/*----------------------------------------------------------------------------*/
const char*
sph_keyword_manager_get_keyword( const char* key ) {
    const char* result = key;

    if (key != NULL) {
        char             fullkey[256] = "SPH.KEYWORDS.";
        char           * keycopy = fullkey + 13;
        sph_keywordnode* node        = NULL;
        const cpl_size   keylen      = strlen(key);
        const char     * search = key;
        const char     * spc;

        if ( !keymanager ) {
            sph_keyword_manager_new();  //check  allocation
            if ( !keymanager ) {
                SPH_NO_SELF;
                return result;
            }
        }
        if (keymanager->root) {

            if (keylen > MAXKEYLEN) {
                (void)cpl_error_set_message(cpl_func, CPL_ERROR_UNSUPPORTED_MODE,
                                            "keylen=%d > "
                                            CPL_STRINGIFY(MAXKEYLEN),
                                            (int)keylen);
                return NULL;
            }

            while (keylen > search - key &&
                   (spc = memchr(search, ' ',
                                 keylen - (search - key))) != NULL) {

                (void)memcpy(keycopy + (search - key), search, spc - search);
                keycopy[search - key] = '_';
                search = spc + 1;
            }

            if (keylen > search - key) {
                (void)memcpy(keycopy + (search - key), search,
                             keylen - (search - key));
            }

            (void)memcpy(keycopy + keylen, ".NAME", 6); /* Incl. NULL-byte */

            node = sph_keywordnode_get_keywordnode_by_name(keymanager->root,
                                                           fullkey);

            if ( node ) {
                result  = sph_keywordnode_get_string(node);
            }
        }
    }

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return a  software representation of a FITS keyword if defined
 *
 * @param fitskey  the FITS keyword to check in the software config
 *
 * @return pointer to software property if the keyword is defined, NULL otherwise
 *
 * This function returns a pointer to the property if the keyword that is given
 * is defined in the
 * software configuration and has a software representaion.
 *
 */
/*----------------------------------------------------------------------------*/
static const cpl_property*
sph_keyword_manager_is_keyword_defined( const char* fitskey ) {
    char                fullkey[256];
    int                    ii            = 0;
    int                    cc            = 0;
    const cpl_property*    p;
    sph_keywordnode*    node        = NULL;
    const char*                astring        = NULL;

    if ( !keymanager ) {
        sph_error_raise( SPH_KEYWORD_MANAGER_NO_ROOT, __FILE__, __func__,
                         __LINE__,
                         SPH_ERROR_WARNING, "No keymanager root defined!");
        return NULL;
    }
    if ( !keymanager->root ) {
        sph_error_raise( SPH_KEYWORD_MANAGER_NO_ROOT, __FILE__, __func__,
                         __LINE__,
                         SPH_ERROR_WARNING, "No keymanager root defined!");
        return NULL;
    }
    if ( !keymanager->plistsoft ) {
        sph_error_raise( SPH_KEYWORD_MANAGER_NOT_FOUND_IN_SOFTPLIST, __FILE__, __func__,
                         __LINE__,
                         SPH_ERROR_WARNING, "No soft propertylist defined!");
        return NULL;
    }

    p = cpl_propertylist_get_const( keymanager->plistsoft, cc );
    for (cc = 0; cc < cpl_propertylist_get_size( keymanager->plistsoft); ++cc) {
        p = cpl_propertylist_get_const( keymanager->plistsoft, cc );
        if ( p ) {
            sprintf( fullkey, "%s.NAME", cpl_property_get_name(p) );
            for (ii = 0; ii < (int)strlen(fullkey); ++ii) {
                if ( ii < 256 ) {
                    if ( fullkey[ii] == ' ' ) {
                        fullkey[ii] = '_';
                    }
                }
            }
            node        = sph_keywordnode_get_keywordnode_by_name( keymanager->root, fullkey );
            astring = sph_keywordnode_get_string( node );
            if ( strcasecmp( astring , fitskey ) == 0 ) {
                return p;
            }
            else {
                ;
        }

        }

    }
    return NULL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a propertylist (keywordlist) by loading from a FITS file
 *
 * @param the filename of the fitsfile to load from
 * @param the extension in the fits file to read from, 0 being the main header
 *
 * @return error code of the operation
 *
 * This function will read the propertylist from a filename -- just the same
 * was that the function cpl_propertylist_load does but using a regex that
 * filters out only the ESO keywords.
 *
 * The error code is returned.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_get_properties_from_file( const char* czFilename, int ext ) {
    cpl_propertylist*       plist       = NULL;

    if ( !keymanager ) {
        sph_keyword_manager_new();
        if ( !keymanager ) {
            SPH_NO_SELF;
            return CPL_ERROR_NULL_INPUT;
        }
    }

    plist = cpl_propertylist_load_regexp( czFilename, ext, ".*ESO.*",0 );
    if ( !plist ) {
        return cpl_error_set_where(cpl_func);
    }

    cpl_frame_set_filename( keymanager->current_frame, czFilename);
    if (keymanager->plist) {
        cpl_propertylist_append( plist, keymanager->plist );
        cpl_propertylist_delete( keymanager->plist );
        keymanager->plist = plist;
        //cpl_propertylist_delete( plist );
    }
    else {
        keymanager->plist = plist;
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a propertylist (keywordlist) by loading from a ASCII file
 *
 * @param the filename of the ASCII file to load from
 * @param the section name  in the ASCII file to read from
 *
 * @return error code of the operation
 *
 * This function will read the propertylist from a filename -- just the same
 * was that the function cpl_propertylist_load does.
 *
 * The error code is returned.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_get_properties_from_ascii( const char* czFilename,
                                               const char* sect ) {
    sph_dictionary* ini         = NULL;
    cpl_propertylist*     plist = NULL;
    char*                 key;
    char                  keyup[2048];
    int                      kk        = 0;
    int                   ii        = 0;
    int                   ival      = 0;
    double                dval      = 0.0;
    int                   nosect    = 0;
    char                 a[ASCIILINESZ + 1];
    char                 b[ASCIILINESZ + 1];

    if ( !keymanager ) {
        sph_keyword_manager_new();
        if ( !keymanager ) {
            SPH_NO_SELF;
            return CPL_ERROR_NULL_INPUT;
        }
    }

    if ( sect == NULL ) {
        sect = "[]";
        nosect = 1;
    }

    plist = keymanager->plist;
    if ( !plist ) {
        keymanager->plist = cpl_propertylist_new();
        plist = keymanager->plist;
    }
    if ( !plist ) {
        return cpl_error_get_code();
    }

    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,"Reading head file from %s",czFilename);
    ini = sph_iniparser_load( czFilename );
    cpl_ensure_code(ini,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ini->size > 0,CPL_ERROR_ILLEGAL_INPUT);
    for ( ii = 0; ii < ini->size; ++ii ) {
        key = NULL;
        if ( nosect ) {
            key = &ini->key[ii][0];
        }
        else if ( !strncmp( ini->key[ii], sect, strlen(sect) ) ) {
            key = &ini->key[ii][strlen(sect)];
        }
        if ( key ) {
            const cpl_property*   p;
            for ( kk = 0; kk < (int)strlen(key); ++kk) {
                keyup[kk]=toupper(key[kk]);
            }
            keyup[strlen(key)]='\0';
            p = sph_keyword_manager_is_keyword_defined( &keyup[1] );
            if ( p ) {
                if ( cpl_property_get_type(p) == CPL_TYPE_BOOL ) {
                    ival = sph_iniparser_getboolean( ini, key, 0);
                    cpl_propertylist_update_bool( plist, &keyup[1], ival );
                }
                else if ( cpl_property_get_type(p) == CPL_TYPE_INT ) {
                    errno = 0;
                    ival = sph_iniparser_getint( ini, ini->key[ii], 0);
                    if ( errno == 0 ) {
                        cpl_propertylist_update_int( plist, &keyup[1], ival );
                    }
                }
                else if ( cpl_property_get_type(p) == CPL_TYPE_FLOAT ||
                        cpl_property_get_type(p) == CPL_TYPE_DOUBLE ) {
                    errno = 0;
                    dval = sph_iniparser_getdouble( ini, ini->key[ii], 0);
                    if ( errno == 0 ) {
                        if ( cpl_property_get_type( p ) == CPL_TYPE_DOUBLE ) {
                            cpl_propertylist_update_double( plist, &keyup[1], dval );
                        }
                        if ( cpl_property_get_type( p ) == CPL_TYPE_FLOAT ) {
                            cpl_propertylist_update_float( plist, &keyup[1], dval );
                        }
                    }
                }
                else if ( cpl_property_get_type(p) == CPL_TYPE_STRING ) {
                    cpl_propertylist_update_string( plist,
                                                    &keyup[1],
                            sph_iniparser_getstring( ini, ini->key[ii],
                                                            "NOTHING" )
                    );
                }
                else {
                    SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,"Unsupported type %d",cpl_property_get_type(p));
                }
            }
            else {
                const char* tmpstr;

                cpl_error_reset();
                SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Keyword %s was not "
                                     "defined in a %s, so am guessing type...",
                                     &keyup[1], keymanager_cfg);

                if ( !strcasecmp( "T", sph_iniparser_getstring(ini, ini->key[ii],
                                                               "K") ) ) {
                    ival = sph_iniparser_getboolean( ini, key, 0);
                    cpl_propertylist_update_bool( plist, &keyup[1], ival );
                    continue;
                }
                if ( !strcasecmp( "F", sph_iniparser_getstring(ini, ini->key[ii],
                                                               "K") ) ) {
                    ival = sph_iniparser_getboolean( ini, key, 0);
                    cpl_propertylist_update_bool( plist, &keyup[1], ival );
                    continue;
                }
                errno = 0;
                ival = sph_iniparser_getint( ini, ini->key[ii], 0);
                dval = sph_iniparser_getdouble( ini, ini->key[ii], 0);
                tmpstr = sph_iniparser_getstring( ini, ini->key[ii],
                                                  "NOTHING" );
                if (sscanf(tmpstr,"%" STRLINESZ "[^.].%" STRLINESZ "[^.]",a,b)
                    < 2 ) {
                    cpl_propertylist_update_int( plist, &keyup[1], ival );
                    continue;
                }
                errno = 0;
                dval = sph_iniparser_getdouble( ini, ini->key[ii], 0);
                if ( errno == 0 ) {
                    cpl_propertylist_update_double( plist, &keyup[1], dval );
                    continue;
                }
                cpl_propertylist_prepend_string( plist,
                        &keyup[1],
                        tmpstr
                );
            }
        }
    }
    sph_iniparser_freedict(ini);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a propertylist (keywordlist) by loading from a FITS file
 *
 * @param the filename of the fitsfile to load from
 * @param the extension in the fits file to read from, 0 being the main header
 *
 * @return error code of the operation
 *
 * This function will read the propertylist from a filename -- just the same
 * was that the function cpl_propertylist_load does.
 *
 * The error code is returned.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_get_properties_from_frame( cpl_frame* frame, int ext ) {
    const char* czFilename            = NULL;

    if ( !keymanager ) {
        sph_keyword_manager_new();
        if ( !keymanager ) {
            SPH_NO_SELF;
            return CPL_ERROR_NULL_INPUT;
        }
    }

    czFilename = cpl_frame_get_filename( frame );
    if ( !czFilename ) {
        return cpl_error_set_where(cpl_func);
    }

    sph_keyword_manager_get_properties_from_file( czFilename, ext );

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Return the integer value of a specific key
 *
 * @param key the software representation of the keyword
 *
 * @return the integer value of the keyword
 *
 * This function returns the keyword value as an integer.
 *
 * Possible error codes set by this function:
 * - CPL_ERROR_NULL_INPUT if key is null or no keyword_manager initialised
 * - SPH_KEYWORD_MANAGER_GENERAL if a general error occured
 * - SPH_KEYWORD_MANAGER_NO_ROOT if there is no root of the tree
 * - SPH_KEYWORD_MANAGER_NO_SOFTPLIST if there is no list of keyws defined
 * - SPH_KEYWORD_MANAGER_NOT_FOUND_IN_SOFTPLIST if the key is not found in list
 *
 */
/*----------------------------------------------------------------------------*/
int
sph_keyword_manager_get_int( const char* key ) {
    sph_keywordnode*    node        = NULL;
    int                 result      = 0;
    cpl_error_code      prestate    = CPL_ERROR_NONE;
    const cpl_size      keylen      = key ? strlen(key) : 0;

    if ( !keymanager ) {
        sph_keyword_manager_new();
        if ( !keymanager ) {
            SPH_NO_SELF;
            return 0;
        }
    }
    prestate = cpl_error_get_code();

    if ( !keymanager->root ) {
        result = cpl_propertylist_get_int( keymanager->plist, key );
        if ( cpl_error_get_code() != prestate ) {
            sph_error_raise( SPH_KEYWORD_MANAGER_NO_ROOT,
                    __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "Could not find the "
                            "keyword %s in the file %s !",
                    key,
                    cpl_frame_get_filename(keymanager->current_frame) );
            return SPH_KEYWORD_MANAGER_NOT_FOUND;
        }
        return result;
    } else if (key != NULL) {
        char* keycopy = memcpy(cpl_malloc(1 + keylen), key, 1 + keylen);
        char* spcfind = memchr(keycopy, ' ', keylen);
        char* fullkeyname;
        char* fullkeyvalue;

        while (spcfind != NULL) {
            *spcfind = '_';
            spcfind = memchr(spcfind, ' ',
                             keylen - (cpl_size)(spcfind - keycopy));
        }

        fullkeyname  = cpl_sprintf("SPH.KEYWORDS.%s.NAME",  keycopy);
        fullkeyvalue = cpl_sprintf("SPH.KEYWORDS.%s.VALUE", keycopy);
        cpl_free(keycopy);

        node   = sph_keywordnode_get_keywordnode_by_name( keymanager->root,
                                                        fullkeyname );
        result = cpl_propertylist_get_int( keymanager->plist,
                                           sph_keywordnode_get_string( node ) );

        cpl_free(fullkeyname);

        if ( cpl_error_get_code() != prestate ) {
            sph_error_raise( SPH_KEYWORD_MANAGER_GENERAL,
                    __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING, "Could not find the "
                    "keyword %s in the file %s !",
                    key,
                    cpl_frame_get_filename(keymanager->current_frame) );
            cpl_error_reset();
            node = sph_keywordnode_get_keywordnode_by_name( keymanager->root,
                                                            fullkeyvalue );

            if ( node ) {
                result      = sph_keywordnode_get_int( node );
            } else {
                result = cpl_propertylist_get_int( keymanager->plist, key );
                if ( cpl_error_get_code() != prestate ) {
                    sph_error_raise( SPH_KEYWORD_MANAGER_NOT_FOUND,
                            __FILE__, __func__, __LINE__,
                            SPH_ERROR_ERROR, "Could not find the "
                                    "keyword %s in the file %s !",
                            key,
                            cpl_frame_get_filename(keymanager->current_frame) );
                    return SPH_KEYWORD_MANAGER_NOT_FOUND;
                }
                cpl_error_reset();
                cpl_free(fullkeyvalue);
                return result;
            }
        }
        cpl_free(fullkeyvalue);
    } else {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return the integer value of a specific key
 *
 * @param key the software representation of the keyword
 *
 * @return the integer value of the keyword
 *
 * This function returns the keyword value as an integer.
 *
 */
/*----------------------------------------------------------------------------*/
const char*
sph_keyword_manager_get_string( const char* key ) {
    sph_keywordnode*    node        = NULL;
    const char*         result      = NULL;
    cpl_error_code      prestate    = CPL_ERROR_NONE;
    const cpl_size      keylen      = key ? strlen(key) : 0;


    if ( !keymanager ) {
        sph_keyword_manager_new();
        if ( !keymanager ) {
            SPH_NO_SELF;
            return NULL;
        }
    }

    prestate = cpl_error_get_code();

    if ( !keymanager->root ) {
        result = cpl_propertylist_get_string( keymanager->plist, key );
        if ( cpl_error_get_code() != prestate ) {
            cpl_error_reset();
            return NULL;
        }
        return result;
    } else if (key != NULL) {
        char* keycopy = memcpy(cpl_malloc(1 + keylen), key, 1 + keylen);
        char* spcfind = memchr(keycopy, ' ', keylen);
        char* fullkeyname;
        char* fullkeyvalue;

        while (spcfind != NULL) {
            *spcfind = '_';
            spcfind = memchr(spcfind, ' ',
                             keylen - (cpl_size)(spcfind - keycopy));
        }

        fullkeyvalue = cpl_sprintf("SPH.KEYWORDS.%s.VALUE", keycopy);
        fullkeyname  = cpl_sprintf("SPH.KEYWORDS.%s.NAME",  keycopy);
        cpl_free(keycopy);

        node        = sph_keywordnode_get_keywordnode_by_name( keymanager->root, fullkeyname );
        result      = cpl_propertylist_get_string( keymanager->plist, sph_keywordnode_get_string( node ) );
        cpl_free(fullkeyname);

        if ( cpl_error_get_code() != prestate ) {
            cpl_error_reset();
            node        = sph_keywordnode_get_keywordnode_by_name( keymanager->root, fullkeyvalue );
            cpl_free(fullkeyvalue);
            if ( node ) {
                result      = sph_keywordnode_get_string( node );
            }
            else {
                result = cpl_propertylist_get_string( keymanager->plist, key );
                if ( cpl_error_get_code() != prestate ) {
                    cpl_error_reset();
                    return NULL;
                }
                cpl_error_reset();
                return result;
            }
        } else {
            cpl_free(fullkeyvalue);
        }
    } else {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Load the keywords.
 *
 * @param czFilename the filename (filename of fits file) to load from
 * @param ext       the extension to use
 * @return error code of the operation
 *
 * This function will load the keywords from a source with the given filename.
 * The keywords are loaded from the FITS file first and then from a *.head
 * file with the same filename base as the
 * FITS file (i.e. the filename with .fits replaced by .head ). This means that
 * any exisiting keywords are overwritten. If this file
 * does not exist, keywords are loaded from the FITS file only.
 * Otherwise, keywords are loaded from both, first from FITS
 * then from .head file.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_load_keywords( const char* czFilename, int ext ) {
    FILE*               fp          = NULL;
    char                czHead[256];
    int                 slen        = 0;
    sph_error_code      rerr        = CPL_ERROR_NONE;

    if ( !keymanager ) {
        sph_keyword_manager_new();
        if ( !keymanager ) {
            SPH_NO_SELF;
            return CPL_ERROR_NULL_INPUT;
        }
    }
    if ( keymanager->plist ) {
        cpl_propertylist_delete( keymanager->plist );
        keymanager->plist = NULL;
    }
    keymanager->plist = cpl_propertylist_new( );
    cpl_frame_set_filename( keymanager->current_frame, czFilename);

    rerr = sph_keyword_manager_get_properties_from_file( czFilename, ext );

    if ( rerr ) {
        return rerr;
    }

    slen = strlen( czFilename );

    if ( slen > 255 ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    strncpy( czHead, czFilename, slen-4 );
    czHead[slen-4]='\0';

    strcat( czHead, "head" );

    fp = fopen( czHead, "r" );

    if ( fp ) {
      fclose( fp );
      sph_keyword_manager_get_properties_from_ascii( czHead, NULL );
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief This function creates a propertylist for a file using the keyword manager
 *
 * @param czFilename the filename of the file to obtain properties from
 *
 * @return the propertylist or NULL in case of error
 *
 * This function uses the full fledged mechanisms of the keyword manager to
 * create a propertylist for a specific FITS file. The function returns a
 * propertylist where property names are the software (key) representations
 * of keywords, e.g. something like SPH.KEYWORDS.IRDIS.FILTER1.ID
 * These are then assigned the values of the actual corresponding keyword
 * either in the FITS file or in the corresponding *.head ASCII file.
 *
 * This function makes it very easy to refactor code that used to use the
 * cpl_proprtylist_load function: simply change the calls of
 * cpl_propertylist_load to sph_keyword_manager_load_properties with the same
 * arguments. Of course any requests to obtain specific properties then have
 * to use the software representations as keys.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_propertylist*
sph_keyword_manager_load_properties( const char* czFilename, int ext )
{
    cpl_propertylist*           result      = NULL;
    const cpl_property*         p;
    cpl_property*               pd          = NULL;
    int                         cc          = 0;
    const char*                 pname       = NULL;
    const char*                        fullkeyname = NULL;
    sph_error_code              rerr        = CPL_ERROR_NONE;
    sph_keyword_manager*        keym        = NULL;
    int                            ii            = 0;
    char                        pnamed[1024];

#ifdef SPH_NO_KEYMANAGER
    return cpl_propertylist_load(czFilename,ext);
#endif
    if ( !keymanager ) {
        sph_keyword_manager_new();
        if ( !keymanager ) {
            SPH_NO_SELF;
            return NULL;
        }
    }

    keym = sph_keyword_manager_new();
    cpl_frame_set_filename( keym->current_frame, czFilename);

    rerr = sph_keyword_manager_load_keywords( czFilename, ext );

    if ( rerr != CPL_ERROR_NONE ) {
        return NULL;
    }
    /* If no root is defined, just return the plain parameterlist as obtained
     * from the FITS file (no software representations!) as the parameterlist
     */
    if ( !keymanager->root || !keymanager->plistsoft ) {
        result = cpl_propertylist_duplicate( keymanager->plist );
        return result;
    }

    result = cpl_propertylist_new();
    if ( ! result ) {
        /* TODO: add error message */
        return NULL;
    }

    p = cpl_propertylist_get_const( keymanager->plistsoft, 0 );

    /* Remove the SPH.KEYWORDS. bit from all the parameters in the software parameterlist
     * and look up the remaining bit in the kewordnode tree to get the corresponding
     * keywordname (this then is not the software representation anymore, but the actual
     * FITS keyword name). Copy the property (if indeed it was found ) and rename to the
     * software representation (without the SPH.KEYWORDS bit).
     */
    for (cc = 0; cc < cpl_propertylist_get_size( keymanager->plistsoft ); ++cc) {
        p = cpl_propertylist_get( keymanager->plistsoft, cc );
        if ( p ) {
            pname = cpl_property_get_name( p );
            strncpy( pnamed,
                    &pname[ strlen( "SPH.KEYWORDS.\0" ) ],
                    1024 - strlen("SPH.KEYWORDS.\0")
            );
            for (ii = 0; ii < (int)strlen(pnamed); ++ii) {
                if ( pnamed[ii] == '_' ) {   // need to translate _ to space to
                    pnamed[ii] = ' ';       // reverse replacement in looking up nodes
                }
            }
            fullkeyname = sph_keyword_manager_get_keyword( pnamed );
            if ( fullkeyname ) {
                if ( cpl_propertylist_has(keymanager->plist, fullkeyname ) ) {
                    cpl_propertylist_copy_property( result, keymanager->plist, fullkeyname );
                    pd = cpl_propertylist_get_property( result, fullkeyname);
                    cpl_property_set_name( pd, pnamed );
                }
                else {
                    cpl_propertylist_copy_property( result, keymanager->plistsoft, pname );
                    pd = cpl_propertylist_get_property( result, pname);
                    cpl_property_set_name( pd, pnamed );
                }
            }
        }
    }
    cpl_error_reset();
    /* Now, add all properties that are defined in the list of FITS keywords that are
     * currently loaded but are not in the software representation list under their
     * FITS keyword name.
     */
    for (cc = 0; cc < cpl_propertylist_get_size( keymanager->plist ); ++cc) {
        p = cpl_propertylist_get_const( keymanager->plist, cc );
        if ( p ) {
            pname = cpl_property_get_name( p );
            if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
                rerr = cpl_error_get_code();
                cpl_error_reset();
            }
            else {
                if ( sph_keyword_manager_is_keyword_defined(pname) || cpl_propertylist_has(result, pname ) ) {
                    ;
                }
                else {
                    rerr = cpl_propertylist_copy_property( result, keymanager->plist, pname );
                    if ( rerr == CPL_ERROR_TYPE_MISMATCH ) {
                        rerr = CPL_ERROR_TYPE_MISMATCH;
                    }
                    else if ( rerr != CPL_ERROR_NONE ) {
                        return NULL;
                    }
                }
            }
        }

    }
    cpl_error_reset();
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Transform back a propertylist, resolving software representations.
 *
 * @param plist        the input propertylist
 *
 * @return the translated propertylist or NULL in case of error.
 *
 * This creates a new propertylist that can be written to a FITS file,
 * translating software representations back to hard keywords,
 * according to the sphere.cfg file.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_propertylist*
sph_keyword_manager_trans_keywords( const cpl_propertylist* plist ) {
    cpl_propertylist*        plisttowrite        = NULL;
        cpl_property*            prop                = NULL;
    const char*                    pname                = NULL;
    const char*                    fullkeyname            = NULL;
    int                        cc                    = 0;

    if ( !plist ) {
        return NULL;
    }
    if ( !keymanager ) {
        sph_keyword_manager_new();
        if ( !keymanager ) {
            SPH_NO_SELF;
            return NULL;
        }
    }
    if ( !keymanager->root || !keymanager->plistsoft ) {
        plisttowrite = cpl_propertylist_duplicate( plist );
        return plisttowrite;
    }

    plisttowrite = cpl_propertylist_new();
    for (cc = 0; cc < cpl_propertylist_get_size( plist ); ++cc) {
      prop = cpl_property_duplicate( cpl_propertylist_get_const( plist, cc ) );
      if ( prop ) {
          pname = cpl_property_get_name( prop );
          if ( pname ) {
              if ( strcmp(pname,"") != 0 && (int)strlen(cpl_property_get_name(prop)) > 1) {
                  fullkeyname = sph_keyword_manager_get_keyword( pname );
                  if ( fullkeyname ) {
                      if ( (int)strlen(fullkeyname) > 1 && pname != fullkeyname) {
                          cpl_property_set_name( prop, fullkeyname );
                          if ( (int)strlen(cpl_property_get_name(prop)) < 1) {
                              SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,"Setting bad.");
                          }
                      }
                  }
                  if ( cpl_propertylist_has( plisttowrite, cpl_property_get_name(prop)) && (int)strlen(cpl_property_get_name(prop)) > 1) {
                      sph_error_raise( SPH_ERROR_GENERAL,
                              __FILE__, __func__, __LINE__,
                              SPH_ERROR_WARNING,
                              "The property %s is present "
                              "twice in the propertylist." ,
                              cpl_property_get_name( prop ) );
                  }
                  else {
                      cpl_propertylist_append_property( plisttowrite, prop);
                  }
              }
          }
          cpl_property_delete(prop);
      }
    }
    cpl_error_reset();
    return plisttowrite;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Udate a keyword of type KEYNAME x
 * @param proplist          the propertylist
 * @param keyname           the KEYNAME
 * @param x                 the x
 * @param value             the value
 *
 * @return error code
 *
 * This update a property of the name KEYNAME_x_y in the propertylist
 * using the cpl_propertylist_update_double function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_update_int1(
        cpl_propertylist* proplist,
        const char* keyname,
        int x,
        int value )
{
    char* str;
    cpl_ensure_code(proplist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(keyname  != NULL, CPL_ERROR_NULL_INPUT);

    str = cpl_sprintf("%s %d", keyname, x);

    cpl_propertylist_update_int(proplist, str, value);
    cpl_free(str);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Udate a keyword of type KEYNAME x
 * @param proplist          the propertylist
 * @param keyname           the KEYNAME
 * @param x                 the x
 * @param value             the value
 *
 * @return error code
 *
 * This update a property of the name KEYNAME_x_y in the propertylist
 * using the cpl_propertylist_update_double function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_update_int_char(
        cpl_propertylist* proplist,
        const char* keyname,
        char x,
        int value )
{
    char* str;
    cpl_ensure_code(proplist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(keyname  != NULL, CPL_ERROR_NULL_INPUT);

    str = cpl_sprintf("%s %c", keyname, x);

    cpl_propertylist_update_int(proplist, str, value);
    cpl_free(str);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Udate a keyword of type KEYNAME x
 * @param proplist          the propertylist
 * @param keyname           the KEYNAME
 * @param x                 the x
 * @param value             the value
 *
 * @return error code
 *
 * This update a property of the name KEYNAME_x_y in the propertylist
 * using the cpl_propertylist_update_double function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_update_string_char(
        cpl_propertylist* proplist,
        const char* keyname,
        char x,
        const char* value )
{
    char* str;
    cpl_ensure_code(proplist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(keyname  != NULL, CPL_ERROR_NULL_INPUT);

    str = cpl_sprintf("%s %c", keyname, x);

    cpl_propertylist_update_string(proplist, str, value);
    cpl_free(str);

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Udate a keyword of type KEYNAME x
 * @param proplist          the propertylist
 * @param keyname           the KEYNAME
 * @param x                 the x
 * @param value             the value
 *
 * @return error code
 *
 * This update a property of the name KEYNAME_x_y in the propertylist
 * using the cpl_propertylist_update_double function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_update_double1(
        cpl_propertylist* proplist,
        const char* keyname,
        int x,
        double value )
{
    char* str;
    cpl_ensure_code(proplist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(keyname  != NULL, CPL_ERROR_NULL_INPUT);

    str = cpl_sprintf("%s %d", keyname, x);

    cpl_propertylist_update_double(proplist, str, value);
    cpl_free(str);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Udate a keyword of type KEYNAME x
 * @param proplist          the propertylist
 * @param keyname           the KEYNAME
 * @param x                 the x
 * @param value             the value
 *
 * @return error code
 *
 * This update a property of the name KEYNAME_x_y in the propertylist
 * using the cpl_propertylist_update_double function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_update_double_char(
        cpl_propertylist* proplist,
        const char* keyname,
        char x,
        double value )
{
    char* str;
    cpl_ensure_code(proplist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(keyname  != NULL, CPL_ERROR_NULL_INPUT);

    str = cpl_sprintf("%s %c", keyname, x);

    cpl_propertylist_update_double(proplist, str, value);
    cpl_free(str);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Udate a keyword of type KEYNAME x
 * @param proplist          the propertylist
 * @param keyname           the KEYNAME
 * @param x                 the character
 * @param n                    the first integer
 * @oaram i                 the second integer
 * @param value             the value
 *
 * @return error code
 *
 * This update a property of the name KEYNAME_x_y in the propertylist
 * using the cpl_propertylist_update_double function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_update_double_char_int2(
        cpl_propertylist* proplist,
        const char* keyname,
        char x,
        int n,
        int i,
        double value )
{
    char* str;
    cpl_ensure_code(proplist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(keyname  != NULL, CPL_ERROR_NULL_INPUT);

    str = cpl_sprintf("%s %c %d_%d", keyname, x, n, i);

    cpl_propertylist_update_double(proplist, str, value);
    cpl_free(str);

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Udate a keyword of type KEYNAME x_y
 * @param proplist          the propertylist
 * @param keyname           the KEYNAME
 * @param x                 the x
 * @param y                 the y
 * @param value             the value
 *
 * @return error code
 *
 * This update a property of the name KEYNAME_x_y in the propertylist
 * using the cpl_propertylist_update_double function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_keyword_manager_update_double2(cpl_propertylist* proplist,
                                                  const char* keyname,
                                                  int x,
                                                  int y,
                                                  double value )
{
    char* str;
    cpl_ensure_code(proplist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(keyname  != NULL, CPL_ERROR_NULL_INPUT);

    str = cpl_sprintf("%s %d_%d", keyname, x, y);

    cpl_propertylist_update_double(proplist, str, value);
    cpl_free(str);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Udate a keyword of type KEYNAME x_y
 * @param proplist          the propertylist
 * @param keyname           the KEYNAME
 * @param x                 the x
 * @param y                 the y
 * @param value             the value
 *
 * @return error code
 *
 * This update a property of the name KEYNAME_x_y in the propertylist
 * using the cpl_propertylist_update_double function.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_update_double_char_int(
        cpl_propertylist* proplist,
        const char* keyname,
        char x,
        int y,
        double value )
{
    char* str;
    cpl_ensure_code(proplist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(keyname  != NULL, CPL_ERROR_NULL_INPUT);

    str = cpl_sprintf("%s %c %d", keyname, x, y);

    cpl_propertylist_update_double(proplist, str, value);
    cpl_free(str);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Update propertylist with value from a vector
 * @param pl            the propertylist to modify
 * @param vec           the vector of values
 * @param keybase       the basename of the keyword to use
 *
 * @return error code
 *
 * This function updates a list of double on the propertylist
 * where each value is taken from the vector and the
 * name of the property is constructed from the basename provided
 * with the running index of the vector, e.g. "KEYBASE 1", ...
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_keyword_manager_vector2proplist(
        cpl_propertylist* pl,
        const cpl_vector* vec,
        const char* keybase)
{
    int             ii          = 0;
    for (ii = 0; ii < cpl_vector_get_size(vec); ++ii) {
        sph_keyword_manager_update_double1(pl,keybase,ii,
                cpl_vector_get(vec,ii));
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the sph_keyword_manager
 *
 * @return error code of the operation
 *
 * The deletes the sph_keyword_manager and releases all resources associated
 * with it.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_keyword_manager_delete( void ) {

    if ( keymanager ) {
        if ( keymanager->root ) {
            sph_keywordnode_destroy( keymanager->root );
            keymanager->root = NULL;
        }
        if ( keymanager->plist ) {
            cpl_propertylist_delete( keymanager->plist );
            keymanager->plist = NULL;
        }
        if ( keymanager->plistsoft ) {
            cpl_propertylist_delete( keymanager->plistsoft );
            keymanager->plistsoft = NULL;
        }
        if ( keymanager->current_frame ) {
            cpl_frame_delete( keymanager->current_frame );
            keymanager->current_frame = NULL;
        }
        cpl_free(keymanager);
        keymanager = NULL;
    }
    return CPL_ERROR_NONE;
}
