/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*---------------------------- Includes ------------------------------------*/

#include "sph_keyword_reader.h"
#include "sph_error.h"

#include <string.h>
#include <stdio.h>

/*---------------------------- Defines -------------------------------------*/
#define ASCIILINESZ         255
#define STRLINESZ CPL_STRINGIFY(ASCIILINESZ)

/*---------------------------------------------------------------------------
                        Private to this module
 ---------------------------------------------------------------------------*/
static
sph_error_code sph_keyword_reader_scanline(const char* line, char* s1, char* s2,
                                           char* s3, char* s4)
    CPL_ATTR_NONNULL;

static sph_error_code
sph_keyword_reader_scanline(const char* line, char* s1, char* s2, char* s3,
                            char* s4) {
    int         rerr        = 0;
    rerr = sscanf(line, "%" STRLINESZ "s \"%" STRLINESZ "[a-zA-Z_ -]\" %"
                  STRLINESZ "s \"%" STRLINESZ "[a-zA-Z -]\"", s1, s2, s3, s4);
    if ( rerr !=4 ) {
        rerr = sscanf(line, "%" STRLINESZ "s \"%" STRLINESZ "[a-zA-Z -]\" %"
                      STRLINESZ "s %" STRLINESZ "s", s1, s2, s3, s4);        
    }
    if ( rerr !=4 ) {
        rerr = sscanf(line, "%" STRLINESZ "s %" STRLINESZ "s %" STRLINESZ
                      "s \"%" STRLINESZ "[a-zA-Z -]\"", s1, s2, s3, s4);        
    }
    if ( rerr !=4 ) {
        rerr = sscanf(line, "%" STRLINESZ "s %" STRLINESZ "s %" STRLINESZ
                      "s %" STRLINESZ "s", s1, s2, s3, s4);        
    }
    return rerr;
}

sph_keyword_reader* 
sph_keyword_reader_new(void) {
    sph_keyword_reader*        result        = NULL;
    
    result = cpl_calloc( 1, sizeof(sph_keyword_reader) );
    result->table = cpl_table_new( 0 );
    cpl_table_new_column( result->table, "TAG", CPL_TYPE_STRING);
    cpl_table_new_column( result->table, "KEYWORD", CPL_TYPE_STRING);
    cpl_table_new_column( result->table, "TYPE", CPL_TYPE_STRING);
    cpl_table_new_column( result->table, "VALUE", CPL_TYPE_STRING);
    result->proplist         = NULL;
    result->alwaysoverride     = 0;
    return result;
}

sph_keyword_reader* 
sph_keyword_reader_new_load( const char* infile ) {
    sph_keyword_reader*        result        = NULL;
    FILE*                   filehandle          = NULL;
    char                line[2056];
    char*                str                        = NULL;
    char                s1[ASCIILINESZ+1];
    char                s2[ASCIILINESZ+1];
    char                s3[ASCIILINESZ+1];
    char                s4[ASCIILINESZ+1];

    result = sph_keyword_reader_new();
    
    if ( !result ) {
        sph_error_raise( SPH_ERROR_GENERAL,
                __FILE__, __func__, 
                __LINE__, SPH_ERROR_ERROR, 
                "Could not create empty keyword reader.");
        return NULL;
    }
    
    filehandle = fopen( infile, "r" );
    str = fgets( line, 2048, filehandle );
    while ( str ) {
        if ( line[0] != '#' ) {
            sph_keyword_reader_scanline( line, s1, s2, s3, s4);
            //printf( "Line: %s --- %s -- %s -- %s\n", s1, s2, s3, s4);
            cpl_table_set_size( result->table, cpl_table_get_nrow(result->table) + 1 );
            cpl_table_set_string( result->table, "TAG", cpl_table_get_nrow(result->table) - 1, s1);
            cpl_table_set_string( result->table, "KEYWORD", cpl_table_get_nrow(result->table) - 1, s2);
            cpl_table_set_string( result->table, "TYPE", cpl_table_get_nrow(result->table) - 1, s3);
            cpl_table_set_string( result->table, "VALUE", cpl_table_get_nrow(result->table) - 1, s4);
        }
        str = fgets( line, 2048, filehandle );
    }
    fclose(filehandle);
    return result;
}

sph_error_code
sph_keyword_reader_set_proplist_from_file( sph_keyword_reader* self,
                                           const char* czFilename,
                                           int extension ) {
    cpl_error_code            rerr = CPL_ERROR_NONE;
    
    if ( !self ) {
        SPH_NO_SELF;
        return SPH_ERROR_GENERAL;
    }
    if ( self->proplist ) {
        cpl_propertylist_delete( self->proplist );
        self->proplist = NULL;
    }
    self->proplist = cpl_propertylist_load( czFilename, extension );
    if ( self->proplist == NULL ) {
        sph_error_raise( SPH_ERROR_GENERAL, 
                __FILE__, __func__, __LINE__, 
                SPH_ERROR_ERROR, "Could not load propertylist.");
        return SPH_ERROR_GENERAL;
    }
    return rerr;
}

char*
sph_keyword_reader_get_value_string( const sph_keyword_reader* self,
                                     const char* czTag ) {
    char*        result;
    const char*        fromprop     = NULL;
    int            ii            = 0;
    int         rowfound    = 0;
    short        found       = 0;
    if ( !self ) {
        SPH_NO_SELF;
        return NULL;
    }
    if ( !self->table ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    result = cpl_calloc( 256, sizeof(char));
    for ( ii = 0; ii < cpl_table_get_nrow(self->table); ++ii ) {
        if ( strcmp( czTag, cpl_table_get_string( self->table, "TAG", ii) ) == 0 ) {
            if ( found == 0 ) {
                strcpy ( result, cpl_table_get_string( self->table, "VALUE", ii) );
                found = 1;
                rowfound = ii;
            }
            else {
                sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__, 
                        __LINE__, SPH_ERROR_WARNING, "Found a multiple tag entry!");
            }
        }        
    }
    if ( !found ) {
        fromprop = cpl_propertylist_get_string( self->proplist, czTag );
    }
    else {
        fromprop = cpl_propertylist_get_string( self->proplist,
                                                cpl_table_get_string( self->table,
                                                                      "KEYWORD",
                                                                      rowfound));
    }
    if ( fromprop != NULL && !self->alwaysoverride ) {
        strncpy ( result, "\0", 2);
        strcat ( result, fromprop );        
    }
    if ( fromprop == NULL && ( found == 0 || self->alwaysoverride ) ) {
        sph_error_raise( SPH_ERROR_GENERAL, 
                __FILE__, __func__, 
                __LINE__, SPH_ERROR_ERROR, "Tag/Keyword not found!");
        return NULL;
    }
    return result;
}

int
sph_keyword_reader_get_value_int( const sph_keyword_reader* self,
                                  const char* czTag ) {
    int         result      = 0;
    int         fromprop    = 0;
    int         ii          = 0;
    int         rowfound    = 0;
    short       found       = 0;
    if ( !self ) {
        SPH_NO_SELF;
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->table ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }

    for ( ii = 0; ii < cpl_table_get_nrow(self->table); ++ii ) {
        if ( strcmp( czTag, cpl_table_get_string( self->table, "TAG", ii) ) == 0 ) {
            if ( found == 0 ) {
                sscanf( cpl_table_get_string( self->table, "VALUE", ii), "%i", &result );
                found = 1;
                rowfound = ii;
            }
            else {
                sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__, 
                        __LINE__, SPH_ERROR_WARNING, "Found a multiple tag entry!");
            }
        }       
    }
    if ( !found ) {
        fromprop = cpl_propertylist_get_int( self->proplist, czTag );
    }
    else {
        fromprop = cpl_propertylist_get_int( self->proplist, cpl_table_get_string( self->table, "KEYWORD", rowfound)  );     
    }
    if ( cpl_error_get_code() == CPL_ERROR_NONE && !self->alwaysoverride ) {
        result = fromprop;
    }
    if ( cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND && ( found == 0 || self->alwaysoverride ) ) {
        sph_error_raise( SPH_ERROR_GENERAL, 
                __FILE__, __func__, 
                __LINE__, SPH_ERROR_ERROR, "Tag/Keyword not found!");
        return 0;
    }
    return result;
}

double
sph_keyword_reader_get_value_double( const sph_keyword_reader* self,
                                     const char* czTag ) {
    double        result    = 0.0;
    double      fromprop    = 0;
    int         ii          = 0;
    int         rowfound    = 0;
    short       found       = 0;
    if ( !self ) {
        SPH_NO_SELF;
        return 0;
    }
    if ( !self->table ) {
        SPH_NULL_ERROR;
        return 0;
    }

    for ( ii = 0; ii < cpl_table_get_nrow(self->table); ++ii ) {
        if ( strcmp( czTag, cpl_table_get_string( self->table, "TAG", ii) ) == 0 ) {
            if ( found == 0 ) {
                sscanf( cpl_table_get_string( self->table, "VALUE", ii), "%lf", &result );
                found = 1;
                rowfound = ii;
            }
            else {
                sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__, 
                        __LINE__, SPH_ERROR_WARNING, "Found a multiple tag entry!");
            }
        }       
    }
    if ( !found ) {
        fromprop = cpl_propertylist_get_double( self->proplist, czTag );
    }
    else {
        fromprop = cpl_propertylist_get_double( self->proplist, cpl_table_get_string( self->table, "KEYWORD", rowfound)  );     
    }
    if ( cpl_error_get_code() == CPL_ERROR_NONE && !self->alwaysoverride ) {
        result = fromprop;
    }
    if ( cpl_error_get_code() == CPL_ERROR_DATA_NOT_FOUND && ( found == 0 || self->alwaysoverride ) ) {
        sph_error_raise( SPH_ERROR_GENERAL, 
                __FILE__, __func__, 
                __LINE__, SPH_ERROR_ERROR, "Tag/Keyword not found!");
        return 0;
    }
    return result;
}

sph_error_code 
sph_keyword_reader_delete( sph_keyword_reader* self ) {
    sph_error_code        rerr = CPL_ERROR_NONE;
    if ( self ) {
        if ( self->table ) {
            cpl_table_delete( self->table );
        }
        if ( self->proplist ) {
        	cpl_propertylist_delete( self->proplist );
        }
        cpl_free( self );
    }
    return rerr;
}
