/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * sph_keywordnode.c
 *
 *  Created on: Apr 2, 2009
 *      Author: pavlov
 */


#include "sph_error.h"
#include "sph_keywordnode.h"

#include <cpl.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>


sph_error_code SPH_KEYWORDNODE_NO_NODE  = SPH_KEYWORDNODE_START + 0;
sph_error_code SPH_KEYWORDNODE_WRONG_FRAME_KEYWORD_TYPE  = SPH_KEYWORDNODE_START + 1;
sph_error_code SPH_KEYWORDNODE_WRONG_FRAMES_KEYWORD_TYPE  = SPH_KEYWORDNODE_START + 2;
sph_error_code SPH_KEYWORDNODE_WRONG_MASTERFRAME_KEYWORD_TYPE = SPH_KEYWORDNODE_START + 3;
sph_error_code SPH_KEYWORDNODE_WRONG_DOUBLEIMAGE_KEYWORD_TYPE = SPH_KEYWORDNODE_START + 4;

 /**
 * Create a new sph_keywordnode-child with no name  from specified sph_keywordnode-parent
 */
sph_keywordnode *sph_keywordnode_new(sph_keywordnode *parent)
{

   // printf("keyword_engine::sph_keywordnode_new... \n");

    sph_keywordnode *pthis;
    pthis = (sph_keywordnode*) cpl_malloc(sizeof(sph_keywordnode));

    pthis->name[0] = '\0';
    pthis->nr = -1;
    pthis->type = keyword_GROUP;
    pthis->parent = parent;
    pthis->prev = NULL;
    pthis->next = NULL;
    // same level
    if (pthis->parent != NULL) {
       if (parent->first == NULL) {
          // first child!
          parent->first = pthis;
          parent->last = pthis;
       } else {
          pthis->prev = parent->last;
          parent->last->next = pthis;
          parent->last = pthis;
       }
    }
    // childs
    pthis->first = NULL;
    pthis->last = NULL;
    pthis->cVal = '\0';
    pthis->iVal = 0;
    pthis->dVal = 0.0;
    pthis->sVal = NULL;
    pthis->sSize = 0;
    pthis->frame = NULL;
    pthis->frames = NULL;
    pthis->masterframe = NULL;
    pthis->doubleimage = NULL;
    return pthis;
}

/**
 * Create a new sph_keywordnode-child with a given name from specified sph_keywordnode-parent
 */

sph_keywordnode *sph_keywordnode_new_name(sph_keywordnode *parent,
                                          const char *name)
{
    sph_keywordnode *pthis;
    pthis = (sph_keywordnode*) cpl_malloc(sizeof(sph_keywordnode));

    pthis->nr = -1;
    pthis->type = keyword_GROUP;
    pthis->parent = parent;
    pthis->prev = NULL;
    pthis->next = NULL;
    // same level
    if (parent != NULL) {
    if (parent->first == NULL) {
        // first child!
        parent->first = pthis;
        parent->last = pthis;
    } else {
        pthis->prev = parent->last;
        parent->last->next = pthis;
        parent->last = pthis;
    }
    }
    // childs
    pthis->first = NULL;
    pthis->last = NULL;
    pthis->cVal = '\0';
    pthis->iVal = 0;
    pthis->dVal = 0.0;
    pthis->sVal = NULL;
    pthis->sSize = 0;
    pthis->frame = NULL;
    pthis->frames = NULL;
    pthis->masterframe = NULL;
    pthis->doubleimage = NULL;
    if (name == NULL) {
    pthis->name[0] = '\0';
    } else {
    strncpy(pthis->name, name, keywordMAX_CFG_NAME_LEN);
    pthis->name[keywordMAX_CFG_NAME_LEN - 1] = '\0';
    }

    return pthis;
}

/**
 * Create a new sph_keywordnode-child with a specific numbering from specified sph_keywordnode-parent
 */
sph_keywordnode *sph_keywordnode_new_nr(sph_keywordnode *parent, int nr)
{
    sph_keywordnode *cfg;
    sph_keywordnode *pthis;

    pthis = (sph_keywordnode*)cpl_malloc(sizeof(sph_keywordnode));

    pthis->name[0] = '\0';
    pthis->type = keyword_GROUP;
    pthis->parent = parent;
    pthis->prev = NULL;
    pthis->next = NULL;
    // same level
    if (parent != NULL) {
    if (parent->first == NULL) {
        // first child!
        parent->first = pthis;
        parent->last = pthis;
    } else {
        pthis->prev = parent->last;
        parent->last->next = pthis;
        parent->last = pthis;
    }
    }
    // childs
    pthis->first = NULL;
    pthis->last = NULL;
    pthis->cVal = '\0';
    pthis->iVal = 0;
    pthis->dVal = 0.0;
    pthis->sVal = NULL;
    pthis->sSize = 0;
    pthis->frame = NULL;
    pthis->frames = NULL;
    pthis->masterframe = NULL;
    pthis->doubleimage = NULL;
    if (nr == 0) {
    // automatic numbering
    if (pthis->prev == NULL) {
        nr = 1;
    } else {
        nr = pthis->prev->nr + 1;
    }
    }
    pthis->nr = nr;
    // insert child in correct position (increasing numbers!)
    while ((pthis->prev != NULL) && (pthis->prev->nr > pthis->nr)){
    pthis->prev->next = pthis->next;
    if (pthis->next != NULL) {
        pthis->next->prev = pthis->prev;
    }
    pthis->next = pthis->prev;
    if (pthis->prev->prev != NULL) {
        pthis->prev->prev->next = pthis;
    }
    pthis->prev = pthis->prev->prev;
    pthis->next->prev = pthis;
    }
    if (parent != NULL) {
    cfg = pthis;
    while (cfg->prev != NULL) {
        cfg = cfg->prev;
    }
    parent->first = cfg;
    cfg = pthis;
    while (cfg->next != NULL) {
        cfg = cfg->next;
    }
    pthis->parent->last = cfg;
    }
    return pthis;
}

/*
 * Destroy a given keyword n-tree (or n-subtree), starting from a given parent
 */
void sph_keywordnode_destroy(sph_keywordnode *pthis)
{
    if (pthis != NULL) {
        while (pthis->first != NULL) {
            /* go down to the last keyword node */
            sph_keywordnode_destroy(pthis->first);
        }
        if (pthis->parent != NULL) {
            if (pthis->parent->first == pthis) {
                pthis->parent->first = pthis->next;
            }
            if (pthis->parent->last == pthis) {
                pthis->parent->last = pthis->prev;
            }
            if (pthis->prev != NULL) {
                pthis->prev->next = pthis->next;
            }
            if (pthis->next != NULL) {
                pthis->next->prev = pthis->prev;
            }
        }
      cpl_free(pthis->sVal );
      cpl_free(pthis); /* DELETE NODE */
    }
}

/**
 * A specific node relative to the current node can be located and/or created by
 * using a relative path. This path uses the syntax for qualified names. If the
 * trailing number is equal to 0 a new node with an automatic generated number is created.
 * @param qalName This argument defines a relative path from the current node.
 * I f this argument is NULL, the current node is used.
 * @param force If the destination node is not found a new node is created (including
 * the intermediate nodes) if this flag is 1.
 */
sph_keywordnode *sph_keywordnode_LocateByName(sph_keywordnode *pthis, const char *qualName, int force)
{
    int  i;
    char bname[keywordMAX_CFG_NAME_LEN];
    int  bnr;
    sph_keywordnode *child;

    //printf("keyword::sph_keywordnode_LocateByName(%s, %d)\n", qualName, force);
    if (qualName == NULL) {
    return pthis;
    } else if (*qualName == '\0') {
    return pthis;
    } else if ((pthis->first == NULL) && !force) {
    return NULL;
    }
    if (isalpha(*qualName)) {
    // parse base name
    i = 0;
    while (isalpha(*qualName) || *qualName=='_' ) {
        bname[i++] = *qualName++;
    }
    bname[i] = '\0';
    //printf("  base name = %s\n", bname);
    if (*qualName == '.') {
        qualName++;
    }
    //printf("  qualName = %s\n", qualName);

    for (child = pthis->first; child != NULL; child = child->next) {
           // printf("  child->name[0] = %s\n", child->name);
        if (child->name[0] != '\0') {
            if (strcmp(child->name, bname) == 0) {
              return sph_keywordnode_LocateByName(child, qualName, force);
           }
        }
    }

    if (force) {
        child = sph_keywordnode_new_name(pthis, bname);
        return sph_keywordnode_LocateByName(child, qualName, force);
    } else {
        return NULL;
    }
    } else if (isdigit(*qualName)) {
    // parse number
    i = 0;
    while (isdigit(*qualName)) {
        bname[i++] = *qualName++;
    }
    bname[i] = '\0';
    bnr = atoi(bname);
    //printf("  base nr = %d\n", bnr);
    if (*qualName == '.') {
        qualName++;
    }
    child = sph_keywordnode_LocateByNr(pthis, bnr, force);
    if (child == NULL) {
        return NULL;
    }
    return sph_keywordnode_LocateByName(child, qualName, force);
    } else {
      // illegal name component!
      return NULL;
    }
}

/*
 * Adapter function to the sph_keywordnode_LocateByName function with force = 0.
 * A specific node relative to the current node is found by
 * using a relative path. This path uses the syntax for qualified names.
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qalName This argument defines a relative path from the current node.
 */
sph_keywordnode *sph_keywordnode_get_keywordnode_by_name(sph_keywordnode *pthis, const char *qualName)
{
    return sph_keywordnode_LocateByName(pthis, qualName, 0);
}


/**
 * A specific child node is located by using a number. If nr is 0 and force is equal to 1, a new
 * node with a new number is generated (automatic numbering).
 * @param nr This argument defines a number (child).
 * @param force If the destination node is not found and force is 1, a new node is created.
 */
sph_keywordnode *sph_keywordnode_LocateByNr(sph_keywordnode *pthis, int nr, int force)
{
    sph_keywordnode *child;

    //printf("keyword::sph_keywordnode_LocateByNr(%d, %d)\n", nr, force);
    for (child = pthis->first; child != NULL; child = child->next) {
    if (child->nr == nr) {
        return child;
    } else if (child->nr > nr) {
        break;
    }
    }
    if (force) {
        child = sph_keywordnode_new_nr(pthis, nr);
        return child;
    } else {
       return NULL;
    }
}

/*
 * Build a qualified name in order to save the n-tree keywords into configuration files in terms of
 * of brunch.leaf
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qualName A qualified name to fill it up (return)
 * @param size  A size of the qualName string /// XXX: Size is not used!
 */
int sph_keywordnode_BuildQualifiedName(const sph_keywordnode *pthis,
                                       char *qualName,
                                       int size)
{
    char nrStr[8];

    if (pthis->parent != NULL) {
       if (!sph_keywordnode_BuildQualifiedName(pthis->parent, qualName, size)) { // XXX: SIze is not used !
          qualName[0] = '\0';
          return 0;
       }
    } else { /// XXX: Dont need this else !
      qualName[0] = '\0';
    }
    if (pthis->name[0] != '\0') {
       if (qualName[0] != '\0') {
         strcat(qualName, ".");
       }
       strcat(qualName, pthis->name);
       return 1;
    } else if (pthis->nr != -1) {
         sprintf(nrStr, "%d", pthis->nr);
         strcat(qualName, nrStr);
         return 1;
    } else {
       qualName[0] = '\0';
       return 1;
    }
}

/*
 * Build a qualified name as above and keywordlist of one given brunch. Keywordlist contains
 * keyword-value pairs: keywords are named nodes whereas values are their numbered childs (if exists).
 *
 * For example, in the case of the following keyword CAMERA1.FILTER1.DIT2 one gets
 * keywordlist:  "NAME"       "VALUE"
 *                 CAMAERA             1
 *              FILTER           1
 *              DIT                 2
 * Note: If named node doesn't have any numbered childs it will be ignored,
 * and not presented in the output  keywordlist
 */
int sph_keywordnode_BuildQualifiedNameList(sph_keywordnode *pthis, char *qualName, int size, cpl_propertylist* keywordlist)
{
    char nrStr[8];
    char keycopy[256];
    int        ii    = 0;

    //printf("Inside BuildQualifiedNameList\n");

    if (pthis->parent != NULL) {
       if (!sph_keywordnode_BuildQualifiedNameList(pthis->parent, qualName, size, keywordlist)) {
          qualName[0] = '\0';
          return 0;
       }
    } else {
      qualName[0] = '\0';
    }
    if (pthis->name[0] != '\0') {
           //cpl_propertylist_append_int(keywordlist, pthis->name, -999);
           //printf("cpl_propertylist size: %d \n", cpl_propertylist_get_size(keywordlist));
        if (qualName[0] != '\0') {
         strcat(qualName, ".");
       }
       strcat(qualName, pthis->name);
       return 1;
    } else if (pthis->nr != -1) {
           //cpl_propertylist_append_int(keywordlist, pthis->parent->name, -999);
           (void)strcpy( keycopy, pthis->parent->name );
           for (ii = 0; ii < (int)strlen(keycopy); ++ii) {
               if ( ii < 256 ) {
                   if ( keycopy[ii] == '_' ) {
                       keycopy[ii] = ' ';
                   }
               }
           }
         cpl_propertylist_append_int(keywordlist, keycopy, pthis->nr);
         sprintf(nrStr, "%d", pthis->nr);
         strcat(qualName, nrStr);
         return 1;
    } else {
       qualName[0] = '\0';
       return 1;
    }
}




/**
 * Set a char value of an existing keyword node or create a new node with the given value.
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qalName This argument defines a relative path from the current node.
 * @param value The character value.
 * @return 1 if the value could be set, otherwise 0.
 */
int sph_keywordnode_SetChar(sph_keywordnode *pthis, const char *qualName, char value)
{
    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 1);
    if (obj == NULL) {
      //write a message in log
       return 0;
    }
    obj->type = keyword_CHAR;
    obj->cVal = value;
    return 1;
}

/**
 * Set an integer value of an existing keyword node or create a new node with the given value.
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qalName This argument defines a relative path from the current node.
 * @param value The integer value.
 * @return 1 if the value could be set, otherwise 0.
 */
int sph_keywordnode_SetInt(sph_keywordnode *pthis, const char *qualName, int value)
{
    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 1);
    if (obj == NULL) {
       //write a message in log
       return 0;
    }
    obj->type = keyword_INT;
    obj->iVal = value;
    return 1;
}

/**
 * Set a double value of an existing keyword node or create a new node with the given value.
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qalName This argument defines a relative path from the current node.
 * @param value The double value.
 * @return 1 if the value could be set, otherwise 0.
 */
int sph_keywordnode_SetDouble(sph_keywordnode *pthis, const char *qualName, double value)
{
    sph_keywordnode *obj;
    //printf("keyword::sph_keywordnode_SetDouble(%s, %f)\n", qualName, value);

    obj = sph_keywordnode_LocateByName(pthis, qualName, 1);
    if (obj == NULL) {
       //write a message in log
       return 0;
    }
    obj->type = keyword_DOUBLE;
    obj->dVal = value;
    return 1;
}


/**
 * Set a string value of an existing keyword node or create a new node with the given value.
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qalName This argument defines a relative path from the current node.
 * @param value The string value.
 * @return 1 if the value could be set, otherwise 0.
 */
int sph_keywordnode_SetString(sph_keywordnode *pthis, const char *qualName, const char *value)
{
    sph_keywordnode *obj;

    //printf("keyword::sph_keywordnode_SetString(%s, %s)\n", qualName, value);
    obj = sph_keywordnode_LocateByName(pthis, qualName, 1);
    if (obj == NULL) {
        //write a message in log
       return 0;
    }
    obj->type = keyword_STRING;
    if (obj->sSize < strlen(value)) {
        obj->sSize = strlen(value)+1;
    }
    if (obj->sVal == NULL) {
        obj->sVal = (char*)cpl_malloc(obj->sSize*sizeof(char));
    } else {
       obj->sVal = (char*)cpl_realloc(obj->sVal, obj->sSize*sizeof(char));
    }
    strncpy(obj->sVal, value, obj->sSize);
    obj->sVal[obj->sSize-1]='\0';
    return 1;
}


/**
 * Set a cpl_frame pointer value of an existing keyword node or create a new node
 * with the given value.
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qalName This argument defines a relative path from the current node.
 * @param value The frame
 * @return 1 if the value could be set, otherwise 0.
 */
int sph_keywordnode_SetSingleFrame(sph_keywordnode *pthis, const char *qualName, const cpl_frame *frame)
{

    sph_keywordnode *obj;
    //cpl_frame* lframe=NULL;
    //lframe = cpl_frameset_get_first(frameset);
    //printf("In setframes1: File name of the first rawframe: %s\n", cpl_frame_get_filename(lframe));
    //lframe = NULL;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 1);
    if (obj == NULL) {
        //write a message in log
       return 0;
    }

    obj->type = keyword_CPLFRAME;
    if (obj->frame == NULL) {
        obj->frame = cpl_frame_duplicate(frame);
     }else {
        cpl_frame_delete(obj->frame);
        obj->frame = cpl_frame_duplicate(frame);
    }
    if (cpl_error_get_code() != CPL_ERROR_NONE){
        SPH_RAISE_CPL;
        return 0;
    }

    //lframe = cpl_frameset_get_first(obj->frames);
    //printf("In setframes2: from tree: File name of the first rawframe: %s\n", cpl_frame_get_filename(lframe));

    return 1;
}


/**
 * Set a cpl_frameset pointer value of an existing keyword node or create a new node
 * with the given value.
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qalName This argument defines a relative path from the current node.
 * @param value The frame
 * @return 1 if the value could be set, otherwise 0.
 */
int sph_keywordnode_SetFrames(sph_keywordnode *pthis, const char *qualName, const cpl_frameset *frameset)
{

    sph_keywordnode *obj;
    //cpl_frame* lframe=NULL;
    //lframe = cpl_frameset_get_first(frameset);
    //printf("In setframes1: File name of the first rawframe: %s\n", cpl_frame_get_filename(lframe));
    //lframe = NULL;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 1);
    if (obj == NULL) {
        //write a message in log
       return 0;
    }

    obj->type = keyword_CPLFRAMESET;
    if (obj->frames == NULL) {
        obj->frames = cpl_frameset_duplicate(frameset);
     }else {
        cpl_frameset_delete(obj->frames);
        obj->frames = cpl_frameset_duplicate(frameset);
    }
    if (cpl_error_get_code() != CPL_ERROR_NONE){
        SPH_RAISE_CPL;
        return 0;
    }

    //lframe = cpl_frameset_get_first(obj->frames);
    //printf("In setframes2: from tree: File name of the first rawframe: %s\n", cpl_frame_get_filename(lframe));

    return 1;
}


/**
 * Set a sph_master_frame pointer value of an existing keyword node or create a new node
 * with the given value.
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qalName This argument defines a relative path from the current node.
 * @param value The frame
 * @return 1 if the value could be set, otherwise 0.
 */
int sph_keywordnode_SetMasterFrame(sph_keywordnode *pthis, const char *qualName, sph_master_frame *masterframe)
{

    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 1);
    if (obj == NULL) {
        //write a message in log
       return 0;
    }

    printf("sph_keywordnode Set Master Frame\n");

    obj->type = keyword_SPHMASTERFRAME;
    if (obj->masterframe == NULL) {
        obj->masterframe = sph_master_frame_duplicate(masterframe);
     }else {
        sph_master_frame_delete(obj->masterframe);
        obj->masterframe = sph_master_frame_duplicate(masterframe);
    }

    /*
    if (sph_error_get_last_code() != CPL_ERROR_NONE){
        SPH_RAISE_CPL;
        return 0;
    }
    */

    return 1;
}

/**
 * Set a sph_double_image pointer value of an existing keyword node or create a new node
 * with the given value.
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @param qalName This argument defines a relative path from the current node.
 * @param value The frame
 * @return 1 if the value could be set, otherwise 0.
 */
int sph_keywordnode_SetDoubleImage(sph_keywordnode *pthis, const char *qualName, sph_double_image *doubleimage)
{

    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 1);
    if (obj == NULL) {
        //write a message in log
       return 0;
    }

    printf("sph_keywordnode Set Double Image\n");

    obj->type = keyword_SPHDOUBLEIMAGE;
    if (obj->doubleimage == NULL) {
        obj->doubleimage = sph_double_image_duplicate(doubleimage);
     }else {
        sph_double_image_delete(obj->doubleimage);
        obj->doubleimage = sph_double_image_duplicate(doubleimage);
    }

    /*
    if (sph_error_get_last_code() != CPL_ERROR_NONE){
        SPH_RAISE_CPL;
        return 0;
    }
    */

    return 1;
}




/**
 * Get the character value of an existing keyword node. If the node does not exists, 0 is returned.
 * @param  pthis This defines a pointer to the given sph_keywordnode.
 * @return value The returned character value.
 */
char sph_keywordnode_get_char(sph_keywordnode *pthis)
{
    sph_keywordnode *obj = pthis;
    char                value = '0';

    if (obj == NULL) {
      //TODO: write a message in log
      return 0;
    }
    switch (obj->type) {
    case keyword_CHAR:
        value = obj->cVal;
        return value;
    case keyword_INT:
        value = (char)(obj->iVal);
        return value;
    case keyword_DOUBLE:
        value = (char)(obj->dVal);
        return value;
    case keyword_STRING:
        value = obj->sVal[0];
        return value;
    default:;
        //TODO: write message in log
        return 0;
    }
}

/**
 * Get the character value of an existing keyword node. If the node does not exists, 0 is returned.
 * @param  pthis This defines a pointer to the given sph_keywordnode.
 *
 * @return the value.
 *
 */
int sph_keywordnode_get_int(sph_keywordnode *pthis )
{
    sph_keywordnode* obj     = NULL;
    int              value   = 0;

    obj = pthis;
    if (obj == NULL) {
        //TODO:raise error
        return 0;
    }
    switch (obj->type) {
    case keyword_CHAR:
        value = (int)(obj->cVal);
        return value;
    case keyword_INT:
        value = obj->iVal;
        return value;
    case keyword_DOUBLE:
        value = (int)(obj->dVal);
        return value;
    case keyword_STRING:
        //TODO:raise error
        return 0;
    default:;
    return 0;
    }
}

/**
 * Get the double value of an existing keyword node. If the node does not exists, 0 is returned.
 * @param  pthis This defines a pointer to the given sph_keywordnode.
 * @param qalName This argument defines a relative path from the current node.
 * @param value The returned character value.
 * @return 1 if the value could be retrieved, otherwise 0.
 */
double sph_keywordnode_get_double(sph_keywordnode *pthis )
{
    sph_keywordnode *obj    = pthis;
    double          value       = 0.0;
    if (obj == NULL) {
       //TODO:write a message in log
       return 0;
    }
    switch (obj->type) {
    case keyword_CHAR:
        value = (double)(obj->cVal);
        return value;
    case keyword_INT:
        value = (double)(obj->iVal);
        return value;
    case keyword_DOUBLE:
        //printf("keyword::sph_keywordnode_GetDouble::keyword_DOUBLE\n");
        value = obj->dVal;
        return value;
    case keyword_STRING:
        return 0;
    default:;
        //TODO:write a message in log
        return 0;
    }
}

/**
 * Get the string value of an existing keyword node. If the node does not exists, 0 is returned.
 * @param  pthis This defines a pointer to the given sph_keywordnode.
 * @param qalName This argument defines a relative path from the current node.
 * @param value The returned character value.
 * @return 1 if the value could be retrieved, otherwise 0.
 */
const char* sph_keywordnode_get_string(sph_keywordnode *pthis )
{
    sph_keywordnode *obj;

    obj = pthis;
    if (obj == NULL) {
      //TODO: write a message in log
      return NULL;
    }
    switch (obj->type) {
    case keyword_CHAR:
        pthis->value[0]=obj->cVal;
        pthis->value[1]='\0';
        return (const char*)pthis->value;
    case keyword_INT:
        snprintf(pthis->value, 256, "%d", obj->iVal);
        return (const char*)pthis->value;
    case keyword_DOUBLE:
        snprintf(pthis->value, 256, "%f", obj->dVal);
        return (const char*)pthis->value;
    case keyword_STRING:
        sprintf(pthis->value, "%s" , obj->sVal);
        return (const char*)pthis->value;
    default:;
        //TODO: write a message in log
        return NULL;
    }
}


/**
 * Get the character value of an existing keyword node. If the node does not exists, 0 is returned.
 * @param  pthis This defines a pointer to the given keywordnode.
 * @param qalName This argument defines a relative path from the current node.
 * @param value The returned character value.
 * @return 1 if the value could be retrieved, otherwise 0.
 */
int sph_keywordnode_GetChar(sph_keywordnode *pthis, const char *qualName, char *value)
{
    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 0);
    if (obj == NULL) {
      //write a message in log
      return 0;
    }
    switch (obj->type) {
    case keyword_CHAR:
        *value = obj->cVal;
        return 1;
    case keyword_INT:
        *value = (char)(obj->iVal);
        return 1;
    case keyword_DOUBLE:
        *value = (char)(obj->dVal);
        return 1;
    case keyword_STRING:
        *value = obj->sVal[0];
        return 1;
    default:;
        return 0;
    }
}

/**
 * Get the character value of an existing keyword node. If the node does not exists, 0 is returned.
 * @param  pthis This defines a pointer to the given keywordnode.
 * @param qalName This argument defines a relative path from the current node.
 * @param value The returned character value.
 * @return 1 if the value could be retrieved, otherwise 0.
 */
int sph_keywordnode_GetInt(sph_keywordnode *pthis, const char *qualName, int *value)
{
    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 0);
    if (obj == NULL) {
    return 0;
    }
    switch (obj->type) {
    case keyword_CHAR:
        *value = (int)(obj->cVal);
        return 1;
    case keyword_INT:
        *value = obj->iVal;
        return 1;
    case keyword_DOUBLE:
        *value = (int)(obj->dVal);
        return 1;
    case keyword_STRING:
        return 0;
    default:;
        return 0;
    }
}


/**
 * Get the double value of an existing keyword node. If the node does not exists, 0 is returned.
 * @param  pthis This defines a pointer to the given keywordnode.
 * @param qalName This argument defines a relative path from the current node.
 * @param value The returned character value.
 * @return 1 if the value could be retrieved, otherwise 0.
 */
int sph_keywordnode_GetDouble(sph_keywordnode *pthis, const char *qualName, double *value)
{
    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 0);
    if (obj == NULL) {
       //write a message in log
       return 0;
    }
    switch (obj->type) {
    case keyword_CHAR:
        *value = (double)(obj->cVal);
        return 1;
    case keyword_INT:
        *value = (double)(obj->iVal);
        return 1;
    case keyword_DOUBLE:
        //printf("keyword::GetDouble::keyword_DOUBLE\n");
        *value = obj->dVal;
        return 1;
    case keyword_STRING:
        return 0;
    default:;
        return 0;
    }
}

/**
 * Get the string value of an existing keyword node. If the node does not exists, 0 is returned.
 * @param  pthis This defines a pointer to the given keywordnode.
 * @param qalName This argument defines a relative path from the current node.
 * @param value The returned character value.
 * @return 1 if the value could be retrieved, otherwise 0.
 */
int sph_keywordnode_GetString(sph_keywordnode *pthis, const char *qualName, char *value, int size)
{
    sph_keywordnode *obj;

    if (size < 2) {
      //write a message in log
      return 0;
    }
    obj = sph_keywordnode_LocateByName(pthis, qualName, 0);
    if (obj == NULL) {
      //write a message in log
      return 0;
    }
    switch (obj->type) {
    case keyword_CHAR:
        value[0] = obj->cVal;
        value[1] = '\0';
        return 1;
    case keyword_INT:
        snprintf(value, size, "%d", obj->iVal);
        return 1;
    case keyword_DOUBLE:
        snprintf(value, size, "%f", obj->dVal);
        return 1;
    case keyword_STRING:
        strncpy(value, obj->sVal, size);
        value[size - 1] = '\0';
        return 1;
    default:;
        return 0;
    }
}


/** Get a pointer to the single cpl_frame of an existing keyword node.
 * @return 1 if the value could be retrieved, otherwise 0.
 */
cpl_frame* sph_keywordnode_GetSingleFrame(sph_keywordnode* pthis, const char *qualName)

{
    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 0);
    if (obj == NULL) {
      //write a message in log
      sph_error_raise(SPH_KEYWORDNODE_NO_NODE,
              __FILE__, __func__, __LINE__,
              SPH_ERROR_WARNING, "The node/object for this keywordname doesn't exist!");
     // return 0;
      return NULL;
    }
    if (obj->type != keyword_CPLFRAME) {
        //write a message in log
         sph_error_raise(SPH_KEYWORDNODE_WRONG_FRAME_KEYWORD_TYPE,
                 __FILE__, __func__, __LINE__,
                 SPH_ERROR_WARNING,
                 "The node/object for this keywordname is not of a cpl_frame type."
                 "The type of this node is %d", sph_keywordnode_GetType(obj));
        //return 0;
        return NULL;
    } else {
      cpl_frame *frame = cpl_frame_duplicate(obj->frame);
     // cpl_frame *lframe = cpl_frameset_get_first(obj->frames);
     // printf("In getframes2: from tree: File name of the first rawframe: %s\n", cpl_frame_get_filename(lframe));
      return frame;
      //return 1;
    }
}

/** Get a pointer to the single cpl_frame of an existing keyword node.
 */
sph_master_frame* sph_keywordnode_GetMasterFrame(sph_keywordnode* pthis, const char *qualName)

{
    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 0);
    if (obj == NULL) {
      //write a message in log
      sph_error_raise(SPH_KEYWORDNODE_NO_NODE,
              __FILE__, __func__, __LINE__,
              SPH_ERROR_WARNING, "The node/object for this keywordname doesn't exist!");
     // return 0;
      return NULL;
    }
    if (obj->type != keyword_SPHMASTERFRAME) {
        //write a message in log
         sph_error_raise(SPH_KEYWORDNODE_WRONG_MASTERFRAME_KEYWORD_TYPE,
                 __FILE__, __func__, __LINE__,
                 SPH_ERROR_WARNING,
                 "The node/object for this keywordname is not of a sph_master_frame type."
                 "The type of this node is %d", sph_keywordnode_GetType(obj));
        //return 0;
        return NULL;
    } else {
      sph_master_frame *masterframe = sph_master_frame_duplicate(obj->masterframe);
      return masterframe;
    }
}

/** Get a pointer to the single cpl_frame of an existing keyword node.
 */
sph_double_image* sph_keywordnode_GetDoubleImage(sph_keywordnode* pthis, const char *qualName)

{
    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 0);
    if (obj == NULL) {
      //write a message in log
      sph_error_raise(SPH_KEYWORDNODE_NO_NODE,
              __FILE__, __func__, __LINE__,
              SPH_ERROR_WARNING, "The node/object for this keywordname doesn't exist!");
     // return 0;
      return NULL;
    }
    if (obj->type != keyword_SPHDOUBLEIMAGE) {
        //write a message in log
         sph_error_raise(SPH_KEYWORDNODE_WRONG_DOUBLEIMAGE_KEYWORD_TYPE,
                 __FILE__, __func__, __LINE__,
                 SPH_ERROR_WARNING,
                 "The node/object for this keywordname is not of a sph_double_image type."
                 "The type of this node is %d", sph_keywordnode_GetType(obj));
        //return 0;
        return NULL;
    } else {
      sph_double_image *doubleimage = sph_double_image_duplicate(obj->doubleimage);
      return doubleimage;
    }
}




/** Get a pointer to the cpl_frameset of an existing node.
 */
cpl_frameset* sph_keywordnode_GetFrames(sph_keywordnode* pthis, const char *qualName)
{
    sph_keywordnode *obj;

    obj = sph_keywordnode_LocateByName(pthis, qualName, 0);
    if (obj == NULL) {
      //write a message in log
      sph_error_raise(SPH_KEYWORDNODE_NO_NODE,
              __FILE__, __func__, __LINE__,
              SPH_ERROR_WARNING, "The node/object for this keywordname doesn't exist!");
     // return 0;
      return NULL;
    }
    if (obj->type != keyword_CPLFRAMESET) {
        //write a message in log
         sph_error_raise(SPH_KEYWORDNODE_WRONG_FRAMES_KEYWORD_TYPE,
                 __FILE__, __func__, __LINE__,
                 SPH_ERROR_WARNING,
                 "The node/object for this keywordname is not of a cpl_frameset type."
                 "The type of this node is %d", sph_keywordnode_GetType(obj));
        //return 0;
        return NULL;
    } else {
      cpl_frameset *frameset = cpl_frameset_duplicate(obj->frames);
     // cpl_frame *lframe = cpl_frameset_get_first(obj->frames);
     // printf("In getframes2: from tree: File name of the first rawframe: %s\n", cpl_frame_get_filename(lframe));
      return frameset;
      //return 1;
    }
}


/*
 * Load a keyword configuration from a given file.
 * @param This argument defines filename with a saved keyword configurations
 * @return Pointer to the keyword root, or NULL.
 */
 sph_keywordnode *sph_keywordnode_load_configuration( const char *cfgName)
{
    sph_keywordnode    *root;
    FILE      *fp;
    char      *lcfglDir;
    char       path[256];
    int        iVal;
    double     dVal;
    int        lineNr = 0;
    char       line[1024];
    char       varName[10*keywordMAX_CFG_NAME_LEN];
    char       varValue[1024];
    char      *strPtr;
    int        i;
    int        j;

    //printf("keyword::sph_keywordnode_LoadConfiguration(%s)\n", cfgName);
    root = sph_keywordnode_new(NULL);

    if ( root == NULL ) {

        sph_error_raise( SPH_ERROR_GENERAL,
                    __FILE__, __func__,
                    __LINE__, SPH_ERROR_ERROR,
                    "Could not create a root sph_keywordnode.");

           return NULL;
    }
    if (cfgName == NULL) {
        //write a message in log
       SPH_ERR("No config filename is provided");
       sph_keywordnode_destroy(root);
       return NULL;
    }

    lcfglDir = getenv("SPHROOT");
    if (lcfglDir == NULL) {
    snprintf(path, 256, "%s", cfgName);
    } else {
    snprintf(path, 256, "%s/cfg/%s", lcfglDir, cfgName);
    }
    //printf("  opening file %s\n", path);
    path[255] = '\0';
    fp = fopen(path, "r");
    if (fp == NULL) {
           sph_error_raise( SPH_ERROR_GENERAL,
                        __FILE__, __func__,
                        __LINE__, SPH_ERROR_WARNING,
                        "Could not open the keywordnode configuration file.");
        sph_keywordnode_destroy(root);
        return NULL;
    }

    //printf("Now parsing %s  \n...", cfgName);
    while (fgets(line, 1024, fp) != NULL) {
     int lineLen = strlen(line);
     lineNr++;

    //printf ("   LINE: %s", line);
    // cut line at first comment character
    for (i = 0; i < lineLen; i++) {
        if (line[i] == '#') {
        line[i] = '\0';
        lineLen = i;
        break;
        }
    }
    if (lineLen == 0) {
        // too short!!!
        continue;
    }
    if (line[lineLen - 1] == '\n') {
        line[lineLen - 1] = '\0';
        lineLen--;
    }
    i = 0;
    // skip whitespaces
    while (isspace(line[i])) {
        i++;
    }
    if (!isalpha(line[i])) {
        continue;
    }
    // copy name
    j = 0;
    while (isalnum(line[i]) || (line[i] == '.') || (line[i] == '_' ) ) {
        varName[j++] = line[i++];
    }
    varName[j] = '\0';
    if ((j == 0) || (line[i] == '\0')) {
        continue;
    }
    // skip whitespaces
    while (isspace(line[i])) {
        i++;
    }
    if ((line[i] != '=') || (line[i] == '\0')) {
        continue;
    }
    i++;
    // skip whitespaces
    while (isspace(line[i])) {
        i++;
    }
    if ((line[i] != '"') || (line[i] == '\0')) {
        sprintf(line, "file %s, line %d: quotes missing", path, lineNr);
        // error handling needed
        continue;
    }
    i++;
    if (line[i] == '\0') {
        sprintf(line, "file %s, line %d: closing quotes missing", path, lineNr);
        // error handling needed
        continue;
    }
    j = 0;
    while ((line[i] != '\0') && (line[i] != '"')) {
        varValue[j++] = line[i++];
    }
    if (line[i] != '"') {
        sprintf(line, "file %s, line %d: closing quotes missing", path, lineNr);
        // error handling needed
        continue;
    }
    varValue[j] = '\0';
    //printf("  var = <%s>, value = <%s>\n", varName, varValue);

    //write it in log
    //printf("  Try to guess the value type...\n");
    if (j == 0) {
       sph_keywordnode_SetString(root, varName, varValue);
       continue;
    }
    iVal = (int)strtol(varValue, &strPtr, 10);
    if (*strPtr == '\0') {
        sph_keywordnode_SetInt(root, varName, iVal);
        continue;
    }
    dVal = strtod(varValue, &strPtr);
    if (*strPtr == '\0') {
        sph_keywordnode_SetDouble(root, varName, dVal);
        continue;
    }
    sph_keywordnode_SetString(root, varName, varValue);
    }
    fclose(fp);
    return root;
}

/**
 * Save a keyword configuration into an opened file (given by pointer)
 * @param  pthis This argument defines a pointer to the given sph_keywordnode.
 * @param fp This argument defines a pointer to the opened file
 * @return
 */
int sph_keywordnode_save(const sph_keywordnode *pthis, FILE *fp)
{
    sph_keywordnode *child;
    char      qualName[10*keywordMAX_CFG_NAME_LEN];

    if (pthis->type == keyword_GROUP) {
    for (child = pthis->first; child != NULL; child = child->next) {
        sph_keywordnode_save(child, fp);
    }
    } else {
      if (sph_keywordnode_BuildQualifiedName(pthis, qualName, 10*keywordMAX_CFG_NAME_LEN)) {
        switch (pthis->type) {
        case keyword_CHAR:
            fprintf(fp, "%s = \"%c\"\n", qualName, pthis->cVal);
            break;
        case keyword_INT:
            fprintf(fp, "%s = \"%d\"\n", qualName, pthis->iVal);
            break;
        case keyword_DOUBLE:
            fprintf(fp, "%s = \"%.8e\"\n", qualName, pthis->dVal);
            break;
        case keyword_STRING:
            fprintf(fp, "%s = \"%s\"\n", qualName, pthis->sVal);
            break;
        default:;
        }
      }
    }
    return 1;
}

/*
 * Save a keyword configuration from a given file.
 * @param cfgName This argument defines filename with a saved keyword configurations
 * @return Pointer to the keyword root, or NULL.
 */
int sph_keywordnode_SaveConfiguration(const sph_keywordnode *pthis,
                                      const char *cfgName)
{
    FILE *fp;
    char *lcfglDir;
    char path[256];

    lcfglDir = getenv("SPHROOT");
    if (lcfglDir == NULL) {
    snprintf(path, 256, "%s", cfgName);
    } else {
    snprintf(path, 256, "%s/cfg/%s", lcfglDir, cfgName);
    }
    path[255] = '\0';
    fp = fopen(path, "w");
    if (fp != NULL) {
    sph_keywordnode_save(pthis, fp);
    fclose(fp);
    return 1;
    }
    return 0;
}


/*
 * Get first child of the given keyword node
 * @param pthis This defines a pointer to the given sph_keywordnode
 * @return Ponter to the first child of given sph_keywordnode
 */
sph_keywordnode *sph_keywordnode_GetFirstChild(sph_keywordnode* pthis)
{
    return pthis->first;
}

/*
 * Get next child of the given keyword node
 * @param pthis This defines a ponter to the given sph_keywordnode
 * @return Ponter to the next child of given sph_keywordnode
 */
sph_keywordnode *sph_keywordnode_GetNext(sph_keywordnode* pthis)
{
    return pthis->next;
}


sph_keywordnode *sph_keywordnode_GetParent(sph_keywordnode* pthis){
    return pthis->parent;
}


/*
 * Get name of the given keyword node
 * @param pthis This defines a ponter to the given sph_keywordnode
 * @return string name of the given sph_keywordnode
 */
char *sph_keywordnode_get_name(sph_keywordnode* pthis)
{
    return pthis->name;
}

/*
 * Get nr of the given keyword node
 * @param pthis This defines a ponter to the given sph_keywordnode
 * @return int nr (numbering) of the given sph_keywordnode
 */
int sph_keywordnode_GetNr(const sph_keywordnode* pthis)
{
    return pthis->nr;
}

/*
 * Get type of the given keyword node
 * @param pthis This defines a ponter to the given sph_keywordnode
 * @return keyword_TYPE type (keyword_GROUP, keyword_CHAR, keyword_INT, keyword_DOUBLE, keyword_STRING)
 *  of the given sph_keywordnode
 */
keyword_TYPE sph_keywordnode_GetType(const sph_keywordnode* pthis)
{
    return pthis->type;
}
