/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#define _sph_ldt_c_
#include "sph_ldt.h"
#include "sph_ifs_lenslet_model.h"
#include "sph_error.h"
#include "sph_spectrum.h"
#include "sph_common_keywords.h"
#include "sph_polygon.h"
#include "sph_image_grid.h"
#include "sph_cube.h"
#include "sph_paint_polygon.h"
#include "sph_pixel_description_table.h"
#include <cpl_error.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <strings.h>
#include <math.h>
#include <limits.h>
#include <float.h>
#include <assert.h>
#include "sph_keyword_manager.h"
#include "sph_interpolation.h"
#include "sph_utils.h"
#undef _sph_ldt_c_

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/

const int SPH_LDT_GENERAL             = SPH_LDT_ERR_START + 0;
const int SPH_LDT_NOT_STANDARDISED    = SPH_LDT_ERR_START + 1;
const int SPH_LDT_NO_REFERENCE_WAVELENGTHS = SPH_LDT_ERR_START + 2;
const float SPH_LDT_WAVETOLERANCE     = 0.001;
const double SPH_LDT_MIN_OVERLAP      = 0.01;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ldt Lenslet Description Table
 *
 * @par Synopsis:
 * @code
 * typedef struct _sph_ldt_
 * {
 *   int                         nlens;  // number lenslets
 *   cpl_propertylist*           properties;
 *   sph_ifs_lenslet_model*      model;
 *   sph_lenslet_descriptor**    arr; // the lenslet descrption array
 *   cpl_vector*                 reflambdas; //wavelengths for standardised LDT
 *} sph_ldt;
 * @endcode
 * @par Descirption:
 * This module provides functions to define and operate on a lenslet
 * description table.
 * @par
 * The lenslet description table (LDT) is of central importance for the wavelength
 * calibration procedure for IFS, but is general enough to also be useable for any
 * multi-spectra instrument
 * @par
 * The lenslet desciption table contains information for every pixel on the detector.
 * This table can be manipulated with the functions provided as part of this
 * module. The functions allow extractor of lenslet descriptors at specific
 * locations, for example.
 * @par
 * Lenslets do not need all have the same wavelength grid. This is a property of
 * the lenslet descriptor, and so may vary from lenslet to lenslet. However,
 * for some functionality a standardised wavelength grid is needed,
 * for example when monochromatic images need to be created.
 * @par
 * The lenslet all have associated positions, both in an orthogonal instrument
 * coordinate system, in a hexagonal grind on on sky. The sky coordinates are only filled in
 * when the astrometry calibration has been performed.
 * @par
 * There are two main saving options: either as a cube of monochromatic images
 * with a table an extra extension or only as a table. Currently, both
 * ways of saving assume that the wavelengths are the same for all
 * lenslets (i.e. a standardised wavelength grid).
 * When writing the table to a FITS file, using sph_ldt_save,
 * the information is
 * written to the file as a FITS table in the format shown here:
 * <table bgcolor="#EEF3F5">
 * <tr>
 * <td>u</td><td>v</td><td>ID</td>
 * <td>x</td><td>y</td><td>sky_x</td><td>sky_y</td><td>N</td>
 * <td>\f$L_1\f$</td><td>...</td><td>\f$L_N\f$</td></tr>
 * <tr>
 * <td>u coord.</td><td>v coord.</td><td>lenslet ID</td>
 * <td>x-pos. (ins coord).</td><td>x-pos. (ins coord).</td>
 * <td>Sky-coord</td><td>Sky-coord</td>
 * <td>#\f$\lambda\f$-points</td>
 * <td>\f$f(\lambda_1)\f$</td><td>...</td><td>f(\f$\lambda_N)\f$</td>
 * </tr>
 * </table>
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-----------------------------------------------------------------------------
 Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new lenslet description table thats empty.
 *
 * @param model             the lenslet model to use
 *
 * @return Pointer to new table or null in case of error.
 *
 * A new lenslet description table is created using the given lenslet
 * model as the base. The input lenslet model should not be deleted,
 * as it is deleted automatically when the lenslet description table
 * is deleted.
 *
 */
/*----------------------------------------------------------------------------*/
sph_ldt* sph_ldt_new( sph_ifs_lenslet_model* model )
{
    sph_ldt*          result      = NULL;
    int                                        uu            = 0;
    int                                        vv            = 0;
    int                                        cc            = 0;
    int                                         id           = 0;
    if ( !model ) {
        return NULL;
    }
    result = cpl_calloc( 1, sizeof(sph_ldt) );
    if ( result == NULL ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL,
                         __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not allocate memory for the "
                         "lenslet description table.\n");
        return NULL;
    }

    sph_ldt_set_model( result, model );

    cc        = 0;

    for ( vv = -model->lenslets_per_side;
            vv <= model->lenslets_per_side; ++vv )
    {
        if ( vv >=0 )
        {
            for ( uu = -model->lenslets_per_side;
                    uu <= model->lenslets_per_side - vv; ++uu )
            {
                id = sph_ifs_lenslet_model_get_index(model,uu,vv);
                result->arr[cc] = sph_lenslet_descriptor_new( id, uu, vv,
                        result->reflambdas );
                cc++;
            }
        }
        else {
            for (uu = 0 - ( model->lenslets_per_side + vv );
                    uu <= model->lenslets_per_side; ++uu)
            {
                id = sph_ifs_lenslet_model_get_index(model,uu,vv);
                result->arr[cc] = sph_lenslet_descriptor_new( id, uu, vv,
                        result->reflambdas );
                cc++;
            }
        }
    }

    if ( cc != result->nlens ) {
        sph_error_raise(SPH_ERROR_GENERAL,
                __FILE__,__func__,__LINE__,
                SPH_ERROR_ERROR,
                "Something is wrong in creation of LDT -- "
                "lenslet count reached %d but have only %d "
                "lenslets in model. ", cc, result->nlens);
    }
    result->properties = cpl_propertylist_new();
    cpl_propertylist_append_double(
            result->properties,
            SPH_COMMON_KEYWORD_SPH_POSANG, 0.0 );
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Get number of wavelengths
 * @param self
 *
 * @return number of wavelength points
 *
 */
/*----------------------------------------------------------------------------*/
int sph_ldt_get_nwavs(
        sph_ldt* self) {
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,-1);
    cpl_ensure(self->reflambdas,CPL_ERROR_NULL_INPUT,-1);
    return cpl_vector_get_size(self->reflambdas);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the lenslet model
 * @param self the LDT
 * @param model the new model
 *
 * @return error code of the operation.
 *
 * This sets a new model for the lenslet description table. If any
 * lenslet data was stored these are deleted.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ldt_set_model( sph_ldt* self, sph_ifs_lenslet_model* model ) {

    if ( !self || !model ) {
        SPH_NO_SELF;
        return CPL_ERROR_NULL_INPUT;
    }

    if ( self->arr )
        sph_ldt_delete_arr( self );

    self->model = model;
    self->nlens = sph_ifs_lenslet_model_get_nlens(model);
    self->arr = cpl_calloc( self->nlens, sizeof(sph_lenslet_descriptor*) );
    if ( self->arr == NULL ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL,
                         __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Could not allocate memory for the "
                         "lenslet description table lenslet array.\n");
        sph_ldt_delete( self );
        return SPH_ERROR_MEMORY_FAIL;
    }

    self->reflambdas = cpl_vector_duplicate(
            sph_ifs_lenslet_model_get_lambda_vector( model ) );
    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Loads the LDT from FITS file
 * @param filename        the file to load from
 *
 * @return new LDT or NULL in case of error.
 *
 * The LDT is expected to have been saved using
 * the sph_ldt_save function.
 *
 */
/*----------------------------------------------------------------------------*/
sph_ldt*
sph_ldt_load( const char* filename ) {
    sph_ifs_lenslet_model*        model = NULL;
    cpl_propertylist*            plist  = NULL;
    sph_ldt*    self = NULL;
    cpl_table*                    tab    = NULL;
    int                            row    = 0;
    sph_lenslet_descriptor*        ldescr = NULL;
    int                            lid = 0;
    int                            lidold = -2;
    int                            cc    = 0;
    int                            bpix = 0;
    int                             next = 0;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure(filename,CPL_ERROR_NULL_INPUT,NULL);

    next = cpl_fits_find_extension(filename,SPH_LDT_EXTID);

    if ( next == -1 )
    {
        // Here come a cpl bug workaround
        // Since cpl_fits_find_extension returns -1 unless all
        // extensions have the EXTNAME keyword defined,
        // just print a warning and go through all extensions manually
        SPH_ERROR_RAISE_WARNING(CPL_ERROR_FILE_IO,
                "Could not find the extension %s in file %s "
                "(ignore this warning, its only for debug purposes)",
                SPH_LDT_EXTID, filename);
        for (next = 0; next <= cpl_fits_count_extensions(filename); ++next) {
            plist = cpl_propertylist_load(filename,next);
            if ( cpl_propertylist_has(plist,"EXTNAME") ) {
                if ( strcasecmp(
                        cpl_propertylist_get_string(plist,"EXTNAME"),
                        SPH_LDT_EXTID) == 0 ) {
                    break;
                }
            }
            cpl_propertylist_delete(plist); plist = NULL;
        }
    }

    tab = cpl_table_load( filename, next, 0 );
    if ( !tab ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO,
                "Could not load table from file %s",
                filename);
        return NULL;
    }

    cpl_propertylist_delete(plist); plist = NULL;
    plist = sph_keyword_manager_load_properties( filename,0 );
    if ( !plist ) {
        SPH_ERR("Could not load lenslet model from file.");
        return NULL;
    }

    model = sph_ifs_lenslet_model_new_from_propertylist(plist);
    cpl_propertylist_delete(plist);
    if ( !model ) {
        SPH_ERR("Could not load lenslet model from file.");
        return NULL;
    }

    self = sph_ldt_new(model);
    if ( cpl_table_get_nrow(tab) !=
            self->nlens * cpl_vector_get_size(self->reflambdas) ||
            cpl_table_get_ncol(tab) != 6 ) {
        SPH_ERR("Table has bad format.");
        cpl_table_delete(tab);
        sph_ldt_delete(self);
        return NULL;
    }

    // Read in the lenslet info...
    lidold = -2;
    ldescr = NULL;
    for (row = 0; row < cpl_table_get_nrow(tab); ++row) {
        lid = cpl_table_get(tab,"LensID",row,&bpix);
        if ( lid != lidold && lid > 0 ) {
            lidold = lid;
            cc = 0;
            ldescr = self->arr[lid-1];
        }
        if ( fabs(cpl_table_get(tab,"Wavelength",row,&bpix)-cpl_vector_get(self->reflambdas,cc)) > 0.001 ) {
            SPH_ERR("Wavelength mismatch between model and wavelength.");
            break;
        }
        if ( ldescr ) {
            ldescr->values[cc] = cpl_table_get(tab,"Value",row,&bpix);
            ldescr->bpix[cc] = cpl_table_get(tab,"Badpix",row,&bpix);
            ldescr->rms[cc] = cpl_table_get(tab,"RMS",row,&bpix);
            ldescr->ncomb[cc] = cpl_table_get(tab,"Weight",row,&bpix);
        }
        cc++;
    }

    cpl_table_delete(tab);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return self;

}

// The function below is currently not used anywhere --
// except in a bunch of tests
// should consider throwing it out.
sph_ldt*
sph_ldt_new_from_pdt_image( sph_pixel_description_table* pdt,
        sph_ifs_lenslet_model* model,
        cpl_image* image ) {
    sph_ldt*  ldt     = NULL;
    sph_master_frame*            mframe        = NULL;

    if ( !pdt || !image ) {
        SPH_NULL_ERROR
        return NULL;
    }

    mframe = sph_master_frame_new_from_cpl_image(image);
    cpl_image_add_scalar(mframe->ncombmap,1.0);
    ldt = sph_ldt_new_from_pdt_master_frame(pdt,model,mframe);
    sph_master_frame_delete(mframe);
    return ldt;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new LDT from a PDT and a master frame
 *
 * @param pdt       the pixel description table to use for
 *                  wavelength associations.
 * @param model		the IFS lenslet model.
 * @param mf        the master_frame to use to fill the LDT
 *                  with values.
 *
 * @return          the new LDT
 *
 * This function fills the LDT with all available information from the master
 * frame given in the input. All the defined images in the master frame are
 * used and the corresponding vector in the LDT filled. The vectors are filled
 * in the following way:
 * <ol>
 *  <li>master_frame->image to lenslet->values</li>
 *  <li>master_frame->badpix to lenslet->bpix</li>
 *  <li>master_frame->rmsmap to lenslet->rms</li>
 *  <li>master_frame->ncombmap to lenslet->ncomb</li>
 * </ol>
 */
/*----------------------------------------------------------------------------*/

sph_ldt*
sph_ldt_new_from_pdt_master_frame(
        sph_pixel_description_table* pdt,
        sph_ifs_lenslet_model* model,
        sph_master_frame* mf ) {

    sph_ldt*  ldt     = NULL;
    sph_lenslet_descriptor*     lenslet     = NULL;
    sph_spectrum*               spec        = NULL;
    int                         nspecs      = 0;
    int                         lensid        = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if ( !pdt || !mf ) {
        SPH_NULL_ERROR
        return NULL;
    }

    nspecs=pdt->nregions;

    ldt = sph_ldt_new( model  );

    if ( !ldt ) {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_MEMORY_FAIL,
                "Could not create empty LDT");
        return NULL;
    }
    for (lensid = 0; lensid < ldt->nlens; ++lensid) {
        lenslet = ldt->arr[ lensid  ];
        sph_lenslet_descriptor_set_bad( lenslet );
    }
    for (int ii = 0; ii < nspecs; ++ii) {
        spec = sph_pixel_description_table_extract_spectrum_mframe( pdt,
                ii + 1, mf, NULL );
        if ( spec == NULL ) {
            // Note: since spec may be NULL without
            // cpl_error being set, the next two lines
            // have no effect in most cases.
        	// TODO: this is not tidy. Should have
        	// better way of dealing with errors in extraction
            SPH_RAISE_CPL;
            SPH_ERROR_RAISE_ERR(cpl_error_get_code(),
                    "Could not extract spectrum.");
        }
        if ( spec && spec->lensid > 0 ) {
            lensid = spec->lensid;
            lenslet = ldt->arr[ lensid - 1 ];
            if ( lenslet ) {
                if ( sph_lenslet_descriptor_fill_spec( lenslet, spec) ==
                        CPL_ERROR_ILLEGAL_INPUT )
                {
                    cpl_error_reset();
                    sph_lenslet_descriptor_set_bad(lenslet);
                }
            }
        }
        sph_spectrum_delete( spec ); spec = NULL;
    }

    if ( cpl_propertylist_has( mf->properties, SPH_COMMON_KEYWORD_SPH_POSANG ) )
    {
      cpl_propertylist_copy_property( ldt->properties, mf->properties,
              SPH_COMMON_KEYWORD_SPH_POSANG );
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return ldt;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return a lenslet descriptor for a specific lenslet.
 *
 * @param nu   The position in u
 * @param nv   The position in v
 *
 * @return Pointer to lenslet descriptor or null in case of error.
 *
 */
/*----------------------------------------------------------------------------*/
sph_lenslet_descriptor* sph_ldt_get( sph_ldt* self,
                                                           int uu,
                                                          int vv )
{
    sph_lenslet_descriptor*     descr       = NULL;
    double                      drings      = 0.0;
    int                         nrings      = 0;
    int                         cc          = 0;


    drings = sqrt( ( self->nlens - 1.0 ) / 3.0 + 0.25 ) - 0.5;
    nrings = round( drings );

    if ( vv >=0 ) {
        cc = nrings * ( nrings + 1 ) + ( nrings - 1 ) * ( nrings ) * 0.5;
        cc += vv * ( 2 * nrings  + 1 ) - ( vv - 1 ) * vv * 0.5;
        cc+= uu + nrings ;
    }
    else {
        cc = ( vv + nrings ) * ( nrings + 1 ) +
                ( vv + nrings - 1 ) * ( vv + nrings ) * 0.5;
        cc+= uu + ( nrings + vv );
    }
    if ( cc < 0 || cc > self->nlens ) {
        return NULL;
    }
    descr = self->arr[cc];
    if ( descr->u != uu  || descr->v !=vv ) {
        return NULL;
    }
    return descr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create new LDT by subtract two input LDT's
 *
 * @param intab         The input LDT to subtract from
 * @param stab          The LDT to subtract
 * @param algorithm     The interpolation algorithm to use
 *
 * This creates a copy of the input table but subtracts the corresponding spectra
 * in the stab LDT for all lenslets.
 * The wavelength of all stab lenslet spectra are interpolated
 * onto the wavelengths of the corresponding lenslet in the input LDT
 * using the algortihm provided as the
 * algorithm parameter
 *
 * @return Pointer to lenslet descriptor or null in case of error.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ldt_subtract( sph_ldt* intab,
        sph_ldt* stab )
{

    sph_error_code            rerr    = CPL_ERROR_NONE;
    sph_lenslet_descriptor*    ldescr    = NULL;
    int                        ii        = 0;

    if ( !intab || !stab ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < intab->nlens; ++ii) {
        ldescr = sph_ldt_get( stab,
                intab->arr[ii]->u, intab->arr[ii]->v );
        if ( ldescr ) {
            sph_lenslet_descriptor_subtract( intab->arr[ii], ldescr );
        }
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create new LDT by dividing two input LDT's
 *
 * @param intab         The input LDT to divide from
 * @param stab          The LDT to divide
 *
 * This divides each lenslet by
 * the corresponding spectra
 * in the stab LDT.
 *
 * @return Pointer to lenslet descriptor or null in case of error.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ldt_divide( sph_ldt* intab,
        sph_ldt* stab )
{
    sph_lenslet_descriptor*    ldescr    = NULL;
    int                        ii        = 0;

    if ( !intab || !stab ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < intab->nlens; ++ii) {
        ldescr = sph_ldt_get( stab,
                intab->arr[ii]->u, intab->arr[ii]->v );
        if ( ldescr ) {
            sph_lenslet_descriptor_divide( intab->arr[ii], ldescr );
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Divide the LDT by median of other LDT
 *
 * @param intab         The input LDT to divide from
 * @param stab          The LDT to divide
 *
 * This divides each lenslet by
 * the corresponding median value of the stab LDT.
 *
 * @return error code.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ldt_divide_median( sph_ldt* intab,
        sph_ldt* stab )
{
    sph_lenslet_descriptor*    ldescr    = NULL;
    int                        ii        = 0;

    if ( !intab || !stab ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < intab->nlens; ++ii) {
        ldescr = sph_ldt_get( stab,
                intab->arr[ii]->u, intab->arr[ii]->v );
        if ( ldescr ) {
            if ( sph_lenslet_descriptor_get_overall_bad( ldescr ) == 0 ) {
                if ( sph_lenslet_descriptor_divide_scalar( intab->arr[ii],
                        sph_lenslet_descriptor_get_median(ldescr,NULL) )
                        != CPL_ERROR_NONE)
                {
                    sph_lenslet_descriptor_set_bad(intab->arr[ii]);
                }
            }
        }
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Divide the lenslet description table by an IFU flat
 * @param self      the lenslet description table
 * @param ldt_tab      the IFU flat as table
 *
 * @return error code of the operation
 *
 * Divided a LDT by the IFU flat values as given in the IFU flat
 * table, which should be a cpl_table. See the code for the headers.
 *
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ldt_divide_ifu_flat(
        sph_ldt* self, cpl_table* ifu_tab)
{
    int             ll  = 0;
    int*          ifu_bpix;
    double*       ifu_medians;
    double*       ifu_rms;
    int             lensid = 0;
    int             bpix = 0;
    int             maxlens = 0;
    double          divisor = 0;

    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ifu_tab,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cpl_table_has_column(ifu_tab,
            SPH_IFS_INSTRUMENT_MODEL_LENSID),CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_table_has_column(ifu_tab,
            SPH_IFS_INSTRUMENT_MODEL_FLATBADPIX),CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_table_has_column(ifu_tab,
            SPH_IFS_INSTRUMENT_MODEL_FLATRMS),CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_table_has_column(ifu_tab,
            SPH_IFS_INSTRUMENT_MODEL_FLATVAL),CPL_ERROR_ILLEGAL_INPUT);

    maxlens = sph_ifs_lenslet_model_get_nlens(self->model);

    ifu_bpix = cpl_calloc(maxlens,sizeof(int));
    ifu_medians = cpl_calloc(maxlens,sizeof(double));
    ifu_rms = cpl_calloc(maxlens,sizeof(double));
    cpl_ensure_code(ifu_rms && ifu_bpix && ifu_medians,
            CPL_ERROR_INCOMPATIBLE_INPUT);

    for (ll = 0; ll < cpl_table_get_nrow(ifu_tab); ++ll) {
        lensid = cpl_table_get(ifu_tab,
                SPH_IFS_INSTRUMENT_MODEL_LENSID,ll,&bpix);

        if ( bpix == 0 && lensid > 0 && lensid <= maxlens )
        {
            ifu_bpix[lensid-1] = cpl_table_get(ifu_tab,
                    SPH_IFS_INSTRUMENT_MODEL_FLATBADPIX,ll,&bpix);
            ifu_medians[lensid-1] = cpl_table_get(ifu_tab,
                    SPH_IFS_INSTRUMENT_MODEL_FLATVAL,ll,&bpix);
            ifu_rms[lensid-1] = cpl_table_get(ifu_tab,
                    SPH_IFS_INSTRUMENT_MODEL_FLATRMS,ll,&bpix);
        }
    }

    for (ll = 0; ll < self->nlens; ++ll) {
        if ( self->arr[ll] ) {
            bpix = sph_lenslet_descriptor_get_overall_bad(self->arr[ll]);
            if ( bpix == 0 ) {
                lensid = self->arr[ll]->lensid;
                if ( lensid < 1 || lensid > maxlens ) {
                    SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                            "Illegal lens id");
                    return CPL_ERROR_ILLEGAL_INPUT;
                }
                if ( ifu_bpix[lensid-1] == 0 ) {
                    divisor = ifu_medians[lensid-1];
                    if (divisor != 0 ) {
                        sph_lenslet_descriptor_divide_scalar(self->arr[ll],
                                divisor);
                    }
                    else {
                        sph_lenslet_descriptor_set_bad(self->arr[ll]);
                    }
                    //TODO: add rms propagation?
                }
                else {
                    sph_lenslet_descriptor_set_bad(self->arr[ll]);
                }
            }
        }
    }

    cpl_free(ifu_bpix); ifu_bpix = NULL;
    cpl_free(ifu_medians); ifu_medians = NULL;
    cpl_free(ifu_rms); ifu_rms = NULL;

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief  Divide LDT by scalar
 *
 * @param intab         The input LDT to divide from
 * @param val            the number to divide by
 *
 * @return error code of the operation
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ldt_divide_scalar( sph_ldt* intab,
                                            double val )
{
    int                        ii        = 0;

    if ( !intab  || val == 0 ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < intab->nlens; ++ii) {
        sph_lenslet_descriptor_divide_scalar( intab->arr[ii], val );
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create new LDT by adding two input LDT's
 *
 * @param intab         The input LDT to add to
 * @param stab          The LDT to add
 * @param algorithm     The interpolation algorithm to use
 *
 * This creates a copy of the input table but adds the corresponding spectra
 * in the stab LDT for all lenslets.
 * The wavelength of all stab lenslet spectra are interpolated
 * onto the wavelengths of the corresponding lenslet in the input LDT
 * using the algortihm provided as the
 * algorithm parameter
 *
 * @return Pointer to lenslet descriptor or null in case of error.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_ldt_add( sph_ldt* intab,
        sph_ldt* stab )
{
    sph_lenslet_descriptor*    ldescr    = NULL;
    int                        ii        = 0;

    if ( !intab || !stab ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    for (ii = 0; ii < intab->nlens; ++ii) {
        ldescr = sph_ldt_get( stab,
                intab->arr[ii]->u, intab->arr[ii]->v );
        if ( ldescr ) {
            sph_lenslet_descriptor_add( intab->arr[ii], ldescr );
        }
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create new LDT by rotating a LDT
 *
 * @param intab         The input LDT to rotate
 * @param angle         The rotation angle (along the wavelength axis)
 *                      in radians
 * @param factor    	The "droplet scale" factor to use
 *
 * This creates a copy of the input table but rotates the lenlet grid by an
 * agnle.
 * The values for all wavelengths for all lenslet spectra are interpolated
 * onto the rotated lenslet grid. The dropplet factor should be a number
 * between 0 and 1 and is the scaling factor performed before the lenslet
 * is "dropped" onto the target grid.
 *
 * @return Pointer to lenslet descriptor or null in case of error.
 *
 */
/*----------------------------------------------------------------------------*/
sph_ldt*
sph_ldt_new_rotate_gimros( sph_ldt* intab,
        double angle,
        double factor )
{
    int                                lens        = 0;
    sph_ldt*    ldt_new        = NULL;
    sph_ifs_lenslet_model*            model        = NULL;
    sph_polygon*                    poly        = NULL;
    int                                u            = 0;
    int                                v            = 0;
    double                            cosang        = cos( angle * CPL_MATH_RAD_DEG );
    double                            sinang        = sin( angle * CPL_MATH_RAD_DEG );
    int                                indexes[10];
    int                                vp            = 0;
    int                                ii            = 0;

    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
      SPH_RAISE_CPL_RESET
    }

    model = sph_ifs_lenslet_model_duplicate( intab->model );
    if ( model ) {
        ldt_new = sph_ldt_new( model );
    }
    if ( !ldt_new ) {
        SPH_NULL_ERROR
        return NULL;
    }

    if ( !sph_ldt_check_ldt_match_ok( intab, ldt_new ) )
    {
        sph_ldt_delete( ldt_new ); ldt_new = NULL;
        return NULL;
    }


    for (ii = 0; ii < 10; ++ii) {
        indexes[ii] = -2;
    }

    SPH_RAISE_CPL_RESET

    for ( lens = 0; lens < intab->nlens; ++lens ) {
        u = intab->arr[ lens ]->u;
        v = intab->arr[ lens ]->v;
        poly = sph_ifs_lenslet_model_get_poly( intab->model, u, v );
        if ( poly )
        {
            sph_polygon_enlarge( poly, factor );
            sph_polygon_rotate_around( poly, 0.0, 0.0, cosang, sinang,
                    0.0, 0.0 );
            sph_ldt_draw_poly( lens,intab, ldt_new,
                    poly, indexes );
            sph_polygon_delete( poly ); poly = NULL;
        }

    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
      SPH_RAISE_CPL_RESET
    }

    // Normalise and Mark all pixels as bad where no value added
    for ( lens = 0; lens < ldt_new->nlens; ++lens )
    {
        for ( vp = 0; vp < cpl_vector_get_size( ldt_new->reflambdas); ++vp )
        {
            if ( ldt_new->arr[lens]->ncomb[vp] <= 0.0 )
            {
                ldt_new->arr[lens]->bpix[vp] = 1;
            }
            else
            {
                ldt_new->arr[lens]->values[vp] /= ldt_new->arr[lens]->ncomb[vp];
            }
        }
    }
    if ( cpl_propertylist_has( intab->properties,
            SPH_COMMON_KEYWORD_SPH_POSANG ) ) {
      cpl_propertylist_update_double( ldt_new->properties,
              SPH_COMMON_KEYWORD_SPH_POSANG,
              cpl_propertylist_get_double( intab->properties,
                      SPH_COMMON_KEYWORD_SPH_POSANG ) + angle );
    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
      SPH_RAISE_CPL_RESET
    }
    return ldt_new;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new master frame from LDT by collapsing the LDT
 *
 * @param self      the LDT to create image from
 * @param angle        rotation angle in degrees CCW
 * @param magfac    a magnification factor to apply
 *
 * @return pointer to the newly created master frame.
 *
 * This routine creates a new sph_master_frame
 * with the given dimenstions.
 * The values used to fill the sph_master_frame are taken from the
 * lenslet descriptor ( using the vectors for SPH_LDT_IMAGE, SPH_LDT_BADPIX,
 * SPH_LDT_RMSMAP, SPH_LDT_NCOMBMAP for the corresponding maste_frame
 * extensions. ).
 * The way the flux is calculated is to assume that total flux in
 * each hexagonal area on the new master_frame equals the flux of one
 * of the pixels at that wavelength for the corresponding wavelength.
 * All the spectrum values are added together so as to create a collapsed
 * image.
 * The magnification factor can be used to show only a central part of the
 * lenslet array on the master frame. Using a value smaller than 0
 * gives the default magnification factor.
 * TODO: this flux normalisation is not correct -- need to take total perp.
 * wavelength direction !
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_ldt_collapse( sph_ldt* intab,
        double angle,
        double magfac )
{
    sph_master_frame*        mframe = NULL;
    sph_master_frame**        mframelist = NULL;
    int                     ll    = 0;
    sph_polygon*            poly  = NULL;
    cpl_image*                overlapim = NULL;
    double                    cosan = cos(CPL_MATH_RAD_DEG*angle);
    double                    sinan = sin(CPL_MATH_RAD_DEG*angle);
    sph_lenslet_descriptor*    ldescr = NULL;
    double                    plane_size = 0.0;
    double                    pixsize = 0.0;
    int                        nn = 0;
    int                        plane = 0;
    int                        nx = 0;
    int                        ny = 0;
    int                        xx    = 0;
    int                        yy    = 0;
    int                        rej = 0;
    double                    val = 0.0;
    double                    rms = 0.0;
    int                        bpix = 0.0;
    double                    ncomb = 0.0;
    sph_polygon_workspace*      ws = NULL;

    if ( cpl_error_get_code( ) ) {
        SPH_RAISE_CPL;
        return NULL;
    }

    ws = sph_polygon_workspace_new();
    nx = ny = intab->model->lenslets_per_side * 2 + 1;

    mframelist = cpl_calloc(cpl_vector_get_size( intab->reflambdas ),
            sizeof(sph_master_frame*) );

    if ( magfac < 0.0 ) magfac = 3.0;
    if ( !mframelist ) {
        return NULL;
    }
    for (nn = 0; nn < cpl_vector_get_size(intab->reflambdas); ++nn) {
        mframelist[nn] = sph_master_frame_new(nx,ny);
        if ( !mframelist[nn] ) {
            SPH_ERROR_RAISE_ERR(
                    SPH_ERROR_MEMORY_FAIL,
                    "Out of memory for creation of master frames.");
            return NULL;
        }
    }

    plane_size = 2.0 * intab->model->lensize_microns *
            intab->model->lenslets_per_side;
    pixsize = plane_size / ( magfac * nx );

    for (ll = 0; ll < intab->nlens; ++ll) {
        ldescr = intab->arr[ll];
        if ( ldescr ) {
            poly = sph_ifs_lenslet_model_get_poly( intab->model,
                    ldescr->u, ldescr->v );
            sph_polygon_rotate_around(poly,0.0,0.0,cosan,sinan,0.0,0.0);
            for (nn = 0; nn < poly->npoints; ++nn) {
                poly->points[nn].x/=pixsize;
                poly->points[nn].y/=pixsize;
                poly->points[nn].x+=nx/2.0;
                poly->points[nn].y+=ny/2.0;
            }
            overlapim = sph_paint_polygon_get_overlap_image(mframelist[0],
                    poly,ws);
            for (plane = 0;
                    plane < cpl_vector_get_size(intab->reflambdas);
                    ++plane)
            {
                sph_paint_polygon_on_master(mframelist[plane],poly,
                        ldescr->values[ plane ],
                        ldescr->bpix[ plane ],
                        ldescr->rms[ plane ],
                        ldescr->ncomb[ plane ], overlapim );
            }
            sph_polygon_delete(poly); poly = NULL;
            cpl_image_delete(overlapim);overlapim = NULL;
        }
    }

    mframe = sph_master_frame_new(nx,ny);

    cpl_propertylist_update_double( mframe->properties,
            SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE,
            pixsize);

    for (yy = 0; yy < ny; ++yy) {
        for (xx = 0; xx < nx; ++xx) {
            val = 0.0;
            bpix = 0;
            ncomb = 0.0;
            rms = 0.0;
            for (nn = 0; nn < cpl_vector_get_size(intab->reflambdas); ++nn) {
                if ( cpl_image_get(
                        mframelist[nn]->badpixelmap,
                        xx+1,yy+1,
                        &rej) == 0 ) {
                    val += cpl_image_get( mframelist[nn]->image,
                            xx+1,yy+1, &rej);
                    ncomb += cpl_image_get( mframelist[nn]->ncombmap,
                            xx+1,yy+1, &rej);
                }
                else {
                    bpix++;
                }
            }
            if ( bpix == cpl_vector_get_size(intab->reflambdas) ) {
                cpl_image_set(mframe->badpixelmap,xx+1,yy+1,1);
            }
            else {
                cpl_image_set(mframe->image,xx+1,yy+1,val);
                cpl_image_set(mframe->rmsmap,xx+1,yy+1,rms);
                cpl_image_set(mframe->ncombmap,xx+1,yy+1,ncomb);
                cpl_image_set(mframe->badpixelmap,xx+1,yy+1,0);
            }
        }
    }
    for (nn = 0; nn < cpl_vector_get_size(intab->reflambdas); ++nn) {
        sph_master_frame_delete(mframelist[nn]); mframelist[nn] = NULL;
    }
    sph_polygon_workspace_delete(ws);
    cpl_free(mframelist); mframelist = NULL;
    return mframe;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new master frame from LDT by collapsing the LDT
 *
 * @param self      the LDT to create image from
 * @param angle        rotation angle in degrees CCW
 * @param magfac    a magnification factor to apply
 *
 * @return pointer to the newly created master frame.
 *
 * This routine creates a new sph_master_frame
 * with the given dimenstions.
 * The values used to fill the sph_master_frame are taken from the
 * lenslet descriptor ( using the vectors for SPH_LDT_IMAGE, SPH_LDT_BADPIX,
 * SPH_LDT_RMSMAP, SPH_LDT_NCOMBMAP for the corresponding maste_frame
 * extensions. ).
 * The way the flux is calculated is to assume that total flux in
 * each hexagonal area on the new master_frame equals the flux of one
 * of the pixels at that wavelength for the corresponding wavelength.
 * All the spectrum values are added together so as to create a collapsed
 * image.
 * The magnification factor can be used to show only a central part of the
 * lenslet array on the master frame. Using a value smaller than 0
 * gives the default magnification factor.
 * TODO: this flux normalisation is not correct -- need to take total perp.
 * wavelength direction !
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_ldt_collapse_interpolate( sph_ldt* intab,
        double angle,
        double magfac )
{
    sph_master_frame*        mframe = NULL;
    int                     ll    = 0;
    cpl_mask*                dummask = NULL;
    cpl_image*                dumim = NULL;
    double                    cosan = cos(CPL_MATH_RAD_DEG*angle);
    double                    sinan = sin(CPL_MATH_RAD_DEG*angle);
    sph_lenslet_descriptor*    ldescr = NULL;
    double                    plane_size = 0.0;
    double                    pixsize = 0.0;
    cpl_array*                  xpoints = NULL;
    cpl_array*                  ypoints = NULL;
    cpl_array*                  zpoints = NULL;
    cpl_array*                  bppoints = NULL;
    cpl_array*                  ncpoints = NULL;
    cpl_array*                  rmspoints = NULL;
    sph_point*                  midp = NULL;
    int                        nx = 0;
    int                        ny = 0;
    sph_interpolation*            interp = NULL;
    double                          rms = 0.0;

    nx = ny = intab->model->lenslets_per_side * 2 + 1;
    if ( magfac < 0.0 ) magfac = 3.0;
    plane_size = 2.0 * intab->model->lensize_microns *
            intab->model->lenslets_per_side;
    pixsize = plane_size / ( magfac * (double)nx );

    mframe = sph_master_frame_new(nx,ny);
    if ( !mframe ) {
        return NULL;
    }
    cpl_propertylist_update_double( mframe->properties,
            SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE,
            pixsize);


    midp = sph_point_new(0.0,0.0);


    xpoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
    ypoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);

    if ( intab->model->lenslet_distortion ) {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "Applying distortion");
    }
    else {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "Not applying distortion");
    }

    for (ll = 0; ll < intab->nlens; ++ll) {
        ldescr = intab->arr[ll];
        if ( ldescr ) {
            sph_ifs_lenslet_model_get_centre(intab->model,
                    ldescr->u,
                    ldescr->v,
                    &midp->x,
                    &midp->y);
            sph_point_rotate_around(midp,0.0,0.0,cosan,sinan,0.0,0.0);
            midp->x/=pixsize;
            midp->y/=pixsize;
            midp->x+=nx/2.0;
            midp->y+=ny/2.0;
            cpl_array_set( xpoints, ll, midp->x);
            cpl_array_set( ypoints, ll, midp->y);
        }
    }
    interp = sph_interpolation_new(xpoints,ypoints,3);
    sph_interpolation_prepare_index_assocs(interp,nx,ny);
    zpoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
    bppoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
    ncpoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
    rmspoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
    for (ll = 0; ll < intab->nlens; ++ll) {
        ldescr = intab->arr[ll];
        if ( ldescr ) {
            cpl_array_set( zpoints, ll,
                    sph_lenslet_descriptor_get_median(ldescr,&rms) );
            cpl_array_set( bppoints, ll,
                    sph_lenslet_descriptor_get_overall_bad(ldescr) );
            if ( sph_lenslet_descriptor_get_overall_bad(ldescr) ) {
                cpl_array_set( ncpoints, ll, 0);
            }
            else {
                cpl_array_set( ncpoints, ll, 1);
            }
            cpl_array_set( rmspoints, ll, rms);
        }
    }

    cpl_image_delete(mframe->image); mframe->image = NULL;
    mframe->image =
            sph_interpolation_interpolate(interp,zpoints);
    cpl_image_delete(mframe->badpixelmap); mframe->badpixelmap = NULL;
    mframe->badpixelmap =
            sph_interpolation_interpolate(interp,bppoints);
    cpl_image_delete(mframe->ncombmap); mframe->ncombmap = NULL;
    mframe->ncombmap =
            sph_interpolation_interpolate(interp,ncpoints);
    cpl_image_delete(mframe->rmsmap); mframe->rmsmap = NULL;
    mframe->rmsmap =
            sph_interpolation_interpolate(interp,rmspoints);

    cpl_array_delete(zpoints); zpoints = NULL;
    cpl_array_delete(bppoints); bppoints = NULL;
    cpl_array_delete(rmspoints); rmspoints = NULL;
    cpl_array_delete(ncpoints); ncpoints = NULL;

    dummask = cpl_mask_threshold_image_create(
            mframe->ncombmap,
            -0.00001,
            0.00001);
    dumim = cpl_image_new_from_mask(dummask);
    cpl_image_add(mframe->badpixelmap,dumim);
    cpl_image_delete(dumim);dumim = NULL;
    cpl_mask_delete(dummask); dummask = NULL;
    sph_point_delete(midp); midp = NULL;
    sph_interpolation_delete(interp); interp = NULL;
    return mframe;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new master frame from LDT
 *
 * @param self      the LDT to create image from
 * @param nx        the size of the new sph_master_frame
 * @param ny           the size of the new sph_master_frame
 * @param plane     the wavelength plane to use
 * @param angle        rotation angle in degrees CCW
 *
 * @return pointer to the newly created CPL image.
 *
 * description This routine creates a new sph_master_frame
 * with the given dimenstions.
 * The values used to fill the sph_master_frame are taken from the
 * lenslet descriptor ( using the vectors for SPH_LDT_IMAGE, SPH_LDT_BADPIX,
 * SPH_LDT_RMSMAP, SPH_LDT_NCOMBMAP for the corresponding maste_frame
 * extensions. ).
 * The way the flux is calculated is to assume that total flux in
 * each hexagonal area on the new master_frame equals the flux of one
 * of the pixels at that wavelength for the corresponding wavelength.
 * TODO: this flux normalisation is not correct -- need to take total perp.
 * wavelength direction !
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_ldt_create_mframe( sph_ldt* intab,
        int nx,
        int ny,
        int plane,
        double angle,
        cpl_image** overlaps )
{
    sph_master_frame*        mframe = NULL;
    int                     ll    = 0;
    sph_polygon*            poly  = NULL;
    sph_polygon_workspace*            ws  = NULL;
    cpl_image*                overlapim = NULL;
    cpl_mask*                dummask = NULL;
    cpl_image*                dumim = NULL;
    double                    cosan = cos(CPL_MATH_RAD_DEG*angle);
    double                    sinan = sin(CPL_MATH_RAD_DEG*angle);
    sph_lenslet_descriptor*    ldescr = NULL;
    double                    plane_size = 0.0;
    double                    pixsize = 0.0;
    int                        nn = 0;
    if ( cpl_error_get_code( ) ) {
        SPH_RAISE_CPL;
        return NULL;
    }
    mframe = sph_master_frame_new(nx,ny);

    ws = sph_polygon_workspace_new();

    if ( !mframe ) {
        return NULL;
    }

    plane_size = 2.0 * intab->model->lensize_microns *
            intab->model->lenslets_per_side;
    pixsize = plane_size / nx;
    for (ll = 0; ll < intab->nlens; ++ll) {
        ldescr = intab->arr[ll];
        if ( ldescr ) {
            poly = sph_ifs_lenslet_model_get_poly( intab->model,
                    ldescr->u,
                    ldescr->v );
            sph_polygon_rotate_around(poly,0.0,0.0,cosan,sinan,0.0,0.0);
            for (nn = 0; nn < poly->npoints; ++nn) {
                poly->points[nn].x/=pixsize;
                poly->points[nn].y/=pixsize;
                poly->points[nn].x+=nx/2.0;
                poly->points[nn].y+=ny/2.0;
            }
        }
        if ( overlaps ) {
            if ( !overlaps[ll] ) {
                overlaps[ll] = sph_paint_polygon_get_overlap_image(mframe,
                        poly,
                        ws);
            }
            overlapim = overlaps[ll];
        }
        if ( ldescr && poly ) {
            sph_paint_polygon_on_master(mframe,poly,
                    ldescr->values[ plane ],
                    ldescr->bpix[ plane ],
                    ldescr->rms[ plane ],
                    ldescr->ncomb[ plane ], overlapim );
            sph_polygon_delete(poly); poly = NULL;
        }
    }

    dummask = cpl_mask_threshold_image_create(
            mframe->ncombmap,
            -0.00001,
            0.00001);
    dumim = cpl_image_new_from_mask(dummask);
    cpl_image_add(mframe->badpixelmap,dumim);
    cpl_image_delete(dumim);dumim = NULL;
    cpl_mask_delete(dummask); dummask = NULL;
    sph_polygon_workspace_delete(ws);
    return mframe;
}



/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new master frame from LDT
 *
 * @param self      the LDT to create image from
 * @param nx        the size of the new sph_master_frame
 * @param ny           the size of the new sph_master_frame
 * @param plane     the wavelength plane to use
 * @param angle        rotation angle in degrees CCW
 *
 * @return pointer to the newly created CPL image.
 *
 * description This routine creates a new sph_master_frame
 * with the given dimenstions.
 * The values used to fill the sph_master_frame are taken from the
 * lenslet descriptor ( using the vectors for SPH_LDT_IMAGE, SPH_LDT_BADPIX,
 * SPH_LDT_RMSMAP, SPH_LDT_NCOMBMAP for the corresponding maste_frame
 * extensions. ).
 * The way the flux is calculated is to assume that total flux in
 * each hexagonal area on the new master_frame equals the flux of one
 * of the pixels at that wavelength for the corresponding wavelength.
 * TODO: this flux normalisation is not correct -- need to take total perp.
 * wavelength direction !
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame* sph_ldt_create_mframe_interpolate(const sph_ldt* intab,
                                                    int nx,
                                                    int ny,
                                                    int plane,
                                                    double angle,
                                                    sph_interpolation** interp)
{
    sph_master_frame*        mframe = NULL;
    int                     ll    = 0;
    cpl_mask*                dummask = NULL;
    cpl_image*                dumim = NULL;
    double                    cosan = cos(CPL_MATH_RAD_DEG*angle);
    double                    sinan = sin(CPL_MATH_RAD_DEG*angle);
    double                    plane_size = 0.0;
    double                    pixsize = 0.0;
    cpl_array*                  xpoints = NULL;
    cpl_array*                  ypoints = NULL;
    cpl_array*                  zpoints = NULL;
    cpl_array*                  bppoints = NULL;
    cpl_array*                  ncpoints = NULL;
    cpl_array*                  rmspoints = NULL;
    sph_point*                  midp = NULL;
    if ( cpl_error_get_code( ) ) {
        SPH_RAISE_CPL;
        return NULL;
    }
    mframe = sph_master_frame_new(nx,ny);
    if ( !mframe ) {
        return NULL;
    }

    midp = sph_point_new(0.0,0.0);


    if ( *interp == NULL ) {
        plane_size = 2.0 * intab->model->lensize_microns *
                intab->model->lenslets_per_side;
        pixsize = plane_size / nx;
        cpl_propertylist_update_double( mframe->properties,
                SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE,
                pixsize);

        xpoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
        ypoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
        for (ll = 0; ll < intab->nlens; ++ll) {
            const sph_lenslet_descriptor* ldescr = intab->arr[ll];
            if ( ldescr ) {
                sph_ifs_lenslet_model_get_centre(intab->model,
                        ldescr->u,
                        ldescr->v,
                        &midp->x,
                        &midp->y);
                sph_point_rotate_around(midp,0.0,0.0,cosan,sinan,0.0,0.0);
                midp->x/=pixsize;
                midp->y/=pixsize;
                midp->x+=nx/2.0;
                midp->y+=ny/2.0;
                cpl_array_set( xpoints, ll, midp->x);
                cpl_array_set( ypoints, ll, midp->y);
            }
        }
        *interp = sph_interpolation_new(xpoints,ypoints,3);
        sph_interpolation_prepare_index_assocs(*interp,nx,ny);
    }
    zpoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
    bppoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
    ncpoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
    rmspoints = cpl_array_new(intab->nlens,CPL_TYPE_DOUBLE);
    for (ll = 0; ll < intab->nlens; ++ll) {
        const sph_lenslet_descriptor* ldescr = intab->arr[ll];
        if ( ldescr ) {
            cpl_array_set( zpoints, ll, ldescr->values[plane]);
            cpl_array_set( bppoints, ll, ldescr->bpix[plane]);
            cpl_array_set( ncpoints, ll, ldescr->ncomb[plane]);
            cpl_array_set( rmspoints, ll, ldescr->rms[plane]);
        }
    }

    cpl_image_delete(mframe->image); mframe->image = NULL;
    mframe->image =
            sph_interpolation_interpolate(*interp,zpoints);
    cpl_image_delete(mframe->badpixelmap); mframe->badpixelmap = NULL;
    mframe->badpixelmap =
            sph_interpolation_interpolate(*interp,bppoints);
    cpl_image_delete(mframe->ncombmap); mframe->ncombmap = NULL;
    mframe->ncombmap =
            sph_interpolation_interpolate(*interp,ncpoints);
    cpl_image_delete(mframe->rmsmap); mframe->rmsmap = NULL;
    mframe->rmsmap =
            sph_interpolation_interpolate(*interp,rmspoints);

    cpl_array_delete(zpoints); zpoints = NULL;
    cpl_array_delete(bppoints); bppoints = NULL;
    cpl_array_delete(rmspoints); rmspoints = NULL;
    cpl_array_delete(ncpoints); ncpoints = NULL;

    dummask = cpl_mask_threshold_image_create(
            mframe->ncombmap,
            -0.00001,
            0.00001);
    dumim = cpl_image_new_from_mask(dummask);
    cpl_image_add(mframe->badpixelmap,dumim);
    cpl_image_delete(dumim);dumim = NULL;
    cpl_mask_delete(dummask); dummask = NULL;
    sph_point_delete(midp); midp = NULL;
    return mframe;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Create detector image from lenslet
 * @param self      the lenslet description table
 *
 * @return new master_frame
 *
 * This creates a new master frame showing how the LDT would look like
 * on the detector, using the lenslet model stored in the LDT to
 * map lenslets to detector regions.
 * The image is shown in the "zero" position for the lenslet model.
 *
 */
/*----------------------------------------------------------------------------*/

sph_master_frame*
sph_ldt_create_detframe(
        sph_ldt* self,
        sph_pixel_description_table* pdtin )
{
    sph_pixel_description_table*            pdt = NULL;
    sph_pixel_descriptor*            pd = NULL;
    sph_master_frame*                       mframe = NULL;
    int                                     xx = 0;
    int                                     yy = 0;
    int                                     y0 = 0;
    sph_polygon*                            poly = NULL;
    mframe = sph_master_frame_new(self->model->detsize_pixels,
            self->model->detsize_pixels);
    if ( pdtin ) {
        pdt = pdtin;
    }
    else {
        pdt = sph_pixel_description_table_new_from_model(self->model, 0.0, 0.0);
    }

    for (yy = 0; yy < self->model->detsize_pixels; ++yy) {
        for (xx = 0;     xx < self->model->detsize_pixels; ++xx){
            pd = sph_pixel_description_table_get_descriptor(pdt, xx, yy);
            if ( pd ) {
                if ( pd->lensid > 0 ) {
                    poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(
                            self->model,
                            self->arr[pd->lensid-1]->u,
                            self->arr[pd->lensid-1]->v, 0.0, 0.0);
                    y0 = (int)sph_polygon_get_bottom(poly)+1;
                    if ( yy-y0 >= 0 ) {
                        cpl_image_set(mframe->image,      xx+1,yy+1, 
                                      self->arr[pd->lensid-1]->values[yy-y0]);
                        cpl_image_set(mframe->badpixelmap,xx+1,yy+1, 
                                      self->arr[pd->lensid-1]->bpix[  yy-y0]);
                        cpl_image_set(mframe->ncombmap,   xx+1,yy+1, 
                                      self->arr[pd->lensid-1]->ncomb[ yy-y0]);
                        cpl_image_set(mframe->rmsmap,     xx+1,yy+1, 
                                      self->arr[pd->lensid-1]->rms[   yy-y0]);
                    } else { 
                           cpl_msg_debug(cpl_func, 
                                         "xx=%d;yy=%d;y0=%d;  yy-y0 = %d\n", 
                                         xx, yy, y0, yy-y0); }
                }
            }
        }
    } cpl_msg_debug(cpl_func, "xx = %d; yy = %d; y0 = %d \n", xx, yy, y0);
    if ( pdtin == NULL ) 
        sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    return mframe;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Save the LDT
 *
 * @param intab         The input LDT to divide from
 * @param filename      The filename to save as
 *
 * Saves the LDT as in the format of a table.
 *
 * @return error code of the operation.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_ldt_save(const sph_ldt* self,
                            const char* filename,
                            unsigned int mode)
{
    cpl_table*            tab    = NULL;
    int                 lensid    = 0;
    int                 ii    = 0;
    int                    row = 0;
    sph_lenslet_descriptor*    ldescr = NULL;
    cpl_propertylist*    pheader = NULL;
    cpl_propertylist*    header = NULL;
    sph_error_code        rerr = CPL_ERROR_NONE;
    SPH_RAISE_CPL_RESET
    if ( !self || !filename ) {
        SPH_NO_SELF;
        return CPL_ERROR_NULL_INPUT;
    }

    tab = cpl_table_new( self->nlens * cpl_vector_get_size(self->reflambdas) );
    if ( !tab ) {
        SPH_ERR("Could not create CPL table for LDT.");
        return SPH_ERROR_MEMORY_FAIL;
    }
    cpl_table_new_column(tab,"LensID", CPL_TYPE_INT);
    cpl_table_new_column(tab,"Wavelength", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab,"Value", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab,"Badpix", CPL_TYPE_INT);
    cpl_table_new_column(tab,"RMS", CPL_TYPE_DOUBLE);
    cpl_table_new_column(tab,"Weight", CPL_TYPE_DOUBLE);
    for (lensid = 0; lensid < self->nlens; ++lensid) {
        for (ii = 0; ii < cpl_vector_get_size( self->reflambdas ); ++ii) {
            row = lensid * cpl_vector_get_size(self->reflambdas) + ii;
            cpl_table_set_int(tab,"LensID",row,lensid+1);
            ldescr = self->arr[lensid];
            if ( ldescr ) {
                rerr = cpl_table_set_double(tab,"Wavelength",
                        row,cpl_vector_get(self->reflambdas,ii));
                rerr += cpl_table_set_double(tab,"Value",
                        row,ldescr->values[ii]);
                rerr += cpl_table_set_int(tab,"Badpix",
                        row,ldescr->bpix[ii]);
                rerr += cpl_table_set_double(tab,"RMS",
                        row,ldescr->rms[ii]);
                rerr += cpl_table_set_double(tab,"Weight",
                        row,ldescr->ncomb[ii]);
                if ( rerr > 0 ) {
                    SPH_RAISE_CPL
                    sph_error_raise(SPH_ERROR_GENERAL,
                            __FILE__,__func__,__LINE__,
                            SPH_ERROR_ERROR,
                            "Bad acces in row %d lensid", row);
                }
            }
        }
    }
    pheader = sph_ifs_lenslet_model_get_as_propertylist( self->model );
    header = cpl_propertylist_new();
    cpl_propertylist_append_string( header, "EXTNAME", SPH_LDT_EXTID );
    cpl_propertylist_append(header, self->properties);
    if ( mode == CPL_IO_CREATE )
        cpl_table_save( tab, pheader,header,filename, CPL_IO_CREATE );
    else {
        cpl_propertylist_append( header, pheader );
        cpl_table_save( tab, NULL, header,filename, mode );
    }
    cpl_propertylist_delete( pheader );pheader = NULL;
    cpl_propertylist_delete( header );header = NULL;
    cpl_table_delete( tab ); tab = NULL;
    return SPH_RAISE_CPL_RESET;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Save the LDT as a FITS cube
 *
 * @param intab         The input LDT to divide from
 * @param filename      The filename to save as
 *
 * Saves the LDT as a FITS cube in the format described above.
 *
 * @return error code of the operation.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_ldt_save_cube(const sph_ldt* intab,
                                 const char* filename,
                                 cpl_frameset* allusedframes,
                                 cpl_frameset* allrecipeframes,
                                 const cpl_frame* template_frame,
                                 const cpl_parameterlist* params,
                                 const char* tag,
                                 const char* recipe,
                                 const char* pipename,
                                 float pixscale,
                                 float parang,
                                 cpl_propertylist* plist)
{
    cpl_propertylist*            proplist2    = NULL;
    sph_interpolation*           interp = NULL;
    cpl_ensure_code( intab && filename, CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    int nx = intab->model->lenslets_per_side * 2 + 1;
    int ny = nx;

    int nplanes = cpl_vector_get_size( intab->reflambdas );
    const double min_lambda = cpl_vector_get_min(intab->reflambdas);

    cpl_imagelist* imlist_im  = cpl_imagelist_new();
    cpl_imagelist* imlist_bpix  = cpl_imagelist_new();
    cpl_imagelist* imlist_rms  = cpl_imagelist_new();
    cpl_imagelist* imlist_wgt  = cpl_imagelist_new();
    cpl_propertylist* proplist = cpl_propertylist_new();

    cpl_propertylist_copy_property_regexp( proplist,
            intab->properties, ".*ESO.*", 0 );
    cpl_propertylist_append( proplist,
            proplist2 = sph_ifs_lenslet_model_get_as_propertylist( intab->model )
            );


    if (tag) cpl_propertylist_update_string( proplist,
            SPH_COMMON_KEYWORD_PRO_CATG, tag );

    for (int ii = 0; ii < nplanes ; ++ii ) {
        sph_master_frame* mf_plane =
            sph_ldt_create_mframe_interpolate(intab, nx, ny, ii, 0.0, &interp);

        if ( mf_plane ) {
            cpl_imagelist_set( imlist_im,
                    mf_plane->image,
                    cpl_imagelist_get_size(imlist_im));
            cpl_imagelist_set( imlist_bpix,
                    mf_plane->badpixelmap,
                    cpl_imagelist_get_size(imlist_bpix));
            cpl_imagelist_set( imlist_rms,
                    mf_plane->rmsmap,
                    cpl_imagelist_get_size(imlist_rms));
            cpl_imagelist_set( imlist_wgt,
                    mf_plane->ncombmap,
                    cpl_imagelist_get_size(imlist_wgt));
            mf_plane->image = NULL;
            mf_plane->ncombmap = NULL;
            mf_plane->rmsmap = NULL;
            mf_plane->badpixelmap = NULL;
            if ( cpl_propertylist_has(
                    mf_plane->properties,
                    SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE) )
            {
                cpl_propertylist_update_double(
                        proplist,
                        SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE,
                        cpl_propertylist_get_double(
                                mf_plane->properties,
                                SPH_COMMON_KEYWORD_LENSLET_IMAGE_SCALE) );
            }
            sph_master_frame_delete( mf_plane );mf_plane = NULL;
        }
        else {
            SPH_RAISE_CPL;
            return cpl_error_get_code();
        }
    }
    if (plist) cpl_propertylist_append(proplist,plist);

    // Saving of science data section...
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLASS,"ESO");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUDOC,"DICD");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUVERS,"DICD version 6");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS1,"IMAGE");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS2,"DATA");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS3,"DATA");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_ERRDATA,"ERROR");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_QUALDATA,"BADMASK");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_SCIDATA,"SCIDATA");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_EXTNAME,"SCIDATA");
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXX,1);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXY,1);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXZ,1);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXXVAL,1);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXYVAL,1);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXZVAL,min_lambda);
#ifdef SPH_USE_WCS
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE,
                                   "RA---TAN");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE,
                                   "DEC--TAN");
#else
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE,
                                   "PIXEL");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE,
                                   "PIXEL");
#endif
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXZTYPE,
                                   "WAVE");
#ifdef SPH_USE_WCS
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT,
                                   "deg");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT,
                                   "deg");
#else
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT,
                                   "pixel");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT,
                                   "pixel");
#endif

    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_WCS_CRPIXZUNIT,
                                   "Microns");
#ifdef SPH_USE_WCS
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD11,
                                   pixscale*cos(parang*CPL_MATH_RAD_DEG));
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD12,
                                   pixscale*sin(parang*CPL_MATH_RAD_DEG));
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD21,
                                   pixscale*sin(parang* -CPL_MATH_RAD_DEG));
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD22,
                                   pixscale*cos(parang*CPL_MATH_RAD_DEG));
#else
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD11, 1.0);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD12, 0.0);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD21, 0.0);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD22, 1.0);
#endif
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD13, 0.0);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD23, 0.0);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD31, 0.0);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD32, 0.0);
    cpl_propertylist_update_double(proplist,SPH_COMMON_KEYWORD_WCS_CD33,
                                   intab->model->dispersion);

    /* Make any final header updates */
    sph_utils_update_header(proplist);

    if ( allusedframes && allrecipeframes ) {
        cpl_dfs_save_imagelist(allrecipeframes, NULL, params,
                allusedframes,
                template_frame,
                imlist_im,
                CPL_TYPE_FLOAT, recipe, proplist,
                NULL, pipename, filename );
    }
    else {
        cpl_msg_warning(cpl_func, "Saving non-DFS imagelist: %s", filename);
        cpl_imagelist_save(imlist_im,filename,CPL_TYPE_DOUBLE,proplist,
                CPL_IO_CREATE);
    }
    cpl_imagelist_delete(imlist_im);imlist_im = NULL;

    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS1,"IMAGE");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS2,"QUALITY");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS3,"MASKZERO");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_EXTNAME,"BADMASK");
    cpl_imagelist_save(imlist_bpix,filename,CPL_BPP_8_UNSIGNED,proplist,
            CPL_IO_EXTEND);
    cpl_imagelist_delete(imlist_bpix);imlist_bpix = NULL;

    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_EXTNAME,"ERROR");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS1,"IMAGE");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS2,"ERROR");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS3,"RMSE");
    cpl_imagelist_save(imlist_rms,filename,CPL_TYPE_DOUBLE,proplist,
            CPL_IO_EXTEND);
    cpl_imagelist_delete(imlist_rms);imlist_rms = NULL;

    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_EXTNAME,"WGT");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS1,"IMAGE");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS2,"ERROR");
    cpl_propertylist_update_string(proplist,SPH_COMMON_KEYWORD_3D_HDUCLAS3,"INVMSE");
    cpl_imagelist_save(imlist_wgt,filename,CPL_TYPE_DOUBLE,proplist,
            CPL_IO_EXTEND);
    cpl_imagelist_delete(imlist_wgt);imlist_wgt = NULL;
    cpl_propertylist_delete(proplist); proplist = NULL;
    cpl_propertylist_delete(proplist2); proplist2 = NULL;

    SPH_RAISE_CPL
    sph_ldt_save( intab, filename, CPL_IO_EXTEND);
    sph_interpolation_delete(interp); interp = NULL;

    return cpl_error_get_code() ? cpl_error_set_where(cpl_func)
        : CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the lenslet description table and free memory.
 *
 * @param self The table to delete
 *
 * @return Error code.
 *
 * The lenslet description table is deleted, as are all associated data
 * strctures. All memory is freed.
 */
/*----------------------------------------------------------------------------*/
void
sph_ldt_delete( sph_ldt* self ) {
    if ( !self ) {
        return;
    }
    if ( self->model ) {
        sph_ifs_lenslet_model_delete( self->model );
        self->model = NULL;
    }
    if ( self->arr ) {
        sph_ldt_delete_arr(self);
    }
    if ( self->properties ) {
        cpl_propertylist_delete( self->properties );
        self->properties = NULL;
    }
    if ( self->reflambdas ) {
        cpl_vector_delete( self->reflambdas );
        self->reflambdas = NULL;
    }
    cpl_free(self);
}

/*----------------------------------------------------------------------------*/
//                         INTERNALS
/*----------------------------------------------------------------------------*/

static
void
sph_ldt_delete_arr( sph_ldt* self ) {
    int n = 0;

    if ( self->arr ) {
        for (n = 0; n < self->nlens; ++n) {
            sph_lenslet_descriptor_delete(self->arr[n]);
            self->arr[n]=NULL;
        }
        cpl_free(self->arr); self->arr = NULL;
    }
}

static
int
sph_ldt_check_ldt_match_ok(
        sph_ldt* ldtA,
        sph_ldt* ldtB )
{
    if ( ldtA->nlens != ldtB->nlens ) {
        sph_error_raise( SPH_ERROR_GENERAL,
                __FILE__, __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "The number of lenses are not the same in the old and "
                "the new created LDT. " );
        return 0;
    }

    if ( cpl_vector_get_size( ldtA->reflambdas ) !=
            cpl_vector_get_size( ldtB->reflambdas ) )
    {
        sph_error_raise( SPH_ERROR_GENERAL,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_WARNING, "The reference wavelengths vector had "
                "different lengths for the new and old tables: %d and %d "
                "respectively. "
                "Currently this is not supported",
                (int)cpl_vector_get_size( ldtA->reflambdas ),
                (int)cpl_vector_get_size( ldtB->reflambdas ) );
        return 0;
    }
    return 1;
}

static
sph_error_code
sph_ldt_draw_poly(
        int lens,
        sph_ldt* intab,
        sph_ldt* ldt_new,
        sph_polygon* poly,
        int* indexes )
{
    sph_error_code rerr = CPL_ERROR_NONE;
    int ii = 0;
    int    u = 0,v = 0;
    sph_polygon* poly2 = NULL;
    int    dindex = 0;
    double overlap = 0.0;
    sph_lenslet_descriptor* descr = NULL;
    int vp = 0;
    sph_polygon_workspace*      ws = NULL;

    ws = sph_polygon_workspace_new();

    for (ii = 0; ii < poly->npoints + 1; ++ii) {
        if ( ii < poly->npoints ) {
            rerr = sph_ifs_lenslet_model_get_coords( ldt_new->model,
                    poly->points[ii].x,
                    poly->points[ii].y, &u, &v);
        }
        else { // Do this since also need to check centre point
            double cx, cy;
            sph_polygon_get_midxy_(poly, &cx, &cy);
            rerr = sph_ifs_lenslet_model_get_coords( ldt_new->model,
                    cx, cy,
                    &u, &v);
        }
        if ( rerr == CPL_ERROR_NONE ) {
            dindex = sph_ldt_check_project_poly_ok(ldt_new->model,
                    u,v,ii,indexes);
            if (  dindex != -1 )
            {
                poly2 = sph_ifs_lenslet_model_get_poly( ldt_new->model,
                        u, v );
                overlap = sph_polygon_calculate_overlap( poly,
                        poly2,ws ) /
                        sph_polygon_area( poly );
                if ( overlap >
                        SPH_LDT_MIN_OVERLAP ) {
                    indexes[ii] = dindex;
                    descr = sph_ldt_get( ldt_new,
                            u, v );
                    if ( descr ) {
                        for ( vp = 0; vp < cpl_vector_get_size(
                                intab->reflambdas); ++vp )
                        {
                            if ( intab->arr[lens]->bpix[vp] == 0 )
                            {
                              descr->values[vp] += overlap *
                                      intab->arr[lens]->values[vp];
                              descr->ncomb[vp] += overlap;
                            }
                        }
                    }
                }
                sph_polygon_delete( poly2 ); poly2 = NULL;
            }
        }
        else rerr = CPL_ERROR_NONE;
    }
    sph_polygon_workspace_delete(ws);
    return rerr;
}



static
int
sph_ldt_check_project_poly_ok( sph_ifs_lenslet_model* model,
        int u, int v, int ii, int* indexes) {
    int dindex = 0;
    int    jj = 0;
    dindex = sph_ifs_lenslet_model_get_index(model,
            u, v );

    if ( dindex == -1 ) {
        return -1;
    }
    for (jj = 0; jj < ii; ++jj) {
        if ( dindex == indexes[jj] ) return -1;
    }
    return dindex;
}









