/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */



#ifndef SPH_LDT_H_
#define SPH_LDT_H_

#include <cpl.h>
#include "sph_ifs_lenslet_model.h"
#include "sph_lenslet_descriptor.h"

#include "sph_pixel_description_table.h"
#include "sph_spectrum.h"
#include "sph_cube.h"
#include "sph_polygon.h"
#include "sph_image_grid.h"
#include "sph_interpolation.h"

#define SPH_IFS_INSTRUMENT_MODEL_LENSID "LENSID"
#define SPH_IFS_INSTRUMENT_MODEL_FLATVAL "FLAT"
#define SPH_IFS_INSTRUMENT_MODEL_FLATRMS "RMS"
#define SPH_IFS_INSTRUMENT_MODEL_FLATBADPIX "BADPIX"

#define SPH_LDT_EXTID "LDTTAB"

static const double SPH_IFS_PIXEL_SCALE = 0.00746;
static const int SPH_IFS_ASTROM_ROTATIONSIGN = 1;

enum sph_ldt_ext {
    SPH_LDT_IMAGE               = 0,
    SPH_LDT_BADPIX              = 1,
    SPH_LDT_RMSMAP              = 2,
    SPH_LDT_NCOMBMAP            = 3,
    SPH_LDT_LAMBDAS             = 4
};

typedef struct _sph_ldt_
{
    int                         nlens;  // number lenslets
    cpl_propertylist*           properties;
    sph_ifs_lenslet_model*      model;
    sph_lenslet_descriptor**    arr; // the lenslet descrption array
    cpl_vector*                 reflambdas; //wavelengths for standardised LDT
} sph_ldt;


sph_ldt*
sph_ldt_new( sph_ifs_lenslet_model* model ) CPL_ATTR_ALLOC;

sph_error_code
sph_ldt_set_model(
        sph_ldt* self,
        sph_ifs_lenslet_model* model );

sph_ldt*
sph_ldt_load( const char* filename ) CPL_ATTR_ALLOC;

sph_lenslet_descriptor*
sph_ldt_get( sph_ldt* self,
        int nu,
        int nv );

sph_error_code
sph_ldt_divide_scalar(
        sph_ldt* intab,
        double val );

sph_error_code
sph_ldt_divide_ifu_flat(
        sph_ldt* self,
        cpl_table* ifu_tab);

sph_error_code
sph_ldt_subtract(
        sph_ldt* intab,
        sph_ldt* stab );

sph_error_code
sph_ldt_divide(
        sph_ldt* intab,
        sph_ldt* stab);

sph_error_code
sph_ldt_divide_median(
        sph_ldt* intab,
        sph_ldt* stab );

sph_error_code
sph_ldt_add(
        sph_ldt* intab,
        sph_ldt* stab );

sph_ldt*
sph_ldt_new_rotate_gimros(
        sph_ldt* intab,
        double angle,
        double factor) CPL_ATTR_ALLOC;

sph_error_code sph_ldt_save_cube(const sph_ldt* intab,
                      const char* filename,
                      cpl_frameset* allusedframes,
                      cpl_frameset* allrecipeframes,
                      const cpl_frame* template_frame,
                      const cpl_parameterlist* params,
                      const char* tag,
                      const char* recipe,
                      const char* pipename,
                      float pixscale,
                      float parang,
                      cpl_propertylist* plist);

sph_error_code sph_ldt_save(const sph_ldt* self,
                            const char* filename,
                            unsigned int mode);

sph_ldt*
sph_ldt_new_from_pdt_image(
        sph_pixel_description_table* pdt,
        sph_ifs_lenslet_model* model,
        cpl_image* image ) CPL_ATTR_ALLOC;

sph_ldt*
sph_ldt_new_from_pdt_master_frame(
        sph_pixel_description_table* pdt,
        sph_ifs_lenslet_model* model,
        sph_master_frame* mf ) CPL_ATTR_ALLOC;


sph_master_frame*
sph_ldt_create_mframe(
        sph_ldt* intab,
		int nx,
		int ny,
        int plane,
        double angle,
        cpl_image** overlaps ) CPL_ATTR_ALLOC;

sph_master_frame*
sph_ldt_create_mframe_interpolate( const sph_ldt* intab,
        int nx,
        int ny,
        int plane,
        double angle,
        sph_interpolation** interp ) CPL_ATTR_ALLOC;

sph_master_frame*
sph_ldt_collapse(
        sph_ldt* intab,
        double angle,
        double magfac) CPL_ATTR_ALLOC;

sph_master_frame*
sph_ldt_collapse_interpolate( sph_ldt* intab,
        double angle,
        double magfac ) CPL_ATTR_ALLOC;
int
sph_ldt_get_nwavs(
        sph_ldt* self );

sph_master_frame*
sph_ldt_create_detframe(
        sph_ldt* self,
        sph_pixel_description_table* pdtin ) CPL_ATTR_ALLOC;

void
sph_ldt_delete( sph_ldt* self );


#ifdef _sph_ldt_c_
const int SPH_LDT_GENERAL;
const int SPH_LDT_NOT_STANDARDISED;
const int SPH_LDT_NO_REFERENCE_WAVELENGTHS ;
const float SPH_LDT_WAVETOLERANCE  ;

static
void
sph_ldt_delete_arr( sph_ldt* self );

static
int
sph_ldt_check_ldt_match_ok(
        sph_ldt* ldtA,
        sph_ldt* ldtB );

static
int
sph_ldt_check_project_poly_ok( sph_ifs_lenslet_model* model,
        int u, int v, int ii, int* indexes);

static
sph_error_code
sph_ldt_draw_poly(
        int lens,
        sph_ldt* intab,
        sph_ldt* ldt_new,
        sph_polygon* poly,
        int* indexes );


#else
extern const int SPH_LDT_GENERAL;
extern const int SPH_LDT_NOT_STANDARDISED;
extern const int SPH_LDT_NO_REFERENCE_WAVELENGTHS ;
extern const float SPH_LDT_WAVETOLERANCE  ;
#endif


#endif /*SPH_LDT_H_*/
