/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */


#include "sph_master_frame.h"
#include <string.h>
#include <assert.h>

#ifndef SPH_MAX_CONTRIB
#define SPH_MAX_CONTRIB 2048
#endif


static cpl_size sph_master_frame_interpolate(sph_master_frame *)
    CPL_ATTR_NONNULL;

/*----------------------------------------------------------------------------*/
/**
  @brief    Interpolate any bad pixel in the master frame and clear the BPM
  @param    self    The master frame to clean
  @return   The number of pixels interpolated, from 0 to all
  @see cpl_detector_interpolate_rejected()
  @note If the interpolation count is 'all' (nx * ny), the interpolation failed
        - also in this case the BPM of each image is deleted

  Iff succesful, the badpixelmap is cleared

 */
/*----------------------------------------------------------------------------*/
static cpl_size sph_master_frame_interpolate(sph_master_frame * self)
{

    const size_t nx = cpl_image_get_size_x(self->badpixelmap);
    const size_t ny = cpl_image_get_size_y(self->badpixelmap);
    cpl_size     nset = 0;

    /* If an image has a BPM, remove it */
    cpl_mask* bpmimg = cpl_image_unset_bpm(self->image);
    cpl_mask* bpmrms = cpl_image_unset_bpm(self->rmsmap);
    cpl_mask* bpmnco = cpl_image_unset_bpm(self->ncombmap);
    cpl_mask* bpmnew = NULL;
    cpl_mask* bpm    = bpmimg ? bpmimg : (bpmrms ? bpmrms :
                                          (bpmnco ? bpmnco : NULL));
    cpl_binary* pbpm;
    cpl_binary* doit;


    if (bpm == NULL) {
        bpm = bpmnew = cpl_mask_new(nx, ny);
    }
    pbpm = cpl_mask_get_data(bpm);

    (void)cpl_mask_threshold_image(bpm, self->badpixelmap, 0.5,
                                   SPH_MAX_CONTRIB, CPL_BINARY_1);

    doit = memchr(pbpm, CPL_BINARY_1, nx * ny * sizeof(*pbpm));

#ifdef SPH_MASTER_FRAME_INTERPOLATE_CHECK
    if (doit == pbpm &&
        memchr(pbpm, CPL_BINARY_0, nx * ny * sizeof(*pbpm)) == NULL) {
        cpl_msg_warning(cpl_func, "No good pixel in masterframe of size "
                        "(%zd,%zd) and type %s", nx, ny, cpl_type_get_name(
                        cpl_image_get_type(self->badpixelmap)));

        nset = nx * ny; /* Interpolation failed */
        doit = NULL;
    }
#endif

    if (doit) {
        double * pimg = cpl_image_get_data_double(self->image);
        double * prms = cpl_image_get_data_double(self->rmsmap);
        double * pnco = cpl_image_get_data_double(self->ncombmap);

        const cpl_binary * prev = doit;
        /* Need a copy of the bpm, to not use bpm as interpolation source
           pixels that have been interpolated in the same pass */
        cpl_binary* pcopynew = NULL;
        cpl_binary* pcopy = NULL;
        cpl_boolean ok; /* All pixels may be bad */

        if (bpmnew == NULL) {
            cpl_mask* copy = bpmnco && bpmnco != bpm ? bpmnco :
                (bpmrms && bpmrms != bpm ? bpmrms : NULL);
            pcopy = copy ? cpl_mask_get_data(copy) : NULL;
        }

        if (pcopy == NULL)
            pcopy = pcopynew = cpl_malloc(nx * ny * sizeof(*pbpm));

        (void)memset(pcopy, CPL_BINARY_0, (size_t)(prev - pbpm));

        do {
            cpl_binary * found = doit;

            assert( *doit );

            /* No more bad pixels between previous and current bad pixel */
            (void)memset(pcopy + (prev - pbpm), CPL_BINARY_0,
                         (size_t)(doit - prev) * sizeof(*pbpm));
            /* Bad pixels may have been cleaned in remaining buffer as well */
            (void)memcpy(pcopy + (doit - pbpm), doit,
                         (nx * ny - (size_t)(doit - pbpm)) * sizeof(*pbpm));
            prev = doit;
            doit = NULL;

            ok = CPL_FALSE;

            do {
                /* Found bad pixel at (i,j) */
                const size_t ij = found - pbpm;
                const size_t j = ij / nx;
                const size_t i = ij - j * nx;

                /* Try to interpolate the bad pixel  */
                size_t npix = 0;
                double sumimg = 0.0;
                double sumrms = 0.0;
                double sumnco = 0.0;


                if (i > 0) { /* The three pixels to the left */
                    if (pcopy[ij - 1] == CPL_BINARY_0) {
                        sumimg += pimg[ij - 1];
                        sumrms += prms[ij - 1];
                        sumnco += pnco[ij - 1];
                        npix++;
                    }
                    if (j > 0 && pcopy[ij - 1 - nx] == CPL_BINARY_0) {
                        sumimg += pimg[ij - 1 - nx];
                        sumrms += prms[ij - 1 - nx];
                        sumnco += pnco[ij - 1 - nx];
                        npix++;
                    }
                    if (j + 1 < ny && pcopy[ij - 1 + nx] == CPL_BINARY_0) {
                        sumimg += pimg[ij - 1 + nx];
                        sumrms += prms[ij - 1 + nx];
                        sumnco += pnco[ij - 1 + nx];
                        npix++;
                    }
                }

                /* The two pixels below and above */
                if (j > 0 && pcopy[ij - nx] == CPL_BINARY_0) {
                    sumimg += pimg[ij - nx];
                    sumrms += prms[ij - nx];
                    sumnco += pnco[ij - nx];
                    npix++;
                }
                if (j + 1 < ny && pcopy[ij + nx] == CPL_BINARY_0) {
                    sumimg += pimg[ij  + nx];
                    sumrms += prms[ij  + nx];
                    sumnco += pnco[ij  + nx];
                    npix++;
                }

                if (i + 1 < nx) { /* The three pixels to the right */
                    if (pcopy[ij + 1] == CPL_BINARY_0) {
                        sumimg += pimg[ij + 1];
                        sumrms += prms[ij + 1];
                        sumnco += pnco[ij + 1];
                        npix++;
                    }
                    if (j > 0 && pcopy[ij + 1 - nx] == CPL_BINARY_0) {
                        sumimg += pimg[ij + 1 - nx];
                        sumrms += prms[ij + 1 - nx];
                        sumnco += pnco[ij + 1 - nx];
                        npix++;
                    }
                    if (j + 1 < ny && pcopy[ij + 1 + nx] == CPL_BINARY_0) {
                        sumimg += pimg[ij + 1 + nx];
                        sumrms += prms[ij + 1 + nx];
                        sumnco += pnco[ij + 1 + nx];
                        npix++;
                    }
                }

                if (npix > 0) {
                    /* Found source pixel(s) for interpolation */
                    pimg[ij] = sumimg / (double)npix;
                    prms[ij] = sumrms / (double)npix;
                    pnco[ij] = sumnco / (double)npix;
                    *found = CPL_BINARY_0;
                    ok = CPL_TRUE;
                    nset++;
                } else if (doit == NULL) {
                    doit = found; /* First bad pixel to redo */
                }

#ifndef SPH_MASTER_FRAME_INTERPOLATE_OPTIMIZE
                found = memchr(found+1, CPL_BINARY_1,
                               (size_t)(nx * ny - ij - 1) * sizeof(*pbpm));
#else
                if (npix > 0) {
                    found = memchr(found+1, CPL_BINARY_1,
                                   (size_t)(nx * ny - ij - 1) * sizeof(*pbpm));
                } else {
                    /* No good pixels nearby, so look for the next as source */
                    cpl_binary* good =
                        memchr(found+1, CPL_BINARY_0,
                               (size_t)(nx * ny - ij - 1) * sizeof(*pbpm));
                    if (good != NULL) {
                        const size_t ijg = good - pbpm;
                        const size_t jg = ijg / nx;
                        const size_t ig = ijg - jg * nx;

                        assert(good > found+1 ); /* No good pixel nearby */
                        assert( jg > 0 || ig > 0);
                        found = good;
                        if (ig > 0) found -= 1;
                        if (jg > 0) found -= nx;
                        assert(found < good ); /* Not a good pixel */
                    } else {
                        cpl_msg_warning(cpl_func, "No good pixel in masterframe"
                                        " of size (%zd,%zd) and type %s",
                                        nx, ny, cpl_type_get_name(
                                        cpl_image_get_type(self->badpixelmap)));

                        found = NULL;
                        assert( doit ); /* There are pixels to interpolate */
                        assert( !ok ); /* No good pixel nearby -> at all */
                    }
                }
#endif

            } while (found);

        } while (doit && ok);

        if (ok) {
            const cpl_type img_type = cpl_image_get_type(self->badpixelmap);

            assert( !doit );

            if (img_type == CPL_TYPE_INT ) {
                (void)memset(cpl_image_get_data_int(self->badpixelmap), 0,
                             nx * ny * sizeof(int));
            } else {
                cpl_msg_warning(cpl_func, "Interpolated (%d) masterframe with "
                                "a bad pixel map of size (%zd,%zd) and type %s",
                                (int)nset, nx, ny, cpl_type_get_name(img_type));
                /* If it is not int, badpixelmap type should be double */
                assert( img_type == CPL_TYPE_DOUBLE );
                (void)memset(cpl_image_get_data_double(self->badpixelmap), 0,
                             nx * ny * sizeof(double));
            }
	} else {
            nset = nx * ny; /* Interpolation failed */
            assert( doit );
        }

        cpl_free(pcopynew);
    }

    cpl_mask_delete(bpmimg);
    cpl_mask_delete(bpmrms);
    cpl_mask_delete(bpmnco);
    cpl_mask_delete(bpmnew);

    return nset;

}
