/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include "sph_pixel_polyfit.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_pixel_polyfit Pixel Polynomial Fit Object
 *
 * @par Synopsis:
 * @par Descirption:
 * This module provides functions to define and operate on a object
 * describing function fits to pixel behaviours.
 * It is an imagelist of type double containing one fit for every pixel.
 * The optional error image contains the reduced chi squared for the fit.
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
 Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new pixel polyfit object by wrapping around the images
 *
 * @param self  The imagelist containing the coefficient images to wrap around
 * @param error Optional image with error of fit to also wrap around
 * @return Pointer to new object or null in case of error.
 * @note The created object will take ownership of the input
 *
 */
/*----------------------------------------------------------------------------*/
sph_pixel_polyfit* sph_pixel_polyfit_wrap(cpl_imagelist* self,
                                          cpl_image*     error)
{
    sph_pixel_polyfit* result = NULL;

    if (self == NULL) {
        (void)cpl_error_set_where(cpl_func);
    } else {
        const cpl_size   ncoeff = cpl_imagelist_get_size(self);
        const cpl_image* im0    = cpl_imagelist_get_const(self, 0);
        const cpl_size   nx     = cpl_image_get_size_x(im0);
        const cpl_size   ny     = cpl_image_get_size_y(im0);

        cpl_ensure(cpl_image_get_type(im0) == CPL_TYPE_DOUBLE,
                   CPL_ERROR_ILLEGAL_INPUT, NULL);

        if (error != NULL) {
            cpl_ensure(cpl_image_get_size_x(error) == nx,
                       CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
            cpl_ensure(cpl_image_get_size_x(error) == ny,
                       CPL_ERROR_INCOMPATIBLE_INPUT, NULL);
            cpl_ensure(cpl_image_get_type(error) == CPL_TYPE_DOUBLE,
                       CPL_ERROR_ILLEGAL_INPUT, NULL);
        }

        result = (sph_pixel_polyfit*)cpl_malloc(sizeof(*result));

        result->fit   = self;
        result->error = error;

        result->pfit = cpl_malloc(ncoeff * sizeof(double*));
        result->pbpm = cpl_malloc(ncoeff * sizeof(cpl_binary*));

        for (cpl_size ii = 0; ii < ncoeff; ++ii) {
            const cpl_image* fit = cpl_imagelist_get_const(self, ii);
            const cpl_mask*  bpm = cpl_image_get_bpm_const(fit);

            result->pfit[ii] = cpl_image_get_data_double_const(fit);
            result->pbpm[ii] = bpm ? cpl_mask_get_data_const(bpm) : NULL;
        }

        result->nx    = nx;
        result->ny    = ny;
        result->order = ncoeff - 1;
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the poly tabe
 *
 * @return void
 * @note Frees up the space and deletes the polyfit object
 */
/*----------------------------------------------------------------------------*/
void sph_pixel_polyfit_delete(sph_pixel_polyfit* self)
{
    if (self != NULL) {
        cpl_imagelist_delete(self->fit);
        cpl_image_delete(self->error);
        cpl_free(self->pfit);
        cpl_free(self->pbpm);
        cpl_free(self);
    }
}

/**@}*/
