/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_pixel_polyfit_table.h"
#include "sph_cube.h"
#include "sph_master_frame.h"

#include <math.h>
/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_pixel_polyfit_table Pixel Polynomial Fit Table
 *
 * @par Synopsis:
 * @code
typedef struct _sph_pixel_polyfit_table_
{
    int                     nx;  // number pixel x
    int                     ny;  // number pixel y
    int                     order; // Order of the polynomials used for fitting
    cpl_table*              table; // The table storing fit parameters
} sph_pixel_polyfit_table;
 * @endcode
 * @par Descirption:
 * This module provides functions to define and operate on a table
 * describing function fits to pixel behaviours.
 * It is essentially just a table, which, contains one row for every pixel.
 * The columns contrain the different fitting coefficients of the polynomial
 * fit.
 * The table can be saved and loaded as/from FITS table extensions.
 * The columns of the table are labelled: pixel-x, pixel-y for the
 * integer pixel positions, Red. Chi sfor the reduced chi squared for the
 * fit and then Order-0, Order-1, ... for all the polynomial coefficients.
 *
 * TODO: at some point this should be changed to a FITS image format, where
 * each coefficient is in one plane or extension of a FITS image file instead
 * of a FITS binary table.
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/
sph_error_code SPH_PIXEL_POLYFIT_TABLE_GENERAL      = SPH_PIXEL_POLYFIT_TABLE_ERR_START + 0;

/*-----------------------------------------------------------------------------
 Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new pixel polyfit table thats empty.
 *
 * @param nx    The number of pixels in x
 * @param ny    The number of pixels in y
 * @param order The order of the polynomials used for fitting
 *
 * @return Pointer to new table or null in case of error.
 *
 * A new pixel polyfit table is created with nx * ny entries.
 *
 */
/*----------------------------------------------------------------------------*/
sph_pixel_polyfit_table*
sph_pixel_polyfit_table_new(int nx, int ny, int order ) {
    sph_pixel_polyfit_table*        self   = NULL;
    char      czOrder[32] = "Order-0";

    if (nx < 0) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                    "Cant create pixel polyfit table with"
                                    "negative number of pixels, nx=%d\n", nx);
        return NULL;
    } else if (ny < 0) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                    "Cant create pixel polyfit table with"
                                    "negative number of pixels, ny=%d\n", ny);
        return NULL;
    } else if (order < 0) {
        (void)cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                    "Cant create pixel polyfit table with"
                                    "negative number of coefficients, n=%d\n",
                                    order);
        return NULL;
    }

    self = cpl_calloc( 1, sizeof(sph_pixel_polyfit_table) );

    self->table = cpl_table_new( nx * ny );

    cpl_table_new_column( self->table, "Red. Chi" , CPL_TYPE_DOUBLE );

    for (int ii = 0; ii <= order;
         ++ii < 10 ? czOrder[6]++ : snprintf(czOrder+6, 25, "%d", ii)) {
        cpl_table_new_column( self->table, czOrder , CPL_TYPE_DOUBLE );
    }

    self->nx = nx;
    self->ny = ny;
    self->order = order;
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new pixel polyfit table using the image list
 *
 * @param imagelist the imagelist containing the coefficient images
 * @param errimage  optional image with error of fit
 *
 * @return Pointer to new table or null in case of error.
 *
 * A new pixel polyfit table is created from the input imagelist.
 *
 */
/*----------------------------------------------------------------------------*/
sph_pixel_polyfit_table*
sph_pixel_polyfit_table_new_from_imlist(const cpl_imagelist* imlist,
                                        const cpl_image*     errimage)
{
    const int ncoeff = cpl_imagelist_get_size(imlist);
    const int nx     = cpl_image_get_size_x(cpl_imagelist_get_const(imlist, 0));
    const int ny     = cpl_image_get_size_y(cpl_imagelist_get_const(imlist, 0));
    sph_pixel_polyfit_table* self =
        sph_pixel_polyfit_table_new(nx, ny, ncoeff-1);
    char      czOrder[32] = "Order-0";

    if ( !self ) {
        (void)cpl_error_set_where(cpl_func);
        return NULL;
    }

    for (int ii = 0; ii < ncoeff;
         ++ii < 10 ? czOrder[6]++ : snprintf(czOrder+6, 25, "%d", ii)) {
        const cpl_image* imp = cpl_imagelist_get_const(imlist, ii);
        for (int xx = 0; xx < nx; ++xx) {
            for (int yy = 0; yy < ny; ++yy) {
                int    bpix = 0;
                const double coeff = cpl_image_get(imp, xx+1, yy+1, &bpix);

                if ( bpix == 0 ) {
                    int bdummy;
                    const double mse = errimage == NULL ? 0.0
                        : cpl_image_get(errimage, xx+1, yy+1, &bdummy);

                    cpl_table_set(self->table, czOrder, xx + yy * nx,
                                  coeff);
                    cpl_table_set(self->table, "Red. Chi", xx + yy * nx,
                                  mse);
                }
            }
        }
    }

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set pixel as invalid.
 *
 * @param nx   The pixel in x
 * @param ny   The pixel in y
 *
 * @return ErrorCode of the operation.
 *
 * Sets the  a pixel in the table as invalid.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_pixel_polyfit_table_set_invalid(sph_pixel_polyfit_table* self,
                                    int nx, int ny )
{
    int                             err         = CPL_ERROR_NONE;
    char      czOrder[32] = "Order-0";

    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    } else if ( !self->table ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    } else if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        SPH_RAISE_CPL
        return cpl_error_get_code();
    }
    for (int ii = 0; ii <= self->order;
         ++ii < 10 ? czOrder[6]++ : snprintf(czOrder+6, 25, "%d", ii)) {
        cpl_table_set_invalid( self->table, czOrder, ny * ( self->nx) + nx );
    }
    err |= cpl_table_set_invalid( self->table, "Red. Chi", ny * ( self->nx) + nx );
    if ( err != CPL_ERROR_NONE ) {
        SPH_RAISE_CPL
    }
    return err;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Set pixel fitting coefficients.
 *
 * @param nx   The pixel in x
 * @param ny   The pixel in y
 * @param mse  The red. chi squared of the fit
 * @param poly The polynomial to set coefficients from
 *
 * @return ErrorCode of the operation.
 *
 * Sets the fitting coefficients for a pixel in the table from an input
 * polynomial. The polynomial is assumed to have been created using the
 * cpl_polynomial_fit_1d_create() function and has to have the same order as
 * the table itself.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_pixel_polyfit_table_set(sph_pixel_polyfit_table* self,
                            int nx, int ny,
                            double mse, cpl_polynomial* poly )
{
    const int poldegree = cpl_polynomial_get_degree( poly );
    char      czOrder[32] = "Order-0";

    if ( !self ) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (cpl_polynomial_get_dimension(poly) != 1) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "polynomial of degree %d has dimension %d "
                                     "!= 1", poldegree,
                                     (int)cpl_polynomial_get_dimension(poly));
    } else if ( self->order <  poldegree ) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "The polynomial and the pixel fit table "
                                     "have different order/degree ( %d for "
                                     "table, %d for polynomial). Cant proceed.",
                                     self->order, poldegree);
    }

    for (cpl_size ii = 0; ii <= CPL_MIN(poldegree, self->order);
         ++ii < 10 ? czOrder[6]++ : snprintf(czOrder+6, 25, "%d", (int)ii)) {
        const double coeff   = cpl_polynomial_get_coeff( poly, &ii );

        if (cpl_table_set(self->table, czOrder, nx + ny * self->nx, coeff )) {
            return cpl_error_set_where(cpl_func);
        }
    }

    if (cpl_table_set( self->table, "Red. Chi", ny * ( self->nx) + nx, mse )) {
        return cpl_error_set_where(cpl_func);
    }

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Get fitting coefficients as sph_master_frame.
 *
 * @param self  The sph_pixel_polyfit_table
 * @param order The order of the coefficent (starting with 0)
 *
 * @return     The polynomial coefficients as a sph_master_frame or NULL in
 *             case of error.
 *
 * Gets the fitting polynomial coefficients for all pixels as an image. The
 * order starts with 0, so that requesting the 0th order gives the offset,
 * order 1 gives the linear slope etc.
 *
 * In case an error occurs a null is returned.
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_pixel_polyfit_table_get_coeff_as_image( const sph_pixel_polyfit_table* self,
                                            int order )
{
    char      czOrder[32] = "Order-0";
    sph_master_frame*               result  = NULL;


    if ( !self ) {
        SPH_NULL_ERROR;
        return NULL;
    } else if ( ! self->table ) {
        SPH_NULL_ERROR;
        return NULL;
    } else if ( order > self->order || order < 0 ) {
        sph_error_raise( SPH_PIXEL_POLYFIT_TABLE_GENERAL,
                         __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Invalid order given. "
                         "Can only retrieve order in the range "
                         "0 to % d", self->order );
        return NULL;
    }

    result = sph_master_frame_new( self->nx, self->ny );

    if ( result == NULL ) {
        sph_error_raise( SPH_PIXEL_POLYFIT_TABLE_GENERAL,
                         __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "Could not create master frame to store coefficient." );
        return NULL;
    }

    
    order < 10 ? czOrder[6] += order : snprintf(czOrder+6, 25, "%d", order);

    for (int xx = 0; xx < self->nx; ++xx) {
        for (int yy = 0; yy < self->ny; ++yy) {
            const int irow = xx + yy * self->nx;
            double mse = -1.0;
            int badval = 0;

            if ( cpl_table_is_valid(self->table, czOrder, irow)) {
                const double coeff =
                    cpl_table_get( self->table, czOrder, irow, &badval);

                cpl_image_set( result->image, xx + 1, yy + 1, coeff );
                cpl_image_set( result->badpixelmap, xx + 1, yy + 1, badval );
                mse = cpl_table_get( self->table, "Red. Chi", irow, &badval );
                cpl_image_set( result->rmsmap, xx + 1, yy + 1, mse );
            }

            if ( mse < 0 ) {
                cpl_image_set( result->badpixelmap, xx + 1, yy + 1, 1 );
            }
        }
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Get a mask of badpixels
 *
 * @param self The pixel polyfit table
 * @param badpix_lowtolerance The lower tolerance bound for "good" pixels
 * @param badpix_uptolerance The upper tolerance bound for "good" pixels
 * @param badpix_chisqtolerance The red. chi squared tolerance bound for "good" pixels
 *
 * @return a new cpl_mask with the mask set to 1 for those pixels that are bad.
 *
 * This method creates a new cpl_mask from the sph_pixel_polyfit_table using the
 * given tolerance values as criteria to decide if a pixel is good or bad.
 * The two values lowtolerance and uptolerance give the lower and upper bound
 * on the slope of the polynomial fit (1st coefficient) and the chisqtolerance
 * gives the maximum chi squared thats tolerated for a good pixel unless it is
 * set to zero in which case it is ignored.
 *
 * For example, to select all pixels as good that have a slope (1st polynomial
 * coefficient) between -2.0 and 5.0 and a chi-squared fit below 3.0 call
 * <code>cpl_mask* badpixmask = sph_pixel_polyfit_table_create_badpixel_
 * mask( pStruct, -2.0, 5.0, 3.0);</code>
 */
/*----------------------------------------------------------------------------*/
cpl_mask*
sph_pixel_polyfit_table_create_badpixel_mask( const sph_pixel_polyfit_table* self,
                                              double badpix_lowtolerance,
                                              double badpix_uptolerance,
                                              double badpix_chisqtolerance)
{
    const char *                    czOrder = "Order-1";
    cpl_mask*                       result  = NULL;


    if ( !self ) {
        SPH_NULL_ERROR;
        return NULL;
    } else if ( !self->table ) {
        SPH_NULL_ERROR;
        return NULL;
    } else if ( self->order < 1 ) {
        sph_error_raise( SPH_PIXEL_POLYFIT_TABLE_GENERAL,
                         __FILE__, __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Invalid order . "
                         "Can only create badpix mask if order > 0 "
                         );
        return NULL;
    }

    result = cpl_mask_new( self->nx, self->ny );

    if ( result == NULL ) {
        sph_error_raise( SPH_PIXEL_POLYFIT_TABLE_GENERAL,
                         __FILE__, __func__, __LINE__, SPH_ERROR_ERROR,
                         "Could not create cpl_mask to store mask." );
        return NULL;
    }

    for (int xx = 0; xx < self->nx; ++xx) {
        for (int yy = 0; yy < self->ny; ++yy) {
            const int irow = xx + yy * self->nx;
            int badval = 0;

            if ( cpl_table_is_valid(self->table, czOrder, irow)) {
                const double coeff =
                    cpl_table_get( self->table, czOrder,  irow, &badval );
                double mse;

                cpl_mask_set(result, xx + 1, yy + 1 ,
                             coeff < badpix_uptolerance &&
                             coeff > badpix_lowtolerance
                             ? 0 : 1);


                mse = cpl_table_get( self->table, "Red. Chi",  irow, &badval );

                if ( mse > badpix_chisqtolerance ) {
                    cpl_mask_set( result, xx + 1, yy + 1 , 1 );
                }
            }
            else {
                cpl_mask_set( result, xx + 1, yy + 1 , 1 );
            }
        }
    }

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Get pixel fitting coefficients.
 *
 * @param nx   The pixel in x
 * @param ny   The pixel in y
 * @param mse  The red. chi squared of the fit (returned)
 *             or NULL if not required
 * @return     The polynomial fit of that pixel or NULL on error.
 *
 * Gets the fitting polynomial for a pixel in the table
 * The polynomial is of the same type as returned by the
 * cpl_polynomial_fit_1d_create() function
 * and has to have the same order as
 * the table itself. If mse is provided, it is filled with the
 * reduced chi squared value.
 *
 */
/*----------------------------------------------------------------------------*/

cpl_polynomial*
sph_pixel_polyfit_table_get( const sph_pixel_polyfit_table* self,
                             int nx, int ny,
                             double* mse )
{
    cpl_polynomial* result = NULL;
    char      czOrder[32] = "Order-0";

    if ( !self ) {
        SPH_NULL_ERROR;
    } else if ( ! self->table ) {
        SPH_NULL_ERROR;
    } else {
        result = cpl_polynomial_new( 1 );

        for (cpl_size ii = 0; ii <= self->order;
             ++ii < 10 ? czOrder[6]++ : snprintf(czOrder+6, 25, "%d", (int)ii)) {
            const cpl_size irow = nx + ny * self->nx;

            if (cpl_table_is_valid(self->table, czOrder, irow)) {
                int isbad;
                const double coeff =
                    cpl_table_get(self->table, czOrder, irow, &isbad);

                cpl_polynomial_set_coeff(result, &ii, coeff);

                if (mse != NULL)
                    *mse = cpl_table_get(self->table, "Red. Chi", irow, &isbad);
            }
        }
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Save the pixel polyfit table.
 *
 * @param self The pixel polyfit table to save
 * @param fitsfilename. The filename to save to
 * @return Error code.
 *
 * Saves the table in FITS format. The routines saves
 * the internal cpl_table to the FITS file specified.
 * The file is created ( if it
 * does not yet exist) or overwritten (if it does).
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_pixel_polyfit_table_save(const sph_pixel_polyfit_table* self,
                             const char* fitsfilename)
{
    int                 rerr        = CPL_ERROR_NONE;

    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }

    rerr |= cpl_table_save( self->table, NULL, NULL, fitsfilename, CPL_IO_DEFAULT);
    if ( rerr ) {
        sph_error_raise( (int)cpl_error_get_code(), __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR, "Problem saving table. "
                                "CPL says: %s", cpl_error_get_message() );
        return (int)cpl_error_get_code();
    }

    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Save the pixel polyfit table as a FITS cube.
 *
 * @param self The pixel polyfit table to save
 * @param fitsfilename. The filename to save to
 * @return Error code.
 *
 * Saves the table in FITS image format, according to the
 * sph_cube specifications.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code
sph_pixel_polyfit_table_save_cube(const sph_pixel_polyfit_table* self,
                                  const char* fitsfilename)
{
    int                 rerr        = CPL_ERROR_NONE;
    sph_master_frame*    mf            = NULL;
    sph_cube*            cube        = NULL;
    int                    ii            = 0;

    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }

    cube = sph_cube_new( fitsfilename );

    if ( !cube ) {
        return SPH_PIXEL_POLYFIT_TABLE_GENERAL;
    }

    for ( ii = 0;  ii <= self->order; ++ ii) {
        mf = sph_pixel_polyfit_table_get_coeff_as_image( self, ii );
        if ( mf ) {
            sph_cube_append_master( cube, mf, ii * 1.0 );
            sph_master_frame_delete( mf );
        }
        else {
            rerr = SPH_PIXEL_POLYFIT_TABLE_GENERAL;
            sph_master_frame_delete( mf );
            break;
        }
    }
    sph_cube_finalise_file(cube->filename);
    sph_cube_delete(cube); cube = NULL;
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Load a pixel polyfit table.
 *
 * @param fitsfilename The filename to load from
 * @return Pointer to the loaded table.
 *
 * Loads a sph_pixel_polyfit_table from a FITS filename. The routine
 * loads the cpl_table and then creates and returns a sph_pixel_polyfit_table
 * structure.
 */
/*----------------------------------------------------------------------------*/
sph_pixel_polyfit_table*
sph_pixel_polyfit_table_load(  const char* fitsfilename ) {
    sph_pixel_polyfit_table*    self   = NULL;
    int                             nx      = 0;
    int                             ny      = 0;
    cpl_table*                      ctab    = NULL;
    int                             order   = 0;

    ctab = cpl_table_load( fitsfilename, 1, 0);

    if ( !ctab ) {
        sph_error_raise(cpl_error_get_code(),
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,"%s",
                        cpl_error_get_message() );
        return NULL;
    }

    nx = (int)sqrt( cpl_table_get_nrow(ctab) );
    ny = nx;
    order = cpl_table_get_ncol( ctab ) - 4;

    if ( order < 0 ) {
        cpl_table_delete(ctab);
        return NULL;
    }

    self = sph_pixel_polyfit_table_new( nx, ny , order );

    if ( ! self ) {
        cpl_table_delete(ctab);
        return NULL;
    }

    self->table = ctab;

    return self;

}
/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the poly tabe
 *
 * @return void
 * @note Frees up the space and deletes the polyfit table
 */
/*----------------------------------------------------------------------------*/
void sph_pixel_polyfit_table_delete( sph_pixel_polyfit_table* self ) {
    if ( self != NULL ) {
        cpl_table_delete( self->table );
        cpl_free(self);
    }
}

/**@}*/
