/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_point.h"

#include "cpl.h"

#include <math.h>
#include <assert.h>

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new point
 * @param x    x value
 * @param y    y value
 *
 * @return new point or NULL
 *
 * Creates a new point in 2D with the given coordinates.
 */
/*----------------------------------------------------------------------------*/
sph_point* sph_point_new(double x, double y)
{
    sph_point* self = (sph_point*)cpl_malloc(sizeof(sph_point));

    assert( self != NULL);

    self->x = x;
    self->y = y;

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Check if two points lie on the same side of a line
 * @param l1     First  point defining line
 * @param l2     Second point defining line
 * @param p1     First  point to check
 * @param p2     Second point to check
 * @return Either non-zero (True) if the points lie on the same side,
 *         or zero (False) if they lie on opposite sides
 * @note No checking for NULL-input for performance reasons. If the line is a
         True is returned for any pair of points.
 * @see sph_point_ccw_test()
 */
/*----------------------------------------------------------------------------*/
cpl_boolean sph_point_same_side(const sph_point* l1,
                                const sph_point* l2,
                                const sph_point* p1,
                                const sph_point* p2)
{
    return
        ((p1->y -l1->y) * (l2->x -l1->x) > (l2->y -l1->y) * (p1->x -l1->x)) ==
        ((p2->y -l1->y) * (l2->x -l1->x) > (l2->y -l1->y) * (p2->x -l1->x));
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Check if two points lie on the same side of a line
 * @param l1     First  point defining line
 * @param l2     Second point defining line
 * @param p1     First  point to check
 * @param p2     Second point to check
 * @return Either non-zero (True) if the points lie on the same side,
 *         or zero (False) if they lie on opposite sides
 * @see sph_point_same_side()
 * @note If one point lies on the line, True is always returned
 */
/*----------------------------------------------------------------------------*/
cpl_boolean sph_point_same_side_(const sph_point* l1,
                                 const sph_point* l2,
                                 const sph_point* p1,
                                 const sph_point* p2)
{
    return
        ((p1->y -l1->y) * (l2->x -l1->x) >= (l2->y -l1->y) * (p1->x -l1->x)) ==
        ((p2->y -l1->y) * (l2->x -l1->x) >= (l2->y -l1->y) * (p2->x -l1->x));
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Check if points are counter clockwise
 * @param A     point A
 * @param B     point B
 * @param C     point C
 *
 * @return Etiher 0 (false) or 1 (True)
 */
/*----------------------------------------------------------------------------*/
short sph_point_ccw_test(sph_point A, sph_point B, sph_point C) {
    return (C.y-A.y) * (B.x-A.x) > (B.y-A.y) * (C.x-A.x);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Rotate a point around another
 * @param self      the point to rottate
 * @param x         the centre of rotation x coord.
 * @param y         the centre of rotation y coord.
 * @param cosangle  the cosine of the rotation angle
 * @param sinangle  the sine of the rotation angle
 * @param dx        optonal shift applied after roation
 * @param dy        optonal shift applied after roation
 *
 * @return error code of the operation
 *
 * This function rotations the point around the given center and
 * optionally applied a shift by dx, dy afer the rotation to the point.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_point_rotate_around(sph_point* self,
                                       double x,
                                       double y,
                                       double cosangle,
                                       double sinangle,
                                       double dx,
                                       double dy)
{

    if (self == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                                     "x=%g, y=%g, dx=%g, dy=%g, cosangle=%g, "
                                     "sinangle=%g", x, y, dx, dy, cosangle,
                                     sinangle);
    } else {
        const double oldx = self->x;

        self->x = cosangle * (oldx - x) - sinangle * (self->y - y) + dx + x;
        self->y = sinangle * (oldx - x) + cosangle * (self->y - y) + dy + y;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Shift a point
 * @param self      the point to shift
 * @param dx        shift in x
 * @param dy        shift in y
 *
 * @return error code of the operation
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_point_shift(sph_point* self, double dx, double dy)
{

    if (self == NULL) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_NULL_INPUT,
                                     "dx=%g, dy=%g", dx, dy);
    } else {

        self->x += dx;
        self->y += dy;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete a point
 * @param self          the point to delete
 */
/*----------------------------------------------------------------------------*/
void sph_point_delete(sph_point* self) {
    cpl_free(self);
}
