/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_polygon.h"

#include "sph_error.h"

#include <cpl.h>

#include <string.h>
#include <math.h>
#include <limits.h>
#include <assert.h>


/**@{*/

/*-----------------------------------------------------------------------------
                        Defines
 -----------------------------------------------------------------------------*/

#ifndef SPH_POLYGON_INISIZE
#define SPH_POLYGON_INISIZE 8
#endif

#define SPH_REDUCE(A, B) if ((B) < A) A = (B)
#define SPH_INCREASE(A, B) if ((B) > A) A = (B)

/*-----------------------------------------------------------------------------
                        Private function prototypes
 -----------------------------------------------------------------------------*/
static
void sph_poly_add_point_( sph_polygon* newpol,
                          const sph_point* p) CPL_ATTR_NONNULL;

static
sph_polygon* sph_polygon_new_(size_t nsize) CPL_ATTR_ALLOC;

static void sph_polygon_grow_(sph_polygon*, size_t) CPL_ATTR_NONNULL;

static
void sph_polygon_copy_(sph_polygon* self, const sph_polygon* other)
    CPL_ATTR_NONNULL;
 

/*-----------------------------------------------------------------------------
                              Function definitions
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief Create a new empty polygon

 @param nothing

 @return the pointer to new polygon.

 Description: create a new empty polygon.

 */
/*----------------------------------------------------------------------------*/
sph_polygon* sph_polygon_new( void ) {

    return sph_polygon_new_( SPH_POLYGON_INISIZE );

}
/*----------------------------------------------------------------------------*/
/**
 @brief Create a new duplicate polygon

 @param self The polygon to copy

 @return the pointer to new polygon.

 Description: create a new duplicate polygon.

 */
/*----------------------------------------------------------------------------*/
sph_polygon* sph_polygon_duplicate( sph_polygon* self ) {
    sph_polygon*        result      = NULL;

    if ( self == NULL ) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_NULL_INPUT,"Null input poly");
    } else {
        result = self->npoints
            ? sph_polygon_new_( 2 * self->npoints )
            : sph_polygon_new();
        sph_polygon_copy_(result, self);
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the size of the polygon
 @param self        The polygon
 @return The number of points in the polygon or negative on error

 */
/*----------------------------------------------------------------------------*/
cpl_size sph_polygon_get_size(const sph_polygon* self)
{
    if (self == NULL) {
        (void)cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
        return -1;
    }

    return self->npoints;
}


/*----------------------------------------------------------------------------*/
/**
 @brief Add a new point to an existing polygon

 @param self        the polygon to add the point to
 @param x            the x-coordinate of point to add
 @param y            the y-coordinate of point to add

 @return error code of the operation.

 Description: Adds a new point to the polygon.

 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_polygon_add_point( sph_polygon* self, double x, double y ) {

    const sph_point p = {x, y};

    sph_poly_add_point_( self, &p );

    return CPL_ERROR_NONE;

}

double sph_polygon_get_top(const sph_polygon* self) {
    double result = 0.0;

    if (self != NULL && self->npoints > 0) {
        result = self->points[0].y;

        for (size_t i = 1; i < (size_t)self->npoints; i++) {
            SPH_INCREASE(result, self->points[i].y);
        }
    }
    return result;
}

double sph_polygon_get_bottom(const sph_polygon* self) {
    double result = 0.0;

    if (self != NULL && self->npoints > 0) {
        result = self->points[0].y;

        for (size_t i = 1; i < (size_t)self->npoints; i++) {
            SPH_REDUCE(result, self->points[i].y);
        }
    }
    return result;
}

double sph_polygon_get_left(const sph_polygon* self) {
    double result = 0.0;

    if (self != NULL && self->npoints > 0) {
        result = self->points[0].x;

        for (size_t i = 1; i < (size_t)self->npoints; i++) {
            SPH_REDUCE(result, self->points[i].x);
        }
    }
    return result;
}

double sph_polygon_get_right(const sph_polygon* self) {
    double result = 0.0;

    if (self != NULL && self->npoints > 0) {
        result = self->points[0].x;

        for (size_t i = 1; i < (size_t)self->npoints; i++) {
            SPH_INCREASE(result, self->points[i].x);
        }
    }
    return result;
}

inline void sph_polygon_get_extent(const sph_polygon* self,
                                   double* ptop,
                                   double* pbottom,
                                   double* pleft,
                                   double* pright)
{
    if (self->npoints > 1) {
        const size_t j = self->points[0].y > self->points[1].y ? 0 : 1;
        const size_t k = self->points[0].x > self->points[1].x ? 0 : 1;

        *ptop    = self->points[j    ].y;
        *pbottom = self->points[1 - j].y;

        *pright  = self->points[k    ].x;
        *pleft   = self->points[1 - k].x;

        for (size_t i = 2; i < (size_t)self->npoints; i++) {
            SPH_REDUCE  (*pleft,  self->points[i].x);
            else
            SPH_INCREASE(*pright, self->points[i].x);

            SPH_INCREASE(*ptop,    self->points[i].y);
            else
            SPH_REDUCE  (*pbottom, self->points[i].y);
        }
    } else if (self->npoints == 1) {
        *ptop  = *pbottom = self->points[0].y;
        *pleft = *pright  = self->points[0].x;
    }
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the coordinate of the geometric centre.

 @param self        the polygon
 @param pmid        The centre coordinate, or undefined on error

 @return CPL_ERROR_NONE on succes or the relevant CPL error
 @note If the polygon is empty, the point is unmodified

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_polygon_get_mid_xy(sph_point * pmid,
                                      const sph_polygon* self)
{
    if (self == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (pmid == NULL) {
        return cpl_error_set(cpl_func, CPL_ERROR_NULL_INPUT);
    } else if (self->npoints > 0) {
        double x = 0.0;
        double y = 0.0;
        size_t ii;

        for ( ii = 0;  ii < self->npoints; ii++) {
            x += self->points[ii].x;
            y += self->points[ii].y;
        }

        pmid->x = x / (double)self->npoints;
        pmid->y = y / (double)self->npoints;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the centre x coordinate

 @param self        the polygon

 @return the x coordinate of the geometric centre

 Description: returns the x coordinate of the geometric centre.

 */
/*----------------------------------------------------------------------------*/
double sph_polygon_get_midx(const sph_polygon* self) {
    double result = 0.0;

    if (self != NULL && self->npoints > 0) {
        for (size_t i = 0; i < (size_t)self->npoints; i++) {
            result += self->points[i].x;
        }
        result /= (double)self->npoints;
    }
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the centre y coordinate

 @param self        the polygon

 @return the y coordinate of the geometric centre

 Description: returns the y coordinate of the geometric centre.

 */
/*----------------------------------------------------------------------------*/
double sph_polygon_get_midy(const sph_polygon* self) {
    double result = 0.0;

    if (self != NULL && self->npoints > 0) {
        for (size_t i = 0; i < (size_t)self->npoints; i++) {
            result += self->points[i].y;
        }
        result /= (double)self->npoints;
    }
    return result;
}


/*----------------------------------------------------------------------------*/
/**
 @brief Return the geometric centre (x,y) coordinate
 @param self   The polygon
 @param px     The X-coordinate
 @param py     The Y-coordinate

 */
/*----------------------------------------------------------------------------*/
void sph_polygon_get_midxy_(const sph_polygon* self, double* px, double* py)
{
    if (self->npoints > 0) {
        *px = 0.0;
        *py = 0.0;
        for (size_t i = 0; i < (size_t)self->npoints; i++) {
            *px += self->points[i].x;
            *py += self->points[i].y;
        }
        *px /= (double)self->npoints;
        *py /= (double)self->npoints;
    }
}


/*----------------------------------------------------------------------------*/
/**
 @brief enlarge the polygon by a factor

 @param self        the polygon
 @param    factor        the factor to enlarge by (lower 1 means shrink)

 @return error code

 Description: enlarge the polygon by a factor. If the factor is less than 1, it
 corresponds to a shrink. The factor must be larger 0.
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_polygon_enlarge( sph_polygon* self, double fac ) {
    double x            = 0.0;
    double y            = 0.0;
    int        ii                = 0;
    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    if ( fac <= DBL_MIN ) {
        SPH_NULL_ERROR
        return CPL_ERROR_NULL_INPUT;
    }
    for ( ii = 0;  ii < self->npoints; ++ ii) {
        x += self->points[ii].x;
        y += self->points[ii].y;
    }
    if ( self->npoints > 0 ) {
        x /= self->npoints;
        y /= self->npoints;
    }
    for ( ii = 0;  ii < self->npoints; ++ ii) {
        self->points[ii].x = x + fac * ( self->points[ii].x - x );
        self->points[ii].y = y + fac * ( self->points[ii].y - y );
    }
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return the edge of a polygon as sph_line

 @param self        the polygon the edge belongs to
 @param edge        the id of the edge (1 for first, see below)
 @return the edge as a new sph_line
 @note For performance reasons, there is no checking for bounds, nor for
        NULL-input.

 Description: returns the edge of a polygon as a new sph_line
 (the pointer to the sph_line is returned). The edge is identified
 by an integer id, between 1 and the number of edges of the polygon.
 The first edge is the one between points 0 and point 1. The last
 edge is the one between the past point and point 0.

 */
/*----------------------------------------------------------------------------*/
sph_line* sph_polygon_get_edge(const sph_polygon* self, int edge) {

    assert(0    <  edge);
    assert(edge <= self->npoints);

    return edge < self->npoints
        ? sph_line_new(self->points[ edge - 1 ].x,
                       self->points[ edge - 1 ].y,
                       self->points[ edge     ].x,
                       self->points[ edge     ].y)
        : sph_line_new(self->points[ edge - 1 ].x,
                       self->points[ edge - 1 ].y,
                       self->points[ 0        ].x,
                       self->points[ 0        ].y);

}
/*----------------------------------------------------------------------------*/
/**
 @brief Clip a polygon to a target polygon.

 @param self      The polygon to clip (may be second poly of work space)
 @param target    The polygon to clip to
 @param workspace The work space
 @return A const pointer to the clipped polygon (from the work space)

 Description: This clips the polygon self, so that it afterward
 corresponds to an area that is the union between the area of self and
 the target polygon.

 */
/*----------------------------------------------------------------------------*/
const sph_polygon* sph_polygon_clip_to_target(const sph_polygon* self,
                                              const sph_polygon* target,
                                              sph_polygon_workspace* workspace)
{
    const sph_polygon* oldpol = self;
    sph_point*         intersect;
    sph_line*          sourceedge;

    cpl_ensure(self      != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(target    != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(workspace != NULL, CPL_ERROR_NULL_INPUT, NULL);

    intersect  = workspace->point1;
    sourceedge = workspace->line;

    for (int ee = 1; ee <= target->npoints; ++ee ) {
        sph_polygon*   newpol = (ee & 1) ? workspace->poly1 : workspace->poly2;
        const sph_line targetedge = {
            target->points + (ee - 1),
            target->points + (ee < target->npoints ? ee : 0)};

        /* FIXME: self & poly2 may be aliased (check relevant on 1st ite) */
        assert(oldpol != newpol);

        newpol->npoints = 0;

        for (int ii = 0; ii < oldpol->npoints; ++ii ) {
            const int nextp = ii + 1 == oldpol->npoints ? 0 : ii + 1;

            if (sph_polygon_same_side( target, ee, oldpol->points + ii)) {
                if (sph_polygon_same_side(target, ee, oldpol->points + nextp)) {
                    sph_poly_add_point_( newpol, oldpol->points + nextp);
                } else {
                    sph_point_copy(sourceedge->p1, oldpol->points + ii);
                    sph_point_copy(sourceedge->p2, oldpol->points + nextp);

                    sph_line_intersect_point(intersect, &targetedge,
                                             sourceedge );

                    sph_poly_add_point_( newpol, intersect);
                }
            } else if (sph_polygon_same_side(target, ee,
                                             oldpol->points + nextp)) {
                sph_point_copy(sourceedge->p1, oldpol->points + ii);
                sph_point_copy(sourceedge->p2, oldpol->points + nextp);

                sph_line_intersect_point(intersect, &targetedge,
                                         sourceedge );
  
                sph_poly_add_point_( newpol, intersect);
                sph_poly_add_point_( newpol, oldpol->points + nextp);
            }
        }
        oldpol = newpol;
    }
    return oldpol;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Calculate the overlap are of the polygon with another.

 @param self     The first polygon
 @param target   The second polygon to overlap.
 @param ws       Workspace with temporary storage

 @return area of overlap or negative on error

 Description: The calculates the area of overlap.

 */
/*----------------------------------------------------------------------------*/
double sph_polygon_calculate_overlap(const sph_polygon* self,
                                     const sph_polygon* target,
                                     sph_polygon_workspace* ws )
{
    double result = 0.0;
    const sph_polygon* clipped = sph_polygon_clip_to_target(self, target, ws);

    if (clipped == NULL) {
        (void)cpl_error_set_where(cpl_func);
    } else {
        result = sph_polygon_area(clipped);
    }

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 @brief NOT IMPLEMENTED YET!

 @param

 @return

 Description:

 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_polygon_sort( sph_polygon* self ) {
    if ( !self ) {
        SPH_NULL_ERROR;
        return CPL_ERROR_NULL_INPUT;
    }
    return CPL_ERROR_UNSPECIFIED;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Test if a point is on the same side of edge as all polygon points

 @param self  The polygon the edge belongs to
 @param ee    The id of the edge (1 for first, see sph_polygon_get_edge)
 @param p     The (x,y)-coord. of point to test.
 @return 1 if point is on same side, 0 if not

 Description: This function tests of a point is on the same side of an edge
 as all polygon points of the polygon self.

 */
/*----------------------------------------------------------------------------*/
cpl_boolean sph_polygon_same_side(const sph_polygon* self,
                                 int ee,
                                 const sph_point* p)
{
    if (ee + 1 < self->npoints) {
        return sph_point_same_side_(&(self->points[ee - 1]),
                                    &(self->points[ee    ]),
                                    &(self->points[ee + 1]),
                                    p);
    } else if (ee + 1 == self->npoints) {
        return sph_point_same_side_(&(self->points[ee - 1]),
                                    &(self->points[ee    ]),
                                    &(self->points[0     ]),
                                    p);
    }

    assert(ee == self->npoints);

    return sph_point_same_side_(&(self->points[ee - 1]),
                                &(self->points[0     ]),
                                &(self->points[1     ]),
                                p);
}

/*----------------------------------------------------------------------------*/
/**
 @brief Test if a point is inside a polygon

 @param self         the polygon
 @param x            the x-coordinate of point to test
 @param y            the y-coordinate of point to test

 @return 0 if not inside, 1 if inside, -1 on error.

 Description: This function tests if a point is inside the
 given polygon. If an error occures (e.g. the polygon has 2 or less points)
 -1 is returned.

 */
/*----------------------------------------------------------------------------*/
short sph_polygon_test_inside(const sph_polygon* self, double x, double y)
{
    double              minx        = 0.0;
    double              miny        = 0.0;
    sph_line*           edge        = NULL;
    sph_line*           toout       = NULL;
    int                 ii          = 0;
    int                 intersectc  = 0;

    if ( !self ) {
        return -1;
    }
    if ( self->npoints < 3 ) {
        return -1;
    }
    minx         = self->points[ 0 ].x;
    miny         = self->points[ 0 ].y;
    for ( ii = 1; ii < self->npoints; ++ii ) {
        if ( self->points[ii].x < minx ) {
            minx = self->points[ii].x;
        }
        if ( self->points[ii].y < miny ) {
            miny = self->points[ii].y;
        }
    }

    toout = sph_line_new( x , y , minx - 0.1, miny - 0.1 );
    for ( ii = 1; ii <= self->npoints; ++ii ) {
        edge = sph_polygon_get_edge( self, ii );
        if ( sph_line_intersect_test( edge, toout) ) {
            intersectc++;
        }
        sph_line_delete( edge );
    }
    sph_line_delete( toout );
    if ( intersectc % 2 ) {
        return 1;
    }
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Shift a polygon.

 @param self        the polygon to shift
 @param dx            the x amount to shift by
 @param dy            the y amount to shift by

 @return error code of the operation

 Description: Shift a polygon.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_polygon_shift( sph_polygon* self,
                   double dx, double dy )
{
    int                 ii          = 0;
    sph_error_code      rerr        = CPL_ERROR_NONE;

    if ( !self ) {
        return CPL_ERROR_NULL_INPUT;
    }

    for ( ii = 0; ii < self->npoints; ++ii ) {
        rerr |= sph_point_shift( &(self->points[ii]), dx, dy );
    }
    return rerr;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Rotate a polygon around a point and shift afterwards.

 @param self        the polygon to rotate and shift
 @param cosangle    the cosine of the angle to rotate by
 @param    sinangle    the sine of the angle to rotate by
 @param dx            the x amount to shift by
 @param dy            the y amount to shift by

 @return error code of the operation

 Description: Rotate and afterwards shift a polygon. The rotation is
 specified by the sine and cosine of the rotation angle (measured counter
 clockwise). This input format is for performance reasons.
 The shift is performed after the rotation.

 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_polygon_rotate_around( sph_polygon* self,
                                          double x, double y,
                                          double cosangle,
                                          double sinangle,
                                          double dx, double dy )
{
    int                 ii          = 0;
    sph_error_code      rerr        = CPL_ERROR_NONE;

    if ( !self ) {
        return CPL_ERROR_NULL_INPUT;
    }

    for ( ii = 0; ii < self->npoints; ++ii ) {
        rerr |= sph_point_rotate_around( &(self->points[ii]),
                                 x, y,
                                 cosangle,
                                 sinangle,
                                 dx, dy );
    }
    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Return a point(vertex) of the polygon

 @param  self     the polygon
 @param  point  the point (vertext number) to return

 @return pointer to the vertex or NULL if vertex does not exist.

 Description: This returns a pointer to a vertex.
 */
/*----------------------------------------------------------------------------*/
sph_point*
sph_polygon_get_point( sph_polygon* self, int p ) {
    if ( !self ) return NULL;
    if ( p<0 || p>=self->npoints ) return NULL;
    return &(self->points[p]);
}

/*----------------------------------------------------------------------------*/
/**
 @brief Calculates the area of a clockwise or counter clockwise sorted polygon
 @param  self     The polygon
 @return Area of the polygon (zero for less than three points)
 @note NULL input is not supported!

 */
/*----------------------------------------------------------------------------*/
double sph_polygon_area(const sph_polygon* self) {

    double area = 0.0;

    switch (self->npoints) {
    default:
        for (size_t i = 4; i < (size_t)self->npoints - 1; i++) {
            area += (self->points[i].x + self->points[i+1].x)
                *   (self->points[i].y - self->points[i+1].y);
        }
    case 5:
        area += (self->points[3].x + self->points[4].x)
            *   (self->points[3].y - self->points[4].y);
    case 4:
        area += (self->points[2].x + self->points[3].x)
            *   (self->points[2].y - self->points[3].y);
    case 3:
        area += (self->points[self->npoints - 1].x + self->points[0].x)
            *   (self->points[self->npoints - 1].y - self->points[0].y)
            +   (self->points[0].x + self->points[1].x)
            *   (self->points[0].y - self->points[1].y)
            +   (self->points[1].x + self->points[2].x)
            *   (self->points[1].y - self->points[2].y);

        area = 0.5 * fabs(area);

    case 2: /* Less than 3 points => zero area */
    case 1:
    case 0:
        break;
    }

    return area;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Delete the polygon and free space

 @param self        the polygon to delete

 @return nothing

 Description: Does just that.

 */
/*----------------------------------------------------------------------------*/
void sph_polygon_delete( sph_polygon* self ) {

    if ( self != NULL ) {
        cpl_free(self->points);
        cpl_free(self);
    }
}

 /*----------------------------------------------------------------------------*/
 /**
 @brief Free all space allocated for this module
 */
/*----------------------------------------------------------------------------*/
void sph_polygon_free_all( void ) {
}
/*----------------------------------------------------------------------------*/
/**
 @brief Free the workspace
 */
/*----------------------------------------------------------------------------*/
void sph_polygon_workspace_delete( sph_polygon_workspace* self ) {
    if ( self ) {
        sph_point_delete(self->point1);
        sph_point_delete(self->point2);
        sph_point_delete(self->point3);
        sph_line_delete(self->line);
        sph_polygon_delete(self->poly1);
        sph_polygon_delete(self->poly2);
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Allocate a new polygon workspace for use in the clip function
 * @return  pointer to new sph_polygon_workspace or NULL
 */
/*----------------------------------------------------------------------------*/
sph_polygon_workspace*
sph_polygon_workspace_new( void ) {
    sph_polygon_workspace*      self = NULL;

    self = (sph_polygon_workspace*)cpl_malloc(sizeof( sph_polygon_workspace ));
    self->line   = sph_line_new(0.0,0.0,0.0,0.0);
    self->point1 = sph_point_new(0.0,0.0);
    self->point2 = sph_point_new(0.0,0.0);
    self->point3 = sph_point_new(0.0,0.0);
    self->poly1  = sph_polygon_new();
    self->poly2  = sph_polygon_new();

    return self;
}


/**@}*/
/*----------------------------------------------------------------------------*/
//          STATIC FUNCTIONS
/*----------------------------------------------------------------------------*/
static
void
sph_poly_add_point_( sph_polygon* self,
                     const sph_point* p)
{
    if (self->npoints == self->nalloc) {
        sph_polygon_grow_( self, self->nalloc );
    }

    sph_point_copy(self->points + self->npoints, p);

    self->npoints++;
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Create a new empty polygon with a pre-reserved size

 @param nsize Pre-reserved size
 @return the pointer to new polygon.

 Description: create a new empty polygon.

 */
/*----------------------------------------------------------------------------*/
static sph_polygon* sph_polygon_new_( size_t nsize ) {
    sph_polygon* self = (sph_polygon*)cpl_malloc( sizeof( *self ) );

    self->nalloc  = nsize; /* Reserve space for this many */
    self->points  = (sph_point*)cpl_malloc(nsize * sizeof(*self->points));
    self->npoints = 0; /* Initially empty */

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Copy the polygon from one to another, space allowing
 @param self  The polygon to copy from
 @param other The polygon to copy to
 @return void
 @note The copying of points stops when the number of source points have been
   copied, or when the size of the destination has been reached, whichever comes
   first

 */
/*----------------------------------------------------------------------------*/
static
void sph_polygon_copy_(sph_polygon* self, const sph_polygon* other) {

    const size_t copysize = other->npoints <= self->nalloc
        ? other->npoints : self->nalloc;

    if (copysize > 0) {
        (void)memcpy(self->points, other->points,
                     copysize * sizeof(*other->points));
    }
    self->npoints = copysize;
}
 

/*----------------------------------------------------------------------------*/
/**
 @internal
 @brief Grow the pre-reserved size
 @param self    The polygon to grow
 @param addsize The size to grow with, a (good) fraction of the current size
 @return void

 */
/*----------------------------------------------------------------------------*/
static void sph_polygon_grow_( sph_polygon* self, size_t addsize ) {

    self->nalloc += addsize ? addsize : 1; /* Increase the size */

    if (self->npoints > 0) {
        /* Copy the points in use */
        void * old = (void*)self->points;
        void * new = cpl_malloc(self->nalloc * sizeof(*self->points));

        self->points = (sph_point*)memcpy(new, old, self->npoints *
                                          sizeof(*self->points));
        cpl_free(old);
    } else {
        cpl_free(self->points);
        self->points = (sph_point*)cpl_malloc(sizeof(*self->points) *
                                              self->nalloc);
    }
}
