/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include <strings.h>
#include "sph_quad_image.h"
#include "sph_double_image.h"
#include "sph_master_frame.h"
#include "sph_zpl_exposure.h"
#include "sph_smart_imagelist.h"
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_utils.h"
#include "sph_version.h"

/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/
sph_error_code SPH_QUAD_IMAGE_GENERAL         = SPH_QUAD_IMAGE_ERR_START + 0;
sph_error_code SPH_QUAD_IMAGE_BAD_ALGORITHM   = SPH_QUAD_IMAGE_ERR_START + 1;
sph_error_code SPH_QUAD_IMAGE_NO_ALGORITHM    = SPH_QUAD_IMAGE_ERR_START + 2;
sph_error_code SPH_QUAD_IMAGE_BAD_TYPE        = SPH_QUAD_IMAGE_ERR_START + 3;
sph_error_code SPH_QUAD_IMAGE_NO_TYPE         = SPH_QUAD_IMAGE_ERR_START + 4;
sph_error_code SPH_QUAD_IMAGE_BAD_IMAGE       = SPH_QUAD_IMAGE_ERR_START + 5;


/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_QUAD_image  A structure to store QUAD image products
 *
 * This structure represents a data product with two values for each pixel.
 * It is mainly used to store "reduced" polarisation in the case of the phase-
 * dependency. It consists of the two double image, i.e. four master frames
 * with the following extensions for each:
 * @par Descirption:
 * <ol>
 * <li> RMSMAP. A map of the same dimension and number of pixels as the
 *      image giving the inverse of the RMS of the image pixel</li>
 * <li> NCombmap. A map of the same dimension and number of pixels as the
 *      image giving the number of pixels that were used to obtain the image</li>
 * <li> Badpixelmap. A map of the same dimension and number of pixels as the
 *      image flagging pixels that were impossible to reduce from the raw frames
 *      either because all input raw pixels at that position were bad or because
 *      the statistics is not calculable.</li>
 * <li> Quality Control Parameter list. A list of quality control parameters.
 *      What quality parameters are listed depends on the recipe used to create
 *      the frame.
 * </ol>
 * Each of these extensions appear twice now, one for each of the two images.
 *
 *
 * @par Synopsis:
 * @code
 * typedef struct _sph_quad_image_
 * {
 *    sph_quad_image*     zero_di;
 *    sph_quad_image*        pi_di;
 * } sph_quad_image;
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_quad_image.
  @return   a pointer to the newly created sph_quad_image or NULL if
              unsuccessful.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_quad_image
 */
/*----------------------------------------------------------------------------*/
sph_quad_image* sph_quad_image_new_empty(void) {
    sph_quad_image* spmi = cpl_calloc( 1, sizeof(sph_quad_image) );

    spmi->properties = NULL;
    spmi->qclist = NULL;
    spmi->zero_image = sph_double_image_new_empty();
    spmi->pi_image = sph_double_image_new_empty();

    return spmi;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_quad_image.
  @return   a pointer to the newly created sph_quad_image or NULL if
              unsuccessful.
  @param    zplexp     a pointer to the sph_zpl_exposure

  @note A CPL error is raised in case there was a problem.

  Construct a new sph_quad_image
 */
/*----------------------------------------------------------------------------*/
sph_quad_image*
sph_quad_image_new_from_zpl_exposures( const sph_zpl_exposure* zplexp )
{
	sph_quad_image*		quadimage = NULL;

	cpl_ensure(zplexp, CPL_ERROR_NULL_INPUT, NULL);

	quadimage = sph_quad_image_new_empty();

   	//copy properties
   	quadimage->properties = cpl_propertylist_duplicate(zplexp->properties);

   	//zero image

   	// zero- image: iframe
    quadimage->zero_image->iframe = sph_master_frame_new_empty();
    quadimage->zero_image->iframe->image = cpl_image_duplicate(zplexp->image_zero_odd);
    if ( quadimage->zero_image->iframe->image ){
              quadimage->zero_image->iframe->badpixelmap = cpl_image_new( cpl_image_get_size_x(quadimage->zero_image->iframe->image),
                              cpl_image_get_size_y( quadimage->zero_image->iframe->image ), CPL_TYPE_INT );
              quadimage->zero_image->iframe->ncombmap = cpl_image_new( cpl_image_get_size_x(quadimage->zero_image->iframe->image),
                              cpl_image_get_size_y( quadimage->zero_image->iframe->image ), CPL_TYPE_DOUBLE );
              quadimage->zero_image->iframe->rmsmap = cpl_image_new( cpl_image_get_size_x(quadimage->zero_image->iframe->image),
                              cpl_image_get_size_y( quadimage->zero_image->iframe->image ), CPL_TYPE_DOUBLE );
    }

    // zero- image: pframe
    quadimage->zero_image->pframe = sph_master_frame_new_empty();
    quadimage->zero_image->pframe->image = cpl_image_duplicate(zplexp->image_zero_even);

    if ( quadimage->zero_image->pframe->image ){
            quadimage->zero_image->pframe->badpixelmap = cpl_image_new( cpl_image_get_size_x( quadimage->zero_image->pframe->image ),
                             cpl_image_get_size_y( quadimage->zero_image->pframe->image ), CPL_TYPE_INT );
            quadimage->zero_image->pframe->ncombmap = cpl_image_new( cpl_image_get_size_x( quadimage->zero_image->pframe->image ),
                              cpl_image_get_size_y( quadimage->zero_image->pframe->image ), CPL_TYPE_DOUBLE );
            quadimage->zero_image->pframe->rmsmap = cpl_image_new( cpl_image_get_size_x( quadimage->zero_image->pframe->image ),
                            cpl_image_get_size_y( quadimage->zero_image->pframe->image ), CPL_TYPE_DOUBLE );
    }


    // pi- image: iframe
    quadimage->pi_image->iframe = sph_master_frame_new_empty();
    quadimage->pi_image->iframe->image = cpl_image_duplicate(zplexp->image_pi_odd );

    if ( quadimage->pi_image->iframe->image ){
            quadimage->pi_image->iframe->badpixelmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->iframe->image ),
                             cpl_image_get_size_y( quadimage->pi_image->iframe->image ), CPL_TYPE_INT );
            quadimage->pi_image->iframe->ncombmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->iframe->image ),
                              cpl_image_get_size_y( quadimage->pi_image->iframe->image ), CPL_TYPE_DOUBLE );
            quadimage->pi_image->iframe->rmsmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->iframe->image ),
                            cpl_image_get_size_y( quadimage->pi_image->iframe->image ), CPL_TYPE_DOUBLE );
    }

    // pi- image pframe
    quadimage->pi_image->pframe = sph_master_frame_new_empty();
    quadimage->pi_image->pframe->image = cpl_image_duplicate( zplexp->image_pi_even );

    if ( quadimage->pi_image->pframe->image ){
            quadimage->pi_image->pframe->badpixelmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->pframe->image ),
                             cpl_image_get_size_y( quadimage->pi_image->pframe->image ), CPL_TYPE_INT );
            quadimage->pi_image->pframe->ncombmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->pframe->image ),
                              cpl_image_get_size_y( quadimage->pi_image->pframe->image ), CPL_TYPE_DOUBLE );
            quadimage->pi_image->pframe->rmsmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->pframe->image ),
                            cpl_image_get_size_y( quadimage->pi_image->pframe->image ), CPL_TYPE_DOUBLE );
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    return quadimage;
}



/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_quad_image.
  @return   a pointer to the newly created sph_quad_image or NULL if
              unsuccessful.
  @param    di_zero        a pointer to the first(zero- phase) double image
  @param    di_pi        a pointer to the second(pi- phase) double image

  @note Input master frames are not duplicated, so don't delete them
  @note A CPL error is raised in case there was a problem.

  Construct a new sph_quad_image
 */
/*----------------------------------------------------------------------------*/
sph_quad_image* sph_quad_image_new_from_double_images(sph_double_image* di_zero,
                                                      sph_double_image* di_pi)
{
    sph_quad_image* spmi = NULL;

    cpl_ensure(di_zero != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(di_pi   != NULL, CPL_ERROR_NULL_INPUT, NULL);

    spmi = cpl_calloc( 1, sizeof(sph_quad_image) );

    spmi->qclist = NULL;
    spmi->properties = NULL;
    spmi->zero_image =  di_zero;
    spmi->pi_image =  di_pi;

    return spmi;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_quad_image from given master frames
  @return   a pointer to the newly created sph_quad_image or NULL if
              unsuccessful.
  @param    mf_zero_odd        a pointer to the zero-phase-odd master frame
  @param    mf_zero_even    a pointer to the zero-phase-even master frame
  @param    mf_pi_odd        a pointer to the pi-phase-odd master frame
  @param    mf_pi_even        a pointer to the pi-phase-even master frame

  @note Input master frames are not duplicated, so don't delete them
  @note A CPL error is raised in case there was a problem.

  Construct a new sph_quad_image
 */
/*----------------------------------------------------------------------------*/
sph_quad_image*
sph_quad_image_new_from_master_frames( sph_master_frame* mf_zero_odd,
                                       sph_master_frame* mf_zero_even,
                                       sph_master_frame* mf_pi_odd,
                                       sph_master_frame* mf_pi_even)
{
    sph_quad_image* spmi = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;

    if ( mf_zero_odd == NULL || mf_zero_even == NULL ||
         mf_pi_odd   == NULL || mf_pi_even     == NULL  ){
         SPH_ERR("Could not create quad image: input master frame(s) has(have) null pointer(s).")
         return NULL;
    }

    spmi = sph_quad_image_new_empty();

    if ( spmi == NULL )
    {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not allocate quad image" );
        return NULL;
    }

    spmi->zero_image->iframe =  mf_zero_odd;
    spmi->zero_image->pframe =  mf_zero_even;
    spmi->pi_image->iframe =  mf_pi_odd;
    spmi->pi_image->pframe =  mf_pi_even;

    //properties
    if ( mf_zero_odd->properties && mf_zero_even->properties && mf_pi_odd->properties && mf_pi_even->properties ) {
    		spmi->properties = cpl_propertylist_new();
    		//ZERO ODD
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_ODD_RAW_MEDIANS_AVG,
    				cpl_propertylist_get_double( mf_zero_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_AVG) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_ODD_RAW_MEDIANS_MED,
    				cpl_propertylist_get_double( mf_zero_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MED) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_ODD_RAW_MEDIANS_MIN,
    				cpl_propertylist_get_double( mf_zero_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MIN) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_ODD_RAW_MEDIANS_MAX,
    				cpl_propertylist_get_double( mf_zero_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MAX) );
    		if ( cpl_propertylist_has(mf_zero_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE ))
    			 cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_ODD_RAW_MEDIANS_RANGE,
    				cpl_propertylist_get_double( mf_zero_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE) );

    		//ZERO EVEN
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_EVEN_RAW_MEDIANS_AVG,
    				cpl_propertylist_get_double( mf_zero_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_AVG) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_EVEN_RAW_MEDIANS_MED,
    				cpl_propertylist_get_double( mf_zero_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MED) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_EVEN_RAW_MEDIANS_MIN,
    				cpl_propertylist_get_double( mf_zero_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MIN) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_EVEN_RAW_MEDIANS_MAX,
    				cpl_propertylist_get_double( mf_zero_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MAX) );
    		if ( cpl_propertylist_has(mf_zero_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE  ))
    			 cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_ZERO_EVEN_RAW_MEDIANS_RANGE,
    				cpl_propertylist_get_double( mf_zero_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE) );

    		//PI ODD
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_ODD_RAW_MEDIANS_AVG,
    			cpl_propertylist_get_double( mf_pi_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_AVG) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_ODD_RAW_MEDIANS_MED,
    			cpl_propertylist_get_double( mf_pi_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MED) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_ODD_RAW_MEDIANS_MIN,
    			cpl_propertylist_get_double( mf_pi_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MIN) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_ODD_RAW_MEDIANS_MAX,
    			cpl_propertylist_get_double( mf_pi_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MAX) );
    		if ( cpl_propertylist_has(mf_pi_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE  ))
    			 cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_ODD_RAW_MEDIANS_RANGE,
    			 cpl_propertylist_get_double( mf_pi_odd->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE) );

    		//PI EVEN
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_EVEN_RAW_MEDIANS_AVG,
    				cpl_propertylist_get_double( mf_pi_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_AVG) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_EVEN_RAW_MEDIANS_MED,
    				cpl_propertylist_get_double( mf_pi_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MED) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_EVEN_RAW_MEDIANS_MIN,
    				cpl_propertylist_get_double( mf_pi_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MIN) );
    		cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_EVEN_RAW_MEDIANS_MAX,
    				cpl_propertylist_get_double( mf_pi_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_MAX) );
    		if ( cpl_propertylist_has(mf_pi_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE  ))
    			 cpl_propertylist_update_double( spmi->properties, SPH_COMMON_KEYWORD_QC_PI_EVEN_RAW_MEDIANS_RANGE,
    				cpl_propertylist_get_double( mf_pi_even->properties, SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE) );

    		if ( cpl_error_get_code() != CPL_ERROR_NONE  ){
    			SPH_INFO_MSG("Some keywords from RAW_MEDIANS group were not properly updated! "
    					"Delete quad image propertylist and reset error system.");
    			cpl_propertylist_delete( spmi->properties ); spmi->properties = NULL;
    			cpl_error_reset();
    		}
    }
    if ( spmi->zero_image->iframe == NULL || spmi->zero_image->pframe == NULL ||
         spmi->pi_image->iframe == NULL || spmi->pi_image->pframe == NULL     )
    {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not assign master frames to newly created quad image" );
        cpl_free(spmi);
        spmi = NULL;
    }
    return spmi;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get/Copy of the zero sph_double_image from the given sph_quad_image.

  @param self   the sph_quad_image frame which zero sph_double_image is to
                retrieve from

  @return a pointer to the newly copied zero sph_double_image or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new copied zero sph_double_image as copy of the zero
  sph_double_image  from the sph_quad_image.

 */
/*----------------------------------------------------------------------------*/
sph_double_image*
sph_quad_image_getcopy_zero_double_image ( const sph_quad_image* self) {

    return self && self->zero_image
        ? sph_double_image_duplicate(self->zero_image) : NULL;

}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get a zero sph_double_image from the given sph_quad_image without
            duplication

  @param self   the sph_quad_image which zero sph_double_image is to
                retrieve from

  @return a pointer to the zero sph_double_image or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return a zero sph_double_image from the given sph_quad_image.

 */
/*----------------------------------------------------------------------------*/
sph_double_image* sph_quad_image_get_zero_double_image ( sph_quad_image* self ){

    return self ? self->zero_image : NULL;

}


/*----------------------------------------------------------------------------*/
/**
  @brief    Get/Copy of the PI sph_double_image from the given sph_quad_image.

  @param self   the sph_quad_image which PI sph_double_image is to
                retrieve from

  @return a pointer to the newly copied PI sph_double_image or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Create/return a new copied PI sph_double_image as copy of the zero
  sph_double_image  from the sph_quad_image.
 */
/*----------------------------------------------------------------------------*/
sph_double_image*
sph_quad_image_getcopy_pi_double_image (sph_quad_image* self) {

    return self && self->pi_image
        ? sph_double_image_duplicate(self->pi_image) : NULL;

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Get a PI sph_double_image from the given sph_quad_image without
            duplication

  @param self   the sph_quad_image  which pi sph_double_image is to
                retrieve from

  @return a pointer to the pi sph_double_image or NULL if
            unsuccessfully.

  @note     A CPL error is raised in case there was a problem.

  Return a pi sph_double_image from the given sph_quad_image.

 */
/*----------------------------------------------------------------------------*/
sph_double_image* sph_quad_image_get_pi_double_image ( sph_quad_image* self ){

    return self ? self->pi_image : NULL;

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_quad_image.
  @param    nx        the pixel size in x
  @param    ny         the pixel size in y
  @return   a pointer to the newly created sph_quad_image or NULL if
              unsuccessful.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_quad_image
 */
/*----------------------------------------------------------------------------*/
sph_quad_image* sph_quad_image_new( int nx, int ny ) {
    sph_quad_image* spmi;

    cpl_ensure(nx > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(ny > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);

    spmi = cpl_calloc( 1, sizeof(sph_quad_image) );

    spmi->qclist = NULL;
    spmi->properties = NULL;
    spmi->zero_image = sph_double_image_new(nx, ny);
    spmi->pi_image = sph_double_image_new(nx, ny);

    if (spmi->zero_image == NULL || spmi->pi_image == NULL) {
        (void)cpl_error_set_where(cpl_func);
        cpl_free(spmi);
        spmi = NULL;
    }
    return spmi;
}

/*----------------------------------------------------------------------------*/
/**
   @brief Copy constructor function for sph_quad_image

   @param    self   the quad image to duplicate

   @return   a pointer to the newly created sph_quad_image or NULL if
            unsuccessfully.
   @note     A CPL error is raised in case there was a problem.

  Construct a new sph_quad_image as copy of the input master frame.
 */
/*----------------------------------------------------------------------------*/
sph_quad_image* sph_quad_image_duplicate( const sph_quad_image* self){
    sph_quad_image*  new_quadimage = NULL;

    if (self == NULL) {
        sph_error_raise( SPH_QUAD_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "An initial quad image to copy has a null pointer."
                                         "Nothing to copy." );
        return NULL;
    }

    //create an empty quad image
    //note that empty double images will be created, so the quadimage->zero_image not NULL as well as quadimage->pi_image not NULL!!!
    new_quadimage = sph_quad_image_new_empty();
    if (new_quadimage == NULL) {
           sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_ERROR, "Could not allocate memory for the quad image" );
                return NULL;
    }
    //delete not empty quadimage->zero_image and quadimage->pi_image,
    //they will be newly created again by duplicating double images
    if (new_quadimage->zero_image) {
    	sph_double_image_delete(new_quadimage->zero_image);
    	new_quadimage->zero_image = NULL;
    }
    if (new_quadimage->pi_image) {
    	sph_double_image_delete(new_quadimage->pi_image);
    	new_quadimage->pi_image = NULL;
    }

    //duplicate properties and qclist if exists
    if ( self->properties != NULL ){
        new_quadimage->properties = cpl_propertylist_duplicate( self->properties );
    }
    if ( self->qclist != NULL ){
        new_quadimage->qclist = cpl_propertylist_duplicate( self->qclist );
    }

    //duplicate zero_image and  pi_image as a double image
    if (self->zero_image !=NULL ){
        new_quadimage->zero_image = sph_double_image_duplicate(self->zero_image);
    } else {
        sph_error_raise( SPH_QUAD_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "zero_image (double image) from the initial quad image to copy has a null pointer." );
    }

    if (self->pi_image != NULL){
        new_quadimage->pi_image = sph_double_image_duplicate(self->pi_image);
    } else {
        sph_error_raise( SPH_QUAD_IMAGE_GENERAL, __FILE__,
                                 __func__, __LINE__,
                                 SPH_ERROR_WARNING, "pi_image (double image) from the initial quad image to copy has a null pointer." );
    }

    return new_quadimage;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief        add a quad image from another quad image
 *
 * @param        self        the quad image to add to
 * @param        qiframe        the quad image to add
 *
 * @return        error code of the operation
 *
 * This function adds quad image "qiframe" to self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_quad_image_add_quad_image( sph_quad_image* self,
                                              const sph_quad_image* qiframe ){
    sph_error_code rerr = CPL_ERROR_NONE;

    rerr = sph_double_image_add_double_image( self->zero_image, qiframe->zero_image );
    // Single frames do not have a valid pi_image
    if (sph_utils_is_single_frame(qiframe->properties)) {
      sph_utils_set_single_frame(self->properties);
    }
    if (!sph_utils_is_single_frame(self->properties)) {
        rerr |= sph_double_image_add_double_image( self->pi_image, qiframe->pi_image );
    }

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief        subtract a quad image from another quad image
 *
 * @param        self        the quad image to subtract from
 * @param        qiframe        the quad image to subtract
 *
 * @return        error code of the operation
 *
 * This function subtracts quad image "qiframe" from self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_quad_image_subtract_quad_image( sph_quad_image* self,
                                    const sph_quad_image* qiframe ) {
    sph_error_code rerr = CPL_ERROR_NONE;

    rerr = sph_double_image_subtract_double_image( self->zero_image, qiframe->zero_image );
    // Single frames do not have a valid pi_image
    if (sph_utils_is_single_frame(qiframe->properties)) {
      sph_utils_set_single_frame(self->properties);
    }
    if (!sph_utils_is_single_frame(self->properties)) {
      rerr |= sph_double_image_subtract_double_image( self->pi_image, qiframe->pi_image );
    }

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief        multiply quad image by scalar double value
 *
 * @param        self         the quad image to multiply
 * @param        value        the double value
 *
 * @return        error code of the operation
 *
 * This function multiplies quad image from self by double "value".
 * Self is changed in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code    sph_quad_image_multiply_double( sph_quad_image* self,
                                                  double value) {
    sph_error_code rerr = CPL_ERROR_NONE;

    rerr = sph_double_image_multiply_double( self->zero_image, value );
    // Single frames do not have a valid pi_image
    if (!sph_utils_is_single_frame(self->properties)) {
      rerr |= sph_double_image_multiply_double( self->pi_image, value );
    }

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief        divide a quad image by another quad image
 *
 * @param        self        the quad image to divide
 * @param        qimage      the quad image to divide by
 *
 * @return        error code of the operation
 *
 * This functions divides the quad image self by the "qimage". Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_quad_image_divide_quad_image( sph_quad_image* self,
                                  const sph_quad_image* qimage) {
    sph_error_code rerr = CPL_ERROR_NONE;

    rerr = sph_double_image_divide_double_image( self->zero_image, qimage->zero_image );
    // Single frames do not have a valid pi_image
    if (sph_utils_is_single_frame(qimage->properties)) {
      sph_utils_set_single_frame(self->properties);
    }
    if (!sph_utils_is_single_frame(self->properties)) {
      rerr |= sph_double_image_divide_double_image( self->pi_image, qimage->pi_image );
    }

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief        subtract a quad image from another quad image
 *
 * @param        self        the quad image to divide
 * @param        mframe      the master frame to divide by
 *
 * @return        error code of the operation
 *
 * This functions divides the quad image self by the master frame "mframe".
 * Self is changed in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_quad_image_divide_master_frame( sph_quad_image* self,
                                    const sph_master_frame* mframe) {
    sph_error_code rerr = CPL_ERROR_NONE;

    rerr = sph_master_frame_divide_master_frame( self->zero_image->iframe, mframe ) ||
           sph_master_frame_divide_master_frame( self->zero_image->pframe, mframe );
    // Single frames do not have a valid pi_image
    if (!sph_utils_is_single_frame(self->properties)) {
      rerr |= sph_master_frame_divide_master_frame( self->pi_image->iframe, mframe ) ||
               sph_master_frame_divide_master_frame( self->pi_image->pframe, mframe );
    }

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}




/*----------------------------------------------------------------------------*/
/**
 * @brief        divide quad image by scalar double value
 *
 * @param        self        the quad image to divide by   (dividend)
 * @param        value        the double value (divisor)
 *
 * @return        error code of the operation
 *
 * This function divides quad image from self by double "value".
 * Self is changed in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code    sph_quad_image_divide_double( sph_quad_image* self,
                                                double value) {
    sph_error_code rerr = CPL_ERROR_NONE;

    rerr = sph_double_image_divide_double( self->zero_image, value );
    // Single frames do not have a valid pi_image
    if (!sph_utils_is_single_frame(self->properties)) {
      rerr |= sph_double_image_divide_double( self->pi_image, value );
    }

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Perform a quality control on the quad image
  @param    self        the quad image

  @return   the error code

  @note     A CPL error is raised in case there was a problem.

  Performs a quality control calculation on the quad image. This means, that
  certain statistics are calculated and added to the qclist first to the each
  master frame of the quad image, and then put it into the main qclist of the

  Currently these statistics are: Mean, median and rms of the image.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_quad_image_quality_check( sph_quad_image* self ){
    int            rerr            = CPL_ERROR_NONE;

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    rerr = sph_master_frame_quality_check( self->zero_image->iframe );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by calculating quality parameters for the "
                "qi->zero_image->iframe master frame.");
        return cpl_error_set_where(cpl_func);
    }

    if ( self->qclist == NULL ) {
        self->qclist = cpl_propertylist_new();
    }

    //Mean
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD,
                cpl_propertylist_get_double( self->zero_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_ODD,
                cpl_propertylist_get_double( self->zero_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEAN of QUAD IMAGE ZERO ODD.")
        SPH_RAISE_CPL;
        return cpl_error_set_where(cpl_func);
    }
    //Median
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_ZERO_ODD ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_ZERO_ODD,
                cpl_propertylist_get_double( self->zero_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_ZERO_ODD,
                cpl_propertylist_get_double( self->zero_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC MEDIAN of QUAD IMAGE ZERO ODD.")
        SPH_RAISE_CPL;
        return cpl_error_set_where(cpl_func);
    }
    //RMS
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_ZERO_ODD ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_ZERO_ODD,
                cpl_propertylist_get_double( self->zero_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_ZERO_ODD,
                cpl_propertylist_get_double( self->zero_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC RMS of QUAD IMAGE ZERO ODD.")
        SPH_RAISE_CPL;
        return cpl_error_set_where(cpl_func);
    }

    // ZERO EVEN
    rerr = sph_master_frame_quality_check( self->zero_image->pframe );
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by calculating quality parameters for the "
                "qi->zero_image->pframe master frame.");
        return cpl_error_set_where(cpl_func);
    }
    //Mean
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN,
                cpl_propertylist_get_double( self->zero_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_ZERO_EVEN,
                cpl_propertylist_get_double( self->zero_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC Mean of QUAD IMAGE ZERO EVEN.")
        SPH_RAISE_CPL;
        return cpl_error_set_where(cpl_func);
    }
    //Median
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_ZERO_EVEN ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_ZERO_EVEN,
                cpl_propertylist_get_double( self->zero_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_ZERO_EVEN,
                cpl_propertylist_get_double( self->zero_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC Median of QUAD IMAGE ZERO EVEN.")
        SPH_RAISE_CPL;
        return cpl_error_set_where(cpl_func);
    }
    //RMS
    if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_ZERO_EVEN ) ){
        rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_ZERO_EVEN,
                cpl_propertylist_get_double( self->zero_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    } else {
        rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_ZERO_EVEN,
                cpl_propertylist_get_double( self->zero_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
    }
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC RMS of QUAD IMAGE ZERO EVEN.")
        return cpl_error_set_where(cpl_func);
    }


    // Single frames do not have a valid pi_image
    if (!sph_utils_is_single_frame(self->properties)) {
      //PI ODD
      rerr = sph_master_frame_quality_check( self->pi_image->iframe );
      if ( rerr != CPL_ERROR_NONE ){
          SPH_ERR("cpl error is raised by calculating quality parameters for the "
                  "qi->pi_image->iframe master frame.")
          return cpl_error_set_where(cpl_func);
      }
      //Mean
      if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD ) ){
          rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD,
                  cpl_propertylist_get_double( self->pi_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
      } else {
          rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_ODD,
                  cpl_propertylist_get_double( self->pi_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
      }
      //Median
      if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_PI_ODD ) ){
          rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_PI_ODD,
                  cpl_propertylist_get_double( self->pi_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
      } else {
          rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_PI_ODD,
                  cpl_propertylist_get_double( self->pi_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
      }
      //RMS
      if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_PI_ODD ) ){
          rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_PI_ODD,
                  cpl_propertylist_get_double( self->pi_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
      } else {
          rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_PI_ODD,
                  cpl_propertylist_get_double( self->pi_image->iframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
      }
      if ( rerr != CPL_ERROR_NONE ){
          SPH_ERR("cpl error is raised by appending/updating QC parameters of QUAD IMAGE PI ODD.")
          SPH_RAISE_CPL;
          return cpl_error_set_where(cpl_func);
      }

      // PI EVEN
      rerr = sph_master_frame_quality_check( self->pi_image->pframe );
      if ( rerr != CPL_ERROR_NONE ){
          SPH_ERR("cpl error is raised by calculating quality parameters for the "
                  "qi->pi_image->pframe master frame.");
          return cpl_error_set_where(cpl_func);
      }
      //Mean
      if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN ) ){
          rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN,
                  cpl_propertylist_get_double( self->pi_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
      } else {
          rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEAN_QUADIMAGE_PI_EVEN,
                  cpl_propertylist_get_double( self->pi_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME ) );
      }
      if ( rerr != CPL_ERROR_NONE ){
          SPH_ERR("cpl error is raised by appending/updating QC Mean of QUAD IMAGE PI EVEN.")
          SPH_RAISE_CPL;
          return cpl_error_set_where(cpl_func);
      }
      //Median
      if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_PI_EVEN ) ){
          rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_PI_EVEN,
                  cpl_propertylist_get_double( self->pi_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
      } else {
          rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_MEDIAN_QUADIMAGE_PI_EVEN,
                  cpl_propertylist_get_double( self->pi_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_MEDIANMASTERFRAME ) );
      }
      if ( rerr != CPL_ERROR_NONE ){
          SPH_ERR("cpl error is raised by appending/updating QC Median  of QUAD IMAGE PI EVEN.")
          SPH_RAISE_CPL;
          return cpl_error_set_where(cpl_func);
      }
      //RMS
      if ( cpl_propertylist_has( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_PI_EVEN ) ){
          rerr = cpl_propertylist_update_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_PI_EVEN,
                  cpl_propertylist_get_double( self->pi_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
      } else {
          rerr = cpl_propertylist_append_double( self->qclist, SPH_COMMON_KEYWORD_QC_RMS_QUADIMAGE_PI_EVEN,
                  cpl_propertylist_get_double( self->pi_image->pframe->qclist, SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME ) );
      }
      if ( rerr != CPL_ERROR_NONE ){
          SPH_ERR("cpl error is raised by appending/updating QC RMS of QUAD IMAGE PI EVEN.")
          return cpl_error_set_where(cpl_func);
      }
    }

    //add number of the badpixels QC keywords to the qclist of the quad image
    rerr |= cpl_propertylist_update_long(self->qclist, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_ODD,
             cpl_image_get_flux(self->zero_image->iframe->badpixelmap));
    rerr |= cpl_propertylist_update_long(self->qclist, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_ZERO_EVEN,
             cpl_image_get_flux(self->zero_image->pframe->badpixelmap));
    // Single frames do not have a valid pi_image
    if (!sph_utils_is_single_frame(self->properties)) {
      rerr |= cpl_propertylist_update_long(self->qclist, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_ODD,
               cpl_image_get_flux(self->pi_image->iframe->badpixelmap));
      rerr |= cpl_propertylist_update_long(self->qclist, SPH_COMMON_KEYWORD_QC_NUMBER_BADPIXELS_QUAD_IMAGE_PI_EVEN,
               cpl_image_get_flux(self->pi_image->pframe->badpixelmap));
    }

    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by appending/updating QC keywords of QUAD IMAGE NUMBER BADPIXELS")
        return cpl_error_set_where(cpl_func);
    }

    return CPL_ERROR_NONE;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create the bad pixel maps for the quad image
  @param    self        the quad image
  @param    sigma       the sigma level for masking
  @return   the error code

  @note     A CPL error is raised in case there was a problem.

  - Create (calculate) the bad pixel maps for the given quad image
  This function is based on the associated function of the master frame which
  current algorithm is an algorithm that sets the bad pixels
  to be all those that are further than sigma * the total RMS of the whole
  master frame away from the median of the master frame.
  - QC keyword value are used from the qclist of each master frame not from the
  quad image qclist
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_quad_image_mask_sigma( sph_quad_image* self, double sigma)
{
    int rerr = sph_master_frame_mask_sigma( self->zero_image->iframe, sigma);

    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by creating mask sigma the "
                "qi->zero_image->iframe (zero odd) master frame.");
        SPH_RAISE_CPL;
        return cpl_error_set_where(cpl_func);
    }

    rerr = sph_master_frame_mask_sigma( self->zero_image->pframe, sigma);
    if ( rerr != CPL_ERROR_NONE ){
        SPH_ERR("cpl error is raised by creating mask sigma the "
                "qi->zero_image->pframe (zero even) master frame.");
        SPH_RAISE_CPL;
        return cpl_error_set_where(cpl_func);
    }

    // Single frames do not have a valid pi_image
    if (!sph_utils_is_single_frame(self->properties)) {
      rerr = sph_master_frame_mask_sigma( self->pi_image->iframe, sigma);
      if ( rerr != CPL_ERROR_NONE ){
          SPH_ERR("cpl error is raised by creating mask sigma the "
                  "qi->pi_image->iframe (pi odd) master frame.");
          SPH_RAISE_CPL;
          return cpl_error_set_where(cpl_func);
      }

      rerr = sph_master_frame_mask_sigma( self->pi_image->pframe, sigma);
      if ( rerr != CPL_ERROR_NONE ){
          SPH_ERR("cpl error is raised by creating mask sigma the "
                  "qi->pi_image->pframe (pi even) master frame.");
          SPH_RAISE_CPL;
          return cpl_error_set_where(cpl_func);
      }
    }

    return CPL_ERROR_NONE;
}



/*----------------------------------------------------------------------------*/
/**
 * @brief        Save the quad image
 *
 * @param        self        the quad_image to save
 * @param       czFilename  the filename to save under
 * @param        pli           a propertylist to append to main header (can be NULL)
 * @return        the error code
 *
 * Save the quad_image as a FITS file. This saves the quad_image as a FITS
 * file with 16 extensions, one each for the image, one for the badpixel map, one
 * for the number of pixels map and one for the rmsmap. First come  8 extensions
 * for double image "zero" , then 8 extensions for double image "pi".
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_quad_image_save( const sph_quad_image* self,
                     const char* czFilename,
                     const cpl_propertylist* pli )
{
    sph_error_code            rerr            = CPL_ERROR_NONE;
    cpl_propertylist*        mainpl            = NULL;
    cpl_propertylist*         extpl            = NULL;
    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->zero_image->iframe || !self->zero_image->pframe ||
            !self->pi_image->iframe || !self->pi_image->pframe ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    extpl = cpl_propertylist_new(); // not really used for the moment; it might be used later

    if ( pli ) {
        mainpl = sph_keyword_manager_trans_keywords( pli );
    }
    else {
        mainpl = cpl_propertylist_new();
    }
    if ( cpl_propertylist_has( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE )){
        rerr = cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE );
    } else {
        cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE  );
    }

    //add qclist created to the main property list
    if ( self->qclist ) {
        rerr = cpl_propertylist_append( mainpl, self->qclist );
    }

    if ( rerr != CPL_ERROR_NONE ){
        return rerr;
    }

    /* Update the header if required */
    sph_utils_update_header(mainpl);

    // phase = zero
    cpl_propertylist_update_string(mainpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_IMAGE_EXTNAME);
    rerr |= cpl_image_save( self->zero_image->iframe->image, czFilename, CPL_TYPE_FLOAT, mainpl, CPL_IO_DEFAULT );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_BADPIXMAP_EXTNAME);
    rerr |= cpl_image_save( self->zero_image->iframe->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_NCOMBMAP_EXTNAME);
    rerr |= cpl_image_save( self->zero_image->iframe->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_RMSMAP_EXTNAME);
    rerr |= cpl_image_save( self->zero_image->iframe->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_IMAGE_EXTNAME);
    rerr |= cpl_image_save( self->zero_image->pframe->image, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_BADPIXMAP_EXTNAME);
    rerr |= cpl_image_save( self->zero_image->pframe->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_NCOMBMAP_EXTNAME);
    rerr |= cpl_image_save( self->zero_image->pframe->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_RMSMAP_EXTNAME);
    rerr |= cpl_image_save( self->zero_image->pframe->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

    //phase = pi
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_IMAGE_EXTNAME);
    rerr |= cpl_image_save( self->pi_image->iframe->image, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_BADPIXMAP_EXTNAME);
    rerr |= cpl_image_save( self->pi_image->iframe->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_NCOMBMAP_EXTNAME);
    rerr |= cpl_image_save( self->pi_image->iframe->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_RMSMAP_EXTNAME);
    rerr |= cpl_image_save( self->pi_image->iframe->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_IMAGE_EXTNAME);
    rerr |= cpl_image_save( self->pi_image->pframe->image, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_BADPIXMAP_EXTNAME);
    rerr |= cpl_image_save( self->pi_image->pframe->badpixelmap, czFilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_NCOMBMAP_EXTNAME);
    rerr |= cpl_image_save( self->pi_image->pframe->ncombmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    cpl_propertylist_update_string(extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_RMSMAP_EXTNAME);
    rerr |= cpl_image_save( self->pi_image->pframe->rmsmap, czFilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );


    cpl_propertylist_delete( mainpl );
    cpl_propertylist_delete( extpl );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;

}
#ifdef SPH_QUAD_IMAGE_UPDATE_PROD_TYPE
/*----------------------------------------------------------------------------*/
/**
 * @brief        Update property list of the quad image DFS product
 *              by SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE
 *
 * @param       czFilename  the filename to load and save quad image with
 *                 correctly updated SPH_COMMON_KEYWORD_SPH_TYPE
 * @return        the error code
 *
 * @note This function is used as a final step to save quad image as a
 * DFS product (see function sph_quad_image_save_dfs). It updates
 * SPH_COMMON_KEYWORD_SPH_TYPE by SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE.
 * It is necessary because cpl_dsf_save_image updates the quad image property
 * list by the property list from the first input frames.
 *
 */
/*----------------------------------------------------------------------------*/
static sph_error_code sph_quad_image_update_prod_type( const char* czFilename )
{
    cpl_propertylist* pli        = NULL;
    cpl_propertylist* pli_regexp = NULL;
    const int         plane      = 0;
    sph_quad_image*   quadimage  = sph_quad_image_load(czFilename, plane);
    sph_error_code    rerr;

    if ( !quadimage ){
        SPH_ERR("No qaud image is loaded.")
        return sph_error_get_last_code();
    }

    pli = cpl_propertylist_load_regexp( czFilename, 0, ".*QC.*", 1);
    pli_regexp = cpl_propertylist_new();
    rerr = cpl_propertylist_copy_property_regexp( pli_regexp, pli,
                                                  ".*COMMENT.*", 1 );

    rerr |= sph_quad_image_save( quadimage, czFilename, pli_regexp);
    if (rerr != CPL_ERROR_NONE ) {
        SPH_ERR("Couldn't  save the current qaud image")
    }
    sph_quad_image_delete( quadimage );
    cpl_propertylist_delete( pli );
    cpl_propertylist_delete( pli_regexp );

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;

}
#endif

/*----------------------------------------------------------------------------*/
/**
 * @brief        Save the quad image as DFS product
 *
 * @param        self        the quad image to save
 * @param       czFilename  the filename to save under
 * @param       allframes    the frameset used to construct the product
 * @param       usedframes    the frameset used to construct the product
 * @param       template_frame  the frame to use as template for header
 * @param       param       the parameterlist used for product
 * @param       tag         the tag of the product
 * @param       recipe      the recipe creating this product
 * @param       pipename    the pipeline creating this product
 * @param        pl            a propertylist to append to main header (can be NULL)
 * @return        the error code
 *
 * Save the master frame as a FITS file according to ESO DFS.
 * This saves the master frame as a FITS file with 16 extensions (4 master frames ).
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_quad_image_save_dfs(const sph_quad_image* self,
                        const char* outfilename,
                        cpl_frameset* allframes,
                        const cpl_frame* template_frame,
                        const cpl_parameterlist* params,
                        const char* tag,
                        const char* recipe,
                        const char* pipename,
                        const cpl_propertylist* pli)
{
    sph_error_code            rerr            = CPL_ERROR_NONE;
    cpl_propertylist*        mainpl            = NULL;
    cpl_propertylist*         extpl            = NULL;
    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->zero_image->iframe || !self->zero_image->pframe ||
            !self->pi_image->iframe || !self->pi_image->pframe ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    if ( pli ) {
        SPH_INFO_MSG("Property list given as an external parameter, load it.")
        mainpl = sph_keyword_manager_trans_keywords( pli );
    } else if ( self->properties ) {
    	 SPH_INFO_MSG("Property list already exists , load it.");
         mainpl = sph_keyword_manager_trans_keywords( self->properties );
    } else {
        SPH_INFO_MSG("No property list presents, a new one is created")
        mainpl = cpl_propertylist_new();
    }

    if ( cpl_propertylist_has( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE )){
        SPH_INFO_MSG("Update a keyword  of the product type, set it to the QUAD IMAGE TYPE")
        rerr = cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE );
    } else {
        SPH_INFO_MSG("Create a  keyword  of the product type of the QUAD IMAGE TYPE")
        rerr = cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE  );
    }


    if ( rerr ) {
    	SPH_ERR("Problem is occured by updating the property list of the main header.");
    	cpl_propertylist_delete(mainpl); mainpl = NULL;
    	//cpl_propertylist_delete(extpl); extpl = NULL;
    	return rerr;
    }
    /*
     * Attention: SPH_COMMON_KEYWORD_SPH_TYPE will be changed by cpl_dfs_save_image function.
     * Unfortunately this keyword will be later updated automatically in cpl_dfs_save_image.
     * In order to avoid this  a function, called sph_quad_image_update_prod_type is used
     * as soon as the dfs product is created (see below)
     *
     */

    //add qclist created to the main property list
    if ( self->qclist ) {
        rerr = cpl_propertylist_append( mainpl, self->qclist );
        if ( rerr == CPL_ERROR_NONE ) {
        	SPH_INFO_MSG("QC parameter list added to the property list")
        } else {
        	SPH_ERR("Error occurs adding QC parameters to the property list")
        	cpl_propertylist_delete(mainpl); mainpl = NULL;
        	//cpl_propertylist_delete(extpl); extpl = NULL;
        	return rerr;
        }
    }

    rerr = cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_PRO_CATG, tag);
    if ( rerr == CPL_ERROR_NONE ) {
    	SPH_INFO_MSG("Updated a keyword of the PRO CATG.")
    } else {
    	SPH_ERR("Error occurs updating of the PRO CATG keyword.")
    	cpl_propertylist_delete(mainpl); mainpl = NULL;
    	//cpl_propertylist_delete(extpl); extpl = NULL;
    	return rerr;
    }


    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPNAME, SPH_QC_COMPNAME );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPVERN, SPH_QC_COMPVERN );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPREVN, SPH_QC_COMPREVN );
    cpl_propertylist_append_string( mainpl, SPH_COMMON_KEYWORD_QC_COMPDATE, SPH_QC_COMPDATE );

    cpl_propertylist_update_string( mainpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_IMAGE_EXTNAME);

    //printf( "SPH_COMMON_KEYWORD_SPH_TYPE = %s \n", SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_QUAD_IMAGE);
    sph_utils_remove_wcs_3d(mainpl);
    /* FIXME: Set WCS to dummy (pixel) value for now */
    sph_utils_reset_wcs_12d(mainpl);

    /* Make any final header updates */
    sph_utils_update_header(mainpl);

    rerr = cpl_dfs_save_image( allframes, NULL, params,
            allframes,
            template_frame,
            self->zero_image->iframe->image,
            CPL_TYPE_FLOAT, recipe, mainpl, NULL, pipename, outfilename );

    if ( rerr == CPL_ERROR_NONE ){
        if (extpl == NULL) extpl = cpl_propertylist_new();
        // phase = zero
        //rerr |= cpl_image_save( self->zero_image->iframe->image, outfilename, CPL_TYPE_FLOAT, mainpl, CPL_IO_DEFAULT ); //already saved in cpl_dfs_save_image
        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_BADPIXMAP_EXTNAME);
        rerr |= cpl_image_save( self->zero_image->iframe->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_NCOMBMAP_EXTNAME);
        rerr |= cpl_image_save( self->zero_image->iframe->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_IFRAME_RMSMAP_EXTNAME);
        rerr |= cpl_image_save( self->zero_image->iframe->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_IMAGE_EXTNAME);
        rerr |= cpl_image_save( self->zero_image->pframe->image, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_BADPIXMAP_EXTNAME);
        rerr |= cpl_image_save( self->zero_image->pframe->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_NCOMBMAP_EXTNAME);
        rerr |= cpl_image_save( self->zero_image->pframe->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_ZERO_PFRAME_RMSMAP_EXTNAME);
        rerr |= cpl_image_save( self->zero_image->pframe->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        //phase = pi
        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_IMAGE_EXTNAME);
        rerr |= cpl_image_save( self->pi_image->iframe->image, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_BADPIXMAP_EXTNAME);
        rerr |= cpl_image_save( self->pi_image->iframe->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_NCOMBMAP_EXTNAME);
        rerr |= cpl_image_save( self->pi_image->iframe->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_IFRAME_RMSMAP_EXTNAME);
        rerr |= cpl_image_save( self->pi_image->iframe->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_IMAGE_EXTNAME);
        rerr |= cpl_image_save( self->pi_image->pframe->image, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_BADPIXMAP_EXTNAME);
        rerr |= cpl_image_save( self->pi_image->pframe->badpixelmap, outfilename, CPL_BPP_8_UNSIGNED, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_NCOMBMAP_EXTNAME);
        rerr |= cpl_image_save( self->pi_image->pframe->ncombmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );

        cpl_propertylist_update_string( extpl, SPH_COMMON_KEYWORD_EXTNAME, SPH_COMMON_KEYWORD_VALUE_QUAD_IMAGE_PI_PFRAME_RMSMAP_EXTNAME);
        rerr |= cpl_image_save( self->pi_image->pframe->rmsmap, outfilename, CPL_TYPE_FLOAT, extpl, CPL_IO_EXTEND );
    } else {
    	SPH_ERR("cpl_dfs_save_image returns error.");
    	if ( mainpl ) {
    	    cpl_propertylist_delete(mainpl); mainpl = NULL;
    	}
    	if ( extpl ) {
    	    cpl_propertylist_delete(extpl); extpl = NULL;
    	}
    	return rerr;
    }

    if ( rerr != CPL_ERROR_NONE ) {
    	SPH_ERR("Error occurs by saving extensions of the quad image");
    	cpl_propertylist_delete(mainpl); mainpl = NULL;
    	//cpl_propertylist_delete(extpl); extpl = NULL;
    	return rerr;
    }
    if ( mainpl ) {
        cpl_propertylist_delete( mainpl );mainpl = NULL;
    }
    if ( extpl ) {
        cpl_propertylist_delete( extpl );extpl = NULL;
    }

#ifdef SPH_QUAD_IMAGE_UPDATE_PROD_TYPE
    // Attention: Update of the created dfs quad image product by the
    // correct SPH_COMMON_KEYWORD_SPH_TYPE  keyword
    rerr = sph_quad_image_update_prod_type( outfilename );
    if ( rerr ) {
    	SPH_ERR("sph_quad_image_update_prod_type returns error.");
    	return rerr;
    }
#endif

    return rerr;

}




/*----------------------------------------------------------------------------*/
/**
 * @brief load a quad image from a file
 *
 * @param    filename        the filename to load from
 * @param     plane            the plane to laod from ( 0..Nplanes)
 *
 * This function creates a new quad image and loads the data from a FITS
 * file with at least one plane and at least 16 extensions (the first 16
 * extensions are read). This function does not check the format is actually
 * a sph_quad_image.
 * Possible re-factoring: use load function from the sph_double_image_load
 *
 */
/*----------------------------------------------------------------------------*/
sph_quad_image*
sph_quad_image_load( const char* czFilename, int plane ) {
    sph_quad_image*        result        = NULL;
    cpl_propertylist*    pl_eso        = NULL;

    result = sph_quad_image_new_empty();

    if ( !result ) {
        return NULL;
    }


    pl_eso = cpl_propertylist_load_regexp( czFilename, 0, ".*ESO.*", 0 );

    if ( pl_eso != NULL ){
        sph_error_code        rerr;
        //initializing and loading properties, qclist for the newly created quad image
        result->properties = cpl_propertylist_new();
        rerr = cpl_propertylist_copy_property_regexp( result->properties, pl_eso, ".*QC.*", 1 );
        if ( rerr != CPL_ERROR_NONE){
            sph_error_raise( SPH_QUAD_IMAGE_GENERAL,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_WARNING,  "Copy regexp properties for the quad image is failed" );
            cpl_error_reset();
            cpl_propertylist_delete( result->properties ); result->properties = NULL;
        }
        result->qclist = cpl_propertylist_new();
        rerr = cpl_propertylist_copy_property_regexp( result->qclist, pl_eso, ".*QC.*", 0 );
        if ( rerr != CPL_ERROR_NONE){
            sph_error_raise( SPH_QUAD_IMAGE_GENERAL,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_WARNING,  "Copy regexp qclist for the quad image is failed" );
            cpl_error_reset();
            cpl_propertylist_delete( result->qclist ); result->qclist = NULL;
        }
        cpl_propertylist_delete( pl_eso );
    } else {
        SPH_RAISE_CPL;
        cpl_error_reset();
    }

    result->zero_image->iframe = sph_master_frame_new_empty();
    result->zero_image->pframe = sph_master_frame_new_empty();

    result->pi_image->iframe = sph_master_frame_new_empty();
    result->pi_image->pframe = sph_master_frame_new_empty();


    // zero image: 8 extensions from 0 to 7
    if ( !result->zero_image->iframe || !result->zero_image->pframe ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->zero_image->iframe->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 0 );
    if ( !result->zero_image->iframe->image ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->zero_image->iframe->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 1 );
    if ( !result->zero_image->iframe->badpixelmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->zero_image->iframe->ncombmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 2 );
    if ( !result->zero_image->iframe->ncombmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->zero_image->iframe->rmsmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 3 );
    if ( !result->zero_image->iframe->rmsmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->zero_image->pframe->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 4 );
    if ( !result->zero_image->pframe->image ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->zero_image->pframe->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 5 );
    if ( !result->zero_image->pframe->badpixelmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->zero_image->pframe->ncombmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 6 );
    if ( !result->zero_image->pframe->ncombmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->zero_image->pframe->rmsmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 7 );
    if ( !result->zero_image->pframe->rmsmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }

    // pi image: 8 extensions from 8 to 15
    if ( !result->pi_image->iframe || !result->pi_image->pframe ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->pi_image->iframe->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 8 );
    if ( !result->pi_image->iframe->image ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->pi_image->iframe->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 9 );
    if ( !result->pi_image->iframe->badpixelmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->pi_image->iframe->ncombmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 10 );
    if ( !result->pi_image->iframe->ncombmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->pi_image->iframe->rmsmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 11 );
    if ( !result->pi_image->iframe->rmsmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->pi_image->pframe->image = cpl_image_load( czFilename, CPL_TYPE_DOUBLE, plane, 12 );
    if ( !result->pi_image->pframe->image ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->pi_image->pframe->badpixelmap = cpl_image_load( czFilename, CPL_TYPE_INT, plane, 13 );
    if ( !result->pi_image->pframe->badpixelmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->pi_image->pframe->ncombmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 14 );
    if ( !result->pi_image->pframe->ncombmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }
    result->pi_image->pframe->rmsmap = cpl_image_load( czFilename, CPL_TYPE_FLOAT, plane, 15 );
    if ( !result->pi_image->pframe->rmsmap ) {
        sph_quad_image_delete( result );
        return NULL;
    }


    return result;
}



/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new quad image frameset from cpl_frameset of zpl exp TYPE
 *
 * @param inframe  the input cpl_frameset (of ZPL_EXP_TYPE)
 *
 * @return pointer to the newly created frameset with quad_image or NULL in
 * case of error.
 *
 * This function creates new sph_quad_image frameset from a given cpl_frameset
 * given as a cpl_frameset of type ZPL_EXP_TYPE.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_quad_image_create_quad_image_frameset_from_zplexp_frameset( const cpl_frameset* inframes )
{
    cpl_frameset*        result    = NULL;
    const cpl_frame*     curframe;
    sph_keyword_manager* keym      = NULL;

    cpl_ensure(inframes,CPL_ERROR_NULL_INPUT,NULL);

    keym = sph_keyword_manager_new();
    if ( !keym ) {
        return NULL;
    }

    result = cpl_frameset_new();

    curframe = cpl_frameset_get_first_const( inframes );

    while ( curframe ) {
        cpl_frame*                quadframe    = NULL;
        //cpl_frame*                tmpframe    = NULL;

        cpl_propertylist*        pl            = NULL;
        //cpl_propertylist*        pl_zero_i    = NULL;
        //cpl_propertylist*        pl_zero_p    = NULL;
        //cpl_propertylist*        pl_pi_i        = NULL;
        //cpl_propertylist*        pl_pi_p        = NULL;
        //sph_master_frame*        mframe        = NULL;
        const char*                czFrame        = NULL;
        //char                    czFrameNew[256];
        const char*                czType        = NULL;
        short                    ok            = 0;

        czFrame = cpl_frame_get_filename( curframe );
        //strncpy( czFrameNew, czFrame, 256 );
        //strcat( czFrameNew, ".tmp");
        quadframe = sph_filemanager_create_temp_frame( czFrame, "NONE", CPL_FRAME_GROUP_CALIB );
        //sph_keyword_manager_load_keywords( czFrame ); // this will read the keywords from ext 0 of the file
        //czType = sph_keyword_manager_get_string( SPH_COMMON_KEYWORD_SPH_TYPE ); // get the value of the  SPH_COMMON_KEYWORD_SPH_TYPE - soft. repres.
        pl = sph_keyword_manager_load_properties( czFrame, 0 );
        czType = cpl_propertylist_get_string(pl, SPH_COMMON_KEYWORD_SPH_TYPE);
        ok = 0;

        if ( !strcasecmp( czType, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP ) ) {
            sph_quad_image* quadimage = sph_quad_image_new_empty();

            if ( quadimage ){

                // zero image iframe
                quadimage->zero_image->iframe = sph_master_frame_new_empty();
                if ( quadimage->zero_image->iframe ){
                      quadimage->zero_image->iframe->image = cpl_image_load(czFrame, CPL_TYPE_DOUBLE, 0, 0 );

                    if ( quadimage->zero_image->iframe->image ){
                        quadimage->zero_image->iframe->badpixelmap = cpl_image_new( cpl_image_get_size_x(quadimage->zero_image->iframe->image),
                                         cpl_image_get_size_y( quadimage->zero_image->iframe->image ), CPL_TYPE_INT );
                        quadimage->zero_image->iframe->ncombmap = cpl_image_new( cpl_image_get_size_x(quadimage->zero_image->iframe->image),
                                          cpl_image_get_size_y( quadimage->zero_image->iframe->image ), CPL_TYPE_INT );
                        quadimage->zero_image->iframe->rmsmap = cpl_image_new( cpl_image_get_size_x(quadimage->zero_image->iframe->image),
                                        cpl_image_get_size_y( quadimage->zero_image->iframe->image ), CPL_TYPE_DOUBLE );
                        //pl_zero_i = sph_keyword_manager_load_properties(czFrame, 0); //read property list from ext 0
                        ok = ok + 1;
                    }
                }

                // zero- image pframe
                quadimage->zero_image->pframe = sph_master_frame_new_empty();
                if ( quadimage->zero_image->pframe ){
                      quadimage->zero_image->pframe->image = cpl_image_load(czFrame, CPL_TYPE_DOUBLE, 0, 1 );

                    if ( quadimage->zero_image->pframe->image ){
                        quadimage->zero_image->pframe->badpixelmap = cpl_image_new( cpl_image_get_size_x( quadimage->zero_image->pframe->image ),
                                         cpl_image_get_size_y( quadimage->zero_image->pframe->image ), CPL_TYPE_INT );
                        quadimage->zero_image->pframe->ncombmap = cpl_image_new( cpl_image_get_size_x( quadimage->zero_image->pframe->image ),
                                          cpl_image_get_size_y( quadimage->zero_image->pframe->image ), CPL_TYPE_INT );
                        quadimage->zero_image->pframe->rmsmap = cpl_image_new( cpl_image_get_size_x( quadimage->zero_image->pframe->image ),
                                        cpl_image_get_size_y( quadimage->zero_image->pframe->image ), CPL_TYPE_DOUBLE );
                        //pl_zero_p = sph_keyword_manager_load_properties(czFrame, 0);
                        // read property list from ext 1, but there is no possibility to do in keyword manager
                        ok = ok + 1;
                    }
                }


                // pi- image iframe
                quadimage->pi_image->iframe = sph_master_frame_new_empty();

                if ( quadimage->pi_image->iframe ){
                      quadimage->pi_image->iframe->image = cpl_image_load(czFrame, CPL_TYPE_DOUBLE, 0, 2 );

                    if ( quadimage->pi_image->iframe->image ){
                        quadimage->pi_image->iframe->badpixelmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->iframe->image ),
                                         cpl_image_get_size_y( quadimage->pi_image->iframe->image ), CPL_TYPE_INT );
                        quadimage->pi_image->iframe->ncombmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->iframe->image ),
                                          cpl_image_get_size_y( quadimage->pi_image->iframe->image ), CPL_TYPE_INT );
                        quadimage->pi_image->iframe->rmsmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->iframe->image ),
                                        cpl_image_get_size_y( quadimage->pi_image->iframe->image ), CPL_TYPE_DOUBLE );
                        ok = ok + 1;
                    }
                }

                // pi- image pframe
                quadimage->pi_image->pframe = sph_master_frame_new_empty();
                if ( quadimage->pi_image->pframe ){
                      quadimage->pi_image->pframe->image = cpl_image_load(czFrame, CPL_TYPE_DOUBLE, 0, 3 );

                    if ( quadimage->pi_image->pframe->image ){
                        quadimage->pi_image->pframe->badpixelmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->pframe->image ),
                                         cpl_image_get_size_y( quadimage->pi_image->pframe->image ), CPL_TYPE_INT );
                        quadimage->pi_image->pframe->ncombmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->pframe->image ),
                                          cpl_image_get_size_y( quadimage->pi_image->pframe->image ), CPL_TYPE_INT );
                        quadimage->pi_image->pframe->rmsmap = cpl_image_new( cpl_image_get_size_x( quadimage->pi_image->pframe->image ),
                                        cpl_image_get_size_y( quadimage->pi_image->pframe->image ), CPL_TYPE_DOUBLE );
                        ok = ok + 1;
                    }
                }

            } // if quadimage

            if ( ok == 4 ){
                //sph_quad_image_save( quadimage, czFrameNew, pl );
                sph_quad_image_save( quadimage, cpl_frame_get_filename( quadframe ), pl);
                sph_quad_image_delete( quadimage ); quadimage = NULL;

                //quadframe = cpl_frame_new();
                //cpl_frame_set_filename( quadframe, czFrameNew );
                cpl_frame_set_tag( quadframe, "None");
                cpl_frameset_insert( result, quadframe );

                sph_error_raise( SPH_QUAD_IMAGE_GENERAL,
                    __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,  " QUAD IMAGE frame with filename = %s is created \n"
                            "(from the PREPROC ZPL EXP frame with filename = %s)", cpl_frame_get_filename ( quadframe ), czFrame );

            } else {
                sph_error_raise( SPH_QUAD_IMAGE_BAD_IMAGE,
                    __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING,  " Frame bad image(s)...ignore this frame (filename: %s)", czFrame );
            }

        } else {             //if czType
            sph_error_raise( SPH_QUAD_IMAGE_BAD_TYPE,
                __FILE__, __func__, __LINE__,
                SPH_ERROR_WARNING,  " Frame bad Type  = %s...ignore this frame (filename: %s)", czType, czFrame );

        }

        cpl_propertylist_delete(pl);
        curframe = cpl_frameset_get_next_const( inframes );

    } // end while

    sph_keyword_manager_delete();
    if (cpl_frameset_get_size(result) == 0) {
    	cpl_frameset_delete(result); result = NULL;
    }
    return result;

}


/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new quad image frameset from the 'cube(s)' of zpl exp TYPE
 *
 * @param inframe  the input cube(s) cpl_frameset  (of ZPL_EXP_TYPE)
 *
 * @return pointer to the newly created frameset with quad_image or NULL in
 * case of error.
 *
 * This function creates new sph_quad_image frameset from a given cpl_frameset
 * given as a cpl_frameset of type ZPL_EXP_TYPE.
 *
 */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_quad_image_create_quad_image_frameset_from_zplexp_cubes( const cpl_frameset* inframes ){
    const cpl_frame*          curframe;
    cpl_frameset*            result                = NULL;
    sph_keyword_manager*    keym                = NULL;

    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);

    keym = sph_keyword_manager_new();
    if ( !keym ) {
    	SPH_ERR("Can't create keyword manager.");
        return NULL;
    }

    result = cpl_frameset_new();

    curframe = cpl_frameset_get_first_const( inframes );
    while ( curframe ) {
        const char*       filename = cpl_frame_get_filename(curframe);
        cpl_frameset*     expframes = cpl_frameset_new();
        cpl_frameset*     result_expframes    = NULL;
        cpl_propertylist* pl = sph_keyword_manager_load_properties(filename, 0);
        cpl_propertylist* pl_proper = cpl_propertylist_load(filename, 0);
        int               nz                  = 0;

        if ( cpl_propertylist_has( pl_proper, "NAXIS3") ) {
            nz = cpl_propertylist_get_int( pl_proper, "NAXIS3");
        } else {
            sph_error_raise(SPH_ERROR_WARNING,
                            __FILE__, __func__, __LINE__,
                            SPH_ERROR_WARNING, "An input frame is not a cube: %s ;"
                            " proceed further anyway",
                            filename );
            nz = 1;
        }

        for (int i = 0; i < nz; i++){
            sph_error_code    rerr;
            cpl_frame*        expframe;
            sph_zpl_exposure* zplexp   = sph_zpl_exposure_load( curframe, i );
            if (!zplexp) {
                sph_error_raise(SPH_ERROR_WARNING,
                                __FILE__, __func__, __LINE__,
                                SPH_ERROR_WARNING, "Could not load an individual zpl exp frame: %s. Goto the next cube",
                                filename );
                break;
            }
            expframe = sph_filemanager_create_temp_frame( filename,
                                                          "NONE",
                                                          CPL_FRAME_GROUP_NONE );

            //TODO:
            //add to the global eso property list of the zpl exp cube particular information about an
            //individual frame such as time stamp, PA, H, ...
            //this particular information must be retrieved(calculated), based on the general keywords
            //of the cube (start time, end time, number of zplexp plane, etc...)


            rerr = sph_zpl_exposure_save( zplexp, expframe, pl);
            sph_zpl_exposure_delete( zplexp );
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_WARNING,
                                __FILE__, __func__, __LINE__,
                                SPH_ERROR_WARNING, "Could not save an individual zpl exp frame: %s. Goto the next cube",
                                cpl_frame_get_filename( expframe ) );
                break;
            }
            rerr = cpl_frameset_insert( expframes, expframe);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_ERROR,
                                __FILE__, __func__, __LINE__,
                                SPH_ERROR_ERROR, "Could not insert expframe with the filename -- %s -- into expframes",
                                cpl_frame_get_filename( expframe ) );
                sph_keyword_manager_delete();
                cpl_frameset_delete( expframes);
                cpl_propertylist_delete(pl);
                cpl_propertylist_delete(pl_proper);
                return NULL;
            }
        } //end for

        result_expframes = sph_quad_image_create_quad_image_frameset_from_zplexp_frameset( expframes );

        if (result_expframes) {
            cpl_frameset_join(result, result_expframes);
            cpl_frameset_delete(result_expframes);
        } else {
            SPH_ERR("Could not create quad image result_frameset from expframes");
        }

        cpl_frameset_delete( expframes);
        cpl_propertylist_delete(pl);
        cpl_propertylist_delete(pl_proper);

        curframe = cpl_frameset_get_next_const(inframes);
    } //end while

    if (cpl_frameset_get_size(result) == 0) {
        sph_error_raise( SPH_QUAD_IMAGE_GENERAL, __FILE__,  __func__, __LINE__,
                         SPH_ERROR_WARNING, "The result frameset has no elements." );
    	cpl_frameset_delete(result);
        result = NULL;
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    delete the sph_quad image and release resources
  @param    self The quad image to delete
  @return    void
 */
/*----------------------------------------------------------------------------*/
void sph_quad_image_delete( sph_quad_image* self ) {
    if ( self ) {
        sph_double_image_delete( self->zero_image );
        sph_double_image_delete( self->pi_image );
        cpl_propertylist_delete( self->properties );
        cpl_propertylist_delete( self->qclist );
        cpl_free( self );
    }
}
