/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_skycalc.h"

#include "sph_error.h"

#include <stdlib.h>
#include <string.h>
#include <math.h>

/*-----------------------------------------------------------------------------
 Declaration of Private Functions
 -----------------------------------------------------------------------------*/

static
void sph_skycalc_set_output(sph_skycalc_output* skyout,
                            double objra, double objdec, double objepoch,
                            double jd, double curep, double mura_arcs,
                            double mura_sec, double mudec, double sid,
                            double lat, double elevsea, double horiz);

void sph_unsph_skycalc_getch(int);
int sph_skycalc_legal_num_part(char);
int sph_skycalc_legal_int_part(char);
int sph_skycalc_legal_command_char(char);
double sph_roundx(double, int);
void sph_put_hrs(double, short, int, int, int);
void sph_skycalc_chomp(char *);
double sph_skycalc_atan_cric(double, double);
void sph_skycalc_min_max_alt(double, double, double*, double*);
double sph_skycalc_ha_alt(double, double, double);
int sph_skycalc_get_pm(double, double*, double*);
int sph_skycalc_get_time(struct date_time *);
double sph_skycalc_date_to_jd(struct date_time);
void sph_skycalc_caldat(double, struct date_time *, short *);
short sph_skycalc_day_of_week(double);
double sph_skycalc_day_of_year(double);
void sph_spkycalc_print_day(short);
void sph_spkycalc_print_all(double);
void sph_spkycalc_print_current(struct date_time, short, short);
void sph_spkycalc_print_calendar(double, short *);
void sph_spkycalc_print_time(double, short);
double sph_skycalc_frac_part(double);
double sph_skycalc_lst(double, double);
void sph_skycalc_lpmoon(double, double, double, double*, double*, double*);
void sph_skycalc_lpsun(double, double*, double*);
void sph_skycalc_eclrot(double, double*, double*, double*);
double sph_skycalc_sph_skycalc_circulo(double);
void sph_skycalc_geocent(double, double, double, double*, double*, double*);
double sph_skycalc_etcorr(double);
void sph_skycalc_flmoon(int, int, double *);
float sph_skycalc_lun_age(double, int *);
double sph_skycalc_jd_moon_alt(double, double, double, double, double);
double sph_skycalc_jd_sun_alt(double, double, double, double);
void sph_skycalc_find_dst_bounds(int, double, int, double *, double *);
double sph_zone(short, double, double, double, double);
double sph_true_jd(struct date_time, short, short, short, double);
void sph_spkycalc_print_tz(double, short, double, double, char);
void sph_xyz_cel(double, double, double, double*, double*);
void sph_aberrate(double,  double[], int);
void sph_skycalc_nutation_params(double, double*, double*);
double sph_skycalc_near_hor_refr(double, double);
double sph_refract_size(double, double);
void sph_refract_corr(double*, double*, double, double, double*, int);
void sph_skycalc_mass_precess(void);
void sph_spkycalc_print_apparent(double, double, double, double, double, double, double, double, double);
void sph_skycalc_galact(double, double, double, double*, double*);
void sph_skycalc_comp_el(double);
void sph_spkycalc_planetxyz(int, double, double*, double*, double*);
void sph_spkycalc_planetvel(int, double, double*, double*, double*);
void sph_xyz2000(double, double, double, double);
void sph_xyz2000xf(double, double*, double*, double*);
void sph_skycalc_earthview(double*, double*, double*, int, double*, double*);
void sph_spkycalc_pposns(double, double, double, short, double*, double*);
void sph_skycalc_barycor(double, double*, double*, double*, double*, double*, double*);
void sph_skycalc_lscrcor(double, double, double, double*);
void sph_skycalc_parellipse(double, double, double, double, double,
                            double, double *, double *, double *, double *);
float sph_skycalc_overlap(double, double, double);
short sph_setup_time_place(struct date_time, double, double, double, short,
                           char *, char, char *, short, short, 
                           double*, double*, double*, double*, double*, double*);

void sph_spkycalc_print_tonight(struct date_time, double, double, double,
                                double, double, char *, double, char *, char,
                                short, double*, double*, short);
void sph_spkycalc_print_circumstances(double, double, double, double,
                                      double, double, double, double,
                                      double, double, double, double);

void sph_skycalc_hourly_airmass(struct date_time, double, double, double,
                                double, short, double, double,
                                double, double, double, double);
void sph_spkycalc_print_params(struct date_time, short, short, double, double,
                               double, char *, double, double, short, double,
                               double, double, double, double, double);
void sph_spkycalc_print_menu(void);
void sph_spkycalc_print_tutorial(void);
void sph_spkycalc_print_examples(void);
void sph_spkycalc_print_accuracy(void);
void sph_spkycalc_print_legaltites(void);
void sph_skycalc_ephemgen(double, double, double, double, double);
double sph_skycalc_hrs_up(double, double, double, double);
void sph_spkycalc_print_air(double, short);
void sph_spkycalc_print_ha_air(double, double, short, short);
void sph_skycalc_obs_season(double, double, double, double, double);
void sph_skycalc_inheap(float *, int *, int *);
void sph_set_heap(float *, int *, int);
void sph_swapheap(float *, int *, int, int, int *);
void sph_skycalc_outheap(float *, int *, int *, int, int *);
void sph_skycalc_indexx(int, float *, int *);
int sph_read_obj_list(void);
int sph_skycalc_find_by_name(double *, double *, double,
                             struct date_time, short, short, short,
                             double, double, double);
void sph_type_list(struct date_time, short, short, short,
                   double, double, double);
int sph_skycalc_find_nearest(double *, double *, double, struct date_time,
                             short, short, short, double, double, double);
void sph_set_zenith(struct date_time, short, short, short,
                    double, double, double, double, double *, double *);
void sph_print1phase(struct date_time, short, short, short,
                     double, double, double, double, double, double);
int sph_set_to_jd(struct date_time*, short, short, short,
                  double, double, int);
void sph_spkycalc_phaselisting(short, short, short, double, double,
                               double, double, double, double);
static void sph_skycalc_oprntf(const char *, ...) CPL_ATTR_PRINTF(1, 2);
char sph_skycalc_getch(void);
short sph_skycalc_get_line(char *);

/* FIXME: These warnings should be fixed upstream... */
CPL_DIAG_PRAGMA_PUSH_IGN(-Wdeclaration-after-statement);
/* FIXME: Mac OSX does not honor the below when specified as an error */
CPL_DIAG_PRAGMA_PUSH_IGN(-Wformat-nonliteral);
CPL_DIAG_PRAGMA_PUSH_IGN(-Wunused-result);
CPL_DIAG_PRAGMA_PUSH_IGN(-Wunused-but-set-variable);
CPL_DIAG_PRAGMA_PUSH_IGN(-Wunused-parameter);
CPL_DIAG_PRAGMA_PUSH_IGN(-Wmaybe-uninitialized);
CPL_DIAG_PRAGMA_PUSH_IGN(-Wshadow);
CPL_DIAG_PRAGMA_PUSH_IGN(-Wparentheses);
CPL_DIAG_PRAGMA_PUSH_IGN(-Wold-style-definition);
#include "skycalc.c"
CPL_DIAG_PRAGMA_POP;
CPL_DIAG_PRAGMA_POP;
CPL_DIAG_PRAGMA_POP;
CPL_DIAG_PRAGMA_POP;
CPL_DIAG_PRAGMA_POP;
CPL_DIAG_PRAGMA_POP;
CPL_DIAG_PRAGMA_POP;
CPL_DIAG_PRAGMA_POP;
CPL_DIAG_PRAGMA_POP;
CPL_DIAG_PRAGMA_POP;

//#define  DEG_IN_RADIAN     57.2957795130823
//#define  EQUAT_RAD         6378137.    /* equatorial radius of earth, meters */

static char hh_string[80];
static char sph_s[80];

/*-----------------------------------------------------------------------------
 Definition of Functions
 -----------------------------------------------------------------------------*/

void sph_skycalc_put_hh_string( char* param_string ){
	snprintf(hh_string, sizeof(hh_string), "%s", param_string );
}

void sph_skycalc_put_sph_s( char* param_string ){
	snprintf(sph_s, sizeof(sph_s), "%s", param_string );
}


static int sph_skycalc_set_timez_local(  sph_skycalc_input* skyinp )
{
	//in fact there is no 'L' command, but it will be interpreted by skycalc like  "unknown command" which is then
	//set the time to the default local time
	skyinp[TIMEZ].command = 'L';
	return 0;
}

static int sph_skycalc_set_timez_ut(  sph_skycalc_input* skyinp )
{
	skyinp[TIMEZ].command = 'g';
	return 0;
}

static int sph_skycalc_set_radec( sph_skycalc_input* skyinp, const char* ra,
                                  const char* dec )
{
	(void)strcpy(skyinp[RA].param_string, ra);
	(void)strcpy(skyinp[DEC].param_string, dec);
	return 0;
}


static int sph_skycalc_set_epoch( sph_skycalc_input *skyinp, int epoch )
{
	sprintf(skyinp[EPOCH].param_string, "%d", epoch);
	return 0;
}


static int sph_skycalc_set_commands( sph_skycalc_input* skyinp)
{
	skyinp[RA].command = 'r';
	skyinp[DEC].command = 'd';
	skyinp[EPOCH].command = 'e';
	//skyinp[TIMEZ].command = 'g'; //set UT time per default
	sph_skycalc_set_timez_ut( skyinp );
	skyinp[DATE].command = 'y';
	skyinp[TIME].command = 't';
	skyinp[EQUAL].command = '=';
	return 0;
}


int sph_skycalc_set_timezone( sph_skycalc_input *skyinp, int timez )
{
	switch ( timez ){
		case LT:
			SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_GENERAL,"LT is set up!")
			sph_skycalc_set_timez_local( skyinp );
			break;
		case UT:
			SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_GENERAL,"UT is set up!")
			sph_skycalc_set_timez_ut( skyinp );
			break;
		default:
			SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "Unknown time sph_zone (expected 0- UT, 1- Local Time): %d", timez)
			return(-1);
	}
	return 0;

}


int sph_skycalc_setup( sph_skycalc_input* skyinp, const char* ra, const char* dec,
                       int epoch, const char* date, const char* time )
{
	sph_skycalc_set_radec( skyinp, ra, dec );
	sph_skycalc_set_epoch( skyinp, epoch );
	sph_skycalc_set_datetime( skyinp, date, time );
	(void)strcpy(skyinp[EQUAL].param_string, "NONE"); //no real parameter for this command
	(void)strcpy(skyinp[TIMEZ].param_string, "NONE"); //no real parameter for this command
	sph_skycalc_set_commands( skyinp );
	return 0;
}

int sph_skycalc_setup_iso8601_string( sph_skycalc_input* skyinp, const char* ra, const char* dec, int epoch, const char* iso8601 )
{
	int 				year, month, day, hh, mm;
	double				ss;

	sph_skycalc_set_radec( skyinp, ra, dec );
	sph_skycalc_set_epoch( skyinp, epoch );
	sph_time_iso8601_from_string(&year, &month, &day, &hh, &mm, &ss, iso8601 );
	sph_skycalc_set_datetime_iso8601(skyinp, year, month, day, hh, mm, ss );
	sph_skycalc_set_commands( skyinp );


	return 0;
}
int sph_skycalc_setup_iso8601( sph_skycalc_input* skyinp,
                               const char* ra, const char* dec, int epoch,
		  int year, int month, int day, int hh, int mm, double ss)
{
	sph_skycalc_set_radec( skyinp, ra, dec );
	sph_skycalc_set_epoch( skyinp, epoch );
	sph_skycalc_set_datetime_iso8601(skyinp, year, month, day, hh, mm, ss );
	sph_skycalc_set_commands( skyinp );

	return 0;
}

int sph_skycalc_setup_mjd( sph_skycalc_input* skyinp,
                           const char* ra, const char* dec, int epoch, double mjd ){
	int 				year, month, day, hh, mm;
	double				ss;

	sph_skycalc_set_radec( skyinp, ra, dec );
	sph_skycalc_set_epoch( skyinp, epoch );
	sph_time_iso8601_from_mjd(&year, &month, &day, &hh, &mm, &ss, mjd );
	sph_skycalc_set_datetime_iso8601(skyinp, year, month, day, hh, mm, ss );
	sph_skycalc_set_commands( skyinp );

	return 0;
}


int sph_skycalc_set_datetime_iso8601( sph_skycalc_input* skyinp,
									  int year, int month, int day, int hh, int mm, double ss)
{
	sprintf(skyinp[DATE].param_string, "%d %d %d", year, month, day);
	sprintf(skyinp[TIME].param_string, "%d %d %f", hh, mm, ss);

	return 0;
}

int sph_skycalc_set_datetime( sph_skycalc_input* skyinp, const char* date, const char* time )
{
	(void)strcpy(skyinp[DATE].param_string, date);
	(void)strcpy(skyinp[TIME].param_string, time);

	return 0;
}

int sph_skycalc_set_time( sph_skycalc_input* skyinp, const char* time )
{
	(void)strcpy(skyinp[TIME].param_string, time);

	return 0;
}

int sph_skycalc_set_mjd ( sph_skycalc_input* skyinp, double mjd ){
	int 				year, month, day, hh, mm;
	double				ss;

	sph_time_iso8601_from_mjd(&year, &month, &day, &hh, &mm, &ss, mjd );
	sph_skycalc_set_datetime_iso8601(skyinp, year, month, day, hh, mm, ss );

	return 0;
}

int sph_skycalc_put_date( struct date_time* date )
{

	int valid_date = 0;
	int niter = 0;

	char* year;
	char* day;
	char* month;
	char sph_s_copy[80]; //--aip

	snprintf( sph_s_copy, sizeof(sph_s_copy), "%s", sph_s );

	//now a nitty-gritty parsing of the sph_s string
	year = strtok( sph_s_copy, " ");
	month = strtok( NULL, " ");
	day = strtok( NULL, " ");

	SPH_ERROR_RAISE_DBG_INFO( SPH_ERROR_INFO, "Year = %s Month = %s Day = %s\n", year, month, day);
	//printf("Enter get short!\n"); //--aip
	while(valid_date == 0 && niter++ < 3) {
	/* have to put in a trap or there's a possible runaway ... */
	 	snprintf( sph_s, sizeof(sph_s), "%s", year );
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "year->sph_s = %s\n", sph_s);
		getshort(&(date->y),-10,2100,
           	   "Give year again, then month and day.\n");
		if(date->y <= 0) return(-1);

	/* scan for mo and day here, *then* error check. */
		snprintf( sph_s, sizeof(sph_s), "%s", month );
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "month->sph_s = %s\n", sph_s);
		getshort(&(date->mo),1,12,
		   "Give month again (as number 1-12), then day.\n");
		snprintf( sph_s, sizeof(sph_s), "%s", day );
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "day->sph_s = %s\n", sph_s);
		getshort(&(date->d),0,32,"Give day again.\n");
		/* a lot of this error checking is redundant with the
		    checks in the new getshort routine.... */
		if(date->y < 100)  {
			date->y = date->y + 1900;
			SPH_ERROR_RAISE_ERR( SPH_ERROR_ERROR, "(Your answer assumed to mean %d)\n",date->y);
		}
		else if((date->y <= 1900 ) || (date->y >= 2100)){
			SPH_ERR("Date out of range: only 1901 -> 2099 allowed.\n");
		}
		/* might be nice to allow weird input dates, but calendrical
		   routines will not necessarily handle these right ... */
		else if((date->d < 0) || (date->d > 32)){
			SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR, "day-of-month %d not allowed!\n",
				date->d);
		}
		else if((date->mo < 1) || (date->mo > 12)){
			SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR, "month %d doesn't exist!\n",
				date->mo);
		}
		else {
			valid_date = 1;
			return(0);  /* success */
		}
	}
	if(valid_date == 0) return(-1);
	return(0);
}



int sph_skycalc_putshort( short* d, short least, short most,
                          const char* errprompt )
{
    //char s[30];
    short success = -1, ndiscard = 0;
    char c, buf[200];

    //scanf("%s",s);
    while(success < 0) {
    	success = sph_spkycalc_parseshort(sph_s,d);
    	if((success == 0) && ((*d < least) || (*d > most))) {
    		SPH_ERROR_RAISE_ERR( SPH_ERROR_ERROR, "%d is out of range; allowed %d to %d -- try again.\n",
			*d,least,most );
    		success = -1;
    	}
    	if(success < 0) {
    		/* if there's error on input, clean out the rest of the line */
    		ndiscard = 0;
    		while((c = getchar()) != '\n')  {
    			buf[ndiscard] = c;
    			ndiscard++;
    		}
    		if(ndiscard > 0) {
    			buf[ndiscard] = '\0';  /* cap the string */
    			SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO,"Rest of input (%s) has been discarded.\n",buf);
    		}
    		SPH_ERROR_RAISE_ERR( SPH_ERROR_ERROR, "%s", errprompt );
    		SPH_ERR( "Parseshort is failed!" );

     		return( (int) success );
    		//printf("%s",errprompt);
	       	//printf("Try again:");
	       	//scanf("%s",s);
      }
    }
    return( (int) success);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Converts global "sph_s" string into a double-precision
 @param    least			double  minimal value
 @param	   most				double  maximal value
 @param    errorprompt      char*   specific error
 @param	   d				double*	output value

 @return   0 = ok, with number, 1 = found a valid command,
	   	   but no number, and -1 = an error of some sort (unexpected char)


  @note It takea a string from the global variable sph_s and
  converts it into a double-precision d. sph_s is set up each time
  in the original sph_skycalc function (substitution of the getdouble).
  */
/*----------------------------------------------------------------------------*/

int sph_skycalc_putdouble(	double *d, double least, double most,
                                const char* errprompt )
{
    //char s[30];  --aip
    char buf[200], c;
    short success = -1, ndiscard = 0;

    //scanf("%s",s); --aip
    while(success < 0) {
	success = sph_skycalc_parsedouble(sph_s,d);
	if((success == 0) && ((*d < least) || (*d > most))) {
	   SPH_ERROR_RAISE_ERR(SPH_ERROR_ERROR, "%g is out of range; allowed %g to %g -- \n",
			*d,least,most);
	   success = -1;
	}
	if(success < 0) {
	   /* if there's error on input, clean out the rest of the line */
	   ndiscard = 0;
	   while((c = getchar()) != '\n')  {
		buf[ndiscard] = c;
		ndiscard++;
	   }
	   if(ndiscard > 0) {
		buf[ndiscard] = '\0';  /* terminate the string */
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "Rest of input (%s) has been discarded.\n",buf);
	   }
  		SPH_ERROR_RAISE_ERR( SPH_ERROR_ERROR, "%s", errprompt );
   		SPH_ERR( "Parsedouble is failed!" );
	}
    }
    return((int) success);
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Converts hh_string into a double-precision  coordinate
 @param    instrng   The input string to check

 @return   converted double precision coordinate

  Take a string with coordinates from the global variable hh_string and
  converts it into a double-precision  coordinate. hh_string is set up each time
  in the original sph_skycalc function (substitution of the get_coord).
 */
/*----------------------------------------------------------------------------*/
double sph_skycalc_put_coord(void)
{
   short sign;
   double hrs, mins, secs = 0.0; /* Avoid uninit warning */
   //char hh_string[6];  /* string with the first coord (hh) */ // this coordinate string is global for this wrapper   -- aip
   char hh1;
   //char errprompt[80]; -- unused varibale
   short i = 0;
   int end_in = 0;

   if(!strcmp(hh_string, "NONE")) return 0.0;

   /* read and handle the hour (or degree) part with sign */

   //scanf("%s",hh_string);   //no read from screen --aip

   if(sph_skycalc_is_delimited(hh_string) == 1) {  /* It's colon-delimited, or something */
	 return(sph_skycalc_conv_delimited_coord(hh_string));
   }

   hh1 = hh_string[i];

   while(hh1 == ' ') {
       /* discard leading blanks */
       i++;
       hh1 = hh_string[i];
   }

   if(hh1 == '-') sign = -1;   /* get sign explicitly */
   else if(hh1 == '+') sign = 1;
   else sign = 1;

    if((end_in = sph_skycalc_parsedouble(hh_string,&hrs)) < 0) {
    	SPH_ERR("Didn't parse correctly -- set parameter to zero!!\n")
    	return(0.0);
   }

   if(sign == -1) hrs = -1. * hrs;

   /* read in the minutes and seconds normally */
   if(end_in == 0)
       	end_in = getdouble(&mins,0.,60.,
	  "Give minutes again, then seconds; no further prompts.\n");
   else return(sign * hrs);
   if(end_in == 0) end_in = getdouble(&secs,0.,60.,
     "Give seconds again, no further prompts.\n");
   else if(end_in == 1) secs = 0.;
   return(sign * (hrs + mins / 60. + secs / 3600.));
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Set the site-specific site data (VLT, Cerro Paranal)
 @return  	longit     = W longitude in decimal hours
			lat        = N latitude in decimal degrees
			stdz       = standard time sph_zone offset, hours
			elevsea    = elevation above sea level (for absolute location)
			elev       = observatory elevation above horizon, meters
			horiz      = (derived) added zenith distance for rise/set due
					to elevation
			use_dst    = 0 don't use it
				     1 use USA convention
				     2 use Spanish convention
				     < 0 Southern hemisphere (reserved, unimplimented)
			zone_name  = name of time sph_zone, e. g. Eastern
			zabr       = single-character abbreviation of time sph_zone
			site_name  = name of site.
*/
/*----------------------------------------------------------------------------*/
void sph_skycalc_put_site(double *longit, double *lat, double *stdz,
                          short *use_dst, char *zone_name, char *zabr,
                          double *elevsea, double *elev, double *horiz,
                          char *site_name)
{
	strcpy(site_name, "VLT, Cerro Paranal");
	strcpy(zone_name, "Chilean");
	*zabr = 'C';
	*use_dst = -1;
	*longit = 4.69356;
	*lat = -24.625;
	*stdz = 4.;
	*elevsea = 2635.;
	*elev = 2635.; /* for ocean horizon, not Andes! */
	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_GENERAL,"\n\n** Will use daylght time, Chilean date conventions. \n\n");
	/* now compute derived quantity "horiz" = depression of horizon.*/
	*horiz = sqrt(2. * *elev / EQUAT_RAD) * DEG_IN_RADIAN;
}


/* Given object, site, and time information, prints the circumstances
   of an observation.  The heart of the "calculator" mode. */

static
void sph_skycalc_set_output(sph_skycalc_output* skyout,
                            double objra, double objdec, double objepoch,
                            double jd, double curep, double mura_arcs,
                            double mura_sec, double mudec, double sid,
                            double lat, double elevsea, double horiz)
{
	double objra_adj,objdec_adj,curra,curdec,ha,alt,az,secz,par; // airm -- unused variables --aip
	char constelname[5];
	float ill_frac,tanz,howbad;
	double ramoon,decmoon,distmoon,ang_moon,
		georamoon, geodecmoon, geodistmoon,
		rasun,decsun,distsun,x,y,z,
		toporasun,topodecsun,
		moon_alt,sun_alt,obj_moon,obj_lunlimb,sun_moon,Vmoon,
		obj_alt,eclong,eclat,tcor,vcor;
	short sph_skycalc_luneclcode = 0;
	double objpar;

	//oprntf("\n\nStd epoch--> RA:");
	//sph_put_coords(objra,3,0);
	//oprntf(", dec: ");
	//sph_put_coords(objdec,2,1);
	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "RA = %f, DEC=%f", objra, objdec);
	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "Epoch  %7.2f  ",objepoch);
    sph_radec_to_constel(objra, objdec, objepoch, constelname);
	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, " (in %s)",constelname);
	if((mura_sec != 0.) | (mura_arcs != 0.) |(mudec != 0.)) {
            double mura_arcs_;
		objra_adj = objra + (curep-objepoch)* mura_sec/3600.;
		objdec_adj = objdec + (curep-objepoch)*mudec/3600.;
		//oprntf("Adj for p.m: RA:");
		//sph_put_coords(objra_adj,3,0);
		//oprntf(", dec: ");
		//sph_put_coords(objdec_adj,2,1);
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "RA_adj = %f, DEC_adj %f", objra_adj, objdec_adj);
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "epoch %7.2f, equinox %7.2f\n", curep,objepoch);
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_GENERAL,"(Annual proper motions:");
		mura_arcs_ = mura_sec * 15. *
		     cos(objdec / DEG_IN_RADIAN);
      	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, " RA: %8.4f sec //%7.3f arcsec, ",mura_sec,mura_arcs_);
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, " dec: %7.3f)\n",mudec);
		sph_skycalc_cooxform(objra_adj,objdec_adj,objepoch,curep,&curra,&curdec,
			XFORM_JUSTPRE,XFORM_FROMSTD);
	}
	else sph_skycalc_cooxform(objra,objdec,objepoch,curep,&curra,&curdec,
		XFORM_JUSTPRE, XFORM_FROMSTD);
	//oprntf("Current  --> RA:");
	//sph_put_coords(curra,3,0);
	//oprntf(", dec: ");
	//sph_put_coords(curdec,2,1);

	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, " RA = %f, Dec= %f", curra, curdec);
	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, " ep %7.2f\n",curep);
	ha = sph_skycalc_adj_time(sid - curra);
	//oprntf("HA: ");
	//sph_put_coords(ha,2,1);
	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, " HA = %f\n",ha);

	alt=sph_skycalc_altit(curdec,ha,lat,&az,&par);
	obj_alt = alt;

	/* test size of sec z to avoid overflowing space provided */
	secz = sph_secant_z(alt);
	if(secz >= 0.) {
          if(secz < 12.){
        	  SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "airmass = %8.3f",sph_true_airmass(secz));
          }
          else if(secz <= 99.){
        	  SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO," sec.z = %8.3f",secz);
          }
	   else SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_GENERAL," Obj very near horizon.");
          if(secz > 3.) SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_GENERAL," -- Large airmass!");
	} else SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_GENERAL," -- BELOW HORIZON.");

	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "sph_skycalc_altitude %6.2f, azimuth %6.2f, ",alt,az);
	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "parallactic angle %4.1f",par);
	objpar = par;
	/* also give +- 180 ..... */
	//if((par <= 180.) && (par > 0.)) oprntf("  [%4.1f]\n",par - 180.);
	// else oprntf("  [%4.1f]\n",par + 180.);
	/* And, also give a measure of how bad the sph_refraction effects will be
	 * if you *don't* set to the parallactic angle ... */
	tanz = tan((90. - alt)/DEG_IN_RADIAN);
	howbad = fabs(tanz * sin(par/DEG_IN_RADIAN));
	if(alt > 3.) {
	  if(howbad < 100.) {
		  SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO,"Parallactic non-correction penalty: %5.2f\n\n",howbad);
	  }
	  else SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO,"Parallactic angle is very critical.\n");
	}

	sph_skycalc_accumoon(jd,lat,sid,elevsea,&georamoon,&geodecmoon,&geodistmoon,
			&ramoon,&decmoon,&distmoon);
	sph_skycalc_accusun(jd,sid,lat,&rasun,&decsun,&distsun,
		&toporasun,&topodecsun,&x,&y,&z);
	alt=sph_skycalc_altit(topodecsun,(sid-toporasun),lat,&az,&par);
	sun_moon = sph_subtend(ramoon,decmoon,toporasun,topodecsun);
	ill_frac= 0.5*(1.-cos(sun_moon)); /* ever so slightly inaccurate ...
	   basis of ancient Greek limit on moon/sun distance ratio! */
	sun_moon = sun_moon * DEG_IN_RADIAN;
/*      printf("geocentric moon: ");
	sph_put_coords(georamoon,4,0);
	printf("  ");
	sph_put_coords(geodecmoon,3,1);
	printf(" %f \n",geodistmoon);  */
	sun_alt = alt;
	if(alt < -18.) {
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_GENERAL,
				"The sun is down - there is no twilight.\n");
	}
	else {
		if (alt < -(0.83+horiz))
			{SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "In twilight, sun alt %4.1f, az %5.1f ",alt,az);}
		else SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "The sun is up, alt %4.1f, az %4.1f",alt,az);
		//oprntf("; Sun at");
		//sph_put_coords(toporasun,3,0);
		//oprntf(", ");
		//sph_put_coords(topodecsun,2,1);
		//oprntf("\n");
		if(sun_moon < 1.5) sph_solecl(sun_moon,distmoon,distsun);
			/* check for solar eclipse if it's close */
		if (alt < -(0.83+horiz))
			SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO,"Clear zenith twilight (blue) approx %4.1f  mag over dark night sky.\n",
				sph_ztwilight(alt));
	}
	moon_alt=sph_skycalc_altit(decmoon,(sid-ramoon),lat,&az,&par);

	if(moon_alt > -2.) {
		//oprntf("Moon:");
		//sph_put_coords(ramoon,2,0);
		//oprntf(", ");
		//sph_put_coords(decmoon,1,1);
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "Moon: ra = %f, dec=%f, alt %5.1f, az %5.1f;",ramoon, decmoon, moon_alt,az);
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "%6.3f illum.\n",ill_frac);
		//sph_spkycalc_print_phase(jd);
		obj_moon = DEG_IN_RADIAN * sph_subtend(ramoon,decmoon,curra,curdec);
		if(fabs(obj_moon) > 10.) {
			SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, " Object is %5.1f degr. from moon.\n",obj_moon);
		}
		else  {
			ang_moon = DEG_IN_RADIAN * asin(RMOON / (distmoon * EQUAT_RAD));
			if((obj_lunlimb = obj_moon - ang_moon) > 0.)
				{SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, " ** NOTE ** Object %4.1f degr. from lunar limb!\n",obj_lunlimb);}
			else SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO," ** NOTE ** You're looking AT the moon!\n");
		}
		if(sun_moon > 176.)
			sph_skycalc_luneclcode = sph_skycalc_lunecl(georamoon,geodecmoon,geodistmoon,
			rasun,decsun,distsun);
		if((moon_alt > 0.) && (obj_alt > 0.5) && (sun_alt < -9.)) {
		  /*if it makes sense to estimate a lunar sky brightness */
		  Vmoon = sph_skycalc_lunskybright(sun_moon,obj_moon,KZEN,moon_alt,
				obj_alt,distmoon);
		  SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "Lunar part of sky bright. = %5.1f V mag/sq.arcsec (estimated).\n",Vmoon);
		  if(sph_skycalc_luneclcode != 0)
			  SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO," NOT including effect of LUNAR ECLIPSE ...!\n");
		}
	}
	else {
		//sph_spkycalc_print_phase(jd);
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "The moon is down.\n");
	}
	sph_skycalc_eclipt(objra,objdec,objepoch,jd,&curep,&eclong,&eclat);
	if(fabs(eclat)<10.) {
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO,"Ecliptic latitude %4.1f; ",eclat);
		SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "watch for low-flying minor planets.\n");
	}
	sph_spkycalc_planet_alert(jd,curra,curdec,PLANET_TOL);
	/* tolerance set to 3 degrees earlier. */
	sph_skycalc_helcor(jd,curra,curdec,ha,lat,elevsea,&tcor,&vcor);
	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "Barycentric corrections: add %6.1f sec, %5.2f km/sec to observed values.\n",tcor,vcor);
	SPH_ERROR_RAISE_DBG_INFO(SPH_ERROR_INFO, "Barycentric Julian date = %14.6f\n",(jd+tcor/SEC_IN_DAY));
	//printf("\nType command, 'f' for fast tour, '?' for a menu:");

	//set the output for sph_skycalc
	skyout->objra = objra;
	skyout->objdec = objdec;
	skyout->objepoch = objepoch;
	skyout->alt = alt;
	skyout->jd = jd;
	skyout->barjd = (jd+tcor/SEC_IN_DAY);
	skyout->pa = objpar;
}


