/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifndef SPH_SKYCALC_H_
#define SPH_SKYCALC_H_

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_time.h"
#include "sph_error.h"
#include <cpl.h>

/*-----------------------------------------------------------------------------
 Type definition
 -----------------------------------------------------------------------------*/
typedef struct _sph_skycalc_input
{
	char	command;
	char	param_string[80];
} sph_skycalc_input;

typedef struct _sph_skycalc_output
{
	double objra;
	double objdec;
	double objepoch;
	double alt;
	double jd;
	double barjd; //Barycentric Julian date (see original skycalc code)
	double pa; //paralactical angle in degrees
} sph_skycalc_output;

enum sph_skycalc_ind  {
		RA,
		DEC,
		EPOCH,
		TIMEZ,
		DATE,
		TIME,
		EQUAL,
		NCOMMAND
};

typedef struct date_time {
	short y;
	short mo;
	short d;
	short h;
	short mn;
	float s;
} sph_date_time ;


#define UT 0 //universal time
#define LT 1 //local time

#define load_site sph_skycalc_put_site
#define get_coord sph_skycalc_put_coord
#define getdouble sph_skycalc_putdouble
#define getshort sph_skycalc_putshort
#define get_date sph_skycalc_put_date

/* a couple of the system-dependent magic numbers are defined here */

#define SYS_CLOCK_OK 2    /* 1 means ANSI-standard time libraries do work,
   2 means they don't.  This is used by compiler switches in file 5 and
   the main program.  */

#define LOG_FILES_OK 1  /* 1 means that log files are enabled.
			Any other value means they're not.  */

#define MAX_OBJECTS 2000
#define MINSHORT -32767   /* min, max short integers and double precision */
#define MAXSHORT 32767
#define MAXDOUBLE 1.0e38
#define MINDOUBLE -1.0e38
#define BUFSIZE 150

#define XFORM_FROMSTD  1  /* defined quantities for apparent place transforms .. */
#define XFORM_TOSTDEP  -1
#define XFORM_JUSTPRE  1
#define XFORM_DOAPPAR  0

/* some (not all) physical, mathematical, and astronomical constants
   used are defined here. */

#define  PI                CPL_MATH_PI
#define  TWOPI             CPL_MATH_2PI
#define  PI_OVER_2         CPL_MATH_PI_2
#define  ARCSEC_IN_RADIAN  (3600.0 * CPL_MATH_DEG_RAD)
#define  DEG_IN_RADIAN     CPL_MATH_DEG_RAD
#define  HRS_IN_RADIAN     (CPL_MATH_DEG_RAD / 15.0)
#define  KMS_AUDAY         1731.45683633   /* km per sec in 1 AU/day */
#define  SPEED_OF_LIGHT    299792.458      /* in km per sec ... exact. */
#define  SS_MASS           1.00134198      /* solar system mass in solar units */
#define  J2000             2451545.        /* Julian date at standard epoch */
#define  SEC_IN_DAY        86400.
#define  FLATTEN           0.003352813   /* flattening of earth, 1/298.257 */
#define  EQUAT_RAD         6378137.    /* equatorial radius of earth, meters */
#define  ASTRO_UNIT        1.4959787066e11 /* 1 AU in meters */
#define  RSUN              6.96000e8  /* IAU 1976 recom. solar radius, meters */
#define  RMOON             1.738e6    /* IAU 1976 recom. lunar radius, meters */
#define  PLANET_TOL        3.          /* flag if nearer than 3 degrees
						to a major planet ... */
#define  KZEN              0.172       /* zenith extinction, mag, for use
				     in lunar sky brightness calculations. */
#define FIRSTJD            2415387.  /* 1901 Jan 1 -- calendrical limit */
#define LASTJD             2488070.  /* 2099 Dec 31 */

/* MAGIC NUMBERS which might depend on how accurately double-
   precision floating point is handled on your machine ... */

#define JDRESOLUTION 3.5e-8   /* 3 milliseconds */

#define  EARTH_DIFF        0.05            /* used in numerical
   differentiation to find earth velocity -- this value gives
   about 8 digits of numerical accuracy on the VAX, but is
   about 3 orders of magnitude larger than the value where roundoff
   errors become apparent. */

#define  MIDN_TOL          0.00001         /* this is no longer
   used -- it was formerly
   how close (in days) a julian date has to be to midnight
   before a warning flag is printed for the reader.  VAX
   double precision renders a Julian date considerably
   more accurately than this.  The day and date are now based
   on the same rounding of the julian date, so they should
   always agree. */

/*  FUNCTION PROTOTYPES and type definitions ....
    These are used in breaking the code into function libraries.
    They work properly on a strictly ANSI compiler, so they
    apparently comply with the ANSI standard format.  */

/* these global variables determine whether the program updates the
time and date whenever asked for output. */


int sph_skycalc( sph_skycalc_input* skyinp, sph_skycalc_output* skyout  );
int sph_skycalc_setup( sph_skycalc_input*, const char*, const char*, int epoch,
                       const char* date, const char* time);
int sph_skycalc_setup_iso8601_string( sph_skycalc_input* skyinp, const char* ra, const char* dec, int epoch, const char* iso8601 );
int sph_skycalc_setup_iso8601( sph_skycalc_input* skyinp, const char* ra, const char* dec, int epoch,
		  int year, int month, int day, int hh, int mm, double ss);
int sph_skycalc_setup_mjd(sph_skycalc_input*, const char*, const char*, int,
                          double);
int sph_skycalc_set_datetime_iso8601( sph_skycalc_input* skyinp,
                                      int year, int month, int day,
                                      int hh, int mm, double ss);
int sph_skycalc_set_datetime( sph_skycalc_input*, const char*, const char* );
int sph_skycalc_set_time( sph_skycalc_input* skyinp, const char* time );
int sph_skycalc_set_mjd ( sph_skycalc_input* skyinp, double mjd );
int sph_skycalc_set_timezone( sph_skycalc_input *skyinp, int timez );

void sph_skycalc_put_site(
		double *longit, double *lat, double *stdz,
		short *use_dst, char *zone_name, char *zabr,
		double *elevsea, double *elev, double *horiz,
		char *site_name );

void sph_skycalc_put_hh_string( char* param_string );
void sph_skycalc_put_sph_s( char* param_string );
double sph_skycalc_put_coord(void);
int sph_skycalc_put_date( struct date_time* date );
int sph_skycalc_putdouble(	double *d, double least, double most,
                                const char* errprompt );
int sph_skycalc_putshort( short* d, short least, short most,
                          const char* errprompt );

//declarations of some functions from original skycalc.c file (small hack :((, not a correct name convention)
int sph_spkycalc_parseshort(	char *s, short *d);
int sph_skycalc_parsedouble( char *s, double *d);
int sph_skycalc_is_delimited(char *instrng);
double sph_skycalc_conv_delimited_coord(char *instrng);
void sph_put_coords(double deci, int prec, int showsign);
void sph_radec_to_constel(double ra, double dec, double epoch, char *constname);
void sph_skycalc_cooxform(double rin, double din, double std_epoch, double date_epoch,
		double* rout, double* dout, int just_precess, int from_std);
double sph_skycalc_altit(double dec, double ha, double lat, double* az, double* parang);
double sph_true_airmass(double secz);
void sph_skycalc_accumoon(double jd, double geolat, double lst, double elevsea,
		double* geora, double* geodec, double* geodist, double* topora, double* topodec, double* topodist);
void sph_skycalc_accusun(double jd, double lst, double geolat,
		double* ra, double* dec, double* doubdist, double* topora, double* topodec,
		double* x, double* y, double* z);
double sph_subtend(double ra1, double dec1, double ra2, double dec2);
void sph_solecl(double sun_moon, double distmoon, double distsun);
float sph_ztwilight(double alt);
void sph_spkycalc_print_phase(double jd);
short sph_skycalc_lunecl(double georamoon,double geodecmoon,double geodistmoon,double rasun,double decsun,double distsun);
double sph_skycalc_lunskybright(double alpha,double rho,double kzen,double altmoon,double alt, double moondist);
void oprntf(char *fmt, ...);
void sph_skycalc_eclipt(double ra, double dec, double epoch, double jd,
		double* curep, double* eclong, double* eclat);
void sph_spkycalc_planet_alert(double jd,double ra,double dec,double tolerance);
void sph_skycalc_helcor(double jd,double ra,double dec,double ha,double lat,double elevsea,
		double* tcor,double* vcor);
double sph_skycalc_adj_time(double x);
double sph_secant_z(double alt);

#endif /* SPH_SKYCALC_H_ */
