/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */
#include "sph_smart_imagelist.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_fits.h"
#include "sph_common_keywords.h"
#include "sph_common_science.h"
#include <cpl.h>

const int SPH_SMART_IMAGELIST_MAX_IMAGES            = 2000;
const long long int SPH_MAX_MEMORY_FOR_FRAMES       = 800000000; /* 800MB Memory in bytes */


/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_smart_imagelist Smart management of many images
 *
 * @par Description:
 * This module provides the functionality to manage lists of frames in such
 * a way that they can all be operated on without running into memory problems.
 *
 * The basic idea is that most operations on many images, like averaging etc.
 * can either be performed sequentially (i.e. only one image needs to be in memory
 * at any time) or in windows, meaning that only a small region of ALL images needs
 * to be in memory at a given time. This module here manages the second case: the
 * images belonging to the frameset are read in in sub-regions.
 *
 * Thus structure should be used in all cases where a property needs to be calculated
 * on a big stack of images that has no x-y dependence but a z-dependence.
 *
 * For example to calculate the clean mean on a stack of 2000 frames, one would put
 * all the frames into a framseset, then create a new sph_smart_framelist() on this frame
 * set, then loop over all sets, getting them using sph_smart_imagelist_get_set(), and
 * for each one calculate a clean mean image. The resulting clean mean image is then added
 * to a stack. After the loop the stack is fed into sph_smart_imagelist_reassemble_imaage().
 *
 * @par Synopsis:
 * @code
 * typedef struct _sph_smart_imagelist
 * {
 *    int                     nx ;        Image size in x
 *    int                     ny ;        Image size in y
 *    int                     nregions;
 *    sph_detector_region*    regions;    array of regions
 *    cpl_frameset*           frameset;   Frameset to use
 *    sph_cube*               cube;       cube to use
 *    int                     extension;
 *    long int                maxmem;
 * } sph_smart_imagelist;
 * @endcode
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/
sph_error_code SPH_SMART_IMAGELIST_GENERAL                  = SPH_SMART_IMAGELIST_ERR_START + 0;
sph_error_code SPH_SMART_IMAGELIST_FRAMES_MISSING           = SPH_SMART_IMAGELIST_ERR_START + 1;
sph_error_code SPH_SMART_IMAGELIST_NO_FRAMES                = SPH_SMART_IMAGELIST_ERR_START + 2;
sph_error_code SPH_SMART_IMAGELIST_NO_SCHEME_FOUND          = SPH_SMART_IMAGELIST_ERR_START + 3;
sph_error_code SPH_SMART_IMAGELIST_BAD_SCHEME               = SPH_SMART_IMAGELIST_ERR_START + 4;


static sph_error_code sph_smart_imagelist_set_frames(sph_smart_imagelist*,
                                                     const cpl_frameset*);


/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new (empty) sph_smart_imagelist
 *
 * @return pointer to newly created smart_imagelist or NULL on error
 *
 * This creates a new empty smart_imagelist.
 */
/*----------------------------------------------------------------------------*/
sph_smart_imagelist* sph_smart_imagelist_new(void) {
    sph_smart_imagelist*   self    = NULL;

    self = cpl_calloc(1, sizeof(sph_smart_imagelist));

    if ( !self ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "could not allocate smart imagelist" );
        return NULL;
    }
    self->frameset = NULL;
    self->regions = NULL;
    self->nregions = 0;
    self->extension = 0;
    self->cube = NULL;
    self->maxmem = SPH_MAX_MEMORY_FOR_FRAMES;
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new sph_smart_imagelist from the given cube
 *
 * @return pointer to newly created smart_imagelist or NULL on error
 *
 * This creates a new empty smart_imagelist.
 */
/*----------------------------------------------------------------------------*/
sph_smart_imagelist*
sph_smart_imagelist_create_manyplanes( sph_smart_imagelist* self,
                                       cpl_frame* inframe,
                                       int extension ) {
    int                     nregions    = 1;
    cpl_image*              image       = NULL;
    int                     nframes     = 0;
    int                     sx          = 0;
    int                     sy          = 0;
    int                     xx          = 0;
    int                     yy          = 0;


    if ( !self )
        self = sph_smart_imagelist_new();

    if ( !self ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "could not allocate smart imagelist" );
        return NULL;
    }
    self->frameset = NULL;
    self->cube = NULL;
    self->manyplaneframe = inframe;
    self->extension = extension;

    if ( !inframe ) {
        sph_error_raise( SPH_SMART_IMAGELIST_FRAMES_MISSING, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "the frame seems empty" );
        return NULL;
    }

    image = cpl_image_load( cpl_frame_get_filename( inframe ), CPL_TYPE_DOUBLE, 0, extension );
    if ( !image ) {
        sph_error_raise( SPH_SMART_IMAGELIST_GENERAL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not load image from first frame." );
        return NULL;
    }

    self->nx = cpl_image_get_size_x(image);
    self->ny = cpl_image_get_size_y(image);

    nframes = sph_fits_get_nplanes( cpl_frame_get_filename(inframe), self->extension );

    nregions = 1;
    while ( (long long int)sizeof(double) * (long long int)self->nx * (long long int)self->ny * (long long int)nframes >
                   self->maxmem * (long long int)nregions * (long long int)nregions ) {
        nregions = nregions * 2;
    }
    if ( nregions >= self->nx ) {
        sph_error_raise( SPH_SMART_IMAGELIST_NO_SCHEME_FOUND, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not find a good division scheme." );
        cpl_image_delete(image); image = NULL;
        return NULL;
    }

    self->regions = cpl_calloc( nregions * nregions , sizeof(sph_detector_region));

    if ( self->nx % nregions || self->ny % nregions ) {
        sph_error_raise( SPH_SMART_IMAGELIST_BAD_SCHEME, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Something went wrong when determining the division scheme." );
        return NULL;
    }

    sx = self->nx / nregions;
    sy = self->ny / nregions;

    self->regions[0].ll_x = 0;
    self->regions[0].ll_y = 0;
    self->regions[0].ur_y = sy;
    self->regions[0].ur_x = sx;

    for ( yy = 0; yy < nregions; yy++ ) {
        for ( xx = 0; xx < nregions; xx++ ) {
            self->regions[ yy * nregions + xx ].ll_x = xx * sx;
            self->regions[ yy * nregions + xx ].ll_y = yy * sy;
            self->regions[ yy * nregions + xx ].ur_x = xx * sx + sx - 1;
            self->regions[ yy * nregions + xx ].ur_y = yy * sy + sy - 1;
        }
    }
    self->nregions = nregions * nregions;
    cpl_image_delete(image); image = NULL;
    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new sph_smart_imagelist from the given cube
 *
 * @return pointer to newly created smart_imagelist or NULL on error
 *
 * This creates a new empty smart_imagelist.
 */
/*----------------------------------------------------------------------------*/
sph_smart_imagelist* sph_smart_imagelist_create_cube( sph_smart_imagelist* self, sph_cube* incube, int extension ) {
    int                     nregions    = 1;
    cpl_image*              image       = NULL;
    int                     nframes     = 0;
    int                     sx          = 0;
    int                     sy          = 0;
    int                     xx          = 0;
    int                     yy          = 0;


    if ( !self )
        self = sph_smart_imagelist_new();

    if ( !self ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "could not allocate smart imagelist" );
        return NULL;
    }
    self->frameset = NULL;
    self->cube = incube;
    self->extension = extension;

    if ( !incube ) {
        sph_error_raise( SPH_SMART_IMAGELIST_FRAMES_MISSING, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "the cube seems empty" );
        return NULL;
    }

    image = cpl_image_load( incube->filename, CPL_TYPE_DOUBLE, 0, extension );
    if ( !image ) {
        sph_error_raise( SPH_SMART_IMAGELIST_GENERAL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not load image from first frame." );
        return NULL;
    }

    self->nx = cpl_image_get_size_x(image);
    self->ny = cpl_image_get_size_y(image);

    nframes = cpl_table_get_nrow( incube->assoc_table );

    nregions = 1;
    while ( (long long int)sizeof(double) * (long long int)self->nx * (long long int)self->ny * (long long int)nframes >
                   self->maxmem * (long long int)nregions * (long long int)nregions ) {
        nregions = nregions * 2;
    }
    if ( nregions >= self->nx ) {
        sph_error_raise( SPH_SMART_IMAGELIST_NO_SCHEME_FOUND, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not find a good division scheme." );
        cpl_image_delete(image); image = NULL;
        return NULL;
    }

    self->regions = cpl_calloc( nregions * nregions , sizeof(sph_detector_region));

    if ( self->nx % nregions || self->ny % nregions ) {
        sph_error_raise( SPH_SMART_IMAGELIST_BAD_SCHEME, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Something went wrong when determining the division scheme." );
        cpl_image_delete(image); image = NULL;
        return NULL;
    }

    sx = self->nx / nregions;
    sy = self->ny / nregions;

    self->regions[0].ll_x = 0;
    self->regions[0].ll_y = 0;
    self->regions[0].ur_y = sy;
    self->regions[0].ur_x = sx;

    for ( yy = 0; yy < nregions; yy++ ) {
        for ( xx = 0; xx < nregions; xx++ ) {
            self->regions[ yy * nregions + xx ].ll_x = xx * sx;
            self->regions[ yy * nregions + xx ].ll_y = yy * sy;
            self->regions[ yy * nregions + xx ].ur_x = xx * sx + sx - 1;
            self->regions[ yy * nregions + xx ].ur_y = yy * sy + sy - 1;
        }
    }
    self->nregions = nregions * nregions;
    cpl_image_delete(image); image = NULL;
    return self;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Return the batch of images that correspond to the region number
 * specified.
 *
 * @param set   The number of the region, starting from the top left with 0
 *
 * @return pointer to newly created cpl_imagelist or NULL on error
 *
 * This creates a new empty cpl_imagelist where the images correspond to the
 * region as identified by the number in all frames in the current frameset of
 * the sph_smart_imagelist.
 *
 */
/*----------------------------------------------------------------------------*/

cpl_imagelist*  sph_smart_imagelist_get_set(sph_smart_imagelist* self,
                                            int set)
{
    cpl_imagelist*          result      = NULL;
    int                     ii          = 0;
    int                        jj            = 0;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL

    if ( self == NULL || self->frameset == NULL || self->regions == NULL ) {
        SPH_NULL_ERROR;
        return NULL;
    }

    if ( set <0 || set >= self->nregions ) {
        sph_error_raise( SPH_SMART_IMAGELIST_NO_FRAMES, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "There are no frames" );
        return NULL;
    }

    result = cpl_imagelist_new();
    if ( result == NULL ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Some problem creating the imagelist" );
        return NULL;
    }

    if ( self->cubemode ) {
        for ( ii = 0; ii < cpl_frameset_get_size(self->frameset ); ii++) {
            cpl_imagelist* dummylist;
            self->manyplaneframe = cpl_frameset_get_position_const(self->frameset,
                                                                   ii);
            dummylist = sph_smart_imagelist_get_set_manyplanes(self, set);
            if ( !dummylist ) {
                cpl_imagelist_delete(result); result = NULL;
                return NULL;
            }
            for ( jj = 0; jj < cpl_imagelist_get_size(dummylist); jj++ )
            {
                cpl_imagelist_set( result, cpl_image_duplicate(cpl_imagelist_get(dummylist,jj)),
                                   cpl_imagelist_get_size(result) );
                if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
                    cpl_imagelist_delete(dummylist); dummylist = NULL;
                    cpl_imagelist_delete(result); result = NULL;
                    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
                }
            }
            cpl_imagelist_delete(dummylist); dummylist = NULL;
        }
    }
    else {
        for ( ii = 0; ii < cpl_frameset_get_size(self->frameset ); ii++) {
            const cpl_frame* frame =
                cpl_frameset_get_position_const( self->frameset, ii );
            cpl_image* image =
                cpl_image_load_window( cpl_frame_get_filename( frame ),
                                       CPL_TYPE_DOUBLE, 0, self->extension,
                                       self->regions[set].ll_x + 1,
                                       self->regions[set].ll_y + 1,
                                       self->regions[set].ur_x + 1,
                                       self->regions[set].ur_y + 1);
            if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
                sph_error_raise( cpl_error_get_code(),
                                 __FILE__, __func__,
                                 __LINE__,
                                 SPH_ERROR_ERROR,"%s",
                                 cpl_error_get_message() );
                cpl_image_delete(image);
                cpl_imagelist_delete(result); result = NULL;
                return NULL;
            }
            cpl_imagelist_set( result, image, ii );
        }
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL
        return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return the batch of images that correspond to the region number
 * specified.
 *
 * @param set   The number of the region, starting from the top left with 0
 *
 * @return pointer to newly created cpl_imagelist or NULL on error
 *
 * This creates a new empty cpl_imagelist where the images correspond to the
 * region as identified by the number in all planes in the current cube of
 * the sph_smart_imagelist.
 *
 */
/*----------------------------------------------------------------------------*/

cpl_imagelist*
sph_smart_imagelist_get_set_manyplanes(const sph_smart_imagelist* self,
                                       int set)
{
    cpl_imagelist* result = NULL;
    const char*    filename;
    int            nplanes;

    if (self == NULL || self->manyplaneframe == NULL || self->regions == NULL) {
        SPH_NULL_ERROR;
        return NULL;
    }

    if (set < 0 || set >= self->nregions) {
        sph_error_raise( SPH_SMART_IMAGELIST_NO_FRAMES, __FILE__,
                         __func__, __LINE__, SPH_ERROR_ERROR,
                         "There is no plane with no %d in %d-cube",
                         set, self->nregions);
        return NULL;
    }

    filename = cpl_frame_get_filename(self->manyplaneframe);
    nplanes = sph_fits_get_nplanes(filename, self->extension);

    if (nplanes < 0) {
        (void)cpl_error_set_where(cpl_func);
    } else {
        result = cpl_imagelist_new();
        for (int ii = 0; ii < nplanes; ii++) {
            cpl_image* image = cpl_image_load_window( filename,
                                           CPL_TYPE_DOUBLE, ii, self->extension,
                                           self->regions[set].ll_x + 1,
                                           self->regions[set].ll_y + 1,
                                           self->regions[set].ur_x + 1,
                                           self->regions[set].ur_y + 1);

            if (cpl_imagelist_set(result, image, ii)) {
                (void)cpl_error_set_where(cpl_func);
                cpl_imagelist_delete(result); result = NULL;
                break;
            }
        }
    }

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Return the batch of images that correspond to the region number
 * specified.
 *
 * @param set   The number of the region, starting from the top left with 0
 *
 * @return pointer to newly created cpl_imagelist or NULL on error
 *
 * This creates a new empty cpl_imagelist where the images correspond to the
 * region as identified by the number in all planes in the current cube of
 * the sph_smart_imagelist.
 *
 */
/*----------------------------------------------------------------------------*/

cpl_imagelist*  sph_smart_imagelist_get_set_cube( sph_smart_imagelist* self,
                                             int set
                                            )
{
    cpl_imagelist*          result      = NULL;
    cpl_image*              image       = NULL;
    int                     ii          = 0;

    if ( self == NULL || self->cube == NULL || self->regions == NULL ) {
        SPH_NULL_ERROR;
        return NULL;
    }

    if ( set <0 || set >= self->nregions ) {
        sph_error_raise( SPH_SMART_IMAGELIST_NO_FRAMES, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "There is no region with no %d in cube", set );
        return NULL;
    }

    result = cpl_imagelist_new();
    if ( result == NULL ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Some problem creating the imagelist" );
        return NULL;
    }

    for ( ii = 0; ii < cpl_table_get_nrow( self->cube->assoc_table ); ii++) {
        image = cpl_image_load_window( self->cube->filename,
                                       CPL_TYPE_DOUBLE, ii, self->extension,
                                       self->regions[set].ll_x + 1,
                                       self->regions[set].ll_y + 1,
                                       self->regions[set].ur_x + 1,
                                       self->regions[set].ur_y + 1);
        if ( !image ) {
            sph_error_raise( cpl_error_get_code(),
                             __FILE__, __func__,
                             __LINE__,
                             SPH_ERROR_ERROR,"%s",
                             cpl_error_get_message() );
            return NULL;
        }
        cpl_imagelist_set( result, image, ii );
    }

    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Re-assemble a full sph_master_frame from sph_master_frames definied
 * in the regions
 *
 * @param self          The smart imagelist
 * @param image_arr     The list of sub-master frames to assamble
 *
 * @return pointer to newly created sph_master_frame of the assembled image
 *
 * This creates a new sph_master_frame which has been assembled from the array
 * of input sph_master_frames. The size of the array of the
 * subimages must be as large as the
 * number of regions defined in the sph_smart_imagelst.
 * Note that the master frames in image_arr are not needed anymore after this function
 * is called (ownership of the data in the image_arr mater_frames is @emph not
 * transferred but the data is copied) so the input array can be deleted safely
 * after
 *
 */
/*----------------------------------------------------------------------------*/
sph_master_frame*
sph_smart_imagelist_reassemble_master_frame( const sph_smart_imagelist* self,
                                             sph_master_frame** image_arr)
{
    sph_master_frame*       result      = NULL;
    sph_master_frame*       cmf         = NULL;
    cpl_image*              resultim    = NULL;
    cpl_image*              image       = NULL;
    cpl_imagelist*          imlist      = NULL;
    int                     ii          = 0;
    int                     extno       = 0;
    int                     cc          = 0;
    int                     maxc        = 0;
    cpl_error_code cpl_entry_err = cpl_error_get_code();

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure(self,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(image_arr,CPL_ERROR_NULL_INPUT,NULL);
    cpl_ensure(self->regions,CPL_ERROR_NULL_INPUT,NULL);

    /* First do the image plane */
    result = sph_master_frame_new_empty();
    cpl_ensure(result,CPL_ERROR_ILLEGAL_OUTPUT,NULL);
    for ( extno = 0; extno < 4; ++extno ) {
        imlist = cpl_imagelist_new();
        if ( imlist == NULL || result == NULL ) {
            sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__, __func__, __LINE__,
                             SPH_ERROR_ERROR,
                             "Could not allocate memory for imlist..."
            );
            sph_master_frame_delete( result );
            return NULL;
        }

        for (cc = 0; cc < self->nregions; ++cc) {
            cmf = image_arr[cc];
            if ( cmf == NULL ) {
                sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                                 SPH_ERROR_ERROR,
                                 "There are elements in the master_frame array missing!");
                if ( cc == 0 ) {
                    cpl_imagelist_delete( imlist );
                }
                sph_master_frame_delete( result );
                return NULL;
            }
            if ( extno == 0 ) {
                cpl_imagelist_set( imlist, cmf->image, cc );
            }
            if ( extno == 1 ) {
                cpl_imagelist_set( imlist, cmf->badpixelmap, cc );
            }
            if ( extno == 2 ) {
                cpl_imagelist_set( imlist, cmf->rmsmap, cc );
            }
            if ( extno == 3 ) {
                cpl_imagelist_set( imlist, cmf->ncombmap, cc );
            }
        }

        if ( extno == 0 ) {
            resultim = cpl_image_new( self->nx, self->ny, cpl_image_get_type( image_arr[0]->image ) );
        }
        if ( extno == 1 ) {
            resultim = cpl_image_new( self->nx, self->ny, cpl_image_get_type( image_arr[0]->badpixelmap ) );
        }
        if ( extno == 2 ) {
            resultim = cpl_image_new( self->nx, self->ny, cpl_image_get_type( image_arr[0]->rmsmap ) );
        }
        if ( extno == 3 ) {
            resultim = cpl_image_new( self->nx, self->ny, cpl_image_get_type( image_arr[0]->ncombmap ) );
        }

        if ( resultim == NULL ) {
            sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                             __func__, __LINE__,
                             SPH_ERROR_ERROR, "Some problem creating the image" );
            for (cc = 0; cc < maxc; ++cc) {
                cpl_imagelist_unset( imlist, 0 );
            }
            cpl_imagelist_delete( imlist );
            sph_master_frame_delete( result );
            return NULL;
        }
        for ( ii = 0; ii < cpl_imagelist_get_size( imlist ); ii++) {
            image = cpl_imagelist_get( imlist, ii );
            cpl_image_copy ( resultim, image, self->regions[ii].ll_x + 1, self->regions[ii].ll_y + 1);
        }
        if ( extno == 0 ) {
            result->image = resultim;
        }
        if ( extno == 1 ) {
            result->badpixelmap = resultim;
        }
        if ( extno == 2 ) {
            result->rmsmap = resultim;
        }
        if ( extno == 3 ) {
            result->ncombmap = resultim;
        }
        if ( cpl_error_get_code() != cpl_entry_err ) {
            SPH_RAISE_CPL
            for (cc = 0; cc < maxc; ++cc) {
                cpl_imagelist_unset( imlist, 0 );
            }
            cpl_imagelist_delete( imlist );
            sph_master_frame_delete( result );
            return NULL;
        }
        maxc = cpl_imagelist_get_size(imlist);
        for (cc = 0; cc < maxc; ++cc) {
            cpl_imagelist_unset( imlist, 0 );
        }
        cpl_imagelist_delete( imlist );
    }
    if ( cpl_error_get_code() != cpl_entry_err ) {
        SPH_RAISE_CPL;
        sph_master_frame_delete( result );
        return NULL;
    }
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete the smart imagelist and de-allocate the space
 *
 * @param self   The smart imagelist
 *
 * @return error code of the operation or 0 otherwise
 *
 * This deletes the sph_smart_imagelist. and de-allocates the space.
 *
 */
/*----------------------------------------------------------------------------*/

int sph_smart_imagelist_delete( sph_smart_imagelist* self) {
    if ( self ) {
        if ( self->regions ) {
            cpl_free( self->regions );
        }

        cpl_free( self );
    }
    return CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Setting the frameset

 @param self        the smart_imagelist
 @param inframes    the frameset to set.

 @return error code

 Description:
 Set the frameset -- note that frames are not duplicated, so dont delete
 the frameset without deleting the sph_smart_imagelist first!

 */
/*----------------------------------------------------------------------------*/
static
sph_error_code sph_smart_imagelist_set_frames(sph_smart_imagelist* self,
                                              const cpl_frameset* inframes )
{
    const cpl_frame* frame = cpl_frameset_get_first_const(inframes);

    if (frame != NULL) {
        const char* filename = cpl_frame_get_filename(frame);
        cpl_image* image = cpl_image_load(filename ,
                                          CPL_TYPE_DOUBLE, 0, self->extension);

        if (image != NULL) {
            cpl_propertylist* pl = cpl_propertylist_load_regexp(filename,
                                                                self->extension,
                                                                "^NAXIS",
                                                                CPL_FALSE);
            const int naxis = cpl_propertylist_get_int(pl, "NAXIS");

            cpl_propertylist_delete(pl);

            if (!cpl_error_get_code()) {

                if (naxis == 3) {
                    cpl_msg_info(cpl_func, "Framecombination in cube mode: %s",
                                 filename);
                    self->cubemode = 1;
                }

                self->frameset = inframes;
                self->nx = cpl_image_get_size_x(image);
                self->ny = cpl_image_get_size_y(image);
            }
            cpl_image_delete(image);
        }
    }

    return cpl_error_get_code()
        ? cpl_error_set_message(cpl_func, cpl_error_get_code(), "Extension=%d. "
                                "Number of frames: %d",
                                self ? self->extension : -1,
                                inframes ? (int)cpl_frameset_get_size(inframes)
                                : 0)
        : CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 @brief Function to calculate the number of subdivisions

 @param self        the sph_smart_imagelist

 @return number of regions

 Description:
 Calculates the number of subdivision regions, by taking the maximal
 memory and the size of the sph_smart_imagelist in x and y pixels.
 The availaible maximal memory is currently not calculated dynamically
 but taken from the corresponding field in the sph_smart_imagelist
 structure.

 */
/*----------------------------------------------------------------------------*/
static
int sph_smart_imagelist_calc_subdivision( sph_smart_imagelist* self ) {
    int nregions    = 1;
    int    nframes        = sph_common_science_get_nraws(self->frameset,self->extension);

    while ( (long long int)sizeof(double) * (long long int)self->nx * (long long int)self->ny * (long long int)nframes >
    self->maxmem * (long long int)nregions * (long long int)nregions ) {
        nregions = nregions * 2;
    }
    cpl_msg_info(cpl_func,"For a max. memory of %ld bytes and %d frames of %d x %d pixels, determined a subdivision into %d regions!",self->maxmem,nframes,self->nx,self->ny,nregions);
    if ( nregions >= self->nx ) {
        sph_error_raise( SPH_SMART_IMAGELIST_NO_SCHEME_FOUND, __FILE__,
                __func__, __LINE__,
                SPH_ERROR_ERROR,
                "Could not find a good division scheme "
                "for the sph_smart_imagelist. This probably means "
                "there is no free memory at all !" );
        return -1;
    }
    // Ooopssss.... The standard IFS lenslet cube has nx = 291 pixels so divisions of e.g 2 are bad. Treat this as special case:
    if ( self->nx == 291 ) {
        // Two cases:
        if (nregions > 1 && nregions < 6) nregions = 3; // High memory -> 3 regions are OK
        if (nregions > 5) nregions = 97; // Low memory -> many tiny regions
    }

    if ( self->nx % nregions || self->ny % nregions ) {
        sph_error_raise( SPH_SMART_IMAGELIST_BAD_SCHEME, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "Something went wrong when determining "
                         "the division scheme. Contact a pipeline "
                         "developer." );
        return -1;
    }
    return nregions;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new sph_smart_imagelist from the given frameset
 *
 * @return pointer to newly created smart_imagelist or NULL on error
 *
 * This creates a new empty smart_imagelist.
 */
/*----------------------------------------------------------------------------*/
sph_smart_imagelist* sph_smart_imagelist_create(const cpl_frameset* inframes,
                                                unsigned int extension ) {
    sph_smart_imagelist*    self        = NULL;
    int                     nregions    = 1;
    int                     sx          = 0;
    int                     sy          = 0;
    int                     xx          = 0;
    int                     yy          = 0;



    self = sph_smart_imagelist_new();
    if ( !self ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR,
                         "could not allocate sph_smart_imagelist" );
        return NULL;
    }

    self->extension = extension;

    if ( sph_smart_imagelist_set_frames(self, inframes) ) {
        sph_smart_imagelist_delete( self );
        return NULL;
    }

    nregions = sph_smart_imagelist_calc_subdivision( self );
    if ( nregions < 0 ) {
        sph_smart_imagelist_delete( self );
        return NULL;
    }
    self->regions = cpl_calloc( nregions * nregions , sizeof( sph_detector_region ) );

    sx = self->nx / nregions;
    sy = self->ny / nregions;

    self->regions[0].ll_x = 0;
    self->regions[0].ll_y = 0;
    self->regions[0].ur_y = sy;
    self->regions[0].ur_x = sx;

    for ( yy = 0; yy < nregions; yy++ ) {
        for ( xx = 0; xx < nregions; xx++ ) {
            self->regions[ yy * nregions + xx ].ll_x = xx * sx;
            self->regions[ yy * nregions + xx ].ll_y = yy * sy;
            self->regions[ yy * nregions + xx ].ur_x = xx * sx + sx - 1;
            self->regions[ yy * nregions + xx ].ur_y = yy * sy + sy - 1;
        }
    }
    self->nregions = nregions * nregions;
    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new propertylist with statistic of raw frame medians
 * @param self      the smart imagelist
 * @param pl        the propertylist to append to
 *
 * @return error code
 *
 * This function creates properties relating to the medians of the raw
 * input frames used for the smart_imagelist. All raw frames set in
 * the the smart imagelist are analysed to give a list of medians which
 * is then used to derive the average median of the raw frames, the median
 * of the medians, the
 * minimum and maximum medians present in the set of raw frames and the
 * range of values.
 * The new properties are appended to the input propertylist.
 *
 */
/*----------------------------------------------------------------------------*/

sph_error_code sph_smart_imagelist_get_raw_median_properties( sph_smart_imagelist* self,
        cpl_propertylist* pl) {
    int             ff          = 0;
    cpl_vector*     medians       = NULL;
    int             nelements = 0;
    int             celement = 0;
    int             pp = 0;

    cpl_ensure_code(self,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(pl,CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(self->frameset,CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_frameset_get_size(self->frameset) > 0,CPL_ERROR_ILLEGAL_INPUT);
    if ( self->cubemode ) {
        for (ff = 0; ff < cpl_frameset_get_size(self->frameset); ++ff) {
            const cpl_frame* aframe =
                cpl_frameset_get_position_const(self->frameset, ff);
            const char* filename = cpl_frame_get_filename(aframe);
            cpl_propertylist* dumlist =
                cpl_propertylist_load(filename, self->extension);

            if ( cpl_propertylist_has(dumlist,"NAXIS3") ) {
                nelements += cpl_propertylist_get_int(dumlist,"NAXIS3");
            }
            else nelements++;
            cpl_propertylist_delete(dumlist);
        }
    }
    else {
        nelements = cpl_frameset_get_size(self->frameset);
    }
    medians = cpl_vector_new(nelements);
    cpl_ensure_code(medians,cpl_error_get_code());

    if ( self->cubemode ) {
        for (ff = 0; ff < cpl_frameset_get_size(self->frameset); ++ff) {
            const cpl_frame* aframe =
                cpl_frameset_get_position_const(self->frameset, ff);
            const char* filename = cpl_frame_get_filename(aframe);
            cpl_propertylist* dumlist =
                cpl_propertylist_load(filename, self->extension);

            if ( cpl_propertylist_has(dumlist,"NAXIS3") ) {
                nelements = cpl_propertylist_get_int(dumlist,"NAXIS3");
            }
            else nelements=1;
            cpl_propertylist_delete(dumlist);
            for (pp = 0; pp < nelements; ++pp) {
                cpl_image*      im = 
                    cpl_image_load(filename, CPL_TYPE_FLOAT, pp, self->extension);
                cpl_ensure_code(im,cpl_error_get_code());
                cpl_vector_set(medians,celement,cpl_image_get_median(im));
                cpl_image_delete(im);
                celement++;
            }
        }
    }
    else {
        for (ff = 0; ff < cpl_frameset_get_size(self->frameset); ++ff) {
            const cpl_frame* aframe =
                cpl_frameset_get_position_const(self->frameset, ff);
            const char* filename = cpl_frame_get_filename(aframe);
            cpl_image*      im =
                cpl_image_load(filename, CPL_TYPE_FLOAT, 0, self->extension);

            cpl_ensure_code(im, cpl_error_get_code());
            cpl_vector_set(medians,ff,cpl_image_get_median(im));
            cpl_image_delete(im);
        }
    }

    cpl_msg_info(cpl_func, "Number of elements for RAW MEDIANS: %i", nelements);

    for(cpl_size i = 0; i < nelements; ++i){
        cpl_msg_info(cpl_func, "Medians element: %lld -> %g", i, cpl_vector_get(medians, i));
    }

    cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_RAW_MEDIANS_AVG,cpl_vector_get_mean(medians));
    cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_RAW_MEDIANS_MED,cpl_vector_get_median(medians));
    cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_RAW_MEDIANS_MIN,cpl_vector_get_min(medians));
    cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_RAW_MEDIANS_MAX,cpl_vector_get_max(medians));
    if ( cpl_vector_get_size(medians)> 1 )
    cpl_propertylist_update_double(pl,SPH_COMMON_KEYWORD_RAW_MEDIANS_RANGE,cpl_vector_get_stdev(medians));
    cpl_vector_delete(medians);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/**@}*/
