/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#include "sph_spectrum.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_dataset.h"
#include "sph_fitting.h"
#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>


/* Private functions declaration */
static cpl_polynomial *
get_poly_from_lines(sph_spectrum * self,
		cpl_vector * calib_lines, const int tol, const int degree);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_spectrum Spectrum object
 *
 * @par Synopsis:
 * @code
 *   typedef struct _sph_spectrum_
{
    short                  lensid; // lenslet id
    short                   specid; // spectrum id
    cpl_vector*            spec;
    cpl_vector*            wavelength;
} sph_spectrum;
 *
 * @endcode
 *
 * @par Description:
 * This object represents an individual spectrum. It consists of
 * a region in absolute pixel coordinates where the spectrum is
 * defined, and a cpl_image with the region.
 * The spectrum structure is a one dimensional representation
 * of a spectrum on the detector -- with the relevant information of the
 * two axis (wavelength and value) being saved in two vectors, spec and wavelength,
 * with the same size.
 *
 * The functions in this module allow spectra to be created and extraced from
 * detector images.
 *
 *
 */
/*----------------------------------------------------------------------------*/
/**@{*/

sph_error_code SPH_SPECTRUM_GENERAL          = SPH_SPECTRUM_ERR_START + 0;
sph_error_code SPH_SPECTRUM_CANT_COLLAPSE    = SPH_SPECTRUM_ERR_START + 1;
sph_error_code SPH_SPECTRUM_OUT_OF_BOUNDS    = SPH_SPECTRUM_ERR_START + 2;
sph_error_code SPH_SPECTRUM_NOT_COLLAPSED    = SPH_SPECTRUM_ERR_START + 3;


/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new (empty) spectrum
 * @param   npoints            number of spectra points ( must be >=2)
 * @param    minlambda        minimum wavelength to assign
 * @param   maxlambda        maximum wavelength to assign
 *
 * @return pointer to newly created spectrum or NULL on error
 *
 * This creates a new empty spectrum. It sets up a vector of npoints ranging
 * from minlambda to maxlambda. The width of each point is
 * maxlambda-minlambda / (npoints-1).
 */
/*----------------------------------------------------------------------------*/
sph_spectrum*
sph_spectrum_new( int npoints, double minlambda, double maxlambda ) {
    sph_spectrum*   self    = NULL;
    double          dx      = 0.0;
    int             ii      = 0;

    self = cpl_calloc(1, sizeof(sph_spectrum));

    if ( !self ) {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not allocate spectrum" );
        return NULL;
    }
    if ( npoints < 2 ) {
        sph_error_raise( SPH_ERROR_INCONSISTENT_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_WARNING, "Cant create "
                                 "spectrum with 1 or 0 points!" );
        return NULL;
    }
    self->wavelengths = cpl_vector_new( npoints );
    self->spec = cpl_vector_new( npoints );

    self->bads = cpl_vector_new( npoints );
    self->rms = cpl_vector_new( npoints );
    self->ncomb = cpl_vector_new( npoints );
    cpl_vector_fill(self->spec,0.0);
    cpl_vector_fill(self->bads,0.0);
    cpl_vector_fill(self->rms,0.0);
    cpl_vector_fill(self->ncomb,0.0);

    dx = ( maxlambda - minlambda ) / ( npoints - 1 );
    if ( dx == 0.0 ) {
        dx = 1.0;
    }
    for ( ii = 0; ii < cpl_vector_get_size( self->spec ); ++ii) {
        cpl_vector_set( self->wavelengths, ii, minlambda + dx * ii );
    }

    return self;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Create a new spectrum loading it from a FITS binary table
 *
 * @param czFilename        the filename of the FITS table to load from
 *
 * @return pointer to newly created spectrum or NULL on error
 *
 * This creates a new spectrum from the specified FITS filename. The table
 * must be in the 1st extension.
 */
/*----------------------------------------------------------------------------*/
sph_spectrum* sph_spectrum_load( const char* czFilename ) {
    sph_spectrum*          self         = NULL;
    cpl_table*              tab         = NULL;
    int                     npoints     = 0;
    double                  lmin        = 0.0;
    double                  lmax        = 0.0;
    int                     rej         = 0;
    int                     ii          = 0;

    tab = cpl_table_load( czFilename, 1, 0 );
    if ( tab == NULL ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "Could not load the spectra table from file.");
        return NULL;
    }
    if ( cpl_table_get_ncol( tab ) != 5 ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "Spectra table does not have 5 columns!");
        return NULL;
    }
    npoints = cpl_table_get_nrow( tab );
    if ( npoints < 2 ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "Spectra table has less than 2 rows!");
        return NULL;
    }
    lmin = cpl_table_get( tab, "Wavelength", 0, &rej );
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "Could not get the minimum lambda from table."
                         "Check that the \"Wavelength\" column is there.");
        return NULL;
    }
    lmax = cpl_table_get( tab, "Wavelength", npoints - 1, &rej );
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise( SPH_ERROR_GENERAL, __FILE__, __func__,
                         __LINE__, SPH_ERROR_ERROR,
                         "Could not get the minimum lambda from table."
                         "Check that the \"Wavelength\" column is there.");
        return NULL;
    }
    self = sph_spectrum_new( npoints, lmin, lmax );

    if ( self ) {
        for (ii = 0; ii < npoints; ++ii) {
            rej = 0;
            const double val = cpl_table_get(tab, "Flux", ii, &rej );
            const double rms = cpl_table_get(tab, "RMS", ii, &rej );
            const double ncomb = cpl_table_get(tab, "Ncomb", ii, &rej );
            rej = rej + cpl_table_get(tab, "Bad", ii, &rej );
            if ( rej == 0 ) {
                cpl_vector_set( self->spec, ii, val);
                cpl_vector_set( self->bads, ii, 0);
                cpl_vector_set( self->rms, ii, rms);
                cpl_vector_set( self->ncomb, ii, ncomb);
            }
            else {
                cpl_vector_set( self->spec, ii, 0.0);
                cpl_vector_set( self->bads, ii, 1);
            }
        }
    }
    cpl_table_delete(tab); tab = NULL;
    return self;

}

/*----------------------------------------------------------------------------*/
/**
 @brief Convert the spectrum to a sph_dataset_structure

 @param the spectrum to convert

 @return the new dataset

 Description:
 Convert the spectrum to a dataset. In principle the spectrum could be implemnted
 as a child "class" of dataset -- since we code in C, use this apprach for now.

 */
/*----------------------------------------------------------------------------*/
sph_dataset*
sph_spectrum_convert_to_dataset( sph_spectrum* spec ) {
    sph_dataset*        result = NULL;
    int                    npoints    = 0;

    if ( !spec ) {
        SPH_NULL_ERROR;
        return NULL;
    }
    npoints = cpl_vector_get_size( spec->wavelengths );
    result = sph_dataset_new( npoints, cpl_vector_get( spec->wavelengths, 0),
            cpl_vector_get( spec->wavelengths, npoints-1) );
    if ( !result ) {
        return NULL;
    }
    cpl_vector_delete( result->xvalues );
    result->xvalues = cpl_vector_duplicate( spec->wavelengths );
    cpl_vector_delete( result->yvalues );
    result->yvalues = cpl_vector_duplicate( spec->spec);
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Fit a gaussian to the 1-D spectrum
 *
 * @param self  The spectrum to fit
 * @param x0    The best fit peak position
 * @param sigma The best fit sigma of the gaussian
 * @param chi   The reduced chi-squared of the best fit
 *
 * @return Error code of the operation
 *
 * A gaussian fit to the 1D spectrum is performed and the best fit values
 * returned
 *
 */
/*----------------------------------------------------------------------------*/
int sph_spectrum_fit_gauss( sph_spectrum* self, double* x0, double* sigma, double* chi ) {
    int             rerr    = CPL_ERROR_NONE;
    double          area    = 0.0;
    double          offset  = 0.0;

    if ( !self ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "No self." );
        return CPL_ERROR_NULL_INPUT;
    }

    if ( !self->spec ) {
        sph_error_raise( SPH_SPECTRUM_NOT_COLLAPSED, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "First collapse the spectrum!" );
        return SPH_SPECTRUM_NOT_COLLAPSED;
    }
    if ( !self->wavelengths ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                __func__, __LINE__,
                SPH_ERROR_ERROR, "could not set 1D spectrum" );
        return CPL_ERROR_NULL_INPUT;
    }


    rerr = cpl_vector_fit_gaussian( self->wavelengths, NULL, self->spec,
                             NULL, CPL_FIT_ALL, x0, sigma,
                             &area, &offset, NULL, chi, NULL);

    return rerr ? cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief get the value at a given wavelength
 *
 * @param self      The spectrum
 * @param wavelength    The wavelength
 *
 * @return value of the spectrum, -1 on error.
 *
 * Returns the value of the spectrum at the given wavelength. If the
 * wavelength requested is outside the boundaries, the routine will
 * return -1 and rais an error.
 */
/*----------------------------------------------------------------------------*/
double sph_spectrum_get_value( sph_spectrum* self, double value ) {
    int                ind        = 0;
    double          result  = 0.0;

    if ( !self ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "No self." );
        return CPL_ERROR_NULL_INPUT;
    }

    if ( !self->spec ) {
            sph_error_raise( SPH_SPECTRUM_NOT_COLLAPSED, __FILE__,
                             __func__, __LINE__,
                             SPH_ERROR_ERROR, "First collapse the spectrum!" );
            return SPH_SPECTRUM_NOT_COLLAPSED;
    }
    if ( !self->wavelengths ) {
        sph_error_raise( CPL_ERROR_NULL_INPUT, __FILE__,
                __func__, __LINE__,
                SPH_ERROR_ERROR, "could not set 1D spectrum" );
        return CPL_ERROR_NULL_INPUT;
    }

    if ( value > cpl_vector_get_max( self->wavelengths) || value < cpl_vector_get_min( self->wavelengths) ) {
        sph_error_raise( SPH_SPECTRUM_OUT_OF_BOUNDS,
                        __FILE__, __func__,
                        __LINE__, SPH_ERROR_ERROR,
                        "The wavelength specified was out of bounds.");
        return -1;
    }

    ind = cpl_vector_find( self->wavelengths, value );
    if ( ind < 0 || ind >= cpl_vector_get_size( self->spec ) ) {
        sph_error_raise( SPH_SPECTRUM_OUT_OF_BOUNDS,
                        __FILE__, __func__,
                        __LINE__, SPH_ERROR_ERROR,
                        "The wavelength specified was out of bounds --"
                        "no coressponding value found.");
        return -1;
    }
    result = cpl_vector_get( self->spec, ind );

    return result;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Save the spectrum.
  @param    self        the spectrum to save
  @param    filename    the filename to save as
  @return   0 on succes, -1 if input pointer was NULL.

  Save the spectrum as a FITS table.
 */
/*----------------------------------------------------------------------------*/
int sph_spectrum_save( sph_spectrum* self, const char* filename) {
    cpl_table*          tab     = NULL;
    int                 ii      = 0;
    int                 np      = 0;

    if ( self == NULL ) {
        return -1;
    }


    np = cpl_vector_get_size( self->wavelengths );

    tab = cpl_table_new( np );

    cpl_table_new_column( tab, "Wavelength", CPL_TYPE_DOUBLE );
    cpl_table_new_column( tab, "Flux", CPL_TYPE_DOUBLE );
    cpl_table_new_column( tab, "Bad", CPL_TYPE_INT );
    cpl_table_new_column( tab, "RMS", CPL_TYPE_DOUBLE );
    cpl_table_new_column( tab, "Ncomb", CPL_TYPE_DOUBLE );

    for (ii = 0; ii < np; ++ii) {
        cpl_table_set( tab, "Wavelength", ii, cpl_vector_get( self->wavelengths, ii ) );
        cpl_table_set( tab, "Flux", ii, cpl_vector_get( self->spec, ii ) );
        cpl_table_set( tab, "Bad", ii, (int)cpl_vector_get( self->bads, ii ) );
        cpl_table_set( tab, "RMS", ii, cpl_vector_get( self->rms, ii ) );
        cpl_table_set( tab, "Ncomb", ii, cpl_vector_get( self->ncomb, ii ) );
    }

    cpl_table_save( tab, NULL, NULL, filename, CPL_IO_DEFAULT );
    /* TODO: handle propertylist */
    cpl_table_delete(tab); tab = NULL;
    return 0;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Dump the spectrum.
  @param    self        the spectrum to dump
  @return   0 on succes, -1 if input pointer was NULL.

  Dump the spectrum to stdout.
 */
/*----------------------------------------------------------------------------*/
int sph_spectrum_dump( sph_spectrum* self ) {
    int                 ii      = 0;
    int                 np      = 0;

    if ( self == NULL ) {
        return -1;
    }


    np = cpl_vector_get_size( self->wavelengths );

    printf( "Point\tLambda\tFlux\tBad\tRMS\tNcomb\n");
    for (ii = 0; ii < np; ++ii) {
        printf( "%5d\t%5.3f\t%5.3f\t%5d\t%5.3f\t%5.3f\n", ii,
                cpl_vector_get( self->wavelengths, ii),
                cpl_vector_get( self->spec, ii ),
                (int)cpl_vector_get( self->bads, ii ),
                cpl_vector_get( self->rms, ii ),
                cpl_vector_get( self->ncomb, ii ));
    }

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief Return the resolving power
 * @param self  the spectrum
 *
 * @return double of the mean resolving power
 *
 * This function calculates sum_lambda/dlambda
 */
/*----------------------------------------------------------------------------*/
double sph_spectrum_get_resolving_power( sph_spectrum* self ) {
    int                 ii      = 0;
    int                 np      = 0;
    cpl_vector*         resol   = NULL;
    double              lam = 0.0;
    double              dlam = 0.0;
    double 				result = 0.0;

    cpl_ensure( self, CPL_ERROR_NULL_INPUT, 0.0 );
    np = cpl_vector_get_size( self->wavelengths );
    resol = cpl_vector_new(np-1);
    cpl_vector_fill(resol,0.0);

    for (ii = 0; ii < np - 1; ++ii) {
    	lam = cpl_vector_get( self->wavelengths, ii);
    	lam = lam + cpl_vector_get( self->wavelengths, ii + 1);
    	lam = lam / 2.0;
    	dlam = fabs(cpl_vector_get( self->wavelengths, ii + 1) -
    			cpl_vector_get( self->wavelengths, ii));
    	if ( (int)cpl_vector_get( self->bads, ii ) == 0 &&
    	     (int)cpl_vector_get( self->bads, ii + 1 ) == 0 &&
    	     dlam > 0.0)
    	{
    		cpl_vector_set( resol, ii, lam / dlam);
    	}
    	else {
    		cpl_vector_set(resol,ii,0.0);
    	}
    }

    result = cpl_vector_get_median(resol);
    cpl_vector_delete(resol); resol = NULL;
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Fit the wavelengths of spectrum new
 * @param self  the spectrum to refit the wavelengths for
 * @param calib_lines    the calibration lines to apply
 * @param degree the fitting degree to use
 * @param tol    the maximum distance from guess line to search
 * @param pdiffs pointer differences (output, NULL allowed)
 * @param pderiv derivate of polynomial (output, NULL allowed)
 * @param coeffs preallocated vector to hold fit coeff (or NULL)
 *
 * @return new wavelength vector
 *
 * This function fits the number of calib_lines to the spectrum. It assumes
 * that the spectrum contains wavelength calibration data (i.e.
 * a set of clearly distinguished lines). If provided, the diffs
 * pointer points to a new cpl_vector that contains the differences
 * of each point along the spectrum between new and old wavelengths.
 * The number of elements in the returned vector and in diffs will be the same
 * as the number of points in the original spectrum.
 * If a coeffs vector is supplied it has to have the same
 * size as the degree. On output it will then be filled with
 * the fitting coefficients of the polynomial.
 */
/*----------------------------------------------------------------------------*/
cpl_vector*
sph_spectrum_fit_wavs( sph_spectrum* self,
        cpl_vector* calib_lines,
        int degree,
        int tol, cpl_vector** pdiffs,
        cpl_vector** pderiv,
        cpl_vector* coeffs)
{
    cpl_vector*         result = NULL;
    cpl_vector*         result_inverse = NULL;
    cpl_vector*         diffs = NULL;
    cpl_vector*         spec_inverse = NULL;
    cpl_vector*         wavs_inverse = NULL;
    int                    ii    = 0;
    int                    nlines = 0;
    int                    np = 0;
    cpl_polynomial*        poly = NULL;

    double                val = 0.0;
    double                 dval = 0.0;
    double                 dl = 0.0;
    int                 inverse_flag  = 0;
    cpl_size           pows[1];

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(calib_lines, CPL_ERROR_NULL_INPUT, NULL);
    if ( coeffs ) {
        cpl_ensure( cpl_vector_get_size(coeffs) == degree,
                CPL_ERROR_ILLEGAL_INPUT, NULL);
    }
    nlines = cpl_vector_get_size(calib_lines);
    if ( nlines < 3 || nlines > 8 ) {
        SPH_ERR("Need between 3 and 8 lines to "
                "calibrated spectrum.");
        return NULL;
    }

    np = cpl_vector_get_size(self->wavelengths);

    if ( cpl_vector_get(self->wavelengths,1) < cpl_vector_get(self->wavelengths,0) ) {
        // Vector is in inverse order (IRDIS has lambda increasing from
        // top to bottom, IFS from bottom to top... sigh...
        inverse_flag = 1;
        spec_inverse = cpl_vector_duplicate(self->spec);
        wavs_inverse = cpl_vector_duplicate(self->wavelengths);
        for (ii = 0; ii < np; ++ii) {
            cpl_vector_set(self->wavelengths,ii,
                    cpl_vector_get(wavs_inverse,np - 1 - ii));
            cpl_vector_set(self->spec,ii,
                    cpl_vector_get(spec_inverse,np - 1 - ii));
        }
    }
    poly = get_poly_from_lines(self, calib_lines, tol, degree);
    if ( !poly ) {
        //SPH_ERROR_RAISE_ERR(cpl_error_get_code(),"Could not perform fit.");
    }
    else {
        result = cpl_vector_new( np );
        if ( pdiffs ) {
            *pdiffs = cpl_vector_new( np );
            diffs = *pdiffs;
        }
        if ( pderiv ) {
            *pderiv = cpl_vector_new( np );
        }
        for (ii = 0; ii < np; ++ii) {
            val = cpl_polynomial_eval_1d(poly,(double)ii,&dl);
            cpl_vector_set(result,ii,val);
            if ( pderiv ) cpl_vector_set(*pderiv,ii,dl);
            if ( diffs ) {
                dval = val - cpl_vector_get(self->wavelengths,ii);
                cpl_vector_set(diffs,ii,dval);
            }
        }
        if ( inverse_flag ) {
            result_inverse = cpl_vector_duplicate(result);
            np = cpl_vector_get_size(result);
            for (ii = 0; ii < np; ++ii) {
                cpl_vector_set(result,ii,cpl_vector_get(result_inverse,np - 1 - ii));
            }
            for (ii = 0; ii < np; ++ii) {
                cpl_vector_set(self->wavelengths,ii,
                        cpl_vector_get(wavs_inverse,ii));
                cpl_vector_set(self->spec,ii,
                        cpl_vector_get(spec_inverse,ii));
            }
            cpl_vector_delete(result_inverse);
        }
    }
    cpl_vector_delete(wavs_inverse); wavs_inverse = NULL;
    cpl_vector_delete(spec_inverse); spec_inverse = NULL;

    if ( coeffs ) {
        for (ii = 0; ii < degree; ++ii) {
            pows[0] = ii;
            cpl_vector_set(coeffs,ii,cpl_polynomial_get_coeff(poly,pows));
        }
    }
    cpl_polynomial_delete(poly); poly = NULL;
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
	cpl_error_reset();
	SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Could not fit spectrum");
    }
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Interpolate to new wavelength points
 * @param self      the spectrim to interpolate
 * @param wavs        the new wavelength points
 *
 * @return pointer to new spectrum
 *
 * This function creates a new spectrum with a possibly different number of points
 * by interpolating the old spectrum onto the new wavelength points in wavs.
 *
 */
/*----------------------------------------------------------------------------*/
sph_spectrum*
sph_spectrum_new_interpolate( sph_spectrum* self, cpl_vector* wavs ) {
    gsl_interp_accel*     acc = NULL;
    int                    np = 0;
    gsl_spline*         spline = NULL;
    double*                x = NULL;
    double*                y = NULL;
    int                    ll = 0;
    double                lam = 0.0;
    double                val = 0.0;
    double                minl = 0.0;
    double                maxl = 0.0;
    sph_spectrum*        newspec = NULL;

    gsl_set_error_handler_off();

    np = cpl_vector_get_size(self->wavelengths);
    x = cpl_vector_get_data(self->wavelengths);
    minl = cpl_vector_get_min( self->wavelengths );
    maxl = cpl_vector_get_max( self->wavelengths );
    newspec = sph_spectrum_new(cpl_vector_get_size(wavs),cpl_vector_get_min(wavs),
            cpl_vector_get_max(wavs));
    if ( !newspec ) return NULL;

    newspec->specid = self->specid;
    newspec->lensid = self->lensid;

    acc = gsl_interp_accel_alloc ();
    spline = gsl_spline_alloc (gsl_interp_linear, np );

    y = cpl_vector_get_data(self->spec);
    if ( gsl_spline_init (spline, x, y, np) == GSL_SUCCESS ) {

        for (ll = 0; ll < cpl_vector_get_size(newspec->wavelengths); ++ll) {
            lam = cpl_vector_get(newspec->wavelengths,ll);
            if ( lam < minl ) lam = minl;
            if ( lam > maxl ) lam = maxl;
            val = gsl_spline_eval (spline, lam, acc);
            if ( isnan(val) ) {
                cpl_vector_set(newspec->spec,ll,0.0);
                cpl_vector_set(newspec->ncomb,ll,0.0);
                cpl_vector_set(newspec->rms,ll,SPH_MASTER_FRAME_BAD_RMS);
                cpl_vector_set(newspec->bads,ll,1);
            }
            else {
                cpl_vector_set(newspec->spec,ll,val);
                cpl_vector_set(newspec->ncomb,ll,1.0);
                cpl_vector_set(newspec->rms,ll,0.0);
                cpl_vector_set(newspec->bads,ll,0);
            }
        }
    }
    else {
        sph_spectrum_delete(newspec);newspec = NULL;
    }
    if ( newspec ) {
        y = cpl_vector_get_data(self->rms);
        if ( gsl_spline_init (spline, x, y, np) == GSL_SUCCESS ) {

            for (ll = 0; ll < cpl_vector_get_size(newspec->wavelengths); ++ll) {
                lam = cpl_vector_get(newspec->wavelengths,ll);
            	if ( lam < minl ) lam = minl;
            	if ( lam > maxl ) lam = maxl;
                val = gsl_spline_eval (spline, lam, acc);
                cpl_vector_set(newspec->rms,ll,val);
            }
        }
        y = cpl_vector_get_data(self->ncomb);
        if ( gsl_spline_init (spline, x, y, np) == GSL_SUCCESS ) {

            for (ll = 0; ll < cpl_vector_get_size(newspec->wavelengths); ++ll) {
                lam = cpl_vector_get(newspec->wavelengths,ll);
            	if ( lam < minl ) lam = minl;
            	if ( lam > maxl ) lam = maxl;
                val = gsl_spline_eval (spline, lam, acc);
                cpl_vector_set(newspec->ncomb,ll,val);
            }
        }
    }
    gsl_spline_free (spline);

    gsl_interp_accel_free (acc);

    return newspec;
}
/*----------------------------------------------------------------------------*/
/**
  @brief    Delete the spectrum.
  @param    self        the spectrum to delete

  Deallocate the spectrum, the associated image and the 1D spectrum.
 */
/*----------------------------------------------------------------------------*/
void sph_spectrum_delete( sph_spectrum* self) {
    if ( self == NULL ) {
        return;
    }
    if ( self->spec ) {
        cpl_vector_delete(self->spec);
    }
    if ( self->wavelengths ) {
        cpl_vector_delete(self->wavelengths);
    }
    if ( self->bads ) {
        cpl_vector_delete(self->bads);
    }
    if ( self->rms ) {
        cpl_vector_delete(self->rms);
    }
    if ( self->ncomb ) {
        cpl_vector_delete(self->ncomb);
    }
    cpl_free(self);
    return;
}


/**@}*/


/*Private functions implementation*/

static cpl_polynomial *
get_poly_from_lines(sph_spectrum * self,
		cpl_vector * calib_lines, const int tol, const int degree){

	const cpl_size nlines = cpl_vector_get_size(calib_lines);
    const cpl_size np = cpl_vector_get_size(self->wavelengths);
    cpl_vector * xpoints = cpl_vector_new(nlines);

	for (cpl_size ii = 0; ii < nlines; ++ii) {
		const double lam0 =
				cpl_vector_get(calib_lines,ii);
		const cpl_size p0 =
				cpl_vector_find( self->wavelengths, lam0 );
		if ( p0 >= 0 ) {
			const cpl_size pstart = p0 - tol < 0 ? 0 : p0 - tol;
			const cpl_size  pend = p0 + tol > np-1 ? np-1 : p0 + tol;
			double val = 0.0, dl = 0.0;
			for (cpl_size jj = pstart; jj <= pend; ++jj)
			{
				const double max = cpl_vector_get(self->spec,jj);
				val += (double)jj * max * max;
				dl += max * max;
			}
			/* xpoints is not pre-initizalized, so when self->spec[pstart:pend]
			   is all zeroes, then keep the zero value */
			if (dl != 0.0) val /= dl;
			cpl_vector_set(xpoints, ii, val);
		} else {
			SPH_ERR("Could not extract");
			SPH_RAISE_CPL_RESET;
		}
	}

	cpl_polynomial * poly = sph_fitting_fit_poly1d( xpoints, calib_lines, NULL,
			0, degree, 0, 0.0, NULL );

	cpl_vector_delete(xpoints);
	return poly;
}
