/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _sph_xtalk_C
#include "sph_error.h"
#include "sph_xtalk.h"
#include "sph_filemanager.h"
#include "sph_fits.h"
#include "sph_utils.h"

#include <cpl.h>
#include <math.h>

#undef _sph_xtalk_C

#ifdef _OPENMP
#include <omp.h>
#endif

#define SPH_XTALK_MIN_COE__ 1.0e-20

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
static
sph_error_code
sph_xtalk_create_array__(
        sph_xtalk* self,
        const cpl_image* inim)
{
    int         xx      = 0;
    int         yy      = 0;
    int         kk      = 0;
    int         nx      = 0;
    int         ny      = 0;
    int         nrows       = 0;
    int         jj          = 0;

    double*         preamp = NULL;
    const double*   imdata = NULL;

    nx = cpl_image_get_size_x(inim);
    ny = cpl_image_get_size_y(inim);

    cpl_ensure_code( nx % self->npreamps == 0, CPL_ERROR_ILLEGAL_INPUT);

    nrows = nx / self->npreamps;
    if ( self->preamp_image ) {
        cpl_image_multiply_scalar(self->preamp_image,0.0);
    }
    else {
        self->preamp_image = cpl_image_new(
                self->npreamps,
                nrows * ny,
                CPL_TYPE_DOUBLE
        );
    }

    preamp = cpl_image_get_data_double(self->preamp_image);
    imdata = cpl_image_get_data_double_const(inim);
#ifdef _OPENMP
#pragma omp parallel for private(kk,jj,xx,yy) shared(preamp,imdata,self,nrows,ny)
#endif
    for (kk = 0; kk < self->npreamps / 2; ++kk)
    {
        for (jj = 0; jj < nrows; ++jj)
        {
            for (yy = 0; yy < ny; ++yy) {
                xx = kk * nrows * 2 + jj;
                preamp [(yy * nrows + jj) * self->npreamps + kk * 2] =
                        imdata[yy * nx + xx];
                xx = (kk + 1) * nrows * 2 - jj - 1;
                preamp [(yy * nrows + jj) * self->npreamps + kk * 2 + 1] =
                        imdata[yy * nx + xx];
            }
        }
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
sph_error_code
sph_xtalk_array_to_image__(
        sph_xtalk* self,
        cpl_image* array,
        cpl_image* result)
{
    int         xx      = 0;
    int         yy      = 0;
    int         kk      = 0;
    int         nrows       = 0;
    int         jj          = 0;
    int         nx          = 0;
    int         ny          = 0;

    double*         preamp = NULL;
    double*         imdata = NULL;

    cpl_ensure_code( self,
            CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( result,
            CPL_ERROR_NULL_INPUT);
    nx = cpl_image_get_size_x(result);
    ny = cpl_image_get_size_y(result);

    cpl_ensure_code( array,
            CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( nx % self->npreamps == 0,
            CPL_ERROR_ILLEGAL_INPUT);
    nrows = nx / self->npreamps;
    cpl_ensure_code( ny * nrows == cpl_image_get_size_y(array),
            CPL_ERROR_ILLEGAL_INPUT);
    preamp = cpl_image_get_data_double(array);
    imdata = cpl_image_get_data_double(result);

#ifdef _OPENMP
#pragma omp parallel for private(kk,jj,xx,yy) shared(preamp,imdata,self,nrows,ny)
#endif
    for (kk = 0; kk < self->npreamps / 2; ++kk)
    {
        for (jj = 0; jj < nrows; ++jj)
        {
            for (yy = 0; yy < ny; ++yy) {
                xx = kk * nrows * 2 + jj;
                imdata[yy * nx + xx] = preamp [(yy * nrows + jj) * self->npreamps + kk * 2];
                xx = (kk + 1) * nrows * 2 - jj - 1;
                imdata[yy * nx + xx] = preamp [(yy * nrows + jj) * self->npreamps + kk * 2 + 1];
            }
        }
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
static
sph_error_code
sph_xtalk_derivative__( sph_xtalk* self ) {
    cpl_image*      deriv = NULL;
    double*         imdat = NULL;
    double*         derivdat = NULL;
    int             kk       = 0;
    int             yy       = 0;
    int             ny       = 0;
    int             npre     = 0;
    cpl_ensure_code( self,
            CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->preamp_image,
            CPL_ERROR_NULL_INPUT);


    deriv = cpl_image_duplicate(self->preamp_image);

    imdat = cpl_image_get_data_double(self->preamp_image);
    derivdat = cpl_image_get_data_double(deriv);

    npre = self->npreamps;
    ny = cpl_image_get_size_y(self->preamp_image);
    for (kk = 0; kk < npre; ++kk) {
        for (yy = 1; yy < ny; ++yy) {
            derivdat[yy * npre + kk] -= imdat[(yy - 1) * npre + kk];
        }
    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        cpl_image_delete(deriv);
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
    }
    if ( self->derivim ) {
        cpl_image_delete( self->derivim );
    }
    self->derivim = deriv;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
static
sph_error_code
sph_xtalk_subtract_xtalk__( sph_xtalk* self ) {
    int             kk  = 0;
    int             jj  = 0;
    int             yy  = 0;
    double*         preamp = NULL;
    double*         coeffs = NULL;
    double*         deriv = NULL;
#ifdef _OPENMP
    static  int     firstcall = 0;
#endif
    int             maxthreads = 1;
    int             width = 0;
    int             yy_start = 0;
    int             yy_end = 0;
    int             ny = 0;
    int             threadnum = 0;
    int             npres = 0;
    double          coe = 0.0;
    cpl_ensure_code( self,
            CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->preamp_image,
            CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->coeffs,
            CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->derivim,
            CPL_ERROR_NULL_INPUT);

    preamp = cpl_image_get_data_double(self->preamp_image);
    coeffs = cpl_image_get_data_double(self->coeffs);
    deriv = cpl_image_get_data_double(self->derivim);

    ny = cpl_image_get_size_y( self->preamp_image);

    npres = self->npreamps;
#ifdef _OPENMP
    maxthreads = omp_get_max_threads();
    if ( maxthreads > 1 &&
            maxthreads % 2 != 0 ) maxthreads--;
    if ( firstcall == 0 ) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Executing in parallel mode with %d threads",
                maxthreads);
    }
    omp_set_num_threads(maxthreads);
    firstcall = 1;
#endif
    threadnum = 0;
    for (kk = 0; kk < self->npreamps; ++kk) {
        for (jj = 0; jj < self->npreamps; ++jj) {
            coe = coeffs[ jj * npres + kk];
            if ( coe > SPH_XTALK_MIN_COE__ ) {
#ifdef _OPENMP
#pragma omp parallel private(threadnum,yy_start,yy_end,yy,width) shared(npres,ny,self,maxthreads)
                threadnum = omp_get_thread_num();
#endif
                width = ny / maxthreads;
                yy_start = threadnum * width;
                yy_end = (threadnum + 1) * width;
                for (yy = yy_start; yy < yy_end; ++yy)
                {
                    preamp[yy * npres + jj] -= coe *
                            deriv[ yy * npres + kk];
                }
            }
        }
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


sph_xtalk*
sph_xtalk_new( int npreamps ) {
    sph_xtalk*      self = NULL;

    self = cpl_calloc(1,sizeof(sph_xtalk));
    self->npreamps = npreamps;
    self->coeffs = cpl_image_new(self->npreamps,
            self->npreamps, CPL_TYPE_DOUBLE);
    cpl_image_add_scalar(self->coeffs,1.0);
    return self;
}

sph_xtalk*
sph_xtalk_load_pattern( const char* filename ) {
    sph_xtalk*      self = NULL;
    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);

    self = sph_xtalk_new(0);
    cpl_ensure(self, cpl_error_get_code(), NULL);
    if ( self->coeffs ) {
        cpl_image_delete(self->coeffs);
        self->coeffs  = NULL;
    }
    self->coeffs = cpl_image_load(filename, CPL_TYPE_DOUBLE, 0, 0 );
    if ( !self->coeffs ) {
        cpl_error_set_where(cpl_func);
        sph_xtalk_delete(self); self = NULL;
    } else {
        self->npreamps = cpl_image_get_size_x(self->coeffs);
    }
    return self;
}

cpl_image*
sph_xtalk_apply( sph_xtalk* self, const cpl_image* imin ) {
    cpl_image*          result = NULL;

    cpl_ensure(self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(imin, CPL_ERROR_NULL_INPUT, NULL);

    cpl_ensure(
            sph_xtalk_create_array__(self,imin) ==
                    CPL_ERROR_NONE,
                    cpl_error_get_code(),
            NULL);
    cpl_ensure(
            sph_xtalk_derivative__(self) ==
                    CPL_ERROR_NONE,
                    cpl_error_get_code(),
            NULL);
    cpl_ensure(
            sph_xtalk_subtract_xtalk__(self) ==
                    CPL_ERROR_NONE,
                    cpl_error_get_code(),
            NULL);

    result = cpl_image_duplicate(imin);
    sph_xtalk_array_to_image__(self,self->preamp_image,result);

    return result;
}

sph_error_code
sph_xtalk_apply_frameset(sph_xtalk* self,
                         const cpl_frameset* rawframes,
                         cpl_frameset* outframes)
{
    const int nf = cpl_frameset_get_size(rawframes);

    cpl_ensure_code(self      != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(rawframes != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(outframes != NULL, CPL_ERROR_NULL_INPUT);

    for (int ff = 0; ff < nf; ++ff)
    {
        const cpl_frame*  aframe =
            cpl_frameset_get_position_const(rawframes, ff);
        const char*       filename = cpl_frame_get_filename(aframe);
        cpl_frame*        dupframe =
            sph_filemanager_get_unique_duplicate_frame( aframe);
        cpl_propertylist* pl = cpl_propertylist_load(filename, 0);
        const int         np = sph_fits_get_nplanes(filename, 0);

        /* Update the header if required */
        sph_utils_update_header(pl);

        for (int pp = 0; pp < np; ++pp) {
            cpl_image* inim  = cpl_image_load(filename, CPL_TYPE_DOUBLE, pp, 0);
            cpl_image* dumim = sph_xtalk_apply(self,inim);

            if ( dumim ) { /* FIXME: Should break on NULL dumin */
                cpl_image_save(dumim, cpl_frame_get_filename(dupframe),
                               CPL_TYPE_DOUBLE, pl,
                               pp == 0 ? CPL_IO_CREATE : CPL_IO_APPEND);
                cpl_image_delete(dumim); dumim = NULL;
            }
            cpl_image_delete(inim); inim = NULL;
            cpl_propertylist_delete(pl); pl = NULL; /* Only used for 1st */
            if ( cpl_error_get_code() != CPL_ERROR_NONE ) break;

        }
        cpl_propertylist_delete(pl); pl = NULL; /* deallocate on fall through */
        if ( cpl_error_get_code() != CPL_ERROR_NONE ) break;
        cpl_frameset_insert(outframes, dupframe);
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

void
sph_xtalk_delete(sph_xtalk* self) {
   if ( self ) {
       cpl_image_delete(self->preamp_image);
       cpl_image_delete(self->derivim);
       cpl_image_delete(self->coeffs);
       cpl_free(self);
   }
}

/**@}*/
