/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <math.h>
#include "sph_zpl_exposure.h"
#include "sph_error.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_fits.h"
#include "sph_cube.h"
#include "sph_utils.h"

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/
#define EXT_ZERO_ODD    0
#define EXT_ZERO_EVEN   1
#define EXT_PI_ODD      2
#define EXT_PI_EVEN     3
#define EXT_OVSC_TABLE	4

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_exposure  zimpol exposure object
 *
 * @par Descirption:
 * This object represents a zimpol exposure in the SPHERE DRH. Zimpol Exposure
 * within the SPHERE DRH is a pre-processed raw zimpol data (single data)
 * <ol>
 * <li> properties  </li>
 * <li> image_zero_odd  </li>
 * <li> image_zero_even </li>
 * <li> image_pi_odd    </li>
 * <li> image_pi_even   </li>
 * <li> ovsc_table		</li>
 * </ol>
 *
 *----------------------------------------------------------------------------*/
/**@{*/


/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor fucntion for sph_zpl_exposure.
  @return   a pointer to the newly created sph_zpl_exposure or NULL if
              unsuccessfull.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_zpl_exposure
 */
/*----------------------------------------------------------------------------*/
sph_zpl_exposure* sph_zpl_exposure_new_empty(void) {
    sph_zpl_exposure*     zplexp         = NULL;
    sph_error_code        rerr         = CPL_ERROR_NONE;


    zplexp = cpl_calloc( 1, sizeof( sph_zpl_exposure ) );

    zplexp->properties = cpl_propertylist_new();
    zplexp->ovsc = cpl_propertylist_new();

    rerr = cpl_propertylist_append_string( zplexp->properties, SPH_COMMON_KEYWORD_SPH_TYPE, SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP );
    if ( rerr != CPL_ERROR_NONE )
    {
        sph_error_raise( SPH_ERROR_MEMORY_FAIL, __FILE__,
                         __func__, __LINE__,
                         SPH_ERROR_ERROR, "Could not allocate memory for zpl exposure" );
        return NULL;
    }
    return zplexp;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_zpl_exposure.

  @param nx     the number of pixels in x
  @param ny     the number of pixels in y

  @return   a pointer to the newly created sph_zpl_exposure or NULL if
            unsuccessfully.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_zpl_exposure of the given n pixel size.
 */
/*----------------------------------------------------------------------------*/
sph_zpl_exposure* sph_zpl_exposure_new( int nx, int ny ) {
    sph_zpl_exposure*     zplexp = NULL;

    zplexp = sph_zpl_exposure_new_empty();
    zplexp->image_zero_odd = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    zplexp->image_zero_even = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    zplexp->image_pi_odd = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);
    zplexp->image_pi_even = cpl_image_new(nx,ny, CPL_TYPE_DOUBLE);

    if (cpl_error_get_code() != CPL_ERROR_NONE ){
           sph_error_raise(SPH_ERROR_ERROR,
              __FILE__, __func__, __LINE__,
               SPH_ERROR_ERROR, "CPL error has occured: %d\n"
                                "CPL error message: %s",
                                cpl_error_get_code(),
                                cpl_error_get_message_default(cpl_error_get_code() ) );
           return NULL;
    }

    return zplexp;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Copy constructor function for sph_zpl_exposure.

  @param self   the sph_zpl_exposure to duplicate

  @return   a pointer to the newly created sph_zpl_exposure or NULL if
            unsuccessfully.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_zpl_exposure as copy of the input sph_zpl_exposure.
 */
/*----------------------------------------------------------------------------*/
sph_zpl_exposure* sph_zpl_exposure_duplicate( sph_zpl_exposure* self ){
    sph_zpl_exposure*    zplexp = NULL;

    zplexp = sph_zpl_exposure_new_empty();

    if ( self-> image_zero_odd ) {
        zplexp->image_zero_odd = cpl_image_duplicate( self->image_zero_odd);
    }
    if ( self-> image_zero_even ) {
        zplexp->image_zero_even = cpl_image_duplicate( self->image_zero_even);
    }
    if ( self-> image_pi_odd ) {
        zplexp->image_pi_odd = cpl_image_duplicate( self->image_pi_odd);
    }
    if ( self-> image_pi_even ) {
        zplexp->image_pi_even = cpl_image_duplicate( self->image_pi_even);
    }

    if ( self->properties ) {
        if ( zplexp->properties ) {
            cpl_propertylist_delete( zplexp->properties );
        }
        zplexp->properties = cpl_propertylist_duplicate( self->properties );
    }

    if( self->ovsc ){
        if ( zplexp->ovsc ) {
            cpl_propertylist_delete( zplexp->ovsc );
        }
        zplexp->ovsc = cpl_propertylist_duplicate( self->ovsc );
    }


    if (cpl_error_get_code() != CPL_ERROR_NONE ){
           sph_error_raise(SPH_ERROR_ERROR,
              __FILE__, __func__, __LINE__,
               SPH_ERROR_ERROR, "CPL error has occured: %d\n"
                                "CPL error message: %s",
                                cpl_error_get_code(),
                                cpl_error_get_message_default(cpl_error_get_code()) );
           return NULL;
    }

    return zplexp;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Constructor function for sph_zpl_exposure from existing cpl images

  @param image_zero_odd     pointer to the first(zero_odd) cpl_image
  @param image_zero_even    pointer to the second(zero_even) cpl_image
  @param image_zero_odd     pointer to the first(zero_odd) cpl_image
  @param image_zero_even    pointer to the second(zero_even) cpl_image

  @return   a pointer to the newly created sph_zpl_exposure or NULL if
            unsuccessfully.
  @note     A CPL error is raised in case there was a problem.

  Construct a new sph_zpl_exposure from the given 4 cpl images.
 */
/*----------------------------------------------------------------------------*/
sph_zpl_exposure* sph_zpl_exposure_new_from_cplimages( cpl_image* image_zero_odd, cpl_image* image_zero_even,
		                                      cpl_image* image_pi_odd, cpl_image* image_pi_even ){
	sph_zpl_exposure*	zplexp = NULL;

	zplexp = sph_zpl_exposure_new_empty();

    if ( image_zero_odd && image_zero_even &&
    		image_pi_odd && image_pi_even ) {
        zplexp->image_zero_odd = cpl_image_duplicate( image_zero_odd );
        zplexp->image_zero_even = cpl_image_duplicate( image_zero_even );
        zplexp->image_pi_odd = cpl_image_duplicate( image_pi_odd );
        zplexp->image_pi_even = cpl_image_duplicate( image_pi_even );
    } else {
        sph_error_raise(SPH_ERROR_ERROR,
           __FILE__, __func__, __LINE__,
            SPH_ERROR_ERROR, "Couldn't create sph zpl exposure from the given images." );
        return NULL;

    }

    if (cpl_error_get_code() != CPL_ERROR_NONE ){
           sph_error_raise(SPH_ERROR_ERROR,
              __FILE__, __func__, __LINE__,
               SPH_ERROR_ERROR, "CPL error has occured: %d\n"
                                "CPL error message: %s",
                                cpl_error_get_code(),
                                cpl_error_get_message_default(cpl_error_get_code()) );
           return NULL;
    }

    return zplexp;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    save the zimpol exposure

  @param    self        the sph_zimpol_exposure to save
  @param    outframe    the cpl frame to save into
  @param     pl            a propertylist to append, it can be NULL
  @return     the error code

  This function saves the sph_zpl_exposure as a FITS file with 4 extensions
  */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_exposure_save( sph_zpl_exposure* self, cpl_frame* outframe, cpl_propertylist* pl){
    sph_error_code           rerr 	            = CPL_ERROR_NONE;
    cpl_propertylist*        tproperties        = NULL;
    cpl_propertylist*        tpl                = NULL;
    cpl_table*				 ovsc_table			= NULL;

    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }
    if ( !self->image_zero_odd || !self->image_zero_even
        || !self->image_pi_odd || !self->image_pi_even ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    if ( self->properties ){
        tproperties = sph_keyword_manager_trans_keywords( self->properties );
    } else {
        tproperties = cpl_propertylist_new();
    }

    if ( pl ){
        tpl = sph_keyword_manager_trans_keywords( pl );
    }

    if ( tproperties && tpl ) {
        for ( int i = 0; i < cpl_propertylist_get_size( tpl ); i++ ) {
            if ( !cpl_propertylist_has( tproperties, cpl_property_get_name( cpl_propertylist_get( tpl, i) ) ) ){
                rerr = cpl_propertylist_append_property( tproperties, cpl_propertylist_get( tpl, i) );
            } else {
                rerr = cpl_propertylist_erase( tproperties, cpl_property_get_name( cpl_propertylist_get( tpl, i) ) );
                rerr = cpl_propertylist_append_property( tproperties, cpl_propertylist_get( tpl, i) );
            }
        }
    }

    if ( cpl_propertylist_has( tproperties, SPH_COMMON_KEYWORD_SPH_TYPE )) {
        cpl_propertylist_update_string( tproperties, SPH_COMMON_KEYWORD_SPH_TYPE,
                                                 SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP );
    } else {
        cpl_propertylist_append_string( tproperties, SPH_COMMON_KEYWORD_SPH_TYPE,
                                                 SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP  );
    }

    /* Update the header if required */
    sph_utils_update_header(tproperties);

    rerr = cpl_image_save( self->image_zero_odd, cpl_frame_get_filename( outframe ), CPL_TYPE_DOUBLE,
                           tproperties, CPL_IO_CREATE );
    rerr = cpl_image_save( self->image_zero_even, cpl_frame_get_filename( outframe ), CPL_TYPE_DOUBLE,
                           NULL, CPL_IO_EXTEND );
    rerr = cpl_image_save( self->image_pi_odd, cpl_frame_get_filename( outframe ), CPL_TYPE_DOUBLE,
                           tproperties, CPL_IO_EXTEND );
    rerr = cpl_image_save( self->image_pi_even, cpl_frame_get_filename( outframe ), CPL_TYPE_DOUBLE,
                           NULL, CPL_IO_EXTEND );

    if ( self->ovsc &&  cpl_propertylist_get_size(self->ovsc) == OVSC_SIZE ){
        ovsc_table = sph_zpl_exposure_ovsc_table_create_empty();
    	sph_zpl_exposure_ovsc_table_add_raw( self, ovsc_table );
    }

    if ( ovsc_table ){
    	SPH_INFO_MSG("Saving table elements...");
    	rerr = cpl_table_save(ovsc_table, NULL, NULL, cpl_frame_get_filename( outframe ), CPL_IO_EXTEND);
        cpl_table_delete( ovsc_table ); ovsc_table = NULL;
    }

    if ( tproperties ) {
        cpl_propertylist_delete( tproperties );
    }
    if ( tpl ) {
        cpl_propertylist_delete( tpl );
    }

    if (rerr != CPL_ERROR_NONE ) {
        SPH_ERR("Problem(s) occure(s) by saving a given zpl exposure")
    }

    return rerr;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    save the zimpol exposure and keep the possibility to append more planes

  @param    self        the sph_zimpol_exposure to save
  @param    outframe    the cpl frame to save into
  @param    pl          a propertylist to save, it can be NULL
  @param    ovsc_table  the overscan table save, it can be NULL

  @return     the error code

  This function saves the sph_zpl_exposure as a FITS file with 4 extensions but
  keeps the possibility to append more zpl_exposure planes
  */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_exposure_save_open( sph_zpl_exposure* self, cpl_frame* outframe,
						    cpl_propertylist* pli, cpl_table* ovsc_table){
    sph_error_code            rerr            = CPL_ERROR_NONE;
    sph_zpl_exposure*		  self_duplicate  = NULL;

    if (!self){
        SPH_NO_SELF;
        return CPL_ERROR_NULL_INPUT;
    }
    cpl_ensure_code(outframe, CPL_ERROR_NULL_INPUT);

    SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO, "filename = %s", cpl_frame_get_filename( outframe ));

    if ( ovsc_table ){
    	SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO, "get size ovsc table  = %d", (int)cpl_table_get_nrow( ovsc_table ));
    	rerr = sph_zpl_exposure_ovsc_table_add_raw( self, ovsc_table );
    	if ( rerr == CPL_ERROR_NONE ){
    		self_duplicate = sph_zpl_exposure_duplicate( self );
    		cpl_propertylist_delete(self_duplicate->ovsc); self_duplicate->ovsc = NULL;
    		rerr = sph_zpl_exposure_save ( self_duplicate, outframe, pli );
    		sph_zpl_exposure_delete( self_duplicate );
    	}else{
    		SPH_ERROR_RAISE_WARNING(SPH_ERROR_WARNING, "Ovesrcan values didn't add to the overscan table. "
    				"zpl exposure cube will be opened/saved without overscan table!");
    		SPH_INFO_MSG("Reset SPH ERROR to handle it properly!")
    		sph_error_reset();
        	rerr = sph_zpl_exposure_save ( self, outframe, pli );
    	}
    } else {
    	rerr = sph_zpl_exposure_save ( self, outframe, pli );
    }

    if ( rerr != CPL_ERROR_NONE) {
    	SPH_ERR("Couldn't save zpl exposure");
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}


/*----------------------------------------------------------------------------*/
/**
  @brief    append the zpl exposure to the existing/opened zpl exp fits frame

  @param    self        the sph_zimpol_exposure to append
  @param    outframe    the cpl frame to append into
  @param    pl          a propertylist to append, it can be NULL
  @return   the error code

  This function appends the sph_zpl_exposure as a plane of 4 extensions to
  the existing/opened FITS files of the sph_zpl_exposure format.
  See also sph_zpl_exposure_save_open function.
  */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_exposure_save_append( sph_zpl_exposure* self, cpl_frame* outframe, cpl_table* ovsc_table ){
    sph_error_code            rerr            = CPL_ERROR_NONE;

    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }

    cpl_ensure_code(outframe, CPL_ERROR_NULL_INPUT);

    if ( !self->image_zero_odd || !self->image_zero_even
        || !self->image_pi_odd || !self->image_pi_even ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    if ( ovsc_table ){
    	sph_zpl_exposure_ovsc_table_add_raw( self, ovsc_table );
    }

    rerr = sph_cube_append_image( cpl_frame_get_filename( outframe ), self->image_zero_odd, NULL, EXT_ZERO_ODD   );
    rerr |= sph_cube_append_image( cpl_frame_get_filename( outframe ), self->image_zero_even, NULL, EXT_ZERO_EVEN );
    rerr |= sph_cube_append_image( cpl_frame_get_filename( outframe ), self->image_pi_odd, NULL, EXT_PI_ODD   );
    rerr |= sph_cube_append_image( cpl_frame_get_filename( outframe ), self->image_pi_even, NULL, EXT_PI_EVEN   );


    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

/*----------------------------------------------------------------------------*/
/**
  @brief    finalize the zpl exposure cube

  @param    self        the sph_zimpol_exposure to append
  @param    outframe    the cpl frame to append into
  @param    pl          a propertylist to append, it can be NULL
  @return   the error code

  This function finalizes the sph_zpl_exposure cube  sph_zpl_exposure format.
  See also sph_zpl_exposure_save_open/sph_zpl_exposure_append function.
  */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_exposure_finalize_file( cpl_frame* outframe, cpl_table* ovsc_table){

	cpl_ensure_code(outframe, CPL_ERROR_NULL_INPUT);

	//if ( !(ovsc_table && cpl_table_get_nrow(ovsc_table) == 1) ) {
		sph_cube_finalise_file( cpl_frame_get_filename( outframe ));
	//}

	//sph_cube_finalise_file( cpl_frame_get_filename( outframe ));
	if ( ovsc_table && cpl_table_get_nrow(ovsc_table) > 0 ) {
        cpl_table_save( ovsc_table, NULL, NULL, cpl_frame_get_filename( outframe ), CPL_IO_EXTEND);
	}
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}


/*----------------------------------------------------------------------------*/
/**
  @brief    save a given overscan table into cpl frame as a new extension

  @param    outframe    the cpl frame to append into
  @param    ovsc_table  the overscan table to save, it can be NULL
  @return   the error code
  */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_exposure_save_ovsc( cpl_frame* outframe, cpl_table* ovsc_table ){

	cpl_ensure_code(outframe, CPL_ERROR_NULL_INPUT);

	if ( ovsc_table && cpl_table_get_nrow(ovsc_table) > 0 ) {
        cpl_table_save( ovsc_table, NULL, NULL, cpl_frame_get_filename( outframe ), CPL_IO_EXTEND);
	}
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}


/*----------------------------------------------------------------------------*/
/**
  @brief    create empty zpl exposure overscan table

  @return   a pointer to the newly created empty overscan table NULL if
            unsuccessfully.
  @note     A CPL error is raised in case there was a problem.


  This function creates empty zpl exposure overscan table
  */
/*----------------------------------------------------------------------------*/
cpl_table*
sph_zpl_exposure_ovsc_table_create_empty( void ){
	cpl_table*		ovsc_table		= NULL;

    ovsc_table = cpl_table_new(0);
    cpl_table_new_column( ovsc_table, ADU1_ZERO_OVSC_MEAN, CPL_TYPE_DOUBLE );
    cpl_table_new_column( ovsc_table, ADU1_ZERO_OVSC_RMS, CPL_TYPE_DOUBLE );
    cpl_table_new_column( ovsc_table, ADU2_ZERO_OVSC_MEAN, CPL_TYPE_DOUBLE );
    cpl_table_new_column( ovsc_table, ADU2_ZERO_OVSC_RMS, CPL_TYPE_DOUBLE );
    cpl_table_new_column( ovsc_table, ADU1_PI_OVSC_MEAN, CPL_TYPE_DOUBLE );
    cpl_table_new_column( ovsc_table, ADU1_PI_OVSC_RMS, CPL_TYPE_DOUBLE );
    cpl_table_new_column( ovsc_table, ADU2_PI_OVSC_MEAN, CPL_TYPE_DOUBLE );
    cpl_table_new_column( ovsc_table, ADU2_PI_OVSC_RMS, CPL_TYPE_DOUBLE );

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL

    return ovsc_table;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    add raw to the ovsc table from ovsc propertylist of the current
  	  	  	zpl_exposure

  @param    self        the sph_zimpol_exposure to extract ovsc properties
  	  	  	  	  	  	to add to the ovsc_table

  @param	ovsc_table	overscan cpl_table
  @return   add a raw to the ovsc table from ovsc propertylist of the current
  	  	  	zpl_exposurenewly created empty overscan table NULL if
            unsuccessfully.
  @note     This function is needed to create a cube of zpl exposures.
            A CPL error is raised in case there was a problem.

  This function adds a raw to the ovsc table from ovsc propertylist of the current
  zpl_exposure.
  */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_exposure_ovsc_table_add_raw( sph_zpl_exposure* self, cpl_table* ovsc_table ){
	int 		tabsize		= 0;

    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }

    cpl_ensure_code(ovsc_table, CPL_ERROR_NULL_INPUT);

	if ( self->ovsc && cpl_propertylist_get_size(self->ovsc) == OVSC_SIZE ){

        tabsize = cpl_table_get_nrow( ovsc_table );
        cpl_table_set_size( ovsc_table, tabsize+1);

         //phase zero
      	cpl_table_set_double( ovsc_table, ADU1_ZERO_OVSC_MEAN, tabsize,
    			cpl_propertylist_get_double( self->ovsc, ADU1_ZERO_OVSC_MEAN ));
     	cpl_table_set_double( ovsc_table, ADU1_ZERO_OVSC_RMS, tabsize,
     			cpl_propertylist_get_double( self->ovsc, ADU1_ZERO_OVSC_RMS ));
     	cpl_table_set_double( ovsc_table, ADU2_ZERO_OVSC_MEAN, tabsize,
     			cpl_propertylist_get_double( self->ovsc, ADU2_ZERO_OVSC_MEAN ));
     	cpl_table_set_double( ovsc_table, ADU2_ZERO_OVSC_RMS, tabsize,
     			cpl_propertylist_get_double( self->ovsc, ADU2_ZERO_OVSC_RMS ));

     	//phase pi
     	cpl_table_set_double( ovsc_table, ADU1_PI_OVSC_MEAN, tabsize,
     			cpl_propertylist_get_double( self->ovsc, ADU1_PI_OVSC_MEAN ));
     	cpl_table_set_double( ovsc_table, ADU1_PI_OVSC_RMS, tabsize,
     			cpl_propertylist_get_double( self->ovsc, ADU1_PI_OVSC_RMS ));
     	cpl_table_set_double( ovsc_table, ADU2_PI_OVSC_MEAN, tabsize,
     			cpl_propertylist_get_double( self->ovsc, ADU2_PI_OVSC_MEAN ));
     	cpl_table_set_double( ovsc_table, ADU2_PI_OVSC_RMS, tabsize,
     			cpl_propertylist_get_double( self->ovsc, ADU2_PI_OVSC_RMS ));

     } else {
    	 SPH_ERR("Wrong overscan propertylist sets!")
     }


    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

}

/*----------------------------------------------------------------------------*/
/**
  @brief    subtract  overscan mean values from zpl_exposure

  @param    self      the sph_zimpol_exposure to subtract from

  @param	ovsc_table	overscan cpl_table

  @return   the error code

  @note     This function is needed to create a cube of zpl exposures.
            A CPL error is raised in case there was a problem.

  This function subtract overscan mean values (kept as ovsc properies) from
  self (zpl_exposure). Self is changed in place.
  */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_exposure_subtract_overscans( sph_zpl_exposure* self ){
	int 		nx			= 0;
	int			ny			= 0;
	int			pis_rjected	= 0;
	double		val			= 0;
        int single_frame = FALSE;

	if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }

	nx = cpl_image_get_size_x( self->image_zero_odd );
	ny = cpl_image_get_size_y( self->image_zero_odd );

        // Check if this is a single frame exposure
        single_frame = sph_utils_is_single_frame(self->properties);

	if ( self->ovsc && cpl_propertylist_get_size(self->ovsc) == OVSC_SIZE ){

		for ( int xx = 0; xx < nx/2; ++xx ) {
	        for (int yy = 0; yy < ny; ++yy){
	        	//image_zero_odd
	        	val = cpl_image_get( self->image_zero_odd, xx + 1, yy + 1, &pis_rjected );
	        	val = val - cpl_propertylist_get_double( self->ovsc,  ADU1_ZERO_OVSC_MEAN );
	        	cpl_image_set( self->image_zero_odd, xx+1, yy+1, val);
	        	//image_zero_even
	        	val = cpl_image_get( self->image_zero_even, xx + 1, yy + 1, &pis_rjected );
	        	val = val - cpl_propertylist_get_double( self->ovsc,  ADU1_ZERO_OVSC_MEAN );
	        	cpl_image_set( self->image_zero_even, xx+1, yy+1, val);
                        // Single frames do not have valid image_pi_odd or image_pi_even
                        if (!single_frame) {
	        	    //image_pi_odd
	        	    val = cpl_image_get( self->image_pi_odd, xx + 1, yy + 1, &pis_rjected );
	        	    val = val - cpl_propertylist_get_double( self->ovsc,  ADU1_PI_OVSC_MEAN );
	        	    cpl_image_set( self->image_pi_odd, xx+1, yy+1, val);
	        	    //image_pi_even
	        	    val = cpl_image_get( self->image_pi_even, xx + 1, yy + 1, &pis_rjected );
	        	    val = val - cpl_propertylist_get_double( self->ovsc,  ADU1_PI_OVSC_MEAN );
	        	    cpl_image_set( self->image_pi_even, xx+1, yy+1, val);
                        }
	        }
		}

		for ( int xx = nx/2; xx < nx; ++xx ) {
	        for (int yy = 0; yy < ny; ++yy){
	        	//image_zero_odd
	        	val = cpl_image_get( self->image_zero_odd, xx + 1, yy + 1, &pis_rjected );
	        	val = val - cpl_propertylist_get_double( self->ovsc,  ADU2_ZERO_OVSC_MEAN );
	        	cpl_image_set( self->image_zero_odd, xx+1, yy+1, val);
	        	//image_zero_even
	        	val = cpl_image_get( self->image_zero_even, xx + 1, yy + 1, &pis_rjected );
	        	val = val - cpl_propertylist_get_double( self->ovsc,  ADU2_ZERO_OVSC_MEAN );
	        	cpl_image_set( self->image_zero_even, xx+1, yy+1, val);
                        // Single frames do not have valid image_pi_odd or image_pi_even
                        if (!single_frame) {
	        	    //image_pi_odd
	        	    val = cpl_image_get( self->image_pi_odd, xx + 1, yy + 1, &pis_rjected );
	        	    val = val - cpl_propertylist_get_double( self->ovsc,  ADU2_PI_OVSC_MEAN );
	        	    cpl_image_set( self->image_pi_odd, xx+1, yy+1, val);
	        	    //image_pi_even
	        	    val = cpl_image_get( self->image_pi_even, xx + 1, yy + 1, &pis_rjected );
	        	    val = val - cpl_propertylist_get_double( self->ovsc,  ADU2_PI_OVSC_MEAN );
	        	    cpl_image_set( self->image_pi_even, xx+1, yy+1, val);
                        }
	        }
		}

	} else {
   	 SPH_ERR("Wrong overscan propertylist sets!")
    }

	SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

}




/*----------------------------------------------------------------------------*/
/**
 * @brief        Update property list of the zpl exposure DFS product
 *               by SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP
 *
 * @param       czFilename  the filename to load and save zpl exposure with
 *              correctly updated SPH_COMMON_KEYWORD_SPH_TYPE
 * @return      the error code
 *
 * @note This function is used as a final step to save zpl exposure as a
 * DFS product (see function sph_double_image_save_dfs). It updates
 * SPH_COMMON_KEYWORD_SPH_TYPE by SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP.
 * It is necessary because cpl_dsf_save_image updates the zpl exposure property
 * list by the property list from the first input frames.
 *
 */
/*----------------------------------------------------------------------------*/
static sph_error_code sph_zpl_exposure_update_prod_type( const char* czFilename )
{
    sph_error_code        rerr 			= CPL_ERROR_NONE;
    sph_zpl_exposure*     zplexp 		= NULL;
    cpl_propertylist*     pli			= NULL;
    cpl_propertylist*     pli_regexp	= NULL;
    cpl_frame*			  frame			= NULL;
    int                   plane			= 0;

    //create frame with the filename czFilename because sph_zpl_exposure load and save use frame as input instead of filename
    frame = cpl_frame_new();
    cpl_frame_set_filename( frame, czFilename );
    cpl_frame_set_tag( frame, "NONE" );
    cpl_frame_set_group( frame, CPL_FRAME_GROUP_NONE );

    zplexp = sph_zpl_exposure_load( frame, plane );

    if ( !zplexp ){
        SPH_ERR("No zpl exposure is loaded.")
        return sph_error_get_last_code();
    }

    pli = cpl_propertylist_load_regexp( czFilename, 0, ".*QC.*", 1);
    pli_regexp = cpl_propertylist_new();
    rerr = cpl_propertylist_copy_property_regexp( pli_regexp, pli, ".*COMMENT.*", 1 );

    rerr = sph_zpl_exposure_save( zplexp, frame, pli_regexp);

    if (rerr != CPL_ERROR_NONE ) {
        SPH_ERR("Couldn't  save a given zpl exposure")
    }
    sph_zpl_exposure_delete( zplexp );
    cpl_frame_delete( frame );

    return rerr;
}



/*----------------------------------------------------------------------------*/
/**
 * @brief       Save the zpl exposure as DFS product
 *
 * @param       self            the zpl exposure to save
 * @param       czFilename      the filename to save under
 * @param       allframes       the frameset used to construct the product
 * @param       usedframes      the frameset used to construct the product
 * @param       template_frame  the frame to use as template for header
 * @param       param           the parameterlist used for product
 * @param       tag             the tag of the product
 * @param       recipe          the recipe creating this product
 * @param       pipename        the pipeline creating this product
 * @param       pl              a propertylist to append to main header (can be NULL)
 * @return      the error code
 *
 * Save the zpl exposure frame as a FITS file according to ESO DFS.
 * This saves the zpl exposure as a FITS file with 4 extensions.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code
sph_zpl_exposure_save_dfs(
		sph_zpl_exposure* self,
        const char* outfilename,
        cpl_frameset* allframes,
        cpl_frame* template_frame,
        cpl_parameterlist* params,
        const char* tag,
        const char* recipe,
        const char* pipename,
        cpl_propertylist* pl)
{
    sph_error_code            rerr				= CPL_ERROR_NONE;
    cpl_propertylist*         tproperties		= NULL;
    cpl_propertylist*         tpl				= NULL;
    cpl_table*				  ovsc_table		= NULL;


    if ( !self ) {
        SPH_NO_SELF
        return CPL_ERROR_NULL_INPUT;
    }

    if ( !self->image_zero_odd || !self->image_zero_even
        || !self->image_pi_odd || !self->image_pi_even ) {
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    if ( self->properties ){
        tproperties = sph_keyword_manager_trans_keywords( self->properties );
    } else {
        tproperties = cpl_propertylist_new();
    }

    if ( pl ){
        tpl = sph_keyword_manager_trans_keywords( pl );
    }

    if ( tproperties && tpl ) {
        for ( int i = 0; i < cpl_propertylist_get_size( tpl ); i++ ) {
            if ( !cpl_propertylist_has( tproperties, cpl_property_get_name( cpl_propertylist_get( tpl, i) ) ) ){
                rerr = cpl_propertylist_append_property( tproperties, cpl_propertylist_get( tpl, i) );
            } else {
                rerr = cpl_propertylist_erase( tproperties, cpl_property_get_name( cpl_propertylist_get( tpl, i) ) );
                rerr = cpl_propertylist_append_property( tproperties, cpl_propertylist_get( tpl, i) );
            }
        }
    }

    if ( cpl_propertylist_has( tproperties, SPH_COMMON_KEYWORD_SPH_TYPE )) {
        cpl_propertylist_update_string( tproperties, SPH_COMMON_KEYWORD_SPH_TYPE,
                                                 SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP );
    } else {
        cpl_propertylist_append_string( tproperties,SPH_COMMON_KEYWORD_SPH_TYPE,
                                                 SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP  );
    }

    /*
      * Attention: SPH_COMMON_KEYWORD_SPH_TYPE will be changed by cpl_dfs_save_image function.
      * Unfortunately this keyword will be later updated automatically in cpl_dfs_save_image.
      * In order to avoid this  a function, called sph_zpl_exposure_update_prod_type is used
      * as soon as the dfs product is created (see below)
      *
      */


    rerr |= cpl_propertylist_update_string( tproperties, SPH_COMMON_KEYWORD_PRO_CATG, tag);

    /* Update the header if required */
    sph_utils_update_header(tproperties);

    if ( rerr != CPL_ERROR_NONE ){
    	SPH_ERR("Error occured by updating keywords");
    	SPH_RAISE_CPL;
        return rerr;
    }

    rerr = cpl_dfs_save_image( allframes, NULL, params,
            allframes,
            template_frame,
            self->image_zero_odd,
            CPL_TYPE_FLOAT, recipe, tproperties, ".*ESO DRS PC PROD TYPE.*", pipename, outfilename );

    if ( rerr == CPL_ERROR_NONE ){
       // rerr |= cpl_image_save( self->image_zero_odd, cpl_frame_get_filename( outframe ), CPL_TYPE_DOUBLE,
       //                        tproperties, CPL_IO_CREATE ); already saved by cpl_dfs_save_imag
        rerr |= cpl_image_save( self->image_zero_even, outfilename, CPL_TYPE_DOUBLE,
                               NULL, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->image_pi_odd, outfilename, CPL_TYPE_DOUBLE,
                               tproperties, CPL_IO_EXTEND );
        rerr |= cpl_image_save( self->image_pi_even, outfilename, CPL_TYPE_DOUBLE,
                               NULL, CPL_IO_EXTEND );

        if ( self->ovsc &&  cpl_propertylist_get_size(self->ovsc) == OVSC_SIZE ){
            ovsc_table = sph_zpl_exposure_ovsc_table_create_empty();
        	sph_zpl_exposure_ovsc_table_add_raw( self, ovsc_table );
        }

        if ( ovsc_table ){
        	rerr = cpl_table_save(ovsc_table, NULL, NULL, outfilename, CPL_IO_EXTEND);
            cpl_table_delete( ovsc_table ); ovsc_table = NULL;
        }

    } else {
      	 SPH_ERR("Could not save first image of the zpl exposure (self->image_zero_odd) using cpl_dfs_save_image");
    }

    if ( tproperties ) {
        cpl_propertylist_delete( tproperties );
    }
    if ( tpl ) {
        cpl_propertylist_delete( tpl );
    }


    // Attention: Update of the created dfs zpl exposure image product by the
    // correct SPH_COMMON_KEYWORD_SPH_TYPE  keyword
    rerr |= sph_zpl_exposure_update_prod_type( outfilename );

     SPH_RAISE_CPL;
     return cpl_error_get_code();

}




/*----------------------------------------------------------------------------*/
/**
  @brief    load sph_zpl_exposure from a frame.
  @param    frame            the frame to load from
  @param    plane            the plane to load from ( 0...n-planes)

  This function creates a new sph_zpl_exposure and loads the data from a FITS
  file with 4 extensions.
  */
/*----------------------------------------------------------------------------*/
sph_zpl_exposure*
sph_zpl_exposure_load( const cpl_frame* inframe, int iplane ) {
    sph_zpl_exposure*   	 result   		    = NULL;
    cpl_table*				 ovsc_table			= NULL;
    //cpl_array*				 selcol				= NULL;
    cpl_array*				 colnames			= NULL;
    char*                    filename            = NULL;
    int single_frame = FALSE;

    cpl_error_reset();
    result = sph_zpl_exposure_new_empty();
    if ( !result ) {
        return NULL;
    }
    if ( result->properties ) {
        cpl_propertylist_delete( result->properties ); result->properties = NULL;
    }

    //result->properties = cpl_propertylist_load_regexp( cpl_frame_get_filename( inframe ), 0, ".*ESO.*", EXT_ZERO_ODD );
    filename = cpl_strdup(cpl_frame_get_filename( inframe ));
    result->properties = sph_keyword_manager_load_properties( filename , EXT_ZERO_ODD );
    cpl_free(filename);
    if ( result->properties == NULL){
           sph_error_raise(SPH_ERROR_INFO,
                  __FILE__, __func__, __LINE__,
                   SPH_ERROR_INFO, "ESO property list doesn't exist in the frame with filename: %s",
                   cpl_frame_get_filename( inframe ));
           cpl_error_reset();
    }

    // Check if this is a single frame exposure
    single_frame = sph_utils_is_single_frame(result->properties);

    result->image_zero_odd = cpl_image_load( cpl_frame_get_filename( inframe ), CPL_TYPE_DOUBLE, iplane, EXT_ZERO_ODD );
    if ( !result->image_zero_odd ){
        sph_zpl_exposure_delete( result );
        SPH_ERROR_RAISE_ERR(cpl_error_get_code(),
        		"Could not load zimpol exposure image_zero_odd "
        		"from file: %s, iplane = %d",
                cpl_frame_get_filename( inframe ), iplane);
        return NULL;
    }

    result->image_zero_even = cpl_image_load( cpl_frame_get_filename( inframe ), CPL_TYPE_DOUBLE, iplane, EXT_ZERO_EVEN );
    if ( !result->image_zero_even ){
        sph_zpl_exposure_delete( result );
        sph_error_raise(SPH_ERROR_ERROR,
                  __FILE__, __func__, __LINE__,
                   SPH_ERROR_ERROR, "Could not load zimpol exposure image_zero_even from file: %s, iplane = %d",
                   cpl_frame_get_filename( inframe ), iplane);
        return NULL;
    }

    if (single_frame) {
      // Set image_pi_odd to all zeros for single frame exposures
      result->image_pi_odd = cpl_image_multiply_scalar_create(result->image_zero_odd, 0.0);
    } else {
      result->image_pi_odd = cpl_image_load( cpl_frame_get_filename( inframe ), CPL_TYPE_DOUBLE, iplane, EXT_PI_ODD );
    }
    if ( !result->image_pi_odd ){
        sph_zpl_exposure_delete( result );
        sph_error_raise(SPH_ERROR_ERROR,
                  __FILE__, __func__, __LINE__,
                   SPH_ERROR_ERROR, "Could not load zimpol exposure image_pi_odd from file: %s, iplane = %d",
                   cpl_frame_get_filename( inframe ), iplane);
        return NULL;
    }
    if (single_frame) {
      // Set image_pi_even to all zeros for single frame exposures
      result->image_pi_even = cpl_image_duplicate(result->image_pi_odd);
    } else {
      result->image_pi_even = cpl_image_load( cpl_frame_get_filename( inframe ), CPL_TYPE_DOUBLE, iplane, EXT_PI_EVEN );
    }
    if ( !result->image_pi_even ){
        sph_zpl_exposure_delete( result );
        sph_error_raise(SPH_ERROR_ERROR,
                  __FILE__, __func__, __LINE__,
                   SPH_ERROR_ERROR, "Could not load zimpol exposure image_pi_even from file: %s, iplane = %d",
                   cpl_frame_get_filename( inframe ), iplane);
        return NULL;
    }


    //SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO, "Number of extensions in the given file: %d", cpl_frame_get_nextensions( inframe ) )
    if ( cpl_frame_get_nextensions( inframe ) == EXT_OVSC_TABLE ) {
    	SPH_INFO_MSG("Load overscan table...");
    	//selcol = cpl_array_new( OVSC_SIZE, CPL_TYPE_DOUBLE );
    	colnames = cpl_array_new (OVSC_SIZE, CPL_TYPE_STRING );
    	cpl_array_set_string(colnames, 0, ADU1_ZERO_OVSC_MEAN);
    	cpl_array_set_string(colnames, 1, ADU1_ZERO_OVSC_RMS);
    	cpl_array_set_string(colnames, 2, ADU2_ZERO_OVSC_MEAN);
    	cpl_array_set_string(colnames, 3, ADU2_ZERO_OVSC_RMS);
    	cpl_array_set_string(colnames, 4, ADU1_PI_OVSC_MEAN);
    	cpl_array_set_string(colnames, 5, ADU1_PI_OVSC_RMS);
    	cpl_array_set_string(colnames, 6, ADU2_PI_OVSC_MEAN);
    	cpl_array_set_string(colnames, 7, ADU2_PI_OVSC_RMS);

    	ovsc_table = cpl_table_load_window( cpl_frame_get_filename( inframe ), EXT_OVSC_TABLE, 1, colnames, iplane, 1 );
    	cpl_array_delete(colnames); colnames = NULL;
    	if ( ovsc_table ){
    		colnames = cpl_table_get_column_names(ovsc_table);
    		for ( int i = 0; i < OVSC_SIZE; i++){
    			cpl_propertylist_update_double( result->ovsc, cpl_array_get_string( colnames, i),
    					cpl_table_get_double( ovsc_table,  cpl_array_get_string( colnames, i ), 0, NULL ) );
    		}
    	}
        cpl_array_delete( colnames );
        cpl_table_delete( ovsc_table );
    }


    if (cpl_error_get_code() != CPL_ERROR_NONE ){
           sph_error_raise(SPH_ERROR_ERROR,
              __FILE__, __func__, __LINE__,
               SPH_ERROR_ERROR, "CPL error has occured: %d\n"
                                "CPL error message: %s",
                                cpl_error_get_code(),
                                cpl_error_get_message_default(cpl_error_get_code()) );
           sph_zpl_exposure_delete( result ); result = NULL;
           return NULL;
    }


    return result;

}

/*----------------------------------------------------------------------------*/
/**
  @brief    Delete the sph zpl exposure.
  @param    self        the sph_zpl_exposure to delete
  @return   the error code

  @note     A CPL error is raised in case there was a problem.

  Deallocate the sph_zpl_exposure.
  */
/*----------------------------------------------------------------------------*/
void sph_zpl_exposure_delete( sph_zpl_exposure* self ){
    if ( self->properties ){
        cpl_propertylist_delete( self->properties );
        self->properties = NULL;
    }
    if ( self->image_zero_odd ){
        cpl_image_delete( self->image_zero_odd );
        self->image_zero_odd = NULL;
    }
    if ( self->image_zero_even ){
        cpl_image_delete( self->image_zero_even );
        self->image_zero_even = NULL;
    }
    if ( self->image_pi_odd ){
        cpl_image_delete( self->image_pi_odd );
        self->image_pi_odd = NULL;
    }
    if ( self->image_pi_even ){
        cpl_image_delete( self->image_pi_even );
        self->image_pi_even = NULL;
    }
    if ( self->ovsc ){
    	cpl_propertylist_delete( self->ovsc );
    	self->ovsc = NULL;
    }

    cpl_free( self );

    return;
}
