/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_irdis_model.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup Testing the IFS instrument model simple simulation code
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_test_irdis_model_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_test_irdis_model_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_irdis_model_new(void) {
    sph_test_irdis_model* insmodel = NULL;
    sph_test_ngc_ir_simulator* ngc = NULL;
    ngc = sph_test_ngc_ir_simulator_new(512, 256);

    insmodel = sph_test_irdis_model_new(ngc);
    cpl_test_assert( insmodel);

    sph_test_irdis_model_delete(insmodel);
    insmodel = NULL;
    sph_test_ngc_ir_simulator_delete(ngc);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_irdis_model_create_pdt_single_window(void) {
    sph_test_irdis_model* insmodel = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_spectral_region* reg = NULL;
    sph_test_ngc_ir_simulator* ngc = NULL;
    ngc = sph_test_ngc_ir_simulator_new(512, 256);
    insmodel = sph_test_irdis_model_new(ngc);

    cpl_test_assert( insmodel);
    sph_test_irdis_model_set_mode(insmodel, SPH_TEST_IRDIS_INSMODE_SPEC_LRS);
    pdt = sph_test_irdis_model_create_pdt_single_window(insmodel);
    cpl_test_nonnull( pdt );
    sph_pixel_description_table_save(pdt, "irdis_tab.fits", NULL);
    cpl_test_eq(pdt->nregions, 1);
    reg = sph_pixel_description_table_get_region(pdt, 1);
    cpl_test_nonnull( reg );
    cpl_test_eq(reg->maxx-reg->minx,
            insmodel->specwin_maxx-insmodel->specwin_minx);
    cpl_test_eq(reg->maxy-reg->miny,
            insmodel->specwin_maxy-insmodel->specwin_miny);

    sph_test_irdis_model_delete(insmodel);
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    insmodel = NULL;
    sph_test_ngc_ir_simulator_delete(ngc);

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_irdis_model_create_pdt(void) {
    sph_test_irdis_model* insmodel = NULL;
    sph_pixel_description_table* pdt = NULL;
    sph_spectral_region* reg = NULL;
    sph_test_ngc_ir_simulator* ngc = NULL;
    ngc = sph_test_ngc_ir_simulator_new(512, 256);

    insmodel = sph_test_irdis_model_new(ngc);

    cpl_test_assert( insmodel);
    sph_test_irdis_model_set_mode(insmodel, SPH_TEST_IRDIS_INSMODE_SPEC_LRS);
    pdt = sph_test_irdis_model_create_pdt(insmodel);
    cpl_test_nonnull( pdt );
    sph_pixel_description_table_save(pdt, "irdis_tab_whole.fits", NULL);
    cpl_test_eq(pdt->nregions, 2);
    reg = sph_pixel_description_table_get_region(pdt, 1);
    cpl_test_nonnull( reg );
    cpl_test_eq(reg->maxx-reg->minx,
            insmodel->specwin_maxx-insmodel->specwin_minx);
    cpl_test_eq(reg->maxy-reg->miny,
            insmodel->specwin_maxy-insmodel->specwin_miny);

    sph_test_irdis_model_delete(insmodel);
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    insmodel = NULL;
    sph_test_ngc_ir_simulator_delete(ngc);

    return;
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_irdis_model_process_image_specmode(void) {
    sph_test_irdis_model* insmodel = NULL;
    cpl_image* inimage = NULL;
    cpl_image* result = NULL;
    sph_pixel_description_table* pdt = NULL;
    cpl_frame* dumframe = NULL;
    int xx, yy;
    sph_test_ngc_ir_simulator* ngc = NULL;
    ngc = sph_test_ngc_ir_simulator_new(512, 256);

    insmodel = sph_test_irdis_model_new(ngc);

    cpl_test_assert( insmodel);

    sph_test_irdis_model_set_mode(insmodel, SPH_TEST_IRDIS_INSMODE_SPEC_LRS);
    pdt = sph_test_irdis_model_create_pdt_single_window(insmodel);
    cpl_test_nonnull( pdt );

    inimage = cpl_image_new(pdt->nx, pdt->ny, CPL_TYPE_DOUBLE);
    for (xx = 0; xx < cpl_image_get_size_x(inimage); ++xx) {
        for (yy = 0; yy < cpl_image_get_size_y(inimage); ++yy) {
            cpl_image_set(inimage, xx + 1, yy + 1, yy + 100);
        }
    }

    result = sph_test_irdis_model_process_image(insmodel, inimage);
    cpl_test_nonnull( result );

    dumframe = sph_test_frame_image(result, CPL_TYPE_DOUBLE, "IRDIS TEST",
            CPL_FRAME_GROUP_RAW);

    sph_test_irdis_model_delete(insmodel);
    sph_pixel_description_table_delete(pdt);
    pdt = NULL;
    insmodel = NULL;
    cpl_image_delete(result);
    result = NULL;
    cpl_image_delete(inimage);
    inimage = NULL;
    sph_test_ngc_ir_simulator_delete(ngc);
    cpl_frame_delete(dumframe);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_irdis_model_process_image_dbi(void) {
    sph_test_irdis_model* insmodel = NULL;
    cpl_image* inimage = NULL;
    cpl_image* result = NULL;
    double flux0 = 0.0;
    double flux = 0.0;
    cpl_frame* dumframe = NULL;
    int xx, yy;
    sph_test_ngc_ir_simulator* ngc = NULL;
    ngc = sph_test_ngc_ir_simulator_new(512, 256);

    insmodel = sph_test_irdis_model_new(ngc);

    cpl_test_assert( insmodel);

    sph_test_irdis_model_set_mode(insmodel, SPH_TEST_IRDIS_INSMODE_DBI);

    inimage = cpl_image_new(insmodel->nx, insmodel->ny, CPL_TYPE_DOUBLE);
    for (xx = 0; xx < cpl_image_get_size_x(inimage); ++xx) {
        for (yy = 0; yy < cpl_image_get_size_y(inimage); ++yy) {
            cpl_image_set(inimage, xx + 1, yy + 1, 100.0);
        }
    }
    flux0 = cpl_image_get_flux(inimage);

    result = sph_test_irdis_model_process_image(insmodel, inimage);
    cpl_test_nonnull( result );

    dumframe = sph_test_frame_image(result, CPL_TYPE_DOUBLE, "IRDIS_DBI_TEST",
            CPL_FRAME_GROUP_RAW);

    flux0 = flux0 / (insmodel->nx * insmodel->ny);
    flux0 = flux0 * CPL_MATH_PI
            * (insmodel->radius_left * insmodel->radius_left
                    + insmodel->radius_right * insmodel->radius_right);
    flux = cpl_image_get_flux(result);
    cpl_test_abs(
            flux,
            flux0,
            CPL_MATH_2PI * 100 * (insmodel->radius_left + insmodel->radius_right));
    sph_test_irdis_model_delete(insmodel);
    insmodel = NULL;
    cpl_image_delete(result);
    result = NULL;
    cpl_image_delete(inimage);
    inimage = NULL;
    sph_test_ngc_ir_simulator_delete(ngc);
    cpl_frame_delete(dumframe);
    return;
}

//*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_test_irdis_model_delete(void) {
    sph_test_irdis_model* insmodel = NULL;
    sph_test_ngc_ir_simulator* ngc = NULL;
    ngc = sph_test_ngc_ir_simulator_new(512, 256);

    insmodel = sph_test_irdis_model_new(ngc);
    cpl_test_assert( insmodel);

    sph_test_irdis_model_delete(insmodel);
    insmodel = NULL;
    sph_test_ngc_ir_simulator_delete(ngc);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testsuite for sph_test_irdis_model",
            cutest_init_test_irdis_model_testsuite,
            cutest_clean_test_irdis_model_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "sph_test_irdis_model_new",
                    cutest_test_irdis_model_new)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_test_irdis_model_create_pdt",
                    cutest_test_irdis_model_create_pdt)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_test_irdis_model_create_pdt_single_window",
                    cutest_test_irdis_model_create_pdt_single_window)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite,
                    "sph_test_irdis_model_process_image_specmode",
                    cutest_test_irdis_model_process_image_specmode)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_test_irdis_model_process_image_dbi",
                    cutest_test_irdis_model_process_image_dbi)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "sph_test_irdis_model_delete",
                    cutest_test_irdis_model_delete)) {
        return sph_test_get_error();
    }
    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
