/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_error.h"
#include "sph_test.h"
#include "sph_utils.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_ifs_distortion_map.h"
#include "sph_ifs_test_fixtures_science_raw.h"
#include "sph_ifs_test_fixtures_calib_products.h"
#include "sph_ifs_tags.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/*-                            Declarations of private functions              */
/*----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_ifs_distortion_map_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_ifs_distortion_map_clean_testsuite(void) {
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_ifs_distortion_map_test_it(void) {
    sph_test_ngc_ir_simulator* ngc = NULL;
    sph_test_ifs_instrument_model* ifsmodel = NULL;
    cpl_frameset* rawframes = NULL;
    sph_ifs_distortion_map* recipe = NULL;
    cpl_parameterlist* params = NULL;

    ngc = sph_test_ngc_ir_simulator_new(512, 512);
    cpl_test_nonnull( ngc );
    sph_test_ngc_ir_simulator_set_add_noise(ngc, 0, 1);
    ifsmodel = sph_test_ifs_instrument_model_new(ngc->det_size_x,
            SPH_IFS_LENSLET_MODEL_LENSLET_SIZE_MICRONS, 40);
    cpl_test_nonnull( ifsmodel );
    rawframes = sph_ifs_test_fixtures_science_raw_point_grid(ngc, ifsmodel);
    cpl_test_nonnull( rawframes );
    cpl_test_eq(
            sph_test_retag_frames_by_group( rawframes, CPL_FRAME_GROUP_RAW, SPH_IFS_TAG_DISTORTION_MAP_RAW),
            CPL_ERROR_NONE);

    cpl_frameset_insert(rawframes,
            sph_ifs_test_fixtures_calib_products_master_dark(ngc));

    cpl_frameset_insert(rawframes,
            sph_ifs_test_fixtures_calib_products_wave_calib(ifsmodel));
    cpl_frameset_insert(rawframes,
            sph_ifs_test_fixtures_calib_products_ifu_flat(ngc, ifsmodel));

    params = sph_ifs_distortion_map_create_paramlist();
    cpl_test_nonnull( params );
    recipe = sph_ifs_distortion_map_new(rawframes, params);
    cpl_test_nonnull( recipe );
    recipe->coll_alg = 1;
    cpl_test_eq_error(sph_ifs_distortion_map_run(recipe), CPL_ERROR_NONE);

    sph_test_ngc_ir_simulator_delete(ngc);
    sph_test_ifs_instrument_model_delete(ifsmodel);
    cpl_frameset_delete(rawframes);
    cpl_parameterlist_delete(params);
    sph_ifs_distortion_map_delete(recipe);
    return;
}

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;

    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("IFS distortion map testsuite",
            cutest_ifs_distortion_map_init_testsuite,
            cutest_ifs_distortion_map_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (cpl_version_get_major() > 7 ||
        (cpl_version_get_major() == 7 && (cpl_version_get_minor() > 1 ||
                                          (cpl_version_get_minor() == 1 &&
                                           cpl_version_get_micro() > 0)))) {
        if (NULL
            == sph_test_do(pSuite, "Testing fit distortion map",
                           cutest_ifs_distortion_map_test_it)) {
            return sph_test_get_error();
        }
    } else {
        cpl_msg_warning(cpl_func, "Skipping tests with too old CPL: %s",
                        cpl_version_get_version());
    }

    return sph_test_end();
}

/**@}*/
