/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#define _sph_ifs_test_fixtures_science_raw_C
#include "sph_ifs_test_fixtures_science_raw.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_master_frame.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ifs_tags.h"
#include "sph_test_ifs_instrument_model.h"
#include "sph_ifs_science_dr.h"
#include "sph_common_keywords.h"
#include "sph_test_image_tools.h"
#include "sph_fits.h"
#include "sph_time.h"
#include "sph_frame_validator.h"
#include <math.h>
#include <cpl.h>
#undef _sph_ifs_test_fixtures_science_raw_C

/*----------------------------------------------------------------------------*/
/**
 * @defgroup A SPHERE API Module
 * @par Synopsis:
 * @code
 * typedef _module_ {
 * } module
 * @endcode
 * @par Desciption:
 *
 * This module provides functionality for apertures, extending the functionality
 * as it exists for cpl_apertures.
 */
/*----------------------------------------------------------------------------*/
/**@{*/
cpl_frameset*
sph_ifs_test_fixtures_science_raw_standard(sph_test_ngc_ir_simulator* ngc,
        sph_test_ifs_instrument_model* insmodel) {
    sph_test_pupilimage_creator* imcreator = NULL;
    cpl_frameset* pupframeset = NULL;
    cpl_frame* pupframe = NULL;
    cpl_image* pupim = NULL;
    cpl_image* insim = NULL;
    cpl_frameset* insframeset = NULL;
    cpl_frameset* result = NULL;
    int ff = 0;
    int nframes = 5;
    int nplanes = 1;
    cpl_test_error(CPL_ERROR_NONE);
    imcreator = sph_test_pupilimage_creator_new();
    if (imcreator) {
        insframeset = cpl_frameset_new();
        sph_test_pupilimage_creator_set_size(imcreator, ngc->det_size_x,
                ngc->det_size_y);
        sph_test_pupilimage_creator_create_binary_star_frames(imcreator, 50.0,
                45.0, 10.0, 1.0e8, 2000.0, 20.0, nframes);
        pupframeset = sph_test_pupilimage_creator_get_frameset(imcreator);
        pupframe = cpl_frameset_get_first(pupframeset);
        while (pupframe) {
            pupim = cpl_image_load(cpl_frame_get_filename(pupframe),
                    CPL_TYPE_DOUBLE, 0, 0);
            insim = sph_test_ifs_instrument_model_disperse(insmodel, pupim, 0.0,
                    0.0);
            cpl_frameset_insert(
                    insframeset,
                    sph_test_frame_image(insim, CPL_TYPE_DOUBLE,
                            SPH_IFS_TAG_SCIENCE_DR_RAW, CPL_FRAME_GROUP_RAW));
            pupframe = cpl_frameset_get_next(pupframeset);
            cpl_image_delete(insim);
            insim = NULL;
            cpl_image_delete(pupim);
            pupim = NULL;
        }
        result = cpl_frameset_new();
        for (ff = 0; ff < nframes; ++ff) {

            cpl_frameset_insert(
                    result,
                    sph_test_ngc_ir_simulator_raw_cube_from_frame(
                            ngc,
                            cpl_frameset_get_position(insframeset, ff),
                            nplanes,
                            "raw_frames_for_ifs_science_dr.fits",
                            ff,
                            sph_test_ifs_instrument_model_get_proplist(
                                    insmodel), SPH_IFS_TAG_SCIENCE_DR_RAW));
        }
        cpl_frameset_delete(insframeset);
        insframeset = NULL;
    }
    sph_test_pupilimage_creator_delete(imcreator);
    imcreator = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    return result;
}

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/
cpl_frameset*
sph_ifs_test_fixtures_science_raw_point_grid(sph_test_ngc_ir_simulator* ngc,
        sph_test_ifs_instrument_model* insmodel) {
    sph_test_pupilimage_creator* imcreator = NULL;
    cpl_frameset* pupframeset = NULL;
    cpl_frame* pupframe = NULL;
    cpl_image* pupim = NULL;
    cpl_image* insim = NULL;
    cpl_frameset* insframeset = NULL;
    cpl_frameset* result = NULL;
    int ff = 0;
    int nframes = 1;
    int nplanes = 1;
    cpl_propertylist* pl = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    imcreator = sph_test_pupilimage_creator_new();
    if (imcreator) {
        insframeset = cpl_frameset_new();
        sph_test_pupilimage_creator_set_size(imcreator, ngc->det_size_x,
                ngc->det_size_y);
        sph_test_pupilimage_creator_set_image(
                imcreator,
                sph_test_image_tools_create_grid_image(ngc->det_size_x,
                        ngc->det_size_y, 12, 12, CPL_TYPE_DOUBLE, 10000000.0,
                        3.0));

        sph_test_pupilimage_creator_create_frames(imcreator, nframes);
        pupframeset = sph_test_pupilimage_creator_get_frameset(imcreator);
        pupframe = cpl_frameset_get_first(pupframeset);
        while (pupframe) {
            pupim = cpl_image_load(cpl_frame_get_filename(pupframe),
                    CPL_TYPE_DOUBLE, 0, 0);
            insim = sph_test_ifs_instrument_model_disperse(insmodel, pupim, 0.0,
                    0.0);
            cpl_frameset_insert(
                    insframeset,
                    sph_test_frame_image(insim, CPL_TYPE_DOUBLE,
                            SPH_IFS_TAG_SCIENCE_DR_RAW, CPL_FRAME_GROUP_RAW));
            pupframe = cpl_frameset_get_next(pupframeset);
            cpl_image_delete(insim);
            insim = NULL;
            cpl_image_delete(pupim);
            pupim = NULL;
        }
        result = cpl_frameset_new();
        for (ff = 0; ff < nframes; ++ff) {
            cpl_frameset_insert(
                    result,
                    sph_test_ngc_ir_simulator_raw_cube_from_frame(
                            ngc,
                            cpl_frameset_get_position(insframeset, ff),
                            nplanes,
                            "raw_frames_for_ifs_science_dr.fits",
                            ff,
                            pl = sph_test_ifs_instrument_model_get_proplist(
                                    insmodel), SPH_IFS_TAG_SCIENCE_DR_RAW));
            cpl_propertylist_delete(pl);
        }
        cpl_frameset_delete(insframeset);
        insframeset = NULL;
    }
    sph_test_pupilimage_creator_delete(imcreator);
    imcreator = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    return result;
}



cpl_frameset*
sph_ifs_test_fixtures_science_raw_constant_flux(sph_test_ngc_ir_simulator* ngc,
        sph_test_ifs_instrument_model* insmodel) {
    return sph_ifs_test_fixtures_raw_constant_flux(ngc,insmodel,SPH_IFS_TAG_SCIENCE_DR_RAW);
}

cpl_frameset*
sph_ifs_test_fixtures_raw_constant_flux(sph_test_ngc_ir_simulator* ngc,
        sph_test_ifs_instrument_model* insmodel,
        const char* tag) {
    cpl_image* pupim = NULL;
    cpl_frameset* insframeset = NULL;
    cpl_frameset* result = NULL;
    cpl_test_error(CPL_ERROR_NONE);

    cpl_test_nonnull(insmodel);

    insframeset = cpl_frameset_new();
    result = cpl_frameset_new();
    pupim = sph_test_image_tools_create_flat_image_double(ngc->det_size_x,
            ngc->det_size_y, 1000.0);
    cpl_frameset_insert(
            result,
            sph_test_frame_image(pupim, CPL_TYPE_DOUBLE,
                    tag, CPL_FRAME_GROUP_RAW));
    cpl_image_delete(pupim);
    pupim = NULL;
    cpl_frameset_delete(insframeset);
    cpl_test_error(CPL_ERROR_NONE);
    return result;
}

cpl_frameset*
sph_ifs_test_fixtures_science_raw_constant_flux_dispersed(
        sph_test_ngc_ir_simulator* ngc, sph_test_ifs_instrument_model* insmodel,
        double value, int nframes, int nplanes) {
    cpl_frame* aframe = NULL;
    cpl_image* pupim = NULL;
    cpl_image* insim = NULL;
    cpl_frameset* result = NULL;
    int ff = 0;
    int pp = 0;
    cpl_image* rawimage = NULL;
    cpl_test_error(CPL_ERROR_NONE);
    result = cpl_frameset_new();
    for (ff = 0; ff < nframes; ++ff) {
        aframe = sph_filemanager_create_temp_frame("IFS_CONST",
                SPH_IFS_TAG_SCIENCE_DR_RAW, CPL_FRAME_GROUP_RAW);
        pupim = sph_test_image_tools_create_flat_image_double(ngc->det_size_x,
                ngc->det_size_y, value);
        insim = sph_test_ifs_instrument_model_disperse(insmodel, pupim, 0.0,
                0.0);
        cpl_image_delete(pupim);
        pupim = NULL;
        for (pp = 0; pp < nplanes; ++pp) {
            rawimage = sph_test_ngc_ir_simulator_create_raw_image(ngc, insim,
                    0.0, 0.0);
            sph_cube_append_image(cpl_frame_get_filename(aframe), rawimage,
                    NULL, 0);
            cpl_image_delete(rawimage);
            rawimage = NULL;
        }
        sph_cube_finalise_file(cpl_frame_get_filename(aframe));
        cpl_frameset_insert(result, aframe);
        cpl_image_delete(insim);
        insim = NULL;
    }
    cpl_test_error(CPL_ERROR_NONE);
    return result;
}

#ifdef SPH_TEST_FIXTURES_SCIENCE
/*----------------------------------------------------------------------------*/
/**
 * @brief Add speckles to an image cube
 * @param inouttlist     the cube to modify
 * @param xcent          the optical centre
 * @param ycent          the optical centre
 * @param lams           the wavelengths for each plane
 * @param xpos           x-positions of the speckles
 * @param ypos           y-positions of the speckles
 * @param siz            sizes of the speckles
 * @param reflambda      the reference lambda
 *
 * @return error code
 *
 * This adds the given speckles to the cube.
 * The positions and sizes
 * are assumed to be the parameters at the reference wavelength.
 * For other planes, these will be scaled according to wavelengths.
 */
/*----------------------------------------------------------------------------*/
static
sph_error_code sph_ifs_test_fixtures_science_add_speckles(
        cpl_imagelist* inoutlist, double xcent, double ycent, cpl_vector* lams,
        cpl_vector* xpos, cpl_vector* ypos, cpl_vector* siz, cpl_vector* vals,
        double reflambda) {
    int nn = 0;
    int pp = 0;
    int nplanes = 0;
    int nspeckles = 0;
    double dx = 0.0;
    double dy = 0.0;
    double s = 0.0;
    double scaling = 0.0;
    cpl_vector* totfluxes = NULL;

    cpl_ensure_code(inoutlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(lams, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(xpos, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ypos, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(siz, CPL_ERROR_NULL_INPUT);

    nplanes = cpl_imagelist_get_size(inoutlist);
    nspeckles = cpl_vector_get_size(xpos);
    cpl_ensure_code(cpl_vector_get_size(lams) == nplanes,
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_vector_get_size(ypos) == nspeckles,
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(cpl_vector_get_size(siz) == nspeckles,
            CPL_ERROR_ILLEGAL_INPUT);

    totfluxes = cpl_vector_new(nplanes);
    for (pp = 0; pp < nplanes; ++pp) {
        scaling = cpl_vector_get(lams, pp) / reflambda;

        for (nn = 0; nn < nspeckles; ++nn) {
            s = cpl_vector_get(siz, nn) * scaling;
            dx = (cpl_vector_get(xpos, nn) - xcent) * scaling;
            dy = (cpl_vector_get(ypos, nn) - ycent) * scaling;
            sph_test_image_tools_add_gauss(cpl_imagelist_get(inoutlist, pp),
                    xcent + dx, ycent + dy, s, cpl_vector_get(vals, nn));
            cpl_error_reset();
        }

        cpl_vector_set(totfluxes, pp,
                cpl_image_get_absflux(cpl_imagelist_get(inoutlist, pp)));

    }
    cpl_vector_dump(totfluxes, stdout);
    cpl_vector_delete(totfluxes);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Add random speckles to an image cube
 * @param inouttlist     the cube to modify
 * @param xcent          the optical centre
 * @param ycent          the optical centre
 * @param nspeck         the number of speckles to add
 * @param lams           the wavelengths for each plane
 * @param reflambda      the reference lambda
 *
 * @return error code
 *
 * This adds random speckles the given speckles to the cube.
 * The positions and sizes
 * are assumed to be the parameters at the reference wavelength.
 * For other planes, these will be scaled according to wavelengths.
 */
/*----------------------------------------------------------------------------*/
static
sph_error_code sph_ifs_test_fixtures_science_add_random_speckles(
        cpl_imagelist* inoutlist, double xcent, double ycent, int nspeck,
        cpl_vector* lams, double reflambda, gsl_rng* pRNG) {
    int nn = 0;
    double x = 0.0;
    double y = 0.0;
    double maxx = 0.0;
    double maxy = 0.0;
    double s = 0.0;
    double val = 0.0;
    cpl_vector* xpos = NULL;
    cpl_vector* ypos = NULL;
    cpl_vector* siz = NULL;
    cpl_vector* vals = NULL;
    int remprng = 0;

    cpl_ensure_code(inoutlist, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(lams, CPL_ERROR_NULL_INPUT);
    if (!pRNG) {
        pRNG = gsl_rng_alloc(gsl_rng_taus);
        gsl_rng_set(pRNG, sph_time_get_time_seed());
        remprng = 1;
    }

    maxx = cpl_image_get_size_x(cpl_imagelist_get(inoutlist, 0));
    maxy = cpl_image_get_size_y(cpl_imagelist_get(inoutlist, 0));

    xpos = cpl_vector_new(nspeck);
    ypos = cpl_vector_new(nspeck);
    siz = cpl_vector_new(nspeck);
    vals = cpl_vector_new(nspeck);

    for (nn = 0; nn < nspeck; ++nn) {
        s = x = y = val = 0.0;
        while (s < 2.0) {
            s = 2.0 + gsl_ran_gaussian(pRNG, 1.0);
        }
        while (x - s <= 0.0 || x + s >= maxx) {
            x = xcent + gsl_ran_gaussian(pRNG, 30.0);
        }
        while (y - s <= 0.0 || y + s >= maxy) {
            y = ycent + gsl_ran_gaussian(pRNG, 30.0);
        }
        while (val <= 0.0) {
            val = gsl_ran_gaussian(pRNG, 1000.0);
        }
        cpl_vector_set(siz, nn, s);
        cpl_vector_set(xpos, nn, x);
        cpl_vector_set(ypos, nn, y);
        cpl_vector_set(vals, nn, val);
    }

    sph_ifs_test_fixtures_science_add_speckles(inoutlist, xcent, ycent, lams,
            xpos, ypos, siz, vals, reflambda);

    cpl_vector_delete(xpos);
    cpl_vector_delete(ypos);
    cpl_vector_delete(vals);
    cpl_vector_delete(siz);
    if (remprng) {
        gsl_rng_free(pRNG);

    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
#endif

/**@}*/
