/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_ird_common_science.h"
#include "sph_test_image_tools.h"
#include "sph_utils.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_irdis_model.h"
#include "sph_common_science.h"
#include "sph_time.h"
#include "sph_fctable.h"
#include "sph_ird_tags.h"
#include "sph_fits.h"
#include "sph_test_ird_fixtures.h"
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    return sph_test_nop_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    return sph_end_test();
}
static cpl_parameterlist*
cutest_sph_ird_common_science_create_paramlist(void) {
    cpl_parameterlist* paramlist = NULL;

    paramlist = cpl_parameterlist_new();
    return paramlist;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_common_science_process_cubes(void) {
    sph_ird_common_science* sci = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* dark_frame = NULL;
    cpl_frame* flat_frame = NULL;
    cpl_frame* bpix_frame = NULL;
    cpl_frame* distort_frame = NULL;
    cpl_frame* filter_frame = NULL;
    sph_ird_instrument_model* model = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* left_out = NULL;
    cpl_frameset* right_out = NULL;
    cpl_frameset* total_out = NULL;
    cpl_frameset* scframes = NULL;
    cpl_frameset* rawframes = NULL;
    cpl_image* im = NULL;
    int bpix = 0;

    cpl_test_error(CPL_ERROR_NONE);
    model = sph_test_ird_fixtures_instrument_model(1024, 512);
    inparams = cutest_sph_ird_common_science_create_paramlist();
    dark_frame = sph_test_ird_fixtures_create_dark(model, 121, 2.0);
    flat_frame = sph_test_ird_fixtures_create_flat(model, 0.1);
    cpl_test_error(CPL_ERROR_NONE);
    inframes = sph_test_ird_fixtures_create_science_frames_no_dither(3, 5,
            model, dark_frame, flat_frame, NULL);
    cpl_test_nonnull( inframes );
    cpl_test_eq(cpl_frameset_get_size(inframes), 3);
    rawframes = cpl_frameset_duplicate(inframes);
    scframes = sph_test_ird_fixtures_create_star_centre_frames_ascii_no_dither(
            3, 5, model, inframes);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_frameset_get_size(scframes),
            cpl_frameset_get_size(inframes));
    sci = sph_ird_common_science_new(inframes, inparams, dark_frame, flat_frame,
              bpix_frame, distort_frame, filter_frame, flat_frame,
                                     NULL, "TEST", "TESTTAG");
    sph_time_start_timer(1);
    cpl_test_eq(
            sph_ird_common_science_process_cubes( sci, rawframes, scframes,
                                                  &left_out, &right_out,
                                                  &total_out,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_LEFT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_RIGHT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_TOTAL_CUBE),
            CPL_ERROR_NONE);
    SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
            "Time taken by processing: %f", sph_time_stop_timer(1));
    cpl_test_eq( cpl_frameset_get_size(left_out),
            cpl_frameset_get_size(rawframes));
    cpl_test_eq( cpl_frameset_get_size(right_out),
            cpl_frameset_get_size(rawframes));
    cpl_test_eq( cpl_frameset_get_size(total_out),
            cpl_frameset_get_size(rawframes));

    im = cpl_image_load(
            cpl_frame_get_filename(cpl_frameset_get_first(total_out)),
            CPL_TYPE_DOUBLE, 1, 0);
    cpl_test_nonnull( im );

    cpl_test(
            cpl_image_get(im, cpl_image_get_size_x(im)/2 + 1, cpl_image_get_size_y(im)/2 + 1, &bpix) > 1000.0);
    cpl_image_delete(im);

    cpl_frameset_delete(left_out);
    left_out = NULL;
    cpl_frameset_delete(right_out);
    right_out = NULL;
    cpl_frameset_delete(total_out);
    total_out = NULL;
    cpl_frameset_delete(scframes);
    scframes = NULL;
    cpl_frame_delete(dark_frame);
    dark_frame = NULL;
    cpl_frame_delete(flat_frame);
    flat_frame = NULL;
    cpl_frame_delete(bpix_frame);
    bpix_frame = NULL;
    cpl_frame_delete(distort_frame);
    distort_frame = NULL;
    cpl_frame_delete(filter_frame);
    filter_frame = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    sph_ird_common_science_delete(sci);
    sph_ird_instrument_model_delete(model);
    model = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_common_science_process_cubes_window(void) {
    sph_ird_common_science* sci = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* dark_frame = NULL;
    cpl_frame* flat_frame = NULL;
    cpl_frame* bpix_frame = NULL;
    cpl_frame* distort_frame = NULL;
    cpl_frame* filter_frame = NULL;
    sph_ird_instrument_model* model = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* left_out = NULL;
    cpl_frameset* right_out = NULL;
    cpl_frameset* total_out = NULL;
    cpl_frameset* scframes = NULL;
    cpl_frameset* rawframes = NULL;
    cpl_image* im = NULL;
    int bpix = 0;

    cpl_test_error(CPL_ERROR_NONE);
    model = sph_test_ird_fixtures_instrument_model(1024, 512);
    inparams = cutest_sph_ird_common_science_create_paramlist();
    dark_frame = sph_test_ird_fixtures_create_dark(model, 121, 2.0);
    flat_frame = sph_test_ird_fixtures_create_flat(model, 0.1);
    cpl_test_error(CPL_ERROR_NONE);
    inframes = sph_test_ird_fixtures_create_science_frames_no_dither(3, 5,
            model, dark_frame, flat_frame, NULL);
    cpl_test_nonnull( inframes );
    cpl_test_eq(cpl_frameset_get_size(inframes), 3);
    rawframes = cpl_frameset_duplicate(inframes);
    scframes = sph_test_ird_fixtures_create_star_centre_frames_ascii_no_dither(
            3, 5, model, inframes);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_frameset_get_size(scframes),
            cpl_frameset_get_size(inframes));
    sci = sph_ird_common_science_new(inframes, inparams, dark_frame, flat_frame,
            bpix_frame, distort_frame, filter_frame, flat_frame,
                                     NULL, "TEST", "TESTTAG");
    sph_time_start_timer(1);
    sph_ird_instrument_model_set_windows(sci->irdmodel, 200);
    cpl_test_eq(
            sph_ird_common_science_process_cubes( sci, rawframes, scframes, &left_out, &right_out, &total_out,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_LEFT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_RIGHT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_TOTAL_CUBE),
            CPL_ERROR_NONE);
    SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
            "Time taken by processing: %f", sph_time_stop_timer(1));
    cpl_test_eq( cpl_frameset_get_size(left_out),
            cpl_frameset_get_size(rawframes));
    cpl_test_eq( cpl_frameset_get_size(right_out),
            cpl_frameset_get_size(rawframes));
    cpl_test_eq( cpl_frameset_get_size(total_out),
            cpl_frameset_get_size(rawframes));

    im = cpl_image_load(
            cpl_frame_get_filename(cpl_frameset_get_first(total_out)),
            CPL_TYPE_DOUBLE, 1, 0);
    cpl_test_nonnull( im );

    cpl_test(
            cpl_image_get(im, cpl_image_get_size_x(im)/2 + 1, cpl_image_get_size_y(im)/2 + 1, &bpix) > 1000.0);
    cpl_image_delete(im);
    cpl_frameset_delete(left_out);
    left_out = NULL;
    cpl_frameset_delete(right_out);
    right_out = NULL;
    cpl_frameset_delete(total_out);
    total_out = NULL;
    cpl_frameset_delete(scframes);
    scframes = NULL;
    cpl_frame_delete(dark_frame);
    dark_frame = NULL;
    cpl_frame_delete(flat_frame);
    flat_frame = NULL;
    cpl_frame_delete(bpix_frame);
    bpix_frame = NULL;
    cpl_frame_delete(distort_frame);
    distort_frame = NULL;
    cpl_frame_delete(filter_frame);
    filter_frame = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    sph_ird_common_science_delete(sci);
    sph_ird_instrument_model_delete(model);
    model = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_common_science_process_cubes_sdi_and_adi(void) {
    sph_ird_common_science* sci = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* dark_frame = NULL;
    cpl_frame* flat_frame = NULL;
    cpl_frame* bpix_frame = NULL;
    cpl_frame* distort_frame = NULL;
    cpl_frame* filter_frame = NULL;
    sph_ird_instrument_model* model = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* left_out = NULL;
    cpl_frameset* right_out = NULL;
    cpl_frameset* total_out = NULL;
    cpl_frameset* scframes = NULL;
    cpl_frameset* rawframes = NULL;

    cpl_test_error(CPL_ERROR_NONE);
    model = sph_test_ird_fixtures_instrument_model(1024, 512);
    inparams = cutest_sph_ird_common_science_create_paramlist();
    dark_frame = sph_test_ird_fixtures_create_dark(model, 121, 2.0);
    flat_frame = sph_test_ird_fixtures_create_flat(model, 0.1);
    cpl_test_error(CPL_ERROR_NONE);
    inframes = cpl_frameset_new();
    cpl_frameset_insert(
            inframes,
            sph_test_ird_fixtures_create_science_frame_stable_fc_rot(10, model,
                    0.0, 0.0, dark_frame, flat_frame, NULL, 0.0, 5.0, 1.1));
    cpl_test_nonnull( inframes );
    cpl_test_eq(cpl_frameset_get_size(inframes), 1);
    rawframes = cpl_frameset_duplicate(inframes);
    scframes = sph_test_ird_fixtures_create_star_centre_frames_ascii_no_dither(
            1, 10, model, inframes);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_eq(cpl_frameset_get_size(scframes),
            cpl_frameset_get_size(inframes));
    sci = sph_ird_common_science_new(inframes, inparams, dark_frame, flat_frame,
            bpix_frame, distort_frame, filter_frame, flat_frame,
                                     NULL, "TEST", "TESTTAG");

    sci->flag_sdi = 1;
    sci->minr = 2.0;
    sci->maxr = 100.0;

    sph_time_start_timer(1);
    cpl_test_eq(
            sph_ird_common_science_process_cubes( sci, rawframes, scframes, &left_out, &right_out, &total_out,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_LEFT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_RIGHT_CUBE,
                                      SPH_IRD_TAG_SCIENCE_DBI_CALIB_TOTAL_CUBE),
            CPL_ERROR_NONE);
    SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
            "Time taken by processing: %f", sph_time_stop_timer(1));
    cpl_test_eq( cpl_frameset_get_size(left_out),
            cpl_frameset_get_size(rawframes));
    cpl_test_eq( cpl_frameset_get_size(right_out),
            cpl_frameset_get_size(rawframes));
    cpl_test_eq( cpl_frameset_get_size(total_out),
            cpl_frameset_get_size(rawframes));
    cpl_frameset_delete(left_out);
    left_out = NULL;
    cpl_frameset_delete(right_out);
    right_out = NULL;
    cpl_frameset_delete(total_out);
    total_out = NULL;
    cpl_frameset_delete(scframes);
    scframes = NULL;
    cpl_frame_delete(dark_frame);
    dark_frame = NULL;
    cpl_frame_delete(flat_frame);
    flat_frame = NULL;
    cpl_frame_delete(bpix_frame);
    bpix_frame = NULL;
    cpl_frame_delete(distort_frame);
    distort_frame = NULL;
    cpl_frame_delete(filter_frame);
    filter_frame = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    cpl_frameset_delete(rawframes);
    rawframes = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    sph_ird_common_science_delete(sci);
    sph_ird_instrument_model_delete(model);
    model = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testing sph_ird_common_science",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing many cubes input",
                    cutest_sph_ird_common_science_process_cubes)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing many cubes input window mode",
                    cutest_sph_ird_common_science_process_cubes_window)) {
        return sph_test_get_error();
    }
    if ( NULL == sph_test_do(pSuite, "Testing many cubes input adi and sdi",
                             cutest_sph_ird_common_science_process_cubes_sdi_and_adi ) )
    {
       return sph_test_get_error();
    }

    return sph_test_end();
}

/**@}*/
