/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_ird_tags.h"
#include "sph_ird_common_science.h"
#include "sph_ird_star_center.h"
#include "sph_ird_star_center.c"
#include "sph_ird_star_center_run.c"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup Star center test with cross schaped waffle
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This is a dedicated test for the star center recipe to verify
 * that a cross shaped input waffle image can be used to determine
 * the star center.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

static
int cutest_init_testsuite(void);
static
int cutest_clean_testsuite(void);
static
void cutest_sph_ird_star_center_run_single_star(void);
static
void cutest_sph_ird_star_center_run_x_pattern(void);
static
void cutest_sph_ird_star_center_run_plus_pattern(void);
/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("Testsuite for star center recipe",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing run with single star",
                    cutest_sph_ird_star_center_run_single_star)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing run with x pattern",
                    cutest_sph_ird_star_center_run_x_pattern)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing run with + pattern",
                    cutest_sph_ird_star_center_run_plus_pattern)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_star_center_run_single_star(void) {
    cpl_image* im = NULL;
    cpl_frame* aframe = NULL;
    sph_ird_star_center* recipe = NULL;
    cpl_frameset* inframes = NULL;
    cpl_parameterlist* inparams = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_fctable* fctab = NULL;
    char* fname = NULL;
    char* fname1 = NULL;
    double bgvalue = 10000.0;
    double peakvalue = 35000.0;
    double vpeak = 0.0;

    cpl_test_error(CPL_ERROR_NONE);

    // Preparing basic image
    im = sph_test_image_tools_create_flat_image_double(2048, 1024, 0.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im,
            cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5,
            cpl_image_get_size_y(im) / 2.0 + 0.5, 3.0, 1000000.0);
    vpeak = cpl_image_get_max(im);
    cpl_image_multiply_scalar(im, peakvalue / vpeak);
    cpl_image_add_scalar(im, bgvalue);

    // Adding noise and keywords
    cpl_test_nonnull( im );
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    aframe = sph_test_frame_image(im, CPL_BPP_32_SIGNED,
            SPH_IRD_TAG_WAFFLE_IMAGE_RAW, CPL_FRAME_GROUP_RAW);
    sph_test_frame_add_key_double(aframe, 0, "MJD-OBS",
            2000.0);
    sph_test_frame_add_key_string(aframe, 0, "ESO DET FRAM UTC",
            "2013-05-18T13:43:21");
    sph_test_frame_add_key_double(aframe, 0, "EXPTIME", 2.0);

    cpl_image_delete(im);
    im = NULL;
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, aframe);

    // Creating dummy dark and flat
    dark = sph_master_frame_new(2048, 1024);
    flat = sph_master_frame_new(2048, 1024);
    cpl_image_add_scalar(flat->image, 1.0);
    cpl_image_add_scalar(dark->image, 10.0);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_DARK",
            SPH_IRD_TAG_DARK_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(dark, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_FLAT",
            SPH_IRD_TAG_FLAT_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(flat, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);

    // Creating recipe
    inparams = sph_ird_star_center_create_paramlist();
    recipe = sph_ird_star_center_new(inframes, inparams);
    cpl_test_nonnull( recipe );
    recipe->tot_fctab = NULL;
    recipe->use_waffle = 0;
    cpl_test_error(CPL_ERROR_NONE);

    // Running recipe
    cpl_test_eq_error(sph_ird_star_center_run(recipe), CPL_ERROR_NONE);
    cpl_error_reset();

    // Checking results
    fname1 = sph_filemanager_new_filename_from_base(
            cpl_frame_get_filename(cpl_frameset_get_position(inframes, 0)),
            "fctable");

    fctab = sph_fctable_load_fits_IRDIS("star_center.fits");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( fctab );
    cpl_test_eq(sph_fctable_get_size(fctab), 1);

    cpl_test_abs(cpl_vector_get(fctab->centresL_x,0), 512.5, 0.03);
    cpl_test_abs(cpl_vector_get(fctab->centresL_y,0), 512.5, 0.03);
    cpl_test_abs(cpl_vector_get(fctab->centresR_x,0), 512.5, 0.03);
    cpl_test_abs(cpl_vector_get(fctab->centresR_y,0), 512.5, 0.03);
    sph_ird_star_center_delete(recipe);
    recipe = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_free(fname);
    fname = NULL;
    cpl_free(fname1);
    fname1 = NULL;
    return;
}

static
void cutest_sph_ird_star_center_run_x_pattern(void) {
    cpl_image* im = NULL;
    cpl_frame* aframe = NULL;
    sph_ird_star_center* recipe = NULL;
    cpl_frameset* inframes = NULL;
    cpl_parameterlist* inparams = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_fctable* fctab = NULL;
    char* fname = NULL;
    char* fname1 = NULL;
    double bgvalue = 10000.0;
    double peakvalue = 35000.0;
    double vpeak = 0.0;
    double distx = 10.0;
    double disty = 10.0;

    cpl_test_error(CPL_ERROR_NONE);

    // Preparing basic image
    /* Left side: */
    im = sph_test_image_tools_create_flat_image_double(2048, 1024, 0.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 - distx,
            cpl_image_get_size_y(im) / 2.0 + 0.5 - disty, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 - distx,
            cpl_image_get_size_y(im) / 2.0 + 0.5 + disty, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 + distx,
            cpl_image_get_size_y(im) / 2.0 + 0.5 + disty, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 + distx,
            cpl_image_get_size_y(im) / 2.0 + 0.5 - disty, 3.0, 1000000.0);

    /* Right side: */
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 + 1024.0 - distx,
            cpl_image_get_size_y(im) / 2.0 + 0.5 - disty, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 + 1024.0 - distx,
            cpl_image_get_size_y(im) / 2.0 + 0.5 + disty, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 + 1024.0 + distx,
            cpl_image_get_size_y(im) / 2.0 + 0.5 + disty, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 + 1024.0 + distx,
            cpl_image_get_size_y(im) / 2.0 + 0.5 - disty, 3.0, 1000000.0);
    vpeak = cpl_image_get_max(im);
    cpl_image_multiply_scalar(im, peakvalue / vpeak);
    cpl_image_add_scalar(im, bgvalue);

    // Adding noise and keywords
    cpl_test_nonnull( im );
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    aframe = sph_test_frame_image(im, CPL_BPP_32_SIGNED,
            SPH_IRD_TAG_WAFFLE_IMAGE_RAW, CPL_FRAME_GROUP_RAW);
    sph_test_frame_add_key_double(aframe, 0, "MJD-OBS",
            2000.0);
    sph_test_frame_add_key_string(aframe, 0, "ESO DET FRAM UTC",
            "2013-05-18T13:43:21");
    sph_test_frame_add_key_double(aframe, 0, "EXPTIME", 2.0);

    cpl_image_delete(im);
    im = NULL;
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, aframe);

    // Creating dummy dark and flat
    dark = sph_master_frame_new(2048, 1024);
    flat = sph_master_frame_new(2048, 1024);
    cpl_image_add_scalar(flat->image, 1.0);
    cpl_image_add_scalar(dark->image, 10.0);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_DARK",
            SPH_IRD_TAG_DARK_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(dark, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_FLAT",
            SPH_IRD_TAG_FLAT_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(flat, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);

    // Creating recipe
    inparams = sph_ird_star_center_create_paramlist();
    recipe = sph_ird_star_center_new(inframes, inparams);
    cpl_test_nonnull( recipe );
    recipe->tot_fctab = NULL;
    cpl_test_error(CPL_ERROR_NONE);

    // Running recipe
    cpl_test_eq_error(sph_ird_star_center_run(recipe), CPL_ERROR_NONE);
    cpl_error_reset();

    // Checking results
    fname1 = sph_filemanager_new_filename_from_base(
            cpl_frame_get_filename(cpl_frameset_get_position(inframes, 0)),
            "fctable");

    fctab = sph_fctable_load_fits_IRDIS("star_center.fits");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( fctab );
    cpl_test_eq(sph_fctable_get_size(fctab), 1);

    cpl_test_abs(cpl_vector_get(fctab->centresL_x,0), 512.0, 0.03);
    cpl_test_abs(cpl_vector_get(fctab->centresL_y,0), 512.0, 0.03);
    cpl_test_abs(cpl_vector_get(fctab->centresR_x,0), 512.0, 0.03);
    cpl_test_abs(cpl_vector_get(fctab->centresR_y,0), 512.0, 0.03);
    sph_ird_star_center_delete(recipe);
    recipe = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_free(fname);
    fname = NULL;
    cpl_free(fname1);
    fname1 = NULL;
    return;
}

static
void cutest_sph_ird_star_center_run_plus_pattern(void) {
    cpl_image* im = NULL;
    cpl_frame* aframe = NULL;
    sph_ird_star_center* recipe = NULL;
    cpl_frameset* inframes = NULL;
    cpl_parameterlist* inparams = NULL;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_fctable* fctab = NULL;
    char* fname = NULL;
    char* fname1 = NULL;
    double bgvalue = 10000.0;
    double peakvalue = 35000.0;
    double vpeak = 0.0;
    double distlarge = 10.0;
    double distsmall = 0.0;

    cpl_test_error(CPL_ERROR_NONE);

    // Preparing basic image
    /* Left side: */
    im = sph_test_image_tools_create_flat_image_double(2048, 1024, 0.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 - distlarge,
            cpl_image_get_size_y(im) / 2.0 + 0.5 - distsmall, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 - distsmall,
            cpl_image_get_size_y(im) / 2.0 + 0.5 + distlarge, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 + distlarge,
            cpl_image_get_size_y(im) / 2.0 + 0.5 + distsmall, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 0.5 + distsmall,
            cpl_image_get_size_y(im) / 2.0 + 0.5 - distlarge, 3.0, 1000000.0);

    /* Right side: */
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5 - distlarge,
            cpl_image_get_size_y(im) / 2.0 + 0.5 - distsmall, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5 - distsmall,
            cpl_image_get_size_y(im) / 2.0 + 0.5 + distlarge, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5 + distlarge,
            cpl_image_get_size_y(im) / 2.0 + 0.5 + distsmall, 3.0, 1000000.0);
    sph_test_image_tools_add_gauss(im, cpl_image_get_size_y(im) / 2.0 + 1024.0 + 0.5 + distsmall,
            cpl_image_get_size_y(im) / 2.0 + 0.5 - distlarge, 3.0, 1000000.0);
    vpeak = cpl_image_get_max(im);
    cpl_image_multiply_scalar(im, peakvalue / vpeak);
    cpl_image_add_scalar(im, bgvalue);

    // Adding noise and keywords
    cpl_test_nonnull( im );
    sph_test_image_tools_apply_poisson_noise(im, NULL);
    aframe = sph_test_frame_image(im, CPL_BPP_32_SIGNED,
            SPH_IRD_TAG_WAFFLE_IMAGE_RAW, CPL_FRAME_GROUP_RAW);
    sph_test_frame_add_key_double(aframe, 0, "MJD-OBS",
            2000.0);
    sph_test_frame_add_key_string(aframe, 0, "ESO DET FRAM UTC",
            "2013-05-18T13:43:21");
    sph_test_frame_add_key_double(aframe, 0, "EXPTIME", 2.0);

    cpl_image_delete(im);
    im = NULL;
    inframes = cpl_frameset_new();
    cpl_frameset_insert(inframes, aframe);

    // Creating dummy dark and flat
    dark = sph_master_frame_new(2048, 1024);
    flat = sph_master_frame_new(2048, 1024);
    cpl_image_add_scalar(flat->image, 1.0);
    cpl_image_add_scalar(dark->image, 10.0);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_DARK",
            SPH_IRD_TAG_DARK_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(dark, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);
    aframe = sph_filemanager_create_temp_frame("TEST_STAR_CENTER_FLAT",
            SPH_IRD_TAG_FLAT_CALIB, CPL_FRAME_GROUP_CALIB);
    sph_master_frame_save(flat, cpl_frame_get_filename(aframe), NULL);
    cpl_frameset_insert(inframes, aframe);

    // Creating recipe
    inparams = sph_ird_star_center_create_paramlist();
    recipe = sph_ird_star_center_new(inframes, inparams);
    cpl_test_nonnull( recipe );
    recipe->tot_fctab = NULL;
    cpl_test_error(CPL_ERROR_NONE);

    // Running recipe
    cpl_test_eq_error(sph_ird_star_center_run(recipe), CPL_ERROR_NONE);
    cpl_error_reset();

    // Checking results
    fname1 = sph_filemanager_new_filename_from_base(
            cpl_frame_get_filename(cpl_frameset_get_position(inframes, 0)),
            "fctable");

    fctab = sph_fctable_load_fits_IRDIS("star_center.fits");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_nonnull( fctab );
    cpl_test_eq(sph_fctable_get_size(fctab), 1);

    cpl_test_abs(cpl_vector_get(fctab->centresL_x,0), 512.0, 0.03);
    cpl_test_abs(cpl_vector_get(fctab->centresL_y,0), 512.0, 0.03);
    cpl_test_abs(cpl_vector_get(fctab->centresR_x,0), 512.0, 0.03);
    cpl_test_abs(cpl_vector_get(fctab->centresR_y,0), 512.0, 0.03);
    sph_ird_star_center_delete(recipe);
    recipe = NULL;
    cpl_parameterlist_delete(inparams);
    inparams = NULL;
    cpl_frameset_delete(inframes);
    inframes = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_free(fname);
    fname = NULL;
    cpl_free(fname1);
    fname1 = NULL;
    return;
}
