/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <math.h>
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_ngc_ir_simulator.h"
#include "sph_test_pupilimage_creator.h"
#include "sph_test_image_tools.h"
#include "sph_ird_wave_calib_run.c"

static sph_ird_wave_calib*
cutest_sph_ird_wave_calib_fixture_standard__(void);

static
int cutest_init_testsuite(void);

static
int cutest_clean_testsuite(void);

static
void cutest_sph_ird_wave_calib_fit_spectrum(void);

static
void cutest_sph_ird_wave_calib_transform_to_coords(void);

static
void cutest_sph_ird_wave_calib_read_wavs_from_inskeys(void);

static
void cutest_sph_ird_wave_calib_find_linepeaks(void);

static
void
cutest_sph_ird_wave_calib_qc_wavelengths(void);

static
void cutest_sph_ird_wave_calib_fit_spectrum_w_linepeaks(void);
/*----------------------------------------------------------------------------*/
/**
 * @defgroup A CUnit Test Suite -- representing a collection of testcases
 * @par Synopsis:
 * @code
 * @endcode
 * @par Desciption:
 *
 * This module provides a collection of tests for one specific, distinct
 * module or set-up. The testing code is implemented using the CUnit
 * framework.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/*-                            INTERNAL HELPER FUNCTIONS                          */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit test MAIN function
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("IRDIS wave calibration testsuite",
            cutest_init_testsuite, cutest_clean_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


    if (NULL
            == sph_test_do(pSuite, "Testing fit spectrum",
                    cutest_sph_ird_wave_calib_fit_spectrum)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing find linepeaks",
                    cutest_sph_ird_wave_calib_find_linepeaks)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing fit spectrum w linepeaks",
                    cutest_sph_ird_wave_calib_fit_spectrum_w_linepeaks)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing qc wavelengths w linepeaks",
                    cutest_sph_ird_wave_calib_qc_wavelengths)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing read ins keys",
                    cutest_sph_ird_wave_calib_read_wavs_from_inskeys)) {
        return sph_test_get_error();
    }
    if (NULL
            == sph_test_do(pSuite, "Testing transform to coords",
                    cutest_sph_ird_wave_calib_transform_to_coords)) {
        return sph_test_get_error();
    }

    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to initiailise the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_init_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     * -------------------------------------------------------------------*/
    sph_test_nop_code();
    return 0;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Function to clean the unit test suite
 */
/*----------------------------------------------------------------------------*/
static
int cutest_clean_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}
/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_wave_calib_read_wavs_from_inskeys(void) {
    cpl_propertylist* plistin = NULL;
    sph_ird_wave_calib* recipe = NULL;
    cpl_frame* aframe = NULL;

    // Setup ....
    recipe = cpl_calloc(1, sizeof(sph_ird_wave_calib));
    cpl_test_nonnull( recipe );
    plistin = cpl_propertylist_new();
    cpl_test_nonnull( plistin );

    cpl_propertylist_update_string(plistin, "ESO INS4 SENS1 ID", "PCMON");
    cpl_propertylist_update_string(plistin, "ESO INS4 SENS1 NAME",
            "DUST PARTICLE COUNTER");

    cpl_propertylist_update_string(plistin, "ESO INS4 SENS10 ID", "CBL5I");
    cpl_propertylist_update_string(plistin, "ESO INS4 SENS10 NAME",
            "Laser 2004 - current");

    cpl_propertylist_update_string(plistin, "ESO INS4 SENS11 ID", "CBL6I");
    cpl_propertylist_update_string(plistin, "ESO INS4 SENS11 NAME",
            "Laser 1080 - current");
    cpl_propertylist_update_string(plistin, "ESO INS4 SENS3 ID", "CBL4T");
    cpl_propertylist_update_string(plistin, "ESO INS4 SENS3 NAME",
            "Laser 100 - current");

    aframe = sph_filemanager_create_temp_frame("plist", "NOTAG",
            CPL_FRAME_GROUP_RAW);
    cpl_test_eq(
            cpl_propertylist_save(plistin, cpl_frame_get_filename(aframe), CPL_IO_CREATE),
            CPL_ERROR_NONE);

    recipe->number_lines = 2;
    recipe->rawframes = cpl_frameset_new();
    cpl_frameset_insert(recipe->rawframes, aframe);
    cpl_test_eq( sph_ird_wave_calib_read_wavs_from_inskeys__(recipe),
            CPL_ERROR_NONE);

    cpl_test_abs( recipe->wavelength_line1, 2004.0, 0.0001);
    cpl_test_abs( recipe->wavelength_line2, 1080.0, 0.0001);

    cpl_vector_delete(recipe->calib_wavelengths);
    cpl_frameset_delete(recipe->rawframes);
    cpl_free(recipe);
    cpl_propertylist_delete(plistin);
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_wave_calib_transform_to_coords(void) {
    int ii = 0;
    double y0 = 30.0;
    double lambda = 0.0;
    double dy = 0.0;
    double c2 = -58.1898233;
    double c3 = 177.6837156;
    double c4 = 113.3309017;
    double l1 = 980.0;
    cpl_vector* calib_lines = NULL;
    cpl_vector* xpoints = NULL;
    cpl_vector* calib_lines_out = NULL;
    cpl_vector* xpoints_out = NULL;
    sph_ird_wave_calib* recipe = NULL;

    recipe = cutest_sph_ird_wave_calib_fixture_standard__();
    cpl_test_nonnull( recipe );

    xpoints = cpl_vector_new(6);
    cpl_vector_set(xpoints, 0, y0);
    cpl_vector_set(xpoints, 1, 40);
    cpl_vector_set(xpoints, 2, 50);
    cpl_vector_set(xpoints, 3, 70);
    cpl_vector_set(xpoints, 4, 80);
    cpl_vector_set(xpoints, 5, 90);

    xpoints_out = cpl_vector_duplicate(xpoints);
    calib_lines_out = cpl_vector_duplicate(recipe->calib_wavelengths);

    cpl_test_eq(
            sph_ird_wave_calib_transform_to_fitcoords__( recipe, xpoints_out, calib_lines_out, l1, y0),
            CPL_ERROR_NONE);

    for (ii = 0; ii < cpl_vector_get_size(calib_lines_out); ++ii) {
        dy = cpl_vector_get(xpoints_out, ii);
        cpl_test_abs( dy, (cpl_vector_get(xpoints,ii) - y0) / 100.0,
                0.01);

        lambda = cpl_vector_get(recipe->calib_wavelengths, ii) - l1
                - c2 * dy * dy - c3 * dy * dy * dy - c4 * dy * dy * dy * dy;
        cpl_test_abs( cpl_vector_get( calib_lines_out, ii), lambda,
                0.01);
    }
    cpl_vector_delete(calib_lines);
    cpl_vector_delete(xpoints);
    cpl_vector_delete(calib_lines_out);
    cpl_vector_delete(xpoints_out);
    cpl_vector_delete(recipe->calib_wavelengths);
    sph_ird_wave_calib_delete(recipe);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_wave_calib_fit_spectrum(void) {
    sph_spectrum* spec = NULL;
    int ii = 0;
    double y0 = 70.0;
    double lambda0 = 1545.07;
    double lambda = 0.0;
    double dy = 0.0;
    double c1 = -980.0;
    cpl_vector* peaks = NULL;
    cpl_vector* wavs = NULL;
    cpl_vector* coeffs = NULL;
    sph_ird_wave_calib* recipe = NULL;
    double chisq = 0.0;
    int np = 200;

    recipe = cutest_sph_ird_wave_calib_fixture_standard__();
    cpl_test_nonnull( recipe );

    peaks = cpl_vector_duplicate(recipe->calib_wavelengths);

    for (ii = 0; ii < np; ++ii) {
        dy = ((double) ii - y0) / 100.0;
        lambda = lambda0 + c1 * dy + recipe->c2 * dy * dy
                + recipe->c3 * dy * dy * dy + recipe->c4 * dy * dy * dy * dy;

        if (lambda >= recipe->wavelength_line1)
            cpl_vector_set(peaks, 0, ii);

        if (lambda >= recipe->wavelength_line2)
            cpl_vector_set(peaks, 1, ii);

        if (lambda >= recipe->wavelength_line3)
            cpl_vector_set(peaks, 2, ii);

        if (lambda >= recipe->wavelength_line4)
            cpl_vector_set(peaks, 3, ii);

        if (lambda >= recipe->wavelength_line5)
            cpl_vector_set(peaks, 4, ii);

        if (lambda >= recipe->wavelength_line6)
            cpl_vector_set(peaks, 5, ii);
    }

    cpl_vector_sort(peaks, CPL_SORT_DESCENDING);
    recipe->fitorder = 1;

    coeffs = cpl_vector_new(recipe->fitorder + 1);

    wavs = sph_ird_wave_calib_spectrum_fit_wavs__(recipe, peaks, np, NULL, NULL,
            coeffs, &chisq);

    cpl_test_nonnull( wavs );
    cpl_test_abs( cpl_vector_get(coeffs,0), y0, 1.1);
    cpl_test_abs( cpl_vector_get(coeffs,1), c1, 10.7);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Fitting result: y0 = %f, c1 = %f, chisq = %f", cpl_vector_get(coeffs,0), cpl_vector_get(coeffs,1), chisq);

    sph_spectrum_delete(spec);
    cpl_vector_delete(peaks);
    cpl_vector_delete(wavs);
    cpl_vector_delete(coeffs);
    cpl_vector_delete(recipe->calib_wavelengths);

    sph_ird_wave_calib_delete(recipe);
    return;
}
static
void cutest_sph_ird_wave_calib_qc_wavelengths(void) {
    sph_ird_wave_calib* rec = NULL;
    int xx = 0;
    int yy = 0;
    sph_pixel_descriptor* descr = NULL;
    cpl_vector* peaks = NULL;
    cpl_vector* lam = NULL;
    cpl_vector* rms = NULL;
    sph_spectral_region* reg = NULL;
    double dy = 0.0;
    double lambda = 0.0;
    double c1 = -980.0;
    double lambda0 = 1545.07;
    double y0 = 70.0;
    double val = 0.0;

    rec = cutest_sph_ird_wave_calib_fixture_standard__();

    rec->pdt = sph_pixel_description_table_new(256, 256, 0.0, 0.0);
    rec->rawmframe = sph_master_frame_new(256, 256);
    reg = sph_spectral_region_new();
    reg->minx = 20;
    reg->maxx = 40;
    reg->miny = 10;
    reg->maxy = 210;

    for (xx = reg->minx; xx < reg->maxx; ++xx) {
        for (yy = reg->miny; yy < reg->maxy; ++yy) {
            descr = sph_pixel_description_table_get_descriptor(rec->pdt, xx,
                    yy);
            dy = ((double) yy - y0 - reg->miny) / 100.0;
            lambda = lambda0 + c1 * dy + rec->c2 * dy * dy
                    + rec->c3 * dy * dy * dy + rec->c4 * dy * dy * dy * dy;
            if (descr) {
                descr->wavelength = lambda;
                descr->dlambda = 0.01
                        * (c1 + 2.0 * rec->c2 * dy + 3.0 * rec->c3 * dy * dy
                                + 4.0 * rec->c4 * dy * dy * dy);
                descr->ddlambda = 0.0001
                        * (2.0 * rec->c2 + 6.0 * rec->c3 * dy
                                + 12.0 * rec->c4 * dy * dy);
                descr->specid = 1;

                val = exp(-pow(lambda - rec->wavelength_line1, 2.0) / 1600.0);
                val += exp(-pow(lambda - rec->wavelength_line2, 2.0) / 1600.0);
                val += exp(-pow(lambda - rec->wavelength_line3, 2.0) / 1600.0);
                val += exp(-pow(lambda - rec->wavelength_line4, 2.0) / 1600.0);
                val += exp(-pow(lambda - rec->wavelength_line5, 2.0) / 1600.0);
                val += exp(-pow(lambda - rec->wavelength_line6, 2.0) / 1600.0);

                cpl_image_set(rec->rawmframe->image, xx + 1, yy + 1,
                        10.0 * val);
            }
        }
    }

    sph_pixel_description_table_save(rec->pdt, "pdt_wave_calib_test.fits",
            NULL);
    sph_master_frame_save(rec->rawmframe, "rawmframe_test.fits", NULL);

    lam = cpl_vector_new(6);
    rms = cpl_vector_new(6);
    peaks = sph_ird_wave_calib_find_linepeaks__(rec->rawmframe, reg, 5.0);
    cpl_test_eq(
            sph_ird_wave_calib_qc_wavelengths__( rec,reg,peaks,lam,rms),
            CPL_ERROR_NONE);

    for (yy = 0; yy < cpl_vector_get_size(rec->calib_wavelengths); ++yy) {
        cpl_test_abs( cpl_vector_get( lam, yy),
                cpl_vector_get(rec->calib_wavelengths,yy), 2.0);
        cpl_test_abs( cpl_vector_get( rms, yy), 0.0, 2.01);
    }
    cpl_vector_delete(peaks);
    cpl_vector_delete(rms);
    cpl_vector_delete(lam);
    sph_spectral_region_delete(reg);
    sph_pixel_description_table_delete(rec->pdt);
    sph_master_frame_delete(rec->rawmframe);
    cpl_vector_delete(rec->calib_wavelengths);
    cpl_free(rec);
    SPH_RAISE_CPL_RESET;

}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_wave_calib_find_linepeaks(void) {
    sph_spectrum* spec = NULL;
    int ii = 0;
    double y0 = 70.0;
    double lambda0 = 1545.07;
    double lambda = 0.0;
    double dy = 0.0;
    double c1 = -980.0;
    double val = 0.0;
    cpl_vector* peaks = NULL;
    cpl_vector* peaks_measured = NULL;
    cpl_vector* wavs = NULL;
    cpl_vector* coeffs = NULL;
    sph_ird_wave_calib* recipe = NULL;
    int xx = 0;
    sph_spectral_region* reg = NULL;
    int np = 200;
    recipe = cutest_sph_ird_wave_calib_fixture_standard__();
    cpl_test_nonnull( recipe );

    peaks = cpl_vector_duplicate(recipe->calib_wavelengths);

    spec = sph_spectrum_new(200, 950.0, 2010.0);

    recipe->rawmframe = sph_master_frame_new(256, 256);

    reg = sph_spectral_region_new();
    reg->lensid = 1;
    reg->minx = 10;
    reg->maxx = 30;
    reg->miny = 20;
    reg->maxy = 220;

    for (ii = 0; ii < np; ++ii) {
        dy = ((double) ii - y0) / 100.0;
        lambda = lambda0 + c1 * dy + recipe->c2 * dy * dy
                + recipe->c3 * dy * dy * dy + recipe->c4 * dy * dy * dy * dy;

        if (lambda >= recipe->wavelength_line1)
            cpl_vector_set(peaks, 0, ii);

        if (lambda >= recipe->wavelength_line2)
            cpl_vector_set(peaks, 1, ii);

        if (lambda >= recipe->wavelength_line3)
            cpl_vector_set(peaks, 2, ii);

        if (lambda >= recipe->wavelength_line4)
            cpl_vector_set(peaks, 3, ii);

        if (lambda >= recipe->wavelength_line5)
            cpl_vector_set(peaks, 4, ii);

        if (lambda >= recipe->wavelength_line6)
            cpl_vector_set(peaks, 5, ii);

        val = exp(-pow(lambda - recipe->wavelength_line1, 2.0) / 1600.0);
        val += exp(-pow(lambda - recipe->wavelength_line2, 2.0) / 1600.0);
        val += exp(-pow(lambda - recipe->wavelength_line3, 2.0) / 1600.0);
        val += exp(-pow(lambda - recipe->wavelength_line4, 2.0) / 1600.0);
        val += exp(-pow(lambda - recipe->wavelength_line5, 2.0) / 1600.0);
        val += exp(-pow(lambda - recipe->wavelength_line6, 2.0) / 1600.0);
        for (xx = reg->minx; xx < reg->maxx; ++xx) {
            cpl_image_set(recipe->rawmframe->image, xx + 1, ii + reg->miny + 1,
                    10.0 * val);
        }
    }

    sph_master_frame_save(recipe->rawmframe, "rawmframe.fits", NULL);

    cpl_vector_sort(peaks, CPL_SORT_DESCENDING);

    peaks_measured = sph_ird_wave_calib_find_linepeaks__(recipe->rawmframe, reg,
            5.0);

    cpl_test_nonnull( peaks_measured );

    for (xx = 0; xx < cpl_vector_get_size(peaks); ++xx) {
        cpl_test_abs( cpl_vector_get(peaks_measured,xx),
                cpl_vector_get(peaks,xx), 1.0);
    }

    sph_master_frame_delete(recipe->rawmframe);
    sph_spectral_region_delete(reg);
    sph_spectrum_delete(spec);
    cpl_vector_delete(peaks);
    cpl_vector_delete(peaks_measured);
    cpl_vector_delete(wavs);
    cpl_vector_delete(coeffs);
    cpl_vector_delete(recipe->calib_wavelengths);

    sph_ird_wave_calib_delete(recipe);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief        A test function of the testsuite
 */
/*----------------------------------------------------------------------------*/
static
void cutest_sph_ird_wave_calib_fit_spectrum_w_linepeaks(void) {
    sph_spectrum* spec = NULL;
    int ii = 0;
    double y0 = 70.0;
    double lambda0 = 1545.07;
    double lambda = 0.0;
    double dy = 0.0;
    double c1 = -980.0;
    double val = 0.0;
    cpl_vector* peaks = NULL;
    cpl_vector* peaks_measured = NULL;
    cpl_vector* wavs = NULL;
    cpl_vector* coeffs = NULL;
    sph_ird_wave_calib* recipe = NULL;
    int xx = 0;
    sph_spectral_region* reg = NULL;
    int np = 200;
    double chisq = 0.0;

    recipe = cutest_sph_ird_wave_calib_fixture_standard__();
    cpl_test_nonnull( recipe );

    peaks = cpl_vector_duplicate(recipe->calib_wavelengths);

    spec = sph_spectrum_new(200, 950.0, 2010.0);

    recipe->rawmframe = sph_master_frame_new(256, 256);

    reg = sph_spectral_region_new();
    reg->lensid = 1;
    reg->minx = 10;
    reg->maxx = 30;
    reg->miny = 20;
    reg->maxy = 220;

    for (ii = 0; ii < np; ++ii) {
        dy = ((double) ii - y0) / 100.0;
        lambda = lambda0 + c1 * dy + recipe->c2 * dy * dy
                + recipe->c3 * dy * dy * dy + recipe->c4 * dy * dy * dy * dy;

        if (lambda >= recipe->wavelength_line1)
            cpl_vector_set(peaks, 0, ii);

        if (lambda >= recipe->wavelength_line2)
            cpl_vector_set(peaks, 1, ii);

        if (lambda >= recipe->wavelength_line3)
            cpl_vector_set(peaks, 2, ii);

        if (lambda >= recipe->wavelength_line4)
            cpl_vector_set(peaks, 3, ii);

        if (lambda >= recipe->wavelength_line5)
            cpl_vector_set(peaks, 4, ii);

        if (lambda >= recipe->wavelength_line6)
            cpl_vector_set(peaks, 5, ii);

        val = exp(-pow(lambda - recipe->wavelength_line1, 2.0) / 600.0);
        val += exp(-pow(lambda - recipe->wavelength_line2, 2.0) / 600.0);
        val += exp(-pow(lambda - recipe->wavelength_line3, 2.0) / 600.0);
        val += exp(-pow(lambda - recipe->wavelength_line4, 2.0) / 600.0);
        val += exp(-pow(lambda - recipe->wavelength_line5, 2.0) / 600.0);
        val += exp(-pow(lambda - recipe->wavelength_line6, 2.0) / 600.0);
        for (xx = reg->minx; xx < reg->maxx; ++xx) {
            cpl_image_set(recipe->rawmframe->image, xx + 1, ii + reg->miny + 1,
                    10.0 * val);
        }
    }

    sph_master_frame_save(recipe->rawmframe, "rawmframe.fits", NULL);

    cpl_vector_sort(peaks, CPL_SORT_DESCENDING);

    peaks_measured = sph_ird_wave_calib_find_linepeaks__(recipe->rawmframe, reg,
            5.0);

    cpl_test_nonnull( peaks_measured );

    cpl_vector_sort(peaks_measured, CPL_SORT_DESCENDING);
    recipe->fitorder = 1;

    coeffs = cpl_vector_new(recipe->fitorder + 1);

    wavs = sph_ird_wave_calib_spectrum_fit_wavs__(recipe, peaks_measured, np,
            NULL, NULL, coeffs, &chisq);

    cpl_test_nonnull( wavs );
    cpl_test_abs( cpl_vector_get(coeffs,0), y0, 0.1);
    cpl_test_abs( cpl_vector_get(coeffs,1), c1, 0.5);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Fitting result: y0 = %f, c1 = %f, chisq = %f", cpl_vector_get(coeffs,0), cpl_vector_get(coeffs,1), chisq);

    sph_master_frame_delete(recipe->rawmframe);
    sph_spectral_region_delete(reg);
    sph_spectrum_delete(spec);
    cpl_vector_delete(peaks);
    cpl_vector_delete(peaks_measured);
    cpl_vector_delete(wavs);
    cpl_vector_delete(coeffs);
    cpl_vector_delete(recipe->calib_wavelengths);

    sph_ird_wave_calib_delete(recipe);
    return;
}

/*----------------------------------------------------------------------------*/
/**
 * Fixtures
 *
 */
/*----------------------------------------------------------------------------*/
static sph_ird_wave_calib*
cutest_sph_ird_wave_calib_fixture_standard__(void) {
    sph_ird_wave_calib* self = NULL;
    self = cpl_calloc(1, sizeof(sph_ird_wave_calib));

    self->bg_raw = NULL;
    self->c2 = -58.1898233;
    self->c3 = 177.6837156;
    self->c4 = 113.3309017;
    self->wavelength_line1 = 987.72;
    self->wavelength_line2 = 1123.71;
    self->wavelength_line3 = 1309.0;
    self->wavelength_line4 = 1545.07;
    self->wavelength_line5 = 1730.23;
    self->wavelength_line6 = 2015.33;

    self->calib_wavelengths = cpl_vector_new(6);
    cpl_vector_set(self->calib_wavelengths, 0, self->wavelength_line1);
    cpl_vector_set(self->calib_wavelengths, 1, self->wavelength_line2);
    cpl_vector_set(self->calib_wavelengths, 2, self->wavelength_line3);
    cpl_vector_set(self->calib_wavelengths, 3, self->wavelength_line4);
    cpl_vector_set(self->calib_wavelengths, 4, self->wavelength_line5);
    cpl_vector_set(self->calib_wavelengths, 5, self->wavelength_line6);

    return self;
}

/**@}*/
