/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <cpl.h>

#include "cutest_zpl_polarization_flat.h"
#include "sph_zpl_polarization_flat.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"

#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_polarization_flat_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     *--------------------------------------------------------------------*/
    sph_test_nop_code();
    return CPL_ERROR_NONE;
}

static int cutest_clean_polarization_flat_testsuite(void) {
    sph_error_dump(SPH_ERROR_ERROR);
    return sph_end_test();
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_polarization_flat only with rawframes
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_polarization_flat_run_1(void) {
    sph_zpl_polarization_flat* zpl_polarization_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_frame* zexpcube1 = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 4;
    //int                         nx                  = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    //int                         ny                  = 100; // -//-

    double dV = 10.0;
    //double              dS						= 0.0;

    double zero_odd = 100.0;
    double zero_even = 200.0;
    double pi_odd = 300.0;
    double pi_even = 400.0;
    cpl_propertylist* pl = NULL;

    inparams = sph_zpl_polarization_flat_create_paramlist();
    inframes = cpl_frameset_new();
    sph_test_simulator_create_zplpreproc_cube_set(&zexpcube, zero_odd,
            zero_even, pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_PFF_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube);

    sph_test_simulator_create_zplpreproc_cube_set(&zexpcube1, zero_odd,
            zero_even, pi_odd, pi_even, dV, nframes);
    cpl_frame_set_tag(zexpcube1, SPH_ZPL_TAG_PFF_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube1, CPL_FRAME_GROUP_RAW);
    rerr = cpl_frameset_insert(inframes, zexpcube1);

    zpl_polarization_flat = sph_zpl_polarization_flat_new(inframes, inparams);
    cpl_test_nonnull( zpl_polarization_flat );

    rerr = sph_zpl_polarization_flat_run(zpl_polarization_flat);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_polarization_flat->polarization_flat_outfilename_cam1, 0);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            1060.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME),
            0.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    pl = cpl_propertylist_load(
            zpl_polarization_flat->polarization_flat_outfilename_cam2, 0);

    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            1060.0, 1e-10);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME),
            0.0, 1e-10);
    cpl_propertylist_delete(pl);
    pl = NULL;

    rerr = sph_zpl_polarization_flat_delete(zpl_polarization_flat);
    cpl_test_error(CPL_ERROR_NONE);

    //Destroy here input parameters and inframes explicitly
    cpl_frameset_delete(inframes);
    cpl_parameterlist_delete(inparams);

    return;
}

#if 0
/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_polarization_flat with rawframes &
 *       calibrations (iff as a quad image)
 *
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_polarization_flat_run_2(void) {
    sph_zpl_polarization_flat* zpl_polarization_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 4;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    double dV = 10.0;
    //double              dS						= 0.0;

    double zero_odd = 100.0;
    double zero_even = 200.0;
    double pi_odd = 300.0;
    double pi_even = 400.0;
    cpl_propertylist* pl = NULL;
    cpl_frame* master_bias_frame = NULL;
    cpl_frame* master_dark_frame = NULL;
    cpl_frame* intensity_flat_frame = NULL;

    inframes = cpl_frameset_new();

    //simulate rawframes
    sph_test_simulator_create_zplpreproc_cube_set( &zexpcube, zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_PFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    //simualate master bias
    zero_odd = 10; zero_even = 11; pi_odd = 12; pi_even = 14;
    master_bias_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_bias_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( master_bias_frame, SPH_ZPL_TAG_BIAS_CALIB );
    cpl_frame_set_group( master_bias_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, master_bias_frame );

    //simualate master dark
    zero_odd = 1; zero_even = 2; pi_odd = 3; pi_even = 5;
    master_dark_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_dark_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( master_dark_frame, SPH_ZPL_TAG_DARK_CALIB );
    cpl_frame_set_group( master_dark_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, master_dark_frame );

    //simualate intensity flat
    zero_odd = 2; zero_even = 2; pi_odd = 2; pi_even = 2;
    intensity_flat_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_intensity_flat_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( intensity_flat_frame, SPH_ZPL_TAG_IFF_CALIB );
    cpl_frame_set_group( intensity_flat_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, intensity_flat_frame );

    inparams = sph_zpl_polarization_flat_create_paramlist();
    zpl_polarization_flat = sph_zpl_polarization_flat_new( inframes, inparams);
    cpl_test_nonnull( zpl_polarization_flat );

    rerr = sph_zpl_polarization_flat_run( zpl_polarization_flat );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_polarization_flat->polarization_flat_outfilename, 0);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), 501.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), -1.0, 1e-10);

    //

    rerr = sph_zpl_polarization_flat_delete( zpl_polarization_flat );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}

/*----------------------------------------------------------------------------*/
/**
 @brief Unit test for the sph_zpl_polarization_flat with rawframes &
 *      calibrations (iff as a master frame)
 */
/*----------------------------------------------------------------------------*/
void cutest_zpl_polarization_flat_run_3(void) {
    sph_zpl_polarization_flat* zpl_polarization_flat = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nframes = 4;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100;// -//-

    double dV = 10.0;
    //double              dS						= 0.0;

    double zero_odd = 100.0;
    double zero_even = 200.0;
    double pi_odd = 300.0;
    double pi_even = 400.0;
    cpl_propertylist* pl = NULL;
    cpl_frame* master_bias_frame = NULL;
    cpl_frame* master_dark_frame = NULL;
    cpl_frame* intensity_flat_frame = NULL;

    inframes = cpl_frameset_new();

    //simulate rawframes
    sph_test_simulator_create_zplpreproc_cube_set( &zexpcube, zero_odd, zero_even, pi_odd, pi_even, dV, nframes );
    cpl_frame_set_tag( zexpcube, SPH_ZPL_TAG_PFF_PREPROC_RAW );
    cpl_frame_set_group( zexpcube, CPL_FRAME_GROUP_RAW );
    rerr = cpl_frameset_insert( inframes, zexpcube );

    //simulate master bias
    zero_odd = 10; zero_even = 11; pi_odd = 12; pi_even = 14;
    master_bias_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_bias_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( master_bias_frame, SPH_ZPL_TAG_BIAS_CALIB );
    cpl_frame_set_group( master_bias_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, master_bias_frame );

    //simulate master dark
    zero_odd = 1; zero_even = 2; pi_odd = 3; pi_even = 5;
    master_dark_frame = sph_test_zpl_simulator_create_quad_image_frame("test_master_dark_calib.fits.tmp", nx, ny, zero_odd, zero_even, pi_odd, pi_even);
    cpl_frame_set_tag( master_dark_frame, SPH_ZPL_TAG_DARK_CALIB );
    cpl_frame_set_group( master_dark_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, master_dark_frame );

    //simulate intensity flat as a masterframe
    intensity_flat_frame = sph_test_zpl_simulator_create_masterframe_frame("test_master_intensity_flat_masterframe_calib.fits.tmp", nx, ny, 2.0);
    cpl_frame_set_tag( intensity_flat_frame, SPH_ZPL_TAG_IFFM_CALIB );
    cpl_frame_set_group( intensity_flat_frame, CPL_FRAME_GROUP_CALIB );
    //printf("master bias tag: %s\n", cpl_frame_get_tag(master_bias_frame));
    rerr = cpl_frameset_insert( inframes, intensity_flat_frame );

    inparams = sph_zpl_polarization_flat_create_paramlist();
    zpl_polarization_flat = sph_zpl_polarization_flat_new( inframes, inparams);
    cpl_test_nonnull( zpl_polarization_flat );

    rerr = sph_zpl_polarization_flat_run( zpl_polarization_flat );
    cpl_test_eq_error(rerr, CPL_ERROR_NONE );

    pl = cpl_propertylist_load( zpl_polarization_flat->polarization_flat_outfilename, 0);

    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME), 501.0, 1e-10);
    cpl_test_abs(cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME), -1.0, 1e-10);

    rerr = sph_zpl_polarization_flat_delete( zpl_polarization_flat );
    cpl_test_error(CPL_ERROR_NONE );

    return;
}
#endif

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_polarization_flat recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    int result = 0;

    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("polarization_flat_test",
            cutest_init_polarization_flat_testsuite,
            cutest_clean_polarization_flat_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }



    if (NULL
            == sph_test_do(pSuite,
                    "sph_polarization_flat_run_1 (pure rawframes)",
                    cutest_zpl_polarization_flat_run_1)) {
        return sph_test_get_error();
    }

    /*
     if ( NULL == sph_test_do(pSuite, "sph_polarization_flat_run_2 (rawframes & all calibrations (iff as a quad imae))",
     cutest_zpl_polarization_flat_run_2 ) )
     {
     return sph_test_get_error();
     }
     if ( NULL == sph_test_do(pSuite, "sph_polarization_flat_run_3 (rawframes & all calibrations (iff as a masterframe))",
     cutest_zpl_polarization_flat_run_3 ) )
     {
     return sph_test_get_error();
     }
     */
    /* Run all tests using the CUnit Basic interface */
    sph_test_nop_int( 0);
    sph_test_nop_char("results.txt");
    result = sph_test_end();
    return result;
}

/**@}*/
