/* $Id: $
 *
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_zpl_science_imaging.h"
#include "sph_zpl_utils.h"
#include "sph_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_keyword_manager.h"
#include "sph_filemanager.h"
#include "sph_keywordnode.h"
#include "sph_test_zpl_simulator.h"
#include "sph_error.h"
#include "sph_test.h"
#include "sph_test_image_tools.h"
#include "sph_fctable.h"

#include <cpl.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>


/*----------------------------------------------------------------------------*/
/**
 * @defgroup techcal_master_test  Unit test of techcal_master recipe and
 *                                  associated functions.
 *
 */
/*----------------------------------------------------------------------------*/

static int cutest_init_science_imaging_testsuite(void) {
    /*--------------------------------------------------------------------
     * -    Prepare CPL and error logging
     *--------------------------------------------------------------------*/
    return sph_test_nop_code();
}

static int cutest_clean_science_imaging_testsuite(void) {
    return sph_end_test();
}

#ifdef SPH_ZPL_SCIENCE_IMAGING
static void cutest_zpl_science_imaging_run_1(void) {
    sph_zpl_science_imaging* zpl_science_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-

    sph_fctable* fctable = NULL;
    cpl_frame* fcframe = NULL;

    cpl_frame* frame_fc = NULL;
    char* fname = NULL;
    char* fname2 = NULL;

    sph_zpl_exposure_imaging* zplexpimg_plane = NULL;

    int nframes = 5;
    double dx[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dy[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double angle[] = { 0, 15, 30, 45, 60 };
    double mean_i[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double mean_p[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double fullmean_i, fullmean_p;

    inparams = sph_zpl_science_imaging_create_paramlist();
    inframes = cpl_frameset_new();

    //simulate fctable
    fcframe = sph_test_simulator_create_fctable(nx, ny, dx, dy, angle, nframes);

    //simulate cube of rotated pre-processed science frames in imaging mode (ZPL EXP IMAGING)
    sph_test_simulator_create_zplpreproc_imaging_science_cube(&zexpcube,
            fcframe, nx, ny);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_SCIENCE_IMAGING_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);

    fullmean_i = 0.0;
    fullmean_p = 0.0;
    for (int ii = 0; ii < nframes; ii++) {
        zplexpimg_plane = sph_zpl_exposure_imaging_load(zexpcube, ii);
        mean_i[ii] = cpl_image_get_mean(zplexpimg_plane->image_odd);
        mean_p[ii] = cpl_image_get_mean(zplexpimg_plane->image_even);
        //printf("mean (%d) = %f", ii, mean[ii]);
        fullmean_i = fullmean_i + mean_i[ii];
        fullmean_p = fullmean_p + mean_p[ii];
        sph_zpl_exposure_imaging_delete(zplexpimg_plane);
    }

    fullmean_i = fullmean_i / nframes;
    fullmean_p = fullmean_p / nframes;
    //printf("fullmean_i %f = ", fullmean_i);

    rerr = cpl_frameset_insert(inframes, zexpcube);

    //load fctable
    fctable = sph_fctable_load_ascii(cpl_frame_get_filename(fcframe));
    fname2 = sph_filemanager_filename_new_ext(cpl_frame_get_filename(zexpcube),
            "txt");
    fname = sph_filemanager_new_filename_from_base(fname2, "fctable");
    sph_fctable_save_ascii(fctable, fname);

    frame_fc = cpl_frame_new();
    cpl_frame_set_filename(frame_fc, fname);
    cpl_frame_set_tag(frame_fc, SPH_ZPL_TAG_FIELD_CENTER_CALIB);
    cpl_frame_set_group(frame_fc, CPL_FRAME_GROUP_CALIB);

    rerr = cpl_frameset_insert(inframes, frame_fc);

    zpl_science_imaging = sph_zpl_science_imaging_new(inframes, inparams);
    cpl_test_nonnull( zpl_science_imaging );

    rerr = sph_zpl_science_imaging_run(zpl_science_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_science_imaging->science_imaging_outfilename_cam1, 0);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            fullmean_i, 1e-3);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME),
            fullmean_p, 1e-3);

    cpl_propertylist_delete(pl);
    pl = NULL;
    sph_zpl_science_imaging_delete(zpl_science_imaging);
    zpl_science_imaging = NULL;

}

static void cutest_zpl_science_imaging_run_2(void) {
    sph_zpl_science_imaging* zpl_science_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-

    cpl_frame* master_bias_imaging_frame = NULL;
    cpl_frame* master_dark_imaging_frame = NULL;
    cpl_frame* intensity_flat_imaging_frame = NULL;
    double iframe_bias = 200.0;
    double pframe_bias = 300.0;

    sph_fctable* fctable = NULL;
    cpl_frame* fcframe = NULL;

    cpl_frame* frame_fc = NULL;
    char* fname = NULL;
    char* fname2 = NULL;

    sph_zpl_exposure_imaging* zplexpimg_plane = NULL;

    int nframes = 5;
    double dx[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dy[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double angle[] = { 0, 15, 30, 45, 60 };
    double mean_i[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double mean_p[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double fullmean_i, fullmean_p;

    inparams = sph_zpl_science_imaging_create_paramlist();
    inframes = cpl_frameset_new();

    //simulate fctable
    fcframe = sph_test_simulator_create_fctable(nx, ny, dx, dy, angle, nframes);

    //simulate cube of rotated pre-processed science frames in imaging mode (ZPL EXP IMAGING)
    sph_test_simulator_create_zplpreproc_imaging_science_cube(&zexpcube,
            fcframe, nx, ny);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_SCIENCE_IMAGING_PREPROC_CAM2);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);

    fullmean_i = 0.0;
    fullmean_p = 0.0;
    for (int ii = 0; ii < nframes; ii++) {
        zplexpimg_plane = sph_zpl_exposure_imaging_load(zexpcube, ii);
        mean_i[ii] = cpl_image_get_mean(zplexpimg_plane->image_odd);
        mean_p[ii] = cpl_image_get_mean(zplexpimg_plane->image_even);
        //printf("mean (%d) = %f", ii, mean[ii]);
        fullmean_i = fullmean_i + mean_i[ii];
        fullmean_p = fullmean_p + mean_p[ii];
        sph_zpl_exposure_imaging_delete(zplexpimg_plane);
    }

    fullmean_i = fullmean_i / nframes;
    fullmean_p = fullmean_p / nframes;
    //printf("fullmean_i = %f\n ", fullmean_i);

    rerr = cpl_frameset_insert(inframes, zexpcube);

    //load fctable
    fctable = sph_fctable_load_ascii(cpl_frame_get_filename(fcframe));
    fname2 = sph_filemanager_filename_new_ext(cpl_frame_get_filename(zexpcube),
            "txt");
    fname = sph_filemanager_new_filename_from_base(fname2, "fctable");
    sph_fctable_save_ascii(fctable, fname);

    frame_fc = cpl_frame_new();
    cpl_frame_set_filename(frame_fc, fname);
    cpl_frame_set_tag(frame_fc, SPH_ZPL_TAG_FIELD_CENTER_CALIB);
    cpl_frame_set_group(frame_fc, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, frame_fc);

    //create master bias imaging (double image frame)
    master_bias_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_si_master_bias.fits.tmp", nx, ny, iframe_bias,
                    pframe_bias);
    cpl_frame_set_tag(master_bias_imaging_frame,
            SPH_ZPL_TAG_BIAS_IMAGING_CALIB_CAM2);
    cpl_frame_set_group(master_bias_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_bias_imaging_frame);

    //create master dark imaging (double image frame)
    master_dark_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_si_master_dark.fits.tmp", nx, ny, iframe_bias + 10.,
                    pframe_bias + 10.);
    cpl_frame_set_tag(master_dark_imaging_frame,
            SPH_ZPL_TAG_DARK_IMAGING_CALIB_CAM2);
    cpl_frame_set_group(master_dark_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_dark_imaging_frame);

    //create intensity flat imaging (double image frame)
    intensity_flat_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_si_intensity_flat.tmp", nx, ny, 2., 3.);
    cpl_frame_set_tag(intensity_flat_imaging_frame,
            SPH_ZPL_TAG_IFF_IMAGING_CALIB_CAM2);
    cpl_frame_set_group(intensity_flat_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, intensity_flat_imaging_frame);

    zpl_science_imaging = sph_zpl_science_imaging_new(inframes, inparams);
    cpl_test_nonnull( zpl_science_imaging );

    rerr = sph_zpl_science_imaging_run(zpl_science_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_science_imaging->science_imaging_outfilename_cam2, 0);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            (fullmean_i-iframe_bias-(iframe_bias+10))/2.0, 1e-3);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME),
            (fullmean_p-pframe_bias-(pframe_bias+10))/3.0, 1e-3);

    cpl_propertylist_delete(pl);
    pl = NULL;
    sph_zpl_science_imaging_delete(zpl_science_imaging);
    zpl_science_imaging = NULL;
}

static void cutest_zpl_science_imaging_run_3(void) {
    sph_zpl_science_imaging* zpl_science_imaging = NULL;
    cpl_parameterlist* inparams = NULL;
    cpl_frameset* inframes = NULL;
    cpl_frame* zexpcube = NULL;
    cpl_propertylist* pl = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    int nx = 100; // it must be only sync changed with sph_test_simulator_create_zplpreproc_cube_plus
    int ny = 100; // -//-

    cpl_frame* master_bias_imaging_frame = NULL;
    cpl_frame* master_dark_imaging_frame = NULL;
    cpl_frame* intensity_flat_imaging_frame = NULL;
    double iframe_bias = 200.0;
    double pframe_bias = 300.0;

    sph_fctable* fctable = NULL;
    cpl_frame* fcframe = NULL;

    cpl_frame* frame_fc = NULL;
    char* fname = NULL;
    char* fname2 = NULL;

    sph_zpl_exposure_imaging* zplexpimg_plane = NULL;

    int nframes = 5;
    double dx[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double dy[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double angle[] = { 0, 15, 30, 45, 60 };
    double mean_i[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };
    double mean_p[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };

    double ovsc[4] = { 100.0, 1.0, 100.0, 2.0 };
    double fullmean_i, fullmean_p;

    inparams = sph_zpl_science_imaging_create_paramlist();
    inframes = cpl_frameset_new();

    //simulate fctable
    fcframe = sph_test_simulator_create_fctable(nx, ny, dx, dy, angle, nframes);

    //simulate cube of rotated pre-processed science frames in imaging mode (ZPL EXP IMAGING)
    zexpcube =
            sph_test_simulator_create_zplpreproc_imaging_science_cube_with_overscan(
                    fcframe, nx, ny, ovsc);
    cpl_frame_set_tag(zexpcube, SPH_ZPL_TAG_SCIENCE_IMAGING_PREPROC_CAM1);
    cpl_frame_set_group(zexpcube, CPL_FRAME_GROUP_RAW);

    fullmean_i = 0.0;
    fullmean_p = 0.0;
    for (int ii = 0; ii < nframes; ii++) {
        zplexpimg_plane = sph_zpl_exposure_imaging_load(zexpcube, ii);
        mean_i[ii] = cpl_image_get_mean(zplexpimg_plane->image_odd);
        mean_p[ii] = cpl_image_get_mean(zplexpimg_plane->image_even);
        //printf("mean (%d) = %f", ii, mean[ii]);
        fullmean_i = fullmean_i + mean_i[ii] - ovsc[0];
        fullmean_p = fullmean_p + mean_p[ii] - ovsc[2];
        sph_zpl_exposure_imaging_delete(zplexpimg_plane);
    }

    fullmean_i = fullmean_i / nframes;
    fullmean_p = fullmean_p / nframes;
    //printf("fullmean_i = %f\n ", fullmean_i);

    rerr = cpl_frameset_insert(inframes, zexpcube);

    //load fctable
    fctable = sph_fctable_load_ascii(cpl_frame_get_filename(fcframe));
    fname2 = sph_filemanager_filename_new_ext(cpl_frame_get_filename(zexpcube),
            "txt");
    fname = sph_filemanager_new_filename_from_base(fname2, "fctable");
    sph_fctable_save_ascii(fctable, fname);

    frame_fc = cpl_frame_new();
    cpl_frame_set_filename(frame_fc, fname);
    cpl_frame_set_tag(frame_fc, SPH_ZPL_TAG_FIELD_CENTER_CALIB);
    cpl_frame_set_group(frame_fc, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, frame_fc);

    //create master bias imaging (double image frame)
    master_bias_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_si_master_bias.fits.tmp", nx, ny, iframe_bias,
                    pframe_bias);
    cpl_frame_set_tag(master_bias_imaging_frame,
            SPH_ZPL_TAG_BIAS_IMAGING_CALIB_CAM1);
    cpl_frame_set_group(master_bias_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_bias_imaging_frame);

    //create master dark imaging (double image frame)
    master_dark_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_si_master_dark.fits.tmp", nx, ny, iframe_bias + 10.,
                    pframe_bias + 10.);
    cpl_frame_set_tag(master_dark_imaging_frame,
            SPH_ZPL_TAG_DARK_IMAGING_CALIB_CAM1);
    cpl_frame_set_group(master_dark_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, master_dark_imaging_frame);

    //create intensity flat imaging (double image frame)
    intensity_flat_imaging_frame =
            sph_test_zpl_simulator_create_double_image_frame(
                    "test_si_intensity_flat.tmp", nx, ny, 2., 3.);
    cpl_frame_set_tag(intensity_flat_imaging_frame,
            SPH_ZPL_TAG_IFF_IMAGING_CALIB_CAM1);
    cpl_frame_set_group(intensity_flat_imaging_frame, CPL_FRAME_GROUP_CALIB);
    rerr = cpl_frameset_insert(inframes, intensity_flat_imaging_frame);

    zpl_science_imaging = sph_zpl_science_imaging_new(inframes, inparams);
    cpl_test_nonnull( zpl_science_imaging );

    rerr = sph_zpl_science_imaging_run(zpl_science_imaging);
    cpl_test_eq_error(rerr, CPL_ERROR_NONE);

    pl = cpl_propertylist_load(
            zpl_science_imaging->science_imaging_outfilename_cam1, 0);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_IFRAME),
            (fullmean_i-iframe_bias-(iframe_bias+10))/2.0, 1e-3);
    cpl_test_abs(
            cpl_propertylist_get_double(pl, SPH_COMMON_KEYWORD_QC_MEAN_DOUBLEIMAGE_PFRAME),
            (fullmean_p-pframe_bias-(pframe_bias+10))/3.0, 1e-3);

    cpl_propertylist_delete(pl);
    pl = NULL;
    sph_zpl_science_imaging_delete(zpl_science_imaging);
    zpl_science_imaging = NULL;
}
#endif

/*----------------------------------------------------------------------------*/
/**
 @brief    Unit tests of techcal_master_dark recipe and associated functions
 */
/*----------------------------------------------------------------------------*/
int main(void) {
    const void* pSuite = NULL;


    if ( 0 != sph_test_init())
        return sph_test_get_error();


    pSuite = sph_add_suite("science_imaging_test",
            cutest_init_science_imaging_testsuite,
            cutest_clean_science_imaging_testsuite);
    if (NULL == pSuite) {
        return sph_test_get_error();
    }


#ifdef SPH_ZPL_SCIENCE_IMAGING
    if (NULL
            == sph_test_do(
                    pSuite,
                    "cutest_zpl_science_imaging_run_1: pure pre-processed frames as input rawframes",
                    cutest_zpl_science_imaging_run_1)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "cutest_zpl_science_imaging_run_2:  pre-processed frames as input rawframes and master bias, dark and iff ",
                    cutest_zpl_science_imaging_run_2)) {
        return sph_test_get_error();
    }

    if (NULL
            == sph_test_do(
                    pSuite,
                    "cutest_zpl_science_imaging_run_3:  pre-processed frames as input rawframes with overscan and master bias, dark and iff ",
                    cutest_zpl_science_imaging_run_3)) {
        return sph_test_get_error();
    }
#endif

    return sph_test_end();
}

/**@}*/
